"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaInvokeFunction = exports.InvocationType = exports.LogType = exports.AwsApiCall = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("aws-cdk-lib/core");
const api_call_base_1 = require("./api-call-base");
const providers_1 = require("./providers");
const waiter_state_machine_1 = require("./waiter-state-machine");
/**
 * Construct that creates a custom resource that will perform
 * a query using the AWS SDK
 */
class AwsApiCall extends api_call_base_1.ApiCallBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_AwsApiCallProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AwsApiCall);
            }
            throw error;
        }
        this.provider = new providers_1.AssertionsProvider(this, 'SdkProvider');
        this.provider.addPolicyStatementFromSdkCall(props.service, props.api);
        this.name = `${props.service}${props.api}`;
        this.api = props.api;
        this.service = props.service;
        if (props.outputPaths) {
            this.outputPaths = [...props.outputPaths];
        }
        this.apiCallResource = new core_1.CustomResource(this, 'Default', {
            serviceToken: this.provider.serviceToken,
            properties: {
                service: props.service,
                api: props.api,
                expected: core_1.Lazy.any({ produce: () => this.expectedResult }),
                actualPath: core_1.Lazy.string({ produce: () => this._assertAtPath }),
                stateMachineArn: core_1.Lazy.string({ produce: () => this.stateMachineArn }),
                parameters: this.provider.encode(props.parameters),
                flattenResponse: core_1.Lazy.string({ produce: () => this.flattenResponse }),
                outputPaths: core_1.Lazy.list({ produce: () => this.outputPaths }),
                salt: Date.now().toString(),
            },
            // Remove the slash from the resource type because when using the v3 package name as the service name,
            // the `service` props includes the slash, but the resource type name cannot contain the slash
            // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-customresource.html#aws-resource-cloudformation-customresource--remarks
            resourceType: `${providers_1.SDK_RESOURCE_TYPE_PREFIX}${this.name}`.substring(0, 60).replace(/[\/]/g, ''),
        });
        // Needed so that all the policies set up by the provider should be available before the custom resource is provisioned.
        this.apiCallResource.node.addDependency(this.provider);
        // if expectedResult has been configured then that means
        // we are making assertions and we should output the results
        core_1.Aspects.of(this).add({
            visit(node) {
                if (node instanceof AwsApiCall) {
                    if (node.expectedResult) {
                        const result = node.apiCallResource.getAttString('assertion');
                        new core_1.CfnOutput(node, 'AssertionResults', {
                            value: result,
                            // Remove the at sign, slash, and hyphen because when using the v3 package name or client name as the service name,
                            // the `id` includes them, but they are not allowed in the `CfnOutput` logical id
                            // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html#outputs-section-syntax
                        }).overrideLogicalId(`AssertionResults${id}`.replace(/[\@\/\-]/g, ''));
                    }
                }
            },
        });
    }
    assertAtPath(path, expected) {
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_ExpectedResult(expected);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.assertAtPath);
            }
            throw error;
        }
        this._assertAtPath = path;
        (this.outputPaths ?? (this.outputPaths = [])).push(path);
        this.expectedResult = expected.result;
        this.flattenResponse = 'true';
        return this;
    }
    waitForAssertions(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_WaiterStateMachineOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.waitForAssertions);
            }
            throw error;
        }
        const waiter = new waiter_state_machine_1.WaiterStateMachine(this, 'WaitFor', {
            ...options,
        });
        this.stateMachineArn = waiter.stateMachineArn;
        this.provider.addPolicyStatementFromSdkCall('states', 'StartExecution');
        waiter.isCompleteProvider.addPolicyStatementFromSdkCall(this.service, this.api);
        this.waiterProvider = waiter.isCompleteProvider;
        return this;
    }
}
exports.AwsApiCall = AwsApiCall;
_a = JSII_RTTI_SYMBOL_1;
AwsApiCall[_a] = { fqn: "@aws-cdk/integ-tests-alpha.AwsApiCall", version: "2.132.0-alpha.0" };
/**
 * Set to Tail to include the execution log in the response.
 * Applies to synchronously invoked functions only.
 */
var LogType;
(function (LogType) {
    /**
     * The log messages are not returned in the response
     */
    LogType["NONE"] = "None";
    /**
     * The log messages are returned in the response
     */
    LogType["TAIL"] = "Tail";
})(LogType || (exports.LogType = LogType = {}));
/**
 * The type of invocation. Default is REQUEST_RESPONSE
 */
var InvocationType;
(function (InvocationType) {
    /**
     * Invoke the function asynchronously.
     * Send events that fail multiple times to the function's
     * dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    InvocationType["EVENT"] = "Event";
    /**
     * Invoke the function synchronously.
     * Keep the connection open until the function returns a response or times out.
     * The API response includes the function response and additional data.
     */
    InvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    /**
     * Validate parameter values and verify that the user
     * or role has permission to invoke the function.
     */
    InvocationType["DRY_RUN"] = "DryRun";
})(InvocationType || (exports.InvocationType = InvocationType = {}));
/**
 * An AWS Lambda Invoke function API call.
 * Use this instead of the generic AwsApiCall in order to
 * invoke a lambda function. This will automatically create
 * the correct permissions to invoke the function
 */
class LambdaInvokeFunction extends AwsApiCall {
    constructor(scope, id, props) {
        super(scope, id, {
            api: 'invoke',
            service: 'Lambda',
            parameters: {
                FunctionName: props.functionName,
                InvocationType: props.invocationType,
                LogType: props.logType,
                Payload: props.payload,
            },
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_LambdaInvokeFunctionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, LambdaInvokeFunction);
            }
            throw error;
        }
        const stack = core_1.Stack.of(this);
        // need to give the assertion lambda permission to invoke
        new core_1.CfnResource(this, 'Invoke', {
            type: 'AWS::Lambda::Permission',
            properties: {
                Action: 'lambda:InvokeFunction',
                FunctionName: props.functionName,
                Principal: this.provider.handlerRoleArn,
            },
        });
        // the api call is 'invoke', but the permission is 'invokeFunction'
        // so need to handle it specially
        this.provider.addPolicyStatementFromSdkCall('Lambda', 'invokeFunction', [stack.formatArn({
                service: 'lambda',
                resource: 'function',
                arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                resourceName: props.functionName,
            })]);
        // If using `waitForAssertions`, do the same for `waiterProvider` as above.
        // Aspects are used here because we do not know if the user is using `waitForAssertions` at this point.
        core_1.Aspects.of(this).add({
            visit(node) {
                if (node instanceof AwsApiCall && node.waiterProvider) {
                    node.waiterProvider.addPolicyStatementFromSdkCall('Lambda', 'invokeFunction', [stack.formatArn({
                            service: 'lambda',
                            resource: 'function',
                            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                            resourceName: props.functionName,
                        })]);
                }
            },
        });
    }
}
exports.LambdaInvokeFunction = LambdaInvokeFunction;
_b = JSII_RTTI_SYMBOL_1;
LambdaInvokeFunction[_b] = { fqn: "@aws-cdk/integ-tests-alpha.LambdaInvokeFunction", version: "2.132.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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