"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isComplete = exports.onTimeout = exports.handler = void 0;
/* eslint-disable no-console */
const assertion_1 = require("./assertion");
const http_1 = require("./http");
const sdk_1 = require("./sdk");
const types = require("./types");
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify({ ...event, ResponseURL: '...' })}`);
    const provider = createResourceHandler(event, context);
    try {
        // if we are deleting the custom resource, just respond
        // with 'SUCCESS' since there is nothing to do.
        if (event.RequestType === 'Delete') {
            await provider.respond({
                status: 'SUCCESS',
                reason: 'OK',
            });
            return;
        }
        const result = await provider.handle();
        // if there is a `stateMachineArn` then we have already started a state machine
        // execution and the rest will be handled there
        if ('stateMachineArn' in event.ResourceProperties) {
            console.info('Found "stateMachineArn", waiter statemachine started');
            return;
        }
        else if ('expected' in event.ResourceProperties) {
            console.info('Found "expected", testing assertions');
            const actualPath = event.ResourceProperties.actualPath;
            // if we are providing a path to make the assertion at, that means that we have
            // flattened the response, otherwise the path to assert against in the entire response
            const actual = actualPath ? result[`apiCallResponse.${actualPath}`] : result.apiCallResponse;
            const assertion = new assertion_1.AssertionHandler({
                ...event,
                ResourceProperties: {
                    ServiceToken: event.ServiceToken,
                    actual,
                    expected: event.ResourceProperties.expected,
                },
            }, context);
            try {
                const assertionResult = await assertion.handle();
                await provider.respond({
                    status: 'SUCCESS',
                    reason: 'OK',
                    // return both the result of the API call _and_ the assertion results
                    data: {
                        ...assertionResult,
                        ...result,
                    },
                });
                return;
            }
            catch (e) {
                await provider.respond({
                    status: 'FAILED',
                    reason: e.message ?? 'Internal Error',
                });
                return;
            }
        }
        await provider.respond({
            status: 'SUCCESS',
            reason: 'OK',
            data: result,
        });
    }
    catch (e) {
        await provider.respond({
            status: 'FAILED',
            reason: e.message ?? 'Internal Error',
        });
        return;
    }
    return;
}
exports.handler = handler;
/**
 * Invoked by the waiter statemachine when the retry
 * attempts are exhausted
 */
async function onTimeout(timeoutEvent) {
    // the event payload is passed through the `errorMessage` in the state machine
    // timeout event
    const isCompleteRequest = JSON.parse(JSON.parse(timeoutEvent.Cause).errorMessage);
    const provider = createResourceHandler(isCompleteRequest, standardContext);
    await provider.respond({
        status: 'FAILED',
        reason: 'Operation timed out: ' + JSON.stringify(isCompleteRequest),
    });
}
exports.onTimeout = onTimeout;
/**
 * Invoked by the waiter statemachine when the user is waiting for a specific
 * result.
 *
 * If the result of the assertion is not successful then it will throw an error
 * which will cause the state machine to try again
 */
async function isComplete(event, context) {
    console.log(`Event: ${JSON.stringify({ ...event, ResponseURL: '...' })}`);
    const provider = createResourceHandler(event, context);
    try {
        const result = await provider.handleIsComplete();
        const actualPath = event.ResourceProperties.actualPath;
        if (result) {
            const actual = actualPath ? result[`apiCallResponse.${actualPath}`] : result.apiCallResponse;
            if ('expected' in event.ResourceProperties) {
                const assertion = new assertion_1.AssertionHandler({
                    ...event,
                    ResourceProperties: {
                        ServiceToken: event.ServiceToken,
                        actual,
                        expected: event.ResourceProperties.expected,
                    },
                }, context);
                const assertionResult = await assertion.handleIsComplete();
                if (!(assertionResult?.failed)) {
                    await provider.respond({
                        status: 'SUCCESS',
                        reason: 'OK',
                        data: {
                            ...assertionResult,
                            ...result,
                        },
                    });
                    return;
                }
                else {
                    console.log(`Assertion Failed: ${JSON.stringify(assertionResult)}`);
                    throw new Error(JSON.stringify(event));
                }
            }
            await provider.respond({
                status: 'SUCCESS',
                reason: 'OK',
                data: result,
            });
        }
        else {
            console.log('No result');
            throw new Error(JSON.stringify(event));
        }
        return;
    }
    catch (e) {
        console.log(e);
        throw new Error(JSON.stringify(event));
    }
}
exports.isComplete = isComplete;
function createResourceHandler(event, context) {
    if (event.ResourceType.startsWith(types.SDK_RESOURCE_TYPE_PREFIX)) {
        return new sdk_1.AwsApiCallHandler(event, context);
    }
    else if (event.ResourceType.startsWith(types.ASSERT_RESOURCE_TYPE)) {
        return new assertion_1.AssertionHandler(event, context);
    }
    else if (event.ResourceType.startsWith(types.HTTP_RESOURCE_TYPE)) {
        return new http_1.HttpHandler(event, context);
    }
    else {
        throw new Error(`Unsupported resource type "${event.ResourceType}`);
    }
}
const standardContext = {
    getRemainingTimeInMillis: () => 90000,
};
//# sourceMappingURL=data:application/json;base64,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