"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomResourceHandler = void 0;
/* eslint-disable no-console */
const https = require("https");
const url = require("url");
const client_sfn_1 = require("@aws-sdk/client-sfn");
// eslint-disable-next-line @typescript-eslint/ban-types
class CustomResourceHandler {
    constructor(event, context) {
        this.event = event;
        this.context = context;
        this.timedOut = false;
        this.timeout = setTimeout(async () => {
            await this.respond({
                status: 'FAILED',
                reason: 'Lambda Function Timeout',
                data: this.context.logStreamName,
            });
            this.timedOut = true;
        }, context.getRemainingTimeInMillis() - 1200);
        this.event = event;
        this.physicalResourceId = extractPhysicalResourceId(event);
    }
    /**
     * Handles executing the custom resource event. If `stateMachineArn` is present
     * in the props then trigger the waiter statemachine
     */
    async handle() {
        try {
            if ('stateMachineArn' in this.event.ResourceProperties) {
                const req = {
                    stateMachineArn: this.event.ResourceProperties.stateMachineArn,
                    name: this.event.RequestId,
                    input: JSON.stringify(this.event),
                };
                await this.startExecution(req);
                return;
            }
            else {
                const response = await this.processEvent(this.event.ResourceProperties);
                return response;
            }
        }
        catch (e) {
            console.log(e);
            throw e;
        }
        finally {
            clearTimeout(this.timeout);
        }
    }
    /**
     * Handle async requests from the waiter state machine
     */
    async handleIsComplete() {
        try {
            const result = await this.processEvent(this.event.ResourceProperties);
            return result;
        }
        catch (e) {
            console.log(e);
            return;
        }
        finally {
            clearTimeout(this.timeout);
        }
    }
    /**
     * Start a step function state machine which will wait for the request
     * to be successful.
     */
    async startExecution(req) {
        try {
            const sfn = new client_sfn_1.SFN({});
            await sfn.startExecution(req);
        }
        finally {
            clearTimeout(this.timeout);
        }
    }
    respond(response) {
        if (this.timedOut) {
            return;
        }
        const cfResponse = {
            Status: response.status,
            Reason: response.reason,
            PhysicalResourceId: this.physicalResourceId,
            StackId: this.event.StackId,
            RequestId: this.event.RequestId,
            LogicalResourceId: this.event.LogicalResourceId,
            NoEcho: false,
            Data: response.data,
        };
        const responseBody = JSON.stringify(cfResponse);
        console.log('Responding to CloudFormation', responseBody);
        const parsedUrl = url.parse(this.event.ResponseURL);
        const requestOptions = {
            hostname: parsedUrl.hostname,
            path: parsedUrl.path,
            method: 'PUT',
            headers: {
                'content-type': '',
                'content-length': Buffer.byteLength(responseBody, 'utf8'),
            },
        };
        return new Promise((resolve, reject) => {
            try {
                const request = https.request(requestOptions, resolve);
                request.on('error', reject);
                request.write(responseBody);
                request.end();
            }
            catch (e) {
                reject(e);
            }
            finally {
                clearTimeout(this.timeout);
            }
        });
    }
}
exports.CustomResourceHandler = CustomResourceHandler;
function extractPhysicalResourceId(event) {
    switch (event.RequestType) {
        case 'Create':
            return event.LogicalResourceId;
        case 'Update':
        case 'Delete':
            return event.PhysicalResourceId;
    }
}
//# sourceMappingURL=data:application/json;base64,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