#!/usr/bin/python3

"""Compute the Irreducible Discovery Rate (IDR) from NarrowPeaks files

Implementation of the IDR methods for two or more replicates.

LI, Qunhua, BROWN, James B., HUANG, Haiyan, et al. Measuring reproducibility
of high-throughput experiments. The annals of applied statistics, 2011,
vol. 5, no 3, p. 1752-1779.

Given a list of peak calls in NarrowPeaks format and the corresponding peak
call for the merged replicate. This tool computes and appends a IDR column to
NarrowPeaks files.
"""

from sys import float_info
import numpy as np
import functools
from scipy.optimize import minimize
from scipy.special import factorial
from scipy.special import logsumexp
from scipy.stats import poisson
from mpmath import polylog

import c_archimedean as c_arch


def lsum(x_values, is_log=True, axis=0):
    """
    compute log sum_i x_i
    :param x_values:
    :param is_log:
    :param axis:
    :return:
    # >>> lsum(np.array([1, 2, 3, 4]))
    # 4.440189698561196
    >>> lsum(np.array([[0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]])
    ... )
    array([2.8032607 , 2.71855056, 2.87969188])
    >>> lsum(np.array([[0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]]),
    ... axis=1)
    array([1.67247612, 1.32058311, 1.69157834, 1.50086587, 1.479448  ,
           1.62823149, 1.73370009, 1.75351973, 1.43774153, 1.67670621])
    """
    return c_arch.lsum(x_values, axis, is_log)


def lssum(x_values, x_sign=np.nan, is_log=True):
    """
    compute log sum_i x_i with sign
    :param x_values:
    :param x_sign:
    :param is_log:
    :return:
    array([3.89773594, 3.89773594, 3.89773594])
    array([2.8032607 , 2.71855056, 2.87969188])
    >>> lssum(np.transpose(np.array([[0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]])),
    ... x_sign=signff(1/3.2, np.array([1, 2, 3]), 3))
    array([1.0729724 , 0.48860043, 0.13450368, 0.28073852, 0.85281759,
           0.77384662, 1.1716088 , 0.18345907, 0.6112606 , 0.29341583])
    """
    return c_arch.llsum(x_values, signff(1/3.2, np.array([1, 2, 3]), 3), True)


def signff(alpha, j, d):
    """
    The sign of choose(alpha*j,d)*(-1)^(d-j)
    :param alpha:
    :param j:
    :param d:
    :return:
    >>> signff(1/1.2, np.array([1,2,3]), 3)
    array([1., 1., 1.])
    >>> signff(1/3.2, np.array([1,2,3]), 3)
    array([ 1., -1.,  1.])
    >>> signff(1/30.2, np.array([1,2,3]), 3)
    array([ 1., -1.,  1.])
    """
    assert 0.0 < alpha, "alpha value:" + str(alpha)
    assert alpha <= 1.0, "alpha value:" + str(alpha)
    assert d >= 0.0, "d value:" + str(d)
    return c_arch.signff(alpha, j, d)


def log1mexpunit(x):
    """
    compute log(1-exp(-a)
    :param x:
    :return:
    >>> log1mexpunit(2.0)
    -0.14541345786885906
    """
    return c_arch.log1mexp(x);
    if x <= np.log(2.0):
        return np.log(-np.expm1(-x))
    else:
        return np.log1p(-np.exp(-x))


def log1mexpvec(x):
    """
    compute log(1-exp(-a)
    :param x:
    :return:
    >>> log1mexpvec(np.array([2.0, 3.0, 4.0]))
    array([-0.14541346, -0.05106918, -0.01848545])
    """
    return np.asarray(c_arch.log1mexpvec(x))
    res = np.empty_like(x)
    eps = np.log(2.0)
    with np.errstate(invalid='ignore'):
        test = x <= eps
        res[test] = np.log(-np.expm1(-x[test]))
        res[np.logical_not(test)] = np.log1p(-np.exp(-x[np.logical_not(test)]))
    return res


def log1mexp(x):
    """
    compute log(1-exp(-a)
    :param x:
    :return:
    """
    if hasattr(x, "__len__"):
        return log1mexpvec(x)
    return log1mexpunit(x)


def log1pexp(x):
    """
    compute log(1 + exp(x))
    :param x:
    :return:
    """
    return np.logaddexp(0.0, x)


def diag_copula(u_values):
    """
    compute theta for a gumbel copula with DMLE
    :param u_values:
    :return: diagonal copula
    >>> diag_copula(np.array([
    ...    [0.72122885, 0.64249391, 0.6771109 ],
    ...    [0.48840676, 0.36490127, 0.27721709],
    ...    [0.63469281, 0.4517949 , 0.62365817],
    ...    [0.87942847, 0.15136347, 0.91851515],
    ...    [0.34839029, 0.05604025, 0.08416331],
    ...    [0.48967318, 0.99356872, 0.66912132],
    ...    [0.60683747, 0.4841944 , 0.22833209],
    ...    [0.30158193, 0.26186022, 0.05502786],
    ...    [0.51942063, 0.73040326, 0.25935125],
    ...    [0.46365886, 0.2459    , 0.83277053]
    ...    ]))
    array([0.72122885, 0.48840676, 0.63469281, 0.91851515, 0.34839029,
           0.99356872, 0.60683747, 0.30158193, 0.73040326, 0.83277053])
    """
    y = np.empty_like(u_values[:, 0])
    for i in range(u_values.shape[0]):
        y[i] = max(u_values[i, :])
    return y


def max_diag_pdf(u_values, diag_pdf, init, bounds):
    """
    find theta using dmle from diagonal pdf
    :param u_values:
    :param diag_pdf:
    :param init:
    :param bounds:
    :return:
    """

    def log_ddelta(theta, u_val):
        """
        helper function to compute the sum of the log pdf diag
        :param theta:
        :param u_val:
        :return:
        """
        return -np.sum(diag_pdf(u_values=u_val, theta=theta, is_log=True))

    res = minimize(
        fun=lambda x: log_ddelta(x, u_values),
        x0=np.array(init),
        bounds=[bounds],
        method="SLSQP"
    )
    return res.x[0]


def dmle_copula_clayton(u_values):
    """
    compute clayton theta with DMLE
    :param u_values:
    :return:
    >>> dmle_copula_clayton( np.array([
    ...    [0.72122885, 0.64249391, 0.6771109 ],
    ...    [0.48840676, 0.36490127, 0.27721709],
    ...    [0.63469281, 0.4517949 , 0.62365817],
    ...    [0.87942847, 0.15136347, 0.91851515],
    ...    [0.34839029, 0.05604025, 0.08416331],
    ...    [0.48967318, 0.99356872, 0.66912132],
    ...    [0.60683747, 0.4841944 , 0.22833209],
    ...    [0.30158193, 0.26186022, 0.05502786],
    ...    [0.51942063, 0.73040326, 0.25935125],
    ...    [0.46365886, 0.2459    , 0.83277053]
    ...    ])
    ... )
    0.2740635891728625
    """
    return max_diag_pdf(
        u_values=u_values,
        diag_pdf=diag_pdf_clayton,
        init=0.5,
        bounds=(float_info.min, 1000.0)
    )


def dmle_copula_frank(u_values):
    """
    compute frank theta with DMLE
    :param u_values:
    :return:
    >>> dmle_copula_frank(np.array([
    ...    [0.72122885, 0.64249391, 0.6771109 ],
    ...    [0.48840676, 0.36490127, 0.27721709],
    ...    [0.63469281, 0.4517949 , 0.62365817],
    ...    [0.87942847, 0.15136347, 0.91851515],
    ...    [0.34839029, 0.05604025, 0.08416331],
    ...    [0.48967318, 0.99356872, 0.66912132],
    ...    [0.60683747, 0.4841944 , 0.22833209],
    ...    [0.30158193, 0.26186022, 0.05502786],
    ...    [0.51942063, 0.73040326, 0.25935125],
    ...    [0.46365886, 0.2459    , 0.83277053]
    ...    ])
    ... )
    3.073701656631533
    """
    return max_diag_pdf(
        u_values=u_values,
        diag_pdf=diag_pdf_frank,
        init=0.5,
        bounds=(float_info.min, 745.0)
    )


def ipsi_clayton(u_values, theta, is_log=False):
    """
    compute Clayton iPsi function
    :param x:
    :param theta:
    :param is_log:
    :return:
    >>> ipsi_clayton(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    0.2)
    array([[0.18457366, 0.40468463, 0.01000981],
           [0.31794958, 0.77863565, 0.24240329],
           [0.04718611, 0.05593388, 0.69746931],
           [1.06021457, 0.16909053, 0.09682607],
           [0.20401552, 1.14748715, 0.09569219],
           [0.08378433, 0.21303069, 0.14035423],
           [0.02046784, 0.38454497, 0.07988283],
           [0.28440895, 0.03868642, 0.05402279],
           [0.34835955, 0.4268666 , 0.11740747],
           [0.08419195, 0.07552085, 0.24603517]])
    >>> ipsi_clayton(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    0.2,
    ...    is_log=True)
    array([[-1.68970664, -0.9046472 , -4.60419006],
           [-1.14586247, -0.25021206, -1.41715244],
           [-3.05365563, -2.88358502, -0.36029677],
           [ 0.05847131, -1.77732101, -2.33483902],
           [-1.58955921,  0.13757446, -2.34661855],
           [-2.47950927, -1.54631906, -1.96358583],
           [-3.88890049, -0.95569455, -2.52719433],
           [-1.25734211, -3.25226671, -2.91834924],
           [-1.05452015, -0.85128374, -2.14210478],
           [-2.47465594, -2.58334647, -1.40228078]])
    """
    return c_arch.ipsi_clayton(u_values, theta, is_log)


def psi_clayton(u_values, theta):
    """
    compute Clayton Psi function
    :param x:
    :param theta:
    :return:
    >>> psi_clayton(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    0.2)
    array([[0.16797341, 0.43186024, 0.03533839],
           [0.32574507, 0.7608775 , 0.23335089],
           [0.05379659, 0.058921  , 0.70980892],
           [0.87552668, 0.15183754, 0.0869198 ],
           [0.18914097, 0.89736349, 0.08605411],
           [0.07726802, 0.19926139, 0.12383307],
           [0.03999973, 0.40771118, 0.07451141],
           [0.28422656, 0.04910704, 0.05777555],
           [0.36343815, 0.45791558, 0.10349543],
           [0.07755952, 0.07150121, 0.23766697]])
    """
    return np.maximum(1.0 + np.sign(theta) * u_values, 0.0) ** (-1.0 / theta)


def cdf_gumbel(u_values, theta, is_log=False):
    """
    compute gumbel copula cdf
    :param u_values:
    :param theta:
    :param is_log:
    :return:
    >>> cdf_gumbel(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266],
    ...    [0.66752741, 0.66752741, 0.66752741]
    ...    ]),
    ...    2)
    array([0.1913089 , 0.4486167 , 0.21746145, 0.35144404, 0.37645985,
           0.14718965, 0.1466199 , 0.11544204, 0.31594332, 0.13185828,
           0.07929224])
    """
    return ipsi_gumbel(
        u_values=np.sum(
            psi_gumbel(
                u_values=u_values,
                theta=theta
            ),
            axis=1)
        ,
        theta=theta,
        is_log=is_log
    )


def cdf_frank(u_values, theta, is_log=False):
    """
    compute frank copula cdf
    :param u_values:
    :param theta:
    :param is_log:
    :return:
    >>> cdf_frank(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266],
    ...    [0.66752741, 0.66752741, 0.66752741]
    ...    ]),
    ...    6)
    array([0.03462875, 0.00121066, 0.0185123 , 0.00254497, 0.00124264,
           0.06281442, 0.05223709, 0.07652003, 0.01163689, 0.06994171,
           0.12129838])
    """
    return ipsi_frank(
        u_values=np.sum(
            psi_frank(
                u_values=u_values,
                theta=theta
            ),
            axis=1)
        ,
        theta=theta,
        is_log=is_log
    )


def cdf_clayton(u_values, theta, is_log=False):
    """
    compute clayton copula cdf
    :param u_values:
    :param theta:
    :param is_log:
    :return:
    >>> cdf_clayton(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266],
    ...    [0.66752741, 0.66752741, 0.66752741]
    ...    ]),
    ...    0.2)
    array([ 0.09501918, -0.0540091 ,  0.03984838, -0.0214099 , -0.03133615,
            0.20090686,  0.13875137,  0.20653633,  0.01574763,  0.20925793,
            0.33859356])
    """
    return ipsi_clayton(
        u_values=np.sum(
            psi_clayton(
                u_values=u_values,
                theta=theta
            ),
            axis=1)
        ,
        theta=theta,
        is_log=is_log
    )

def pdf_clayton(u_values, theta, is_log=False):
    """
    compute clayton copula pdf
    :param u_values:
    :param theta:
    :param is_log:
    :return:
    >>> pdf_clayton(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    -0.2,
    ...    is_log=True)
    array([ 0.64386088, -0.09219481,  0.87324848,  0.90978732,  0.81025042,
            0.453309  ,  0.53709342,  0.3984068 ,  0.66544891,  0.42404594])
    >>> pdf_clayton(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    0.2,
    ...    is_log=True)
    array([-0.12264018,  0.13487358, -0.40809375, -0.4061165 , -0.39266393,
            0.04690954, -0.10905049,  0.00406707,  0.00732412,  0.03587759])
    >>> pdf_clayton(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    0.2)
    array([0.88458189, 1.1443921 , 0.66491654, 0.66623254, 0.67525564,
           1.0480272 , 0.89668514, 1.00407535, 1.007351  , 1.03652896])
    """
    return c_arch.pdf_clayton(u_values, theta, is_log)
    d = float(u_values.shape[1])
    lu = np.sum(np.log(u_values), axis=1)
    t_var = np.sum(ipsi_clayton(u_values, theta), axis=1)
    res = np.zeros(shape=u_values.shape[0])
    if theta == 0.0:
        return res
    if theta < 0.0:
        for i in range(u_values.shape[0]):
            if t_var[i] < 1.0:
                res[i] = np.log1p(theta)
                res[i] -= (1.0 + theta) * lu[i]
                res[i] -= (d + 1.0 / theta) * np.log1p(-t_var[i])
            else:
                res[i] = -np.Inf
    else:
        res = np.sum(
            np.log1p(
                theta * np.linspace(start=1.0, stop=d - 1.0, num=int(d) - 1)
            ),
            axis=0
        ) - (1.0 + theta) * lu - (d + 1.0 / theta) * np.log1p(t_var)
    if is_log:
        return res
    else:
        return np.exp(res)


def diag_pdf_clayton(u_values, theta, is_log=False):
    """
    compute clayton copula diagonal pdf
    :param u_values:
    :param theta:
    :param is_log:
    :return:
    >>> diag_pdf_clayton(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    0.2,
    ...    is_log=True)
    array([ 0.98084835, -0.87814578,  0.58088657,  0.12305888,  0.13268952,
            0.23601368,  0.86262679,  0.66752968, -0.04581703,  0.31014911])
    >>> diag_pdf_clayton(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    0.2)
    array([2.66671759, 0.41555273, 1.78762258, 1.13095101, 1.14189541,
           1.26619164, 2.36937639, 1.94941569, 0.95521672, 1.36362842])
    """
    y = diag_copula(u_values)
    d = float(u_values.shape[1])
    if is_log:
        return np.log(d) - (1.0 + 1.0 / theta) * np.log1p((d - 1.0) *
                                                          (1.0 - y ** theta))
    return d * (1.0 + (d - 1.0) * (1.0 - y ** theta)) ** (- (1.0 + 1.0 /
                                                             theta))


def dmle_copula_gumbel(u_values):
    """
    compute theta for a gumbel copula with DMLE
    :param u_values:
    :return: theta
    >>> dmle_copula_gumbel(np.array([
    ...    [0.72122885, 0.64249391, 0.6771109 ],
    ...    [0.48840676, 0.36490127, 0.27721709],
    ...    [0.63469281, 0.4517949 , 0.62365817],
    ...    [0.87942847, 0.15136347, 0.91851515],
    ...    [0.34839029, 0.05604025, 0.08416331],
    ...    [0.48967318, 0.99356872, 0.66912132],
    ...    [0.60683747, 0.4841944 , 0.22833209],
    ...    [0.30158193, 0.26186022, 0.05502786],
    ...    [0.51942063, 0.73040326, 0.25935125],
    ...    [0.46365886, 0.2459    , 0.83277053]
    ...    ]))
    1.5138454093002933
    >>> dmle_copula_gumbel(np.array([
    ...   [0.42873569, 0.18285458, 0.9514195],
    ...   [0.25148149, 0.05617784, 0.3378213],
    ...   [0.79410993, 0.76175687, 0.0709562],
    ...   [0.02694249, 0.45788802, 0.6299574],
    ...   [0.39522060, 0.02189511, 0.6332237],
    ...   [0.66878367, 0.38075101, 0.5185625],
    ...   [0.90365653, 0.19654621, 0.6809525],
    ...   [0.28607729, 0.82713755, 0.7686878],
    ...   [0.22437343, 0.16907646, 0.5740400],
    ...   [0.66752741, 0.69487362, 0.3329266]
    ...    ]))
    1.1658220337182064
    """
    return min([1.0 + float_info.min,
                max_diag_pdf(
                    u_values=u_values,
                    diag_pdf=diag_pdf_gumbel,
                    init=1.5,
                    bounds=(1.0 + float_info.min, 100.0)
                )])


def ipsi_frank(u_values, theta, is_log=False):
    """
    Compute iPsi function for Frank copula
    :param u_values:
    :param theta:
    :param is_log:
    :return:
    >>> ipsi_frank(np.array([
    ...   [0.42873569, 0.18285458, 0.9514195],
    ...   [0.25148149, 0.05617784, 0.3378213],
    ...   [0.79410993, 0.76175687, 0.0709562],
    ...   [0.02694249, 0.45788802, 0.6299574],
    ...   [0.39522060, 0.02189511, 0.6332237],
    ...   [0.66878367, 0.38075101, 0.5185625],
    ...   [0.90365653, 0.19654621, 0.6809525],
    ...   [0.28607729, 0.82713755, 0.7686878],
    ...   [0.22437343, 0.16907646, 0.5740400],
    ...   [0.66752741, 0.69487362, 0.3329266]
    ...   ]),
    ...   0.2)
    array([[0.791148  , 1.61895993, 0.04510005],
           [1.30709475, 2.78651154, 1.02049626],
           [0.21055968, 0.24900271, 2.55444583],
           [3.51840984, 0.72823612, 0.42610361],
           [0.86923904, 3.72534678, 0.42125181],
           [0.37009377, 0.90510924, 0.60976894],
           [0.09197708, 1.54811406, 0.35325138],
           [1.1816307 , 0.17302414, 0.24062066],
           [1.41846307, 1.69593035, 0.51357717],
           [0.37185113, 0.33437415, 1.03460728]])
    """

    def mapping_function(x):
        """
        helper function for vectorize
        :param x:
        :return:
        """
        if x <= 0.01 * np.abs(theta):
            return - np.log(np.expm1(-x * theta) / np.expm1(-theta))
        else:
            if np.exp(-theta) > 0 and np.abs(theta - x * theta) < 1.0 / 2.0:
                return -np.log1p(np.exp(-theta) * np.expm1(theta - x * theta) /
                                 np.expm1(-theta))
            else:
                return -np.log1p((np.exp(-x * theta) - np.exp(-theta)) /
                                 np.expm1(-theta))

    mapping_function = np.vectorize(mapping_function)
    if is_log:
        return np.log(mapping_function(u_values))
    return mapping_function(u_values)


def psi_frank(u_values, theta):
    """
    Compute Psi function for Frank copula
    :param u_values:
    :param theta:
    :return:
    >>> psi_frank(np.array([
    ...   [0.42873569, 0.18285458, 0.9514195],
    ...   [0.25148149, 0.05617784, 0.3378213],
    ...   [0.79410993, 0.76175687, 0.0709562],
    ...   [0.02694249, 0.45788802, 0.6299574],
    ...   [0.39522060, 0.02189511, 0.6332237],
    ...   [0.66878367, 0.38075101, 0.5185625],
    ...   [0.90365653, 0.19654621, 0.6809525],
    ...   [0.28607729, 0.82713755, 0.7686878],
    ...   [0.22437343, 0.16907646, 0.5740400],
    ...   [0.66752741, 0.69487362, 0.3329266]
    ...   ]),
    ...   0.2)
    array([[0.62819295, 0.81834625, 0.36287933],
           [0.75972015, 0.93988774, 0.69230893],
           [0.42741191, 0.44210586, 0.92474177],
           [0.97065876, 0.60899809, 0.50765389],
           [0.65105904, 0.97608253, 0.50591179],
           [0.48734711, 0.66120387, 0.57101593],
           [0.38132619, 0.80627025, 0.4811596 ],
           [0.73189809, 0.41293628, 0.4389142 ],
           [0.78231684, 0.83069672, 0.53847762],
           [0.48799062, 0.47418202, 0.69595363]])
    >>> psi_frank(np.array([
    ...   [0.42873569, 0.18285458, 0.9514195],
    ...   [0.25148149, 0.05617784, 0.3378213],
    ...   [0.79410993, 0.76175687, 0.0709562],
    ...   [0.02694249, 0.45788802, 0.6299574],
    ...   [0.39522060, 0.02189511, 0.6332237],
    ...   [0.66878367, 0.38075101, 0.5185625],
    ...   [0.90365653, 0.19654621, 0.6809525],
    ...   [0.28607729, 0.82713755, 0.7686878],
    ...   [0.22437343, 0.16907646, 0.5740400],
    ...   [0.66752741, 0.69487362, 0.3329266]
    ...   ]),
    ...   -40)
    array([[0.98928161, 0.99542864, 0.97621451],
           [0.99371296, 0.99859555, 0.99155447],
           [0.98014725, 0.98095608, 0.9982261 ],
           [0.99932644, 0.9885528 , 0.98425106],
           [0.99011948, 0.99945262, 0.98416941],
           [0.98328041, 0.99048122, 0.98703594],
           [0.97740859, 0.99508634, 0.98297619],
           [0.99284807, 0.97932156, 0.9807828 ],
           [0.99439066, 0.99577309, 0.985649  ],
           [0.98331181, 0.98262816, 0.99167684]])
    >>> psi_frank(np.array([
    ...   [0.42873569, 0.18285458, 0.9514195],
    ...   [0.25148149, 0.05617784, 0.3378213],
    ...   [0.79410993, 0.76175687, 0.0709562],
    ...   [0.02694249, 0.45788802, 0.6299574],
    ...   [0.39522060, 0.02189511, 0.6332237],
    ...   [0.66878367, 0.38075101, 0.5185625],
    ...   [0.90365653, 0.19654621, 0.6809525],
    ...   [0.28607729, 0.82713755, 0.7686878],
    ...   [0.22437343, 0.16907646, 0.5740400],
    ...   [0.66752741, 0.69487362, 0.3329266]
    ...   ]),
    ...   -10)
    array([[0.95712886, 0.98171545, 0.90486527],
           [0.97485315, 0.99438248, 0.96621969],
           [0.92059451, 0.9238295 , 0.99290471],
           [0.99730587, 0.95421383, 0.93700824],
           [0.96048014, 0.99781059, 0.93668164],
           [0.93312595, 0.961927  , 0.94814684],
           [0.90964101, 0.98034637, 0.93190918],
           [0.97139377, 0.91729209, 0.92313647],
           [0.9775638 , 0.98309319, 0.94259952],
           [0.93325157, 0.93051719, 0.96670913]])
    """
    if theta > 0.0:
        return -log1mexp(u_values - log1mexp(theta)) / theta
    elif theta == 0.0:
        return np.exp(-u_values)
    elif theta < np.log(np.finfo(float).eps):
        return -log1pexp(-(u_values + theta)) / theta
    return - np.log1p(np.exp(-u_values) * np.expm1(-theta)) / theta


def diag_pdf_frank(u_values, theta, is_log=False):
    """
    compute frank copula diagonal pdf
    :param u_values:
    :param theta:
    :param is_log:
    :return:
    >>> diag_pdf_frank(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    0.2,
    ...    is_log=True)
    array([ 0.9904959 , -1.00142163,  0.6200179 ,  0.17221735,  0.18204756,
            0.28624782,  0.88191526,  0.70127801, -0.00374368,  0.35967931])
    >>> diag_pdf_frank(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    0.2)
    array([2.6925694 , 0.36735682, 1.85896133, 1.187936  , 1.19967124,
           1.33142237, 2.41552162, 2.01632796, 0.99626332, 1.43286983])
    """
    yt = diag_copula(u_values) * theta
    d = float(u_values.shape[1])

    def delt_dcom(x):
        """

        :param x:
        :return:
        """
        ep = ((np.exp(-x) - np.exp(x - theta)) / (-np.expm1(-x)))
        delt = np.exp(-x) * (1.0 + ep)
        d1 = d - 1.0
        dcom = d + d1 * ep
        dcom_time = (1.0 + ep) * delt
        return delt, dcom, dcom_time, d1

    def ddiagepoly2(x):
        """

        :param x:
        :return:
        """
        delt, dcom, dcom_time, d1 = delt_dcom(x)
        res = d1 * (d - 2.0) / 2.0 * (1.0 + (d - 3.0) / 3.0 *
                                      delt)
        res *= dcom_time
        res += dcom
        if is_log:
            return np.log(d) - np.log(res)
        return d / res

    def ddiagepoly4(x):
        """

        :param x:
        :return:
        """
        delt, dcom, dcom_time, d1 = delt_dcom(x)
        res = (d - 1.0) * (d - 2.0) / 2.0 * (1.0 + (d - 3.0) / 3.0 *
                                             delt *
                                             (1 + (d - 4.0) / 4.0 * delt * (
                                                     1.0 + (d - 5.0) / 5 * delt
                                             )))
        res *= dcom_time
        res += dcom
        if is_log:
            return np.log(d) - np.log(res)
        return d / res

    def ddiagem1(x):
        """

        :param x:
        :return:
        """
        h = -np.expm1(-theta)
        ie = -np.expm1(-x)
        res = (h / ie) ** (d - 1.0) - ie
        if is_log:
            return np.log(d) - x - np.log(res)
        return d * np.exp(-x) / res

    def mapping_function(x):
        """
        helper function to vectorize
        :param x:
        :return:
        """
        if x > 25:
            return ddiagepoly2(x)
        elif x < 0.1:
            return ddiagem1(x)
        else:
            return ddiagepoly4(x)

    mapping_function = np.vectorize(mapping_function)
    return mapping_function(yt)


def eulerian(n, m):
    """
    compute eulerian numbers
    :param n:
    :param m:
    :return:
    """
    dp = np.full((n + 1, m + 1), 0)
    # For each row from 1 to n
    for i in range(1, n + 1):
        # For each column from 0 to m
        for j in range(m + 1):
            # If i is greater than j
            if i > j:
                # If j is 0, then make that
                # state as 1.
                dp[i, j] = 1 if j == 0 else (((i - j) * dp[i - 1, j - 1]) +
                                ((j + 1) * dp[i - 1, j]))
    return dp[n, m]


@functools.lru_cache(maxsize=1, typed=False)
def eulerian_all(n):
    """
    compute eulerian number
    :param n:
    :return:
    >>> eulerian_all(10)
    array([1.000000e+00, 1.013000e+03, 4.784000e+04, 4.551920e+05,
           1.310354e+06, 1.310354e+06, 4.551920e+05, 4.784000e+04,
           1.013000e+03, 1.000000e+00])
    """
    return np.asarray(c_arch.eulerian_all(n))


def polyneval(coef, x, is_log_z=False):
    """
    :param coef:
    :param x:
    :return:
    >>> polyneval(np.array(eulerian_all(10)), np.array([4, 3]))
    array([4.42208200e+09, 8.80405504e+08])
    >>> polyneval(np.array(eulerian_all(10)), np.log(np.array([4, 3])), True)
    array([4.42208200e+09, 8.80405504e+08])
    """
    return c_arch.polyneval(coef, x, is_log_z)


def polylog(z, s, is_log_z=True):
    """
    :param z:
    :param s:
    :param is_log_z:
    :return:
    >>> polylog(np.array([0.01556112, 0.00108968, 0.00889932]), -2, False)
    array([-4.1004881 , -6.81751129, -4.68610299])
    >>> polylog(np.log(np.array([0.01556112, 0.00108968, 0.00889932])), -2,
    ... True)
    array([-4.1004881 , -6.81751129, -4.68610299])
    """
    return c_arch.polylog(z, int(s), is_log_z)


def pdf_frank(u_values, theta, is_log=False):
    """
    compute frank copula pdf
    :param u_values:
    :param theta:
    :param is_log:
    :return:
    >>> pdf_frank(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    5.0)
    array([0.1523755 , 1.83967837, 0.02685253, 0.1513627 , 0.18938125,
           1.44811361, 0.09058269, 0.21205362, 1.08804125, 0.74757727])
    >>> pdf_frank(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    5.0,
    ...    is_log=True)
    array([-1.88140742,  0.60959076, -3.6173953 , -1.88807634, -1.66399309,
            0.37026175, -2.40149211, -1.55091612,  0.08437906, -0.29091761])
    """
    return c_arch.pdf_frank(u_values, theta, is_log)


def ipsi_gumbel(u_values, theta, is_log=False):
    """
    Compute iPsi function for gumbel copula
    :param u_values:
    :param theta:
    :param is_log:
    :return:
    >>> ipsi_gumbel(np.array([
    ...   [0.42873569, 0.18285458, 0.9514195],
    ...   [0.25148149, 0.05617784, 0.3378213],
    ...   [0.79410993, 0.76175687, 0.0709562],
    ...   [0.02694249, 0.45788802, 0.6299574],
    ...   [0.39522060, 0.02189511, 0.6332237],
    ...   [0.66878367, 0.38075101, 0.5185625],
    ...   [0.90365653, 0.19654621, 0.6809525],
    ...   [0.28607729, 0.82713755, 0.7686878],
    ...   [0.22437343, 0.16907646, 0.5740400],
    ...   [0.66752741, 0.69487362, 0.3329266]
    ...   ]),
    ...   1.2)
    array([[0.81923327, 1.88908593, 0.02733237],
           [1.47231458, 3.55739554, 1.10313864],
           [0.17190186, 0.20976237, 3.21401011],
           [4.67297101, 0.74347847, 0.3959922 ],
           [0.91460233, 4.99665702, 0.39068015],
           [0.33531508, 0.95887481, 0.60372092],
           [0.06408581, 1.79316182, 0.31736126],
           [1.30892336, 0.13611697, 0.20141246],
           [1.61947932, 1.99408403, 0.49340591],
           [0.33719654, 0.29741158, 1.1209654 ]])
    >>> ipsi_gumbel(np.array([
    ...   [0.42873569, 0.18285458, 0.9514195],
    ...   [0.25148149, 0.05617784, 0.3378213],
    ...   [0.79410993, 0.76175687, 0.0709562],
    ...   [0.02694249, 0.45788802, 0.6299574],
    ...   [0.39522060, 0.02189511, 0.6332237],
    ...   [0.66878367, 0.38075101, 0.5185625],
    ...   [0.90365653, 0.19654621, 0.6809525],
    ...   [0.28607729, 0.82713755, 0.7686878],
    ...   [0.22437343, 0.16907646, 0.5740400],
    ...   [0.66752741, 0.69487362, 0.3329266]
    ...   ]),
    ...   1.2, is_log=True)
    array([[-0.19938642,  0.63609307, -3.59968356],
           [ 0.38683571,  1.26902869,  0.09815943],
           [-1.76083155, -1.56177998,  1.16751941],
           [ 1.54179506, -0.29641547, -0.92636075],
           [-0.08926592,  1.60876909, -0.93986609],
           [-1.09268464, -0.04199476, -0.50464323],
           [-2.74753233,  0.58398044, -1.14771453],
           [ 0.26920494, -1.9942407 , -1.60240044],
           [ 0.48210469,  0.69018481, -0.70642309],
           [-1.08708931, -1.21263832,  0.11419028]])
    """
    return c_arch.ipsi_gumbel(u_values, theta, is_log)


def psi_gumbel(u_values, theta):
    """
    Compute Psi function for Frank copula
    :param u_values:
    :param theta:
    :return:
    >>> psi_gumbel(np.array([
    ...   [0.42873569, 0.18285458, 0.9514195],
    ...   [0.25148149, 0.05617784, 0.3378213],
    ...   [0.79410993, 0.76175687, 0.0709562],
    ...   [0.02694249, 0.45788802, 0.6299574],
    ...   [0.39522060, 0.02189511, 0.6332237],
    ...   [0.66878367, 0.38075101, 0.5185625],
    ...   [0.90365653, 0.19654621, 0.6809525],
    ...   [0.28607729, 0.82713755, 0.7686878],
    ...   [0.22437343, 0.16907646, 0.5740400],
    ...   [0.66752741, 0.69487362, 0.3329266]
    ...   ]),
    ...   1.2)
    array([[0.61034427, 0.78449875, 0.38314216],
           [0.72866953, 0.91322228, 0.66711104],
           [0.43814072, 0.45063321, 0.89558443],
           [0.95198356, 0.59359729, 0.50641834],
           [0.63043035, 0.95944934, 0.50493238],
           [0.48911264, 0.63939466, 0.56071577],
           [0.39890033, 0.77277837, 0.48384529],
           [0.70297971, 0.42582847, 0.44791995],
           [0.74988568, 0.79662481, 0.53276337],
           [0.48966058, 0.47790784, 0.67038358]])
    """
    return np.exp(-u_values ** (1.0 / theta))


def diag_pdf_gumbel(u_values, theta, is_log=False):
    """
    compute frank copula diagonal pdf
    :param u_values:
    :param theta:
    :param is_log:
    :return:
    >>> diag_pdf_gumbel(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    0.2,
    ...    is_log=True)
    array([  -6.55858673, -257.13458817,  -50.29601565, -106.33588414,
           -105.08436706,  -91.86224008,  -19.02297494,  -40.4347328 ,
           -128.83053864,  -82.60105914])
    >>> diag_pdf_gumbel(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    0.2)
    array([1.41788815e-003, 2.12748865e-112, 1.43455743e-022, 6.59040780e-047,
           2.30377070e-046, 1.27272929e-040, 5.47553997e-009, 2.75054445e-018,
           1.12100608e-056, 1.33910865e-036])
    """
    y = diag_copula(u_values)
    d = float(u_values.shape[1])
    alpha = 1.0 / theta
    da = d ** alpha
    if is_log:
        return (da - 1.0) * np.log(y) + alpha * np.log(d)
    return da * y ** (da - 1.0)


def log_polyg(lx_var, alpha_var, d_var):
    """
    compute gumbel polylog
    :param lx_var:
    :param alpha_var:
    :param d_var:
    :return:
    >>> lsum(np.transpose(ipsi_gumbel(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    1.2,
    ...    is_log=True
    ...    )))
    array([1.00636964, 1.81365937, 1.27973155, 1.76000074, 1.84085744,
           0.64075371, 0.77684883, 0.49862315, 1.41268535, 0.56279559])
    >>> log_polyg(
    ...    lsum(np.transpose(ipsi_gumbel(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    1.2,
    ...    is_log=True
    ...    ))) * 1/1.2, 1/1.2, 3
    ... )
    array([2.24028738, 4.12345214, 2.86724735, 3.99567951, 4.1883271 ,
           1.42510898, 1.72500906, 1.11696417, 3.17657604, 1.25542124])
    >>> log_polyg(
    ...    lsum(np.transpose(ipsi_gumbel(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    3.2,
    ...    is_log=True
    ...    ))) * 1/3.2, 1/3.2, 3
    ... )
    array([ 0.35110025,  1.31419104,  1.07707314,  1.68854151,  1.80435943,
           -0.43406987,  0.23166651, -0.18316099,  0.62329368, -0.35013782])
    """
    return c_arch.log_polyg(lx_var, alpha_var, d_var)


def pdf_gumbel(u_values, theta, is_log=False):
    """
    compute frank copula pdf
    :param u_values:
    :param theta:
    :param is_log:
    :return:
    >>> pdf_gumbel(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    1.2)
    array([0.62097606, 1.39603813, 0.58225969, 0.85072331, 0.88616848,
           1.10022557, 0.66461897, 0.82092565, 1.15561848, 1.01957628])
    >>> pdf_gumbel(np.array([
    ...    [0.42873569, 0.18285458, 0.9514195],
    ...    [0.25148149, 0.05617784, 0.3378213],
    ...    [0.79410993, 0.76175687, 0.0709562],
    ...    [0.02694249, 0.45788802, 0.6299574],
    ...    [0.39522060, 0.02189511, 0.6332237],
    ...    [0.66878367, 0.38075101, 0.5185625],
    ...    [0.90365653, 0.19654621, 0.6809525],
    ...    [0.28607729, 0.82713755, 0.7686878],
    ...    [0.22437343, 0.16907646, 0.5740400],
    ...    [0.66752741, 0.69487362, 0.3329266]
    ...    ]),
    ...    1.2,
    ...    is_log=True)
    array([-0.47646275,  0.33363832, -0.54083873, -0.16166834, -0.12084819,
            0.09551522, -0.40854139, -0.19732273,  0.14463568,  0.01938713])
    >>> pdf_gumbel(np.array([
    ...    [0.42873569, 0.18285458],
    ...    [0.25148149, 0.05617784],
    ...    [0.79410993, 0.76175687],
    ...    [0.02694249, 0.45788802],
    ...    [0.39522060, 0.02189511],
    ...    [0.66878367, 0.38075101],
    ...    [0.90365653, 0.19654621],
    ...    [0.28607729, 0.82713755],
    ...    [0.22437343, 0.16907646],
    ...    [0.66752741, 0.69487362]
    ...    ]),
    ...    3.2,
    ...    is_log=True)
    array([-0.06797204,  0.11058147,  1.26625419, -2.13290342, -1.75890767,
           -0.4483621 , -5.15135971, -2.95009109,  0.94790989,  1.04000301])
    """
    return c_arch.pdf_gumbel(u_values, theta, is_log)
