# AKBS: The **A**ll **K**nowing **B**uild **S**ystem for C, C++ and Assembly

## Requirements
Python 3>=

## Installation
```bash
# Linux
python3 -m pip install akbs
# Windows (UNSUPPORTED!)
python -m pip install akbs


# Usage
python3 -m akbs
```
## Speed
To test the build system, I made a build script for [basic_math_operations](https://github.com/avighnac/basic_math_operations),

**Operating System**: Debian Bullseye
**Host**: Linux 5.10.102.1-microsoft-standard-WSL2
**Architecture**: x86_64

**AKBS**: 
Version: 1.0.5
Time: 1.351s

**CMake w/ Makefile**:
CMake Version: 3.26.0
Make Version: 4.3
Time: 3.360s

**CMake w/ Ninja**:
CMake Version: 3.26.0
Ninja Version: 1.11.1
Time: 2.750s

The `build.akbs` file is as follows
```
set(C_STD, 17)
set(CXX_STD, 17)
set(FILES, wildcard$(src/library/**/*.cpp) wildcard$(src/library/**/*.c))
set(OUTPUT_DIR, dist)
set(BUILD_DIR, build)

if(eq$($PLATFORM, POSIX))
set(FILES, wildcard$(src/library/linux/*.asm) remove$($FILES, src/library/cross-platform/addp.c, src/library/cross-platform/multiply_whole.c))
endif

check_for(C, CXX, ASM_INTEL, SHARED, STATIC)

set(OUTPUT, libbasic_math_operations.so)
compile(SHARED, $FILES)
set(OUTPUT, libbasic_math_operations.a)
compile(STATIC, $FILES)
```
**Note:** The `remove$()` helper function is used because the `multiply_whole` and `addp` functions are already present in Assembly for Linux, not for Windows

## How do you use AKBS?
By default, the build script is called `build.akbs`, similar to `Makefile` and `CMakeLists.txt`
However, you can use the `--file` option to specify a file

To enable languages, you use the `check_for` function
```
check_for(C, CXX, ASM_INTEL, ASM_ATT, STATIC, SHARED)
```
Right now, only these 4 languages (and a static and shared library linker) (3 if you count AT&T and Intel syntax Assembly as one language) are supported

If you want to set the standard of C and C++, set the C_STD or CXX_STD variable
```
set(C_STD, 17)
set(CXX_STD, 17)
```

To compile a list of files, use the compile function
```
compile(SHARED/STATIC, src/a.c src/b.c src/c.c)
```

To print a statement you can use the print function and to exit a program, use exit
```
print($PLATFORM)
exit(1)
```
To use a variable, use `$VARIABLENAME`

The `$PLATFORM` variable comes predefined and is set to `os.name`

For conditions, use if (else and else if are not implemented yet) and endif.

```
if(set$(PLATFORM))
print($PLATFORM)
if(eq$($PLATFORM, UNIX))
print Yay, we're in UNIX land
endif
endif
```

Also, there is a rudimentary pre-processor, with `%define`

```
%define ifend endif
if(set$(PLATFORM))
ifend
```

Comments work by adding a semi-colon at the start of the line
```
; these   
; lines   
; will    
; be      
; skipped 
```
**Comments only work from the start of the line**

There is also a list of helper functions

| Function Name | Arguments | Description | Introduced
|---|---|---|---|
| wildcard$ | str1 | Evaluates a list of space separated globs into a space separated list of files | v1.0.0
| remove$ | str1, str2, str3... | Removes str2 onwards from a space separated list of strings | v1.0.0
| replace$ | str1, str2, str3... | Replaces str2,4,6,8... with str3,5,7,9... in str1 | v1.0.3
| eq$ | arg1, arg2 | Checks if two strings are equal | v1.0.0
| neq$ | arg1, arg2 | Checks if two strings are unequal | v1.0.3
| gt$ | arg1, arg2 | Checks if arg1 is greater than arg2 | v1.0.3
| lt$ | arg1, arg2 | Checks if arg1 is lesser than arg2 | v1.0.3
| gte$ | arg1, arg2 | Checks if arg1 is greater than or equal to arg2 | v1.0.3
| lte$ | arg1, arg2 | Checks if arg1 is lesser than or equal to arg2 | v1.0.3
| set$ | arg1 | | Checks if there is a variable with the name arg1 | v1.0.3 
| notset$ | arg1 | Checks if there is not a variable with the name arg1 | v1.0.3 
| and$ | arg1, arg2, arg3... | Ands all the booleans | v1.0.4
| or$ | arg1, arg2, arg3... | Ors all the booleans | v1.0.4
| not$ | arg1 | Nots the boolean | v1.0.4


A list of important variables are
| Variable | Is Set | Description | Introduced |
|---|---|---|---|
| PLATFORM | Yes | Equivalent of `os.name` | v1.0.0 |
| C_COMPILER | No | C compiler location set by `check_for` | v1.0.0 |
| CXX_COMPILER | No | C++ compiler location set by `check_for` | v1.0.0 |
| ASM_INTEL_COMPILER | No | Intel Assembly assembler location set by `check_for` | v1.0.0 |
| ASM_ATT_COMPILER | No | AT&T Assembly assembler location set by `check_for` | v1.0.0 |
| SHARED_COMPILER | No | Linker location for shared libraries set by `check_for` | v1.0.0 |
| STATIC_COMPILER | No | Linker location for static libraries set by `check_for` | v1.0.2 |
| OUTPUT | No | Output file generated by linking | v1.0.0 |
| C_STD | No | The C std used (just the number like 17, 11, etc.) | v1.0.0 |
| CXX_STD | No | The C++ std used (just the number like 17, 11, etc.) | v1.0.0 |
| BUILD_DIR | No | The directory to build the objects in | v1.0.1 |
| OUTPUT_DIR | No | The directory to output the finished objects in in | v1.0.1 |
| C_FLAGS | No | Flags passed to C compiler | v1.0.2 |
| CXX_FLAGS | No | Flags passed to C++ compiler | v1.0.2 |
| ASM_INTEL_FLAGS | No | Flags passed to Intel Assembly assembler | v1.0.2 |
| ASM_ATT_FLAGS | No | Flags passed to AT&T Assembly assembler | v1.0.2 |
| SHARED_FLAGS | No | Flags passed to shared library linker | v1.0.2 |
| STATIC_FLAGS | No | Flags passed to static library linker | v1.0.2 |

## How to clean the files
```bash
python3 -m akbs --clean
```

## To-Do
* [x] Build directory (milestone 1.0.1)
* [x] Nested functions
* [ ] Windows support
* [ ] Optimization
* [x] Subdirectories
* [x] Ability to set compilers from `set()` and environment variables
* [x] More if conditions
* [x] `replace$()` helper function
* [x] --clean command
* [x] Documenting my code + Readable variables
* [x] C_FLAGS, CXX_FLAGS, ASM...
* [x] Cache install locations (milestone 1.0.2)
* [x] Ability to set target architecture (Use CFLAGS and CXXFLAGS)
* [ ] Plugin Support
* [x] Make `print` and `exit` a function, not a statement
* [ ] Make comments work in the middle of a line