"""
Created by Epic at 12/20/20
"""
from discord.ext.commands import Bot, Group, Context, errors, Command, converter
from discord.http import Route
from discord import Message, Guild, User, TextChannel, Role, VoiceChannel
from logging import getLogger

param_types = {
    int: 4,
    converter.IDConverter: 4,
    bool: 5,
    User: 6,
    converter.UserConverter: 6,
    TextChannel: 7,
    VoiceChannel: 7,
    converter.TextChannelConverter: 7,
    converter.VoiceChannelConverter: 7,
    Role: 8,
    converter.RoleConverter: 8,
}


class Slashify:
    def __init__(self, bot: Bot):
        self.bot = bot
        self.bot.add_listener(self.on_dispatch, "on_socket_response")
        self.logger = getLogger("slashify")
        self.bot.loop.create_task(self.start())

    def parse_param(self, old_param):
        param_name, param_value = old_param
        param = {
            "name": param_name,
            "description": "Parameter %s" % param_name,
            "optional": False
        }
        annotation = param_value.annotation

        if isinstance(annotation, Optional):
            param["required"] = False
            annotation = annotation.basetype
        else:
            param["required"] = True
        if isinstance(annotation, Choices):
            param["type"] = param_types.get(annotation.basetype, 3)
            param["choices"] = [{"name": name, "value": value} for name, value in annotation.choices]
        else:
            param["type"] = param_types.get(annotation, 3)
        return param

    def parse_params(self, params):
        new_params = []
        for param_name, param_value in params.items():
            param = self.parse_param((param_name, param_value))
            new_params.append(param)
        return new_params

    async def start(self):
        await self.bot.wait_until_ready()
        for command in self.bot.commands:
            command_data = self.add_command(command)
            route = Route("POST", "/applications/{client_id}/commands", client_id=self.bot.user.id)
            await self.bot.http.request(route, json=command_data)

    def add_command(self, command):
        if isinstance(command, Group):
            self.logger.error("Error while adding command", exc_info=TypeError("Groups are not supported yet."))
            return
        command_data = {
            "name": command.name,
            "description": command.description or "Command autogenerated by Slashify",
            "options": self.parse_params(command.clean_params)
        }
        self.logger.info("Added slash command: %s" % command_data["name"])
        return command_data

    async def parse_args(self, context, command: Command, values):
        mappings = {item["name"]: item["value"] for item in values}
        parsed = {}
        for name, param in command.clean_params.items():
            converter = command._get_converter(param)
            mapping = mappings.get(name)
            if mapping is None:
                continue
            parsed[name] = await command.do_conversion(context, converter, mapping, name)
        return parsed

    async def on_dispatch(self, data):
        if data["t"] != "INTERACTION_CREATE":
            return
        data = data["d"]
        guild: Guild = self.bot.get_guild(int(data["guild_id"]))
        channel = guild.get_channel(int(data["channel_id"]))

        # Hacking together fields lol
        data["attachments"] = []
        data["embeds"] = []
        data["edited_timestamp"] = None
        data["pinned"] = False
        data["mention_everyone"] = False
        data["tts"] = False
        data["content"] = None
        data["author"] = data["member"]["user"]

        message = Message(state=self.bot._connection, channel=channel, data=data)
        command_name = data["data"]["name"]
        context = SlashContext(
            prefix="/",
            message=message,
            bot=self.bot,
            invoked_with=command_name,
            command=self.bot.get_command(command_name),
            args=[]
        )
        self.bot.dispatch('command', context)
        try:
            if not await self.bot.can_run(context):
                raise errors.CheckFailure('The global check once functions failed.')
        except errors.CommandError as exc:
            await context.command.dispatch_error(context, exc)
        else:
            self.bot.dispatch('command_completion', context)
        params = await self.parse_args(context, context.command, data["data"].get("options", []))
        context.kwargs = params
        await context.command(context, **params)


class CustomAnnotation:
    def __init__(self, basetype=None):
        self.basetype = basetype

    async def convert(self, ctx, arg):
        if self.basetype is not None:
            return await self.basetype.convert(ctx, arg)
        return arg


class Choices(CustomAnnotation):
    def __init__(self, choices, basetype=None):
        self.choices = choices
        super().__init__(basetype)


class Optional(CustomAnnotation):
    pass


class SlashContext(Context):
    pass
