# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/05_pds.ipynb (unless otherwise specified).

__all__ = ['CatalogFilter']

# Cell
from datetime import date as dtdate
from datetime import timedelta
from pathlib import Path

from yarl import URL

import hvplot.pandas  # noqa
from nbverbose.showdoc import show_doc
from planetarypy.pds.apps import find_indexes, get_index
from planetarypy import utils

# Cell
class CatalogFilter:
    "useful to make class to keep the base catalog in memory"

    def __init__(self, date: str = None):
        self.date = date
        df = get_index("cassini.uvis", "index")
        df["product_id"] = df.FILE_NAME.map(lambda x: Path(x).stem)
        df["volume"] = df.FILE_NAME.map(lambda x: x.split("/")[1])
        new_cols = ["product_id", "volume"] + list(df.columns)[:-2]
        df = df[new_cols]
        df.set_index("product_id", inplace=True)
        df.OBSERVATION_TYPE.fillna("UNKNOWN", inplace=True)
        self.df = df

    @property
    def date(self):
        return self._date

    @date.setter
    def date(self, value: str):
        date = value.replace("_", "-")
        tokens = date.split("-")
        if len(tokens) == 2:
            date = utils.nasa_date_to_datetime(date)
        elif len(tokens) == 3:
            tokens = [int(i) for i in tokens]
            date = dtdate(*tokens)
        self._date = date

    def set_next_day(self):
        next_day = self.date + timedelta(days=1)
        self.date = next_day.isoformat()[:10]

    @property
    def isodate(self):
        return self.date.isoformat()[:10]

    @property
    def nasadate(self):
        return utils.iso_to_nasa_date(self.isodate)

    @property
    def piddate(self):
        return self.nasadate.replace("-","_")
    @property
    def fuvdate(self):
        return "FUV"+self.piddate

    @property
    def euvdate(self):
        return "EUV"+self.piddate

    @property
    def uv(self):
        "return E/F_UV only"
        return self.df[self.df.index.map(lambda x: x[:3] in ["EUV", "FUV"])]

    def _get_clean_date_filter(self, date: str = None):
        date = self.date if date is None else date
        if "-" in date:
            date = date.replace("-", "_")
        return date

    def get_fuv_date(self, date: str = None):  # datestring like yyyy-jjj, e.g. 2010-198
        date = self.date if date is None else date
        return self.df[self.df.index.str.startswith(self.fuvdate)]

    def get_euv_date(self, date: str = None):  # datestring like yyyy-jjj
        date = self.date if date is None else date
        return self.df[self.df.index.str.startswith(self.euvdate)]

    @property
    def ustare_stars(self):
        return self.uv.query("OBSERVATION_TYPE=='USTARE' and TARGET_NAME=='STAR'")

    @property
    def target_calib(self):
        return self.uv.query("TARGET_NAME=='CALIB'")

    def get_ustare_stars_date(
        self,
        date: str = None,  # datestring like yyyy-jjj
        uvtype: str = "",  # add EUV/FUV filter
    ):
        date = self.date if date is None else date
        _filter = uvtype + self.nasadate
        df = self.ustare_stars
        return df[df.index.str.contains(_filter)]