# -*- coding: utf-8 -*-
"""Traffic_Network_model.ipynb

Automatically generated by Colaboratory.

Original file is located at
    https://colab.research.google.com/drive/1Xsbp-8Ax61ZXnB3xkBWNe5J61coYWmXF

# IMPORT
"""

#All the needed imports for our code.
import numpy as np
import numpy.random as rd
import math
rd.seed(seed=1)# Seed positive for repetable result.
 
import matplotlib as mpl
import matplotlib.pyplot as plt
import matplotlib.cm as cm
from matplotlib.colors import Normalize

import networkx as nx
import inspect
import os, sys
import copy
#from termcolor import colored

from google.colab import drive
import cv2
import glob

drive.mount('/content/drive')

"""# ERROR DEFINITION

Implementation of the error use in our code
"""

class Error(Exception): 
    """
    Abstract class defining how we create and call our error.
    """
    def __init__(self,message=""):
        self.message=message #string giving the error message to print
    
    def __str__(self):
        return self.message

class ConnectionError(Error):
    """
    Error class generating all errors related to a bad network creation. (missing road/intersection, missing network conection...)
    """

class TypeCarError(Error):
    """
    Error class generating all errors related to a wrong type (type of car, right now between 0 and 3).
    """

class TypeRoadError(Error):
    """
    Error class generating all errors related to a wrong type (type of car, right now : (normal, entry, exit)).
    """

class CompileError(Error):
    """
    Error class generating all errors related to a network with bad or no conditioning (missing one function call, or to big step)
    """

class PriorityError(Error): #reprendre le docstring
    """
    Error class generating error related to wrong list of pod for priority  
    """

class NotImplementedError(Error):
    """
    Error class generating all errors linked to an abstract class method not been proprely inmplemented
    """

class ModelImportingError(Error): 
    """
    Error class generating all errors linked to unability to imported a model
    """

class OtherErrors(): 
    """
    Error class generating all errors we can't relly define (all the error tha don't fall in the above category).
    """

"""# UTILITIES"""

class HiddenPrints: # https://stackoverflow.com/questions/8391411/how-to-block-calls-to-print
    """
    Class define for hiding print in part of the code
    """
    def __enter__(self):
        self._original_stdout = sys.stdout
        sys.stdout = open(os.devnull, 'w')

    def __exit__(self, exc_type, exc_val, exc_tb):
        sys.stdout.close()
        sys.stdout = self._original_stdout

class Utilities:
    #Class used to get all class and method descriptions of the module.
    """
    Class used to stock all the utility functions
    most are for personal use and understanding of the model, some are use in our model.
    """
    def __init__(self):
        """
        no attribute
        """
    
    def show_information(self,function):
        """
        show the doc of a function or a class

        Parameters
        ----------
            Function : the function about which we want the doc
            
        Returns 
        -------
            None
        """
        print(function.__doc__)
    
    def get_function_parameters(self,function): 
        """
        return the list of the attributs and default value of a function

        Parameters
        ----------
            Function : the function about which we want the param
            
        Returns 
        -------
            (list tuple) : list of tuple, each tuple correspond to one arg, 
                          tuple [arg_name, type,  default value] : (string, string, Object)
                          if no default value then [arg_name, type]
        """
        signature = inspect.signature(function)
        list_arg=[]
        for i, j in (signature.parameters.items()):
            if (j.default is inspect.Parameter.empty):
                tmp=(i,type(j.default))
            else:
                tmp=(i,type(j.default),j.default)
            list_arg.append(tmp)

        return list_arg
    
    def get_list_function_name(self,Class):
        """
        return the list of the name of function for one class

        Parameters
        ----------
            Class : the class about wich we want the list of function
            
        Returns 
        -------
            (list str) : the list of the name of the function
        """
        tmp=inspect.getmembers(Class, predicate=inspect.isfunction)
        return [ x[0] for x in tmp ]
    
    def get_list_function(self,Class):
        """
        return the list of the of function for one class (the object, not it's name)

        Parameters
        ----------
            Class : the class about wich we want the list of function
            
        Returns 
        -------
            (list object) : the list of the name of the function
        """
        tmp=inspect.getmembers(Class, predicate=inspect.isfunction)
        return [ x[1] for x in tmp ]
    
    def get_list_class_name(self):
        """
        return the list of the name of class in our model (list of class in memory right now)

        Parameters
        ----------
            None
            
        Returns 
        -------
            (list str) : the list of the name of the class
        """
        tmp = inspect.getmembers(sys.modules[__name__], inspect.isclass)
        return [ x[0] for x in tmp ]
    
    def get_list_class(self):
        """
        return the list of the class in our model (the object, not it's name) (list of class in memory right now)

        Parameters
        ----------
            None
            
        Returns 
        -------
            (list object) : the list of the name of the function
        """
        tmp = inspect.getmembers(sys.modules[__name__], inspect.isclass)
        return [ x[1] for x in tmp ]

    def get_class_attribute(self,Class):
        """
        return the list of the attribute name of a class
        Parameters
        ----------
            Class (object) : the class about wich we want the attribute
            
        Returns 
        -------
            (list object) : the list of the name of the attribute
        """
        class_name=str(Class.__class__.__name__)
        private_attribute_name_begin="_"+class_name
        list_attribute_name=[]
        for i in inspect.getmembers(Class):  
            # To remove methods
            if not inspect.ismethod(i[1]):
                # to remove attribute bound to object and only keeping the one from our actual class
                if (not i[0].startswith('_')):
                    list_attribute_name.append(i[0])
                elif (i[0].startswith(private_attribute_name_begin)):
                    tmp=i[0].replace(private_attribute_name_begin,"")
                    list_attribute_name.append(tmp)
        return list_attribute_name
    
    def show_all(self,more=False):
        """
        print the list of all information concerning our model: all the class, and all there method

        Parameters
        ----------
            more (boolean) optional : bolean defining if we also print the list of arg for each function; by default False
            
        Returns 
        -------
            None
        """
        class_list=self.get_list_class()
        class_list_name=self.get_list_class_name()
        s0=""
        indent="    "
        for i in range(len(class_list)):
            s0+=str(class_list_name[i])
            s0+="\n"
            s1=""
            if (more):
                function_list=self.get_list_function(class_list[i])
            function_list_name=self.get_list_function_name(class_list[i])
            for j in range(len(function_list_name)):
                s1+=indent
                s1+=str(function_list_name[j])
                s1+="\n"
                if (more):
                    arg_list=self.get_function_parameters(function_list[j])
                    param_name = " ".join([ str(x[0]) for x in arg_list])
                    s1+=indent*2
                    s1+=param_name
                    s1+="\n"
            s0+=s1
        print(s0)
    
    def get_parent_class(self,Class):
        """
        give the list of parent class (if a class inherite of several class, it will give a list with multiple element, otherwise, only a list of one class)

        Parameters
        ----------
            Class (Class) : the class about wich we want the information
            
        Returns 
        -------
            (list Class) : list of parent class
        """
        try:
            iter=Class.__bases__
        except AttributeError:
            tmp=Class.__class__
            iter=tmp.__bases__
        except Exception as e:
            print("Unespected error")
            raise e
        
        list_parent_class=[] #list because of multi-heritage
        for i in iter:
            list_parent_class.append(i)
        return list_parent_class
    
    def get_ancestor_class(self,Class):
        """
        give the list of ancestor class (give all the class from wich this class has inherited, directly or not)

        Parameters
        ----------
            (Class Class) : the class about wich we want the information
        
        Returns 
        -------
            (list Class) : list of ancestor class
        """
        try:
            iter=Class.__mro__
        except AttributeError:
            tmp=Class.__class__
            iter=tmp.__mro__
        except Exception as e:
            print("Unespected error")
            raise e
        
        list_parent_class=[] #list because of multi-heritage
        for i in iter:
            list_parent_class.append(i)
        return list_parent_class
    
    def get_subclasses(self,Class):
        """
        give the list of ancestor class (give all the class from wich this class has inherited, directly or not)

        Parameters
        ----------
            (Class Class) : the class about wich we want the information
        
        Returns 
        -------
            (list Class) : list of ancestor class
        """
        try:
            tmp=Class.__subclasses__()
            return tmp
        except AttributeError:
            tmp=Class.__class__
            tmp2=tmp.__subclasses__()
            return tmp2
        except Exception as e:
            print("unespected error")
            raise e
    
    # def get_car_type(self): #déplacer dans la class Cars en tant que méthode 
    #     """
    #     give the dict of the different type of cars (class that inherited of Cars)

    #     Parameters
    #     ----------
    #         None
        
    #     Returns 
    #     -------
    #         (dict Class) : dict of child class of Cars (the key of the dict are the type number of a class)
    #     """
    #     list_class=self.get_subclasses(Cars)
    #     dict_class=dict()
    #     for i in range(len(list_class)):
    #         class_object=list_class[i](0,0,0,0)  #create an instance of Cars to then get the number associated with the type of driver
    #         if (class_object.type_name in dict_class.keys()):
    #             print("Warning, key name confict for Car type")
    #             message=""
    #             message+="class : "
    #             message+=str(dict_class[i].__name__)
    #             message+=" is already named "
    #             message+=str(dict_class[i](0,0,0,0).type_name)
    #             message+="\n"
    #             message+="class : "
    #             message+=str(class_object.__class__.__name__)
    #             message+=" will replace it"
    #             print(message)
    #         dict_class[class_object.type_name]=list_class[i]
    #     return dict_class

"""# ACCESS"""

class Access:
    """
    class to manage everything linked to drive access, reading and wrtiting
    """
    def __init__(self):
        self.__path="/content/drive/My Drive/Road_Network/result/"  #path toward the project directory
        self.__tmp_video="tmp_video/" #name of the folder for storing picture for video
        self.__video="video/" #name of the folder for storing video
        self.__model="model/" #name of the folder where we store network configuration

        #format
        self.__picture_format=".png" #picture format
        self.__video_format=".mp4" #video format
        self.__text_format=".txt" #text format
    
    def __repr__(self):
        """ redefining print """
        s=""
        s+="path "
        s+="\n"
        s+="_"*20
        s+="\n"
        s+="global path : "
        s+=str(self.__path)
        s+="\n"
        s+="temporary picture folder : "
        s+=str(self.__tmp_video)
        s+="\n"
        s+="video folder : "
        s+=str(self.__video)
        s+="\n"
        s+="model folder : "
        s+=str(self.__model)
        s+="\n"
        s+="\n"

        s+="format"
        s+="\n"
        s+="_"*20
        s+="\n"
        s+="picture format : "
        s+=str(self.__picture_format)
        s+="\n"
        s+="video format : "
        s+=str(self.__video_format)
        s+="\n"
        s+="text format : "
        s+=str(self.__text_format)

        return s
    
    def contain_file(self,file_name,path_to_directory): #directory_contains_file
        """
        test if a file is in the directory

        Parameters
        ----------
            file_name (string)              : the name of the file we want to test
            path_to_directory (string) : the path to the directory
        
        Returns 
        -------
            Boolean
        """
        list_file = os.listdir(path_to_directory)

        return (file_name in list_file)

    def save_picture(self,picture,name):
        """
        save the picture in tmp folder

        Parameters
        ----------
            picture (picture object from plt) : the picture object we save
            name (string) : the name of the picture we save
        
        Returns 
        -------
            None
        """
        picture.savefig(self.__path+self.__tmp_video+str(name)+self.__picture_format)
        plt.close()
    
    def transform_list_picture_to_video(self,fps,name): #transform_pictures_to_video
        """
        transform the list of pictures from the temporary picture directory to a video
        and save this video in the video file

        Parameters
        ----------
            fps (int) : frame per second, the number of pictures we place per second in our video
            name (string) : the name of the video we save
        
        Returns 
        -------
            None
        """
        file_name=name+self.__video_format
        path_to_directory=self.__path+self.__video
        if (self.contain_file(file_name,path_to_directory)):
            raise OtherErrors("name already used, please use another name")

        img_array = []
        for filename in glob.glob(self.__path+self.__tmp_video+"*"+self.__picture_format):
            img = cv2.imread(filename)
            height, width, layers = img.shape
            size = (width,height)
            img_array.append(img)
 
        fourcc = cv2.VideoWriter_fourcc(*'MP4V')
        
        out = cv2.VideoWriter(self.__path+self.__video+str(name)+self.__video_format,fourcc, fps, size)
 
        for i in range(len(img_array)):
            out.write(img_array[i])
        out.release()
    
    def suppres_temporary_video_directory(self):   #suppress
        """
        use with cautions
        suppresse all png file in result/tmp/video

        Parameters
        ----------
            None

        Returns 
        -------
            None
        """
        mydir=self.__path+self.__tmp_video
        for i in os.listdir(mydir):
            tmp="".join(reversed(i[-1: -5: -1]))
            if (tmp==".png"):
                file_name=mydir+i
                os.remove(file_name)
    
    def save_model(self,string,name):
        """
        save the network config in the model folder

        Parameters
        ----------
            string (string) :
            name (string) : the name of the file we create
        
        Returns 
        -----
            None
        """
        file_name=name+self.__text_format
        path_to_directory=self.__path+self.__model
        if (self.contain_file(file_name,path_to_directory)):
            raise OtherErrors("name already use, please use another name")
        
        path=self.__path+self.__model+str(name)+self.__text_format
        with open(path, 'w') as model:
            model.write(string)
    
    def load_model(self,name):
        """
        load the network config from the model folder

        Parameters
        ----------
            name (string) : the name of the model we load
        
        Returns 
        -----
            string : the string corresponding to the model we want to load.
        """
        path=self.__path+self.__model+str(name)+self.__text_format
        with open(path, 'r') as model:
            return (model.read())

"""# Speed

The different types of speed used in the road class for the calculation of speeds in the latter.
"""

class Speed:
    """
    Abstract class defining speed class
    """
    
    def speed(*args, **kwargs):
        """
        abstract method defining the speed calcultion metric use for computing speed on a road.
        """
        raise NotImplementedError()

class LinearSpeed(Speed): #remplacer "Linear" par autre chose
    """
    Linear speed
    """

    def __init__(self,coeff=1):
        """
        Parameters
        ----------
            coeff (int) optional : epsilon coefficient of the speed calculation (the smaller it is the higher the speed d)
                                   should be in the interval ]0,1] by default 1
        """
        self.epsilon=coeff #epsilon coefficient of the speed calculation (should be small)

    def speed(self,vmax,l,d):
        """
        calculates the speed on a road
            implement the abstract method

        Parameters
        ----------
            vmax (int) : maximum speed on the road
            l (int) : length of the car
            d (int) : distance from the previous car

        Returns 
        -------
            (int) : the calculated speed
        """
        if (d<10**-6): #to avoid an error following a division by zeros
            return 0
        v=max(vmax*min(1,(1-(l/d))/self.epsilon),0)
        return v
    
    def __repr__(self):
        s="linear speed calculation function"
        s+=", epsilon : "+str(self.epsilon)
        return s

"""# Weight

The different types of weights, wich are, the different ways of calculating the shortest path.

Each weights corresponds to a different way of calculating the weights assigned to the roads during the shortest path.

2 examples are the length of the roads, or the time needed to cross it if it's empty.
"""

class Abstract_weight:
    """
    Abstract class
    class used to define different type of weight for the road cost.
    """
    def __init__(self):
        """
        abstract class
        """

    def __hash__(self):
        """redefine hash to use weight types as key"""
        return hash(str(self))
    
    def __eq__(self, other):
        """redefine eq to use weight types as key"""
        return (self.__class__ == other.__class__)
    
    def calculate_road_weight(self,road):
        """
        Abstract method to redefine
        """
        raise NotImplementedError("this is an abstract method, you must redefine it for your class")

class Abstract_static_weight(Abstract_weight):
    """
    Abstract class
    every static class weight must inerited this class.
    Only use for differiencing static from dynamic weight.
    (static weight doesn't change during the simulation, dynamic does, so it must be recompute at each weight.)
    """

class Abstract_dynamic_weight(Abstract_weight):
    """
    Abstract class
    every dynamic class weight must inerited this class.
    Only use for differiencing static from dynamic weight.
    (static weight doesn't change during the simulation, dynamic does so it must be recompute at each weight.)
    """

"""Real class"""

class Weight(Abstract_static_weight):
    """
    class used to define different type of weight for the length of a road.
    here: road length
    """
    
    def calculate_road_weight(self,road):
        """
        calculates the weight of the road

        Parameters
        ----------
            road (Road) : the road whose weight is calculated

        Returns 
        -------
            (float) : the weight of the road (here its length)
        """
        return road.l

    def __repr__(self):
        return "Weight"

class Weight_duration_empty(Abstract_static_weight):
    """
    Class used to define different type of weight for the length of a road.
    Here time that a car would take to cover it when it's empty
    (length/speed with speed = max speed of the road)
    """
    def calculate_road_weight(self,road):
        """
        calculates the weight of the road

        Parameters
        ----------
            road (Road) : the road whose weight is calculated

        Returns 
        -------
            (float) : the weight of the road (here the time it will take to cover it (when it's empty))
        """
        t=road.l/road.vmax
        return t
    
    def __repr__(self):
        return "Weight_duration_empty"

class Weight_duration(Abstract_dynamic_weight):
    """
    Class used to define different type of weight for the length of a road.
    Here time that a car would take to cover it on average.
    (length/speed with speed = speed of the road with distance d = average distance between the different cars)
    """
    def calculate_road_weight(self,road):
        """
        calculates the weight of the road

        Parameters
        ----------
            road (Road) : the road whose weight is calculated

        Returns 
        -------
            (float) : the weight of the road (here the time it will take to cover it)
        """
        vmax=road.vmax
        if (len(road.cars)>0):
            d=road.l/(1+len(road.cars))
            l=road.cars[0].lenght
            v=road.class_speed.speed(vmax,l,d)
            if (v>10**-6):
                t=road.l/v
            else:
                t=(road.l)/10**(-6)
        else:
            t=road.l/vmax
        return t
    
    def __repr__(self):
        return "Weight_duration_empty"

class Weight_duration_discretized(Abstract_dynamic_weight):
    """
    Class used to define different type of weight for the length of a road.
    Here time that a car would take to cover it by discretizing .
    T (time = some of tk)
    tk : dk/vk | dk= pk+1-pk distance beetween a car and the car befor it
                 vk speed for the car pk 
    """
    def __division(self,d,v):
        """
        divide d by v, make sure they don't cause a problem of dividing by zeros by sending a huge number (if zeros or close to it is detect)

        Parameters
        ----------
            d (float) : the distance
            v (float) : the speed

        Returns 
        -------
            (float) : the time : d/v
        """
        if (v>10**(-6)):
            return d/v
        else:
            return d*10**6


    def calculate_road_weight(self,road):
        """
        calculates the weight of the road

        Parameters
        ----------
            road (Road) : the road whose weight is calculated

        Returns 
        -------
            (float) : the weight of the road (here the time it will take to cover it)
        """
        n=len(road.cars) #number of cars on the road.
        vmax=road.vmax #max speed on the road
        d=road.l # length of the road
        if (n==0):
            return self.__division(d,vmax)
        else:
            #time between the entry and the car closest to the entry.
            T=0
            d0=road.cars[n-1].position
            l0=road.cars[n-1].lenght
            v0=road.class_speed.speed(vmax,l0,d0)
            T+=self.__division(d0,v0)

            for i in range(n-1,0,-1): #time between the car i+1 and i (car i+1 is)
                di=road.cars[i].position
                di_1=road.cars[i-1].position
                li=road.cars[i].lenght
                vi=road.class_speed.speed(vmax,li,di_1-di)
                T+=self.__division(di_1-di,vi)
            
            #time between the car closest to the exit and the exit
            dn=road.cars[0].position
            ln=road.cars[0].lenght
            vn=road.class_speed.speed(vmax,ln,d-dn)
            T+=self.__division(d-dn,v0)

            return T
        
    
    def __repr__(self):
        return "Weight_duration_discretized"

"""# Cars

The different types of cars/drivers that circulate on the road network.

Each type corresponds to a different driving behavior and is associated with a different weight (a way of calculating its shortest path).

Example shortest path in time, shortest path in distance.
"""

class Cars: # expliquer ce qu'est type_name 
  """
  Class Abstract representing cars on the road network.

  named -1
  """
  def __init__(self,type0,num_road,number0,p,l=1):
      self.number=number0 #number of the car in question
      self.position=p #the position of the car on the road
      
      self.type_road=type0 #the type of the road
      self.position_road=num_road #the number of the road on which the car is.
      self.lenght=l #the length of the car

      self.color="" #the color of the basic cars (for the graphs)
      self.type_name=-1  #Type of driver
      self.weight=Abstract_weight() #the type of weight used to calculate the path of this type of car

  def get_path(self,previous_state):
      """
      Return the path that the car must follow (the next road key)

      Parameters
      ----------
          previous_state (RoadNetwork) : the road network on which our car travels.

      Returns :
      -------
          (tuple) : (string,int) the key of the next road we want to take
      """
      path=previous_state.get_next_road(self)

      return path

  def __repr__(self):
      """ redefining print """
      s="car number : "
      s+=str(self.number)
      s+=" of lenght : "
      s+=str(self.lenght)
      s+=" on the road : "
      s+=str(self.position_road)
      s+=" in : "
      s+=str(self.position)
      s+=" "
      s+=repr(self.weight)

      return s


  def get_car_type(self):  #all_car_types ?
        """
        give the dict of the different type of cars (class that inherited of Cars)

        Parameters
        ----------
            None
        
        Returns 
        -------
            (dict Class) : dict of child class of Cars (the key of the dict are the type number of a class)
        """
        utilities = Utilities()
        list_class=utilities.get_subclasses(self)
        dict_class=dict()
        for i in range(len(list_class)):
            class_object=list_class[i](0,0,0,0)  #create an instance of Cars to then get the number associated with the type of driver
            if (class_object.type_name in dict_class.keys()):
                print("Warning, key name confict for Car type")
                message=""
                message+="class : "
                message+=str(dict_class[i].__name__)
                message+=" is already named "
                message+=str(dict_class[i](0,0,0,0).type_name)
                message+="\n"
                message+="class : "
                message+=str(class_object.__class__.__name__)
                message+=" will replace it"
                print(message)
            dict_class[class_object.type_name]=list_class[i]
        return dict_class

class Predifine(Cars):  #type_name = 5 au lieu de 4 ?
    """
    cars that follow one specific path impose to them at there creation

    named 5
    """
    def __init__(self,type0,num_road,number0,p,l=1,dict_choice=None):
        super().__init__(type0,num_road,number0,p,l)
        self.color="orange" #the color of NonInterferer cars (for graphs)
        self.type_name=5
        self.weight=None #this car don't use a pod
        self.path=dict_choice #a dict compose of the next road to take : the key is the intersection label : the value is the next road key
    
    def get_path(self,previous_state):
        """
        Returns the path that the car must follow next (the list of road numbers)

        Parameters
        ----------
            previous_state (RoadNetwork) : the road network on which our car travels.

        Returns : 
        -------
            (tuple) : (string,int) the key of the next road we want to take
        """
        inter_num=previous_state.roads[self.type_road,self.position_road].i2 #number of the next intersections

        if len(previous_state.intersections[inter_num].roads_exit)==1:
            return previous_state.intersections[inter_num].roads_exit[0]

        elif len(previous_state.intersections[inter_num].roads_exit)==0:
            raise OtherErrors("no road exit after this one")
        else:
            return self.path[inter_num]
    
    def __repr__(self):
        """ redefining print """
        s=super().__repr__()
        s+=" Predifine"
        s+="\n"
        s+=repr(self.path)

        return s

class NonInterfererDistance(Cars):
    """
    choose the shortest way for them (in distance) (don't care about the others, don't know the traffic state)

    named 0
    """
    def __init__(self,type0,num_road,number0,p,l=1):
        super().__init__(type0,num_road,number0,p,l)
        self.color="black" #the color of NonInterferer cars (for graphs)
        self.type_name=0
        self.weight=Weight() #the type of weight used to calculate the path of this type of car
    
    def __repr__(self):
        """ redefining print """
        s=super().__repr__()
        s+=" NonInterfererDistance"

        return s

class NonInterferer(Cars):
    """
    choose the shortest way for them (in time) (don't care about the others, don't know the traffic state)

    named 1
    """
    def __init__(self,type0,num_road,number0,p,l=1):
        super().__init__(type0,num_road,number0,p,l)
        self.color="purple" #the color of NonInterferer cars (for graphs)
        self.type_name=1
        self.weight=Weight_duration_empty() #the type of weight used to calculate the path of this type of car
    
    def __repr__(self):
        """ redefining print """
        s=super().__repr__()
        s+=" NonInterferer"

        return s

class Player(Cars):
    """
    the cars knowing the traffic state.

    named 2
    """
    def __init__(self,type0,num_road,number0,p,l=1):
        super().__init__(type0,num_road,number0,p,l)
        self.color="red" #the color of the players (for the graphs)
        self.type_name=2
        self.weight=Weight_duration() #the type of weight used to calculate the path of this type of car
    
    def __repr__(self):
        """ redefining print """
        s=super().__repr__()
        s+=" Player"

        return s

class Optimizer(Cars):
    """
    the cars knowing the traffic state, precizely.

    named 3
    """
    def __init__(self,type0,num_road,number0,p,l=1):
        super().__init__(type0,num_road,number0,p,l)
        self.color="yellow" #the color of the players (for the graphs)
        self.type_name=3
        self.weight=Weight_duration_discretized() #the type of weight used to calculate the path of this type of car
    
    def __repr__(self):
        """ redefining print """
        s=super().__repr__()
        s+=" Optimizer"

        return s

"""# Road / Intersection

The different static objects that make up the network: roads and intersections.
"""

class Intersection:
    """
    Class representing the intersections of the road network.
    """
    def __init__(self,number0,coordinate_X=0,coordinate_Y=0):
        self.number=number0 # intersection number
        self.roads_entry=[]  # list of roads connected to the intersection (list of dople (string,numbers): -road that enter the intersection
        self.roads_exit=[]  #                                                                              -road that  exit  the intersection
        self.cordX=coordinate_X #graphical position of the intersection
        self.cordY=coordinate_Y

        self.next_road=dict() #the next road to take dictionary on the different types of weight
        self.min_path=dict()  #the shortest path (list of roads), dictionary on the different types of static weight (calculated at the beginning of the system)
        self.min_lenght=dict()  #the shortest path (its duration), dictionary on the different types of static weight (calculated at the beginning of the system)
        
    def __repr__(self):
        """ redefining print """
        s="intersection "
        s+=str(self.number)
        s+=" positioned in : "
        s+=str(self.cordX)
        s+=" "
        s+=str(self.cordY)
        s+="\n"
        
        s+="connect to roads : "
        for i in range(len(self.roads_entry)):
            s+=str(self.roads_entry[i])
            s+=" "
        s+="/ "
        for i in range(len(self.roads_exit)):
            s+=str(self.roads_exit[i])
            s+=" "
        
        s+="\n"
        s+="shortest path : "
        s+="\n"
        for key in self.min_path.keys():
            s+="    "
            s+="next road : "
            s+=str(self.next_road[key])
            s+=" path : "
            s+=str(self.min_path[key])
            s+=" in "
            s+=str(self.min_lenght[key])
            s+=" type : "
            s+=str(key)
            s+="\n"
        
        return s
    
    def add_road_entree(self,type_road,number0):
        """
        Linked a road to the intersection (the road end at the intersection).

        Parameters
        ----------
            number0 (int) :  number of the road we linked to the intersection

        Returns
        -------
            None.
        """
        self.roads_entry.append((type_road,number0))
    
    def add_road_sortie(self,type_road,number0):
        """
        Linked a road to the intersection (the road begin at the intersection).

        Parameters
        ----------
            number0 (int) :  number of the road we linked to the intersection

        Returns
        -------
            None.


        """
        self.roads_exit.append((type_road,number0))

class Road:
    """
    Class representing the roads of the road network.
    """
    def __init__(self,type0,number0,insersection1,insersection2,l1,vm=20,V=LinearSpeed()):
        self.number=number0 #road number
        self.type_road=type0

        self.i1=insersection1 #number of the 1st intersection connected to the road (it's input) (-1 if the road is connected to only one intersection (its output))
        self.i2=insersection2 #number of the 2nd intersection connected to the road (it's exit) (-1 if the road is connected to only one intersection (its entry))
        self.l=l1 #length of the road
        self.vmax=vm #maximum possible speed on the road (20 m.s-1 = 72 km)

        self.next_roads=[] #list of roads that can be taken from this road
        self.priority=0 #priority weight (the probability for the road to have priority at the next intersections.)
        self.class_speed=V#the method of calculation of the speed on this road by default the one of the mother class

        self.has_priority=False #init the boolean definnig if a car can pass at the intersection.
        
        self.cars=[] #list of cars (class Car) positioned on this road (the 1st car in the list is the one closest to the exit of the road
            #inversely the last one is the one closest to the road entrance)
    
    def reinitialise_road(self):
        """
        Reset the list of next_roads.
            (removes all roads from next_roads)

        Parameters
        ----------
            None

        Returns 
        -------
            None
        """
        self.next_roads=[]
    
    def add_next_road(self,number0):
        """
        Add a road (its number) to the following list of road.

        Parameters
        ----------
            number (int) : the number of the road added

        Returns 
        -------
            None
        """
        self.next_roads.append(number0)
    
    def __create_car(self,number0,p,l,type0):
        """
        Creates a car of a chosen type and returns it.
        (private method)

        Parameters
        ----------
            type_road (string) : the road type
            number0 (int) : the car number
            p (int) : the position of the car on the road
            l (int) : the length of the car
            type0 (int) : the type of car : 0 = NonInterfererDistance, 1 NonInterferer, 2 Player, 3 Optimizer

        Returns 
        -------
            (Cars) : The cars that we end up creating
        """
        num_road=self.number
        type_road=self.type_road
        try:
            utilities=Utilities()
            cars = Cars(0,0,0,0)
            with HiddenPrints(): #for blocking print in case two car type have the same number (one warning at the creation of the network is enough)
                type_car_dict=cars.get_car_type()#get the dict of all the Car type and there key (type number)
            car_object=type_car_dict[type0](type_road,num_road,number0,p,l)#creation of the car
        except KeyError:
            raise TypeCarError("number "+str(type0)+" doesn't match a type in the car type dict")
        except Exception as e:
            print("unespected error")
            raise e
        return car_object

    
    def reinitialise_car(self):
        """
        Reset the list of cars.
            (removes all cars from cars list)

        Parameters
        ----------
            None

        Returns 
        -------
            None
        """
        self.cars=[]
    
    def add_car(self,position0,num_car,type0,l):
        """
        Creates and places a car of a chosen type in its place on the road.

        Parameters
        ----------
            position0 (int) : the position of the car on the road
            num_car (int) : the car number
            type0 (int) : the type of car : 0 = NonInterfererDistance, 1 NonInterferer, 2 Player, 3 Optimizer
            l (int) optional : the length of the car by default 1

        Returns 
        -------
            None
        """
        is_place=False #bollean saying if the car has not yet been placed on the road (False = it has not been placed)
        for i in range(len(self.cars)):
            if (position0>self.cars[i].position and not is_place):
                    self.cars.insert(i,self.__create_car(num_car,position0,l,type0))
                    is_place=True
        if (len(self.cars) == 0):
            self.cars.append(self.__create_car(num_car,position0,l,type0))
        else:
            if (position0<self.cars[len(self.cars)-1].position):
                self.cars.append(self.__create_car(num_car,position0,l,type0))
    
    def add_car_object(self,car):
        """
        Add a car at the beginning of a road.

        Parameters
        ----------
            car (Cars) : the car we want to add

        Returns 
        -------
            None
        """
        self.cars.append(car)
    
    def get_liste_car_pos(self):
        """
        Returns the list of the positions of the cars on this road.
        (list starting from the cars near the end of the road)

        Parameters
        ----------
            None

        Returns 
        -------
            (list int) : the list of cars positions
        """
        liste=[]
        for i in range(len(self.cars)):
            liste.append(self.cars[i].position)
        return liste
    
    def has_car(self,number_car):
        """
        Test if the car in question belongs to the road.

        Parameters
        ----------
            number_car (int) : the number of the car you are looking for

        Returns 
        -------
            (bool) : boolean saying if the car is on the road or not
        """
        boole=False
        for i in range(len(self.cars)):
            if (self.cars[i].number==number_car):
                boole=True
        return boole

            
    def __repr__(self):
        """ redefining print """

        #general information
        s0="road "
        s0+=str(self.number)
        s0+=" from "
        s0+=str(self.i1)
        s0+=" to "
        s0+=str(self.i2)
        s0+=" length : "
        s0+=str(self.l)
        s0+=" priority weight : "
        s0+=str(self.priority)
        s0+=" max speed : "
        s0+=str(self.vmax)
        s0+=" "
        s0+=repr(self.class_speed)
        s0+=" ; car has priority : "
        s0+=str(self.has_priority)

        #next road information
        sn="next road list :"
        for i in range(len(self.next_roads)):
            sn+=str(self.next_roads[i])
            sn+=' , '
        if (len(self.next_roads)>0):
            sn=sn.rstrip(sn[-1])
            sn=sn.rstrip(sn[-1])
            sn=sn.rstrip(sn[-1])

        #cars information
        sv="list cars :"
        for i in range(len(self.cars)):
            sv+="\n"
            sv+="    "
            sv+=repr(self.cars[i])

        s=""
        s+=s0
        s+="\n"
        s+=sn
        s+="\n"
        s+=sv
        return s
    
    def get_pos_cars(self):
        """
        local part of show_pos_car do all the necessary operation on one road and send back the string
        """
        s="road : "
        s+=str(self.type_road)
        s+=" "
        s+=str(self.number)
        s+="\n"
        s+="    "
        for j in range(len(self.cars)): #loop on the cars
            s+=str(self.cars[j].number)
            s+=": "
            s+=str(self.cars[j].position)
            s+=" ; "
        if (len(self.cars)>0):
            s=s.rstrip(s[-1])
            s=s.rstrip(s[-1])
            s=s.rstrip(s[-1])
        return s

"""# ROAD NETWORK

The class representing the road network.

The main class that manage the dynamic of the system.
"""

class RoadNetwork:
    """
    Class representing a road network.
    """
    def __init__(self):
        utilities=Utilities() #create an utility object in order to have access to the Utility method
        cars = Cars(0,0,0,0)

        self.roads=dict() #dictionary storing the Road (for each type of roads)
                          #dict key : (type, num_road)
                          #type :(string: "normal", "entry", "exit")
        self.intersections=dict() #dictionary storing the Intersection
                          #dict key : (int) number of the road
        
        self.__static_weight_list=[ x() for x in utilities.get_subclasses(Abstract_static_weight) ] #list of the different types of static weight used for shortest path calculations
        self.__dynamic_weight_list=[ x() for x in utilities.get_subclasses(Abstract_dynamic_weight) ] #list of the different types of dynamic weight used for shortest path calculations
        self.path=dict() #dictionary storing the shortest paths for each road and each type of weight.
                         #dict key : (Weight, num_road), Weight: object of a class inheriting from Weight (each class = a different key)
        
        self.__car_type_dict=cars.get_car_type()#dict of all the car type (the key is the num of the car type, the value is the class.)

        #set of parameters to ensure system stability
        #v_max*dt_max<l is required
        self.__v_max=1 #
        self.__l_min=1 #arbitrary initialization redefined in connect
        self.__dt_max=1#

        self.__length_infinite_road=10**6 #the lenght of the infinite road (just a number big enought to never reach the end.) 

        self.__dynamic_weight_list_use=None #list of the dynamic pod use during this simuation

        self.__is_well_define=True #as long as nothing is added to the system, all elements are properly connected.
                                   #becomes false as soon as we modify the system. returns to True when we call the connect() function
        self.__is_conditioned=True #same but for cars (dynamic part)
    
    def __repr__(self):
        """
        Returns all the information of the object.
        """
        srn="normal road"
        srn+="\n"
        s0en="entry road"
        s0en+="\n"
        s0ex="exit road"
        s0ex+="\n"
        for i in self.roads: #loop on all the road of the network (get all information about all the road)
            if (self.roads[i].type_road=="normal"):
                srn+=repr(self.roads[i])
                srn+="\n"
            elif (self.roads[i].type_road=="entry"):
                s0en+=repr(self.roads[i])
                s0en+="\n"
            elif (self.roads[i].type_road=="exit"):
                s0ex+=repr(self.roads[i])
                s0ex+="\n"
            else:
                raise TypeRoadError("unidentified road type found")
        
        si="intersections"
        si+="\n"
        for i in (self.intersections):
            si+=repr(self.intersections[i])
            si+="\n"
        
        #part for processing the car type
        soc=""
        for i in self.__car_type_dict:
            soc+=str(i)
            soc+=" "
            soc+=str(type(self.__car_type_dict[i](-1,-1,-1,-1)).__name__)
            soc+=" ; "
        soc=soc.rstrip(soc[-1])
        soc=soc.rstrip(soc[-1])

        so=""
        so+="length infinite road : "
        so+=str(self.__length_infinite_road)
        so+="\n"
        so+="list of static Pods : "
        so+=str(self.__static_weight_list)
        so+="\n"
        so+="list of dynamic Pods : "
        so+=str(self.__dynamic_weight_list)
        so+="\n"
        so+="list of Cars type : "
        so+=soc
        so+="\n"
        so+="v_max  : "
        so+=str(self.__v_max)
        so+="\n"
        so+="l_min  : "
        so+=str(self.__l_min)
        so+="\n"
        so+="dt_max : "
        so+=str(self.__dt_max)
        so+="\n"
        so+="est bien définie     : "
        so+=str(self.__is_well_define)
        so+="\n"
        so+="est bien conditionné : "
        so+=str(self.__is_conditioned)

        s=""
        s+=s0en #add information about entry road
        s+="_"*25
        s+="\n"
        s+=s0ex #add information about exit road
        s+="_"*25
        s+="\n"
        s+=srn #add information about normal road
        s+="_"*35
        s+="\n"
        s+=si #add information about intersection
        s+="_"*50
        s+="\n"
        s+=so #add all the other information

        return s
    
    def is_well_defined(self):
        """
        show if the network is well defined

        Parameters
        ----------
            None

        Returns 
        -------
            boolean
        """
        return copy.deepcopy(self.__is_well_define) #copy in order to not get the actual object (stop people from modifing the object)
    
    def set_not_defined(self):
        """
        set the network to not defined. Use by some function outside the class: you should not use it yourself

        Parameters
        ----------
            None

        Returns 
        -------
            None
        """
        self.__is_well_define=False

    def get_dt_max(self):
        """
        give the network max_dt

        Parameters
        ----------
            None

        Returns 
        -------
            int : the max dt
        """
        return copy.deepcopy(self.__dt_max)
    
    def count_car_in_network(self):
        """
        count the number of cars in the network

        Parameters
        ----------
            None

        Returns 
        -------
            (int) : the number of cars in the network
        """
        n=0
        for i in self.roads:
            n+=len(self.roads[i].cars)
        return n
    
    def count_car_by_type(self):
        """
        count the number of cars in the network for each car type

        Parameters
        ----------
            None

        Returns 
        -------
            (dict int) : the number of cars in the network (the key is the car type)
        """
        dict_car_number=dict()
        for i in self.__car_type_dict: #initializing the number for each car type
            dict_car_number[i]=0
        
        for i in self.roads: #loop on the roads
            for j in range(len(self.roads[i].cars)): #loop on the cars of the road
                dict_car_number[self.roads[i].cars[j].type_name]+=1
        
        return dict_car_number
    
    def get_all_car_info(self):
        """
        get all information about the system state at one point (concerning cars)

        Parameters
        ----------
            None

        Returns 
        -------
            (dict tuple) : dict of tuple (car type num, car position on the road)
                           the key are (road_type, road_num, position in the road)
                           position on the road : the distance the car has travel on the road.
                           position in the road : number of car bettween the car and the exit of the road.
        """
        dict_cars=dict()
        
        for i in self.roads: #loop on the roads
            for j in range(len(self.roads[i].cars)): #loop on the cars of the road
                dict_cars[i,j]=(self.roads[i].cars[j].type_name,self.roads[i].cars[j].position)
        
        return dict_cars
    
    #############################################################################################################################
    ##
    ##                        Functions to create and modify the network
    ##
    #############################################################################################################################
        
    def add_intersection(self, label, cordX=0, cordY=0):
        """
        Add an intersection to the network
        (allows to define these graphical coordinates (these coordinates have no impact on the dynamics of the system))

        Parameters
        ----------
            label (int) : the number of the intersection
            cordX (float) optional : the graphical coordinate in x of the intersection by default 0
            cordy (float) optional : the graphical coordinate in y of the intersection by default 0

        Returns 
        -------
            None
        """
        self.__is_well_define=False
        self.intersections[label]=Intersection(label,cordX,cordY)
    
    def set_entree(self,num_inter):
        """
        Creates the network entry road.

        Parameters
        ----------
            num_inter (int) : the number of the intersection to which the road is attached

        Returns 
        -------
            None
        """
        self.__is_well_define=False

        if ("entry",0) in self.roads:
            road_key=("entry",0)
            inter_num2=self.roads[road_key].i2
            self.intersections[inter_num2].roads_entry.pop(self.intersections[inter_num2].roads_entry.index(road_key)) #pop on list

        self.roads["entry",0]=Road("entry",0,-1,num_inter,self.__length_infinite_road)
        self.intersections[num_inter].add_road_entree("entry",0)
    
    def set_sortie(self,num_inter):
        """
        Creates the network exit road.

        Parameters
        ----------
            num_inter (int) : the number of the intersection to which the road is attached

        Returns 
        -------
            None
        """
        self.__is_well_define=False

        if ("exit",0) in self.roads:   
            road_key=("exit",0)
            inter_num1=self.roads[road_key].i1
            self.intersections[inter_num1].roads_exit.pop(self.intersections[inter_num1].roads_exit.index(road_key)) #pop on list

        self.roads["exit",0]=Road("exit",0,num_inter,-1,self.__length_infinite_road)
        self.intersections[num_inter].add_road_sortie("exit",0)

    def add_road(self,label,intersections1,intersections2,L=10,vm=20,vclass=LinearSpeed()):
        """
        Adds a road to the road network
        does not allow to create the entry and exit roads of the network.

        Parameters
        ----------
            label (int) : the label (number) of the road
            intersections1 (int) : the intersection where the road start
            intersections2 (int) : the intersection where the road finished
            L (int) optional : length of the road; by default 10
            vm (int) optional : maximum speed on the road; by default 20
            vclass (Speed) optional : class of speed calculation on a road; by default LinearSpeed()
        
        Returns
        -------
            None
        """
        self.__is_well_define=False

        self.roads["normal",label]=Road("normal",label,intersections1,intersections2,L,vm,vclass)
        self.intersections[intersections1].add_road_sortie("normal",label)
        self.intersections[intersections2].add_road_entree("normal",label)
    
    def set_intersection_priority(self,num_inter,liste_priority):
        """
        Change the priority of all the road arriving to an intersection.

        Parameters
        ----------
            num_inter (int) : the number of the intersection considered
            liste_priority (list tuple) : list compose of the key and priority of the road,
                                          list of : (( type_road , num_road ) , priority):
                                          type_road (string) : the type of the road
                                          num_road (int) : the number of the road
                                          priority (float) : the priority pods we set to the road (must be beetween 0 and 1)

        Returns 
        -------
            None
        """

        n=len(self.intersections[num_inter].roads_entry)
        k=len(liste_priority)
        total_weight=0 #sum of the weight we set to the road
        for i in range(k):
            weight=liste_priority[i][1]
            if (weight<0):
                raise OtherErrors("invalid priority weight : weight can't be negative")
            total_weight+=weight

        
        if (k>n):
            message="to many road specify for the corresponding intersection"
            message+="\n"
            message+="intersections : "+str(num_inter)+" has "+str(n)+" road arriving to this intersection, "
            message+=str(k)+" road priority where given."
            raise  ConnectionError(message)
        elif (total_weight>1):
            message="for intersection "+str(num_inter)+", invalid priority weight : sum of weight can't be greater than 1"
            raise PriorityError(message)
        elif (k==n): #case where we specify the priority of each road
            if (total_weight!=1):
                message="for intersection "+str(num_inter)+", sum of weight not equal to 1"
                raise PriorityError(message)
            else:
                remaining_weight=0
        else:
            remaining_weight=(1-total_weight)/(n-k)#the weight of the road we didn't specify (there all equal and complete the total sum of weight to reach 1)

        dic_of_weight=dict()
        for i in range(n):
            dic_of_weight[self.intersections[num_inter].roads_entry[i]]=remaining_weight
        for i in range(k):
            dic_of_weight[liste_priority[i][0]]=liste_priority[i][1]

        for j in dic_of_weight:
            self.roads[j].priority=dic_of_weight[j]
    
    #############################################################################################################################
    ##
    ##                        Utility functions to perform various static operations on the network
    ##
    #############################################################################################################################
    
    
    def get_pod_path(self,path,weight=Weight()):
        """
        Takes a list of path as argument (a path is a list of road) and computes their total length.

        Parameters
        ----------
            path (list list int) : the numbers of the roads through which we pass
            weight (Weight) optional : an object defining the type of weight used for the calculation by default Weight()

        Returns 
        -------
            (list int) : the list of length of the path in question
        """
        lenght_list=[]
        for i in range(len(path)):#loop on the differente path
            lenght=0
            for j in range(len(path[i])): #loop on the road of the path
                lenght+=weight.calculate_road_weight(self.roads[path[i][j]])
            lenght_list.append(lenght)
        return lenght_list
    
    def __reset_passage_priority(self,inter_num):
        """
        Private function that reset the passage priority of every road in an intersection.

        Parameters
        ----------
            inter_num (int) : the number of the intersection where we reset the passage priority

        Returns 
        -------
            None
        """
        list_inter_road=self.intersections[inter_num].roads_entry
        for i in list_inter_road:
            self.roads[i].has_priority=False

    def __compute_order_of_passing(self,inter_num):
        """
        Private function that compute and set the order of passing of cars at one intersections.
        the order of passage is compute randomly using the priority weight of each road

        Parameters
        ----------
            inter_num (int) : the number of the intersections where we set the order of passage

        Returns 
        -------
            (list int) : the list of the passage order of cars (0 priority road, 1 second in order ...)
        """
        weight_sum=0 #init to 0, should go up until being equal to 1
        list_weight_sum=[weight_sum] #list of float going up beetwen 0 and 1. use to choose the road using the weight priority of a road
        list_inter_road=self.intersections[inter_num].roads_entry #list of all the road arriving to the intersections.
        n=len(list_inter_road)

        for i in range(n): #loop on the road arriving to the intersection.
            weight_sum+=self.roads[list_inter_road[i]].priority
            list_weight_sum.append(weight_sum)
        
        i=0
        order=[-1]*n
        p=rd.uniform(0,1)#draw of an float use to define the priority road
        while ( not ((list_weight_sum[i]<p) and (p<list_weight_sum[i+1]))): #loop to find the priority road
            i+=1
        
        order=[-1]*n
        for j in range(i,n): #loop to define the priority order of the other roads (road after the priority road in the list)
            order[j]=(j-i)
        for j in range(0,i): #loop to define the priority order of the other roads (road before the priority road in the list)
            order[j]=(i+j)
        return order
    
    def __set_priority(self,inter_num):
        """
        Change on each road the priority boolean deciding if cars on this road can pass.
        (all road before the first road with a car at the end have priority True)
        (all the later are False)

        Parameters
        ----------
            inter_num (int) : the number of the intersection where we do our operation.

        Returns 
        -------
            None
        """
        order_of_passing=self.__compute_order_of_passing(inter_num)#chose the order of priority (wich road has priority, wich road has priority after the first road...)
        list_inter_road=self.intersections[inter_num].roads_entry#list of road arriving to the intersection.

        begin=order_of_passing.index(0)
        n=len(list_inter_road)
        i=begin
        can_pass=False
        self.__reset_passage_priority(inter_num)

        while ((i<n) and not (can_pass)): #loop on the list from road with higest priority to end of the road (include).
            if (len(self.roads[list_inter_road[i]].cars)>0): #test if there is a cars on the road
                  self.roads[list_inter_road[i]].has_priority=True
                  if (self.roads[list_inter_road[i]].l-self.roads[list_inter_road[i]].cars[0].position)<self.roads[list_inter_road[i]].cars[0].lenght: #test if the cars is at the end of the road (closer than it's lenght)
                      can_pass=True
            i+=1
        i=0
        while ((i<begin) and not (can_pass)): #loop on the list from the begin to road with higest priority (not include).
            if (len(self.roads[list_inter_road[i]].cars)>0): #test if there is a cars on the road
                self.roads[list_inter_road[i]].has_priority=True
                if (self.roads[list_inter_road[i]].l-self.roads[list_inter_road[i]].cars[0].position)<self.roads[list_inter_road[i]].cars[0].lenght: #test if the cars is at the end of the road (closer than it's lenght)
                    can_pass=True
            i+=1

    def connect(self):
        """
        Makes all the connections and calculations that can only be done after adding all the roads and intersections.
        Must be executed after each modification of the system architecture.

        recalculates all the next/shortest path for each road and updates this information, connects the roads (in their attribute), ...

        Parameters
        ----------
            None

        Returns 
        -------
            None
        """
        self.__v_max=0
        #does the connection to the road
        for i in self.roads:
            if (self.roads[i].type_road=="normal" or self.roads[i].type_road=="entry"):
                
                inter=self.intersections[self.roads[i].i2]
                self.roads[i].reinitialise_road() #reinitialize the list of roads connected to the end of this road
                for j in range(len(inter.roads_exit)):
                    sortie=inter.roads_exit[j] #the number of the road starting from the intersection (in the list of roads starting from the intersection)
                    self.roads[i].add_next_road(sortie)
            
            elif (self.roads[i].type_road=="exit"):
                pass
            else:
                message="name : "+str(self.roads[i].type_road)+" is not a valid type name"
                message+="\n"
                message+="normal ; exit ; entry ; are the current valid name"
                raise ConnectionError(message)

            if (self.roads[i].vmax>self.__v_max):#test if the road has a higher speed to find the maximum speed of all roads.
                self.__v_max=self.roads[i].vmax
        

        #calculate the weight of each segment (the time it would take the car to cover it) and create the corresponding graph

        for weight in (self.__static_weight_list): #loop on static weight types
            for i in self.intersections:#loop on the intersection.
                try:
                    self.__set_path(weight,i)
                except nx.NetworkXNoPath:
                    message="this configuration doesn't allow to be able to reach the exit from everywhere, please change this before carrying on"
                    raise ConnectionError(message)
                except Exception as e:
                    print("unespected error")
                    raise e
        
        for weight in (self.__dynamic_weight_list): #Loop on the types of dynamic weights
            for i in self.intersections:#loop on the intersection.
                self.__set_path(weight,i)
        
        for i in self.intersections: #init the weight priority of the road at the intersection.
            self.set_intersection_priority(i,[])

        self.__is_conditioned=False #some information like road max speed can have change.
        self.__is_well_define=True #we have made all the connections so the network is well defined
    
    def compile_car_system(self):
        """
        Makes all the calculations that can be done only after adding all the cars.
        Executed after each addition/removal of car from the system.

        Makes the calculations of l_min and dt_max of the system so that it is well conditioned.

        Parameters
        ----------
            None

        Returns 
        -------
            None
        """
        if (self.__is_well_define):
            self.__l_min=10**6
            liste_poid=[]
            for i in (self.roads):
                for j in range(len(self.roads[i].cars)):
                    car=self.roads[i].cars[j]
                    if (car.lenght<self.__l_min):#test if the road has a higher speed to find the maximum speed of all roads.
                        self.__l_min=self.roads[i].cars[j].lenght
                    if (not (car.weight in liste_poid)):#test if a car already use one of the pod
                        liste_poid.append(car.weight)#added the new pod to the list.
            liste_poid_dynamic_tmp=[]
            for i in range(len(self.__dynamic_weight_list)):
                if (self.__dynamic_weight_list[i] in liste_poid):
                    liste_poid_dynamic_tmp.append(self.__dynamic_weight_list[i])
              
            self.__dt_max=self.__l_min/self.__v_max
            self.__dynamic_weight_list_use=liste_poid_dynamic_tmp #the dynamic pod we use
            self.__is_conditioned=True
        else:
            raise ConnectionError("before compiling the car info you must connect the roads with connect()")

    def road_reset(self):
        """
        Supresses all the cars in the network to get an empty network.

        Parameters
        ----------
            None

        Returns 
        -------
            None
        """
        for i in (self.roads):
            self.roads[i].cars=[]
    
    def position_car(self,type_road,road_num,list_type,l):
        """
        Positions n cars on the road road_num, positions them in the middle of the road.
        the type is specify by list type (private function) (n is the length of the list)

        Parameters
        ----------
            type_road (int) : the type of the road
            road_num (int) : the number of the road on which the cars are placed
            list_type (list int)  : list of the type we place: 0 = NonInterfererDistance, 1 NonInterferer, 2 Player, 3 Optimizer
            l (float) : length of the cars

        Returns 
        -------
            None
        """
        self.__is_conditioned=False
        
        n=len(list_type)

        if (type(road_num)==int or type(road_num)==float): #test if road_num is a number
            num_tmp=road_num
        else:
            num_tmp=-1
        
        self.roads[type_road,road_num].reinitialise_car()

        if (type_road=="normal"): #case normal road
            k=self.roads[type_road,road_num].l
            for j in range(n):
                p=k*(j+1)/(n+1)
                self.roads[(type_road,road_num)].add_car(p,num_tmp*n+j,list_type[j],l)
        
        elif (type_road=="entry"): #case entry road
            ecart=self.roads[type_road,0].l-(n+1)*2*l #the place on the road where we start putting cars
            k=(n+1)*2*l
            for j in range(n):
                p=ecart+(k*(j+1)/(n+1))
                self.roads[type_road,0].add_car(p,num_tmp*n+j,list_type[j],l)
        
        elif (type_road=="exit"): #case exit road
            k=(n+1)*2*l
            for j in range(n):
                p=k*(j+1)/(n+1)
                self.roads[type_road,0].add_car(p,num_tmp*n+j,list_type[j],l)
        
        else:
            raise IndexError("the road type "+str(type_road)+" doesn't exist")
    
    
    def __shortest_path(self,p,inter_num):
        """
        Private function
        compute all the shortest path beetween one intersections, and the intersection connect to the exit road

        Parameters
        ----------
            p (Weight) : class defining the type of weight used for the shortest path algorithm.
                calcul_poid_road is the important function to define when creating a weight class.
            inter_num (int) : the number of the intersection of wich we compute the shortest path toward the exit.
        Returns 
        -------
            (list list tuple) a list of path: each path is a list of tuple
                                list : the key of the road composing  the shortest path, tuple : the key of the road : (type_road, num_road)
        """

        G = nx.DiGraph()
        #calculate the weight of each segment (the time it would take the car to cover it) and create the corresponding graph
        for i in (self.roads):
            road=self.roads[i]
            weight=p.calculate_road_weight(road)
            G.add_edge(road.i1,road.i2,weight=weight,num_road=i)
                
        num_exit_inter=self.roads["exit",0].i1

        path=[]
        
        tmp=nx.all_shortest_paths(G,source=inter_num,target=num_exit_inter,weight='weight')#compute the shortest path

        for pathway_inter in tmp: #loop on the different path beetween 2 points
            if (len(pathway_inter)>1):#test if a path exist.
                pathway_road=[]
                for j in range(len(pathway_inter)-1):#loop on one path 
                    pathway_road.append(G[pathway_inter[j]][pathway_inter[j+1]]['num_road'])
                path.append(pathway_road)
        
        return path
    
    def __set_next_road_path(self,pod,inter_num,path):
        """
        Private function
        compute and set the list of all the fisrt road in the shortest path from one intersection to the exit.

        Parameters
        ----------
            pod (Weight) : class defining the pod that was use for the shortest path
            inter_num (int) : the number of the intersection
            path (list list tuple) :  a list of path: each path is a list of tuple
                                list : the key of the road composing  the shortest path, tuple : the key of the road : (type_road, num_road)
        
        Returns 
        -------
            (None)
        """
        num_exit=self.roads["exit",0].i1

        self.intersections[inter_num].min_path[pod]=path
        self.intersections[inter_num].min_lenght[pod]=self.get_pod_path(path,pod)

        if (inter_num==num_exit):
            self.intersections[inter_num].next_road[pod]=[("exit",0)]#set the next road to go with this pod when the next intersection is the exit intersection
        else:
            if (len(path)>0):
                tmp=[]
                for j in range(len(path)):
                    tmp.append(path[j][0])
                self.intersections[inter_num].next_road[pod]=tmp#set the list of next road to go (one road per path) in the general case
            else :
                self.intersections[inter_num].next_road[pod]=None#set None when no path is availlaible to reach the exit intersection from this intersection.

    def __set_path(self,pod,inter_num):
        """
        set all the information link to the path with a specific weight.

        Parameters
        ----------
            p (Weight) : class defining the type of weight used for the shortest path algorithm.
                calcul_poid_road is the important function to define when creating a weight class.
            inter_num (int) :
        Returns 
        -------
            (None)
        """
        path=self.__shortest_path(pod,inter_num)
        self.__set_next_road_path(pod,inter_num,path)

    def get_next_road(self,car):
        """
        return the next road the car must take

        Parameters
        ----------
            car (Cars) : the car about wich we want to find the next road
        Returns 
        -------
            (tuple) : (string,int) the key of the next road we want to take
        """
        pod=car.weight
        road_key=(car.type_road,car.position_road)
        next_roads_list=self.intersections[self.roads[road_key].i2].next_road[pod]#we get the list of the next possible road for this car, in order to take the shortest path

        n=len(next_roads_list)
        if (n==1):
            return next_roads_list[0]
        elif (n>1):
            return next_roads_list[rd.randint(n)]
        else:
            raise OtherErrors("no available path toward exit for this car.")


    #############################################################################################################################
    ##
    ##                        Function to interact with cars on the network
    ##
    #############################################################################################################################
    def __inter_has_car_close(self,inter_num):
        """
        return True if a car coming toward the intersection is close to it (closer than it's length l)

        Parameters
        ----------
            inter_num (int) : the number of the intersection considered
        Returns 
        -------
            (bollean) : True if a car coming toward the intersection is close to it
        """
        list_road=self.intersections[inter_num].roads_entry#list of roads arriving to the intersections

        for i in range(len(list_road)): #loop on the roads arriving to the intersections
            if (len(self.roads[list_road[i]].cars)>0): #test if there are cars on the road
                l=self.roads[list_road[i]].cars[0].lenght
                p=self.roads[list_road[i]].cars[0].position
                L=self.roads[list_road[i]].l
                if (L-p<l): #test if the car is close to the intersection
                    return True
        return False

    def get_distance_previous_car(self,type0,num_road,num_car):
        """
        Finds the distance at which the car is in front of the car we are considering.
        If the car has no car in front of it on this road returns the distance from the car to the end of the road.

        Parameters
        ----------
            type0 (string) : the type of road
            num_road (int) : the number of the road on which the car is considered
            num_car (int) : the position in the car list of the car being considered
        Returns 
        -------
            (int) : the distance to the previous car
        """
        road=self.roads[type0,num_road]
        position1=road.cars[num_car].position #our car
        if (num_car == 0):
            position2=road.l
        else:
            position2=road.cars[num_car-1].position #the car in front of us.
        return (position2-position1)

    def part_move(self,v,dt,type_road,num_road,num_car,d,l):
        """
        Moves the car with dt, v, its route and its index in it.

        Parameters
        ----------
            v (int) : the speed at which the car should go
            dt (int) : the time step
            num_road (int) : the number of the road in question
            num_car (int) : the index of the car in the list of cars on the road
            d (int) : the distance the car has in front of it (distance to the next intersection if it does not have priority, or to the car in front of it in other cases).
            
        Returns 
        -------
            None
        """
        #test if you can hit the car in front of you
        if (d-v*dt<l):
            increment=max(0,d-l) #move while not going closer to the car before, than the lenght of our car. (can also be distance to the intersection)
        else:
            increment=v*dt
        
        self.roads[(type_road,num_road)].cars[num_car].position+=increment


    def step_road_move(self,dt,previous_state,road):
        """
        Function called by step and moving all cars on a road.

        Parameters
        ----------
            dt (int) : the step
            previous_state (RoadNetwork) : copy of the network (the previous state of the network)
            road (Road) : the road on which we move the cars
            
        Returns 
        -------
            None
        """
        #case car at the end of the road.
        if (len(road.cars)>0): #test if there is a car on the road.
            #information on the road
            vmax=road.vmax
            L=road.l
            num_road=road.number
            type_road=road.type_road

           #vehicle information
            d=previous_state.get_distance_previous_car(type_road,num_road,0)
            l=road.cars[0].lenght
            position=road.cars[0].position
            
            if (type_road!="exit"):
                #num_next_road=self.get_next_road(road.cars[0])
                num_next_road=road.cars[0].get_path(self)
            else:
                num_next_road=None #we shouldn't need it for cars on the exit road

            if (position>=L-l): #test if our car is at the end of the road (criterion distance lower than the length of the vehicle.)
                boole=previous_state.roads[type_road,num_road].has_priority
                if (boole): #the car has priority
                    k=len(previous_state.roads[num_next_road].cars)
                    if (k>0):
                        d+=previous_state.roads[num_next_road].cars[k-1].position
                    else:
                        d+=previous_state.roads[num_next_road].l
                
                v=road.class_speed.speed(vmax,l,d)
            else:
                v=vmax #we go at the maximum possible speed (see 1381 paper)
            
            self.part_move(v,dt,type_road,num_road,0,d,l) # moving the car
        
        #the other cars
        for j in range(1,len(road.cars)):
            d=previous_state.get_distance_previous_car(type_road,num_road,j)
            l=road.cars[j].lenght
            vmax=road.vmax
            v=road.class_speed.speed(vmax,l,d)

            self.part_move(v,dt,type_road,num_road,j,d,l) # moving the car
    
    def step_road_transfere(self,road):
        """
        Function that transfers the cars that have passed the end of the road to the next corresponding road.

        Parameters
        ----------
            road (Road) : the road where we will perhaps transfer the car
            
        Returns 
        -------
            None
        """
        #check if the car at the end of the road is out of it
        num_road=road.number
        type_road=road.type_road
        index=(type_road,num_road)

        if (type_road=="normal" or type_road=="entry"): #case normal road
            if (len(self.roads[index].cars)>0):
                if (self.roads[index].cars[0].position>self.roads[index].l): #case car at the end of the road
                    car=self.roads[index].cars[0]
                    car.position=car.position-self.roads[index].l
                    self.roads[index].cars.pop(0) #removal of the car from the old road

                    num_next_road=road.cars[0].get_path(self)

                    (car.type_road,car.position_road)=num_next_road
                    self.roads[num_next_road].add_car_object(car) #adding the car to the new road

        elif (type_road=="exit"): #case road exit
            pass #we don't do anything here
        else:
            raise IndexError("error road "+str(num_road)+" has an invalid road type")
    
    def step(self,dt):
        """
        Makes all cars move one step (takes care of the system dynamics).

        Parameters
        ----------
            dt (int) : the time step, by how much do we advance
            
        Returns 
        -------
            None
        """
        if (self.__is_well_define):
            if (self.__is_conditioned):
                if (0<dt<self.__dt_max):
                    #calculates the weights of the road and the new shortest paths
                    for i in self.intersections: #loop on the intersection.
                        if (self.__inter_has_car_close(i)): #test if there a car close to the intersection coming toward it.
                            for weight in (self.__dynamic_weight_list_use): #loop on dynamic weight types
                                self.__set_path(weight,i)
                    
                    for i in self.intersections: #chose randomly the priority of the road for this intersection.
                        self.__set_priority(i)
                    
                    previous_state=copy.deepcopy(self)

                    #makes the cars move
                    for i in (previous_state.roads): #loop on normal roads
                        self.step_road_move(dt,previous_state,previous_state.roads[i])

                    #changes the road of the cars that have passed the end of the road
                    for i in (previous_state.roads): #loop on normal roads
                        self.step_road_transfere(previous_state.roads[i])
                else:
                    message="Warning : step dt "+str(dt)+" must be positive and smaller than "+str(self.__dt_max)
                    message+="\n"
                    message+=" new step automatically chosen : "+str(self.__dt_max/2)
                    print(message)
                    self.step(self.__dt_max/2) #try again with impossibility to end up in the same case.
            else:
                self.compile_car_system()
                if (not self.__is_conditioned):
                    raise CompileError("unable to suceffuly compile network")
                else:
                    self.step(dt) #try again with impossibility to end up in the same case.
        else:
            self.connect()
            if (not self.__is_well_define):
                raise ConnectionError("unable to suceffuly compile network")
            else:
                self.step(dt) #try again with impossibility to end up in the same case.
    
    def replace(self,lenght=100):
        """
        Transfers the cars from the exit road to the entry road.

        Parameters
        ----------
            lenght (int) optional : the simulated length of the exit / entry road, 100 by default
            
        Returns 
        -------
            None
        """
        for i in range(len(self.roads["exit",0].cars)):
            car=self.roads["exit",0].cars[0]
            car.position=self.roads["entry",0].l-lenght+car.position
            car.position_road=0
            car.type_road="entry"
            self.roads["exit",0].cars.pop(0)
            self.roads["entry",0].add_car_object(car)

    def run(self,dt,n):
        """
        Execute n step.

        Parameters
        ----------
            dt (int) : the time step, by how much do we advance
            n (int) : how many time steps we execute
            
        Returns 
        -------
            None
        """        
        for i in range(n):
            self.step(dt)
    
    def run_cycle(self,dt,n):
        """
        Execute n step.
        puts the cars from the outgoing road on the incoming road

        Parameters
        ----------
            dt (int) : the time step, by how much do we advance
            n (int) : how many time steps we execute
            
        Returns 
        -------
            None
        """             
        l=50
        for i in range(n):
            self.step(dt)
            self.replace(l*dt)
    
    #############################################################################################################################
    ##
    ##                        Dynamic calculation functions
    ##
    #############################################################################################################################
    
    def list_car(self):
        """
        return the list of all the car number in the RoadNetwork (can have duplicate if two car use the same number)

        Parameters
        ----------
            None
            
        Returns 
        -------
            (list int) : list of car numbers
        """
        liste_num_car=[]
        for i in (self.roads): #loop on roads
            for j in range(len(self.roads[i].cars)):
                liste_num_car.append(self.roads[i].cars[j].number)
        return liste_num_car
    
    #############################################################################################################################
    ##
    ##                        Graphic functions
    ##
    #############################################################################################################################
    
    def show_pos_car(self):
        """
        Print the roads (their number) and the position of the cars on them (and their number).

        Parameters
        ----------
            None
            
        Returns 
        -------
            None
        """     
        s=""
        s_n=""
        s_en=""
        s_ex=""
        
        #loop on the roads
        for i in (self.roads):
            if (self.roads[i].type_road=="normal"):#case normal road
                s_n+=self.roads[i].get_pos_cars()
                s_n+="\n"
            elif (self.roads[i].type_road=="entry"):#case entry road
                s_en+=self.roads[i].get_pos_cars()
                s_en+="\n"
            elif (self.roads[i].type_road=="exit"):#case exit road
                s_ex+=self.roads[i].get_pos_cars()
                s_ex+="\n"
            else:
                raise TypeRoadError("unidentified road type found")
        s+=s_en
        s+=s_ex
        s+=s_n
        print(s)
    
    def draw_static(self):
        """
        Draws the graph of roads and intersections representing the road network
        (static drawing does not show cars).

        Parameters
        ----------
            None
            
        Returns 
        -------
            None
        """     

        G = nx.DiGraph()
        edge_labels = dict()
        color_map = []

        #Create the nodes of the graph (here the intersections of the road network)
        for j in (self.intersections):
            G.add_node(j,pos=(self.intersections[j].cordX,self.intersections[j].cordY))
        
        #Create the edges of the graph (here the roads of the road network)
        for i in (self.roads):
            if (self.roads[i].type_road=="normal"):
                inter1=self.roads[i].i1
                inter2=self.roads[i].i2
            
                if not(inter1 == -1 or inter2 == -1):
                    G.add_edge(inter1,inter2)
                    edge_labels[(inter1,inter2)] = self.roads[i].number#add the road number
        
        pos=nx.get_node_attributes(G,'pos')#defines the positions of the nodes (here intersections)


        node_labels = {node:node for node in G.nodes()}#added node label (adds intersection numbers)

        #add color label (adds a particular color to the entry and exit nodes)
        try :
            num_entree=self.roads["entry",0].i2
        except:
            num_entree=None
        try :
            num_sortie=self.roads["exit",0].i1
        except:
            num_sortie=None
        
        for node in G:
            if node == num_entree:
                color_map.append('red')
            elif node == num_sortie:
                color_map.append('green')
            else:
                color_map.append('cyan')
        
        nx.draw_networkx_labels(G, pos, labels=node_labels)
        nx.draw_networkx_edge_labels(G, pos=pos, edge_labels=edge_labels)
        nx.draw(G,pos,node_color=color_map)
        plt.show()

"""# CREATE FUNCTION"""

class CreateNetwork:
    """
    Class use to create a road network.
    """

    ############################################
    ####            Inner class             #### 
    ############################################

    class Inner:
        """
        inner class only use for switching label
        """
        def __init__(self,label):
            self.label=label
        
        def __repr__(self):
            """ redefining print (use for debugging) """
            s="Inner :"
            s+=repr(self.label)
            return s

        def __hash__(self):
            """redefine hash to use weight types as key"""
            return hash(str(self))
    
        def __eq__(self, other):
            """redefine eq to use Inner types as key"""
            if (self.__class__ == other.__class__):
                return (self.label==other.label)
            return False
        
        def __getitem__(self,i):
            return self.label[i]
    
    ############################################
    ####           core methode             #### 
    ############################################

    def __init__(self):
        self.__road_network=RoadNetwork() #the road network we create.
        self.__acess=Access() #the access class to manage everything linked to saving and downloading
    
    def __repr__(self):
        """ redefining print """
        s=""
        s+=repr(self.__road_network)
        return s

    def draw_static(self):
        """
        Draws the graph of roads and intersections representing the road network
        (static drawing does not show cars).

        Parameters
        ----------
            None
            
        Returns 
        -------
            None
        """
        self.__road_network.draw_static()
    
    def get_copy_network(self):
        """function to get the network, don't use it"""
        self.__road_network.set_not_defined()
        return copy.deepcopy(self.__road_network)
    
    #############################################################################################################################
    ##
    ##                        Functions to  modify the network
    ##
    #############################################################################################################################

    def return_network(self):
        """
        returns the network we created
        Parameters
        ----------
            None
            
        Returns 
        -------
            RoadNetwork
        """
        if not (self.__road_network.is_well_defined()):
            self.__road_network.connect()
        return self.__road_network
    
    def set_entry(self,entry_num):
        """
        set the new entry

        Parameters
        ----------
            entry_num (int) : the label of the intersection connect to the entry
            
        Returns 
        -------
            None
        """
        self.__road_network.set_entree(entry_num)
    
    def set_exit(self,exit_num):
        """
        set the new exit

        Parameters
        ----------
            entry_num (int) : the label of the intersection connect to the exit
            
        Returns 
        -------
            None
        """
        self.__road_network.set_sortie(exit_num)
    
    def add_intersection(self, label, cordX=0, cordY=0):
        """
        Add an intersection to the network
        (allows to define these graphical coordinates (these coordinates have no impact on the dynamics of the system))

        Parameters
        ----------
            label (int) : the number of the intersection
            cordX (float) optional : the graphical coordinate in x of the intersection by default 0
            cordy (float) optional : the graphical coordinate in y of the intersection by default 0

        Returns 
        -------
            None
        """
        keys=self.__road_network.intersections.keys()
        if label in keys:
            raise OtherErrors("label already use please use another one")
        self.__road_network.add_intersection(label,cordX,cordY)
    
    def add_road(self,label,intersections1,intersections2,L=10,vm=20,vclass=LinearSpeed()):
        """
        Adds a road to the road network
        does not allow to create the entry and exit roads of the network.

        Parameters
        ----------
            label (int) : the label (number) of the road
            intersections1 (int) : the intersection where the road starts
            intersections2 (int) : the intersection where the road finished
            L (int) optional : length of the road; by default 10
            vm (int) optional : maximum speed on the road; by default 20
            vclass (Speed) optional : class of speed calculation on a road; by default LinearSpeed()
        
        Returns
        -------
            None
        """
        keys=self.__road_network.roads.keys()
        key=("normal",label)
        if key in keys:
            raise OtherErrors("label already use please use another one")
        self.__road_network.add_road(label,intersections1,intersections2,L,vm,vclass)
    
    def set_intersection_priority(self,num_inter,liste_priority):
        """
        Change the priority of all the road arriving to an intersection.

        Parameters
        ----------
            num_inter (int) : the number of the intersection considered
            liste_priority (list tuple) : list compose of the key and priority of the road,
                                          list of : (( type_road , num_road ) , priority):
                                          type_road (string) : the type of the road
                                          num_road (int) : the number of the road
                                          priority (float) : the priority pods we set to the road (must be beetween 0 and 1)

        Returns 
        -------
            None
        """
        self.__road_network.set_intersection_priority(num_inter,liste_priority)
    
    def revert_roads(self,list_roads):
        """
        revert the direction of all the road we specify

        Parameters
        ----------
            list_roads (list tuple) : list of the key of the road we must revert
            
        Returns 
        -------
            None
        """
        for i in range(len(list_roads)): #loop on the road we revert
            road_key=list_roads[i]
            inter_num1=self.__road_network.roads[road_key].i1
            inter_num2=self.__road_network.roads[road_key].i2
            self.__road_network.roads[road_key].i1=inter_num2
            self.__road_network.roads[road_key].i2=inter_num1

            self.__road_network.intersections[inter_num1].roads_exit.pop(self.__road_network.intersections[inter_num1].roads_exit.index(road_key)) #pop on list
            self.__road_network.intersections[inter_num2].roads_entry.pop(self.__road_network.intersections[inter_num2].roads_entry.index(road_key)) #pop on list

            self.__road_network.intersections[inter_num1].roads_entry.append(road_key)
            self.__road_network.intersections[inter_num2].roads_exit.append(road_key)
        try:
            self.__road_network.connect()
        except:
            self.__road_network.set_not_defined()
            print("Warning: the network isn't well defined")
    
    def change_road_param(self,list_roads,L=None,vmax=None,class_speed=None):
        """
        change some parameter of a list of roads

        if an optional parameter isn't specify, all road keep there actual value for this parameter

        Parameters
        ----------
            list_roads    (list tuple)       : list of the key of the roads we consider
            L               (float) optional : the length of the roads
            vmax            (float) optional : the max speed of the roads
            class_speed (Speed) optional : the speed class that is applied to the roads for speed calculation
            
        Returns 
        -------
            None
        """
        for i in range(len(list_roads)):
            road_key=list_roads[i]
            if (L!=None):
                self.__road_network.roads[road_key].l=L
            if (vmax!=None):
                self.__road_network.roads[road_key].vmax=vmax
            if (class_speed!=None):
                self.__road_network.roads[road_key].class_speed=class_speed
    
    def change_intersection_coord(self,label,coordX=None,coordY=None):
        """
        change the intersection coordinate

        Parameters
        ----------
            label (int) : the label of the intersection
            coordX (float) optional : the X coordinate of the intersection
            coordY (float) optional : the Y coordinate of the intersection
                                      if a coordinate isn't specify, we don't change it
            
        Returns 
        -------
            None
        """
        if (coordX!=None):
            self.__road_network.intersections[label].cordX=coordX
        if (coordY!=None):
            self.__road_network.intersections[label].cordY=coordY
    
    def __switch_label_road(self,prior_label,new_label):
        """

        Parameters
        ----------
            prior_label (tuple) : the prior key of the road (type_road, num_road)
            new_label (tuple)   : the new key of the road   (type_road, num_road)
            
        Returns 
        -------
            None
        """
        road=self.__road_network.roads[prior_label]
        inter_num1=road.i1
        inter_num2=road.i2

        #change on the road
        self.__road_network.roads.pop(prior_label) #pop on dict (suppresing the actual road)
        road.type_road=new_label[0]
        road.number=new_label[1]
        self.__road_network.roads[new_label]=road

        #change on the intersections
        self.__road_network.intersections[inter_num1].roads_exit[self.__road_network.intersections[inter_num1].roads_exit.index(prior_label)]=new_label
        self.__road_network.intersections[inter_num2].roads_entry[self.__road_network.intersections[inter_num2].roads_entry.index(prior_label)]=new_label
    
    def __switch_label_intersection(self,prior_label,new_label):
        """

        Parameters
        ----------
            prior_label (int) : the prior key of the intersection
            new_label (int)   : the new key of the intersection
            
        Returns 
        -------
            None
        """
        intersection=self.__road_network.intersections[prior_label]

        #change on the intersection
        self.__road_network.intersections.pop(prior_label) #pop on dict
        intersection.number=new_label
        self.__road_network.intersections[new_label]=intersection

        #change on the road
        for i in range(len(intersection.roads_entry)): #loop on all the roads that enter the intersection
            self.__road_network.roads[intersection.roads_entry[i]].i2=new_label
        
        for i in range(len(intersection.roads_exit)): #loop on all the roads that exit the intersection
            self.__road_network.roads[intersection.roads_exit[i]].i1=new_label

    
    def switch_labels(self,prior_road_label_list=[],new_road_label_list=[],prior_intersection_label_list=[],new_intersection_label_list=[]):
        
        if (len(prior_road_label_list)!=len(new_road_label_list)) or (len(prior_intersection_label_list)!=len(new_intersection_label_list)):
            raise OtherErrors("prior and new label list must be of equal size")

        #road case
        for i in range(len(prior_road_label_list)):
            prior_road_key=("normal",prior_road_label_list[i])
            new_road_key=("normal",new_road_label_list[i])
            tmp=self.Inner(new_road_key) #temporary label for relabeling
            self.__switch_label_road(prior_road_key,tmp)

        for i in range(len(prior_road_label_list)):
            prior_road_key=("normal",prior_road_label_list[i])
            new_road_key=("normal",new_road_label_list[i])
            tmp=self.Inner(new_road_key) #temporary label for relabeling
            self.__switch_label_road(tmp,new_road_key)

        #intersection case
        for i in range(len(prior_intersection_label_list)):
            tmp=self.Inner(new_intersection_label_list[i]) #temporary label for relabeling
            self.__switch_label_intersection(prior_intersection_label_list[i],tmp)
        
        for i in range(len(prior_intersection_label_list)):
            tmp=self.Inner(new_intersection_label_list[i]) #temporary label for relabeling
            self.__switch_label_intersection(tmp,new_intersection_label_list[i])
    
    def merge(self,other_network_in_creation,change_label=True):
        """
        merge two network, but don't do any conection beetween the two.
        (it just place them in the same network)

        Parameters
        ----------
            other_network_in_creation (CreateNetwork) : the network we want to add
            change_label (boolean) optional : define if we keep the label of the second network, or if we rename it automatically; by default True
            
        Returns 
        -------
            None
        """

        self.__road_network.set_not_defined()
        other_rn=other_network_in_creation.get_copy_network()

        #change to suppress exit road and entry road on the second network
        road_key_en=("entry",0)
        inter_num2=other_rn.roads[road_key_en].i2
        other_rn.intersections[inter_num2].roads_entry.pop(other_rn.intersections[inter_num2].roads_entry.index(road_key_en)) #pop on list
        other_rn.roads.pop(road_key_en) #pop on dict

        road_key_ex=("exit",0)
        inter_num1=other_rn.roads[road_key_ex].i1
        other_rn.intersections[inter_num1].roads_exit.pop(other_rn.intersections[inter_num1].roads_exit.index(road_key_ex)) #pop on list
        other_rn.roads.pop(road_key_ex) #pop on dict

        if (change_label): #automatique relabeling (only work with positive int as label)
            #intersection
            try:
                inter_max_label1=max(self.__road_network.intersections.keys())
                inter_min_label2=min(other_network_in_creation.__road_network.intersections.keys())
                
                new_label_list=list(other_network_in_creation.__road_network.intersections.keys())
                new_label_list_intersection = [x+inter_max_label1-inter_min_label2+1 for x in new_label_list]

            except TypeError:
                raise TypeError("you cant use change_label=True with none int label")
            except Exception as e:
                print("unespected error")
                raise e
            
            #road
            road_label1 = [ x[1] for x in self.__road_network.roads.keys()]
            road_label2 = [ x[1] for x in other_network_in_creation.__road_network.roads.keys()]
            try:
                road_max_label1=max(road_label1)
                road_min_label2=min(road_label2)

                new_label_list_road = [x+inter_max_label1-inter_min_label2+1 for x in new_label_list]
            except TypeError:
                raise TypeError("you cant use change_label=True with none int label")
            except Exception as e:
                print("unespected error")
                raise e
            new_key_list_road = [ ("normal",x) for x in new_label_list_road]
            print(new_key_list_road)

        else:
            #fuse the two network
            self.__road_network.intersections.update(other_rn.intersections)
            self.__road_network.roads.update(other_rn.roads)
        
    def linked_road(self,list_begin_intersection,list_exit_intersection,list_label,L=None,vmax=None,class_speed=None):
        if (len(list_begin_intersection)!=len(list_exit_intersection)):
            raise OtherErrors("list_begin_intersection and list_exit_intersection must be of equal size")
        
        #yes it's not prety
        if (L==None):
            if (vmax==None):
                if (class_speed==None):
                    for i in range(len(list_begin_intersection)):
                        self.add_road(label=list_label[i], intersections1=list_begin_intersection[i] ,intersections2=list_exit_intersection[i])
                else:
                    for i in range(len(list_begin_intersection)):
                        self.add_road(label=list_label[i], intersections1=list_begin_intersection[i] ,intersections2=list_exit_intersection[i], vclass=class_speed)
            else:
                if (class_speed==None):
                    for i in range(len(list_begin_intersection)):
                        self.add_road(label=list_label[i], intersections1=list_begin_intersection[i] ,intersections2=list_exit_intersection[i], vm=vmax)
                else:
                    for i in range(len(list_begin_intersection)):
                        self.add_road(label=list_label[i], intersections1=list_begin_intersection[i] ,intersections2=list_exit_intersection[i], vm=vmax, vclass=class_speed)
        else:
            if (vmax==None):
                if (class_speed==None):
                    for i in range(len(list_begin_intersection)):
                        self.add_road(label=list_label[i], intersections1=list_begin_intersection[i] ,intersections2=list_exit_intersection[i], L=L)
                else:
                    for i in range(len(list_begin_intersection)):
                        self.add_road(label=list_label[i], intersections1=list_begin_intersection[i] ,intersections2=list_exit_intersection[i], L=L, vclass=class_speed)
            else:
                if (class_speed==None):
                    for i in range(len(list_begin_intersection)):
                        self.add_road(label=list_label[i], intersections1=list_begin_intersection[i] ,intersections2=list_exit_intersection[i], L=L, vm=vmax)
                else:
                    for i in range(len(list_begin_intersection)):
                        self.add_road(label=list_label[i], intersections1=list_begin_intersection[i] ,intersections2=list_exit_intersection[i], L=L, vm=vmax, vclass=class_speed)
    
    #####################################################################
    ##
    ##                        Graphical modification
    ##
    ######################################################################
    
    def swap(self,axis,axis_pos):
        """
        swap the cordinate of the intersection of our network in along an axis define by the parameter

        Parameters
        ----------
            axis (string) : define if the axis is 'either horizontal' or 'vertical'
            axis_pos : the position of our axis: (the X coord if we swap in horizontal OR the Y coord if we swap in vertical)
            
        Returns 
        -------
            None
        """
        if (axis=="vertical"):
            for i in self.__road_network.intersections:
                self.__road_network.intersections[i].cordX=axis_pos+(axis_pos-self.__road_network.intersections[i].cordX)
        elif (axis=="horizontal"):
            for i in self.__road_network.intersections:
                self.__road_network.intersections[i].cordY=axis_pos+(axis_pos-self.__road_network.intersections[i].cordY)
        else:
            raise OtherErrors("argument axis must be either vertical or horizontal")
    
    def __rotate_point(self,point,center,angle):
        """
        rotate one point (private) (doesn't need the object)

        Parameters
        ----------
            point (tuple float) : the coordinate of the point we rotate
            center (tuple float) : the coordinate of the point around wich we rotate
            angle (float) : the angle of the rotation in radian
            
        Returns 
        -------
            (tuple float) : the coordinate of the point we have rotate
        """
        point_tmp=(point[0]-center[0],point[1]-center[1])
        point_tmp=(point_tmp[0]*np.cos(angle)+point_tmp[1]*np.sin(angle),-point_tmp[0]*np.sin(angle)+point_tmp[1]*np.cos(angle))
        point2=(point_tmp[0]+center[0],point_tmp[1]+center[1])
        return point2

    def rotate(self,center,angle):
        """
        rotate the network around one point with a certain angle (rotate in the trigonometric direction) base 2pi

        Parameters
        ----------
            center (tuple float) : the X, Y cord of the center of our rotation
            angle (float) : the angle of our rotation in radian
            
        Returns 
        -------
            None
        """
        for i in self.__road_network.intersections:
            point=(self.__road_network.intersections[i].cordX,self.__road_network.intersections[i].cordY)
            new_coord=self.__rotate_point(point,center,angle)
            self.__road_network.intersections[i].cordX=new_coord[0]
            self.__road_network.intersections[i].cordY=new_coord[1]

    def transpose(self,vector):
        """
        transpose the network

        Parameters
        ----------
            vector (tuple float) : the X, Y cord of the transposed vector.
            
        Returns 
        -------
            None
        """
        for i in self.__road_network.intersections:
            self.__road_network.intersections[i].cordX+=vector[0]
            self.__road_network.intersections[i].cordY+=vector[1]
    
    def round_coord(self,decimal=0):
        """
        round the intersection cordinate

        Parameters
        ----------
            decimal (int) : the number of decimal places
            
        Returns 
        -------
            None
        """
        for i in self.__road_network.intersections:
            self.__road_network.intersections[i].cordX=round(self.__road_network.intersections[i].cordX,decimal)
            self.__road_network.intersections[i].cordY=round(self.__road_network.intersections[i].cordY,decimal)
    
    #############################################################################################################################
    ##
    ##                        Functions to create the network automatically
    ##
    #############################################################################################################################

    def create_grid(self,height_road_number,height_road_size,length_road_number=None,length_road_size=None,vm=20,vclass=LinearSpeed(),size=1):
        """
        create a grid-shaped network

        Parameters
        ----------
            height_road_number (int)          : the height number of our grid (how many level we want)
            height_road_size (float)          : the height of our grid (the size we want our road to have)
            length_road_number (int) optional : the lenght number of our grid (how many level we want); by default the same as height_road_number
            length_road_size (float) optional : the lenght of our grid (how big we want our representationto be); by default the same as height_road_size

            vm             (float) optional : the max speed on the road; 20 by default
            vclass       (Speed) optional : the type of speed we want to use in our grid; LinearSpeed by default
            size           (float) optional : multiplier coefficient of the graphic size of everithing inside the network; 1 by default

        Returns 
        -------
            None
        """ 
        self.__road_network=RoadNetwork()#reseting network

        #defining undefine parameters
        if (length_road_number==None):
            length_road_number=height_road_number
        if (length_road_size==None):
            length_road_size=height_road_size
        
        entry_road_label=0
        exit_road_label=length_road_number*height_road_number-1
        
        if (height_road_number<=0 or length_road_number<=0):
            raise OtherErrors("cannot create a network with negative or zero lenght")
        
        #placing the intersection
        for i in range(height_road_number):
            for j in range(length_road_number):
                self.__road_network.add_intersection(label=i*length_road_number+j,cordX=i*height_road_size*size,cordY=j*length_road_size*size)
        
        #road going in vertical axes
        for i in range(height_road_number-1):
            for j in range(length_road_number):
                self.__road_network.add_road(label=length_road_number*i+j, intersections1=i*length_road_number+j, intersections2=(i+1)*length_road_number+j, L=height_road_size, vm=vm, vclass=vclass)
        
        #road going in horizontal axes
        n=(height_road_number-1)*length_road_number
        for j in range(length_road_number-1):
            for i in range(height_road_number):
                self.__road_network.add_road(label=n+height_road_number*j+i,intersections1=i*length_road_number+j,intersections2=i*length_road_number+j+1,L=length_road_size, vm=vm, vclass=vclass)

        self.__road_network.set_entree(entry_road_label)
        self.__road_network.set_sortie(exit_road_label)

        self.__road_network.connect()

    def create_traffic_circle(self,road_length,number_of_intersection,out_road_length,list_direction=None,vm=20,vclass=LinearSpeed(),coeff_size=1):
        """
        create a roundabout network

        Parameters
        ----------
            road_length (float)          : lenght of the road in the traffic_circle
            number_of_intersection (int) : number of intersection in the traffic_circle
            out_road_length (float)      : lenght of the road outside the traffic_circle

            list_direction (int) optional : list of the direction of the road going in and out of the traffic circle:
                                            0 : into, 1 : out, 2 : both, 3 : no road | by default list of 0
            vm           (float) optional : the max speed on the road; 20 by default
            vclass     (Speed) optional : the type of speed we want to use in our grid; LinearSpeed by default
            coeff_size   (float) optional : multiplier coefficient of the graphic size of everithing inside the network; 1 by default

        Returns 
        -------
            None
        """
        self.__road_network=RoadNetwork() #reseting network
        
        if (list_direction==None):
            list_direction=[0]*number_of_intersection
        entry_road_label=0
        exit_road_label=1

        radius=road_length*number_of_intersection/(2*np.pi)
        for i in range(number_of_intersection): #adding the intersection
            theta=i/number_of_intersection*2*np.pi
            self.__road_network.add_intersection(label=i,cordX=-np.sin(theta)*radius*coeff_size,cordY=np.cos(theta)*radius*coeff_size) #intersection in the trafic circle
            self.__road_network.add_intersection(label=number_of_intersection+i,cordX=(-np.sin(theta)*(radius+out_road_length))*coeff_size,cordY=(np.cos(theta)*(radius+out_road_length))*coeff_size) #intersection next to the trafic circle
        
        #adding the road of the circle
        for i in range(number_of_intersection-1):
            self.__road_network.add_road(label=i, intersections1=i, intersections2=i+1, L=road_length, vm=vm, vclass=vclass)
        self.__road_network.add_road(label=number_of_intersection-1, intersections1=number_of_intersection-1, intersections2=0, L=road_length, vm=vm, vclass=vclass)
        
        #adding the road outside the circle
        label_number_road=number_of_intersection
        for i in range(number_of_intersection):
            if (list_direction[i]==0):
                self.__road_network.add_road(label=label_number_road, intersections1=i+number_of_intersection, intersections2=i, L=road_length, vm=vm, vclass=vclass)
                label_number_road+=1
            elif (list_direction[i]==1):
                self.__road_network.add_road(label=label_number_road, intersections1=i, intersections2=i+number_of_intersection, L=road_length, vm=vm, vclass=vclass)
                label_number_road+=1
            elif (list_direction[i]==2):

                self.__road_network.add_road(label=label_number_road, intersections1=i+number_of_intersection, intersections2=i, L=road_length, vm=vm, vclass=vclass)
                label_number_road+=1
                self.__road_network.add_road(label=label_number_road, intersections1=i, intersections2=i+number_of_intersection, L=road_length, vm=vm, vclass=vclass)
                label_number_road+=1
            elif (list_direction[i]==3):
                pass
            else:
                message="unknow type of direction in list_direction."
                message+=" "+str(list_direction[i])+" is not a valid number"
                raise OtherErrors(message)
        
        self.__road_network.set_entree(entry_road_label)
        self.__road_network.set_sortie(exit_road_label)

        try:
            self.__road_network.connect()
        except ConnectionError:
            message="Warning: this configuration doesn't allow cars to be able to reach the exit from everywhere, please change this and call connect() before carrying on"
            print(message)
        except PriorityError:
            message="Warning: some intersection doesn't have a road arriving toward it, please change this and call connect() before carrying on" 
            print(message)
        except Exception as e:
            print("unespected error")
            raise e
         
            
    #############################################################################################################################
    ##
    ##                        Functions to save and load the network
    ##
    #############################################################################################################################

    def __to_number(self,string,force=False):
        """
        try to tranform the string to a number, if impossible, return the string
        """
        try :
            return int(string)
        except :
            try :
                return float(string)
            except :
                if (force):
                    raise TypeError("type must be converted to int or float, it wasn't possible")
                return string
    
    def __transform_string_to_speed(self,string):
        """
        Private method
        transform the string corresponding to the speed print to the speed object

        Parameters
        ----------
            string (string) : the string corresponding to the print of the object

        Returns 
        -------
            speed : the speed class corresponding to the string
        """
        if (string=="constant speed calculation function"):
            return Speed()
        elif (string[:33]=="linear speed calculation function"):
            index=string.index(" epsilon : ")
            epsilon=self.__to_number(string[index+11:],True)
            return LinearSpeed(epsilon)
        else:
            raise ModelImportingError(string+" is unknow speed class")
    
    def __transform_string_to_intersection(self,string):
        """
        Private method
        transform the string corresponding to the intersection print to the intersection object

        Parameters
        ----------
            string (string) : the string corresponding to the print of the object

        Returns 
        -------
            Intersection : the intersection corresponding to the string
        """
        intersection=Intersection(0)

        index1=string.index(" positioned in : ")
        intersection.number=self.__to_number(string[:index1])


        index2=string[index1+17:].index(" ")+index1+17
        intersection.cordX=self.__to_number(string[index1+17:index2],True)

        index3=string[index2+1:].index("\n")+index2+1
        intersection.cordY=self.__to_number(string[index2+1:index3],True)

        index4=string.index("connect to roads : ")
        index5=string[index4:].index(" / ")+index4
        entry_road=string[index4+19+1:index5-1]
        entry_road_list=entry_road.split(") (")

        for i in range(len(entry_road_list)):
            key=entry_road_list[i].split(",")
            if (len(key)!=2):
                raise ModelImportingError("wrong size of key for road in intersection creation")
            new_key=(key[0][1:-1],self.__to_number(key[1][1:]))
            intersection.roads_entry.append(new_key)

        index6=string.index("shortest path : ")

        exit_road=string[index5+4:index6-3]
        exit_road_list=exit_road.split(") (")

        for i in range(len(exit_road_list)):
            key=exit_road_list[i].split(",")
            if (len(key)!=2):
                raise ModelImportingError("wrong size of key for road in intersection creation")
            new_key=(key[0][1:-1],self.__to_number(key[1][1:]))
            intersection.roads_exit.append(new_key)

        return intersection
    
    def __transform_string_to_road(self,string):
        """"
        Private method
        transform the string corresponding to the road print to the road object

        Parameters
        ----------
            string (string) : the string corresponding to the print of the object

        Returns 
        -------
            Road : the road corresponding to the string
        """
        road=Road(0,0,0,0,0)#arbitrary initialisation, we will modify all the parameter afterward.

        index1=string.index(" from ")
        road.number=self.__to_number(string[0:index1])

        index2=string.index(" to ")
        road.i1=self.__to_number(string[index1+6:index2])

        index3=string.index(" length : ")
        road.i2=self.__to_number(string[index2+4:index3])

        if (road.i1==-1):
            road.type_road="entry"
        elif (road.i2==-1):
            road.type_road="exit"
        else:
            road.type_road="normal"

        index4=string.index(" priority weight : ")
        road.l=self.__to_number(string[index3+10:index4],True)

        index5=string.index(" max speed : ")
        road.priority=self.__to_number(string[index4+19:index5],True)

        index6=index5+13+string[index5+13:].index(" ")
        road.vmax=self.__to_number(string[index5+13:index6],True)

        index7=string.index(" ; car has priority : ")

        speed_function_string=string[index6+1:index7]
        road.class_speed=self.__transform_string_to_speed(speed_function_string)

        index8=string[index7+22:].index("\n")+index7+22

        boolean=string[index7+22:index8]
        if (boolean=="False"):
            road.has_priority=False
        elif (boolean=="True"):
            road.has_priority=True
        else:
            raise ModelImportingError("non boolean found for road.has_priority")
    
        road.cars=[]
    
        return road
    
    def __transform_string_to_model(self,string):
        """"
        Private method
        transform the string corresponding to the print of our model to the road_network model corresponding.

        Parameters
        ----------
            string (string) : the string corresponding to the print of the object

        Returns 
        -------
            Road_network : the model corresponding to the string
        """
        road_network=RoadNetwork()

        list_string=string.split("_"*20)

        #slice the print substring into all the big different categorie
        new_list_string=[]
        for i in range(len(list_string)):
            if (len(list_string[i])>0):
                if (list_string[i][0]=="_"):
                    index=list_string[i].index("\n")
                    list_string[i]=list_string[i][index+1:]
            if (len(list_string[i])>0):
                new_list_string.append(list_string[i])

        #separating all the different information
        entry_road_string=new_list_string[0]
        exit_road_string=new_list_string[1]
        normal_road_string=new_list_string[2]

        intersections_string=new_list_string[3]
        additional_information=new_list_string[4]


        if (len(new_list_string)!=5):
            raise ModelImportingError("error in decoding file, wrong splitting in different model part")
    
        #entry roads part
        if (entry_road_string[0:10]!="entry road"):
            raise ModelImportingError("file doesn't follow the correct configuration for model")
    
        entry_road_list_string=entry_road_string.split("\n"+"road ")
        for i in range(1,len(entry_road_list_string)):
            road=self.__transform_string_to_road(entry_road_list_string[i])
            road_network.roads[road.type_road,road.number]=road


        #exit roads part
        if (exit_road_string[0:9]!="exit road"):
            raise ModelImportingError("file doesn't follow the correct configuration for model")
    
        exit_road_list_string=exit_road_string.split("\n"+"road ")
        for i in range(1,len(exit_road_list_string)):
            road=self.__transform_string_to_road(exit_road_list_string[i])
            road_network.roads[road.type_road,road.number]=road


        #normal roads part
        if (normal_road_string[0:11]!="normal road"):
            raise ModelImportingError("file doesn't follow the correct configuration for model")
    
        normal_road_list_string=normal_road_string.split("\n"+"road ")
        for i in range(1,len(normal_road_list_string)):
            road=self.__transform_string_to_road(normal_road_list_string[i])
            road_network.roads[road.type_road,road.number]=road

        #intersections part
        if (intersections_string[0:13]!="intersections"):
            raise ModelImportingError("file doesn't follow the correct configuration for model")
    

        intersections_list_string=intersections_string.split("\n"+"intersection ")
        for i in range(1,len(intersections_list_string)):
            intersection=self.__transform_string_to_intersection(intersections_list_string[i])
            road_network.intersections[intersection.number]=intersection
        
        road_network.connect()
        return road_network
    
    def save_model(self,name):
        """
        save the model in the model directory

        Parameters
        ----------
            name (string) : the name of the model

        Returns 
        -------
            None
        """
        try :
            self.__road_network.connect()
        except ConnectionError:
            raise ConnectionError("your model must be well deifned in order to save it")
        except Exception as e:
            print("unespected error")
            raise e
        
        string=repr(self.__road_network)
        self.__acess.save_model(string,name)
    
    def load_model(self,name):
        """
        load the model from the model directory.

        Parameters
        ----------
            name (string) : the name of the model

        Returns 
        -------
            None
        """
        string=self.__acess.load_model(name)
        self.__road_network=self.__transform_string_to_model(string)

"""# SIMULATE NETWORK"""

class SimulateNetwork:
    """
    Class use to create a road network.
    """
    def __init__(self,road_network):
        self.__road_network=road_network #the road network we use for the simulation, bound to the class.
        self.__acess=Access() #the access class to manage everything linked to saving and downloading
    
    def __repr__(self):
        """ redefining print """
        s=""
        s+=repr(self.__road_network)
        return s
    
    #############################################################################################################################
    ##
    ##                        Functions to place car on the network
    ##
    #############################################################################################################################

    def road_reset(self):
        """
        Supresses all the cars in the network to get an empty network.

        Parameters
        ----------
            None

        Returns 
        -------
            None
        """
        self.__road_network.road_reset()

    def compile_car_system(self):
        """
        Makes all the calculations that can be done only after adding all the cars.
        Executed after each addition/removal of car from the system.

        Makes the calculations of l_min and dt_max of the system so that it is well conditioned.

        Parameters
        ----------
            None

        Returns 
        -------
            None
        """
        self.__road_network.compile_car_system()
    
    def position_car_road(self,n,type_road,road_num,type0=0,l=1):
        """
        Positions n cars on the road road_num, positions them in the middle of the road.

        Parameters
        ----------
            n (int) : the number of cars added to the road
            type_road (int) : the type of the road
            road_num (int) : the number of the road on which the cars are placed
            type0 (int) optional : the type of car we place : 0 = NonInterfererDistance, 1 NonInterferer, 2 Player, 3 Optimizer, by default 0
            l (int) optional : length of the cars; 1 by default

        Returns 
        -------
            None
        """
        list_type=[type0]*n
        self.__road_network.position_car(type_road,road_num,list_type,l)
    
    def __dispatch_type_car(self,types,pack_size):
        """
        take a list of int (the index corresponds to the type of car, the number to how many car we place inside)

        Parameters
        ----------
            type0 (list int)  : the type of car we place : 0 = NonInterfererDistance, 1 NonInterferer, 2 Player, 3 Optimizer, by default 0
            pack_size (int) : the size of a pack of value (the randomize positionning of car is done on subset of size pack_size)

        Returns 
        -------
            (list int) the list of the type of car we use (the number is the type of the car, the index is where we place it)
        """
        list_type=[]
        
        for i in range(pack_size):
            list_type_tmp=[]
            for j in range(len(types)):
                list_type_tmp+=[j]*(int(types[j]*(i+1)/pack_size)-int(types[j]*(i/pack_size)))
            rd.shuffle(list_type_tmp)
            list_type+=list_type_tmp

        return list_type

    def __dispatch_type_car_uniforme(self,types):
        """
        take a list of int (the index correspond to the type of car, the number to how many car we place inside it)

        Parameters
        ----------
            type0 (list int)  : the type of car we place : 0 = NonInterfererDistance, 1 NonInterferer, 2 Player, 3 Optimizer, by default 0
            pack_size (int) : the size of a pack of value (the randomize positionning of car is done on subset of size pack_size)

        Returns 
        -------
            (list int) the list of the type of car we use (the number is the type of the car, the indes is where we place it)
        """
        k=0
        n=0
        for i in range(len(types)):
            if types[i]>0:
                if k==0:
                    type_use=i
                    number_type_use=types[i]
                elif k==1:
                    type_use_other=i
                else:
                    raise OtherErrors("this functiononly work with two different type, not more.")
                k+=1
                n+=types[i]  

        liste=[-1]*n
        for x in range(number_type_use):
            liste[math.ceil(n/number_type_use/2+x*n/number_type_use)-1]=type_use
        
        for i in range(len(liste)):
            if liste[i]==-1:
                liste[i]=type_use_other

        return liste
    
    def __get_dict_choice(self,initial_road,list_list_choice):
        """
        Place Normal and predefine cars ont the road. set the predifine path for the predifine cars

        Parameters
        ----------
            initial_road (tuple) : the key of the road were we place the cars
            list_list_choice (list list) : the list of the choice the cars take the sublist is compose of 0 and 1 with sum equal to 0.
                                                                            1 place designe wich road the cars choose (from the smallest to the biggest label)

        Returns 
        -------
            dict key : int; value : tuple : the key are the intersection label, the value the road key. 
        """
        i=0
        inter_already_pass_by=[]
        finish=False
        n=len(list_list_choice)

        road_key_current=initial_road
        inter_key_current=self.__road_network.roads[road_key_current].i2

        dict_choice=dict()

        while (inter_key_current not in inter_already_pass_by and i<n):
            if (len(self.__road_network.intersections[inter_key_current].roads_exit)>1):
                inter_already_pass_by.append(inter_key_current)

                if (sum(list_list_choice[i])!=1):
                    raise OtherErrors("list_list_choice must have a sum either o or 1 has value and sum must be equal to one in a sublist ")
                liste_road_key=self.__road_network.intersections[inter_key_current].roads_exit
                liste_road_key.sort(key = lambda x: x[1])

                road_key_current=liste_road_key[list_list_choice[i].index(1)]
                dict_choice[inter_key_current]=road_key_current

                inter_key_current=self.__road_network.roads[road_key_current].i2

                i+=1
        return dict_choice
    
    
    def place_mix_controlled_cars(self,n,l,initial_road,tensor_car_decision):
        """
        Place Normal and predefine cars ont the road. set the predifine path for the predifine cars

        Parameters
        ----------
            n (int)  : total number of cars
            l (float) : the size of the cars
            initial_road (tuple) : the key of the road were we place the cars
            tensor_car_decision (list list list) : tensor of the path cars take:
                                                  first dim, the car from entry of the road to exit of the road
                                                  second dim, the number of intersection wherer the car choose
                                                  third dim, the choice : list of 0 and 1 (only 1 1), the one signify wich road we take.
                                                  (first position mean the road with the lowest label...)

        Returns 
        -------
            None
        """
        k=len(tensor_car_decision)
        h=0
        if (n<k):
            raise OtherErrors("total number of cars can't be smaller than controlled cars.")

        #place cars in the most uniform way
        cars_list_type=self.__dispatch_type_car_uniforme([0,n-k,0,0,0,k])
        self.__road_network.position_car(initial_road[0],initial_road[1],cars_list_type,l)

        for i in range(len(cars_list_type)):
            if (cars_list_type[i]==5): #predefine car type num
                self.__road_network.roads[initial_road].cars[-i-1].path=self.__get_dict_choice(initial_road,tensor_car_decision[h])
                h+=1
                #a dict compose of the next road to take : the key is the intersection label : the value is the next road key
    
    def position_car_road_mix_type(self,type_road,road_num,types,l=1,pack_size=None):
        """
        Positions n cars on the road road_num, positions them in the middle of the road.

        Parameters
        ----------
            type_road (int) : the type of the road
            road_num (int) : the number of the road on which the cars are placed
            types (list int)  : 
                          the type of car we place : 0 = NonInterfererDistance, 1 NonInterferer, 2 Player, 3 Optimizer, by default 0
            l (int) optional : length of the cars; 1 by default
            pack_size (int) optional : the size of a pack of value (the randomize positionning of car is done on subset of size pack_size); by default the size of our sample

        Returns 
        -------
            None
        """
        if (pack_size==None): #set pack_size to the size of our sample
            pack_size=sum(types)
        
        list_type=self.__dispatch_type_car(types,pack_size)
        self.__road_network.position_car(type_road,road_num,list_type,l)
    
    def position_car_uniform(self,n,type0=0,l=1):
        """
        Position n cars on each road, position them in the middle of the road (except for entry and exit).

        Parameters
        ----------
            n (int) : the number of cars added on each road
            type0 (int) optional : the type of car we place : 0 = NonInterfererDistance, 1 NonInterferer, 2 Player, 3 Optimizer, by default 0
            l (int) optional : length of the cars; 1 by default

        Returns 
        -------
            None
        """
        self.road_reset()
        for i in (self.__road_network.roads):
            if (self.__road_network.roads[i].type_road=="normal"):
                self.position_car_road(n,"normal",self.__road_network.roads[i].number,type0,l)
        self.compile_car_system()
    
    def position_car_uniform_mix_type(self,types,l=1,pack_size=None):
        """
        position n cars on each road, position them in the middle of the road (except for entry and exit)
        uses different types
        types is a list of the number of each type to put

        Parameters
        ----------
            types (list int) : list of the number of cars we put per road. the position in the list corresponds to the type of car we place: 
                pos 0 = NonInterfererDistance, 1 nonInterferer, 2 Player, 3 Optimizer
            l (int) optional : length of the cars; 1 by default
            pack_size (int) optional : the size of a pack of value (the randomize positionning of car is done on subset of size pack_size); by default the size of our sample

        Returns 
        -------
            None
        """
        if (pack_size==None): #set pack_size to the size of our sampling
            pack_size=sum(types)

        self.road_reset() 
        for i in (self.__road_network.roads):
            if (self.__road_network.roads[i].type_road=="normal"):
                self.position_car_road_mix_type("normal",self.__road_network.roads[i].number,types,l)
        self.compile_car_system()
      
    #############################################################################################################################
    ##
    ##                        Functions to advance time in the network and simulate traffic
    ##
    #############################################################################################################################

    def run(self,dt,n):
        """
        Execute n step.

        Parameters
        ----------
            dt (int) : the time step, by how much do we advance
            n (int) : how many time steps we execute
            
        Returns 
        -------
            None
        """        
        self.__road_network.run(dt,n)
    
    def run_cycle(self,dt,n):
        """
        Execute n step.
        puts the cars from the outgoing road on the incoming road

        Parameters
        ----------
            dt (int) : the time step, by how much do we advance
            n (int) : how many time steps we execute
            
        Returns 
        -------
            None
        """             
        self.__road_network.run_cycle(dt,n)

    #############################################################################################################################
    ##
    ##                        Functions to compute travel time on the network
    ##
    #############################################################################################################################

    def run_get_all_info(self,dt,Nmax,n_pass_over=1):
        """
        run at most Nmax step, stop when all the car have reach the exit road (they are supress when they arrive on the exit road)
        return all information regarding the car at each time of the system.

        Parameters
        ----------
            dt (int) : the time step, by how much do we advance
            Nmax (int) : the maximum number of steps we execute before stopping (Nmax reached means the car has not covered the whole system)
            n_pass_over (int) optional : the number of step we do before getting all infos of one state

        Returns 
        -------
            (list dict tuple) : list at each consider step of :
                           dict of tuple (car type num, car position on the road)
                           the key are (road_type, road_num, position in the road)
                           position on the road : the distance the car has travel on the road.
                           position in the road : mean number of car bettween the car and the exit of the road.
        """
        if (type(Nmax)!=int or type(n_pass_over)!=int):
            raise TypeError("type must be int")
        elif (Nmax<=0 or n_pass_over<=0):
            raise OtherErrors("Nmax and n_pass_over msut be positive integer and not 0")

        list_info=[]
        list_info.append(self.__road_network.get_all_car_info())
        q=Nmax//n_pass_over
        r=Nmax%n_pass_over
        n=self.__road_network.count_car_in_network() #the number of car at the beggin of the test
        i=0 #cars that are still in the network
        j=0 #count the number of block of step

        while (i<n and j<q): #loop to make the network advance
            k=0 #count the number of step inside one block of step
            while (i<n and k<n_pass_over):
                self.__road_network.roads["exit",0].cars=[]
                self.__road_network.step(dt)
                i+=len(self.__road_network.roads["exit",0].cars)
                k+=1
            list_info.append(self.__road_network.get_all_car_info())
            j+=1
        
        k=0
        while (i<n and k<r): #loop on the rest to make the network advance it's last few step.
            self.__road_network.roads["exit",0].cars=[]
            self.__road_network.step(dt)
            i+=len(self.__road_network.roads["exit",0].cars)
            k+=1
        
        if (k>0): #test if we end up doing at least one step of the above while
            list_info.append(self.__road_network.get_all_car_info())
        
        return list_info
    
    def time_calculation(self,dt,Nmax):
        """
        Calculates the time it will take for the cars to cross the road system,
            can only be done on a system containing cars.

        Parameters
        ----------
            dt (int) : the time step, by how much do we advance
            Nmax (int) : the maximum number of steps we execute before stopping (Nmax reached means the car has not covered the whole system)
            
        Returns 
        -------
            (int) : the time it takes the car to go through the system
        """
        l=50
        network=copy.deepcopy(self.__road_network)
        n=0
        k=0
        is_car_find=False
        car_find=False
        while (not is_car_find): #loop advancing the system to find a car
            if (len(network.roads["entry",0].cars)):
                car_number=network.roads["entry",0].cars[0].number
                is_car_find=True
                car_find=True
            else:
                network.step(dt)
                network.replace(l*dt)
                n+=1
                if (n>=Nmax):
                    is_car_find=True #we have used up our budget
                    print("budget atteint")
        
        if (car_find): #test if we have found a car
            has_car_enter=False
            while (not has_car_enter): #loop advancing the system to get the car off the entry road.
                network.replace(l*dt)
                network.step(dt)
                n+=1
                has_car_enter=not(network.roads["entry",0].has_car(car_number))
                if (n>=Nmax):
                    has_car_enter=True #we have used up our budget
                    print("budget atteint")
            
            has_car_exit=(network.roads["exit",0].has_car(car_number))
            while (not has_car_exit): #loop moving the system forward to get the car through the system. (stops when it reaches the entrance road)
                network.replace(l*dt)
                network.step(dt)
                k+=1
                n+=1
                has_car_exit=(network.roads["exit",0].has_car(car_number))
                if (n>=Nmax):
                    has_car_exit=True #we have used up our budget
                    print("budget atteint")
        
        return k*dt


    def time_calculation_mean(self,dt,Nmax,show=False):
        """
        Calculates the average time that cars will take to cross the road system.
            Can only be done on a system containing cars.

        Parameters
        ----------
            dt (int) : the time step, by how much do we advance
            Nmax (int) : the maximum number of steps we execute before stopping (Nmax reached means the car has not covered the whole system)
            show (boolean) optional : if True show additional information; False by default
            
        Returns 
        -------
            (int) : the time it takes the car to go through the system
        """ 
        l=50
        network=copy.deepcopy(self.__road_network)
        n=0
        is_car_find=False
        car_find=False
        liste_num_car=network.list_car()
        k=len(liste_num_car)
        liste_etat_car=[0]*k #Cars status : 0 : the car is not yet on the entrance road; 1 the car is on the entrance road (for the 1st time);
        # 2 : the car has entered but not exited (it has passed through the entrance road 1 time), 3 the car has exited (it has passed through the exit road (after passing through the entrance))
        liste_chrono=[0]*k
        number_car_finish=0

        while (n<Nmax and not(number_car_finish==k)): #loop making the cars move forward on the network
            for i in range(k): #loop on the cars

                if (liste_etat_car[i]==0): #The initial state, the car has not yet passed through the entrance, the timer has not been started
                    if (network.roads["entry",0].has_car(liste_num_car[i])):
                        liste_etat_car[i]=1

                elif (liste_etat_car[i]==1): #the car has arrived on the entrance road, we are about to start the timer
                    if (not(network.roads["entry",0].has_car(liste_num_car[i]))): #on déclenche le chrono
                        liste_etat_car[i]=2
                        liste_chrono[i]=1
                
                elif (liste_etat_car[i]==2): #the car went into the network, the timer has started
                    liste_chrono[i]+=1
                    if (network.roads["exit",0].has_car(liste_num_car[i])):
                        liste_etat_car[i]=3
                        number_car_finish+=1

                elif (liste_etat_car[i]==3): #the car has made a turn (we have measured our time), we wait for the others
                    pass
                else:
                    print("error, invalid car state")
            
            n+=1
            network.replace(l*dt)
            network.step(dt)
        
        n_budget_atteint=0 # the number of cars for which the budget has been reached without 1 lap.
        for i in range(k):
            if (liste_etat_car[i]!=3):
                n_budget_atteint+=1
        print("budget atteint pour "+str((n_budget_atteint/k)*100)+" % des cas")

        if (show):
            print(liste_chrono)
        return sum(liste_chrono)/k*dt
    

    def time_calculation_mean_lap(self,dt,Ntour,show=False):
        """
        Calculates the average time that cars will take to cross the road system.
            Can only be done on a system containing cars.

        Parameters
        ----------
            dt (int) : the time step, by how much do we advance
            Ntour (int) : the maximum number of steps we execute before stopping (Nmax reached means the car has not covered the whole system)
            show (boolean) optional : if True show additional information; False by default
            
        Returns 
        -------
            (int) : the time it takes the car to go through the system
        """ 

        l=50 #invisible road length
        network=copy.deepcopy(self.__road_network)

        liste_num_car=network.list_car()
        k=len(liste_num_car)
        liste_etat_car=[0]*k #Cars status : 0 : the car is not yet on the entrance road; 1 the car is on the entrance road (for the 1st time);
        # 2 : the car has entered but not exited (it has passed through the entrance road 1 time), 3 the car has exited (it has passed through the exit road (after passing through the entrance))
        liste_chrono_boucle=[0]*k

        liste_chrono_total=[]#list of all the time
        

        for i in range(Ntour): #loop making the cars move forward on the network
            for j in range(k): #loop on the cars

                if (liste_etat_car[j]==0): #The initial state, the car has not yet passed through the entrance, the timer has not been started
                    if (network.roads["entry",0].has_car(liste_num_car[j])):
                        liste_etat_car[j]=1

                elif (liste_etat_car[j]==1): #the car has arrived on the entrance road, we are about to start the timer
                    if (not(network.roads["entry",0].has_car(liste_num_car[j]))): #we start the stopwatch (for this cars)
                        liste_etat_car[j]=2
                        liste_chrono_boucle[j]=1
                
                elif (liste_etat_car[j]==2): #the car went into the network, the timer has started (for this cars)
                    liste_chrono_boucle[j]+=1
                    if (network.roads["exit",0].has_car(liste_num_car[j])):
                        liste_etat_car[j]=3
                
                elif (liste_etat_car[j]==3): #the car went out of the network, the timer has finish (for this cars)
                    liste_etat_car[j]=0
                    liste_chrono_total.append(liste_chrono_boucle[j])
                    liste_chrono_boucle[j]=0

                else:
                    print("erreur, invalid car state")
            
            network.replace(l*dt)
            network.step(dt)
        
        #we have finished the simulation, we order our result.
        if (len(liste_chrono_total)==0):
            total_time=0
            print("zeros car has done a tour.")
        else:
            total_time=sum(liste_chrono_total)/(len(liste_chrono_total))*dt
        
        if (show): #additional information
            n_not_finish=0
            time_not_finish=0
            for i in range(len(liste_etat_car)):
                if (liste_etat_car[i]==2):
                    n_not_finish+=1
                    time_not_finish+=liste_chrono_boucle[i]
            
            if (n_not_finish>0):
                mean_time_not_finish=(time_not_finish/n_not_finish)*dt
            else:
                mean_time_not_finish=0
            
            
            s1="number of car that finish : "
            s1+=str(len(liste_chrono_total))
            s1+=" ; average travel time : "
            s1+=str(total_time)

            s2="number of car that did not finish : "
            s2+=str(n_not_finish)
            s2+=" ; average travel time : "
            s2+=str(mean_time_not_finish)

            if (len(liste_chrono_total)!=0):
                var=np.var([i*dt for i in liste_chrono_total])
                min=np.min(liste_chrono_total)*dt
                max=np.max(liste_chrono_total)*dt

                q10=np.percentile(liste_chrono_total,10)*dt
                q25=np.percentile(liste_chrono_total,25)*dt
                median=np.percentile(liste_chrono_total,50)*dt
                q75=np.percentile(liste_chrono_total,75)*dt
                q90=np.percentile(liste_chrono_total,90)*dt

                s=""
                s+=s1
                s+="\n"
                s+=s2
                s+="\n"
                s+="var : "+str(var)+" ; min : "+str(min)+" ; max : "+str(max)
                s+="\n"
                s+="Qartile : 10% : "+str(q10)+" ; 25% : "+str(q25)+" ; median : "+str(median)+" ; 75% : "+str(q75)+" ; 90% : "+str(q90)
            else:
                s=""
                s+=s1
                s+="\n"
                s+=s2
                s+="\n"
                s+="no car finish : no further information like variance, min, max"
            print(s)
        
        return total_time

    def time_calculation_one_run(self,dt,Nmax,show=False):
        """
        Calculates the time it will take for the cars to cross the road system,
            can only be done on a system containing cars. (compute the time each cars stay in the network and sum them) then divide by the number of cars

        Parameters
        ----------
            dt (int) : the time step, by how much do we advance
            Nmax (int) : the maximum number of steps we execute before stopping (Nmax reached means the car has not covered the whole system)
            
        Returns 
        -------
            (int) : the time it takes the car to go through the system
        """
        network=copy.deepcopy(self.__road_network)
        #compute the number of cars
        list_time=[]
        n=self.__road_network.count_car_in_network()
        i=0
        k=0
        j=len(network.roads["exit",0].cars)
        if (j>0):
            network.roads["exit",0].cars=[]
            list_time.extend([0]*j)
            i+=j

        while (k<Nmax and i<n):
            k+=1
            network.step(dt)
            j=len(network.roads["exit",0].cars)
            if (j>0):
                network.roads["exit",0].cars=[]
                list_time.extend([dt*k]*j)
                i+=j
        if (i<n):
            print("budget atteint")
        if (show):
            if (len(list_time)>0):
                var=np.var(list_time)
                min=np.min(list_time)
                max=np.max(list_time)

                q10=np.percentile(list_time,10)
                q25=np.percentile(list_time,25)
                median=np.percentile(list_time,50)
                q75=np.percentile(list_time,75)
                q90=np.percentile(list_time,90)

                s=""
                s+=str(i)+" cars have finish"
                s+="\n"
                s+="var : "+str(var)+" ; min : "+str(min)+" ; max : "+str(max)
                s+="\n"
                s+="Qartile : 10% : "+str(q10)+" ; 25% : "+str(q25)+" ; median : "+str(median)+" ; 75% : "+str(q75)+" ; 90% : "+str(q90)

                print(s)
            else:
                print("no car did finish")
        
        if (n==0):
            return 0
        return sum(list_time)/n
    
    #############################################################################################################################
    ##
    ##                        Functions to draw the network
    ##
    #############################################################################################################################

    def show_pos_car(self):
        """
        Print the roads (their number) and the position of the cars on them (and their number).

        Parameters
        ----------
            None
            
        Returns 
        -------
            None
        """
        self.__road_network.show_pos_car()

    def draw_static(self):
        """
        Draws the graph of roads and intersections representing the road network
        (static drawing does not show cars).

        Parameters
        ----------
            None
            
        Returns 
        -------
            None
        """
        self.__road_network.draw_static()
    
    def __get_graph_pos_element(self):
        """
        Calculates the position in space of all the elements of the network (intersections, roads, cars),
        as well as the corresponding information (car number, car color),
        according to the coordinates given at the intersections.
        Returns this information.

        Parameters
        ----------
            None
            
        Returns 
        -------
            (list list float) : 2D tensor containing the list of intersections coordinates 
                first dimension : list of this information
                second dimension: [coordX, coordY]
            
            (list int) : list of the label of the intersections

            (list list list float) : 3D tensor containing the list of roads coordinates 
                first dimension : list of this information
                second dimension: [coord in, coord out]
                third dimension : [coordX, coordY]
            
            (list int) : list of the label of the roads

            (list list float) : 2D tensor containing the list of cars coordinates 
                first dimension : list of this information
                second dimension: [coordX, coordY]

            (list int) : list of car numbers (corresponds to the list of car coordinates)

            (list string) : list of car colors (corresponds to the list of car coordinates)

        """     
        inter_pos=[] #type [[cordX1,cordY1],[cordX2,cordY2], ... ] rank 2 tensor (dim: posX/ posY ; intersections)
        #loop which retrieves the coordinates of the intersections
        inter_label=[] #the number of the intersection
        for i in (self.__road_network.intersections):
            inter_pos.append([self.__road_network.intersections[i].cordX, self.__road_network.intersections[i].cordY])
            inter_label.append(self.__road_network.intersections[i].number)
        
        road_pos=[] #type [[[cordXe1,cordY1e],[cordXs1,cordYs1]],[[cordXe2,cordYe2],[cordXs2,cordYs2]],] rank 3 tensor (dim: posX/ posY; in/out; roads)
        road_label=[]#the number of the road

        car_pos=[] #type [[cordX1,cordY1],[cordX2,cordY2], ... ] rank 2 tensor (dim: posX/ posY ; car)
        car_color=[] #type [color1, color2, ... ] list of car colors
        car_num=[] #type [num1, num2, ... ] list of car numbers

        #loop that retrieves the coordinates of the roads and cars
        for i in (self.__road_network.roads):
            if (self.__road_network.roads[i].type_road=="normal"):
                inter_e_num=self.__road_network.roads[i].i1
                inter_s_num=self.__road_network.roads[i].i2
                tmp1=[self.__road_network.intersections[inter_e_num].cordX,self.__road_network.intersections[inter_e_num].cordY]
                tmp2=[self.__road_network.intersections[inter_s_num].cordX,self.__road_network.intersections[inter_s_num].cordY]
                road_pos.append([tmp1,tmp2])

                road_label.append(self.__road_network.roads[i].number)

                #loop on the road cars
                for j in range(len(self.__road_network.roads[i].cars)):
                    rapport=(self.__road_network.roads[i].cars[j].position)/(self.__road_network.roads[i].l)
                    cordX=tmp1[0]*(1-rapport)+tmp2[0]*rapport
                    cordY=tmp1[1]*(1-rapport)+tmp2[1]*rapport
                    car_pos.append([cordX,cordY])

                    car_num.append(self.__road_network.roads[i].cars[j].number)
                    car_color.append(self.__road_network.roads[i].cars[j].color)
        
        return inter_pos,inter_label,road_pos,road_label,car_pos,car_num,car_color

    def draw_car(self,label=False,cooef_size=1.5):
        """
        Uses get_graph_pos_car, calculates the coordinates of the system elements (intersections, roads, cars); 
        and some associated information (car number, car color).

        Draws the dynamic state of the network with intersections, roads and cars.
        can show the number of the different components.

        Parameters
        ----------
            label (Bool) optional : boolean defining if we show the numbers of the system components (True = we show them); False by default
            cooef_size (float) optional : cooef seting the size of all the object in the plot; 1.5 by default
            
        Returns 
        -------
            None
        """
        inter_pos,inter_label,road_pos,road_label,car_pos,car_num,car_colors=self.__get_graph_pos_element()

        #offset calculation, (how to place the annotations)
        mat=np.array(inter_pos)
        cord_min=np.amin(mat,0)
        cord_max=np.amax(mat,0)

        size_x=(cord_max[0]-cord_min[0])
        size_y=(cord_max[1]-cord_min[1])
        
        #intersections
        x_inter=[]
        y_inter=[]
        for i in range(len(inter_pos)):
            x_inter.append(inter_pos[i][0])
            y_inter.append(inter_pos[i][1])
        plt.scatter(x_inter,y_inter,s=100*cooef_size,color='green',zorder=2)
        
        #roads
        ox=[]
        oy=[]
        vx=[]
        vy=[]
        for i in range(len(road_pos)):
            ox.append(road_pos[i][0][0])
            oy.append(road_pos[i][0][1])
            vx.append(road_pos[i][1][0]-road_pos[i][0][0])
            vy.append(road_pos[i][1][1]-road_pos[i][0][1])
        plt.quiver(ox, oy, vx, vy, angles='xy', scale_units='xy', scale=1 ,color="blue",width=0.01*cooef_size)
        
        #cars
        x_car=[]
        y_car=[]
        for i in range(len(car_pos)):
            x_car.append(car_pos[i][0])
            y_car.append(car_pos[i][1])
        
        plt.scatter(x_car,y_car,s=50*cooef_size,color=car_colors,zorder=2)
        
        if (label):
            
            #intersections
            offset1=-0.01
            for i in range(len(inter_pos)):
                plt.annotate(inter_label[i], (x_inter[i]+offset1*size_x, y_inter[i]+offset1*size_y))
            
            #roads
            offset2=0.1
            for i in range(len(road_pos)):
                s='r$\textcolor{blue}'
                s+='{'
                s+=str(1)
                s+='}'
                plt.annotate(road_label[i], ((2*ox[i]+vx[i])/2, (2*oy[i]+vy[i])/2+offset2*size_y))
        
            #cars
            offset3=0.02
            for i in range(len(car_pos)):
                plt.annotate(car_num[i], (x_car[i]+offset3*size_x, y_car[i]))

            plt.show()

    def __get_graph_pos_element_without_car(self):
        """
        calculates the position in space of all the elements of the network (intersections, roads, cars), 
        as well as the corresponding information (car number, car color),
        according to the coordinates given at the intersections.
        returns this information

        Parameters
        ----------
            None
            
        Returns 
        -------
            (list list float) : 2D tensor containing the list of intersection coordinates 
                first dimension : list of this information
                second dimension: [coordX, coordY]
            
            (list int) : list of the label of the intersections

            (list list list float) : 3D tensor containing the list of road coordinates 
                first dimension : list of this information
                second dimension: [coord in, coord out].     
                third dimension: [coordX, coordY]
            
            (list int) : list of the label of the roads
            
            (list float) : list of road density


        """     
        inter_pos=[] #type [[cordX1,cordY1],[cordX2,cordY2], ... ] rank 2 tensor (dim: posX/ posY ; intersections)
        #loop that retrieves the coordinates of the intersections
        inter_label=[] #the number of the intersection
        for i in (self.__road_network.intersections):
            inter_pos.append([self.__road_network.intersections[i].cordX, self.__road_network.intersections[i].cordY])
            inter_label.append(self.__road_network.intersections[i].number)
        
        road_pos=[] #type [[[cordXe1,cordY1e],[cordXs1,cordYs1]],[[cordXe2,cordYe2],[cordXs2,cordYs2]],] rank 3 tensor (dim: posX/ posY; in/out; roads)
        road_density=[]#the density of the road
        road_label=[]#the number of the road

        #loop which retrieves the coordinates of the roads
        for i in (self.__road_network.roads):
            if (self.__road_network.roads[i].type_road=="normal"):
                inter_e_num=self.__road_network.roads[i].i1
                inter_s_num=self.__road_network.roads[i].i2
                tmp1=[self.__road_network.intersections[inter_e_num].cordX,self.__road_network.intersections[inter_e_num].cordY]
                tmp2=[self.__road_network.intersections[inter_s_num].cordX,self.__road_network.intersections[inter_s_num].cordY]
                road_pos.append([tmp1,tmp2])

                road_density.append(len(self.__road_network.roads[i].cars)/self.__road_network.roads[i].l)
                road_label.append(self.__road_network.roads[i].number)
        
        return inter_pos,inter_label,road_pos,road_label,road_density
    
    def draw_density(self,label=True,cooef_size=2):
        """
        Uses __get_graph_pos_element_without_car, computes the coordinates of the elements of the system (intersections, roads); 
        and some associated information (road_density).

        Draws the dynamic state of the network with intersections, roads colored by the density of cars.
        can show the number of the different components.

        Parameters
        ----------
            label (Bool) optional : boolean defining if we show the numbers of the system components (True = we show them); True by default
            cooef_size (float) optional : the thickness of the drawing; 2 by default
            
        Returns 
        -------
            None
        """
        inter_pos,inter_label,road_pos,road_label,road_density=self.__get_graph_pos_element_without_car()

        #calculation of road colors
        norm = Normalize()
        norm.autoscale(road_density)
        color_map=cm.autumn_r(norm(np.array(road_density)))

        #offset calculation, (how to place the annotations)
        mat=np.array(inter_pos)
        cord_min=np.amin(mat,0)
        cord_max=np.amax(mat,0)

        size_x=(cord_max[0]-cord_min[0])
        size_y=(cord_max[1]-cord_min[1])
        
        #intersections
        x_inter=[]
        y_inter=[]
        for i in range(len(inter_pos)):
            x_inter.append(inter_pos[i][0])
            y_inter.append(inter_pos[i][1])
        plt.scatter(x_inter,y_inter,s=100*cooef_size,color='green',zorder=2)
        
        #roads
        ox=[]
        oy=[]
        vx=[]
        vy=[]
        for i in range(len(road_pos)):
            ox.append(road_pos[i][0][0])
            oy.append(road_pos[i][0][1])
            vx.append(road_pos[i][1][0]-road_pos[i][0][0])
            vy.append(road_pos[i][1][1]-road_pos[i][0][1])
        plt.quiver(ox, oy, vx, vy, angles='xy', scale_units='xy', scale=1 ,color=color_map, width=0.01*cooef_size)
        plt.colorbar(cm.ScalarMappable(norm=norm, cmap=cm.autumn_r))
        
        if (label):
            
            #intersections
            offset1=-0.01
            for i in range(len(inter_pos)):
                plt.annotate(inter_label[i], (x_inter[i]+offset1*size_x, y_inter[i]+offset1*size_y))
            
            #roads
            offset2=-0.01
            for i in range(len(road_pos)):
                s='r$\textcolor{blue}'
                s+='{'
                s+=str(1)
                s+='}'
                plt.annotate(road_label[i], ((2*ox[i]+vx[i])/2+offset2*size_y, (2*oy[i]+vy[i])/2+offset2*size_y))

            plt.show()
    
    def draw_dynamic(self,dt,n_step,n_plot,label=False,cooef_size=1.5):
        """
        Execute n step.
        puts the cars from the outgoing road on the incoming road
        draw them each n_plot step

        Parameters
        ----------
            dt (int) : the time step, by how much do we advance
            n_step (int) : how many time steps we execute
            n_plot (int) : how many step we execute beetween each plot
            label (Bool) optional : boolean defining if we show the numbers of the system components (True = we show them)
            cooef_size (int) optional : cooef seting the size of all the object in the plot; 1.5 by default
            
        Returns 
        -------
            None
        """             
        l=50
        k=n_step//n_plot
        r=k%n_plot
        self.draw_car(label,cooef_size)
        for i in range(k):
            for j in range(n_plot):
                self.__road_network.step(dt)
            plt.pause(0.01)
            self.draw_car(label,cooef_size)
        for j in range(r):
            self.__road_network.step(dt)
        self.draw_car(label,cooef_size)
    
    def draw_dynamic_circle(self,dt,n_step,n_plot,label=False,cooef_size=1.5):
        """
        Execute n step.
        puts the cars from the outgoing road on the incoming road
        draw them each n_plot step

        Parameters
        ----------
            dt (int) : the time step, by how much do we advance
            n_step (int) : how many time steps we execute
            n_plot (int) : how many step we execute beetween each plot
            label (Bool) optional : boolean defining if we show the numbers of the system components (True = we show them)
            cooef_size (int) optional : cooef seting the size of all the object in the plot; 1.5 by default
            
        Returns 
        -------
            None
        """             
        l=50
        k=n_step//n_plot
        r=k%n_plot
        self.draw_car(label,cooef_size)
        for i in range(k):
            for j in range(n_plot):
                self.__road_network.step(dt)
                self.__road_network.replace(l*dt)
            plt.pause(0.01)
            self.draw_car(label,cooef_size)
        for j in range(r):
            self.__road_network.step(dt)
            self.__road_network.replace(l*dt)
        self.draw_car(label,cooef_size)
    
    def save_picture(self,name,label=False,cooef_size=1.5):
        """
        Uses get_graph_pos_car, calculates the coordinates of the system elements (intersections, roads, cars); 
        and some associated information (car number, car color).

        Draws the dynamic state of the network with intersections, roads and cars.
        can show the number of the different components.

        Parameters
        ----------
            label (Bool) optional : boolean defining if we show the numbers of the system components (True = we show them); False by default
            cooef_size (float) optional : cooef seting the size of all the object in the plot; 1.5 by default
            
        Returns 
        -------
            None
        """
        inter_pos,inter_label,road_pos,road_label,car_pos,car_num,car_colors=self.__get_graph_pos_element()

        #offset calculation, (how to place the annotations)
        mat=np.array(inter_pos)
        cord_min=np.amin(mat,0)
        cord_max=np.amax(mat,0)

        size_x=(cord_max[0]-cord_min[0])
        size_y=(cord_max[1]-cord_min[1])
        
        #intersections
        x_inter=[]
        y_inter=[]
        for i in range(len(inter_pos)):
            x_inter.append(inter_pos[i][0])
            y_inter.append(inter_pos[i][1])
        plt.scatter(x_inter,y_inter,s=100*cooef_size,color='green',zorder=2)
        
        #roads
        ox=[]
        oy=[]
        vx=[]
        vy=[]
        for i in range(len(road_pos)):
            ox.append(road_pos[i][0][0])
            oy.append(road_pos[i][0][1])
            vx.append(road_pos[i][1][0]-road_pos[i][0][0])
            vy.append(road_pos[i][1][1]-road_pos[i][0][1])
        plt.quiver(ox, oy, vx, vy, angles='xy', scale_units='xy', scale=1 ,color="blue",width=0.01*cooef_size)
        
        #cars
        x_car=[]
        y_car=[]
        for i in range(len(car_pos)):
            x_car.append(car_pos[i][0])
            y_car.append(car_pos[i][1])
        
        plt.scatter(x_car,y_car,s=50*cooef_size,color=car_colors,zorder=2)
        
        if (label):
            
            #intersections
            offset1=-0.01
            for i in range(len(inter_pos)):
                plt.annotate(inter_label[i], (x_inter[i]+offset1*size_x, y_inter[i]+offset1*size_y))
            
            #roads
            offset2=0.1
            for i in range(len(road_pos)):
                s='r$\textcolor{blue}'
                s+='{'
                s+=str(1)
                s+='}'
                plt.annotate(road_label[i], ((2*ox[i]+vx[i])/2, (2*oy[i]+vy[i])/2+offset2*size_y))
        
            #cars
            offset3=0.02
            for i in range(len(car_pos)):
                plt.annotate(car_num[i], (x_car[i]+offset3*size_x, y_car[i]))
        
        self.__acess.save_picture(plt,name)

    def save_video(self,dt,n_step,n_plot,fps,name,label=False,cooef_size=1.5):
        """
        Execute n step.
        puts the cars from the outgoing road on the incoming road
        save them each n_plot step

        Parameters
        ----------
            dt (int)     : the time step, by how much do we advance
            n_step (int) : how many time steps we execute
            n_plot (int) : how many step we execute beetween each plot
            fps (int)    : the number of picure per second in the video
            name (string): the name of the video
            label (Bool)     optional : boolean defining if we show the numbers of the system components (True = we show them)
            cooef_size (int) optional : cooef seting the size of all the object in the plot; 1.5 by default
            
        Returns 
        -------
            None
        """
        print("simulating model and saving picture")
        print("0%")
        self.__acess.suppres_temporary_video_directory() #clear the temporary directory of all picture it contain

        l=50
        k=n_step//n_plot
        r=k%n_plot
        self.save_picture(0,label,cooef_size)

        progression=0

        for i in range(k):
            for j in range(n_plot):
                self.__road_network.step(dt)
            plt.pause(0.01)
            self.save_picture(i+1,label,cooef_size)
            if ((i+1)/(k+1)*100>progression+0.5):
                progression=round((i+1)/(k+1)*100)
                print(str(progression)+"%")
        
        for j in range(r):
            self.__road_network.step(dt)
        self.save_picture(k+1,label,cooef_size)

        print("converting to video")

        self.__acess.transform_list_picture_to_video(fps,name)

        self.__acess.suppres_temporary_video_directory() #clear the temporary directory of all picture it contain

        print("process finish")
    
    def compute_global_time_with_control(self,n,l,initial_road,tensor_car_decision):
        """
        Place Normal and predefine cars ont the road. set the predifine path for the predifine cars
        then launch the simulation of our model and compute the mean time for all the cars

        Parameters
        ----------
            n (int)  : total number of cars
            l (float) : the size of the cars
            initial_road (tuple) : the key of the road were we place the cars
            tensor_car_decision (list list list) : tensor of the path cars take:
                                                  first dim, the car from entry of the road to exit of the road
                                                  second dim, the number of intersection wherer the car choose
                                                  third dim, the choice : list of 0 and 1 (only one 1), the 1 signify wich road we take.
                                                  (first position mean the road with the lowest label...)

        Returns 
        -------
            None
        """
        self.road_reset()
        self.place_mix_controlled_cars(n=n,l=l,initial_road=initial_road,tensor_car_decision=tensor_car_decision)
        self.compile_car_system()
        time=self.time_calculation_one_run(self.__road_network.get_dt_max()/2,10**5)# 10**5 should be big enought for or case (might not always be true)
        return time

"""# END"""

#compute only once
special_network=CreateNetwork()
special_network.load_model("initial config")
special_network.draw_static()

special_network_simulate=SimulateNetwork(special_network.return_network())#defining the network we use for our cost function

def f(matrix_car_decision):
    """
    the cost function for one configuration

    Parameters
    ----------
        matrix_car_decision (list list int) : matrix of zeros and 1 with one 1 and 2 zeros in each sublist
                                            the outside list represent the number of car
                                            each sublist (the position of the 1) define the choice of road the car follow:
                                            [1,0,0] : (lowest label) road 1 : middle road
                                            [0,1,0] : road 3 : bottom road
                                            [0,0,1] : (the higuest label) road 5 : the top road
            
    Returns 
    -------
        (float) : the average time it take for the car to exit the system.
    """
    tensor_car_decision=[]
    n=10 #the number of cars place on the initial road
    l=0.01 #the size of the cars
    initial_road=("normal",7) #the road where we initially place the cars

    #transformation of the matrix choice in the tensor choice.
    for i in range(len(matrix_car_decision)):
        tensor_car_decision.append([matrix_car_decision[i]])

    #call to general method for our road network with the metaparameter define above
    return special_network_simulate.compute_global_time_with_control(n=n,l=l,initial_road=initial_road,tensor_car_decision=tensor_car_decision)

def f2(matrix_car_decision):
    """
    the cost function for one configuration

    Parameters
    ----------
        matrix_car_decision (list list int) : matrix of zeros and 1 with one 1 and 2 zeros in each sublist
                                            the outside list represent the number of car
                                            each sublist (the position of the 1) define the choice of road the car follow:
                                            [1,0,0] : (lowest label) road 1 : middle road
                                            [0,1,0] : road 3 : bottom road
                                            [0,0,1] : (the higuest label) road 5 : the top road
            
    Returns 
    -------
        (float) : the average time it take for the car to exit the system.
    """
    tensor_car_decision=[]
    n=30 #the number of cars place on the initial road
    l=0.01 #the size of the cars
    initial_road=("normal",7) #the road where we initially place the cars

    #transformation of the matrix choice in the tensor choice.
    for i in range(len(matrix_car_decision)):
        tensor_car_decision.append([matrix_car_decision[i]])

    #call to general method for our road network with the metaparameter define above
    return special_network_simulate.compute_global_time_with_control(n=n,l=l,initial_road=initial_road,tensor_car_decision=tensor_car_decision)

acess=Access()
acess.contain_file(file_name="test_several_car_type.mp4",path_to_directory="/content/drive/My Drive/Road_Network/result/video")