"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const sns = require("../lib");
module.exports = {
    'create a subscription'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription', {
            Endpoint: 'endpoint',
            Protocol: 'lambda',
            TopicArn: {
                Ref: 'TopicBFC7AF6E'
            }
        }));
        test.done();
    },
    'with filter policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            filterPolicy: {
                color: sns.SubscriptionFilter.stringFilter({
                    whitelist: ['red', 'green'],
                    blacklist: ['white', 'orange'],
                    matchPrefixes: ['bl', 'ye'],
                }),
                price: sns.SubscriptionFilter.numericFilter({
                    whitelist: [100, 200],
                    between: { start: 300, stop: 350 },
                    greaterThan: 500,
                    lessThan: 1000,
                    betweenStrict: { start: 2000, stop: 3000 },
                    greaterThanOrEqualTo: 1000,
                    lessThanOrEqualTo: -2,
                })
            },
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription', {
            FilterPolicy: {
                color: [
                    'red',
                    'green',
                    { 'anything-but': ['white', 'orange'] },
                    { prefix: 'bl' },
                    { prefix: 'ye' }
                ],
                price: [
                    { numeric: ['=', 100] },
                    { numeric: ['=', 200] },
                    { numeric: ['>', 500] },
                    { numeric: ['>=', 1000] },
                    { numeric: ['<', 1000] },
                    { numeric: ['<=', -2] },
                    { numeric: ['>=', 300, '<=', 350] },
                    { numeric: ['>', 2000, '<', 3000] },
                ]
            },
        }));
        test.done();
    },
    'with existsFilter'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            filterPolicy: {
                size: sns.SubscriptionFilter.existsFilter(),
            },
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription', {
            FilterPolicy: {
                size: [{ exists: true }]
            },
        }));
        test.done();
    },
    'throws with raw delivery for protocol other than http, https or sqs'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // THEN
        test.throws(() => new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
            rawMessageDelivery: true
        }), /Raw message delivery/);
        test.done();
    },
    'throws with more than 5 attributes in a filter policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        const cond = { conditions: [] };
        // THEN
        test.throws(() => new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
            filterPolicy: {
                a: cond,
                b: cond,
                c: cond,
                d: cond,
                e: cond,
                f: cond,
            },
        }), /5 attribute names/);
        test.done();
    },
    'throws with more than 100 conditions in a filter policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // THEN
        test.throws(() => new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
            filterPolicy: {
                a: { conditions: [...Array.from(Array(2).keys())] },
                b: { conditions: [...Array.from(Array(10).keys())] },
                c: { conditions: [...Array.from(Array(6).keys())] },
            },
        }), /\(120\) must not exceed 100/);
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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