#!/usr/bin/env python

#  Copyright (c) 2019 JD Williams
#
#  This file is part of Firefly, a Python SOA framework built by JD Williams. Firefly is free software; you can
#  redistribute it and/or modify it under the terms of the GNU General Public License as published by the
#  Free Software Foundation; either version 3 of the License, or (at your option) any later version.
#
#  Firefly is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
#  Public License for more details. You should have received a copy of the GNU Lesser General Public
#  License along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
#  You should have received a copy of the GNU General Public License along with Firefly. If not, see
#  <http://www.gnu.org/licenses/>.

"""Using the JSON dumped by the dump-dependency-json generator,
generate input suitable for graphviz to render a dependency graph of
targets."""

import collections
import json
import sys


def ParseTarget(target):
  target, _, suffix = target.partition('#')
  filename, _, target = target.partition(':')
  return filename, target, suffix


def LoadEdges(filename, targets):
  """Load the edges map from the dump file, and filter it to only
  show targets in |targets| and their depedendents."""

  file = open('dump.json')
  edges = json.load(file)
  file.close()

  # Copy out only the edges we're interested in from the full edge list.
  target_edges = {}
  to_visit = targets[:]
  while to_visit:
    src = to_visit.pop()
    if src in target_edges:
      continue
    target_edges[src] = edges[src]
    to_visit.extend(edges[src])

  return target_edges


def WriteGraph(edges):
  """Print a graphviz graph to stdout.
  |edges| is a map of target to a list of other targets it depends on."""

  # Bucket targets by file.
  files = collections.defaultdict(list)
  for src, dst in edges.items():
    build_file, target_name, toolset = ParseTarget(src)
    files[build_file].append(src)

  print 'digraph D {'
  print '  fontsize=8'  # Used by subgraphs.
  print '  node [fontsize=8]'

  # Output nodes by file.  We must first write out each node within
  # its file grouping before writing out any edges that may refer
  # to those nodes.
  for filename, targets in files.items():
    if len(targets) == 1:
      # If there's only one node for this file, simplify
      # the display by making it a box without an internal node.
      target = targets[0]
      build_file, target_name, toolset = ParseTarget(target)
      print '  "%s" [shape=box, label="%s\\n%s"]' % (target, filename,
                                                     target_name)
    else:
      # Group multiple nodes together in a subgraph.
      print '  subgraph "cluster_%s" {' % filename
      print '    label = "%s"' % filename
      for target in targets:
        build_file, target_name, toolset = ParseTarget(target)
        print '    "%s" [label="%s"]' % (target, target_name)
      print '  }'

  # Now that we've placed all the nodes within subgraphs, output all
  # the edges between nodes.
  for src, dsts in edges.items():
    for dst in dsts:
      print '  "%s" -> "%s"' % (src, dst)

  print '}'


def main():
  if len(sys.argv) < 2:
    print >>sys.stderr, __doc__
    print >>sys.stderr
    print >>sys.stderr, 'usage: %s target1 target2...' % (sys.argv[0])
    return 1

  edges = LoadEdges('dump.json', sys.argv[1:])

  WriteGraph(edges)
  return 0


if __name__ == '__main__':
  sys.exit(main())
