"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.URLShortener = void 0;
const path = require("path");
const core_1 = require("@aws-cdk/core");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_apigateway_1 = require("@aws-cdk/aws-apigateway");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const utils_1 = require("./utils");
/**
 * Represents a URL shortener.
 *
 * Use `addDomainName` to configure a custom domain.
 *
 * By default, this construct will deploy:
 *
 * - An API Gateway API that can be accessed from a public endpoint.
 * - A DynamoDB table for storing links.
 * - A Lambda Function for shortening the link and storing it to DynamoDB table.
 */
class URLShortener extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._stack = core_1.Stack.of(this);
        this._apigw = new aws_apigateway_1.RestApi(this, 'Api', {
            restApiName: `${id}-URLShortener-Api`,
        });
        this._dynamoTable = (props === null || props === void 0 ? void 0 : props.dynamoTable) || this._makeTable();
        this._dynamoTableKeyName = utils_1.getKeySchemaProperty(this._dynamoTable, 'HASH').attributeName;
        this._shortenFn = this._makeShortenLambdaFn();
        const validationModel = this._makeValidationModel(this._apigw);
        const apigwDynamodbRole = this._makeRole(this._dynamoTable.tableArn);
        this._makeUsagePlan(this._apigw);
        const routes = {
            root: {
                childResources: {
                    '{id}': {
                        methods: [
                            {
                                method: 'GET',
                                integration: new aws_apigateway_1.AwsIntegration({
                                    service: 'dynamodb',
                                    action: 'UpdateItem',
                                    options: {
                                        credentialsRole: apigwDynamodbRole,
                                        passthroughBehavior: aws_apigateway_1.PassthroughBehavior.WHEN_NO_TEMPLATES,
                                        integrationResponses: [
                                            {
                                                selectionPattern: '200',
                                                responseParameters: {
                                                    'method.response.header.Location': 'integration.response.body.Attributes.url.S',
                                                    'method.response.header.Content-Type': "'text/html; charset=UTF-8'",
                                                },
                                                responseTemplates: {
                                                    'text/html; charset=UTF-8': [
                                                        "#if($input.path('$.Attributes.url.S').isEmpty() == false)",
                                                        "#set($url = $input.path('$.Attributes.url.S'))",
                                                        '#set($context.responseOverride.header.Location = $url)',
                                                        '<!DOCTYPE html><html><head><meta charset="UTF-8" />',
                                                        '<meta http-equiv="refresh" content="0;url=$url" />',
                                                        '<title>Redirecting to $url</title></head>',
                                                        '<body>Redirecting to <a href="$url">$url</a>.</body></html>',
                                                        '#else',
                                                        '#set($context.responseOverride.status = 404)',
                                                        '#end',
                                                    ].join(''),
                                                },
                                                statusCode: '301',
                                            },
                                            {
                                                statusCode: '404',
                                                responseTemplates: {
                                                    'text/html': 'Not Found',
                                                },
                                            },
                                        ],
                                        requestTemplates: {
                                            'application/json': JSON.stringify({
                                                TableName: this._dynamoTable.tableName,
                                                Key: {
                                                    [this._dynamoTableKeyName]: {
                                                        S: "$input.params('id')",
                                                    },
                                                },
                                                UpdateExpression: 'set clicks = clicks + :num, last_click_ts = :rt',
                                                ConditionExpression: `attribute_exists(${this._dynamoTableKeyName})`,
                                                ProjectionExpression: `${this._dynamoTableKeyName}, url`,
                                                ExpressionAttributeValues: {
                                                    ':num': {
                                                        N: '1',
                                                    },
                                                    ':rt': {
                                                        N: '$context.requestTimeEpoch',
                                                    },
                                                },
                                                ReturnValues: 'ALL_NEW',
                                            }),
                                        },
                                    },
                                }),
                                methodOptions: {
                                    methodResponses: [
                                        {
                                            responseParameters: {
                                                'method.response.header.Location': true,
                                                'method.response.header.Content-Type': true,
                                            },
                                            statusCode: '301',
                                        },
                                        {
                                            statusCode: '404',
                                        },
                                    ],
                                },
                            },
                        ],
                    },
                },
                methods: [
                    {
                        method: 'GET',
                        integration: new aws_apigateway_1.MockIntegration({
                            passthroughBehavior: aws_apigateway_1.PassthroughBehavior.WHEN_NO_TEMPLATES,
                            integrationResponses: [
                                {
                                    statusCode: '404',
                                    responseTemplates: {
                                        'text/html': 'Not Found',
                                    },
                                },
                            ],
                            requestTemplates: {
                                'application/json': JSON.stringify({
                                    statusCode: 404,
                                }),
                            },
                        }),
                        methodOptions: {
                            methodResponses: [
                                {
                                    responseParameters: {
                                        'method.response.header.Location': true,
                                        'method.response.header.Content-Type': true,
                                    },
                                    statusCode: '301',
                                },
                                {
                                    statusCode: '404',
                                },
                            ],
                        },
                    },
                    {
                        method: 'POST',
                        integration: new aws_apigateway_1.LambdaIntegration(this._shortenFn),
                        methodOptions: {
                            requestParameters: {
                                'method.request.header.Content-Type': true,
                            },
                            requestValidatorOptions: {
                                validateRequestBody: true,
                            },
                            requestModels: {
                                'application/json': validationModel,
                            },
                            apiKeyRequired: true,
                        },
                    },
                ],
            },
        };
        this._buildApiGateway(this._apigw.root, routes.root);
    }
    _buildApiGateway(parentResource, resourceConfiguration) {
        const { childResources = {}, methods = [] } = resourceConfiguration;
        methods.forEach(({ method, integration, methodOptions }) => {
            parentResource.addMethod(method, integration, methodOptions);
        });
        Object.entries(childResources).forEach(([pathPart, resourceConfiguration]) => {
            const childResource = parentResource.addResource(pathPart);
            this._buildApiGateway(childResource, resourceConfiguration);
        });
    }
    _makeTable() {
        return new aws_dynamodb_1.Table(this, 'Table', URLShortener.defaultDynamoTableProps);
    }
    _makeShortenLambdaFn() {
        const fn = new aws_lambda_1.Function(this, 'Function', {
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'lambda-fns', 'shorten')),
            handler: 'index.handler',
            environment: {
                TABLE_NAME: this._dynamoTable.tableName,
                KEY_NAME: this._dynamoTableKeyName,
            },
        });
        this._dynamoTable.grantWriteData(fn);
        return fn;
    }
    _makeValidationModel(api) {
        return api.addModel('ValidationModel', {
            modelName: 'ValidationModel',
            schema: {
                schema: aws_apigateway_1.JsonSchemaVersion.DRAFT4,
                type: aws_apigateway_1.JsonSchemaType.OBJECT,
                properties: {
                    url: {
                        type: aws_apigateway_1.JsonSchemaType.STRING,
                        format: 'uri',
                        pattern: '^https?://',
                    },
                },
                required: ['url'],
            },
            contentType: 'application/json',
        });
    }
    _makeRole(tableArn) {
        return new aws_iam_1.Role(this, 'Role', {
            assumedBy: new aws_iam_1.ServicePrincipal('apigateway.amazonaws.com'),
            inlinePolicies: {
                APIGatewayDynamoDBUpdateItem: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: ['dynamodb:UpdateItem'],
                            resources: [tableArn],
                        }),
                    ],
                }),
            },
        });
    }
    _makeUsagePlan(api) {
        const apiKey = api.addApiKey('ApiKey', {
            apiKeyName: `${this._stack.stackName}-URLShortener-ApiKey`,
        });
        new core_1.CfnOutput(this, 'ApiKeyURL', {
            value: `https://console.aws.amazon.com/apigateway/home?region=${this._stack.region}#/api-keys/${apiKey.keyId}`,
        });
        return api
            .addUsagePlan('UsagePlan', {
            name: `${this._stack.stackName}-URLShortener-UsagePlan`,
            apiKey,
            description: `Usage plan for ${this._stack.stackName} url shortener api`,
        })
            .addApiStage({
            stage: api.deploymentStage,
        });
    }
    addDomainName(options) {
        const { zone, domainName, certificate } = options;
        const domainNameHash = utils_1.hash(domainName);
        const domain = new aws_apigateway_1.DomainName(this, `DomainName${domainNameHash}`, {
            domainName,
            certificate: certificate ||
                new aws_certificatemanager_1.DnsValidatedCertificate(this, `DnsValidatedCertificate${domainNameHash}`, {
                    domainName,
                    hostedZone: zone,
                }),
        });
        domain.addBasePathMapping(this._apigw);
        const aliasRecord = new aws_route53_1.ARecord(this, `AliasRecord${domainNameHash}`, {
            zone,
            recordName: domainName,
            target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.ApiGatewayDomain(domain)),
        });
        new core_1.CfnOutput(this, `CustomDomainApiEndpoint${domainNameHash}`, {
            value: `https://${aliasRecord.domainName}`,
        });
        return this;
    }
}
exports.URLShortener = URLShortener;
/**
 * Default table props with partition key set to `id`, you can use it to extend
 * your `TableProps`.
 *
 * @example
 * const tableProps = {
 *   ...URLShortener.defaultDynamoTableProps,
 *   stream: StreamViewType.NEW_AND_OLD_IMAGES,
 * });
 */
URLShortener.defaultDynamoTableProps = {
    partitionKey: {
        name: 'id',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
};
//# sourceMappingURL=data:application/json;base64,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