import { Construct } from '@aws-cdk/core';
import { Table, TableProps } from '@aws-cdk/aws-dynamodb';
import { ICertificate } from '@aws-cdk/aws-certificatemanager';
import { IHostedZone } from '@aws-cdk/aws-route53';
/**
 * Properties to configure a URL shortener
 */
export interface URLShortenerProps {
    /**
     * The DynamoDB table used for storing links.
     *
     * A static property `defaultDynamoTableProps` is exposed with default
     * partition key set to `id`. You can extend it for your own `TableProps`.
     *
     * @example
     * const tableProps = {
     *   ...URLShortener.defaultDynamoTableProps,
     *   stream: StreamViewType.NEW_AND_OLD_IMAGES,
     * });
     *
     * @default - A new DynamoDB Table is created.
     */
    readonly dynamoTable?: Table;
}
/**
 * Properties to configure a domain name
 */
export interface CustomDomainOptions {
    /**
     * Domain name to be associated with URL shortener service, supports apex
     * domain and subdomain.
     */
    readonly domainName: string;
    /**
     * Hosted zone of the domain which will be used to create alias record(s)
     * from domain name in the hosted zone to URL shortener API endpoint.
     */
    readonly zone: IHostedZone;
    /**
     * The AWS Certificate Manager (ACM) certificate that will be associated with
     * the URL shortener that will be created.
     *
     * @default - A new DNS validated certificate is created in the same region.
     */
    readonly certificate?: ICertificate;
}
/**
 * Represents a URL shortener.
 *
 * Use `addDomainName` to configure a custom domain.
 *
 * By default, this construct will deploy:
 *
 * - An API Gateway API that can be accessed from a public endpoint.
 * - A DynamoDB table for storing links.
 * - A Lambda Function for shortening the link and storing it to DynamoDB table.
 */
export declare class URLShortener extends Construct {
    private readonly _stack;
    private readonly _apigw;
    private readonly _dynamoTable;
    private readonly _dynamoTableKeyName;
    private readonly _shortenFn;
    /**
     * Default table props with partition key set to `id`, you can use it to extend
     * your `TableProps`.
     *
     * @example
     * const tableProps = {
     *   ...URLShortener.defaultDynamoTableProps,
     *   stream: StreamViewType.NEW_AND_OLD_IMAGES,
     * });
     */
    static readonly defaultDynamoTableProps: TableProps;
    constructor(scope: Construct, id: string, props?: URLShortenerProps);
    private _buildApiGateway;
    private _makeTable;
    private _makeShortenLambdaFn;
    private _makeValidationModel;
    private _makeRole;
    private _makeUsagePlan;
    addDomainName(options: CustomDomainOptions): this;
}
