"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const spec_1 = require("@jsii/spec");
const fs_extra_1 = require("fs-extra");
const os_1 = require("os");
const path_1 = require("path");
const compiler_1 = require("../lib/compiler");
describe(compiler_1.Compiler, () => {
    describe('generated tsconfig', () => {
        test('default is tsconfig.json', () => {
            const sourceDir = (0, fs_extra_1.mkdtempSync)((0, path_1.join)((0, os_1.tmpdir)(), 'jsii-compiler-watch-mode-'));
            const compiler = new compiler_1.Compiler({
                projectInfo: _makeProjectInfo(sourceDir, 'index.d.ts'),
            });
            compiler.emit();
            expect((0, fs_extra_1.readJsonSync)((0, path_1.join)(sourceDir, 'tsconfig.json'), 'utf-8')).toEqual(expectedTypeScriptConfig());
        });
        test('file name can be customized', () => {
            const sourceDir = (0, fs_extra_1.mkdtempSync)((0, path_1.join)((0, os_1.tmpdir)(), 'jsii-compiler-watch-mode-'));
            const compiler = new compiler_1.Compiler({
                projectInfo: _makeProjectInfo(sourceDir, 'index.d.ts'),
                generateTypeScriptConfig: 'tsconfig.jsii.json',
            });
            compiler.emit();
            expect((0, fs_extra_1.readJsonSync)((0, path_1.join)(sourceDir, 'tsconfig.jsii.json'), 'utf-8')).toEqual(expectedTypeScriptConfig());
        });
    });
    test('"watch" mode', async () => {
        // This can be a little slow, allowing 15 seconds maximum here (default is 5 seconds)
        jest.setTimeout(15000);
        const sourceDir = (0, fs_extra_1.mkdtempSync)((0, path_1.join)((0, os_1.tmpdir)(), 'jsii-compiler-watch-mode-'));
        try {
            (0, fs_extra_1.writeFileSync)((0, path_1.join)(sourceDir, 'index.ts'), 'export class MarkerA {}');
            // Intentionally using lower case name - it should be case-insensitive
            (0, fs_extra_1.writeFileSync)((0, path_1.join)(sourceDir, 'readme.md'), '# Test Package');
            const compiler = new compiler_1.Compiler({
                projectInfo: _makeProjectInfo(sourceDir, 'index.d.ts'),
                failOnWarnings: true,
                projectReferences: false,
            });
            let firstCompilation = true;
            let onWatchClosed;
            let onWatchFailed;
            const watchClosed = new Promise((ok, ko) => {
                onWatchClosed = ok;
                onWatchFailed = ko;
            });
            const watch = await compiler.watch({
                nonBlocking: true,
                // Ignore diagnostics reporting (not to pollute test console output)
                reportDiagnostics: () => null,
                // Ignore watch status reporting (not to pollute test console output)
                reportWatchStatus: () => null,
                // Verify everything goes according to plan
                compilationComplete: (emitResult) => {
                    try {
                        expect(emitResult.emitSkipped).toBeFalsy();
                        const output = JSON.stringify((0, spec_1.loadAssemblyFromPath)(sourceDir));
                        if (firstCompilation) {
                            firstCompilation = false;
                            expect(output).toContain('"MarkerA"');
                            (0, fs_extra_1.writeFileSync)((0, path_1.join)(sourceDir, 'index.ts'), 'export class MarkerB {}');
                            return;
                        }
                        expect(output).toContain('"MarkerB"');
                        watch.close();
                        // Tell the test suite we're done here!
                        onWatchClosed();
                    }
                    catch (e) {
                        watch.close();
                        onWatchFailed(e);
                    }
                },
            });
            await watchClosed;
        }
        finally {
            (0, fs_extra_1.removeSync)(sourceDir);
        }
    });
    test('rootDir is added to assembly', () => {
        const outDir = 'jsii-outdir';
        const rootDir = 'jsii-rootdir';
        const sourceDir = (0, fs_extra_1.mkdtempSync)((0, path_1.join)((0, os_1.tmpdir)(), 'jsii-tmpdir'));
        (0, fs_extra_1.ensureDirSync)((0, path_1.join)(sourceDir, rootDir));
        try {
            (0, fs_extra_1.writeFileSync)((0, path_1.join)(sourceDir, rootDir, 'index.ts'), 'export class MarkerA {}');
            // Intentionally using lower case name - it should be case-insensitive
            (0, fs_extra_1.writeFileSync)((0, path_1.join)(sourceDir, rootDir, 'readme.md'), '# Test Package');
            const compiler = new compiler_1.Compiler({
                projectInfo: {
                    ..._makeProjectInfo(sourceDir, (0, path_1.join)(outDir, 'index.d.ts')),
                    tsc: {
                        outDir,
                        rootDir,
                    },
                },
                failOnWarnings: true,
                projectReferences: false,
            });
            compiler.emit();
            const assembly = (0, spec_1.loadAssemblyFromPath)(sourceDir);
            expect(assembly.metadata).toEqual(expect.objectContaining({
                tscRootDir: rootDir,
            }));
        }
        finally {
            (0, fs_extra_1.removeSync)(sourceDir);
        }
    });
    test('emits declaration map when feature is enabled', () => {
        const sourceDir = (0, fs_extra_1.mkdtempSync)((0, path_1.join)((0, os_1.tmpdir)(), 'jsii-tmpdir'));
        try {
            (0, fs_extra_1.writeFileSync)((0, path_1.join)(sourceDir, 'index.ts'), 'export class MarkerA {}');
            const compiler = new compiler_1.Compiler({
                projectInfo: {
                    ..._makeProjectInfo(sourceDir, 'index.d.ts'),
                    tsc: {
                        declarationMap: true,
                    },
                },
                generateTypeScriptConfig: 'tsconfig.jsii.json',
            });
            compiler.emit();
            expect(() => {
                (0, fs_extra_1.readFileSync)((0, path_1.join)(sourceDir, 'index.d.ts.map'), 'utf-8');
            }).not.toThrow();
        }
        finally {
            (0, fs_extra_1.removeSync)(sourceDir);
        }
    });
    describe('compressed assembly option', () => {
        test('creates a gzipped assembly file', () => {
            const sourceDir = (0, fs_extra_1.mkdtempSync)((0, path_1.join)((0, os_1.tmpdir)(), 'jsii-tmpdir'));
            try {
                (0, fs_extra_1.writeFileSync)((0, path_1.join)(sourceDir, 'index.ts'), 'export class MarkerA {}');
                const compiler = new compiler_1.Compiler({
                    projectInfo: _makeProjectInfo(sourceDir, 'index.d.ts'),
                    compressAssembly: true,
                });
                compiler.emit();
                expect((0, fs_extra_1.existsSync)((0, path_1.join)(sourceDir, spec_1.SPEC_FILE_NAME_COMPRESSED))).toBeTruthy();
            }
            finally {
                (0, fs_extra_1.removeSync)(sourceDir);
            }
        });
        test('creates file equivalent to uncompressed file', () => {
            const uncompressedSourceDir = (0, fs_extra_1.mkdtempSync)((0, path_1.join)((0, os_1.tmpdir)(), 'jsii-tmpdir'));
            const compressedSourceDir = (0, fs_extra_1.mkdtempSync)((0, path_1.join)((0, os_1.tmpdir)(), 'jsii-tmpdir-2'));
            try {
                const fileContents = 'export class MarkerA {}';
                (0, fs_extra_1.writeFileSync)((0, path_1.join)(uncompressedSourceDir, 'index.ts'), fileContents);
                (0, fs_extra_1.writeFileSync)((0, path_1.join)(compressedSourceDir, 'index.ts'), fileContents);
                const uncompressedJsiiCompiler = new compiler_1.Compiler({
                    projectInfo: _makeProjectInfo(uncompressedSourceDir, 'index.d.ts'),
                });
                const compressedJsiiCompiler = new compiler_1.Compiler({
                    projectInfo: _makeProjectInfo(compressedSourceDir, 'index.d.ts'),
                    compressAssembly: true,
                });
                uncompressedJsiiCompiler.emit();
                compressedJsiiCompiler.emit();
                // The files we expect are there
                expect((0, fs_extra_1.existsSync)((0, path_1.join)(uncompressedSourceDir, spec_1.SPEC_FILE_NAME))).toBeTruthy();
                expect((0, fs_extra_1.existsSync)((0, path_1.join)(compressedSourceDir, spec_1.SPEC_FILE_NAME_COMPRESSED))).toBeTruthy();
                const uncompressedJsii = (0, spec_1.loadAssemblyFromPath)(uncompressedSourceDir);
                const compressedJsii = (0, spec_1.loadAssemblyFromPath)(compressedSourceDir);
                expect(compressedJsii).toEqual(uncompressedJsii);
            }
            finally {
                (0, fs_extra_1.removeSync)(uncompressedSourceDir);
                (0, fs_extra_1.removeSync)(compressedSourceDir);
            }
        });
    });
});
function _makeProjectInfo(sourceDir, types) {
    return {
        projectRoot: sourceDir,
        packageJson: undefined,
        types,
        main: types.replace(/(?:\.d)?\.ts(x?)/, '.js$1'),
        name: 'jsii',
        version: '0.0.1',
        jsiiVersionFormat: 'short',
        license: 'Apache-2.0',
        author: { name: 'John Doe', roles: ['author'] },
        repository: { type: 'git', url: 'https://github.com/aws/jsii.git' },
        dependencies: {},
        peerDependencies: {},
        dependencyClosure: [],
        bundleDependencies: {},
        targets: {},
        excludeTypescript: [],
        tsc: {
            // NOTE: these are the default values jsii uses when none are provided in package.json.
            inlineSourceMap: true,
            inlineSources: true,
        },
    };
}
function expectedTypeScriptConfig() {
    return {
        _generated_by_jsii_: 'Generated by jsii - safe to delete, and ideally should be in .gitignore',
        compilerOptions: {
            alwaysStrict: true,
            charset: 'utf8',
            composite: false,
            declaration: true,
            experimentalDecorators: true,
            incremental: true,
            inlineSourceMap: true,
            inlineSources: true,
            lib: ['es2020'],
            module: 'CommonJS',
            newLine: 'lf',
            noEmitOnError: true,
            noFallthroughCasesInSwitch: true,
            noImplicitAny: true,
            noImplicitReturns: true,
            noImplicitThis: true,
            noUnusedLocals: true,
            noUnusedParameters: true,
            resolveJsonModule: true,
            skipLibCheck: true,
            strict: true,
            strictNullChecks: true,
            strictPropertyInitialization: true,
            stripInternal: false,
            target: 'ES2020',
            tsBuildInfoFile: 'tsconfig.tsbuildinfo',
        },
        exclude: ['node_modules'],
        include: [(0, path_1.join)('**', '*.ts')],
    };
}
//# sourceMappingURL=compiler.test.js.map