"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.stripAnsi = exports.findUp = exports.findPackageJsonUp = exports.findDependencyDirectory = exports.parseRepository = exports.parsePerson = exports.logDiagnostic = exports._formatDiagnostic = exports.formatDiagnostic = exports.diagnosticsLogger = exports.JSII_DIAGNOSTICS_CODE = exports.DIAGNOSTICS = void 0;
const fs = require("fs-extra");
const log4js = require("log4js");
const path = require("path");
const ts = require("typescript");
const jsii_diagnostic_1 = require("./jsii-diagnostic");
/**
 * Name of the logger for diagnostics information
 */
exports.DIAGNOSTICS = 'diagnostics';
/**
 * Diagnostic code for JSII-generated messages.
 */
exports.JSII_DIAGNOSTICS_CODE = 9999;
/**
 * Obtains the relevant logger to be used for a given diagnostic message.
 *
 * @param logger     the ``log4js.Logger`` to use for emitting the message.
 * @param diagnostic the message for which a logger is requested.
 *
 * @returns a logger method of the ``logger`` for the appropriate level.
 */
function diagnosticsLogger(logger, diagnostic) {
    switch (diagnostic.category) {
        case ts.DiagnosticCategory.Error:
            if (!logger.isErrorEnabled()) {
                return undefined;
            }
            return logger.error.bind(logger);
        case ts.DiagnosticCategory.Warning:
            if (!logger.isWarnEnabled()) {
                return undefined;
            }
            return logger.warn.bind(logger);
        case ts.DiagnosticCategory.Message:
            if (!logger.isDebugEnabled()) {
                return undefined;
            }
            return logger.debug.bind(logger);
        case ts.DiagnosticCategory.Suggestion:
        default:
            if (!logger.isTraceEnabled()) {
                return undefined;
            }
            return logger.trace.bind(logger);
    }
}
exports.diagnosticsLogger = diagnosticsLogger;
/**
 * Formats a diagnostic message with color and context, if possible.
 *
 * @param diagnostic  the diagnostic message ot be formatted.
 * @param projectRoot the root of the TypeScript project.
 *
 * @returns a formatted string.
 */
function formatDiagnostic(diagnostic, projectRoot) {
    if (jsii_diagnostic_1.JsiiDiagnostic.isJsiiDiagnostic(diagnostic)) {
        // Ensure we leverage pre-rendered diagnostics where available.
        return diagnostic.format(projectRoot);
    }
    return _formatDiagnostic(diagnostic, projectRoot);
}
exports.formatDiagnostic = formatDiagnostic;
/**
 * Formats a diagnostic message with color and context, if possible. Users
 * should use `formatDiagnostic` instead, as this implementation is intended for
 * internal usafe only.
 *
 * @param diagnostic  the diagnostic message ot be formatted.
 * @param projectRoot the root of the TypeScript project.
 *
 * @returns a formatted string.
 */
function _formatDiagnostic(diagnostic, projectRoot) {
    const formatDiagnosticsHost = {
        getCurrentDirectory: () => projectRoot,
        getCanonicalFileName: (fileName) => fileName,
        getNewLine: () => ts.sys.newLine,
    };
    const message = diagnostic.file != null
        ? ts.formatDiagnosticsWithColorAndContext([diagnostic], formatDiagnosticsHost)
        : ts.formatDiagnostic(diagnostic, formatDiagnosticsHost);
    if (!jsii_diagnostic_1.JsiiDiagnostic.isJsiiDiagnostic(diagnostic)) {
        return message;
    }
    // This is our own diagnostics, so we'll format appropriately (replacing TS#### with JSII####).
    return message.replace(` TS${exports.JSII_DIAGNOSTICS_CODE}: `, ` JSII${diagnostic.jsiiCode}: `);
}
exports._formatDiagnostic = _formatDiagnostic;
function logDiagnostic(diagnostic, projectRoot) {
    const logFunc = diagnosticsLogger(log4js.getLogger(exports.DIAGNOSTICS), diagnostic);
    if (!logFunc) {
        return;
    }
    logFunc(formatDiagnostic(diagnostic, projectRoot).trim());
}
exports.logDiagnostic = logDiagnostic;
const PERSON_REGEX = /^\s*(.+?)(?:\s*<([^>]+)>)?(?:\s*\(([^)]+)\))?\s*$/;
/**
 * Parses a string-formatted person entry from `package.json`.
 * @param value the string-formatted person entry.
 *
 * @example
 *  parsePerson("Barney Rubble <b@rubble.com> (http://barnyrubble.tumblr.com/)");
 *  // => { name: "Barney Rubble", email: "b@rubble.com", url: "http://barnyrubble.tumblr.com/" }
 */
function parsePerson(value) {
    const match = PERSON_REGEX.exec(value);
    if (!match) {
        throw new Error(`Invalid stringified "person" value: ${value}`);
    }
    const [, name, email, url] = match;
    const result = {
        name: name.trim(),
    };
    if (email) {
        result.email = email.trim();
    }
    if (url) {
        result.url = url.trim();
    }
    return result;
}
exports.parsePerson = parsePerson;
const REPOSITORY_REGEX = /^(?:(github|gist|bitbucket|gitlab):)?([A-Za-z\d_-]+\/[A-Za-z\d_-]+)$/;
function parseRepository(value) {
    const match = REPOSITORY_REGEX.exec(value);
    if (!match) {
        return { url: value };
    }
    const [, host, slug] = match;
    switch (host ?? 'github') {
        case 'github':
            return { url: `https://github.com/${slug}.git` };
        case 'gist':
            return { url: `https://gist.github.com/${slug}.git` };
        case 'bitbucket':
            return { url: `https://bitbucket.org/${slug}.git` };
        case 'gitlab':
            return { url: `https://gitlab.com/${slug}.git` };
        default:
            throw new Error(`Unknown host service: ${host}`);
    }
}
exports.parseRepository = parseRepository;
/**
 * Find the directory that contains a given dependency, identified by its 'package.json', from a starting search directory
 *
 * (This code is duplicated among jsii/jsii-pacmak/jsii-reflect. Changes should be done in all
 * 3 locations, and we should unify these at some point: https://github.com/aws/jsii/issues/3236)
 */
function findDependencyDirectory(dependencyName, searchStart) {
    // Explicitly do not use 'require("dep/package.json")' because that will fail if the
    // package does not export that particular file.
    const entryPoint = require.resolve(dependencyName, {
        paths: [searchStart],
    });
    // Search up from the given directory, looking for a package.json that matches
    // the dependency name (so we don't accidentally find stray 'package.jsons').
    const depPkgJsonPath = findPackageJsonUp(dependencyName, path.dirname(entryPoint));
    if (!depPkgJsonPath) {
        throw new Error(`Could not find dependency '${dependencyName}' from '${searchStart}'`);
    }
    return depPkgJsonPath;
}
exports.findDependencyDirectory = findDependencyDirectory;
/**
 * Find the package.json for a given package upwards from the given directory
 *
 * (This code is duplicated among jsii/jsii-pacmak/jsii-reflect. Changes should be done in all
 * 3 locations, and we should unify these at some point: https://github.com/aws/jsii/issues/3236)
 */
function findPackageJsonUp(packageName, directory) {
    return findUp(directory, (dir) => {
        const pjFile = path.join(dir, 'package.json');
        return (fs.pathExistsSync(pjFile) && fs.readJsonSync(pjFile).name === packageName);
    });
}
exports.findPackageJsonUp = findPackageJsonUp;
/**
 * Find a directory up the tree from a starting directory matching a condition
 *
 * Will return `undefined` if no directory matches
 *
 * (This code is duplicated among jsii/jsii-pacmak/jsii-reflect. Changes should be done in all
 * 3 locations, and we should unify these at some point: https://github.com/aws/jsii/issues/3236)
 */
function findUp(directory, pred) {
    const result = pred(directory);
    if (result) {
        return directory;
    }
    const parent = path.dirname(directory);
    if (parent === directory) {
        return undefined;
    }
    return findUp(parent, pred);
}
exports.findUp = findUp;
const ANSI_REGEX = 
// eslint-disable-next-line no-control-regex
/[\u001b\u009b][[()#;?]*(?:[0-9]{1,4}(?:;[0-9]{0,4})*)?[0-9A-ORZcf-nqry=><]/g;
function stripAnsi(x) {
    return x.replace(ANSI_REGEX, '');
}
exports.stripAnsi = stripAnsi;
//# sourceMappingURL=utils.js.map