import * as ts from 'typescript';
import { Emitter } from './emitter';
import { ProjectInfo } from './project-info';
export declare const DIAGNOSTICS = "diagnostics";
export declare const JSII_DIAGNOSTICS_CODE = 9999;
export interface CompilerOptions {
    /** The information about the project to be built */
    projectInfo: ProjectInfo;
    /** Whether the compiler should watch for changes or just compile once */
    watch?: boolean;
    /** Whether to detect and generate TypeScript project references */
    projectReferences?: boolean;
    /** Whether to fail when a warning is emitted */
    failOnWarnings?: boolean;
    /** Whether to strip deprecated members from emitted artifacts */
    stripDeprecated?: boolean;
    /** The path to an allowlist of FQNs to strip if stripDeprecated is set */
    stripDeprecatedAllowListFile?: string;
    /** Whether to add warnings for deprecated elements */
    addDeprecationWarnings?: boolean;
    /**
     * The name of the tsconfig file to generate
     * @default "tsconfig.json"
     */
    generateTypeScriptConfig?: string;
    /**
     * Whether to compress the assembly
     * @default false
     */
    compressAssembly?: boolean;
}
export interface TypescriptConfig {
    compilerOptions: ts.CompilerOptions;
    include?: string[];
    exclude?: string[];
    references?: ts.ProjectReference[];
}
export declare class Compiler implements Emitter {
    private readonly options;
    private readonly compilerHost;
    private typescriptConfig?;
    private rootFiles;
    private readonly configPath;
    private readonly projectReferences;
    constructor(options: CompilerOptions);
    /**
     * Compiles the configured program.
     *
     * @param files can be specified to override the standard source code location logic. Useful for example when testing "negatives".
     */
    emit(...files: string[]): ts.EmitResult;
    /**
     * Watches for file-system changes and dynamically recompiles the project as needed. In non-blocking mode, this
     * returns the TypeScript watch handle for the application to use.
     *
     * @internal
     */
    watch(opts: NonBlockingWatchOptions): Promise<ts.Watch<ts.BuilderProgram>>;
    /**
     * Watches for file-system changes and dynamically recompiles the project as needed. In blocking mode, this results
     * in a never-resolving promise.
     */
    watch(): Promise<never>;
    /**
     * Prepares the project for build, by creating the necessary configuration
     * file(s), and assigning the relevant root file(s).
     *
     * @param files the files that were specified as input in the CLI invocation.
     */
    private _prepareForBuild;
    /**
     * Do a single build
     */
    private _buildOnce;
    private _consumeProgram;
    /**
     * Build the TypeScript config object
     *
     * This is the object that will be written to disk.
     */
    private buildTypeScriptConfig;
    /**
     * Creates a `tsconfig.json` file to improve the IDE experience.
     *
     * @return the fully qualified path to the `tsconfig.json` file
     */
    private writeTypeScriptConfig;
    /**
     * Find all dependencies that look like TypeScript projects.
     *
     * Enumerate all dependencies, if they have a tsconfig.json file with
     * "composite: true" we consider them project references.
     *
     * (Note: TypeScript seems to only correctly find transitive project references
     * if there's an "index" tsconfig.json of all projects somewhere up the directory
     * tree)
     */
    private findProjectReferences;
    /**
     * Find source files using the same mechanism that the TypeScript compiler itself uses.
     *
     * Respects includes/excludes/etc.
     *
     * This makes it so that running 'typescript' and running 'jsii' has the same behavior.
     */
    private determineSources;
    /**
     * Resolve the given dependency name from the current package, and find the associated tsconfig.json location
     *
     * Because we have the following potential directory layout:
     *
     *   package/node_modules/some_dependency
     *   package/tsconfig.json
     *
     * We resolve symlinks and only find a "TypeScript" dependency if doesn't have 'node_modules' in
     * the path after resolving symlinks (i.e., if it's a peer package in the same monorepo).
     *
     * Returns undefined if no such tsconfig could be found.
     */
    private findMonorepoPeerTsconfig;
    private diagsHaveAbortableErrors;
}
/**
 * Options for Watch in non-blocking mode.
 *
 * @internal
 */
export interface NonBlockingWatchOptions {
    /**
     * Signals non-blocking execution
     */
    readonly nonBlocking: true;
    /**
     * Configures the diagnostics reporter
     */
    readonly reportDiagnostics: ts.DiagnosticReporter;
    /**
     * Configures the watch status reporter
     */
    readonly reportWatchStatus: ts.WatchStatusReporter;
    /**
     * This hook gets invoked when a compilation cycle (complete with Assembler execution) completes.
     */
    readonly compilationComplete: (emitResult: ts.EmitResult) => void;
}
//# sourceMappingURL=compiler.d.ts.map