"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_redshift_1 = require("aws-cdk-lib/aws-redshift");
const nag_rules_1 = require("../../nag-rules");
/**
 * Redshift clusters have user user activity logging enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_redshift_1.CfnCluster) {
        const clusterParameterGroupName = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.clusterParameterGroupName);
        if (clusterParameterGroupName === undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        let found = false;
        for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_redshift_1.CfnClusterParameterGroup) {
                const childParameterGroupName = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, child.ref);
                if (childParameterGroupName === clusterParameterGroupName) {
                    found = isCompliantClusterParameterGroup(child);
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether a given cluster parameter group is compliant
 * @param node the CfnClusterParameterGroup to check
 * returns whether the Cluster Parameter group is compliant
 */
function isCompliantClusterParameterGroup(node) {
    const resolvedParameters = aws_cdk_lib_1.Stack.of(node).resolve(node.parameters);
    if (resolvedParameters == undefined) {
        return false;
    }
    for (const parameter of resolvedParameters) {
        const resolvedParam = aws_cdk_lib_1.Stack.of(node).resolve(parameter);
        if (resolvedParam.parameterName === 'enable_user_activity_logging' &&
            resolvedParam.parameterValue === 'true') {
            return true;
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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