"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_kinesisanalytics_1 = require("aws-cdk-lib/aws-kinesisanalytics");
const nag_rules_1 = require("../../nag-rules");
/**
 * Kinesis Data Analytics Flink Applications have checkpointing enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_kinesisanalytics_1.CfnApplicationV2) {
        if (node.runtimeEnvironment.toLowerCase().startsWith('flink')) {
            const applicationConfiguration = aws_cdk_lib_1.Stack.of(node).resolve(node.applicationConfiguration);
            if (applicationConfiguration == undefined) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
            const flinkApplicationConfiguration = aws_cdk_lib_1.Stack.of(node).resolve(applicationConfiguration.flinkApplicationConfiguration);
            if (flinkApplicationConfiguration == undefined) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
            const checkpointConfiguration = aws_cdk_lib_1.Stack.of(node).resolve(flinkApplicationConfiguration.checkpointConfiguration);
            if (checkpointConfiguration == undefined) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
            if (nag_rules_1.NagRules.resolveIfPrimitive(node, checkpointConfiguration.configurationType) == 'CUSTOM') {
                const enabled = nag_rules_1.NagRules.resolveIfPrimitive(node, checkpointConfiguration.checkpointingEnabled);
                if (!enabled) {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiS2luZXNpc0RhdGFBbmFseXRpY3NGbGlua0NoZWNrcG9pbnRpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvcnVsZXMva2luZXNpcy9LaW5lc2lzRGF0YUFuYWx5dGljc0ZsaW5rQ2hlY2twb2ludGluZy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBOzs7RUFHRTtBQUNGLCtCQUE2QjtBQUM3Qiw2Q0FBaUQ7QUFDakQsMkVBQW9FO0FBQ3BFLCtDQUE4RDtBQUU5RDs7O0dBR0c7QUFDSCxrQkFBZSxNQUFNLENBQUMsY0FBYyxDQUNsQyxDQUFDLElBQWlCLEVBQXFCLEVBQUU7SUFDdkMsSUFBSSxJQUFJLFlBQVksdUNBQWdCLEVBQUU7UUFDcEMsSUFBSSxJQUFJLENBQUMsa0JBQWtCLENBQUMsV0FBVyxFQUFFLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQzdELE1BQU0sd0JBQXdCLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUNyRCxJQUFJLENBQUMsd0JBQXdCLENBQzlCLENBQUM7WUFDRixJQUFJLHdCQUF3QixJQUFJLFNBQVMsRUFBRTtnQkFDekMsT0FBTyw2QkFBaUIsQ0FBQyxhQUFhLENBQUM7YUFDeEM7WUFDRCxNQUFNLDZCQUE2QixHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FDMUQsd0JBQXdCLENBQUMsNkJBQTZCLENBQ3ZELENBQUM7WUFDRixJQUFJLDZCQUE2QixJQUFJLFNBQVMsRUFBRTtnQkFDOUMsT0FBTyw2QkFBaUIsQ0FBQyxhQUFhLENBQUM7YUFDeEM7WUFDRCxNQUFNLHVCQUF1QixHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FDcEQsNkJBQTZCLENBQUMsdUJBQXVCLENBQ3RELENBQUM7WUFDRixJQUFJLHVCQUF1QixJQUFJLFNBQVMsRUFBRTtnQkFDeEMsT0FBTyw2QkFBaUIsQ0FBQyxhQUFhLENBQUM7YUFDeEM7WUFDRCxJQUNFLG9CQUFRLENBQUMsa0JBQWtCLENBQ3pCLElBQUksRUFDSix1QkFBdUIsQ0FBQyxpQkFBaUIsQ0FDMUMsSUFBSSxRQUFRLEVBQ2I7Z0JBQ0EsTUFBTSxPQUFPLEdBQUcsb0JBQVEsQ0FBQyxrQkFBa0IsQ0FDekMsSUFBSSxFQUNKLHVCQUF1QixDQUFDLG9CQUFvQixDQUM3QyxDQUFDO2dCQUNGLElBQUksQ0FBQyxPQUFPLEVBQUU7b0JBQ1osT0FBTyw2QkFBaUIsQ0FBQyxhQUFhLENBQUM7aUJBQ3hDO2FBQ0Y7U0FDRjtRQUNELE9BQU8sNkJBQWlCLENBQUMsU0FBUyxDQUFDO0tBQ3BDO1NBQU07UUFDTCxPQUFPLDZCQUFpQixDQUFDLGNBQWMsQ0FBQztLQUN6QztBQUNILENBQUMsRUFDRCxNQUFNLEVBQ04sRUFBRSxLQUFLLEVBQUUsWUFBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDLElBQUksRUFBRSxDQUNsQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLypcbkNvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiovXG5pbXBvcnQgeyBwYXJzZSB9IGZyb20gJ3BhdGgnO1xuaW1wb3J0IHsgQ2ZuUmVzb3VyY2UsIFN0YWNrIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgQ2ZuQXBwbGljYXRpb25WMiB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1raW5lc2lzYW5hbHl0aWNzJztcbmltcG9ydCB7IE5hZ1J1bGVDb21wbGlhbmNlLCBOYWdSdWxlcyB9IGZyb20gJy4uLy4uL25hZy1ydWxlcyc7XG5cbi8qKlxuICogS2luZXNpcyBEYXRhIEFuYWx5dGljcyBGbGluayBBcHBsaWNhdGlvbnMgaGF2ZSBjaGVja3BvaW50aW5nIGVuYWJsZWRcbiAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICovXG5leHBvcnQgZGVmYXVsdCBPYmplY3QuZGVmaW5lUHJvcGVydHkoXG4gIChub2RlOiBDZm5SZXNvdXJjZSk6IE5hZ1J1bGVDb21wbGlhbmNlID0+IHtcbiAgICBpZiAobm9kZSBpbnN0YW5jZW9mIENmbkFwcGxpY2F0aW9uVjIpIHtcbiAgICAgIGlmIChub2RlLnJ1bnRpbWVFbnZpcm9ubWVudC50b0xvd2VyQ2FzZSgpLnN0YXJ0c1dpdGgoJ2ZsaW5rJykpIHtcbiAgICAgICAgY29uc3QgYXBwbGljYXRpb25Db25maWd1cmF0aW9uID0gU3RhY2sub2Yobm9kZSkucmVzb2x2ZShcbiAgICAgICAgICBub2RlLmFwcGxpY2F0aW9uQ29uZmlndXJhdGlvblxuICAgICAgICApO1xuICAgICAgICBpZiAoYXBwbGljYXRpb25Db25maWd1cmF0aW9uID09IHVuZGVmaW5lZCkge1xuICAgICAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5OT05fQ09NUExJQU5UO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IGZsaW5rQXBwbGljYXRpb25Db25maWd1cmF0aW9uID0gU3RhY2sub2Yobm9kZSkucmVzb2x2ZShcbiAgICAgICAgICBhcHBsaWNhdGlvbkNvbmZpZ3VyYXRpb24uZmxpbmtBcHBsaWNhdGlvbkNvbmZpZ3VyYXRpb25cbiAgICAgICAgKTtcbiAgICAgICAgaWYgKGZsaW5rQXBwbGljYXRpb25Db25maWd1cmF0aW9uID09IHVuZGVmaW5lZCkge1xuICAgICAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5OT05fQ09NUExJQU5UO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IGNoZWNrcG9pbnRDb25maWd1cmF0aW9uID0gU3RhY2sub2Yobm9kZSkucmVzb2x2ZShcbiAgICAgICAgICBmbGlua0FwcGxpY2F0aW9uQ29uZmlndXJhdGlvbi5jaGVja3BvaW50Q29uZmlndXJhdGlvblxuICAgICAgICApO1xuICAgICAgICBpZiAoY2hlY2twb2ludENvbmZpZ3VyYXRpb24gPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgcmV0dXJuIE5hZ1J1bGVDb21wbGlhbmNlLk5PTl9DT01QTElBTlQ7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKFxuICAgICAgICAgIE5hZ1J1bGVzLnJlc29sdmVJZlByaW1pdGl2ZShcbiAgICAgICAgICAgIG5vZGUsXG4gICAgICAgICAgICBjaGVja3BvaW50Q29uZmlndXJhdGlvbi5jb25maWd1cmF0aW9uVHlwZVxuICAgICAgICAgICkgPT0gJ0NVU1RPTSdcbiAgICAgICAgKSB7XG4gICAgICAgICAgY29uc3QgZW5hYmxlZCA9IE5hZ1J1bGVzLnJlc29sdmVJZlByaW1pdGl2ZShcbiAgICAgICAgICAgIG5vZGUsXG4gICAgICAgICAgICBjaGVja3BvaW50Q29uZmlndXJhdGlvbi5jaGVja3BvaW50aW5nRW5hYmxlZFxuICAgICAgICAgICk7XG4gICAgICAgICAgaWYgKCFlbmFibGVkKSB7XG4gICAgICAgICAgICByZXR1cm4gTmFnUnVsZUNvbXBsaWFuY2UuTk9OX0NPTVBMSUFOVDtcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5DT01QTElBTlQ7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5OT1RfQVBQTElDQUJMRTtcbiAgICB9XG4gIH0sXG4gICduYW1lJyxcbiAgeyB2YWx1ZTogcGFyc2UoX19maWxlbmFtZSkubmFtZSB9XG4pO1xuIl19