"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NagPack = exports.NagMessageLevel = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const fs_1 = require("fs");
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ignore_suppression_conditions_1 = require("./ignore-suppression-conditions");
const nag_rules_1 = require("./nag-rules");
const nag_suppression_helper_1 = require("./utils/nag-suppression-helper");
const VALIDATION_FAILURE_ID = 'CdkNagValidationFailure';
const SUPPRESSION_ID = 'CdkNagSuppression';
/**
 * The severity level of the rule.
 */
var NagMessageLevel;
(function (NagMessageLevel) {
    NagMessageLevel["WARN"] = "Warning";
    NagMessageLevel["ERROR"] = "Error";
})(NagMessageLevel = exports.NagMessageLevel || (exports.NagMessageLevel = {}));
/**
 * Base class for all rule packs.
 */
class NagPack {
    constructor(props) {
        this.reportStacks = new Array();
        this.packName = '';
        this.verbose =
            props == undefined || props.verbose == undefined ? false : props.verbose;
        this.logIgnores =
            props == undefined || props.logIgnores == undefined
                ? false
                : props.logIgnores;
        this.reports =
            props == undefined || props.reports == undefined ? true : props.reports;
        this.userGlobalSuppressionIgnore = props?.suppressionIgnoreCondition;
    }
    get readPackName() {
        return this.packName;
    }
    get readReportStacks() {
        return this.reportStacks;
    }
    /**
     * Create a rule to be used in the NagPack.
     * @param params The @IApplyRule interface with rule details.
     */
    applyRule(params) {
        if (this.packName === '') {
            throw Error('The NagPack does not have a pack name, therefore the rule could not be applied. Set a packName in the NagPack constructor.');
        }
        const allSuppressions = nag_suppression_helper_1.NagSuppressionHelper.getSuppressions(params.node);
        const ruleSuffix = params.ruleSuffixOverride
            ? params.ruleSuffixOverride
            : params.rule.name;
        const ruleId = `${this.packName}-${ruleSuffix}`;
        try {
            const ruleCompliance = params.rule(params.node);
            if (this.reports === true) {
                this.initializeStackReport(params);
                if (ruleCompliance === nag_rules_1.NagRuleCompliance.COMPLIANT) {
                    this.writeToStackComplianceReport(params, ruleId, ruleCompliance);
                }
            }
            if (this.isNonCompliant(ruleCompliance)) {
                const findings = this.asFindings(ruleCompliance);
                for (const findingId of findings) {
                    const suppressionReason = this.ignoreRule(allSuppressions, ruleId, findingId, params.node, params.level, params.ignoreSuppressionCondition);
                    if (this.reports === true) {
                        this.writeToStackComplianceReport(params, ruleId, nag_rules_1.NagRuleCompliance.NON_COMPLIANT, suppressionReason);
                    }
                    if (suppressionReason) {
                        if (this.logIgnores === true) {
                            const message = this.createMessage(SUPPRESSION_ID, findingId, `${ruleId} was triggered but suppressed.`, `Provided reason: "${suppressionReason}"`);
                            aws_cdk_lib_1.Annotations.of(params.node).addInfo(message);
                        }
                    }
                    else {
                        const message = this.createMessage(ruleId, findingId, params.info, params.explanation);
                        if (params.level == NagMessageLevel.ERROR) {
                            aws_cdk_lib_1.Annotations.of(params.node).addError(message);
                        }
                        else if (params.level == NagMessageLevel.WARN) {
                            aws_cdk_lib_1.Annotations.of(params.node).addWarning(message);
                        }
                    }
                }
            }
        }
        catch (error) {
            const reason = this.ignoreRule(allSuppressions, VALIDATION_FAILURE_ID, '', params.node, params.level, params.ignoreSuppressionCondition);
            if (this.reports === true) {
                this.writeToStackComplianceReport(params, ruleId, 'UNKNOWN', reason);
            }
            if (reason) {
                if (this.logIgnores === true) {
                    const message = this.createMessage(SUPPRESSION_ID, '', `${VALIDATION_FAILURE_ID} was triggered but suppressed.`, reason);
                    aws_cdk_lib_1.Annotations.of(params.node).addInfo(message);
                }
            }
            else {
                const information = `'${ruleId}' threw an error during validation. This is generally caused by a parameter referencing an intrinsic function. For more details enable verbose logging.'`;
                const message = this.createMessage(VALIDATION_FAILURE_ID, '', information, error.message);
                aws_cdk_lib_1.Annotations.of(params.node).addWarning(message);
            }
        }
    }
    /**
     * Check whether a specific rule should be ignored.
     * @param suppressions The suppressions listed in the cdk-nag metadata.
     * @param ruleId The id of the rule to ignore.
     * @param resource The resource being evaluated.
     * @param findingId The id of the finding that is being checked.
     * @returns The reason the rule was ignored, or an empty string.
     */
    ignoreRule(suppressions, ruleId, findingId, resource, level, ignoreSuppressionCondition) {
        for (let suppression of suppressions) {
            if (nag_suppression_helper_1.NagSuppressionHelper.doesApply(suppression, ruleId, findingId)) {
                const ignoreMessage = new ignore_suppression_conditions_1.SuppressionIgnoreOr(this.userGlobalSuppressionIgnore ?? new ignore_suppression_conditions_1.SuppressionIgnoreNever(), this.packGlobalSuppressionIgnore ?? new ignore_suppression_conditions_1.SuppressionIgnoreNever(), ignoreSuppressionCondition ?? new ignore_suppression_conditions_1.SuppressionIgnoreNever()).createMessage({
                    resource,
                    reason: suppression.reason,
                    ruleId,
                    findingId,
                    ruleLevel: level,
                });
                if (ignoreMessage) {
                    let id = findingId ? `${ruleId}[${findingId}]` : `${ruleId}`;
                    aws_cdk_lib_1.Annotations.of(resource).addInfo(`The suppression for ${id} was ignored for the following reason(s).\n\t${ignoreMessage}`);
                }
                else {
                    if (!suppression.appliesTo) {
                        // the rule is not granular so it always applies
                        return suppression.reason;
                    }
                    else {
                        return `[${findingId}] ${suppression.reason}`;
                    }
                }
            }
        }
        return '';
    }
    /**
     * The message to output to the console when a rule is triggered.
     * @param ruleId The id of the rule.
     * @param findingId The id of the finding.
     * @param info Why the rule was triggered.
     * @param explanation Why the rule exists.
     * @returns The formatted message string.
     */
    createMessage(ruleId, findingId, info, explanation) {
        let message = findingId
            ? `${ruleId}[${findingId}]: ${info}`
            : `${ruleId}: ${info}`;
        return this.verbose ? `${message} ${explanation}\n` : `${message}\n`;
    }
    /**
     * Write a line to the rule pack's compliance report for the resource's Stack
     * @param params The @IApplyRule interface with rule details.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     */
    writeToStackComplianceReport(params, ruleId, compliance, explanation = '') {
        const line = this.createComplianceReportLine(params, ruleId, compliance, explanation);
        const outDir = aws_cdk_lib_1.App.of(params.node)?.outdir;
        const stackName = params.node.stack.nested
            ? aws_cdk_lib_1.Names.uniqueId(params.node.stack)
            : params.node.stack.stackName;
        const fileName = `${this.packName}-${stackName}-NagReport.csv`;
        const filePath = path_1.join(outDir ? outDir : '', fileName);
        fs_1.appendFileSync(filePath, line);
    }
    /**
     * Initialize the report for the rule pack's compliance report for the resource's Stack if it doesn't exist
     * @param params
     */
    initializeStackReport(params) {
        const stackName = params.node.stack.nested
            ? aws_cdk_lib_1.Names.uniqueId(params.node.stack)
            : params.node.stack.stackName;
        const fileName = `${this.packName}-${stackName}-NagReport.csv`;
        if (!this.reportStacks.includes(fileName)) {
            const outDir = aws_cdk_lib_1.App.of(params.node)?.outdir;
            const filePath = path_1.join(outDir ? outDir : '', fileName);
            this.reportStacks.push(fileName);
            fs_1.writeFileSync(filePath, 'Rule ID,Resource ID,Compliance,Exception Reason,Rule Level,Rule Info\n');
        }
    }
    /**
     * Helper function to create a line for the compliance report
     * @param params The @IApplyRule interface with rule details.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     */
    createComplianceReportLine(params, ruleId, compliance, explanation = '') {
        //| Rule ID | Resource ID | Compliance | Exception Reason | Rule Level | Rule Info
        const line = Array();
        line.push(ruleId);
        line.push(params.node.node.path);
        if ((compliance === nag_rules_1.NagRuleCompliance.NON_COMPLIANT ||
            compliance === 'UNKNOWN') &&
            explanation !== '') {
            line.push('Suppressed');
            line.push(explanation);
        }
        else {
            line.push(compliance);
            line.push('N/A');
        }
        line.push(params.level);
        line.push(params.info);
        return line.map((i) => '"' + i.replace(/"/g, '""') + '"').join(',') + '\n';
    }
    isNonCompliant(ruleResult) {
        return (ruleResult === nag_rules_1.NagRuleCompliance.NON_COMPLIANT ||
            Array.isArray(ruleResult));
    }
    asFindings(ruleResult) {
        if (Array.isArray(ruleResult)) {
            return ruleResult;
        }
        else {
            return [''];
        }
    }
}
exports.NagPack = NagPack;
_a = JSII_RTTI_SYMBOL_1;
NagPack[_a] = { fqn: "cdk-nag.NagPack", version: "2.23.4" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibmFnLXBhY2suanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvbmFnLXBhY2sudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTs7O0VBR0U7QUFDRiwyQkFBbUQ7QUFDbkQsK0JBQTRCO0FBQzVCLDZDQUE0RTtBQUU1RSxtRkFJeUM7QUFFekMsMkNBQWdGO0FBQ2hGLDJFQUFzRTtBQUV0RSxNQUFNLHFCQUFxQixHQUFHLHlCQUF5QixDQUFDO0FBQ3hELE1BQU0sY0FBYyxHQUFHLG1CQUFtQixDQUFDO0FBOEQzQzs7R0FFRztBQUNILElBQVksZUFHWDtBQUhELFdBQVksZUFBZTtJQUN6QixtQ0FBZ0IsQ0FBQTtJQUNoQixrQ0FBZSxDQUFBO0FBQ2pCLENBQUMsRUFIVyxlQUFlLEdBQWYsdUJBQWUsS0FBZix1QkFBZSxRQUcxQjtBQUVEOztHQUVHO0FBQ0gsTUFBc0IsT0FBTztJQVMzQixZQUFZLEtBQW9CO1FBTHRCLGlCQUFZLEdBQUcsSUFBSSxLQUFLLEVBQVUsQ0FBQztRQUNuQyxhQUFRLEdBQUcsRUFBRSxDQUFDO1FBS3RCLElBQUksQ0FBQyxPQUFPO1lBQ1YsS0FBSyxJQUFJLFNBQVMsSUFBSSxLQUFLLENBQUMsT0FBTyxJQUFJLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzNFLElBQUksQ0FBQyxVQUFVO1lBQ2IsS0FBSyxJQUFJLFNBQVMsSUFBSSxLQUFLLENBQUMsVUFBVSxJQUFJLFNBQVM7Z0JBQ2pELENBQUMsQ0FBQyxLQUFLO2dCQUNQLENBQUMsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDO1FBQ3ZCLElBQUksQ0FBQyxPQUFPO1lBQ1YsS0FBSyxJQUFJLFNBQVMsSUFBSSxLQUFLLENBQUMsT0FBTyxJQUFJLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzFFLElBQUksQ0FBQywyQkFBMkIsR0FBRyxLQUFLLEVBQUUsMEJBQTBCLENBQUM7SUFDdkUsQ0FBQztJQUVELElBQVcsWUFBWTtRQUNyQixPQUFPLElBQUksQ0FBQyxRQUFRLENBQUM7SUFDdkIsQ0FBQztJQUNELElBQVcsZ0JBQWdCO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQztJQUMzQixDQUFDO0lBTUQ7OztPQUdHO0lBQ08sU0FBUyxDQUFDLE1BQWtCO1FBQ3BDLElBQUksSUFBSSxDQUFDLFFBQVEsS0FBSyxFQUFFLEVBQUU7WUFDeEIsTUFBTSxLQUFLLENBQ1QsNEhBQTRILENBQzdILENBQUM7U0FDSDtRQUNELE1BQU0sZUFBZSxHQUFHLDZDQUFvQixDQUFDLGVBQWUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDMUUsTUFBTSxVQUFVLEdBQUcsTUFBTSxDQUFDLGtCQUFrQjtZQUMxQyxDQUFDLENBQUMsTUFBTSxDQUFDLGtCQUFrQjtZQUMzQixDQUFDLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7UUFDckIsTUFBTSxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsUUFBUSxJQUFJLFVBQVUsRUFBRSxDQUFDO1FBQ2hELElBQUk7WUFDRixNQUFNLGNBQWMsR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUNoRCxJQUFJLElBQUksQ0FBQyxPQUFPLEtBQUssSUFBSSxFQUFFO2dCQUN6QixJQUFJLENBQUMscUJBQXFCLENBQUMsTUFBTSxDQUFDLENBQUM7Z0JBQ25DLElBQUksY0FBYyxLQUFLLDZCQUFpQixDQUFDLFNBQVMsRUFBRTtvQkFDbEQsSUFBSSxDQUFDLDRCQUE0QixDQUFDLE1BQU0sRUFBRSxNQUFNLEVBQUUsY0FBYyxDQUFDLENBQUM7aUJBQ25FO2FBQ0Y7WUFDRCxJQUFJLElBQUksQ0FBQyxjQUFjLENBQUMsY0FBYyxDQUFDLEVBQUU7Z0JBQ3ZDLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsY0FBYyxDQUFDLENBQUM7Z0JBQ2pELEtBQUssTUFBTSxTQUFTLElBQUksUUFBUSxFQUFFO29CQUNoQyxNQUFNLGlCQUFpQixHQUFHLElBQUksQ0FBQyxVQUFVLENBQ3ZDLGVBQWUsRUFDZixNQUFNLEVBQ04sU0FBUyxFQUNULE1BQU0sQ0FBQyxJQUFJLEVBQ1gsTUFBTSxDQUFDLEtBQUssRUFDWixNQUFNLENBQUMsMEJBQTBCLENBQ2xDLENBQUM7b0JBRUYsSUFBSSxJQUFJLENBQUMsT0FBTyxLQUFLLElBQUksRUFBRTt3QkFDekIsSUFBSSxDQUFDLDRCQUE0QixDQUMvQixNQUFNLEVBQ04sTUFBTSxFQUNOLDZCQUFpQixDQUFDLGFBQWEsRUFDL0IsaUJBQWlCLENBQ2xCLENBQUM7cUJBQ0g7b0JBRUQsSUFBSSxpQkFBaUIsRUFBRTt3QkFDckIsSUFBSSxJQUFJLENBQUMsVUFBVSxLQUFLLElBQUksRUFBRTs0QkFDNUIsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FDaEMsY0FBYyxFQUNkLFNBQVMsRUFDVCxHQUFHLE1BQU0sZ0NBQWdDLEVBQ3pDLHFCQUFxQixpQkFBaUIsR0FBRyxDQUMxQyxDQUFDOzRCQUNGLHlCQUFXLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7eUJBQzlDO3FCQUNGO3lCQUFNO3dCQUNMLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxhQUFhLENBQ2hDLE1BQU0sRUFDTixTQUFTLEVBQ1QsTUFBTSxDQUFDLElBQUksRUFDWCxNQUFNLENBQUMsV0FBVyxDQUNuQixDQUFDO3dCQUNGLElBQUksTUFBTSxDQUFDLEtBQUssSUFBSSxlQUFlLENBQUMsS0FBSyxFQUFFOzRCQUN6Qyx5QkFBVyxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDO3lCQUMvQzs2QkFBTSxJQUFJLE1BQU0sQ0FBQyxLQUFLLElBQUksZUFBZSxDQUFDLElBQUksRUFBRTs0QkFDL0MseUJBQVcsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQzt5QkFDakQ7cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGO1FBQUMsT0FBTyxLQUFLLEVBQUU7WUFDZCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsVUFBVSxDQUM1QixlQUFlLEVBQ2YscUJBQXFCLEVBQ3JCLEVBQUUsRUFDRixNQUFNLENBQUMsSUFBSSxFQUNYLE1BQU0sQ0FBQyxLQUFLLEVBQ1osTUFBTSxDQUFDLDBCQUEwQixDQUNsQyxDQUFDO1lBQ0YsSUFBSSxJQUFJLENBQUMsT0FBTyxLQUFLLElBQUksRUFBRTtnQkFDekIsSUFBSSxDQUFDLDRCQUE0QixDQUFDLE1BQU0sRUFBRSxNQUFNLEVBQUUsU0FBUyxFQUFFLE1BQU0sQ0FBQyxDQUFDO2FBQ3RFO1lBQ0QsSUFBSSxNQUFNLEVBQUU7Z0JBQ1YsSUFBSSxJQUFJLENBQUMsVUFBVSxLQUFLLElBQUksRUFBRTtvQkFDNUIsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FDaEMsY0FBYyxFQUNkLEVBQUUsRUFDRixHQUFHLHFCQUFxQixnQ0FBZ0MsRUFDeEQsTUFBTSxDQUNQLENBQUM7b0JBQ0YseUJBQVcsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQztpQkFDOUM7YUFDRjtpQkFBTTtnQkFDTCxNQUFNLFdBQVcsR0FBRyxJQUFJLE1BQU0sMEpBQTBKLENBQUM7Z0JBQ3pMLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxhQUFhLENBQ2hDLHFCQUFxQixFQUNyQixFQUFFLEVBQ0YsV0FBVyxFQUNWLEtBQWUsQ0FBQyxPQUFPLENBQ3pCLENBQUM7Z0JBQ0YseUJBQVcsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQzthQUNqRDtTQUNGO0lBQ0gsQ0FBQztJQUVEOzs7Ozs7O09BT0c7SUFDTyxVQUFVLENBQ2xCLFlBQWtDLEVBQ2xDLE1BQWMsRUFDZCxTQUFpQixFQUNqQixRQUFxQixFQUNyQixLQUFzQixFQUN0QiwwQkFBa0Q7UUFFbEQsS0FBSyxJQUFJLFdBQVcsSUFBSSxZQUFZLEVBQUU7WUFDcEMsSUFBSSw2Q0FBb0IsQ0FBQyxTQUFTLENBQUMsV0FBVyxFQUFFLE1BQU0sRUFBRSxTQUFTLENBQUMsRUFBRTtnQkFDbEUsTUFBTSxhQUFhLEdBQUcsSUFBSSxtREFBbUIsQ0FDM0MsSUFBSSxDQUFDLDJCQUEyQixJQUFJLElBQUksc0RBQXNCLEVBQUUsRUFDaEUsSUFBSSxDQUFDLDJCQUEyQixJQUFJLElBQUksc0RBQXNCLEVBQUUsRUFDaEUsMEJBQTBCLElBQUksSUFBSSxzREFBc0IsRUFBRSxDQUMzRCxDQUFDLGFBQWEsQ0FBQztvQkFDZCxRQUFRO29CQUNSLE1BQU0sRUFBRSxXQUFXLENBQUMsTUFBTTtvQkFDMUIsTUFBTTtvQkFDTixTQUFTO29CQUNULFNBQVMsRUFBRSxLQUFLO2lCQUNqQixDQUFDLENBQUM7Z0JBQ0gsSUFBSSxhQUFhLEVBQUU7b0JBQ2pCLElBQUksRUFBRSxHQUFHLFNBQVMsQ0FBQyxDQUFDLENBQUMsR0FBRyxNQUFNLElBQUksU0FBUyxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsTUFBTSxFQUFFLENBQUM7b0JBQzdELHlCQUFXLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxDQUFDLE9BQU8sQ0FDOUIsdUJBQXVCLEVBQUUsZ0RBQWdELGFBQWEsRUFBRSxDQUN6RixDQUFDO2lCQUNIO3FCQUFNO29CQUNMLElBQUksQ0FBQyxXQUFXLENBQUMsU0FBUyxFQUFFO3dCQUMxQixnREFBZ0Q7d0JBQ2hELE9BQU8sV0FBVyxDQUFDLE1BQU0sQ0FBQztxQkFDM0I7eUJBQU07d0JBQ0wsT0FBTyxJQUFJLFNBQVMsS0FBSyxXQUFXLENBQUMsTUFBTSxFQUFFLENBQUM7cUJBQy9DO2lCQUNGO2FBQ0Y7U0FDRjtRQUNELE9BQU8sRUFBRSxDQUFDO0lBQ1osQ0FBQztJQUVEOzs7Ozs7O09BT0c7SUFDTyxhQUFhLENBQ3JCLE1BQWMsRUFDZCxTQUFpQixFQUNqQixJQUFZLEVBQ1osV0FBbUI7UUFFbkIsSUFBSSxPQUFPLEdBQUcsU0FBUztZQUNyQixDQUFDLENBQUMsR0FBRyxNQUFNLElBQUksU0FBUyxNQUFNLElBQUksRUFBRTtZQUNwQyxDQUFDLENBQUMsR0FBRyxNQUFNLEtBQUssSUFBSSxFQUFFLENBQUM7UUFDekIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxHQUFHLE9BQU8sSUFBSSxXQUFXLElBQUksQ0FBQyxDQUFDLENBQUMsR0FBRyxPQUFPLElBQUksQ0FBQztJQUN2RSxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ08sNEJBQTRCLENBQ3BDLE1BQWtCLEVBQ2xCLE1BQWMsRUFDZCxVQUdhLEVBQ2IsY0FBc0IsRUFBRTtRQUV4QixNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsMEJBQTBCLENBQzFDLE1BQU0sRUFDTixNQUFNLEVBQ04sVUFBVSxFQUNWLFdBQVcsQ0FDWixDQUFDO1FBQ0YsTUFBTSxNQUFNLEdBQUcsaUJBQUcsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLE1BQU0sQ0FBQztRQUMzQyxNQUFNLFNBQVMsR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNO1lBQ3hDLENBQUMsQ0FBQyxtQkFBSyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQztZQUNuQyxDQUFDLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBQ2hDLE1BQU0sUUFBUSxHQUFHLEdBQUcsSUFBSSxDQUFDLFFBQVEsSUFBSSxTQUFTLGdCQUFnQixDQUFDO1FBQy9ELE1BQU0sUUFBUSxHQUFHLFdBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3RELG1CQUFjLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFRDs7O09BR0c7SUFDTyxxQkFBcUIsQ0FBQyxNQUFrQjtRQUNoRCxNQUFNLFNBQVMsR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNO1lBQ3hDLENBQUMsQ0FBQyxtQkFBSyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQztZQUNuQyxDQUFDLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBQ2hDLE1BQU0sUUFBUSxHQUFHLEdBQUcsSUFBSSxDQUFDLFFBQVEsSUFBSSxTQUFTLGdCQUFnQixDQUFDO1FBQy9ELElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsRUFBRTtZQUN6QyxNQUFNLE1BQU0sR0FBRyxpQkFBRyxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxDQUFDO1lBQzNDLE1BQU0sUUFBUSxHQUFHLFdBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1lBQ3RELElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQ2pDLGtCQUFhLENBQ1gsUUFBUSxFQUNSLHdFQUF3RSxDQUN6RSxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ08sMEJBQTBCLENBQ2xDLE1BQWtCLEVBQ2xCLE1BQWMsRUFDZCxVQUdhLEVBQ2IsY0FBc0IsRUFBRTtRQUV4QixrRkFBa0Y7UUFDbEYsTUFBTSxJQUFJLEdBQUcsS0FBSyxFQUFVLENBQUM7UUFDN0IsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUNsQixJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2pDLElBQ0UsQ0FBQyxVQUFVLEtBQUssNkJBQWlCLENBQUMsYUFBYTtZQUM3QyxVQUFVLEtBQUssU0FBUyxDQUFDO1lBQzNCLFdBQVcsS0FBSyxFQUFFLEVBQ2xCO1lBQ0EsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztZQUN4QixJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1NBQ3hCO2FBQU07WUFDTCxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1lBQ3RCLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7U0FDbEI7UUFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUN4QixJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2QixPQUFPLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLEdBQUcsR0FBRyxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsR0FBRyxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsSUFBSSxDQUFDO0lBQzdFLENBQUM7SUFFTyxjQUFjLENBQUMsVUFBeUI7UUFDOUMsT0FBTyxDQUNMLFVBQVUsS0FBSyw2QkFBaUIsQ0FBQyxhQUFhO1lBQzlDLEtBQUssQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQzFCLENBQUM7SUFDSixDQUFDO0lBRU8sVUFBVSxDQUFDLFVBQXlCO1FBQzFDLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsRUFBRTtZQUM3QixPQUFPLFVBQVUsQ0FBQztTQUNuQjthQUFNO1lBQ0wsT0FBTyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1NBQ2I7SUFDSCxDQUFDOztBQS9TSCwwQkFnVEMiLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuKi9cbmltcG9ydCB7IGFwcGVuZEZpbGVTeW5jLCB3cml0ZUZpbGVTeW5jIH0gZnJvbSAnZnMnO1xuaW1wb3J0IHsgam9pbiB9IGZyb20gJ3BhdGgnO1xuaW1wb3J0IHsgQW5ub3RhdGlvbnMsIEFwcCwgQ2ZuUmVzb3VyY2UsIElBc3BlY3QsIE5hbWVzIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgSUNvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHtcbiAgSU5hZ1N1cHByZXNzaW9uSWdub3JlLFxuICBTdXBwcmVzc2lvbklnbm9yZU5ldmVyLFxuICBTdXBwcmVzc2lvbklnbm9yZU9yLFxufSBmcm9tICcuL2lnbm9yZS1zdXBwcmVzc2lvbi1jb25kaXRpb25zJztcbmltcG9ydCB7IE5hZ1BhY2tTdXBwcmVzc2lvbiB9IGZyb20gJy4vbW9kZWxzL25hZy1zdXBwcmVzc2lvbic7XG5pbXBvcnQgeyBOYWdSdWxlQ29tcGxpYW5jZSwgTmFnUnVsZUZpbmRpbmdzLCBOYWdSdWxlUmVzdWx0IH0gZnJvbSAnLi9uYWctcnVsZXMnO1xuaW1wb3J0IHsgTmFnU3VwcHJlc3Npb25IZWxwZXIgfSBmcm9tICcuL3V0aWxzL25hZy1zdXBwcmVzc2lvbi1oZWxwZXInO1xuXG5jb25zdCBWQUxJREFUSU9OX0ZBSUxVUkVfSUQgPSAnQ2RrTmFnVmFsaWRhdGlvbkZhaWx1cmUnO1xuY29uc3QgU1VQUFJFU1NJT05fSUQgPSAnQ2RrTmFnU3VwcHJlc3Npb24nO1xuXG4vKipcbiAqIEludGVyZmFjZSBmb3IgY3JlYXRpbmcgYSBOYWcgcnVsZSBwYWNrLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIE5hZ1BhY2tQcm9wcyB7XG4gIC8qKlxuICAgKiBXaGV0aGVyIG9yIG5vdCB0byBlbmFibGUgZXh0ZW5kZWQgZXhwbGFuYXRvcnkgZGVzY3JpcHRpb25zIG9uIHdhcm5pbmcsIGVycm9yLCBhbmQgbG9nZ2VkIGlnbm9yZSBtZXNzYWdlcyAoZGVmYXVsdDogZmFsc2UpLlxuICAgKi9cbiAgcmVhZG9ubHkgdmVyYm9zZT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgb3Igbm90IHRvIGxvZyB0cmlnZ2VyZWQgcnVsZXMgdGhhdCBoYXZlIGJlZW4gc3VwcHJlc3NlZCBhcyBpbmZvcm1hdGlvbmFsIG1lc3NhZ2VzIChkZWZhdWx0OiBmYWxzZSkuXG4gICAqL1xuICByZWFkb25seSBsb2dJZ25vcmVzPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdG8gZ2VuZXJhdGUgQ1NWIGNvbXBsaWFuY2UgcmVwb3J0cyBmb3IgYXBwbGllZCBTdGFja3MgaW4gdGhlIEFwcCdzIG91dHB1dCBkaXJlY3RvcnkgKGRlZmF1bHQ6IHRydWUpLlxuICAgKi9cbiAgcmVhZG9ubHkgcmVwb3J0cz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIENvbmRpdGlvbmFsbHkgcHJldmVudCBydWxlcyBmcm9tIGJlaW5nIHN1cHByZXNzZWQgKGRlZmF1bHQ6IG5vIHVzZXIgcHJvdmlkZWQgY29uZGl0aW9uKS5cbiAgICovXG4gIHJlYWRvbmx5IHN1cHByZXNzaW9uSWdub3JlQ29uZGl0aW9uPzogSU5hZ1N1cHByZXNzaW9uSWdub3JlO1xufVxuXG4vKipcbiAqIEludGVyZmFjZSBmb3IgSlNJSSBpbnRlcm9wZXJhYmlsaXR5IGZvciBwYXNzaW5nIHBhcmFtZXRlcnMgYW5kIHRoZSBSdWxlIENhbGxiYWNrIHRvIEBhcHBseVJ1bGUgbWV0aG9kLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElBcHBseVJ1bGUge1xuICAvKipcbiAgICogT3ZlcnJpZGUgZm9yIHRoZSBzdWZmaXggb2YgdGhlIFJ1bGUgSUQgZm9yIHRoaXMgcnVsZVxuICAgKi9cbiAgcnVsZVN1ZmZpeE92ZXJyaWRlPzogc3RyaW5nO1xuICAvKipcbiAgICogV2h5IHRoZSBydWxlIHdhcyB0cmlnZ2VyZWQuXG4gICAqL1xuICBpbmZvOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBXaHkgdGhlIHJ1bGUgZXhpc3RzLlxuICAgKi9cbiAgZXhwbGFuYXRpb246IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBhbm5vdGF0aW9ucyBtZXNzYWdlIGxldmVsIHRvIGFwcGx5IHRvIHRoZSBydWxlIGlmIHRyaWdnZXJlZC5cbiAgICovXG4gIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWw7XG4gIC8qKlxuICAgKiBBIGNvbmRpdGlvbiBpbiB3aGljaCBhIHN1cHByZXNzaW9uIHNob3VsZCBiZSBpZ25vcmVkLlxuICAgKi9cbiAgaWdub3JlU3VwcHJlc3Npb25Db25kaXRpb24/OiBJTmFnU3VwcHJlc3Npb25JZ25vcmU7XG4gIC8qKlxuICAgKiBUaGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICovXG4gIG5vZGU6IENmblJlc291cmNlO1xuICAvKipcbiAgICogVGhlIGNhbGxiYWNrIHRvIHRoZSBydWxlLlxuICAgKiBAcGFyYW0gbm9kZSBUaGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2suXG4gICAqL1xuICBydWxlKG5vZGU6IENmblJlc291cmNlKTogTmFnUnVsZVJlc3VsdDtcbn1cblxuLyoqXG4gKiBUaGUgc2V2ZXJpdHkgbGV2ZWwgb2YgdGhlIHJ1bGUuXG4gKi9cbmV4cG9ydCBlbnVtIE5hZ01lc3NhZ2VMZXZlbCB7XG4gIFdBUk4gPSAnV2FybmluZycsXG4gIEVSUk9SID0gJ0Vycm9yJyxcbn1cblxuLyoqXG4gKiBCYXNlIGNsYXNzIGZvciBhbGwgcnVsZSBwYWNrcy5cbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIE5hZ1BhY2sgaW1wbGVtZW50cyBJQXNwZWN0IHtcbiAgcHJvdGVjdGVkIHZlcmJvc2U6IGJvb2xlYW47XG4gIHByb3RlY3RlZCBsb2dJZ25vcmVzOiBib29sZWFuO1xuICBwcm90ZWN0ZWQgcmVwb3J0czogYm9vbGVhbjtcbiAgcHJvdGVjdGVkIHJlcG9ydFN0YWNrcyA9IG5ldyBBcnJheTxzdHJpbmc+KCk7XG4gIHByb3RlY3RlZCBwYWNrTmFtZSA9ICcnO1xuICBwcm90ZWN0ZWQgdXNlckdsb2JhbFN1cHByZXNzaW9uSWdub3JlPzogSU5hZ1N1cHByZXNzaW9uSWdub3JlO1xuICBwcm90ZWN0ZWQgcGFja0dsb2JhbFN1cHByZXNzaW9uSWdub3JlPzogSU5hZ1N1cHByZXNzaW9uSWdub3JlO1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzPzogTmFnUGFja1Byb3BzKSB7XG4gICAgdGhpcy52ZXJib3NlID1cbiAgICAgIHByb3BzID09IHVuZGVmaW5lZCB8fCBwcm9wcy52ZXJib3NlID09IHVuZGVmaW5lZCA/IGZhbHNlIDogcHJvcHMudmVyYm9zZTtcbiAgICB0aGlzLmxvZ0lnbm9yZXMgPVxuICAgICAgcHJvcHMgPT0gdW5kZWZpbmVkIHx8IHByb3BzLmxvZ0lnbm9yZXMgPT0gdW5kZWZpbmVkXG4gICAgICAgID8gZmFsc2VcbiAgICAgICAgOiBwcm9wcy5sb2dJZ25vcmVzO1xuICAgIHRoaXMucmVwb3J0cyA9XG4gICAgICBwcm9wcyA9PSB1bmRlZmluZWQgfHwgcHJvcHMucmVwb3J0cyA9PSB1bmRlZmluZWQgPyB0cnVlIDogcHJvcHMucmVwb3J0cztcbiAgICB0aGlzLnVzZXJHbG9iYWxTdXBwcmVzc2lvbklnbm9yZSA9IHByb3BzPy5zdXBwcmVzc2lvbklnbm9yZUNvbmRpdGlvbjtcbiAgfVxuXG4gIHB1YmxpYyBnZXQgcmVhZFBhY2tOYW1lKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHRoaXMucGFja05hbWU7XG4gIH1cbiAgcHVibGljIGdldCByZWFkUmVwb3J0U3RhY2tzKCk6IHN0cmluZ1tdIHtcbiAgICByZXR1cm4gdGhpcy5yZXBvcnRTdGFja3M7XG4gIH1cbiAgLyoqXG4gICAqIEFsbCBhc3BlY3RzIGNhbiB2aXNpdCBhbiBJQ29uc3RydWN0LlxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHZpc2l0KG5vZGU6IElDb25zdHJ1Y3QpOiB2b2lkO1xuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBydWxlIHRvIGJlIHVzZWQgaW4gdGhlIE5hZ1BhY2suXG4gICAqIEBwYXJhbSBwYXJhbXMgVGhlIEBJQXBwbHlSdWxlIGludGVyZmFjZSB3aXRoIHJ1bGUgZGV0YWlscy5cbiAgICovXG4gIHByb3RlY3RlZCBhcHBseVJ1bGUocGFyYW1zOiBJQXBwbHlSdWxlKTogdm9pZCB7XG4gICAgaWYgKHRoaXMucGFja05hbWUgPT09ICcnKSB7XG4gICAgICB0aHJvdyBFcnJvcihcbiAgICAgICAgJ1RoZSBOYWdQYWNrIGRvZXMgbm90IGhhdmUgYSBwYWNrIG5hbWUsIHRoZXJlZm9yZSB0aGUgcnVsZSBjb3VsZCBub3QgYmUgYXBwbGllZC4gU2V0IGEgcGFja05hbWUgaW4gdGhlIE5hZ1BhY2sgY29uc3RydWN0b3IuJ1xuICAgICAgKTtcbiAgICB9XG4gICAgY29uc3QgYWxsU3VwcHJlc3Npb25zID0gTmFnU3VwcHJlc3Npb25IZWxwZXIuZ2V0U3VwcHJlc3Npb25zKHBhcmFtcy5ub2RlKTtcbiAgICBjb25zdCBydWxlU3VmZml4ID0gcGFyYW1zLnJ1bGVTdWZmaXhPdmVycmlkZVxuICAgICAgPyBwYXJhbXMucnVsZVN1ZmZpeE92ZXJyaWRlXG4gICAgICA6IHBhcmFtcy5ydWxlLm5hbWU7XG4gICAgY29uc3QgcnVsZUlkID0gYCR7dGhpcy5wYWNrTmFtZX0tJHtydWxlU3VmZml4fWA7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IHJ1bGVDb21wbGlhbmNlID0gcGFyYW1zLnJ1bGUocGFyYW1zLm5vZGUpO1xuICAgICAgaWYgKHRoaXMucmVwb3J0cyA9PT0gdHJ1ZSkge1xuICAgICAgICB0aGlzLmluaXRpYWxpemVTdGFja1JlcG9ydChwYXJhbXMpO1xuICAgICAgICBpZiAocnVsZUNvbXBsaWFuY2UgPT09IE5hZ1J1bGVDb21wbGlhbmNlLkNPTVBMSUFOVCkge1xuICAgICAgICAgIHRoaXMud3JpdGVUb1N0YWNrQ29tcGxpYW5jZVJlcG9ydChwYXJhbXMsIHJ1bGVJZCwgcnVsZUNvbXBsaWFuY2UpO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgICBpZiAodGhpcy5pc05vbkNvbXBsaWFudChydWxlQ29tcGxpYW5jZSkpIHtcbiAgICAgICAgY29uc3QgZmluZGluZ3MgPSB0aGlzLmFzRmluZGluZ3MocnVsZUNvbXBsaWFuY2UpO1xuICAgICAgICBmb3IgKGNvbnN0IGZpbmRpbmdJZCBvZiBmaW5kaW5ncykge1xuICAgICAgICAgIGNvbnN0IHN1cHByZXNzaW9uUmVhc29uID0gdGhpcy5pZ25vcmVSdWxlKFxuICAgICAgICAgICAgYWxsU3VwcHJlc3Npb25zLFxuICAgICAgICAgICAgcnVsZUlkLFxuICAgICAgICAgICAgZmluZGluZ0lkLFxuICAgICAgICAgICAgcGFyYW1zLm5vZGUsXG4gICAgICAgICAgICBwYXJhbXMubGV2ZWwsXG4gICAgICAgICAgICBwYXJhbXMuaWdub3JlU3VwcHJlc3Npb25Db25kaXRpb25cbiAgICAgICAgICApO1xuXG4gICAgICAgICAgaWYgKHRoaXMucmVwb3J0cyA9PT0gdHJ1ZSkge1xuICAgICAgICAgICAgdGhpcy53cml0ZVRvU3RhY2tDb21wbGlhbmNlUmVwb3J0KFxuICAgICAgICAgICAgICBwYXJhbXMsXG4gICAgICAgICAgICAgIHJ1bGVJZCxcbiAgICAgICAgICAgICAgTmFnUnVsZUNvbXBsaWFuY2UuTk9OX0NPTVBMSUFOVCxcbiAgICAgICAgICAgICAgc3VwcHJlc3Npb25SZWFzb25cbiAgICAgICAgICAgICk7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgaWYgKHN1cHByZXNzaW9uUmVhc29uKSB7XG4gICAgICAgICAgICBpZiAodGhpcy5sb2dJZ25vcmVzID09PSB0cnVlKSB7XG4gICAgICAgICAgICAgIGNvbnN0IG1lc3NhZ2UgPSB0aGlzLmNyZWF0ZU1lc3NhZ2UoXG4gICAgICAgICAgICAgICAgU1VQUFJFU1NJT05fSUQsXG4gICAgICAgICAgICAgICAgZmluZGluZ0lkLFxuICAgICAgICAgICAgICAgIGAke3J1bGVJZH0gd2FzIHRyaWdnZXJlZCBidXQgc3VwcHJlc3NlZC5gLFxuICAgICAgICAgICAgICAgIGBQcm92aWRlZCByZWFzb246IFwiJHtzdXBwcmVzc2lvblJlYXNvbn1cImBcbiAgICAgICAgICAgICAgKTtcbiAgICAgICAgICAgICAgQW5ub3RhdGlvbnMub2YocGFyYW1zLm5vZGUpLmFkZEluZm8obWVzc2FnZSk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIGNvbnN0IG1lc3NhZ2UgPSB0aGlzLmNyZWF0ZU1lc3NhZ2UoXG4gICAgICAgICAgICAgIHJ1bGVJZCxcbiAgICAgICAgICAgICAgZmluZGluZ0lkLFxuICAgICAgICAgICAgICBwYXJhbXMuaW5mbyxcbiAgICAgICAgICAgICAgcGFyYW1zLmV4cGxhbmF0aW9uXG4gICAgICAgICAgICApO1xuICAgICAgICAgICAgaWYgKHBhcmFtcy5sZXZlbCA9PSBOYWdNZXNzYWdlTGV2ZWwuRVJST1IpIHtcbiAgICAgICAgICAgICAgQW5ub3RhdGlvbnMub2YocGFyYW1zLm5vZGUpLmFkZEVycm9yKG1lc3NhZ2UpO1xuICAgICAgICAgICAgfSBlbHNlIGlmIChwYXJhbXMubGV2ZWwgPT0gTmFnTWVzc2FnZUxldmVsLldBUk4pIHtcbiAgICAgICAgICAgICAgQW5ub3RhdGlvbnMub2YocGFyYW1zLm5vZGUpLmFkZFdhcm5pbmcobWVzc2FnZSk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSBjYXRjaCAoZXJyb3IpIHtcbiAgICAgIGNvbnN0IHJlYXNvbiA9IHRoaXMuaWdub3JlUnVsZShcbiAgICAgICAgYWxsU3VwcHJlc3Npb25zLFxuICAgICAgICBWQUxJREFUSU9OX0ZBSUxVUkVfSUQsXG4gICAgICAgICcnLFxuICAgICAgICBwYXJhbXMubm9kZSxcbiAgICAgICAgcGFyYW1zLmxldmVsLFxuICAgICAgICBwYXJhbXMuaWdub3JlU3VwcHJlc3Npb25Db25kaXRpb25cbiAgICAgICk7XG4gICAgICBpZiAodGhpcy5yZXBvcnRzID09PSB0cnVlKSB7XG4gICAgICAgIHRoaXMud3JpdGVUb1N0YWNrQ29tcGxpYW5jZVJlcG9ydChwYXJhbXMsIHJ1bGVJZCwgJ1VOS05PV04nLCByZWFzb24pO1xuICAgICAgfVxuICAgICAgaWYgKHJlYXNvbikge1xuICAgICAgICBpZiAodGhpcy5sb2dJZ25vcmVzID09PSB0cnVlKSB7XG4gICAgICAgICAgY29uc3QgbWVzc2FnZSA9IHRoaXMuY3JlYXRlTWVzc2FnZShcbiAgICAgICAgICAgIFNVUFBSRVNTSU9OX0lELFxuICAgICAgICAgICAgJycsXG4gICAgICAgICAgICBgJHtWQUxJREFUSU9OX0ZBSUxVUkVfSUR9IHdhcyB0cmlnZ2VyZWQgYnV0IHN1cHByZXNzZWQuYCxcbiAgICAgICAgICAgIHJlYXNvblxuICAgICAgICAgICk7XG4gICAgICAgICAgQW5ub3RhdGlvbnMub2YocGFyYW1zLm5vZGUpLmFkZEluZm8obWVzc2FnZSk7XG4gICAgICAgIH1cbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIGNvbnN0IGluZm9ybWF0aW9uID0gYCcke3J1bGVJZH0nIHRocmV3IGFuIGVycm9yIGR1cmluZyB2YWxpZGF0aW9uLiBUaGlzIGlzIGdlbmVyYWxseSBjYXVzZWQgYnkgYSBwYXJhbWV0ZXIgcmVmZXJlbmNpbmcgYW4gaW50cmluc2ljIGZ1bmN0aW9uLiBGb3IgbW9yZSBkZXRhaWxzIGVuYWJsZSB2ZXJib3NlIGxvZ2dpbmcuJ2A7XG4gICAgICAgIGNvbnN0IG1lc3NhZ2UgPSB0aGlzLmNyZWF0ZU1lc3NhZ2UoXG4gICAgICAgICAgVkFMSURBVElPTl9GQUlMVVJFX0lELFxuICAgICAgICAgICcnLFxuICAgICAgICAgIGluZm9ybWF0aW9uLFxuICAgICAgICAgIChlcnJvciBhcyBFcnJvcikubWVzc2FnZVxuICAgICAgICApO1xuICAgICAgICBBbm5vdGF0aW9ucy5vZihwYXJhbXMubm9kZSkuYWRkV2FybmluZyhtZXNzYWdlKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgd2hldGhlciBhIHNwZWNpZmljIHJ1bGUgc2hvdWxkIGJlIGlnbm9yZWQuXG4gICAqIEBwYXJhbSBzdXBwcmVzc2lvbnMgVGhlIHN1cHByZXNzaW9ucyBsaXN0ZWQgaW4gdGhlIGNkay1uYWcgbWV0YWRhdGEuXG4gICAqIEBwYXJhbSBydWxlSWQgVGhlIGlkIG9mIHRoZSBydWxlIHRvIGlnbm9yZS5cbiAgICogQHBhcmFtIHJlc291cmNlIFRoZSByZXNvdXJjZSBiZWluZyBldmFsdWF0ZWQuXG4gICAqIEBwYXJhbSBmaW5kaW5nSWQgVGhlIGlkIG9mIHRoZSBmaW5kaW5nIHRoYXQgaXMgYmVpbmcgY2hlY2tlZC5cbiAgICogQHJldHVybnMgVGhlIHJlYXNvbiB0aGUgcnVsZSB3YXMgaWdub3JlZCwgb3IgYW4gZW1wdHkgc3RyaW5nLlxuICAgKi9cbiAgcHJvdGVjdGVkIGlnbm9yZVJ1bGUoXG4gICAgc3VwcHJlc3Npb25zOiBOYWdQYWNrU3VwcHJlc3Npb25bXSxcbiAgICBydWxlSWQ6IHN0cmluZyxcbiAgICBmaW5kaW5nSWQ6IHN0cmluZyxcbiAgICByZXNvdXJjZTogQ2ZuUmVzb3VyY2UsXG4gICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbCxcbiAgICBpZ25vcmVTdXBwcmVzc2lvbkNvbmRpdGlvbj86IElOYWdTdXBwcmVzc2lvbklnbm9yZVxuICApOiBzdHJpbmcge1xuICAgIGZvciAobGV0IHN1cHByZXNzaW9uIG9mIHN1cHByZXNzaW9ucykge1xuICAgICAgaWYgKE5hZ1N1cHByZXNzaW9uSGVscGVyLmRvZXNBcHBseShzdXBwcmVzc2lvbiwgcnVsZUlkLCBmaW5kaW5nSWQpKSB7XG4gICAgICAgIGNvbnN0IGlnbm9yZU1lc3NhZ2UgPSBuZXcgU3VwcHJlc3Npb25JZ25vcmVPcihcbiAgICAgICAgICB0aGlzLnVzZXJHbG9iYWxTdXBwcmVzc2lvbklnbm9yZSA/PyBuZXcgU3VwcHJlc3Npb25JZ25vcmVOZXZlcigpLFxuICAgICAgICAgIHRoaXMucGFja0dsb2JhbFN1cHByZXNzaW9uSWdub3JlID8/IG5ldyBTdXBwcmVzc2lvbklnbm9yZU5ldmVyKCksXG4gICAgICAgICAgaWdub3JlU3VwcHJlc3Npb25Db25kaXRpb24gPz8gbmV3IFN1cHByZXNzaW9uSWdub3JlTmV2ZXIoKVxuICAgICAgICApLmNyZWF0ZU1lc3NhZ2Uoe1xuICAgICAgICAgIHJlc291cmNlLFxuICAgICAgICAgIHJlYXNvbjogc3VwcHJlc3Npb24ucmVhc29uLFxuICAgICAgICAgIHJ1bGVJZCxcbiAgICAgICAgICBmaW5kaW5nSWQsXG4gICAgICAgICAgcnVsZUxldmVsOiBsZXZlbCxcbiAgICAgICAgfSk7XG4gICAgICAgIGlmIChpZ25vcmVNZXNzYWdlKSB7XG4gICAgICAgICAgbGV0IGlkID0gZmluZGluZ0lkID8gYCR7cnVsZUlkfVske2ZpbmRpbmdJZH1dYCA6IGAke3J1bGVJZH1gO1xuICAgICAgICAgIEFubm90YXRpb25zLm9mKHJlc291cmNlKS5hZGRJbmZvKFxuICAgICAgICAgICAgYFRoZSBzdXBwcmVzc2lvbiBmb3IgJHtpZH0gd2FzIGlnbm9yZWQgZm9yIHRoZSBmb2xsb3dpbmcgcmVhc29uKHMpLlxcblxcdCR7aWdub3JlTWVzc2FnZX1gXG4gICAgICAgICAgKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBpZiAoIXN1cHByZXNzaW9uLmFwcGxpZXNUbykge1xuICAgICAgICAgICAgLy8gdGhlIHJ1bGUgaXMgbm90IGdyYW51bGFyIHNvIGl0IGFsd2F5cyBhcHBsaWVzXG4gICAgICAgICAgICByZXR1cm4gc3VwcHJlc3Npb24ucmVhc29uO1xuICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICByZXR1cm4gYFske2ZpbmRpbmdJZH1dICR7c3VwcHJlc3Npb24ucmVhc29ufWA7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICAgIHJldHVybiAnJztcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgbWVzc2FnZSB0byBvdXRwdXQgdG8gdGhlIGNvbnNvbGUgd2hlbiBhIHJ1bGUgaXMgdHJpZ2dlcmVkLlxuICAgKiBAcGFyYW0gcnVsZUlkIFRoZSBpZCBvZiB0aGUgcnVsZS5cbiAgICogQHBhcmFtIGZpbmRpbmdJZCBUaGUgaWQgb2YgdGhlIGZpbmRpbmcuXG4gICAqIEBwYXJhbSBpbmZvIFdoeSB0aGUgcnVsZSB3YXMgdHJpZ2dlcmVkLlxuICAgKiBAcGFyYW0gZXhwbGFuYXRpb24gV2h5IHRoZSBydWxlIGV4aXN0cy5cbiAgICogQHJldHVybnMgVGhlIGZvcm1hdHRlZCBtZXNzYWdlIHN0cmluZy5cbiAgICovXG4gIHByb3RlY3RlZCBjcmVhdGVNZXNzYWdlKFxuICAgIHJ1bGVJZDogc3RyaW5nLFxuICAgIGZpbmRpbmdJZDogc3RyaW5nLFxuICAgIGluZm86IHN0cmluZyxcbiAgICBleHBsYW5hdGlvbjogc3RyaW5nXG4gICk6IHN0cmluZyB7XG4gICAgbGV0IG1lc3NhZ2UgPSBmaW5kaW5nSWRcbiAgICAgID8gYCR7cnVsZUlkfVske2ZpbmRpbmdJZH1dOiAke2luZm99YFxuICAgICAgOiBgJHtydWxlSWR9OiAke2luZm99YDtcbiAgICByZXR1cm4gdGhpcy52ZXJib3NlID8gYCR7bWVzc2FnZX0gJHtleHBsYW5hdGlvbn1cXG5gIDogYCR7bWVzc2FnZX1cXG5gO1xuICB9XG5cbiAgLyoqXG4gICAqIFdyaXRlIGEgbGluZSB0byB0aGUgcnVsZSBwYWNrJ3MgY29tcGxpYW5jZSByZXBvcnQgZm9yIHRoZSByZXNvdXJjZSdzIFN0YWNrXG4gICAqIEBwYXJhbSBwYXJhbXMgVGhlIEBJQXBwbHlSdWxlIGludGVyZmFjZSB3aXRoIHJ1bGUgZGV0YWlscy5cbiAgICogQHBhcmFtIHJ1bGVJZCBUaGUgaWQgb2YgdGhlIHJ1bGUuXG4gICAqIEBwYXJhbSBjb21wbGlhbmNlIFRoZSBjb21wbGlhbmNlIHN0YXR1cyBvZiB0aGUgcnVsZS5cbiAgICogQHBhcmFtIGV4cGxhbmF0aW9uIFRoZSBleHBsYW5hdGlvbiBmb3Igc3VwcHJlc3NlZCBydWxlcy5cbiAgICovXG4gIHByb3RlY3RlZCB3cml0ZVRvU3RhY2tDb21wbGlhbmNlUmVwb3J0KFxuICAgIHBhcmFtczogSUFwcGx5UnVsZSxcbiAgICBydWxlSWQ6IHN0cmluZyxcbiAgICBjb21wbGlhbmNlOlxuICAgICAgfCBOYWdSdWxlQ29tcGxpYW5jZS5DT01QTElBTlRcbiAgICAgIHwgTmFnUnVsZUNvbXBsaWFuY2UuTk9OX0NPTVBMSUFOVFxuICAgICAgfCAnVU5LTk9XTicsXG4gICAgZXhwbGFuYXRpb246IHN0cmluZyA9ICcnXG4gICk6IHZvaWQge1xuICAgIGNvbnN0IGxpbmUgPSB0aGlzLmNyZWF0ZUNvbXBsaWFuY2VSZXBvcnRMaW5lKFxuICAgICAgcGFyYW1zLFxuICAgICAgcnVsZUlkLFxuICAgICAgY29tcGxpYW5jZSxcbiAgICAgIGV4cGxhbmF0aW9uXG4gICAgKTtcbiAgICBjb25zdCBvdXREaXIgPSBBcHAub2YocGFyYW1zLm5vZGUpPy5vdXRkaXI7XG4gICAgY29uc3Qgc3RhY2tOYW1lID0gcGFyYW1zLm5vZGUuc3RhY2submVzdGVkXG4gICAgICA/IE5hbWVzLnVuaXF1ZUlkKHBhcmFtcy5ub2RlLnN0YWNrKVxuICAgICAgOiBwYXJhbXMubm9kZS5zdGFjay5zdGFja05hbWU7XG4gICAgY29uc3QgZmlsZU5hbWUgPSBgJHt0aGlzLnBhY2tOYW1lfS0ke3N0YWNrTmFtZX0tTmFnUmVwb3J0LmNzdmA7XG4gICAgY29uc3QgZmlsZVBhdGggPSBqb2luKG91dERpciA/IG91dERpciA6ICcnLCBmaWxlTmFtZSk7XG4gICAgYXBwZW5kRmlsZVN5bmMoZmlsZVBhdGgsIGxpbmUpO1xuICB9XG5cbiAgLyoqXG4gICAqIEluaXRpYWxpemUgdGhlIHJlcG9ydCBmb3IgdGhlIHJ1bGUgcGFjaydzIGNvbXBsaWFuY2UgcmVwb3J0IGZvciB0aGUgcmVzb3VyY2UncyBTdGFjayBpZiBpdCBkb2Vzbid0IGV4aXN0XG4gICAqIEBwYXJhbSBwYXJhbXNcbiAgICovXG4gIHByb3RlY3RlZCBpbml0aWFsaXplU3RhY2tSZXBvcnQocGFyYW1zOiBJQXBwbHlSdWxlKTogdm9pZCB7XG4gICAgY29uc3Qgc3RhY2tOYW1lID0gcGFyYW1zLm5vZGUuc3RhY2submVzdGVkXG4gICAgICA/IE5hbWVzLnVuaXF1ZUlkKHBhcmFtcy5ub2RlLnN0YWNrKVxuICAgICAgOiBwYXJhbXMubm9kZS5zdGFjay5zdGFja05hbWU7XG4gICAgY29uc3QgZmlsZU5hbWUgPSBgJHt0aGlzLnBhY2tOYW1lfS0ke3N0YWNrTmFtZX0tTmFnUmVwb3J0LmNzdmA7XG4gICAgaWYgKCF0aGlzLnJlcG9ydFN0YWNrcy5pbmNsdWRlcyhmaWxlTmFtZSkpIHtcbiAgICAgIGNvbnN0IG91dERpciA9IEFwcC5vZihwYXJhbXMubm9kZSk/Lm91dGRpcjtcbiAgICAgIGNvbnN0IGZpbGVQYXRoID0gam9pbihvdXREaXIgPyBvdXREaXIgOiAnJywgZmlsZU5hbWUpO1xuICAgICAgdGhpcy5yZXBvcnRTdGFja3MucHVzaChmaWxlTmFtZSk7XG4gICAgICB3cml0ZUZpbGVTeW5jKFxuICAgICAgICBmaWxlUGF0aCxcbiAgICAgICAgJ1J1bGUgSUQsUmVzb3VyY2UgSUQsQ29tcGxpYW5jZSxFeGNlcHRpb24gUmVhc29uLFJ1bGUgTGV2ZWwsUnVsZSBJbmZvXFxuJ1xuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogSGVscGVyIGZ1bmN0aW9uIHRvIGNyZWF0ZSBhIGxpbmUgZm9yIHRoZSBjb21wbGlhbmNlIHJlcG9ydFxuICAgKiBAcGFyYW0gcGFyYW1zIFRoZSBASUFwcGx5UnVsZSBpbnRlcmZhY2Ugd2l0aCBydWxlIGRldGFpbHMuXG4gICAqIEBwYXJhbSBydWxlSWQgVGhlIGlkIG9mIHRoZSBydWxlLlxuICAgKiBAcGFyYW0gY29tcGxpYW5jZSBUaGUgY29tcGxpYW5jZSBzdGF0dXMgb2YgdGhlIHJ1bGUuXG4gICAqIEBwYXJhbSBleHBsYW5hdGlvbiBUaGUgZXhwbGFuYXRpb24gZm9yIHN1cHByZXNzZWQgcnVsZXMuXG4gICAqL1xuICBwcm90ZWN0ZWQgY3JlYXRlQ29tcGxpYW5jZVJlcG9ydExpbmUoXG4gICAgcGFyYW1zOiBJQXBwbHlSdWxlLFxuICAgIHJ1bGVJZDogc3RyaW5nLFxuICAgIGNvbXBsaWFuY2U6XG4gICAgICB8IE5hZ1J1bGVDb21wbGlhbmNlLkNPTVBMSUFOVFxuICAgICAgfCBOYWdSdWxlQ29tcGxpYW5jZS5OT05fQ09NUExJQU5UXG4gICAgICB8ICdVTktOT1dOJyxcbiAgICBleHBsYW5hdGlvbjogc3RyaW5nID0gJydcbiAgKTogc3RyaW5nIHtcbiAgICAvL3wgUnVsZSBJRCB8IFJlc291cmNlIElEIHwgQ29tcGxpYW5jZSB8IEV4Y2VwdGlvbiBSZWFzb24gfCBSdWxlIExldmVsIHwgUnVsZSBJbmZvXG4gICAgY29uc3QgbGluZSA9IEFycmF5PHN0cmluZz4oKTtcbiAgICBsaW5lLnB1c2gocnVsZUlkKTtcbiAgICBsaW5lLnB1c2gocGFyYW1zLm5vZGUubm9kZS5wYXRoKTtcbiAgICBpZiAoXG4gICAgICAoY29tcGxpYW5jZSA9PT0gTmFnUnVsZUNvbXBsaWFuY2UuTk9OX0NPTVBMSUFOVCB8fFxuICAgICAgICBjb21wbGlhbmNlID09PSAnVU5LTk9XTicpICYmXG4gICAgICBleHBsYW5hdGlvbiAhPT0gJydcbiAgICApIHtcbiAgICAgIGxpbmUucHVzaCgnU3VwcHJlc3NlZCcpO1xuICAgICAgbGluZS5wdXNoKGV4cGxhbmF0aW9uKTtcbiAgICB9IGVsc2Uge1xuICAgICAgbGluZS5wdXNoKGNvbXBsaWFuY2UpO1xuICAgICAgbGluZS5wdXNoKCdOL0EnKTtcbiAgICB9XG4gICAgbGluZS5wdXNoKHBhcmFtcy5sZXZlbCk7XG4gICAgbGluZS5wdXNoKHBhcmFtcy5pbmZvKTtcbiAgICByZXR1cm4gbGluZS5tYXAoKGkpID0+ICdcIicgKyBpLnJlcGxhY2UoL1wiL2csICdcIlwiJykgKyAnXCInKS5qb2luKCcsJykgKyAnXFxuJztcbiAgfVxuXG4gIHByaXZhdGUgaXNOb25Db21wbGlhbnQocnVsZVJlc3VsdDogTmFnUnVsZVJlc3VsdCkge1xuICAgIHJldHVybiAoXG4gICAgICBydWxlUmVzdWx0ID09PSBOYWdSdWxlQ29tcGxpYW5jZS5OT05fQ09NUExJQU5UIHx8XG4gICAgICBBcnJheS5pc0FycmF5KHJ1bGVSZXN1bHQpXG4gICAgKTtcbiAgfVxuXG4gIHByaXZhdGUgYXNGaW5kaW5ncyhydWxlUmVzdWx0OiBOYWdSdWxlUmVzdWx0KTogTmFnUnVsZUZpbmRpbmdzIHtcbiAgICBpZiAoQXJyYXkuaXNBcnJheShydWxlUmVzdWx0KSkge1xuICAgICAgcmV0dXJuIHJ1bGVSZXN1bHQ7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBbJyddO1xuICAgIH1cbiAgfVxufVxuIl19