# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pytest_cache_assert', 'pytest_cache_assert._check_assert']

package_data = \
{'': ['*']}

install_requires = \
['attrs-strict>=0.2',
 'beartype>=0.9.0',
 'dictdiffer>=0.9',
 'pendulum>=2.1.2,<3.0.0',
 'pytest>=5']

entry_points = \
{'pytest11': ['pytest_cache_assert = pytest_cache_assert.plugin']}

setup_kwargs = {
    'name': 'pytest-cache-assert',
    'version': '1.0.0rc0',
    'description': 'Cache assertion data to simplify regression testing of complex serializable data',
    'long_description': '# pytest_cache_assert\n\nCache assertion data to simplify regression testing of complex serializable data\n\n## Installation\n\n`poetry add pytest_assert_check --dev`\n\n## Quick Start\n\nThe primary use case of this package is regression testing of large dictionaries. You may have some parameterized test cases where you need to assert that a resulting dictionary is the same, but you don’t want to manually generate the expected fields and values and couple the test case to the source code. Instead you can cache or “record” the expected serialized data structure, check the expected dictionary into version control, then regenerate on changes\n\nThis package can minimize test case logic, while improving test thoroughness\n\nThis project was heavily inspired by the excellent [pytest-recording](https://github.com/kiwicom/pytest-recording)\n\n### Basic Example\n\nYou\'ve created a new project called `package_a` with one file `package_a/source_file.py` and test `tests/test_file.py`\n\n```py\n"""package_a/source_file.py"""\n\nimport sys\nfrom datetime import datetime\nfrom typing import Any, Dict, List, Optional\n\nfrom beartype import beartype\nfrom pydantic import BaseModel\n\n\nclass User(BaseModel):  # noqa: H601\n    """Example from pydantic documentation."""\n\n    id: int  # noqa: A003,VNE003\n    name = \'John Doe\'\n    signup_ts: Optional[datetime] = None\n    friends: List[int] = []\n\n\n@beartype\ndef create_data(name: str) -> Dict[str, Any]:\n    """Arbitrary function that returns a dictionary.\n\n    This demonstration uses pydantic, but any dictionary can be tested!\n\n    """\n    return User(id=sys.maxsize, name=name).dict()\n```\n\n```py\n"""tests/test_file.py"""\n\nimport pytest\n\nfrom package_a.source_file import create_data\n\n\n@pytest.mark.parametrize(\'name\', [\'Test Name 1\', \'Test Name 2\'])\ndef test_create_data(name, assert_against_cache):\n    """Basic test of create_data()."""\n    result = create_data(name=name)\n\n    # One could manually create the expected dictionary\n    cache = {\'id\': 9223372036854775807, \'signup_ts\': None, \'friends\': [], \'name\': name}\n    assert result == cache\n    # ----------------------------------------------------------------------------------\n    # Or utilize the pytest_cache_assert fixture to compare against the last cached version\n    assert_against_cache(result)\n```\n\n`pytest_cache_assert` will automatically create: `tests/cache-assert/source_file/test_file/test_create_data-[Test Name 1].json` (and `test_create_data[Test Name 2].json`) for each of the parameters when first run by caching the `result`. Below is the example for `test_create_data-[Test Name 1].json`\n\n```json\n{\n  "_info": [\n    {\n      "func_args": {\n        "name": "Test Name 1"\n      },\n      "test_file": "test_readme.py",\n      "test_name": "test_create_data"\n    }\n  ],\n  "_json": {\n    "friends": [],\n    "id": 9223372036854775807,\n    "name": "Test Name 1",\n    "signup_ts": null\n  }\n}\n```\n\nThe cached JSON files must be checked into version control and if needed, can be manually edited or deleted so that they will be regenerated when the test suite is next run\n\n### More Examples\n\nIn your cached dictionary, you may have variable values with more complex logic to verify, such as dates, UUIDs, etc. These can be selectively ignored, matched-if-null, or some other user-specified check:\n\n```py\n"""tests/test_main.py."""\n\nfrom uuid import uuid4\nfrom datetime import datetime, timedelta\n\nfrom pytest_cache_assert import KeyRule, check_suppress, check_type, Wildcards\n\n\ndef test_assert_against_cache_key_rules(assert_against_cache):\n    """Demonstrate use of `key_rules`."""\n    # NOTE: all data passed to assert_against_cache must be JSON Serializable (i.e. from Marshmallow or Pydantic, etc.)\n    now = datetime.now()\n    cached_data = {\'date\': str(now), {\'nested\': {\'uuid\': str(uuid4())}}, {\'ignored\': {\'a\': 1, \'b\': 2}}}\n    test_data = {\'date\': str(now + timedelta(hours=3)), {\'nested\': {\'uuid\': str(uuid4())}}}\n    #\n    key_rules = [\n      # Suppress keys \'ignored.a\' and \'ignored.b\' with the SINGLE wildcard\n      # If there were a \'ignored.c.something\', the Wildcard.RECURSIVE could be used\n      KeyRule(pattern=[\'ignored\', Wildcards.SINGLE], func=check_suppress),\n      # Ensure that values are of the same type. This is useful if you expect datetime objects and want to raise\n      #   if one value is Null (or you have a number stored as a string or UUID, etc.)\n      KeyRule(pattern=[\'date\'], func=check_type),\n      KeyRule(pattern=[\'nested\', \'uuid\'], func=check_type),\n      # The KeyRule function can be replaced with any function that accepts keyword arguments \'old\' and \'new\'\n    ]\n    #\n    # Run twice to ensure that the initial cache file exists and that the key_rules suppress any errors\n    assert_against_cache(cached_data, key_rules=key_rules)\n    assert_against_cache(test_data, key_rules=key_rules)\n\n    # Verify that without key rules, an AssertionError is raised\n    with pytest.raises(AssertionError, match=DEF_ERROR_MESSAGE):\n        assert_against_cache(test_data)  # act\n```\n\nOr you may want to write your own custom checks against the serialized data, such as with Cerberus or another library. This is possible with the `validator` callable. The default validator is a no-op and that may be replaced with any custom function that raises an Exception on error.\n\n```py\n"""tests/test_main.py."""\n\nimport re\n\nimport pytest\nfrom beartype import beartype\nfrom cerberus import Validator\nfrom cerberus.schema import SchemaError\n\n\n@beartype\ndef cerberus_validator(test_data: TEST_DATA_TYPE) -> None:\n    """Cerberus custom validator example."""\n    validator = Validator({\'result\': {\'type\': \'int\'}})\n    assert validator.validate(test_data)\n\n\ndef test_assert_against_cache_validator(assert_against_cache):\n    """Test the validator."""\n    expected = re.escape("{\'result\': [{\'type\': [\'Unsupported types: int\']}]}")\n\n    with pytest.raises(SchemaError, match=expected):\n        assert_against_cache({\'result\': False}, validator=cerberus_validator)  # act\n```\n\n### Even More Example\n\nFor more examples, see [Scripts](https://github.com/kyleking/pytest_cache_assert/scripts) or [Tests](https://github.com/kyleking/pytest_cache_assert/tests)\n\n## Global Configuration Options\n\n- `DEF_CACHE_DIR_KEY`: set a custom relative path from the `tests/` directory. Default is `assert-cache/`\n\n```py\nimport pytest\n\nfrom pytest_cache_assert import DEF_CACHE_DIR_KEY\n\n\n@pytest.fixture(scope=\'module\')\ndef cache_assert_config():\n    return {\n        DEF_CACHE_DIR_KEY: \'custom/cache/dir\',\n    }\n```\n\n### Planned Global Configuration Options\n\nThese are ideas for future options that are not currently implemented, but could be if there is enough interest:\n\n- PLANNED: Consider a record mode that will always-write to regenerate the cache while working on development\n    - The other edge case where a `mode` might be helpful is when file names or test names are changed and the cache metadata has too many duplicates and needs to be refreshed\n- PLANNED: [Provide CLI arguments like `pytest-recording`](https://github.com/kiwicom/pytest-recording/blob/484bb887dd43fcaf44149160d57b58a7215e2c8a/src/pytest_recording/plugin.py#L37-L70) (`request.config.getoption("--record-mode") or "none"`) for one-time changes to configuration\n- PLANNED: Consider filters to prevent secrets from being cached: `filter_headers=[[\'authorization\', \'id\'], [\'authorization\', \'cookies\']]`\n\n## Roadmap\n\nSee the `Open Issues` and `Milestones` for current status and [./docs/CODE_TAG_SUMMARY.md](./docs/CODE_TAG_SUMMARY.md) for annotations in the source code.\n\nFor release history, see the [./docs/CHANGELOG.md](./docs/CHANGELOG.md)\n\n## Contributing\n\nSee the Developer Guide, Contribution Guidelines, etc\n\n- [./docs/DEVELOPER_GUIDE.md](./docs/DEVELOPER_GUIDE.md)\n- [./docs/STYLE_GUIDE.md](./docs/STYLE_GUIDE.md)\n- [./docs/CONTRIBUTING.md](./docs/CONTRIBUTING.md)\n- [./docs/CODE_OF_CONDUCT.md](./docs/CODE_OF_CONDUCT.md)\n- [./docs/SECURITY.md](./docs/SECURITY.md)\n\n## License\n\n[LICENSE](https://github.com/kyleking/pytest_cache_assert/LICENSE)\n',
    'author': 'Kyle King',
    'author_email': 'dev.act.kyle@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/kyleking/pytest_cache_assert',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
