# Copyright The PyTorch Lightning team.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
import torch
from torch import Tensor

from torchmetrics.utilities.checks import _check_same_shape


def si_sdr(preds: Tensor, target: Tensor, zero_mean: bool = False) -> Tensor:
    """
    Calculates Scale-invariant signal-to-distortion ratio (SI-SDR) metric. The SI-SDR value
    is in general considered an overall measure of how good a source sound.

    Args:
        preds:
            shape ``[...,time]``
        target:
            shape ``[...,time]``
        zero_mean:
            If to zero mean target and preds or not

    Returns:
        si-sdr value of shape [...]

    Example:
        >>> from torchmetrics.functional.audio import si_sdr
        >>> target = torch.tensor([3.0, -0.5, 2.0, 7.0])
        >>> preds = torch.tensor([2.5, 0.0, 2.0, 8.0])
        >>> si_sdr_val = si_sdr(preds, target)
        >>> si_sdr_val
        tensor(18.4030)

    References:
        [1] Le Roux, Jonathan, et al. "SDR half-baked or well done." IEEE International Conference on Acoustics, Speech
        and Signal Processing (ICASSP) 2019.
    """
    _check_same_shape(preds, target)
    EPS = torch.finfo(preds.dtype).eps

    if zero_mean:
        target = target - torch.mean(target, dim=-1, keepdim=True)
        preds = preds - torch.mean(preds, dim=-1, keepdim=True)

    alpha = (torch.sum(preds * target, dim=-1, keepdim=True) + EPS) / (torch.sum(target**2, dim=-1, keepdim=True) + EPS)
    target_scaled = alpha * target

    noise = target_scaled - preds

    si_sdr_value = (torch.sum(target_scaled**2, dim=-1) + EPS) / (torch.sum(noise**2, dim=-1) + EPS)
    si_sdr_value = 10 * torch.log10(si_sdr_value)

    return si_sdr_value
