#!/usr/bin/env python

from collections import OrderedDict
import unittest

from mock import call, MagicMock  # type: ignore
from pandas import DataFrame, to_datetime  # type: ignore
from pandas.util.testing import (assert_frame_equal,  # type: ignore
                                 assert_series_equal)
import mock
import numpy as np  # type: ignore
import pandas as pd

from edpanalyst import (PopulationSchema, guess_schema,
                        Population, PopulationModel,
                        PopulationModelExperimental)
import edpanalyst


# The population model metadata you use if you don't care about it in your test
PM = {'id': 'pm-17',
      'parent_id': 'p-14',
      'name': 'dontcare',
      'creation_time': 1234,
      'build_progress': {'status': 'built'},
      'user_metadata': {}}
WXYZ_SCHEMA = PopulationSchema.from_json(
    {'columns': [
        {'name': 'w', 'stat_type': 'realAdditive'},
        {'name': 'x', 'stat_type': 'realAdditive'},
        {'name': 'y', 'stat_type': 'realAdditive'},
        {'name': 'z', 'stat_type': 'realAdditive'}]})
XYZ_NAME_SCHEMA = PopulationSchema.from_json(
    {'identifying_columns': ['name'],
     'columns': [{'name': 'name', 'stat_type': 'void'},
                 {'name': 'x', 'stat_type': 'realAdditive'},
                 {'name': 'y', 'stat_type': 'realAdditive'},
                 {'name': 'z', 'stat_type': 'realAdditive'}]})


class SessionTest(unittest.TestCase):

    def setUp(self):
        self._mock_endpoint = MagicMock()
        self._session = edpanalyst.Session(
            client=MagicMock(), endpoint=self._mock_endpoint)

    def test_list(self):
        pm1 = {'id': 'pm-1',
               'parent_id': 'p-1',
               'name': 'first model',
               'creation_time': 123,
               'build_progress': {'status': 'built'},
               'user_metadata': {}}
        pm2 = {'id': 'pm-2',
               'parent_id': 'p-22',
               'name': 'second model',
               'creation_time': 456,
               'build_progress': {'status': 'built'},
               'user_metadata': {}}

        # Test that if you search without a keyword, all population models are
        # returned.
        data = {'id': ['pm-1', 'pm-2'],
                'name': ['first model', 'second model'],
                'parent_id': ['p-1', 'p-22'],
                'creation_time': to_datetime([123, 456], unit='s'),
                'status': ['built', 'built']}
        columns = ['id', 'name', 'parent_id', 'creation_time', 'status']
        expected_pms = DataFrame(data, columns=columns)
        self._mock_endpoint.population_model.get.return_value = FakeResponse(
            [pm1, pm2])
        got_pms = self._session.list().reset_index(drop=True)
        assert_frame_equal(got_pms, expected_pms)

        # Test a match on the name field.
        data = {'id': ['pm-1'],
                'name': ['first model'],
                'parent_id': ['p-1'],
                'creation_time': to_datetime([123], unit='s'),
                'status': ['built']}
        expected_pms = DataFrame(data, columns=columns)
        got_pms = self._session.list('firs').reset_index(drop=True)
        assert_frame_equal(got_pms, expected_pms)

        # Test a string that doesn't match gives a zero-row data frame.
        self.assertEqual(self._session.list('missing substring').shape[0], 0)

    def test_list_pops(self):
        pop1 = {'id': 'p-1',
                'name': 'first pop',
                'creation_time': 234,
                'user_metadata': {},
                'models': ['pm-1']}
        pop2 = {'id': 'p-2',
                'name': 'second pop',
                'creation_time': 789,
                'user_metadata': {},
                'models': []}
        # Test that if you search without a keyword, all populatons are
        # returned.
        data = {'id': ['p-1', 'p-2'],
                'name': ['first pop', 'second pop'],
                'creation_time': to_datetime([234, 789], unit='s'),
                'num_models': [1, 0]}
        columns = ['id', 'name', 'creation_time', 'num_models']
        expected_pops = DataFrame(data, columns=columns)
        self._mock_endpoint.population.get.return_value = FakeResponse(
            [pop1, pop2])
        got_pops = self._session.list_populations().reset_index(drop=True)
        assert_frame_equal(got_pops, expected_pops)

        # Test a match on the name field.
        data = {'id': ['p-1'],
                'name': ['first pop'],
                'creation_time': to_datetime([234], unit='s'),
                'num_models': [1]}
        expected_pops = DataFrame(data, columns=columns)
        got_pops = self._session.list_populations('fir').reset_index(drop=True)
        assert_frame_equal(got_pops, expected_pops)

        # Test a string that doesn't match gives a zero-row data frame.
        self.assertEqual(self._session.list_populations(
            'missing substring').shape[0], 0)

    def test_complains_on_creation(self):
        mock_endpoint = MagicMock()
        mock_endpoint.population.get.return_value = FakeResponse(
            None, status_code=401)
        with self.assertRaises(ValueError):
            edpanalyst.Session(client=None, endpoint=mock_endpoint)


class TestUpload(unittest.TestCase):
    def test_upload(self):
        mock_client = MagicMock()
        mock_client.upload_population.return_value = 'fakepid'

        ses = edpanalyst.Session(mock_client)
        data = DataFrame({'foo': [1, 2, 3]})
        schema = guess_schema(data)
        schema.set_identifying_columns(['foo'])
        ses.upload(data, 'mynewpop', schema, autobuild=False)

        mock_client.upload_population.assert_called_once_with(
            data={'foo': ['1', '2', '3']},
            schema=schema,
            name='mynewpop')

    def test_upload_guesses_schema(self):
        mock_client = MagicMock()
        mock_client.upload_population.return_value = 'fakepid'

        ses = edpanalyst.Session(mock_client)
        data = DataFrame({'foo': [1, 2, 3]})
        ses.upload(data, 'mynewpop', autobuild=False)

        mock_client.upload_population.assert_called_once_with(
            data={'foo': ['1', '2', '3']},
            schema=mock.ANY,
            name='mynewpop')
        # Check that we got a reasonable schema
        schema_arg = mock_client.upload_population.call_args[1]['schema']
        self.assertEqual(['foo'], schema_arg.columns())


def _pop_with_mocked_backend(desc, schema=None):
    mock_endpoint = MagicMock()
    mock_endpoint.get.return_value = FakeResponse(desc)
    if schema:
        mock_endpoint.schema.get.return_value = FakeResponse(schema.to_json())
    pop = Population(desc['id'], client=None, endpoint=mock_endpoint)
    mock_client = MagicMock()
    mock_client.config.edp_url = 'baseurl'
    pop._client = mock_client
    return pop, mock_endpoint


class TestPopulation(unittest.TestCase):
    def test_repr_html(self):
        desc = {'id': 'p-7',
                'name': 'MyPop',
                'models': [PM]
                }
        pop, mock_endpoint = _pop_with_mocked_backend(desc, WXYZ_SCHEMA)
        self.assertRegexpMatches(
            pop._repr_html_(),
            'href="baseurl/explorer/population_model/pm-17"')

    def test_latest(self):
        desc = {'id': 'p-7',
                'name': 'MyPop',
                'models': [
                    {'id': 'pm-72',
                     'build_progress': {'status': 'built'},
                     'creation_time': 123},
                    {'id': 'pm-12',
                     'build_progress': {'status': 'built'},
                     'creation_time': 456},
                    {'id': 'pm-81',
                     'build_progress': {'status': 'unbuilt'},
                     'creation_time': 789},
                    {'id': 'pm-19',
                     'build_progress': {'status': 'built'},
                     'creation_time': 0},
                    ]
                }
        pop, mock_endpoint = _pop_with_mocked_backend(desc, WXYZ_SCHEMA)
        self.assertEqual('pm-12', pop.latest._pmid)

    def test_latest_with_no_built_models(self):
        desc = {'id': 'p-7',
                'name': 'MyPop',
                'models': [
                    {'id': 'pm-81',
                     'build_progress': {'status': 'unbuilt'},
                     'creation_time': 789},
                    ]
                }
        pop, mock_endpoint = _pop_with_mocked_backend(desc, WXYZ_SCHEMA)
        with self.assertRaises(ValueError):
            pop.latest


def _popmod_with_mocked_backend(desc, schema=None):
    mock_endpoint = MagicMock()
    mock_endpoint.get.return_value = FakeResponse(desc)
    if schema:
        mock_endpoint.schema.get.return_value = FakeResponse(schema.to_json())
    popmod = PopulationModel(desc['id'], client=None, endpoint=mock_endpoint)
    return popmod, mock_endpoint


class TestPopulationModel(unittest.TestCase):
    def test_repr_html(self):
        desc = {'id': PM['id'],
                'parent_id': None,
                'name': 'myname',
                'creation_time': 0,
                'build_progress': {'status': 'built'},
                'user_metadata': {}}
        popmod, mock_endpoint = _popmod_with_mocked_backend(desc, WXYZ_SCHEMA)
        mock_client = MagicMock()
        mock_client.config.edp_url = 'baseurl'
        popmod._client = mock_client
        self.assertEqual(
            '<a href="baseurl/explorer/population_model/pm-17" '
            'target="_blank">Explore myname</a>',
            popmod._repr_html_())

    def test_repr_html_unbuilt(self):
        desc = {'id': PM['id'],
                'parent_id': None,
                'name': 'myname',
                'creation_time': 0,
                'build_progress': {'status': 'in_progress'},
                'user_metadata': {}}
        popmod, mock_endpoint = _popmod_with_mocked_backend(desc, WXYZ_SCHEMA)
        # Even if schema() is raising we should still be able to make a
        # population with a working _repr_html_.
        mock_endpoint.schema.get.side_effect = edpanalyst.ModelNotBuiltError()
        popmod = PopulationModel(
            popmod._pmid,
            client=None, endpoint=popmod._endpoint)
        mock_client = MagicMock()
        mock_client.config.edp_url = 'baseurl'
        popmod._client = mock_client
        self.assertEqual(
            '<a href="baseurl/explorer/population_model/pm-17" '
            'target="_blank">Explore myname (unbuilt)</a>',
            popmod._repr_html_())

    def test_experimental_property(self):  # type: () -> None
        popmod, _ = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)
        self.assertIsInstance(popmod.experimental, PopulationModelExperimental)


class TestSelect(unittest.TestCase):
    def test_select(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.select.post.return_value = FakeResponse({
            'columns': {'x': [2, 3, 4, 5, 6],
                        'y': [9, 8, 7, 6, 5]},
            'rowids': [0, 1, 2, 3, 4]})

        expected = DataFrame(
            {'x': [2, 3, 4, 5, 6],
             'y': [9, 8, 7, 6, 5]},
            columns=['x', 'y'],
            index=[0, 1, 2, 3, 4])
        sel = popmod.select(['x', 'y'])
        assert_frame_equal(sel, expected)

        expected_req = {'target': ['x', 'y']}
        mock_endpoint.select.post.assert_called_once_with(
            json=expected_req)

    def test_select_with_given(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.select.post.return_value = FakeResponse({
            'columns': {'x': [2, 3, 4, 5, 6],
                        'y': [9, 8, 7, 6, 5]},
            'rowids': [0, 1, 2, 3, 4]})

        expected = DataFrame(
            {'x': [2, 3, 4, 5, 6],
             'y': [9, 8, 7, 6, 5]},
            columns=['x', 'y'],
            index=[0, 1, 2, 3, 4])
        sel = popmod.select(['x', 'y'], where={'z': 8})
        assert_frame_equal(sel, expected)

        expected_req = {'target': ['x', 'y'], 'where': {'z': 8}}
        mock_endpoint.select.post.assert_called_once_with(
            json=expected_req)

    def test_select_with_index_as_rowids(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.select.post.return_value = FakeResponse({
            'columns': {'x': [2, 3, 4, 5, 6],
                        'y': [9, 8, 7, 6, 5]},
            'rowids': [0, 1, 2, 3, 4]})

        popmod.select(['x', 'y'], rowids=pd.Index([8, 1]))

        expected_req = {'target': ['x', 'y'], 'rowids': [8, 1]}
        mock_endpoint.select.post.assert_called_once_with(
            json=expected_req)


class TestColumnAssociation(unittest.TestCase):
    def test_mutual_information(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.column_association.post.return_value = FakeResponse(
            {'target': ['x', 'y'], 'elements': [1, .75, 1]})

        expected = DataFrame([
            ('x', 'x', 1),
            ('x', 'y', .75),
            ('y', 'x', .75),
            ('y', 'y', 1)
            ], columns=['X', 'Y', 'I']).set_index(['X', 'Y'])['I']
        mi = popmod.mutual_information(['x', 'y'])
        assert_series_equal(mi, expected)

        expected_req = {'target': ['x', 'y'],
                        'statistic': 'mutual information'}
        mock_endpoint.column_association.post.assert_called_once_with(
            json=expected_req)

    def test_mutual_information_with_given_values(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.column_association.post.return_value = FakeResponse(
            {'target': ['w', 'x', 'z'], 'elements': [1, .75, 1, .1, .2, 1]})

        popmod.mutual_information(given_values={'y': 19})

        # Drops out 'y' because it's in givens
        expected_req = {'target': ['w', 'x', 'z'],
                        'given': {'y': 19},
                        'statistic': 'mutual information'}
        mock_endpoint.column_association.post.assert_called_once_with(
            json=expected_req)

    def test_mutual_information_with_given_columns(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.column_association.post.return_value = FakeResponse(
            {'target': ['w', 'x', 'z'], 'elements': [1, .75, 1, .1, .2, 1]})

        popmod.mutual_information(given_columns=['y'])

        # Drops out 'y' because it's in givens
        expected_req = {'target': ['w', 'x', 'z'],
                        'given_columns': ['y'],
                        'statistic': 'mutual information'}
        mock_endpoint.column_association.post.assert_called_once_with(
            json=expected_req)

    def test_mutual_information_with_overlapping_cols(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        # 'y' is in both columns and givens
        with self.assertRaises(ValueError):
            popmod.mutual_information(columns=['x', 'y'],
                                      given_values={'y': 19, 'z': 13})

    def test_mutual_information_diffable(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.column_association.post.side_effect = [
            FakeResponse({'target': ['x', 'y'], 'elements': [1, .75, 1]}),
            FakeResponse({'target': ['x', 'y'], 'elements': [1, .5, 1]}),
            ]

        expected = DataFrame([
            ('x', 'x', 0),
            ('x', 'y', .25),
            ('y', 'x', .25),
            ('y', 'y', 0)
            ], columns=['X', 'Y', 'I']).set_index(['X', 'Y'])['I']
        mi1 = popmod.mutual_information(['x', 'y'])
        mi2 = popmod.mutual_information(['x', 'y'])
        assert_series_equal(mi1 - mi2, expected)

    def test_classic_dep_prob(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.column_association.post.return_value = FakeResponse(
            {'target': ['x', 'y'], 'elements': [1, .75, 1]})

        popmod.classic_dep_prob(['x', 'y'])

        expected_req = {'target': ['x', 'y'],
                        'statistic': 'classic dep prob'}
        mock_endpoint.column_association.post.assert_called_once_with(
            json=expected_req)

    def test_column_relevance(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.column_association.post.return_value = FakeResponse(
            {'target': ['x', 'y'], 'elements': [None, .4, None]})

        expected = DataFrame([
            ('x', 'x', np.nan),
            ('x', 'y', .6),
            ('y', 'x', .6),
            ('y', 'y', np.nan)
            ], columns=['X', 'Y', 'I']).set_index(['X', 'Y'])['I']
        mi = popmod.column_relevance(['x', 'y'])
        assert_series_equal(mi, expected)

        expected_req = {'target': ['x', 'y'],
                        'statistic': 'mutual information',
                        'distribution': .1}
        mock_endpoint.column_association.post.assert_called_once_with(
            json=expected_req)

    def test_relevant_columns(self):
        col_objs = [{'name': colname, 'stat_type': 'realAdditive'}
                    for colname in ['x', 'a', 'b', 'c']]
        schema = PopulationSchema.from_json({'columns': col_objs})
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, schema)

        mock_endpoint.column_association.post.return_value = FakeResponse({
            'target': ['x', 'a', 'b', 'c'],
            'elements': [1,
                         .5, 1,
                         .01, .7, 1,
                         .8, .2, .3, 1]})

        relevant_cols = popmod.relevant_columns('x', num_cols=3)
        expected = DataFrame([('x', 1), ('c', .8), ('a', .5)],
                             columns=['column', 'depprob'])
        assert_frame_equal(relevant_cols, expected)

        expected_req = {'target': ['x', 'a', 'b', 'c'],
                        'statistic': 'classic dep prob'}
        mock_endpoint.column_association.post.assert_called_once_with(
            json=expected_req)


class TestProbability(unittest.TestCase):
    def test_joint_probability(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.logpdf_rows.post.return_value = FakeResponse(
            np.log([.2, .35]))

        results = popmod.joint_probability(
            OrderedDict([('x', [1, 2]), ('y', [3, 1])]),
            probability_column='pr')

        expected = DataFrame({
            'x': [1, 2],
            'y': [3, 1],
            'pr': [.2, .35]},
            columns=['x', 'y', 'pr'])
        assert_frame_equal(results, expected)

        expected_req = {'targets': {'x': [1, 2], 'y': [3, 1]}}
        mock_endpoint.logpdf_rows.post.assert_called_once_with(
            json=expected_req)

    def test_joint_probability_with_givens(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.logpdf_rows.post.return_value = FakeResponse(
            np.log([.2, .35]))

        results = popmod.joint_probability(
            OrderedDict([('x', [1, 2]), ('y', [3, 1])]),
            given={'z': 5},
            probability_column='pr')

        expected = DataFrame({
            'x': [1, 2],
            'y': [3, 1],
            'pr': [.2, .35]},
            columns=['x', 'y', 'pr'])
        assert_frame_equal(results, expected)

        expected_req = {
            'targets': {'x': [1, 2], 'y': [3, 1]},
            'given': {'z': 5}}
        mock_endpoint.logpdf_rows.post.assert_called_once_with(
            json=expected_req)


class TestLikelihood(unittest.TestCase):
    def test_joint_likelihood(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.select.post.return_value = FakeResponse({
            'columns': {'x': [2, 3, 4, 5, 6],
                        'y': [9, 8, 7, 6, 5]},
            'rowids': [0, 1, 2, 3, 4]})
        mock_endpoint.logpdf_observed.post.return_value = FakeResponse(
            np.log([.025, .04, .03, .01, .05]))

        results = popmod.joint_likelihood(['x', 'y'], probability_column='pr')

        expected = DataFrame({
            'x': [2, 3, 4, 5, 6],
            'y': [9, 8, 7, 6, 5],
            'pr': [.025, .04, .03, .01, .05]},
            columns=['x', 'y', 'pr'],
            index=[0, 1, 2, 3, 4])
        assert_frame_equal(results, expected)

        mock_endpoint.select.post.assert_called_once_with(
            json={'target': ['x', 'y']})
        # We didn't specify givens so implicitly use all non-target columns
        expected_req = {
            'targets': ['x', 'y'],
            'givens': ['w', 'z'],
            'rowids': [0, 1, 2, 3, 4]}
        mock_endpoint.logpdf_observed.post.assert_called_once_with(
            json=expected_req)

    def test_joint_likelihood_with_givens(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.select.post.return_value = FakeResponse({
            'columns': {'x': [2, 3, 4, 5, 6],
                        'y': [9, 8, 7, 6, 5]},
            'rowids': [0, 1, 2, 3, 4]})
        mock_endpoint.logpdf_observed.post.return_value = FakeResponse(
            np.log([.025, .04, .03, .01, .05]))

        popmod.joint_likelihood(['x', 'y'], givens=['z'])

        expected_req = {
            'targets': ['x', 'y'],
            'givens': ['z'],
            'rowids': [0, 1, 2, 3, 4]}
        mock_endpoint.logpdf_observed.post.assert_called_once_with(
            json=expected_req)

    def test_joint_likelihood_with_overlapping_columns(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        # `y` is in both `targets` and `givens`
        with self.assertRaises(ValueError):
            popmod.joint_likelihood(['x', 'y'], givens=['y', 'z'])

    def test_joint_likelihood_with_id_cols(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(
            PM, XYZ_NAME_SCHEMA)

        mock_endpoint.select.post.side_effect = [
            FakeResponse({'columns': {
                              'x': [2, 3, 4, 5, 6],
                              'y': [9, 8, 7, 6, 5]},
                          'rowids': [0, 1, 2, 3, 4]}),
            FakeResponse({'columns': {
                              'name': ['a', 'b', 'c', 'd', 'e']},
                          'rowids': [0, 1, 2, 3, 4]})]
        mock_endpoint.logpdf_observed.post.return_value = FakeResponse(
            np.log([.025, .04, .03, .01, .05]))

        results = popmod.joint_likelihood(['x', 'y'], probability_column='pr')

        expected = DataFrame({
            'name': ['a', 'b', 'c', 'd', 'e'],
            'x': [2, 3, 4, 5, 6],
            'y': [9, 8, 7, 6, 5],
            'pr': [.025, .04, .03, .01, .05]},
            columns=['name', 'x', 'y', 'pr'],
            index=[0, 1, 2, 3, 4])
        assert_frame_equal(results, expected)

        mock_endpoint.select.post.assert_has_calls([
            call(json={'target': ['x', 'y']}),
            call(json={'target': ['name'], 'rowids': [0, 1, 2, 3, 4]})])

        # Make sure we can turn off id columns
        mock_endpoint.reset_mock()
        mock_endpoint.select.post.side_effect = [
            FakeResponse({'columns': {
                              'x': [2, 3, 4, 5, 6],
                              'y': [9, 8, 7, 6, 5]},
                          'rowids': [0, 1, 2, 3, 4]})]
        popmod.return_identifying_columns(False)
        popmod.joint_likelihood(['x', 'y'], probability_column='pr')
        mock_endpoint.select.post.assert_called_once_with(
            json={'target': ['x', 'y']})

    def test_joint_likelihood_doesnt_stomp_on_columns(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)
        with self.assertRaises(ValueError):
            popmod.joint_likelihood(['x', 'pro'], probability_column='pro')

    def test_likelihood(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.select.post.return_value = FakeResponse({
            'columns': {'x': [2, 3, 4, 5, 6],
                        'y': [9, 8, None, 6, 5]},
            'rowids': [0, 1, 2, 3, 4]})
        mock_endpoint.logpdf_observed.post.side_effect = [
            FakeResponse(np.log([.025, .04, np.nan, .01, .05])),
            FakeResponse(np.log([.1, .2, .3, .4, .5]))]

        results = popmod.likelihood(['x', 'y'], probability_suffix='_pr')

        expected = DataFrame({
            'x': [2, 3, 4, 5, 6],
            'x_pr': [.025, .04, None, .01, .05],
            'y': [9, 8, None, 6, 5],
            'y_pr': [.1, .2, .3, .4, .5]},
            columns=['x', 'x_pr', 'y', 'y_pr'],
            index=[0, 1, 2, 3, 4])
        assert_frame_equal(results, expected)

        mock_endpoint.select.post.assert_called_once_with(
            json={'target': ['x', 'y']})
        # Make sure there were separate calls for each column
        mock_endpoint.logpdf_observed.post.assert_has_calls([
            call(json={'targets': ['x'], 'givens': ['w', 'y', 'z'],
                       'rowids': [0, 1, 2, 3, 4]}),
            call(json={'targets': ['y'], 'givens': ['w', 'x', 'z'],
                       'rowids': [0, 1, 2, 3, 4]})],
            any_order=True)

    def test_likelihood_with_givens(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.select.post.return_value = FakeResponse({
            'columns': {'x': [2, 3, 4, 5, 6],
                        'y': [9, 8, None, 6, 5]},
            'rowids': [0, 1, 2, 3, 4]})
        mock_endpoint.logpdf_observed.post.side_effect = [
            FakeResponse(np.log([.025, .04, np.nan, .01, .05])),
            FakeResponse(np.log([.1, .2, .3, .4, .5]))]

        popmod.likelihood(['x', 'y'], givens=['w', 'x', 'z'])

        mock_endpoint.select.post.assert_called_once_with(
            json={'target': ['x', 'y']})
        # Make sure there were separate calls for each column
        mock_endpoint.logpdf_observed.post.assert_has_calls([
            # 'y' is not a given col, so it's not conditioned on
            call(json={'targets': ['x'], 'givens': ['w', 'z'],
                       'rowids': [0, 1, 2, 3, 4]}),
            call(json={'targets': ['y'], 'givens': ['w', 'x', 'z'],
                       'rowids': [0, 1, 2, 3, 4]})],
            any_order=True)

    def test_likelihood_with_id_cols_and_select(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(
            PM, XYZ_NAME_SCHEMA)

        # First is the select for the likelihood, then is the select for the
        # `select` columns.
        mock_endpoint.select.post.side_effect = [
            FakeResponse({'columns': {
                              'x': [2, 3, 4, 5, 6],
                              'y': [9, 8, None, 6, 5]},
                          'rowids': [0, 1, 2, 3, 4]}),
            FakeResponse({'columns': {
                              'otherselect': [9, 8, 7, 6, 5],
                              'name': ['a', 'b', 'c', 'd', 'e']},
                          'rowids': [0, 1, 2, 3, 4]})]
        mock_endpoint.logpdf_observed.post.side_effect = [
            FakeResponse(np.log([.025, .04, np.nan, .01, .05])),
            FakeResponse(np.log([.1, .2, .3, .4, .5]))]

        results = popmod.likelihood(['x', 'y'], select=['otherselect'],
                                    probability_suffix='_pr')

        expected = DataFrame({
            'name': ['a', 'b', 'c', 'd', 'e'],
            'otherselect': [9, 8, 7, 6, 5],
            'x': [2, 3, 4, 5, 6],
            'x_pr': [.025, .04, None, .01, .05],
            'y': [9, 8, None, 6, 5],
            'y_pr': [.1, .2, .3, .4, .5]},
            columns=['name', 'otherselect', 'x', 'x_pr', 'y', 'y_pr'],
            index=[0, 1, 2, 3, 4])
        assert_frame_equal(results, expected)

        mock_endpoint.select.post.assert_has_calls([
            call(json={'target': ['x', 'y']}),
            call(json={'target': ['name', 'otherselect'],
                       'rowids': [0, 1, 2, 3, 4]})
            ])
        # Make sure there were separate calls for each column
        mock_endpoint.logpdf_observed.post.assert_has_calls([
            call(json={'targets': ['x'], 'givens': ['y', 'z'],
                       'rowids': [0, 1, 2, 3, 4]}),
            call(json={'targets': ['y'], 'givens': ['x', 'z'],
                       'rowids': [0, 1, 2, 3, 4]})],
            any_order=True)

    def test_likelihood_doesnt_stomp_on_columns(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)
        with self.assertRaises(ValueError):
            popmod.likelihood(['x', 'x_mysuffix'],
                              probability_suffix='_mysuffix')


class TestAddDataColumns(unittest.TestCase):
    def test_add_data_columns(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(
            PM, XYZ_NAME_SCHEMA)

        mock_endpoint.select.post.return_value = FakeResponse(
            {'columns': {'name': [8, 1],
                         'z': [3, 19]},
             'rowids': [7, 3]})
        df = DataFrame(
            {'foo': [1, 17]},
            index=[7, 3])
        expected = DataFrame(
            {'foo': [1, 17],
             'name': [8, 1],
             'z': [3, 19]},
            columns=['name', 'z', 'foo'],
            index=[7, 3])
        results = popmod.add_data_columns(df, ['z'])
        assert_frame_equal(results, expected)

        mock_endpoint.select.post.assert_called_once_with(
                json={'target': ['name', 'z'], 'rowids': [7, 3]})


class TestInfer(unittest.TestCase):
    def test_infer(self):
        schema = PopulationSchema.from_json(
            {'identifying_columns': ['x', 'z'],
             'columns': [{'name': 'x', 'stat_type': 'realAdditive'},
                         {'name': 'y', 'stat_type': 'realAdditive'},
                         {'name': 'z', 'stat_type': 'realAdditive'}]})
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, schema)

        mock_endpoint.select.post.return_value = FakeResponse(
            {'columns': {'z': [5, 4, 3, 2, 1]},
             'rowids': [0, 1, 2, 3, 4]})
        mock_endpoint.infer.post.return_value = FakeResponse(
            {'columns': {'x': [0,    6,  None, 4,  None],
                         'y': [None, 7,  None, 5,  7]},
             'uncertainty': {'x': [.3,   .3, None, .4, None],
                             'y': [None, .1, None, .9, .2]},
             'rowids': [0, 1, 2, 3, 4]})

        results = popmod.infer(['x', 'y'])
        expected = DataFrame({
            'z':          [5,     4,     3, 2,      1],
            'x':          [0,     6,  None, 4,   None],
            'x_conf':     [.3,   .3,  None, .4,  None],
            'y': [None, 7,   None, 5,   7],
            'y_conf':     [None, .1,  None, .9,  .2]},
            columns=['z', 'x', 'x_conf', 'y', 'y_conf'],
            index=[0, 1, 2, 3, 4])
        assert_frame_equal(results, expected)
        mock_endpoint.select.post.assert_called_with(
            json={'target': ['z'], 'rowids': [0, 1, 2, 3, 4]})
        mock_endpoint.infer.post.assert_called_with(
            json={'target': ['x', 'y'],
                  'rowids': [0, 1, 2, 3, 4],
                  'infer_present': False})

    def test_infer_doesnt_stomp_on_columns(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)
        mock_endpoint.select.post.return_value = FakeResponse(
            {'columns': {'z': [5, 4, 3, 2, 1]},
             'rowids': [0, 1, 2, 3, 4]})
        with self.assertRaises(ValueError):
            popmod.infer(['x', 'x_mysuffix'], confidence_suffix='_mysuffix')


class TestSimulate(unittest.TestCase):
    def test_implicit_targets(self):
        schema = PopulationSchema.from_json(
            {'columns': [{'name': 'x', 'stat_type': 'realAdditive'}]})
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, schema)

        mock_endpoint.simulate_row.post.return_value = FakeResponse(
            {'columns': {'x': [4, 7]}})

        expected = DataFrame({'x': [4, 7]})
        assert_frame_equal(popmod.simulate(n=2), expected)

        mock_endpoint.simulate_row.post.assert_called_once_with(
            json={'target': ['x'], 'n': 2})

    def test_target(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.simulate_row.post.return_value = FakeResponse(
            {'columns': {'z': [4, 7]}})

        expected = DataFrame({'z': [4, 7]})
        assert_frame_equal(popmod.simulate(targets=['z'], n=2),
                           expected)

        mock_endpoint.simulate_row.post.assert_called_once_with(
            json={'target': ['z'], 'n': 2})

    def test_rowid(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.select.post.return_value = FakeResponse(
            {'columns': {'x': [2], 'y': [9], 'z': [None]}, 'rowids': ['3']})
        mock_endpoint.simulate_row.post.return_value = FakeResponse(
            {'columns': {'x': [4, 7]}})

        expected = DataFrame({'x': [4, 7]})
        assert_frame_equal(
            popmod.simulate(targets=['x'], rowid='3', n=2),
            expected)

        # `x` is the target so we drop it from givens, and `z` is missing in
        # the row so it's not given either.
        mock_endpoint.simulate_row.post.assert_called_once_with(
            json={'target': ['x'], 'given': {'y': 9}, 'n': 2})

    def test_given_overrides_rowid(self):
        popmod, mock_endpoint = _popmod_with_mocked_backend(PM, WXYZ_SCHEMA)

        mock_endpoint.select.post.return_value = FakeResponse(
            {'columns': {'x': [2], 'y': [9], 'z': [13]}, 'rowids': ['3']})
        mock_endpoint.simulate_row.post.return_value = FakeResponse(
            {'columns': {'x': [4, 7]}})

        expected = DataFrame({'x': [4, 7]})
        assert_frame_equal(popmod.simulate(
            targets=['x'], rowid='3', given={'z': 42}, n=2),
            expected)

        # `y` comes from rowid, `z` from givens
        mock_endpoint.simulate_row.post.assert_called_once_with(
                json={'target': ['x'], 'given': {'y': 9, 'z': 42}, 'n': 2})


class FakeResponse(object):
    """A testing class that mimics a requests.Response."""
    def __init__(self, json, status_code=200):
        self._json = json
        self.status_code = status_code

    def json(self):
        self.raise_for_status()
        return self._json

    def raise_for_status(self):
        if self.status_code != 200:
            raise ValueError('Is this right?')


if __name__ == '__main__':
    unittest.main()
