"""
``Analytics`` are intended to more closely rank or compare a GEM subset, rather than the entire GEM.
These functions are intended for analyzing and comparing subsets generated by the functions found in ``prospectors``.
"""

import numpy as np
import param
import xarray as xr
from sklearn.base import clone

from ..models import CallableInterface
from ..utils._operations import shuffle_along_axis, null_rank_distribution

__all__ = [
    "RankGenesByModel",
    "nFDR",
    "mProbes",
]


class RankGenesByModel(CallableInterface):
    """
    Given some machine learning model, runs `n_iterations` and returns a summary of the ranking results.

    This operation uses the ``Interface`` base class.

    Parameters
    ----------
    model :
        A model that can be trained via ``.fit(x, y)``.

    n_iterations : int
        Number of gene ranking iterations to use an instance of `model`.

    """
    model = param.Parameter()
    n_iterations = param.Integer(default=1)

    @staticmethod
    def rank_genes_by_model(counts: xr.DataArray,
                            labels: xr.DataArray,
                            model) -> xr.DataArray:
        model = model.fit(counts.values, labels.values)
        attrs = {'Ranking Model': str(model),
                 "count_variable": counts.name,
                 "annotation_variables": labels.name}

        data = xr.DataArray(data=model.feature_importances_,
                            coords=[(counts.coords.dims[1], counts.coords[counts.coords.dims[1]].values)],
                            name="feature_importances",
                            attrs=attrs)
        return data

    def __call__(self, *args, **params):

        if len(self.annotation_variables) > 1:
            raise ValueError(f"This operation only accepts a single entry for `annotation_variables`.")

        counts = self.x_count_data
        targets = self.y_annotation_data

        if self.n_iterations == 1:
            return self.rank_genes_by_model(counts=counts, labels=targets, model=self.model).to_dataset()

        else:
            # return [self.rank_genes_by_model(counts=counts, labels=targets, model=self.model)
            #                         for _ in range(self.n_iterations)]
            ranking_ds = xr.concat([self.rank_genes_by_model(counts=counts, labels=targets, model=self.model)
                                    for _ in range(self.n_iterations)],
                                   dim='model_iteration')
            # ranking_ds["feature_importance_mean"] = ranking_ds['feature_importance'].mean(dim="model_iteration")
            # ranking_ds["feature_importance_std"] = ranking_ds['feature_importance'].std(dim="model_iteration")
            return ranking_ds


class nFDR(CallableInterface):
    """
    nFDR (False Discovery Rate) [method_compare]_.

    nFDR trains two models and compares their ``feature_importances_`` attributes to estimate
    the false discovery rate.

    The FDR estimated is the percent of instances a shuffled output feature has a higher feature
    importance score than the same non-shuffled feature score.

    This is repeated up to ``n_iterations``.
    """

    model = param.Parameter()
    n_iterations = param.Integer(default=1)

    @staticmethod
    def nFDR(counts: xr.DataArray,
             labels: xr.DataArray,
             model) -> xr.DataArray:

        y_shadowed = np.random.permutation(labels.values)

        real_model = clone(model).fit(counts.values, labels.values)
        shadow_model = clone(model).fit(counts.values, y_shadowed)

        null_rank_dist = null_rank_distribution(real_model.feature_importances_,
                                                shadow_model.feature_importances_)
        attrs = {'Ranking Model': str(model),
                 "count_variable": counts.coords.dims[1],
                 "annotation_variables": labels.name}

        nrd_xarray = xr.DataArray(data=null_rank_dist,
                                  coords=[(counts.coords.dims[1], counts.coords[counts.coords.dims[1]].values)],
                                  name="nFDR",
                                  attrs=attrs)
        return nrd_xarray

    def __call__(self, *args, **params):

        counts = self.x_count_data
        targets = self.y_annotation_data

        if self.n_iterations == 1:
            return self.nFDR(counts=counts, labels=targets, model=self.model).to_dataset()

        else:
            ranking_ds = xr.concat([self.nFDR(counts=counts, labels=targets, model=self.model)
                                    for _ in range(self.n_iterations)],
                                   dim='model_iteration')

            # ranking_ds["nFDR_mean"] = ranking_ds['nFDR'].mean(dim="model_iteration")
            # ranking_ds["nFDR_std"] = ranking_ds['nFDR'].std(dim="model_iteration")

            return ranking_ds


class mProbes(CallableInterface):
    """
    mProbes [method_compare]_ works by randomly permuting the feature values in the supplied data.
    e.g. count values are shuffled within each samples feature (gene) array.

    It then ranks the real and shadowed features (for ``n_iterations``) with the supplied ``model``
    via a call to ``model.fit()``. It then examines ``model.feature_importances_`` for the feature
    importance values, and then calculates the null rank distribution.

    This is repeated upto ``n_iterations``.
    """

    model = param.Parameter()
    n_iterations = param.Integer(default=1)

    @staticmethod
    def mProbes(counts: xr.DataArray,
                labels: xr.DataArray,
                model) -> xr.DataArray:

        shadowed_array = shuffle_along_axis(counts.values, 1)
        x_shadowed = np.hstack((counts.values, shadowed_array))
        model = model.fit(x_shadowed, labels.values)
        ranks = model.feature_importances_
        real, shadow = ranks.reshape((2, counts.values.shape[1]))
        null_rank_dist = null_rank_distribution(real, shadow)

        attrs = {'Ranking Model': str(model),
                 "count_variable": counts.coords.dims[1],
                 "annotation_variables": labels.name}

        mprobes_xarray = xr.DataArray(
            data=null_rank_dist,
            coords=[(counts.coords.dims[1], counts.coords[counts.coords.dims[1]].values)],
            name="nFDR",
            attrs=attrs)

        return mprobes_xarray

    def __call__(self, *args, **params):

        counts = self.x_count_data
        targets = self.y_annotation_data

        if self.n_iterations == 1:
            return self.nFDR(counts=counts, labels=targets, model=self.model).to_dataset()

        else:
            mprobes_data = np.vstack([self.mProbes(counts=counts, labels=targets, model=self.model)
                                      for i in range(self.n_iterations)])

            ranking_ds = xr.Dataset(
                data_vars={
                    'mProbes': (
                        ('model_iteration', counts.coords.dims[1]),  # dimensions
                        mprobes_data,  # values
                    ),
                },
                coords={
                    counts.coords.dims[1]: counts[counts.coords.dims[1]].values,
                    'model_iteration': np.arange(self.n_iterations),
                },
            )
            # ranking_ds["mProbes_mean"] = ranking_ds['mProbes'].mean(dim="model_iteration")
            # ranking_ds["mProbes_std"] = ranking_ds['mProbes'].std(dim="model_iteration")

            return ranking_ds

        # x_data = self.x_count_data.values
        # y_data = self.y_annotation_data.values
        #
        # if self.n_iterations == 1:
        #     return self._mprobes_fdr_to_xarray(self.mProbes(x_data, y_data, self.model))
        # else:
        #     fdrs = [self.mProbes(x_data, y_data, self.model) for i in range(self.n_iterations)]
        #     fdrs = [self._mprobes_fdr_to_xarray(values) for values in fdrs]
        #     ranking_ds = xr.concat(fdrs, "mProbes_iter")
        #     ranking_ds.name = "mProbes NRD"
        #     ranking_ds["mProbes_iter"] = (["mProbes_iter", ], np.arange(self.n_iterations))
        #     ranking_ds["mProbes_mean"] = ([self.gem.gene_index_name], ranking_ds.mean(dim="mProbes_iter"))
        #     ranking_ds["mProbes_std"] = ([self.gem.gene_index_name], ranking_ds.std(dim="mProbes_iter"))
        #     return ranking_ds.to_dataset()
