from kubragen.configfile import ConfigFileExtension, ConfigFile, ConfigFileExtensionData
from kubragen.options import OptionGetter


class PrometheusConfigFileExt_Kubernetes(ConfigFileExtension):
    """
    Prometheus configuration extension for Kubernetes scrape config.

    Based on `prometheus/prometheus <https://github.com/prometheus/prometheus/blob/master/documentation/examples/prometheus-kubernetes.yml>`_.
    """
    def process(self, configfile: ConfigFile, data: ConfigFileExtensionData, options: OptionGetter) -> None:
        if 'scrape_configs' not in data.data:
            data.data['scrape_configs'] = []
        data.data['scrape_configs'].extend([{
            'job_name': 'kubernetes-apiservers',
            'kubernetes_sd_configs': [{
                'role': 'endpoints'
            }],
            'scheme': 'https',
            'tls_config': {
                'ca_file': '/var/run/secrets/kubernetes.io/serviceaccount/ca.crt'
            },
            'bearer_token_file': '/var/run/secrets/kubernetes.io/serviceaccount/token',
            'relabel_configs': [{
                'source_labels': ['__meta_kubernetes_namespace',
                                  '__meta_kubernetes_service_name',
                                  '__meta_kubernetes_endpoint_port_name'],
                'action': 'keep',
                'regex': 'default;kubernetes;https'
            }]
        },
        {
            'job_name': 'kubernetes-nodes',
            'scheme': 'https',
            'tls_config': {'ca_file': '/var/run/secrets/kubernetes.io/serviceaccount/ca.crt'},
            'bearer_token_file': '/var/run/secrets/kubernetes.io/serviceaccount/token',
            'kubernetes_sd_configs': [{
                'role': 'node'
            }],
            'relabel_configs': [{
                'action': 'labelmap',
                'regex': '__meta_kubernetes_node_label_(.+)'
            }]
        },
        {
            'job_name': 'kubernetes-cadvisor',
            'scheme': 'https',
            'metrics_path': '/metrics/cadvisor',
            'tls_config': {
                'ca_file': '/var/run/secrets/kubernetes.io/serviceaccount/ca.crt'
            },
            'bearer_token_file': '/var/run/secrets/kubernetes.io/serviceaccount/token',
            'kubernetes_sd_configs': [{
                'role': 'node'
            }],
            'relabel_configs': [{
                'action': 'labelmap',
                'regex': '__meta_kubernetes_node_label_(.+)'
            }]
        },
        {
            'job_name': 'kubernetes-service-endpoints',
            'kubernetes_sd_configs': [{
                'role': 'endpoints'
            }],
            'relabel_configs': [{
                'action': 'labelmap',
                'regex': '__meta_kubernetes_service_label_(.+)'
            },
            {
                'source_labels': ['__meta_kubernetes_namespace'],
                'action': 'replace',
                'target_label': 'kubernetes_namespace'
            },
            {
                'source_labels': ['__meta_kubernetes_service_name'],
                'action': 'replace',
                'target_label': 'kubernetes_name'
            }]
        },
        {
            'job_name': 'kubernetes-services',
            'metrics_path': '/probe',
            'params': {
                'module': ['http_2xx']
            },
            'kubernetes_sd_configs': [{
                'role': 'service'
            }],
            'relabel_configs': [{
                'source_labels': ['__address__'],
                'target_label': '__param_target'
            },
            {
                'target_label': '__address__',
                'replacement': 'blackbox-exporter.example.com:9115'
            },
            {
                'source_labels': ['__param_target'],
                'target_label': 'instance'
            },
            {
                'action': 'labelmap',
                'regex': '__meta_kubernetes_service_label_(.+)'
            },
            {
                'source_labels': ['__meta_kubernetes_namespace'],
                'target_label': 'kubernetes_namespace'
            },
            {
                'source_labels': ['__meta_kubernetes_service_name'],
                'target_label': 'kubernetes_name'
            }]
        },
        {
            'job_name': 'kubernetes-ingresses',
            'metrics_path': '/probe',
            'params': {
                'module': ['http_2xx']
            },
            'kubernetes_sd_configs': [{
                'role': 'ingress'
            }],
            'relabel_configs': [{
                'source_labels': ['__meta_kubernetes_ingress_scheme',
                                  '__address__',
                                  '__meta_kubernetes_ingress_path'],
                'regex': '(.+);(.+);(.+)',
                'replacement': '${1}://${2}${3}',
                'target_label': '__param_target'
            },
            {
                'target_label': '__address__',
                'replacement': 'blackbox-exporter.example.com:9115'
            },
            {
                'source_labels': ['__param_target'],
                'target_label': 'instance'
            },
            {
                'action': 'labelmap',
                'regex': '__meta_kubernetes_ingress_label_(.+)'
            },
            {
                'source_labels': ['__meta_kubernetes_namespace'],
                'target_label': 'kubernetes_namespace'
            },
            {
                'source_labels': ['__meta_kubernetes_ingress_name'],
                'target_label': 'kubernetes_name'
            }]
        },
        {
            'job_name': 'kubernetes-pods',
            'kubernetes_sd_configs': [{
                'role': 'pod'
            }],
            'relabel_configs': [{
                'action': 'labelmap',
                'regex': '__meta_kubernetes_pod_label_(.+)'
            },
            {
                'source_labels': ['__meta_kubernetes_namespace'],
                'action': 'replace',
                'target_label': 'kubernetes_namespace'
            },
            {
                'source_labels': ['__meta_kubernetes_pod_name'],
                'action': 'replace',
                'target_label': 'kubernetes_pod_name'
            }]
        }])
