"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeType = exports.ElasticacheAutoMonitor = void 0;
const cdk = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const sns_sub = require("@aws-cdk/aws-sns-subscriptions");
const actions = require("@aws-cdk/aws-cloudwatch-actions");
const cpu_monitor_1 = require("./cpu-monitor");
const swap_usage_monitor_1 = require("./swap-usage-monitor");
const evictions_monitor_1 = require("./evictions-monitor");
const current_connections_monitor_1 = require("./current-connections-monitor");
const freeable_memory_monitor_1 = require("./freeable-memory-monitor");
/**
 * ElasticacheAutoMonitor allows you to send email, sms, slack, or trigger aws sns topic when an alarm occurs.
 * You will get the following monitoring:
 *  1. Cpu Monitor: Should be less than 90%. (See below reference)
 *  2. SwapUsage Monitor: Should be less than 50M.
 *  3. Evictions Monitor: Should not have evictions value.
 *  4. CurrConnections Monitor: According to your business needs, default every 1 vcup is equal to 200 connections.
 *  5. FreeableMemory Monitor: Not less than 10%
 *
 * Reference: https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/CacheMetrics.WhichShouldIMonitor.html
 */
class ElasticacheAutoMonitor extends cdk.Construct {
    static setUpWithLambda(scope, cacheClusterId, props) {
        const topic = new sns.Topic(scope, 'alarm-' + cacheClusterId + '-lambda');
        topic.addSubscription(new sns_sub.LambdaSubscription(props.lambda));
        ElasticacheAutoMonitor.setup(scope, cacheClusterId, topic, props);
    }
    static setUpWithSlack(scope, cacheClusterId, props) {
        let channel = props.channel || 'cloudwatch-alarm';
        let username = props.username || 'Webhookbot';
        let icon_emoji = props.iconEmoji || ':scream:';
        const fn = new lambda.Function(scope, 'alarm-' + cacheClusterId + '-to-slack', {
            handler: 'index.handler',
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset('./src/asset/lambda'),
            environment: {
                SLACK_WEBHOOK_URL: props.webHookUrl,
                CHANNEL: channel,
                USERNAME: username,
                ICON_EMOJI: icon_emoji,
            },
        });
        const topic = new sns.Topic(scope, 'alarm-' + cacheClusterId + '-slack');
        topic.addSubscription(new sns_sub.LambdaSubscription(fn));
        ElasticacheAutoMonitor.setup(scope, cacheClusterId, topic, props);
    }
    static setUpWithEmail(scope, cacheClusterId, props) {
        const topic = new sns.Topic(scope, 'alarm-' + cacheClusterId + '-with-email-topic');
        for (let email in props.emails) {
            topic.addSubscription(new sns_sub.EmailSubscription(email));
        }
        ElasticacheAutoMonitor.setup(scope, cacheClusterId, topic, props);
    }
    static setUpWithSms(scope, cacheClusterId, props) {
        const topic = new sns.Topic(scope, 'alarm-' + cacheClusterId + '-with-sms-topic');
        for (let phone in props.phones) {
            topic.addSubscription(new sns_sub.SmsSubscription(phone));
        }
        ElasticacheAutoMonitor.setup(scope, cacheClusterId, topic, props);
    }
    static setup(scope, cacheClusterId, topic, props) {
        // bind default and props
        const nodeType = props.nodeType || NodeType.DEFAULT;
        const _props = {
            nodeType,
            currConnectionsPeriod: cdk.Duration.minutes(1),
            currConnectionsThreshold: nodeType.vcupCount * 200,
            ...props,
        };
        cpu_monitor_1.CpuMonitor.setup(scope, cacheClusterId, _props.nodeType).addAlarmAction(new actions.SnsAction(topic));
        swap_usage_monitor_1.SwapUsageMomiton.setup(scope, cacheClusterId).addAlarmAction(new actions.SnsAction(topic));
        evictions_monitor_1.EvictionsMomiton.setup(scope, cacheClusterId).addAlarmAction(new actions.SnsAction(topic));
        current_connections_monitor_1.CurrConnectionsMomiton.setup(scope, cacheClusterId, _props.currConnectionsPeriod, _props.currConnectionsThreshold).addAlarmAction(new actions.SnsAction(topic));
        freeable_memory_monitor_1.FreeableMemoryMomiton.setup(scope, cacheClusterId, _props.nodeType).addAlarmAction(new actions.SnsAction(topic));
    }
}
exports.ElasticacheAutoMonitor = ElasticacheAutoMonitor;
class NodeType {
    constructor(name, vcupCount, memory) {
        this.name = name;
        this.vcupCount = vcupCount;
        this.memory = memory;
    }
}
exports.NodeType = NodeType;
NodeType.T2_MIRCO = new NodeType('cache.t2.mirco', 1, 0.555 * 1024 * 1024 * 1024);
NodeType.T2_SMALL = new NodeType('cache.t2.small', 1, 1.55 * 1024 * 1024 * 1024);
NodeType.T2_MEDIUM = new NodeType('cache.t2.medium', 2, 3.22 * 1024 * 1024 * 1024);
NodeType.T3_MICRO = new NodeType('cache.t3.micro', 2, 0.5 * 1024 * 1024 * 1024);
NodeType.T3_SMALL = new NodeType('cache.t3.small', 2, 1.37 * 1024 * 1024 * 1024);
NodeType.T3_MEDIUM = new NodeType('cache.t3.medium', 2, 3.09 * 1024 * 1024 * 1024);
NodeType.M4_LARGE = new NodeType('cache.m4.large', 2, 6.42 * 1024 * 1024 * 1024);
NodeType.M4_XLARGE = new NodeType('cache.m4.xlarge', 4, 14.28 * 1024 * 1024 * 1024);
NodeType.M4_2XLARGE = new NodeType('cache.m4.2xlarge', 8, 29.70 * 1024 * 1024 * 1024);
NodeType.M4_4XLARGE = new NodeType('cache.m4.4xlarge', 16, 60.78 * 1024 * 1024 * 1024);
NodeType.M4_10XLARGE = new NodeType('cache.m4.10xlarge', 40, 154.64 * 1024 * 1024 * 1024);
NodeType.M5_LARGE = new NodeType('cache.m5.large', 2, 6.38 * 1024 * 1024 * 1024);
NodeType.M5_XLARGE = new NodeType('cache.m5.xlarge', 4, 12.93 * 1024 * 1024 * 1024);
NodeType.M5_2XLARGE = new NodeType('cache.m5.2xlarge', 8, 26.04 * 1024 * 1024 * 1024);
NodeType.M5_4XLARGE = new NodeType('cache.m5.4xlarge', 16, 52.26 * 1024 * 1024 * 1024);
NodeType.M5_12XLARGE = new NodeType('cache.m5.12xlarge', 48, 157.12 * 1024 * 1024 * 1024);
NodeType.M5_24XLARGE = new NodeType('cache.m5.12xlarge', 96, 314.32 * 1024 * 1024 * 1024);
NodeType.R4_LARGE = new NodeType('cache.r4.large', 2, 12.3 * 1024 * 1024 * 1024);
NodeType.R4_XLARGE = new NodeType('cache.r4.xlarge', 4, 25.05 * 1024 * 1024 * 1024);
NodeType.R4_2XLARGE = new NodeType('cache.r4.2xlarge', 8, 50.47 * 1024 * 1024 * 1024);
NodeType.R4_4XLARGE = new NodeType('cache.r4.4xlarge', 16, 101.38 * 1024 * 1024 * 1024);
NodeType.R4_8XLARGE = new NodeType('cache.r4.8xlarge', 32, 203.26 * 1024 * 1024 * 1024);
NodeType.R4_16XLARGE = new NodeType('cache.r4.16xlarge', 64, 407 * 1024 * 1024 * 1024);
NodeType.R5_LARGE = new NodeType('cache.r5.large', 2, 13.07 * 1024 * 1024 * 1024);
NodeType.R5_XLARGE = new NodeType('cache.r5.xlarge', 4, 26.32 * 1024 * 1024 * 1024);
NodeType.R5_2XLARGE = new NodeType('cache.r5.2xlarge', 8, 52.82 * 1024 * 1024 * 1024);
NodeType.R5_4XLARGE = new NodeType('cache.r5.4xlarge', 16, 105.81 * 1024 * 1024 * 1024);
NodeType.R5_12XLARGE = new NodeType('cache.r5.12xlarge', 48, 317.77 * 1024 * 1024 * 1024);
NodeType.R5_24LARGE = new NodeType('cache.r5.12xlarge', 96, 635.61 * 1024 * 1024 * 1024);
NodeType.DEFAULT = NodeType.M5_LARGE;
//# sourceMappingURL=data:application/json;base64,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