import * as cdk from '@aws-cdk/core';
import * as lambda from '@aws-cdk/aws-lambda';
/**
 * Elasticache auto monitor set up base props.
 */
interface ISetUpProps {
    /**
     * Default elasticache node type.
     * It is strongly recommended to set according to the actual value.
     *
     * @default NodeType.M5_LARGE
     */
    readonly nodeType?: NodeType;
    /**
     * The period over which the specified statistic is applied.
     *
     * @default Duration.minutes(1)
     */
    readonly currConnectionsPeriod?: cdk.Duration;
    /**
     * CurrConnections threshold
     *
     * Every 1 vcup is equal to 50 connections
     *
     * @default 100
     */
    readonly currConnectionsThreshold?: number;
}
/**
 * Elasticache auto monitor set up with labmda props.
 */
export interface ISetUpWithLambdaProps extends ISetUpProps {
    readonly lambda: lambda.Function;
}
export interface ISetUpWithSlackProps extends ISetUpProps {
    /**
     * Go to this(https://slack.com/apps/A0F7XDUAZ-incoming-webhooks) link to apply for webhook
     */
    readonly webHookUrl: string;
    /**
     * Setting channel.
     * What is a channel: (https://slack.com/intl/en-cn/help/articles/360017938993-What-is-a-channel)
     *
     * @default cloudwatch-alarm
     */
    readonly channel?: string;
    /**
     * Setting Slack bot name.
     *
     * @default Webhookbot
     */
    readonly username?: string;
    /**
     * Emoji for bot.
     *
     * @default :scream:
     */
    readonly iconEmoji?: string;
}
export interface ISetUpWithEmailProps extends ISetUpProps {
    /**
     * Mailing list to be sent.
     * Note that email subscriptions require confirmation by visiting the link sent to the email address.
     */
    readonly emails: string[];
}
export interface ISetUpWithSmsProps extends ISetUpProps {
    /**
     * Include country code and phone number, for example: +15551231234
     */
    readonly phones: string[];
}
/**
 * ElasticacheAutoMonitor allows you to send email, sms, slack, or trigger aws sns topic when an alarm occurs.
 * You will get the following monitoring:
 *  1. Cpu Monitor: Should be less than 90%. (See below reference)
 *  2. SwapUsage Monitor: Should be less than 50M.
 *  3. Evictions Monitor: Should not have evictions value.
 *  4. CurrConnections Monitor: According to your business needs, default every 1 vcup is equal to 200 connections.
 *  5. FreeableMemory Monitor: Not less than 10%
 *
 * Reference: https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/CacheMetrics.WhichShouldIMonitor.html
 */
export declare class ElasticacheAutoMonitor extends cdk.Construct {
    static setUpWithLambda(scope: cdk.Construct, cacheClusterId: string, props: ISetUpWithLambdaProps): void;
    static setUpWithSlack(scope: cdk.Construct, cacheClusterId: string, props: ISetUpWithSlackProps): void;
    static setUpWithEmail(scope: cdk.Construct, cacheClusterId: string, props: ISetUpWithEmailProps): void;
    static setUpWithSms(scope: cdk.Construct, cacheClusterId: string, props: ISetUpWithSmsProps): void;
    private static setup;
}
export declare class NodeType {
    readonly name: string;
    readonly vcupCount: number;
    readonly memory: number;
    private constructor();
    static readonly T2_MIRCO: NodeType;
    static readonly T2_SMALL: NodeType;
    static readonly T2_MEDIUM: NodeType;
    static readonly T3_MICRO: NodeType;
    static readonly T3_SMALL: NodeType;
    static readonly T3_MEDIUM: NodeType;
    static readonly M4_LARGE: NodeType;
    static readonly M4_XLARGE: NodeType;
    static readonly M4_2XLARGE: NodeType;
    static readonly M4_4XLARGE: NodeType;
    static readonly M4_10XLARGE: NodeType;
    static readonly M5_LARGE: NodeType;
    static readonly M5_XLARGE: NodeType;
    static readonly M5_2XLARGE: NodeType;
    static readonly M5_4XLARGE: NodeType;
    static readonly M5_12XLARGE: NodeType;
    static readonly M5_24XLARGE: NodeType;
    static readonly R4_LARGE: NodeType;
    static readonly R4_XLARGE: NodeType;
    static readonly R4_2XLARGE: NodeType;
    static readonly R4_4XLARGE: NodeType;
    static readonly R4_8XLARGE: NodeType;
    static readonly R4_16XLARGE: NodeType;
    static readonly R5_LARGE: NodeType;
    static readonly R5_XLARGE: NodeType;
    static readonly R5_2XLARGE: NodeType;
    static readonly R5_4XLARGE: NodeType;
    static readonly R5_12XLARGE: NodeType;
    static readonly R5_24LARGE: NodeType;
    static readonly DEFAULT: NodeType;
}
export {};
