# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['stix2xspec', 'stix2xspec.data']

package_data = \
{'': ['*']}

install_requires = \
['rich>=10.14.0,<11.0.0', 'typer[all]>=0.4.0,<0.5.0']

extras_require = \
{':python_version < "3.8"': ['importlib_metadata>=4.5.0,<5.0.0']}

entry_points = \
{'console_scripts': ['stix2xspec = stix2xspec.__main__:app']}

setup_kwargs = {
    'name': 'stix2xspec',
    'version': '0.1.0',
    'description': 'Convert STIX science data (L1A, L1, or L4 spectrograms or pixel data) to a format compatible with XSPEC',
    'long_description': '# stix2xspec\n\n<div align="center">\n\n[![Build status](https://github.com/stix2xspec/stix2xspec/workflows/build/badge.svg?branch=master&event=push)](https://github.com/stix2xspec/stix2xspec/actions?query=workflow%3Abuild)\n[![Python Version](https://img.shields.io/pypi/pyversions/stix2xspec.svg)](https://pypi.org/project/stix2xspec/)\n[![Dependencies Status](https://img.shields.io/badge/dependencies-up%20to%20date-brightgreen.svg)](https://github.com/stix2xspec/stix2xspec/pulls?utf8=%E2%9C%93&q=is%3Apr%20author%3Aapp%2Fdependabot)\n\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![Security: bandit](https://img.shields.io/badge/security-bandit-green.svg)](https://github.com/PyCQA/bandit)\n[![Pre-commit](https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white)](https://github.com/stix2xspec/stix2xspec/blob/master/.pre-commit-config.yaml)\n[![Semantic Versions](https://img.shields.io/badge/%20%20%F0%9F%93%A6%F0%9F%9A%80-semantic--versions-e10079.svg)](https://github.com/stix2xspec/stix2xspec/releases)\n[![License](https://img.shields.io/github/license/stix2xspec/stix2xspec)](https://github.com/stix2xspec/stix2xspec/blob/master/LICENSE)\n![Coverage Report](assets/images/coverage.svg)\n\nConvert STIX science data (L1A, L1, or L4 spectrograms or pixel data) to a format compatible with XSPEC\n\n</div>\n\n<!--\n\n## Very first steps\n\n### Initialize your code\n\n1. Initialize `git` inside your repo:\n\n```bash\ncd stix2xspec && git init\n```\n\n2. If you don\'t have `Poetry` installed run:\n\n```bash\nmake poetry-download\n```\n\n3. Initialize poetry and install `pre-commit` hooks:\n\n```bash\nmake install\nmake pre-commit-install\n```\n\n4. Run the codestyle:\n\n```bash\nmake codestyle\n```\n\n5. Upload initial code to GitHub:\n\n```bash\ngit add .\ngit commit -m ":tada: Initial commit"\ngit branch -M main\ngit remote add origin https://github.com/stix2xspec/stix2xspec.git\ngit push -u origin main\n```\n\n### Set up bots\n\n- Set up [Dependabot](https://docs.github.com/en/github/administering-a-repository/enabling-and-disabling-version-updates#enabling-github-dependabot-version-updates) to ensure you have the latest dependencies.\n- Set up [Stale bot](https://github.com/apps/stale) for automatic issue closing.\n\n### Poetry\n\nWant to know more about Poetry? Check [its documentation](https://python-poetry.org/docs/).\n\n<details>\n<summary>Details about Poetry</summary>\n<p>\n\nPoetry\'s [commands](https://python-poetry.org/docs/cli/#commands) are very intuitive and easy to learn, like:\n\n- `poetry add numpy@latest`\n- `poetry run pytest`\n- `poetry publish --build`\n\netc\n</p>\n</details>\n\n### Building and releasing your package\n\nBuilding a new version of the application contains steps:\n\n- Bump the version of your package `poetry version <version>`. You can pass the new version explicitly, or a rule such as `major`, `minor`, or `patch`. For more details, refer to the [Semantic Versions](https://semver.org/) standard.\n- Make a commit to `GitHub`.\n- Create a `GitHub release`.\n- And... publish 🙂 `poetry publish --build`\n\n## 🎯 What\'s next\n\nWell, that\'s up to you 💪🏻. I can only recommend the packages and articles that helped me.\n\n- [`Typer`](https://github.com/tiangolo/typer) is great for creating CLI applications.\n- [`Rich`](https://github.com/willmcgugan/rich) makes it easy to add beautiful formatting in the terminal.\n- [`Pydantic`](https://github.com/samuelcolvin/pydantic/) – data validation and settings management using Python type hinting.\n- [`Loguru`](https://github.com/Delgan/loguru) makes logging (stupidly) simple.\n- [`tqdm`](https://github.com/tqdm/tqdm) – fast, extensible progress bar for Python and CLI.\n- [`IceCream`](https://github.com/gruns/icecream) is a little library for sweet and creamy debugging.\n- [`orjson`](https://github.com/ijl/orjson) – ultra fast JSON parsing library.\n- [`Returns`](https://github.com/dry-python/returns) makes you function\'s output meaningful, typed, and safe!\n- [`Hydra`](https://github.com/facebookresearch/hydra) is a framework for elegantly configuring complex applications.\n- [`FastAPI`](https://github.com/tiangolo/fastapi) is a type-driven asynchronous web framework.\n\nArticles:\n\n- [Open Source Guides](https://opensource.guide/).\n- [A handy guide to financial support for open source](https://github.com/nayafia/lemonade-stand)\n- [GitHub Actions Documentation](https://help.github.com/en/actions).\n- Maybe you would like to add [gitmoji](https://gitmoji.carloscuesta.me/) to commit names. This is really funny. 😄\n\n## 🚀 Features\n\n### Development features\n\n- Supports for `Python 3.7` and higher.\n- [`Poetry`](https://python-poetry.org/) as the dependencies manager. See configuration in [`pyproject.toml`](https://github.com/stix2xspec/stix2xspec/blob/master/pyproject.toml) and [`setup.cfg`](https://github.com/stix2xspec/stix2xspec/blob/master/setup.cfg).\n- Automatic codestyle with [`black`](https://github.com/psf/black), [`isort`](https://github.com/timothycrosley/isort) and [`pyupgrade`](https://github.com/asottile/pyupgrade).\n- Ready-to-use [`pre-commit`](https://pre-commit.com/) hooks with code-formatting.\n- Type checks with [`mypy`](https://mypy.readthedocs.io); docstring checks with [`darglint`](https://github.com/terrencepreilly/darglint); security checks with [`safety`](https://github.com/pyupio/safety) and [`bandit`](https://github.com/PyCQA/bandit)\n- Testing with [`pytest`](https://docs.pytest.org/en/latest/).\n- Ready-to-use [`.editorconfig`](https://github.com/stix2xspec/stix2xspec/blob/master/.editorconfig), [`.dockerignore`](https://github.com/stix2xspec/stix2xspec/blob/master/.dockerignore), and [`.gitignore`](https://github.com/stix2xspec/stix2xspec/blob/master/.gitignore). You don\'t have to worry about those things.\n\n### Deployment features\n\n- `GitHub` integration: issue and pr templates.\n- `Github Actions` with predefined [build workflow](https://github.com/stix2xspec/stix2xspec/blob/master/.github/workflows/build.yml) as the default CI/CD.\n- Everything is already set up for security checks, codestyle checks, code formatting, testing, linting, docker builds, etc with [`Makefile`](https://github.com/stix2xspec/stix2xspec/blob/master/Makefile#L89). More details in [makefile-usage](#makefile-usage).\n- [Dockerfile](https://github.com/stix2xspec/stix2xspec/blob/master/docker/Dockerfile) for your package.\n- Always up-to-date dependencies with [`@dependabot`](https://dependabot.com/). You will only [enable it](https://docs.github.com/en/github/administering-a-repository/enabling-and-disabling-version-updates#enabling-github-dependabot-version-updates).\n- Automatic drafts of new releases with [`Release Drafter`](https://github.com/marketplace/actions/release-drafter). You may see the list of labels in [`release-drafter.yml`](https://github.com/stix2xspec/stix2xspec/blob/master/.github/release-drafter.yml). Works perfectly with [Semantic Versions](https://semver.org/) specification.\n\n### Open source community features\n\n- Ready-to-use [Pull Requests templates](https://github.com/stix2xspec/stix2xspec/blob/master/.github/PULL_REQUEST_TEMPLATE.md) and several [Issue templates](https://github.com/stix2xspec/stix2xspec/tree/master/.github/ISSUE_TEMPLATE).\n- Files such as: `LICENSE`, `CONTRIBUTING.md`, `CODE_OF_CONDUCT.md`, and `SECURITY.md` are generated automatically.\n- [`Stale bot`](https://github.com/apps/stale) that closes abandoned issues after a period of inactivity. (You will only [need to setup free plan](https://github.com/marketplace/stale)). Configuration is [here](https://github.com/stix2xspec/stix2xspec/blob/master/.github/.stale.yml).\n- [Semantic Versions](https://semver.org/) specification with [`Release Drafter`](https://github.com/marketplace/actions/release-drafter).\n-->\n\n## Installation\n\n```bash\npip install -U stix2xspec\n```\n\nor install with `Poetry`\n\n```bash\npoetry add stix2xspec\n```\n\nThen you can run\n\n```bash\nstix2xspec --help\n```\n\nor with `Poetry`:\n\n```bash\npoetry run stix2xspec --help\n```\n\n## Dependencies\n\nThis software requires the files stored in [STIX-CONF](https://github.com/i4Ds/STIX-CONF). Set up an environment variable that points to this directory.\n\nThese files can also be found in the [IDL ground software](https://github.com/i4Ds/STIX-GSW) _dbase_ directory.  \n\n```bash\nexport STX_CONF=/path/to/STIX-CONF\n```\n\n## Example - Background-subtract and convert a STIX FITS file \n\n### [Try it out in CoLab](https://colab.research.google.com/drive/1bXTpKqWAwyp92lM9alSqrAPZtRw2ocMT?usp=sharing)\n\nDownload a FITS file from the [STIX Data Center (SDC)](https://datacenter.stix.i4ds.net/). More details about STIX data products, along with tutorials, can be found on the [STIX wiki](https://datacenter.stix.i4ds.net/wiki/index.php?title=STIX_Data_Products). Official science data products are level 1 (L1). These can be found on the [STIX archive server](http://dataarchive.stix.i4ds.net/data/fits/). \n\nPre-release data products (L1A) are supported by this software but not officially recommended for use. There are two types of such data, spectrogram data and pixel data. Spectrogram data have _stix-sci-spectrogram_ in their filename and can be searched for using _product_type=\'xray-spec\'_ in  [stixdcpy FitsQuery](https://github.com/i4Ds/stix0dcpy ). They tend to cover very long time intervals. Pixel data have _stix-sci-xray_ in the filename, and are most often generated for single events (solar flares). Both kinds are returned when searching with _product_type=\'l1\'_. \n\nDownload the corresponding background file. Background files can be found by using the filter _filter=\'BKG\'_ or by looking at the file description on the SDC.\n\nSubtracting the background and converting the file to an OGIP-compatible spectrogram is done via the following:\n\n```python\nfrom stix2xspec.stix2xspec import convert_spectrogram\n\nfitsfile = \'solo_L1A_stix-sci-spectrogram-2207238956_20220723T122007-20220723T182511_079258_V01.fits\' # full path\nbgfile = \'solo_L1A_stix-sci-xray-l1-2207235029_20220723T113947-20220723T122747_079205_V01.fits\'\noutfile = convert_spectrogram(fitsfile, bgfile, to_fits = True)\n```\n\n![](spectrogram_readme.png)\n\n**Figure 1**: A portion of the converted spectrogram\n\n\n![](spectrogram_idl_readme.png)\n\n**Figure 2**: The same portion of the same input and background files, converted using the official IDL STIX ground software (_convert_spectrogram.pro_)\n\n\nAlong with the important data processing steps of applying the error lookup table (ELUT) and performing livetime-correction, background subtraction is performed, counts are converted to count rate and an energy-dependent systematic error term is generated, which is useful when using XSPEC. Any necessary FITS header quantities are calculated and added to the existing header as needed.\n\nA .srm file containing the spectral response matrix is also written. This is not yet generated via the appropriate calculations; rather, an existing .srm file is edited to match the energy channels contained in the input file. The STIX spectral response matrix is relatively stable over time, but it can be generated using the official [IDL ground software](https://github.com/i4Ds/STIX-GSW).  \n\n## Example - apply ELUT and livetime correction to spectrogram or pixel data\n\nData processing can be performed with or without the final step of conversion to count rate.\n\n```python\nfrom stix2xspec.spectrogram import Spectrogram \n\nspec = Spectrogram(fitsfile)\nspec.apply_elut()\nspec.correct_counts()\n```\n\nCounts can be converted to count rate:\n\n```python\nspec.to_rate()\n```\n\nThe same can be done for background files:\n\n```python\nfrom stix2xspec.spectrogram import Spectrogram \n\nspec_bg = Spectrogram(bgfile, background = True, use_discriminators = False)\nspec_bg.apply_elut()\nspec_bg.correct_counts()\n```\n\n## Example - fit STIX spectrum with solar-specific models in XSPEC\n\nThis requires additional installation of  [sunpy/sunxspex](https://github.com/sunpy/sunxspex) and of course [XSPEC](https://heasarc.gsfc.nasa.gov/xanadu/xspec/), which comes together with [pyxspec](https://heasarc.gsfc.nasa.gov/xanadu/xspec/python/html/index.html). For now, the XSPEC solar models are found in [this fork of sunxspex](https://github.com/elastufka/sunxspex/tree/xspec_functions). \n\nBe sure to enable XSPEC via command line before starting a Python session.\n\n```bash\n. $HEADAS/headas-init.sh\n```\n\nAdd the thermal bremsstrahlung model _vth_ and the non-thermal thick-target bremsstrahlung model _bremsstrahlung_thick_target_ to XSPEC, then fit them first individually and then together.\n\n```python \nimport xspec\nfrom stix2xspec.xspec_utils import *\nfrom sunxspex import xspec_models\n\nmod_th = sunxspex.xspec_models.ThermalModel()\nxspec.AllModels.addPyMod(mod_th.model, mod_th.ParInfo, \'add\')\nmod_th.print_ParInfo() # see the initial configuration of parameters\n\nmod_nt = sunxspex.xspec_models.ThickTargetModel()\nxspec.AllModels.addPyMod(mod_nt.model, mod_nt.ParInfo, \'add\')\nmod_nt.print_ParInfo() # see the initial configuration of parameters\n\nxspec.AllData.clear() # get rid of any data that is still loaded from previous runs\nxspec.AllData(f"1:1 {\'stx_spectrum_20220723_122031.fits\'}{{1140}}") # fit the 1140th data row in the converted spectrogram file. make sure the .srm file is in the same folder as the spectrogram file.\n\nspectime = fits_time_to_datetime(\'stx_spectrum_20220723_122031.fits\', idx=1140)\nplot_data(xspec, erange = [4,50],title = f\'STIX spectrum at {spectime:%Y-%m-%d %H:%M:%S}\').show()\n```\n![](spectrum_readme.png)\n\nIf desired, a time interval rather than a single row (time bin) of the spectrogram can be chosen for fitting. A new FITS file containing only one row must be generated.\n\n```python\nspectrum_from_time_interval(stx_spectrum_20220723_122031.fits, \'2022-07-23T17:55:00\', \'2022-07-23T18:05:00\', out_fitsname=\'stx_spectrum_integrated.fits\')\n```\nThen load this data using the usual pyxspec commands.\n\nFitting with a thermal and/or non-thermal solar model can easily be done with the following. Other commonly used models native to Xspec are:\n\n- [apec]() \n- [powerlaw]()\n- [bknpower]()\n\n```python\nmodel, chisq = fit_thermal_nonthermal(xspec, thmodel = \'vth\', ntmodel = \'bremsstrahlung_thick_target\', lowErange = [3,10])\n```\n\nXSPEC will display fitted model parameters either in the terminal or directly in the Python/Jupyter session, depending on how standard output is configured. You can also print and plot using the following commands (requires pandas and plotLy).\n\n```python\nshow_model(model, df=True)\n```\n| Model par | Model comp | Component | Parameter | Unit | Value | Sigma |\n| :---: | :---: | :---: | --- |--- | --- | --- |\n|1 |1 | vth|EM| 1e49| 0.30 | frozen| \n|2 |1 | vth|kT| keV| 1.88 | ± 0.26| \n|3 |1 | vth|abund| | 1.00 | frozen| \n|4 |1 | vth|norm| | 1.86e-03 | ± 7.77e-04| \n|5 |2 | bremsstrahlung_thick_target|p| | 2.65 | ± 346.01| \n|6 |2 | bremsstrahlung_thick_target|eebrk| keV| 16.93 | ± 421.75| \n|7 |2 | bremsstrahlung_thick_target|q| | 6.64 | ± 71.89| \n|8 |2 | bremsstrahlung_thick_target|eelow| keV| 10.43 | ± 176.56| \n|9 |2 | bremsstrahlung_thick_target|eehigh| keV| 1.00e+07 | ± 3.39e+09| \n|10 |2 | bremsstrahlung_thick_target|norm| | 0.57 | ± 11.83|\n\n```\nfig = plot_fit(xspec, model, fitrange = [3,30])\nfittext = annotate_plot(model, chisq=chisq, exclude_parameters = [\'norm\',\'Abundanc\',\'Redshift\'], MK=True)\nfig.update_layout(width=650, yaxis_range = [-2,3])\nfig.add_annotation(x=1.75,y=.5,text=fittext,xref=\'paper\',yref=\'paper\', showarrow = False)\nfig.show()\n```\n![](specfit_readme.png)\n\n<!--\n## Example - Convert STIX FITS file and background file independently\n\nno use case for this at the moment\n-->\n\n<!--\n### Makefile usage\n\n[`Makefile`](https://github.com/stix2xspec/stix2xspec/blob/master/Makefile) contains a lot of functions for faster development.\n\n<details>\n<summary>1. Download and remove Poetry</summary>\n<p>\n\nTo download and install Poetry run:\n\n```bash\nmake poetry-download\n```\n\nTo uninstall\n\n```bash\nmake poetry-remove\n```\n\n</p>\n</details>\n\n<details>\n<summary>2. Install all dependencies and pre-commit hooks</summary>\n<p>\n\nInstall requirements:\n\n```bash\nmake install\n```\n\nPre-commit hooks coulb be installed after `git init` via\n\n```bash\nmake pre-commit-install\n```\n\n</p>\n</details>\n\n<details>\n<summary>3. Codestyle</summary>\n<p>\n\nAutomatic formatting uses `pyupgrade`, `isort` and `black`.\n\n```bash\nmake codestyle\n\n# or use synonym\nmake formatting\n```\n\nCodestyle checks only, without rewriting files:\n\n```bash\nmake check-codestyle\n```\n\n> Note: `check-codestyle` uses `isort`, `black` and `darglint` library\n\nUpdate all dev libraries to the latest version using one comand\n\n```bash\nmake update-dev-deps\n```\n\n<details>\n<summary>4. Code security</summary>\n<p>\n\n```bash\nmake check-safety\n```\n\nThis command launches `Poetry` integrity checks as well as identifies security issues with `Safety` and `Bandit`.\n\n```bash\nmake check-safety\n```\n\n</p>\n</details>\n\n</p>\n</details>\n\n<details>\n<summary>5. Type checks</summary>\n<p>\n\nRun `mypy` static type checker\n\n```bash\nmake mypy\n```\n\n</p>\n</details>\n\n<details>\n<summary>6. Tests with coverage badges</summary>\n<p>\n\nRun `pytest`\n\n```bash\nmake test\n```\n\n</p>\n</details>\n\n<details>\n<summary>7. All linters</summary>\n<p>\n\nOf course there is a command to ~~rule~~ run all linters in one:\n\n```bash\nmake lint\n```\n\nthe same as:\n\n```bash\nmake test && make check-codestyle && make mypy && make check-safety\n```\n\n</p>\n</details>\n\n<details>\n<summary>8. Docker</summary>\n<p>\n\n```bash\nmake docker-build\n```\n\nwhich is equivalent to:\n\n```bash\nmake docker-build VERSION=latest\n```\n\nRemove docker image with\n\n```bash\nmake docker-remove\n```\n\nMore information [about docker](https://github.com/stix2xspec/stix2xspec/tree/master/docker).\n\n</p>\n</details>\n\n<details>\n<summary>9. Cleanup</summary>\n<p>\nDelete pycache files\n\n```bash\nmake pycache-remove\n```\n\nRemove package build\n\n```bash\nmake build-remove\n```\n\nDelete .DS_STORE files\n\n```bash\nmake dsstore-remove\n```\n\nRemove .mypycache\n\n```bash\nmake mypycache-remove\n```\n\nOr to remove all above run:\n\n```bash\nmake cleanup\n```\n\n</p> -->\n</details>\n\n## 📈 Releases\n\nYou can see the list of available releases on the [GitHub Releases](https://github.com/stix2xspec/stix2xspec/releases) page.\n\nWe follow [Semantic Versions](https://semver.org/) specification.\n\nWe use [`Release Drafter`](https://github.com/marketplace/actions/release-drafter). As pull requests are merged, a draft release is kept up-to-date listing the changes, ready to publish when you’re ready. With the categories option, you can categorize pull requests in release notes using labels.\n\n### List of labels and corresponding titles\n\n|               **Label**               |  **Title in Releases**  |\n| :-----------------------------------: | :---------------------: |\n|       `enhancement`, `feature`        |       🚀 Features       |\n| `bug`, `refactoring`, `bugfix`, `fix` | 🔧 Fixes & Refactoring  |\n|       `build`, `ci`, `testing`        | 📦 Build System & CI/CD |\n|              `breaking`               |   💥 Breaking Changes   |\n|            `documentation`            |    📝 Documentation     |\n|            `dependencies`             | ⬆️ Dependencies updates |\n\nYou can update it in [`release-drafter.yml`](https://github.com/stix2xspec/stix2xspec/blob/master/.github/release-drafter.yml).\n\nGitHub creates the `bug`, `enhancement`, and `documentation` labels for you. Dependabot creates the `dependencies` label. Create the remaining labels on the Issues tab of your GitHub repository, when you need them.\n\n## 🛡 License\n\n[![License](https://img.shields.io/github/license/stix2xspec/stix2xspec)](https://github.com/stix2xspec/stix2xspec/blob/master/LICENSE)\n\nThis project is licensed under the terms of the `GNU GPL v3.0` license. See [LICENSE](https://github.com/stix2xspec/stix2xspec/blob/master/LICENSE) for more details.\n\n## 📃 Citation\n\n```bibtex\n@misc{stix2xspec,\n  author = {stix2xspec},\n  title = {Convert STIX science data (L1A, L1, or L4 spectrograms or pixel data) to a format compatible with XSPEC},\n  year = {2022},\n  publisher = {GitHub},\n  journal = {GitHub repository},\n  howpublished = {\\url{https://github.com/stix2xspec/stix2xspec}}\n}\n```\n\n## Credits [![🚀 Your next Python package needs a bleeding-edge project structure.](https://img.shields.io/badge/python--package--template-%F0%9F%9A%80-brightgreen)](https://github.com/TezRomacH/python-package-template)\n\nThis project was generated with [`python-package-template`](https://github.com/TezRomacH/python-package-template)\n',
    'author': 'stix2xspec',
    'author_email': 'elastufka@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/stix2xspec/stix2xspec',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
