"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventsRuleToKinesisStreams = void 0;
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const defaults = require("@aws-solutions-constructs/core");
const core_2 = require("@aws-solutions-constructs/core");
class EventsRuleToKinesisStreams extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the EventsRuleToKinesisStreams class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Set up the Kinesis Stream
        this.kinesisStream = defaults.buildKinesisStream(this, {
            existingStreamObj: props.existingStreamObj,
            kinesisStreamProps: props.kinesisStreamProps,
        });
        // Create an events service role
        this.eventsRole = new iam.Role(this, 'eventsRole', {
            assumedBy: new iam.ServicePrincipal('events.amazonaws.com'),
            description: 'Events Rule Role',
        });
        // Grant permission to events service role to allow event rule to send events data to the kinesis stream
        this.kinesisStream.grantWrite(this.eventsRole);
        // Set up the Kinesis Stream as the target for event rule
        const kinesisStreamEventTarget = {
            bind: () => ({
                id: '',
                arn: this.kinesisStream.streamArn,
                role: this.eventsRole
            })
        };
        // Set up the events rule props
        const defaultEventsRuleProps = defaults.DefaultEventsRuleProps([kinesisStreamEventTarget]);
        const eventsRuleProps = core_2.overrideProps(defaultEventsRuleProps, props.eventRuleProps, true);
        // Setup up the event rule
        this.eventsRule = new events.Rule(this, 'EventsRule', eventsRuleProps);
        if (props.createCloudWatchAlarms === undefined || props.createCloudWatchAlarms) {
            // Deploy best practices CW Alarms for Kinesis Stream
            this.cloudwatchAlarms = defaults.buildKinesisStreamCWAlarms(this);
        }
    }
}
exports.EventsRuleToKinesisStreams = EventsRuleToKinesisStreams;
//# sourceMappingURL=data:application/json;base64,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