"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualRouter = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const mesh_1 = require("./mesh");
const route_1 = require("./route");
const shared_interfaces_1 = require("./shared-interfaces");
class VirtualRouterBase extends cdk.Resource {
    /**
     * Add a single route to the router
     */
    addRoute(id, props) {
        const route = new route_1.Route(this, id, {
            ...props,
            routeName: id,
            mesh: this.mesh,
            virtualRouter: this,
        });
        return route;
    }
}
class VirtualRouter extends VirtualRouterBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualRouterName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        this.listeners = new Array();
        this.mesh = props.mesh;
        this.addListener(props.listener || { portMapping: { port: 8080, protocol: shared_interfaces_1.Protocol.HTTP } });
        const router = new appmesh_generated_1.CfnVirtualRouter(this, 'Resource', {
            virtualRouterName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                listeners: this.listeners,
            },
        });
        this.virtualRouterName = this.getResourceNameAttribute(router.attrVirtualRouterName);
        this.virtualRouterArn = this.getResourceArnAttribute(router.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualRouter`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualRouter given an ARN
     */
    static fromVirtualRouterArn(scope, id, virtualRouterArn) {
        return new ImportedVirtualRouter(scope, id, { virtualRouterArn });
    }
    /**
     * Import an existing VirtualRouter given names
     */
    static fromVirtualRouterName(scope, id, meshName, virtualRouterName) {
        return new ImportedVirtualRouter(scope, id, { meshName, virtualRouterName });
    }
    /**
     * Import an existing virtual router given attributes
     */
    static fromVirtualRouterAttributes(scope, id, attrs) {
        return new ImportedVirtualRouter(scope, id, attrs);
    }
    /**
     * Add port mappings to the router
     */
    addListener(listener) {
        this.listeners.push({
            portMapping: listener.portMapping,
        });
    }
}
exports.VirtualRouter = VirtualRouter;
/**
 * Used to import a VirtualRouter and perform actions or read properties from
 */
class ImportedVirtualRouter extends VirtualRouterBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.mesh) {
            this._mesh = props.mesh;
        }
        if (props.meshName) {
            if (props.mesh) {
                throw new Error('Supply either \'mesh\' or \'meshName\', but not both');
            }
            this._mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', props.meshName);
        }
        if (props.virtualRouterArn) {
            this.virtualRouterArn = props.virtualRouterArn;
            this.virtualRouterName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualRouterArn).resourceName));
        }
        else if (props.virtualRouterName && props.meshName) {
            this.virtualRouterName = props.virtualRouterName;
            this.virtualRouterArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualRouter`,
                resourceName: this.virtualRouterName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
    /**
     * The AppMesh mesh the VirtualRouter belongs to
     */
    get mesh() {
        if (!this._mesh) {
            throw new Error('Please supply either \'mesh\' or \'meshName\' when calling \'fromVirtualRouterAttributes\'');
        }
        return this._mesh;
    }
}
//# sourceMappingURL=data:application/json;base64,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