"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MainPipeline = void 0;
const aws_events_1 = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const constructs_1 = require("constructs");
const customNodejsFunction_1 = require("./customNodejsFunction");
const path = require("path");
const notificationsTopic_1 = require("./notificationsTopic");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const lodash_1 = require("lodash");
const context_1 = require("../util/context");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const types_1 = require("../util/types");
const appStage_1 = require("./appStage");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
class MainPipeline extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const source = pipelines_1.CodePipelineSource.codeCommit(props.codeCommitRepository, props.repository.defaultBranch);
        const pipeline = new pipelines_1.CodePipeline(this, 'Pipeline', lodash_1.merge({
            pipelineName: aws_cdk_lib_1.Stack.of(this).stackName,
            synth: new pipelines_1.ShellStep('Synth', {
                input: source,
                installCommands: [
                    ...(props.commands.preInstall || []),
                    ...(props.commands.install || []),
                ],
                commands: [
                    ...(props.commands.buildAndTest || []),
                    ...props.commands.synthPipeline,
                ],
                primaryOutputDirectory: props.cdkOutputDirectory,
            }),
            crossAccountKeys: true,
            codeBuildDefaults: props.codeBuild,
        }, props.codePipeline));
        props.pipeline.forEach(step => {
            if (this.isWave(step)) {
                this.addWaveDeployment(pipeline, step, props.stacks);
            }
            else {
                this.addEnvironmentDeployment(pipeline, step, props.stacks);
            }
        });
        pipeline.buildPipeline();
        this.createPipelineBuildNotifications(pipeline, props.repository.host, props.repositoryApiDestination);
        this.failuresTopic = this.createPipelineFailuresTopic(pipeline);
    }
    isWave(waveOrEnvironment) {
        return 'wave' in waveOrEnvironment;
    }
    addWaveDeployment(pipeline, step, stacks) {
        const wave = pipeline.addWave(step.wave);
        step.environments.forEach(env => {
            const environmentDeployment = {
                ...env,
                pre: [
                    ...(step.preEachEnvironment || []),
                    ...(env.pre || []),
                ],
                post: [
                    ...(env.post || []),
                    ...(step.postEachEnvironment || []),
                ],
            };
            this.addEnvironmentDeployment(wave, environmentDeployment, stacks, `Wave${capitalizeFirstLetter(step.wave)}`, { WAVE_NAME: step.wave });
        });
        if (step.pre && step.pre.length > 0) {
            wave.addPre(new pipelines_1.ShellStep(`PreWave${capitalizeFirstLetter(step.wave)}`, {
                env: { WAVE_NAME: step.wave },
                commands: step.pre,
            }));
        }
        if (step.post && step.post.length > 0) {
            wave.addPost(new pipelines_1.ShellStep(`PostWave${capitalizeFirstLetter(step.wave)}`, {
                env: { WAVE_NAME: step.wave },
                commands: step.post,
            }));
        }
    }
    addEnvironmentDeployment(parent, step, stacks, idPrefix = '', envVariables) {
        const stage = parent.addStage(new appStage_1.AppStage(this, `${idPrefix}DeployEnv${capitalizeFirstLetter(step.environment)}`, {
            envName: step.environment,
            env: context_1.getEnvironmentConfig(this, step.environment),
            stacks,
        }));
        if (step.pre && step.pre.length > 0) {
            stage.addPre(new pipelines_1.ShellStep(`${idPrefix}PreEnv${capitalizeFirstLetter(step.environment)}`, {
                env: {
                    ...envVariables,
                    ENV_NAME: step.environment,
                },
                commands: step.pre,
            }));
        }
        if (step.post && step.post.length > 0) {
            stage.addPost(new pipelines_1.ShellStep(`${idPrefix}PostEnv${capitalizeFirstLetter(step.environment)}`, {
                env: {
                    ...envVariables,
                    ENV_NAME: step.environment,
                },
                commands: step.post,
            }));
        }
    }
    createPipelineFailuresTopic(pipeline) {
        const failuresTopic = new notificationsTopic_1.NotificationsTopic(this, 'PipelineFailuresTopic', {
            projectName: context_1.getProjectName(this),
            notificationName: 'pipelineFailures',
        });
        // for better visibility, use EventBridge Rules instead of CodeStar Notifications that are generated with pipeline.notifyOn()
        pipeline.pipeline.onStateChange('OnPipelineFailure', {
            eventPattern: {
                detail: {
                    state: ['FAILED'],
                },
            },
            target: new targets.SnsTopic(failuresTopic.topic),
        });
        return failuresTopic.topic;
    }
    /**
     * To send CodePipeline build status back to repository:
     * - trigger Lambda function on CodePipeline state change events,
     * - in Lambda:
     *   - get CodePipeline execution details to get commit SHA,
     *   - send custom event to EventBridge including the commit SHA,
     * - use EventBridge to send build status to repository.
     */
    createPipelineBuildNotifications(pipeline, repositoryType, repositoryApiDestination) {
        const pipelineBuildStatusEventsSourceName = `${aws_cdk_lib_1.Stack.of(this).stackName}.pipelineBuildStatus`;
        const pipelineBuildStatusFunction = new customNodejsFunction_1.CustomNodejsFunction(this, 'PipelineBuildStatus', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', 'lambda', 'pipelineBuildStatus')),
            environment: {
                'REPOSITORY_TYPE': repositoryType,
                'EVENT_SOURCE_NAME': pipelineBuildStatusEventsSourceName,
            },
        });
        pipelineBuildStatusFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['codepipeline:GetPipelineExecution'],
            resources: [`arn:aws:codepipeline:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:${pipeline.pipeline.pipelineName}`],
        }));
        pipelineBuildStatusFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['events:PutEvents'],
            resources: [`arn:aws:events:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:event-bus/default`],
            conditions: {
                StringEquals: {
                    'events:source': pipelineBuildStatusEventsSourceName,
                },
            },
        }));
        pipeline.pipeline.onStateChange('OnPipelineStateChange', {
            target: new targets.LambdaFunction(pipelineBuildStatusFunction),
        });
        new aws_events_1.Rule(this, 'SendPipelineStatusToRepositoryRule', {
            eventPattern: {
                source: [pipelineBuildStatusEventsSourceName],
                detailType: ['CodePipeline Pipeline Execution State Change'],
            },
            targets: [
                new targets.ApiDestination(repositoryApiDestination, {
                    pathParameterValues: ['$.detail.commit-sha'],
                    event: this.createStatusEvent(repositoryType),
                }),
            ],
        });
    }
    createStatusEvent(repositoryType) {
        switch (repositoryType) {
            case 'github':
                return aws_events_1.RuleTargetInput.fromObject({
                    'state': aws_events_1.EventField.fromPath('$.detail.state'),
                    'target_url': `https://${aws_events_1.EventField.fromPath('$.region')}.console.aws.amazon.com/codesuite/codepipeline/pipelines/${aws_events_1.EventField.fromPath('$.detail.pipeline-name')}/executions/${aws_events_1.EventField.fromPath('$.detail.execution-id')}`,
                    'context': aws_events_1.EventField.fromPath('$.detail.pipeline-name'),
                });
            case 'bitbucket':
                return aws_events_1.RuleTargetInput.fromObject({
                    'key': 'AWS-PIPELINE-BUILD',
                    'state': aws_events_1.EventField.fromPath('$.detail.state'),
                    'name': aws_events_1.EventField.fromPath('$.detail.pipeline-name'),
                    'description': 'AWS CodePipeline',
                    'url': `https://${aws_events_1.EventField.fromPath('$.region')}.console.aws.amazon.com/codesuite/codepipeline/pipelines/${aws_events_1.EventField.fromPath('$.detail.pipeline-name')}/executions/${aws_events_1.EventField.fromPath('$.detail.execution-id')}`,
                });
            default:
                return types_1.assertUnreachable(repositoryType);
        }
    }
}
exports.MainPipeline = MainPipeline;
const capitalizeFirstLetter = (str) => str.charAt(0).toUpperCase() + str.slice(1);
//# sourceMappingURL=data:application/json;base64,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