"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FeatureBranchBuilds = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const constructs_1 = require("constructs");
const customNodejsFunction_1 = require("./customNodejsFunction");
const path = require("path");
const notificationsTopic_1 = require("./notificationsTopic");
const types_1 = require("../util/types");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const context_1 = require("../util/context");
class FeatureBranchBuilds extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const source = aws_codebuild_1.Source.codeCommit({ repository: props.codeCommitRepository });
        const deployProject = this.createDeployProject(source, props.codeBuild, props.commands, props.codeCommitRepository, props.repository.defaultBranch);
        this.createDeployNotifications(deployProject, props.repository.host, props.repositoryApiDestination);
        this.failuresTopic = this.createBuildFailuresTopic(deployProject);
        this.createDestroyProject(source, props.codeBuild, props.commands, props.codeCommitRepository, props.repository.defaultBranch);
    }
    createDeployProject(source, codeBuild, commands, repository, defaultBranch) {
        const deployProject = new aws_codebuild_1.Project(this, 'DeployProject', {
            projectName: `${aws_cdk_lib_1.Stack.of(this).stackName}-featureBranch-deploy`,
            source,
            timeout: codeBuild.timeout,
            environment: codeBuild.buildEnvironment,
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: [
                            'ENV_NAME=$(echo ${BRANCH_NAME} | awk \'{print tolower($0)}\')',
                            ...(commands.preInstall || []),
                            ...(commands.install || []),
                        ],
                    },
                    build: {
                        commands: [
                            ...(commands.buildAndTest || []),
                            ...commands.deployEnvironment,
                        ],
                    },
                },
            }),
        });
        this.grantAssumeCDKRoles(deployProject);
        repository.onCommit('OnBranchCommit', this.createProjectTriggerOptions(deployProject, defaultBranch, true));
        return deployProject;
    }
    createDeployNotifications(deployProject, repositoryType, repositoryApiDestination) {
        const deployStatusEventSourceName = `${aws_cdk_lib_1.Stack.of(this).stackName}.featureBranchDeployStatus`;
        const deployStatusFunction = new customNodejsFunction_1.CustomNodejsFunction(this, 'DeployStatus', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', 'lambda', 'featureBranchDeployStatus')),
            environment: {
                'REPOSITORY_TYPE': repositoryType,
                'EVENT_SOURCE_NAME': deployStatusEventSourceName,
            },
        });
        deployStatusFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['events:PutEvents'],
            resources: [`arn:aws:events:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:event-bus/default`],
            conditions: {
                StringEquals: {
                    'events:source': deployStatusEventSourceName,
                },
            },
        }));
        deployProject.onStateChange('OnDeployStateChange', {
            target: new targets.LambdaFunction(deployStatusFunction),
        });
        new aws_events_1.Rule(this, 'SendDeployStatusToRepositoryRule', {
            eventPattern: {
                source: [deployStatusEventSourceName],
                detailType: ['CodeBuild Build State Change'],
            },
            targets: [
                new targets.ApiDestination(repositoryApiDestination, {
                    pathParameterValues: ['$.detail.commit-sha'],
                    event: this.createStatusEvent(repositoryType),
                }),
            ],
        });
    }
    createStatusEvent(repositoryType) {
        switch (repositoryType) {
            case 'github':
                return aws_events_1.RuleTargetInput.fromObject({
                    'state': aws_events_1.EventField.fromPath('$.detail.state'),
                    'target_url': `https://${aws_events_1.EventField.fromPath('$.region')}.console.aws.amazon.com/codesuite/codebuild/projects/${aws_events_1.EventField.fromPath('$.detail.project-name')}/build/${aws_events_1.EventField.fromPath('$.detail.build-id')}`,
                    'context': aws_events_1.EventField.fromPath('$.detail.project-name'),
                });
            case 'bitbucket':
                return aws_events_1.RuleTargetInput.fromObject({
                    'key': 'AWS-PIPELINE-BUILD',
                    'state': aws_events_1.EventField.fromPath('$.detail.state'),
                    'name': aws_events_1.EventField.fromPath('$.detail.project-name'),
                    'description': 'Feature branch deployment on AWS CodeBuild',
                    'url': `https://${aws_events_1.EventField.fromPath('$.region')}.console.aws.amazon.com/codesuite/codebuild/projects/${aws_events_1.EventField.fromPath('$.detail.project-name')}/build/${aws_events_1.EventField.fromPath('$.detail.build-id')}`,
                });
            default:
                return types_1.assertUnreachable(repositoryType);
        }
    }
    createBuildFailuresTopic(deployProject) {
        const failuresTopic = new notificationsTopic_1.NotificationsTopic(this, 'FeatureBranchBuildFailuresTopic', {
            projectName: context_1.getProjectName(this),
            notificationName: 'featureBranchBuildFailures',
        });
        // for better visibility, use EventBridge Rules instead of CodeStar Notifications that are generated with project.notifyOnBuildFailed()
        deployProject.onBuildFailed('OnFeatureBuildFailure', {
            target: new targets.SnsTopic(failuresTopic.topic),
        });
        return failuresTopic.topic;
    }
    createDestroyProject(source, codeBuild, commands, repository, defaultBranch) {
        const destroyProject = new aws_codebuild_1.Project(this, 'DestroyProject', {
            projectName: `${aws_cdk_lib_1.Stack.of(this).stackName}-featureBranch-destroy`,
            source,
            timeout: codeBuild.timeout,
            environment: codeBuild.buildEnvironment,
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: [
                            'ENV_NAME=$(echo ${BRANCH_NAME} | awk \'{print tolower($0)}\')',
                            ...(commands.preInstall || []),
                            ...(commands.install || []),
                        ],
                    },
                    build: {
                        commands: [
                            ...commands.destroyEnvironment,
                        ],
                    },
                },
            }),
        });
        this.grantAssumeCDKRoles(destroyProject);
        repository.onReferenceDeleted('OnBranchRemoval', this.createProjectTriggerOptions(destroyProject, defaultBranch));
        return destroyProject;
    }
    createProjectTriggerOptions(targetProject, defaultBranch, withSourceVersion = false) {
        return {
            eventPattern: {
                detail: {
                    referenceType: ['branch'],
                    referenceName: [
                        { 'anything-but': [defaultBranch] },
                    ],
                    referenceFullName: [
                        { 'anything-but': { prefix: 'refs/remotes/' } },
                    ],
                },
            },
            target: new aws_events_targets_1.CodeBuildProject(targetProject, {
                event: aws_events_1.RuleTargetInput.fromObject({
                    sourceVersion: withSourceVersion ? aws_events_1.EventField.fromPath('$.detail.commitId') : undefined,
                    environmentVariablesOverride: [
                        {
                            name: 'BRANCH_NAME',
                            value: aws_events_1.EventField.fromPath('$.detail.referenceName'),
                            type: 'PLAINTEXT',
                        },
                    ],
                }),
            }),
        };
    }
    grantAssumeCDKRoles(project) {
        const qualifier = aws_cdk_lib_1.Stack.of(this).synthesizer.bootstrapQualifier || 'hnb659fds';
        project.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [
                `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk-${qualifier}-deploy-role-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`,
                `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk-${qualifier}-file-publishing-role-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`,
                `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk-${qualifier}-image-publishing-role-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`,
                `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk-${qualifier}-lookup-role-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`,
            ],
        }));
    }
}
exports.FeatureBranchBuilds = FeatureBranchBuilds;
//# sourceMappingURL=data:application/json;base64,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