"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CIStack = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lodash_1 = require("lodash");
const aws_codecommit_1 = require("aws-cdk-lib/aws-codecommit");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const featureBranchBuilds_1 = require("./constructs/featureBranchBuilds");
const mainPipeline_1 = require("./constructs/mainPipeline");
const types_1 = require("./util/types");
const applicationProps_1 = require("./applicationProps");
const aws_chatbot_1 = require("aws-cdk-lib/aws-chatbot");
const notificationsTopic_1 = require("./constructs/notificationsTopic");
const context_1 = require("./util/context");
const defaultCommands = {
    npm: {
        install: [
            'npm install --location=global aws-cdk@2',
            'npm ci',
        ],
    },
    pnpm: {
        install: [
            'npm install --location=global aws-cdk@2 pnpm',
            'pnpm install --frozen-lockfile',
        ],
    },
};
class CIStack extends aws_cdk_lib_1.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        const resolvedProps = this.resolveProps(props);
        const projectName = context_1.getProjectName(this);
        const repository = this.createCodeCommitRepository();
        const repositoryApiDestination = this.createRepositoryApiDestination(props.repository);
        const mainPipeline = new mainPipeline_1.MainPipeline(this, 'MainPipeline', {
            ...resolvedProps,
            codeCommitRepository: repository,
            repositoryApiDestination,
        });
        const featureBranchBuilds = new featureBranchBuilds_1.FeatureBranchBuilds(this, 'FeatureBranchBuilds', {
            ...resolvedProps,
            codeCommitRepository: repository,
            repositoryApiDestination,
        });
        if (resolvedProps.slackNotifications.workspaceId && resolvedProps.slackNotifications.channelId) {
            this.createSlackNotifications(projectName, resolvedProps.slackNotifications, mainPipeline.failuresTopic, featureBranchBuilds.failuresTopic);
        }
    }
    resolveProps(props) {
        if (props.packageManager) {
            lodash_1.merge(applicationProps_1.defaultProps, { commands: defaultCommands[props.packageManager] });
        }
        return lodash_1.defaultsDeep(lodash_1.cloneDeep(props), applicationProps_1.defaultProps);
    }
    createCodeCommitRepository() {
        const repository = new aws_codecommit_1.Repository(this, 'Repository', {
            repositoryName: context_1.getProjectName(this),
        });
        const mirrorUser = new aws_iam_1.User(this, 'RepositoryMirrorUser', {
            userName: `${this.stackName}-repository-mirror-user`,
        });
        repository.grantPullPush(mirrorUser);
        return repository;
    }
    createRepositoryApiDestination(repository) {
        const projectName = context_1.getProjectName(this);
        switch (repository.host) {
            case 'github':
                return new aws_events_1.ApiDestination(this, 'GitHubDestination', {
                    connection: new aws_events_1.Connection(this, 'GitHubConnection', {
                        authorization: aws_events_1.Authorization.apiKey('Authorization', aws_cdk_lib_1.SecretValue.secretsManager(`/${projectName}/githubAuthorization`, { jsonField: 'header' })),
                        description: 'GitHub repository connection',
                        headerParameters: {
                            'Accept': aws_events_1.HttpParameter.fromString('application/vnd.github+json'),
                            'X-GitHub-Api-Version': aws_events_1.HttpParameter.fromString('2022-11-28'),
                        },
                    }),
                    endpoint: `https://api.github.com/repos/${repository.name}/statuses/*`,
                    httpMethod: aws_events_1.HttpMethod.POST,
                });
            case 'bitbucket':
                return new aws_events_1.ApiDestination(this, 'BitbucketDestination', {
                    connection: new aws_events_1.Connection(this, 'BitbucketConnection', {
                        authorization: aws_events_1.Authorization.apiKey('Authorization', aws_cdk_lib_1.SecretValue.secretsManager(`/${projectName}/bitbucketAuthorization`, { jsonField: 'header' })),
                        description: 'Bitbucket repository connection',
                    }),
                    endpoint: `https://api.bitbucket.org/2.0/repositories/${repository.name}/commit/*/statuses/build`,
                    httpMethod: aws_events_1.HttpMethod.POST,
                });
            default:
                return types_1.assertUnreachable(repository.host);
        }
    }
    createSlackNotifications(projectName, config, mainPipelineFailuresTopic, featureBranchFailuresTopic) {
        const alarmsTopic = new notificationsTopic_1.NotificationsTopic(this, 'SlackAlarmsTopic', {
            projectName: projectName,
            notificationName: 'slackAlarms',
        });
        const slack = new aws_chatbot_1.SlackChannelConfiguration(this, 'SlackChannelConfiguration', {
            slackChannelConfigurationName: this.stackName,
            slackWorkspaceId: config.workspaceId,
            slackChannelId: config.channelId,
            notificationTopics: [
                alarmsTopic.topic,
                config.mainPipelineFailures ? mainPipelineFailuresTopic : undefined,
                config.featureBranchFailures ? featureBranchFailuresTopic : undefined,
            ].filter(types_1.notEmpty),
        });
        slack.role?.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchReadOnlyAccess'));
    }
}
exports.CIStack = CIStack;
//# sourceMappingURL=data:application/json;base64,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