'''
# Opinionated CDK CI Pipeline

[![NPM](https://img.shields.io/npm/v/opinionated-ci-pipeline?color=blue)](https://www.npmjs.com/package/opinionated-ci-pipeline)
[![PyPI](https://img.shields.io/pypi/v/opinionated-ci-pipeline?color=blue)](https://pypi.org/project/opinionated-ci-pipeline/)

CI/CD utilizing [CDK Pipelines](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.pipelines-readme.html).

Features:

* pipeline deploying application from the default branch
  to multiple environments on multiple accounts,
* feature branch deployments to ephemeral environments,
* development environments deployments from the local CLI,
* build status notifications to repository commits,
* build failures notifications to SNS.

Currently supported source repositories are GitHub and Bitbucket.

## Table of contents

* [Table of contents](#table-of-contents)
* [Usage](#usage)

  * [1. Install](#1-install)
  * [2. Set context parameters](#2-set-context-parameters)
  * [3. Create `CDKApplication`](#3-create-cdkapplication)
  * [4. Create repository access token secret](#4-create-repository-access-token-secret)

    * [GitHub](#github)
    * [Bitbucket](#bitbucket)
  * [5. Bootstrap the CDK](#5-bootstrap-the-cdk)
  * [6. Deploy the CI Stack](#6-deploy-the-ci-stack)
  * [7. Setup source repository mirroring](#7-setup-source-repository-mirroring)

    * [GitHub](#github)
    * [Bitbucket](#bitbucket)
  * [Deploy development environment](#deploy-development-environment)
* [Parameters](#parameters)
* [Notifications and alarms](#notifications-and-alarms)
* [Library development](#library-development)

## Usage

To set up, you need to complete the following steps:

1. Install the library in your project.
2. Specify target account and region for deployed environments.
3. Create `CDKApplication` with build process configuration.
4. Create repository access token for build status notifications.
5. Bootstrap the CDK on the AWS account(s).
6. Deploy the CI.
7. Setup source repository mirroring to CodeCommit.

At the end, you will have CI pipeline in place,
and be able to deploy your own custom environment from the CLI as well.

### 1. Install

For Node.js:

```bash
npm install -D opinionated-ci-pipeline
```

For Python:

```bash
pip install opinionated-ci-pipeline
```

### 2. Set context parameters

Add project name and environments config in the `cdk.json` as `context` parameters.
Each environment must have `account` and `region` provided.

```json
{
  "app": "...",
  "context": {
    "projectName": "myproject",
    "environments": {
      "default": {
        "account": "111111111111",
        "region": "us-east-1"
      },
      "prod": {
        "account": "222222222222",
        "region": "us-east-1"
      }
    }
  }
}
```

The project name will be used as a prefix for the deployed CI Stack name.

Environment names should match environments provided later
in the `CDKApplication` configuration.

The optional `default` environment configuration is used as a fallback.

The CI pipeline itself is deployed to the `ci` environment,
with a fallback to the `default` environment as well.

### 3. Create `CDKApplication`

In the CDK entrypoint script referenced by the `cdk.json` `app` field,
replace the content with an instance of `CDKApplication`:

```python
#!/usr/bin/env node
import 'source-map-support/register';
import {ExampleStack} from '../lib/exampleStack';
import {CDKApplication} from 'opinionated-ci-pipeline';

new CDKApplication({
    stacks: {
        create: (scope, projectName, envName) => {
            new ExampleStack(scope, 'ExampleStack', {stackName: `${projectName}-${envName}-ExampleStack`});
        },
    },
    repository: {
        host: 'github',
        name: 'organization/repository',
    },
    packageManager: 'npm',
    pipeline: [
        {
            environment: 'test',
            post: [
                'echo "do integration tests here"',
            ],
        },
        {
            environment: 'prod',
        },
    ],
});
```

This configures the application with one Stack
and a pipeline deploying to an environment `test`,
running integration tests, and deploying to environment `prod`.

The `test` and `prod` environments will be deployed
from the branch `main` (by default).
All other branches will be deployed to separate environments.
Those feature-branch environments will be destroyed after the branch is removed.

To allow deployment of multiple environments,
the Stack(s) name must include the environment name.

### 4. Create repository access token secret

An access to the source repository is required
to send build status notifications,
visible in commit status and Pull Requests.

#### GitHub

Create [a fine-grained personal access token](https://docs.github.com/en/authentication/keeping-your-account-and-data-secure/creating-a-personal-access-token#creating-a-fine-grained-personal-access-token)
with "commit statuses" read and write access.

Then create a secret in AWS Secrets Manager
named `/PROJECT_NAME/githubAuthorization` with the generated token:

```json
{
  "header": "Bearer xxx"
}
```

#### Bitbucket

Create an access token in Bitbucket repository settings
with `repository:write` access.

Then create a secret in AWS Secrets Manager
named `/PROJECT_NAME/bitbucketAuthorization` with the generated token:

```json
{
  "header": "Bearer xxx"
}
```

### 5. Bootstrap the CDK

[Bootstrap the CDK](https://docs.aws.amazon.com/cdk/v2/guide/bootstrapping.html)
on the account holding the CI pipeline
and all other accounts the pipeline will be deploying to.

When bootstrapping other accounts, add the `--trust` parameter
with the account ID of the account holding the pipeline.

### 6. Deploy the CI Stack

Run:

```bash
cdk deploy -c ci=true
```

### 7. Setup source repository mirroring

Because of [multiple drawbacks](adr/connecting-repositories.md)
of "native" CodePipeline and CodeBuild integrations with GitHub and Bitbucket,
the CI builds use CodeCommit as a source.
You must configure GitHub / Bitbucket to mirror the repository to CodeCommit.

CI stack creates a CodeCommit repository (named the same as the `projectName` context value)
and IAM user `{PROJECT_NAME}-ci-repository-mirror-user` with access to it.

#### GitHub

1. Create an SSH key pair with `ssh-keygen -t rsa -b 4096` command.
2. Upload the public key as an SSH key for AWS CodeCommit
   in the created IAM user security credentials settings.
3. Create secrets in the GitHub repository settings under
   "Secrets and variables" -> "Actions":

   * `CODECOMMIT_SSH_PRIVATE_KEY` - the private key generated in step 1.
   * `CODECOMMIT_SSH_PRIVATE_KEY_ID` - the SSH key ID of uploaded key from IAM user.
4. Create `.github/workflows/mirror-repository.yml` file
   (update the `REGION` and `PROJECT_NAME` in `target_repo_url`):

```yml
name: Mirror to CodeCommit
on: [ push, delete ]
jobs:
  sync:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v3
        with:
          fetch-depth: 0
      - uses: pixta-dev/repository-mirroring-action@v1
        with:
          target_repo_url: ssh://git-codecommit.REGION.amazonaws.com/v1/repos/PROJECT_NAME
          ssh_private_key: ${{ secrets.CODECOMMIT_SSH_PRIVATE_KEY }}
          ssh_username: ${{ secrets.CODECOMMIT_SSH_PRIVATE_KEY_ID }}
```

#### Bitbucket

1. Create an SSH key pair in the Bitbucket repository Pipeline config.
2. Upload the public key as an SSH key for AWS CodeCommit
   in the created IAM user security credentials settings.
3. Create secrets in the Bitbucket repository Pipeline settings:

   * `CODECOMMIT_SSH_PRIVATE_KEY` - the private key generated in step 1.
   * `CODECOMMIT_SSH_PRIVATE_KEY_ID` - the SSH key ID of uploaded key from IAM user.
4. Put `git-codecommit.REGION.amazonaws.com` in Bitbucket repository SSH Keys Known hosts
   (update the `REGION`).
5. Create `bitbucket-pipelines.yml` file (update the `REGION` and `PROJECT_NAME`):

```yml
image: atlassian/default-image:4

clone:
  depth: full

pipelines:
  default:
    - step:
        name: Sync to CodeCommit
        script:
          - echo "User ${SSH_KEY_ID}" >> ~/.ssh/config
          # fetch all branches to mirror the whole repository
          - for remote in `git branch -r | grep -v -- '->' | grep -v -- "${BITBUCKET_BRANCH}"`; do git branch --track ${remote#origin/} $remote; done
          # mirror repository; it will remove deleted branches as well
          - git push --mirror ssh://git-codecommit.REGION.amazonaws.com/v1/repos/REPOSITORY-NAME
```

### Deploy development environment

Run:

```bash
cdk deploy -c env=MYENV --all
```

to deploy arbitrary environments.

## Parameters

<table>
    <tr>
        <th>Name</th>
        <th>Type</th>
        <th>Description</th>
    </tr>
    <tr>
        <td>stacks</td>
        <td>object</td>
        <td>
An object with a create() method to create Stacks for the application.
<br/>
The same Stacks will be deployed with main pipeline, feature-branch builds, and local deployments.
        </td>
    </tr>
    <tr>
        <td>packageManager</td>
        <td>npm | pnpm</td>
        <td>
Package manager used in the repository.
<br/>
If provided, the install commands will be set to install dependencies using given package manager.
        </td>
    </tr>
    <tr>
        <td>commands</td>
        <td>object</td>
        <td>
Commands executed to build and deploy the application.
<br/>
Commands executed on particular builds:

* main pipeline:

  * `preInstall`
  * `install`
  * `buildAndTest`
  * `synthPipeline`
* feature branch environment deployment:

  * `preInstall`
  * `install`
  * `buildAndTest`
  * `deployEnvironment`
* feature branch environment destruction:

  * `preInstall`
  * `install`
  * `destroyEnvironment`

    </td>
    </tr>
    <tr>
        <td>cdkOutputDirectory</td>
        <td>string</td>
        <td>

The location where CDK outputs synthetized files.
Corresponds to the CDK Pipelines `ShellStepProps#primaryOutputDirectory`.

</td>
      </tr>
      <tr>
          <td>pipeline</td>
          <td>object[]</td>
          <td>
CodePipeline deployment pipeline for the main repository branch.
<br/>
Can contain environments to deploy
and waves that deploy multiple environments in parallel.
<br/>
Each environment and wave can have pre and post commands
that will be executed before and after the environment or wave deployment.
            </td>
      </tr>
      <tr>
          <td>codeBuild</td>
          <td>object</td>
          <td>
Override CodeBuild properties, used for the main pipeline
as well as feature branch ephemeral environments deploys and destroys.
</td>
      </tr>
      <tr>
          <td>codePipeline</td>
          <td>object</td>
          <td>Override CodePipeline properties.</td>
      </tr>
      <tr>
          <td>slackNotifications</td>
          <td>object</td>
          <td>
Configuration for Slack notifications.
Requires configuring AWS Chatbot client manually first.
</td>
      </tr>
</table>

## Notifications and alarms

Stack creates SNS Topics with notifications for
main pipeline failures and feature branch build failures.
Their ARNs are saved in SSM Parameters and outputed by the stack:

* main pipeline failures:

  * SSM: `/{projectName}/ci/pipelineFailuresTopicArn`
  * Stack exported output: `{projectName}-ci-pipelineFailuresTopicArn`
* feature branch build failures:

  * SSM: `/{projectName}/ci/featureBranchBuildFailuresTopicArn`
  * Stack exported output: `{projectName}-ci-featureBranchBuildFailuresTopicArn`

If you setup Slack notifications,
you can configure those failure notifications to be sent to Slack.

Moreover, if you setup Slack notifications,
an additional SNS Topic will be created
to which you can send CloudWatch Alarms.
It's ARN is provided:

* SSM: `/{projectName}/ci/slackAlarmsTopicArn`
* Stack exported output: `{projectName}-ci-slackAlarmsTopicArn`

## Library development

Project uses [jsii](https://aws.github.io/jsii/)
to generate packages for different languages.

Install dependencies:

```bash
npm install
```

Build:

```bash
npm run build
```

Install and deploy example application:

```bash
cd example
pnpm install
pnpm cdk deploy -c ci=true
```
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from typeguard import check_type

from ._jsii import *

import aws_cdk as _aws_cdk_ceddda9d
import aws_cdk.aws_iam as _aws_cdk_aws_iam_ceddda9d
import aws_cdk.pipelines as _aws_cdk_pipelines_ceddda9d
import constructs as _constructs_77d1e7e8


@jsii.data_type(
    jsii_type="opinionated-ci-pipeline.ApplicationProps",
    jsii_struct_bases=[],
    name_mapping={
        "pipeline": "pipeline",
        "repository": "repository",
        "stacks": "stacks",
        "cdk_output_directory": "cdkOutputDirectory",
        "code_build": "codeBuild",
        "code_pipeline": "codePipeline",
        "commands": "commands",
        "package_manager": "packageManager",
        "slack_notifications": "slackNotifications",
    },
)
class ApplicationProps:
    def __init__(
        self,
        *,
        pipeline: typing.Sequence[typing.Union[typing.Union["WaveDeployment", typing.Dict[builtins.str, typing.Any]], typing.Union["EnvironmentDeployment", typing.Dict[builtins.str, typing.Any]]]],
        repository: typing.Union["RepositoryProps", typing.Dict[builtins.str, typing.Any]],
        stacks: "IStacksCreation",
        cdk_output_directory: typing.Optional[builtins.str] = None,
        code_build: typing.Optional[typing.Union[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        code_pipeline: typing.Optional[typing.Union["CodePipelineOverrides", typing.Dict[builtins.str, typing.Any]]] = None,
        commands: typing.Optional[typing.Union["BuildCommands", typing.Dict[builtins.str, typing.Any]]] = None,
        package_manager: typing.Optional[builtins.str] = None,
        slack_notifications: typing.Optional[typing.Union["SlackNotifications", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''
        :param pipeline: (experimental) CodePipeline deployment pipeline for the main repository branch. Can contain environments to deploy and waves that deploy multiple environments in parallel. Each environment and wave can have pre and post commands that will be executed before and after the environment or wave deployment.
        :param repository: 
        :param stacks: (experimental) An object with a create() method to create Stacks for the application. The same Stacks will be deployed with main pipeline, feature-branch builds, and local deployments.
        :param cdk_output_directory: (experimental) The location where CDK outputs synthetized files. Corresponds to the CDK Pipelines ShellStepProps#primaryOutputDirectory. Default: cdk.out
        :param code_build: (experimental) Override CodeBuild properties, used for the main pipeline as well as feature branch ephemeral environments deploys and destroys. Default: 30 minutes timeout, compute type MEDIUM with Linux build image Standard 6.0
        :param code_pipeline: (experimental) Override CodePipeline properties.
        :param commands: (experimental) Commands executed to build and deploy the application.
        :param package_manager: (experimental) Package manager used in the repository. If provided, the install commands will be set to install dependencies using given package manager.
        :param slack_notifications: (experimental) Configuration for Slack notifications. Requires configuring AWS Chatbot client manually first.

        :stability: experimental
        '''
        if isinstance(repository, dict):
            repository = RepositoryProps(**repository)
        if isinstance(code_build, dict):
            code_build = _aws_cdk_pipelines_ceddda9d.CodeBuildOptions(**code_build)
        if isinstance(code_pipeline, dict):
            code_pipeline = CodePipelineOverrides(**code_pipeline)
        if isinstance(commands, dict):
            commands = BuildCommands(**commands)
        if isinstance(slack_notifications, dict):
            slack_notifications = SlackNotifications(**slack_notifications)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d22629910ce624c4144f7d3a67b45ba1c7b1018575c9f66d6177bc139efb6eb9)
            check_type(argname="argument pipeline", value=pipeline, expected_type=type_hints["pipeline"])
            check_type(argname="argument repository", value=repository, expected_type=type_hints["repository"])
            check_type(argname="argument stacks", value=stacks, expected_type=type_hints["stacks"])
            check_type(argname="argument cdk_output_directory", value=cdk_output_directory, expected_type=type_hints["cdk_output_directory"])
            check_type(argname="argument code_build", value=code_build, expected_type=type_hints["code_build"])
            check_type(argname="argument code_pipeline", value=code_pipeline, expected_type=type_hints["code_pipeline"])
            check_type(argname="argument commands", value=commands, expected_type=type_hints["commands"])
            check_type(argname="argument package_manager", value=package_manager, expected_type=type_hints["package_manager"])
            check_type(argname="argument slack_notifications", value=slack_notifications, expected_type=type_hints["slack_notifications"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "pipeline": pipeline,
            "repository": repository,
            "stacks": stacks,
        }
        if cdk_output_directory is not None:
            self._values["cdk_output_directory"] = cdk_output_directory
        if code_build is not None:
            self._values["code_build"] = code_build
        if code_pipeline is not None:
            self._values["code_pipeline"] = code_pipeline
        if commands is not None:
            self._values["commands"] = commands
        if package_manager is not None:
            self._values["package_manager"] = package_manager
        if slack_notifications is not None:
            self._values["slack_notifications"] = slack_notifications

    @builtins.property
    def pipeline(
        self,
    ) -> typing.List[typing.Union["WaveDeployment", "EnvironmentDeployment"]]:
        '''(experimental) CodePipeline deployment pipeline for the main repository branch.

        Can contain environments to deploy
        and waves that deploy multiple environments in parallel.

        Each environment and wave can have pre and post commands
        that will be executed before and after the environment or wave deployment.

        :stability: experimental
        '''
        result = self._values.get("pipeline")
        assert result is not None, "Required property 'pipeline' is missing"
        return typing.cast(typing.List[typing.Union["WaveDeployment", "EnvironmentDeployment"]], result)

    @builtins.property
    def repository(self) -> "RepositoryProps":
        '''
        :stability: experimental
        '''
        result = self._values.get("repository")
        assert result is not None, "Required property 'repository' is missing"
        return typing.cast("RepositoryProps", result)

    @builtins.property
    def stacks(self) -> "IStacksCreation":
        '''(experimental) An object with a create() method to create Stacks for the application.

        The same Stacks will be deployed with main pipeline, feature-branch builds, and local deployments.

        :stability: experimental
        '''
        result = self._values.get("stacks")
        assert result is not None, "Required property 'stacks' is missing"
        return typing.cast("IStacksCreation", result)

    @builtins.property
    def cdk_output_directory(self) -> typing.Optional[builtins.str]:
        '''(experimental) The location where CDK outputs synthetized files.

        Corresponds to the CDK Pipelines ShellStepProps#primaryOutputDirectory.

        :default: cdk.out

        :stability: experimental
        '''
        result = self._values.get("cdk_output_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def code_build(
        self,
    ) -> typing.Optional[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions]:
        '''(experimental) Override CodeBuild properties, used for the main pipeline as well as feature branch ephemeral environments deploys and destroys.

        :default: 30 minutes timeout, compute type MEDIUM with Linux build image Standard 6.0

        :stability: experimental
        '''
        result = self._values.get("code_build")
        return typing.cast(typing.Optional[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions], result)

    @builtins.property
    def code_pipeline(self) -> typing.Optional["CodePipelineOverrides"]:
        '''(experimental) Override CodePipeline properties.

        :stability: experimental
        '''
        result = self._values.get("code_pipeline")
        return typing.cast(typing.Optional["CodePipelineOverrides"], result)

    @builtins.property
    def commands(self) -> typing.Optional["BuildCommands"]:
        '''(experimental) Commands executed to build and deploy the application.

        :stability: experimental
        '''
        result = self._values.get("commands")
        return typing.cast(typing.Optional["BuildCommands"], result)

    @builtins.property
    def package_manager(self) -> typing.Optional[builtins.str]:
        '''(experimental) Package manager used in the repository.

        If provided, the install commands will be set to install dependencies using given package manager.

        :stability: experimental
        '''
        result = self._values.get("package_manager")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def slack_notifications(self) -> typing.Optional["SlackNotifications"]:
        '''(experimental) Configuration for Slack notifications.

        Requires configuring AWS Chatbot client manually first.

        :stability: experimental
        '''
        result = self._values.get("slack_notifications")
        return typing.cast(typing.Optional["SlackNotifications"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApplicationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="opinionated-ci-pipeline.BuildCommands",
    jsii_struct_bases=[],
    name_mapping={
        "build_and_test": "buildAndTest",
        "deploy_environment": "deployEnvironment",
        "destroy_environment": "destroyEnvironment",
        "install": "install",
        "pre_install": "preInstall",
        "synth_pipeline": "synthPipeline",
    },
)
class BuildCommands:
    def __init__(
        self,
        *,
        build_and_test: typing.Optional[typing.Sequence[builtins.str]] = None,
        deploy_environment: typing.Optional[typing.Sequence[builtins.str]] = None,
        destroy_environment: typing.Optional[typing.Sequence[builtins.str]] = None,
        install: typing.Optional[typing.Sequence[builtins.str]] = None,
        pre_install: typing.Optional[typing.Sequence[builtins.str]] = None,
        synth_pipeline: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''
        :param build_and_test: 
        :param deploy_environment: 
        :param destroy_environment: 
        :param install: 
        :param pre_install: 
        :param synth_pipeline: 

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__784009b669b30e314c38fdf9a7262dc95229d2370f3ae7a3bda7d6cf264f0c25)
            check_type(argname="argument build_and_test", value=build_and_test, expected_type=type_hints["build_and_test"])
            check_type(argname="argument deploy_environment", value=deploy_environment, expected_type=type_hints["deploy_environment"])
            check_type(argname="argument destroy_environment", value=destroy_environment, expected_type=type_hints["destroy_environment"])
            check_type(argname="argument install", value=install, expected_type=type_hints["install"])
            check_type(argname="argument pre_install", value=pre_install, expected_type=type_hints["pre_install"])
            check_type(argname="argument synth_pipeline", value=synth_pipeline, expected_type=type_hints["synth_pipeline"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if build_and_test is not None:
            self._values["build_and_test"] = build_and_test
        if deploy_environment is not None:
            self._values["deploy_environment"] = deploy_environment
        if destroy_environment is not None:
            self._values["destroy_environment"] = destroy_environment
        if install is not None:
            self._values["install"] = install
        if pre_install is not None:
            self._values["pre_install"] = pre_install
        if synth_pipeline is not None:
            self._values["synth_pipeline"] = synth_pipeline

    @builtins.property
    def build_and_test(self) -> typing.Optional[typing.List[builtins.str]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("build_and_test")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def deploy_environment(self) -> typing.Optional[typing.List[builtins.str]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("deploy_environment")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def destroy_environment(self) -> typing.Optional[typing.List[builtins.str]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("destroy_environment")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def install(self) -> typing.Optional[typing.List[builtins.str]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("install")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def pre_install(self) -> typing.Optional[typing.List[builtins.str]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("pre_install")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def synth_pipeline(self) -> typing.Optional[typing.List[builtins.str]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("synth_pipeline")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BuildCommands(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CDKApplication(
    _aws_cdk_ceddda9d.App,
    metaclass=jsii.JSIIMeta,
    jsii_type="opinionated-ci-pipeline.CDKApplication",
):
    '''
    :stability: experimental
    '''

    def __init__(
        self,
        *,
        analytics_reporting: typing.Optional[builtins.bool] = None,
        auto_synth: typing.Optional[builtins.bool] = None,
        context: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        default_stack_synthesizer: typing.Optional[_aws_cdk_ceddda9d.IReusableStackSynthesizer] = None,
        outdir: typing.Optional[builtins.str] = None,
        post_cli_context: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        stack_traces: typing.Optional[builtins.bool] = None,
        tree_metadata: typing.Optional[builtins.bool] = None,
        pipeline: typing.Sequence[typing.Union[typing.Union["WaveDeployment", typing.Dict[builtins.str, typing.Any]], typing.Union["EnvironmentDeployment", typing.Dict[builtins.str, typing.Any]]]],
        repository: typing.Union["RepositoryProps", typing.Dict[builtins.str, typing.Any]],
        stacks: "IStacksCreation",
        cdk_output_directory: typing.Optional[builtins.str] = None,
        code_build: typing.Optional[typing.Union[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        code_pipeline: typing.Optional[typing.Union["CodePipelineOverrides", typing.Dict[builtins.str, typing.Any]]] = None,
        commands: typing.Optional[typing.Union[BuildCommands, typing.Dict[builtins.str, typing.Any]]] = None,
        package_manager: typing.Optional[builtins.str] = None,
        slack_notifications: typing.Optional[typing.Union["SlackNotifications", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''
        :param analytics_reporting: Include runtime versioning information in the Stacks of this app. Default: Value of 'aws:cdk:version-reporting' context key
        :param auto_synth: Automatically call ``synth()`` before the program exits. If you set this, you don't have to call ``synth()`` explicitly. Note that this feature is only available for certain programming languages, and calling ``synth()`` is still recommended. Default: true if running via CDK CLI (``CDK_OUTDIR`` is set), ``false`` otherwise
        :param context: Additional context values for the application. Context set by the CLI or the ``context`` key in ``cdk.json`` has precedence. Context can be read from any construct using ``node.getContext(key)``. Default: - no additional context
        :param default_stack_synthesizer: The stack synthesizer to use by default for all Stacks in the App. The Stack Synthesizer controls aspects of synthesis and deployment, like how assets are referenced and what IAM roles to use. For more information, see the README of the main CDK package. Default: - A ``DefaultStackSynthesizer`` with default settings
        :param outdir: The output directory into which to emit synthesized artifacts. You should never need to set this value. By default, the value you pass to the CLI's ``--output`` flag will be used, and if you change it to a different directory the CLI will fail to pick up the generated Cloud Assembly. This property is intended for internal and testing use. Default: - If this value is *not* set, considers the environment variable ``CDK_OUTDIR``. If ``CDK_OUTDIR`` is not defined, uses a temp directory.
        :param post_cli_context: Additional context values for the application. Context provided here has precedence over context set by: - The CLI via --context - The ``context`` key in ``cdk.json`` - The ``AppProps.context`` property This property is recommended over the ``AppProps.context`` property since you can make final decision over which context value to take in your app. Context can be read from any construct using ``node.getContext(key)``. Default: - no additional context
        :param stack_traces: Include construct creation stack trace in the ``aws:cdk:trace`` metadata key of all constructs. Default: true stack traces are included unless ``aws:cdk:disable-stack-trace`` is set in the context.
        :param tree_metadata: Include construct tree metadata as part of the Cloud Assembly. Default: true
        :param pipeline: (experimental) CodePipeline deployment pipeline for the main repository branch. Can contain environments to deploy and waves that deploy multiple environments in parallel. Each environment and wave can have pre and post commands that will be executed before and after the environment or wave deployment.
        :param repository: 
        :param stacks: (experimental) An object with a create() method to create Stacks for the application. The same Stacks will be deployed with main pipeline, feature-branch builds, and local deployments.
        :param cdk_output_directory: (experimental) The location where CDK outputs synthetized files. Corresponds to the CDK Pipelines ShellStepProps#primaryOutputDirectory. Default: cdk.out
        :param code_build: (experimental) Override CodeBuild properties, used for the main pipeline as well as feature branch ephemeral environments deploys and destroys. Default: 30 minutes timeout, compute type MEDIUM with Linux build image Standard 6.0
        :param code_pipeline: (experimental) Override CodePipeline properties.
        :param commands: (experimental) Commands executed to build and deploy the application.
        :param package_manager: (experimental) Package manager used in the repository. If provided, the install commands will be set to install dependencies using given package manager.
        :param slack_notifications: (experimental) Configuration for Slack notifications. Requires configuring AWS Chatbot client manually first.

        :stability: experimental
        '''
        props = CDKApplicationProps(
            analytics_reporting=analytics_reporting,
            auto_synth=auto_synth,
            context=context,
            default_stack_synthesizer=default_stack_synthesizer,
            outdir=outdir,
            post_cli_context=post_cli_context,
            stack_traces=stack_traces,
            tree_metadata=tree_metadata,
            pipeline=pipeline,
            repository=repository,
            stacks=stacks,
            cdk_output_directory=cdk_output_directory,
            code_build=code_build,
            code_pipeline=code_pipeline,
            commands=commands,
            package_manager=package_manager,
            slack_notifications=slack_notifications,
        )

        jsii.create(self.__class__, self, [props])


@jsii.data_type(
    jsii_type="opinionated-ci-pipeline.CDKApplicationProps",
    jsii_struct_bases=[_aws_cdk_ceddda9d.AppProps, ApplicationProps],
    name_mapping={
        "analytics_reporting": "analyticsReporting",
        "auto_synth": "autoSynth",
        "context": "context",
        "default_stack_synthesizer": "defaultStackSynthesizer",
        "outdir": "outdir",
        "post_cli_context": "postCliContext",
        "stack_traces": "stackTraces",
        "tree_metadata": "treeMetadata",
        "pipeline": "pipeline",
        "repository": "repository",
        "stacks": "stacks",
        "cdk_output_directory": "cdkOutputDirectory",
        "code_build": "codeBuild",
        "code_pipeline": "codePipeline",
        "commands": "commands",
        "package_manager": "packageManager",
        "slack_notifications": "slackNotifications",
    },
)
class CDKApplicationProps(_aws_cdk_ceddda9d.AppProps, ApplicationProps):
    def __init__(
        self,
        *,
        analytics_reporting: typing.Optional[builtins.bool] = None,
        auto_synth: typing.Optional[builtins.bool] = None,
        context: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        default_stack_synthesizer: typing.Optional[_aws_cdk_ceddda9d.IReusableStackSynthesizer] = None,
        outdir: typing.Optional[builtins.str] = None,
        post_cli_context: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        stack_traces: typing.Optional[builtins.bool] = None,
        tree_metadata: typing.Optional[builtins.bool] = None,
        pipeline: typing.Sequence[typing.Union[typing.Union["WaveDeployment", typing.Dict[builtins.str, typing.Any]], typing.Union["EnvironmentDeployment", typing.Dict[builtins.str, typing.Any]]]],
        repository: typing.Union["RepositoryProps", typing.Dict[builtins.str, typing.Any]],
        stacks: "IStacksCreation",
        cdk_output_directory: typing.Optional[builtins.str] = None,
        code_build: typing.Optional[typing.Union[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        code_pipeline: typing.Optional[typing.Union["CodePipelineOverrides", typing.Dict[builtins.str, typing.Any]]] = None,
        commands: typing.Optional[typing.Union[BuildCommands, typing.Dict[builtins.str, typing.Any]]] = None,
        package_manager: typing.Optional[builtins.str] = None,
        slack_notifications: typing.Optional[typing.Union["SlackNotifications", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''
        :param analytics_reporting: Include runtime versioning information in the Stacks of this app. Default: Value of 'aws:cdk:version-reporting' context key
        :param auto_synth: Automatically call ``synth()`` before the program exits. If you set this, you don't have to call ``synth()`` explicitly. Note that this feature is only available for certain programming languages, and calling ``synth()`` is still recommended. Default: true if running via CDK CLI (``CDK_OUTDIR`` is set), ``false`` otherwise
        :param context: Additional context values for the application. Context set by the CLI or the ``context`` key in ``cdk.json`` has precedence. Context can be read from any construct using ``node.getContext(key)``. Default: - no additional context
        :param default_stack_synthesizer: The stack synthesizer to use by default for all Stacks in the App. The Stack Synthesizer controls aspects of synthesis and deployment, like how assets are referenced and what IAM roles to use. For more information, see the README of the main CDK package. Default: - A ``DefaultStackSynthesizer`` with default settings
        :param outdir: The output directory into which to emit synthesized artifacts. You should never need to set this value. By default, the value you pass to the CLI's ``--output`` flag will be used, and if you change it to a different directory the CLI will fail to pick up the generated Cloud Assembly. This property is intended for internal and testing use. Default: - If this value is *not* set, considers the environment variable ``CDK_OUTDIR``. If ``CDK_OUTDIR`` is not defined, uses a temp directory.
        :param post_cli_context: Additional context values for the application. Context provided here has precedence over context set by: - The CLI via --context - The ``context`` key in ``cdk.json`` - The ``AppProps.context`` property This property is recommended over the ``AppProps.context`` property since you can make final decision over which context value to take in your app. Context can be read from any construct using ``node.getContext(key)``. Default: - no additional context
        :param stack_traces: Include construct creation stack trace in the ``aws:cdk:trace`` metadata key of all constructs. Default: true stack traces are included unless ``aws:cdk:disable-stack-trace`` is set in the context.
        :param tree_metadata: Include construct tree metadata as part of the Cloud Assembly. Default: true
        :param pipeline: (experimental) CodePipeline deployment pipeline for the main repository branch. Can contain environments to deploy and waves that deploy multiple environments in parallel. Each environment and wave can have pre and post commands that will be executed before and after the environment or wave deployment.
        :param repository: 
        :param stacks: (experimental) An object with a create() method to create Stacks for the application. The same Stacks will be deployed with main pipeline, feature-branch builds, and local deployments.
        :param cdk_output_directory: (experimental) The location where CDK outputs synthetized files. Corresponds to the CDK Pipelines ShellStepProps#primaryOutputDirectory. Default: cdk.out
        :param code_build: (experimental) Override CodeBuild properties, used for the main pipeline as well as feature branch ephemeral environments deploys and destroys. Default: 30 minutes timeout, compute type MEDIUM with Linux build image Standard 6.0
        :param code_pipeline: (experimental) Override CodePipeline properties.
        :param commands: (experimental) Commands executed to build and deploy the application.
        :param package_manager: (experimental) Package manager used in the repository. If provided, the install commands will be set to install dependencies using given package manager.
        :param slack_notifications: (experimental) Configuration for Slack notifications. Requires configuring AWS Chatbot client manually first.

        :stability: experimental
        '''
        if isinstance(repository, dict):
            repository = RepositoryProps(**repository)
        if isinstance(code_build, dict):
            code_build = _aws_cdk_pipelines_ceddda9d.CodeBuildOptions(**code_build)
        if isinstance(code_pipeline, dict):
            code_pipeline = CodePipelineOverrides(**code_pipeline)
        if isinstance(commands, dict):
            commands = BuildCommands(**commands)
        if isinstance(slack_notifications, dict):
            slack_notifications = SlackNotifications(**slack_notifications)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__be38be79ae4e169cdad1ff5fa548afd5af816580eaf5ec1eca5799f6bf3ace32)
            check_type(argname="argument analytics_reporting", value=analytics_reporting, expected_type=type_hints["analytics_reporting"])
            check_type(argname="argument auto_synth", value=auto_synth, expected_type=type_hints["auto_synth"])
            check_type(argname="argument context", value=context, expected_type=type_hints["context"])
            check_type(argname="argument default_stack_synthesizer", value=default_stack_synthesizer, expected_type=type_hints["default_stack_synthesizer"])
            check_type(argname="argument outdir", value=outdir, expected_type=type_hints["outdir"])
            check_type(argname="argument post_cli_context", value=post_cli_context, expected_type=type_hints["post_cli_context"])
            check_type(argname="argument stack_traces", value=stack_traces, expected_type=type_hints["stack_traces"])
            check_type(argname="argument tree_metadata", value=tree_metadata, expected_type=type_hints["tree_metadata"])
            check_type(argname="argument pipeline", value=pipeline, expected_type=type_hints["pipeline"])
            check_type(argname="argument repository", value=repository, expected_type=type_hints["repository"])
            check_type(argname="argument stacks", value=stacks, expected_type=type_hints["stacks"])
            check_type(argname="argument cdk_output_directory", value=cdk_output_directory, expected_type=type_hints["cdk_output_directory"])
            check_type(argname="argument code_build", value=code_build, expected_type=type_hints["code_build"])
            check_type(argname="argument code_pipeline", value=code_pipeline, expected_type=type_hints["code_pipeline"])
            check_type(argname="argument commands", value=commands, expected_type=type_hints["commands"])
            check_type(argname="argument package_manager", value=package_manager, expected_type=type_hints["package_manager"])
            check_type(argname="argument slack_notifications", value=slack_notifications, expected_type=type_hints["slack_notifications"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "pipeline": pipeline,
            "repository": repository,
            "stacks": stacks,
        }
        if analytics_reporting is not None:
            self._values["analytics_reporting"] = analytics_reporting
        if auto_synth is not None:
            self._values["auto_synth"] = auto_synth
        if context is not None:
            self._values["context"] = context
        if default_stack_synthesizer is not None:
            self._values["default_stack_synthesizer"] = default_stack_synthesizer
        if outdir is not None:
            self._values["outdir"] = outdir
        if post_cli_context is not None:
            self._values["post_cli_context"] = post_cli_context
        if stack_traces is not None:
            self._values["stack_traces"] = stack_traces
        if tree_metadata is not None:
            self._values["tree_metadata"] = tree_metadata
        if cdk_output_directory is not None:
            self._values["cdk_output_directory"] = cdk_output_directory
        if code_build is not None:
            self._values["code_build"] = code_build
        if code_pipeline is not None:
            self._values["code_pipeline"] = code_pipeline
        if commands is not None:
            self._values["commands"] = commands
        if package_manager is not None:
            self._values["package_manager"] = package_manager
        if slack_notifications is not None:
            self._values["slack_notifications"] = slack_notifications

    @builtins.property
    def analytics_reporting(self) -> typing.Optional[builtins.bool]:
        '''Include runtime versioning information in the Stacks of this app.

        :default: Value of 'aws:cdk:version-reporting' context key
        '''
        result = self._values.get("analytics_reporting")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def auto_synth(self) -> typing.Optional[builtins.bool]:
        '''Automatically call ``synth()`` before the program exits.

        If you set this, you don't have to call ``synth()`` explicitly. Note that
        this feature is only available for certain programming languages, and
        calling ``synth()`` is still recommended.

        :default:

        true if running via CDK CLI (``CDK_OUTDIR`` is set), ``false``
        otherwise
        '''
        result = self._values.get("auto_synth")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def context(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Additional context values for the application.

        Context set by the CLI or the ``context`` key in ``cdk.json`` has precedence.

        Context can be read from any construct using ``node.getContext(key)``.

        :default: - no additional context
        '''
        result = self._values.get("context")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def default_stack_synthesizer(
        self,
    ) -> typing.Optional[_aws_cdk_ceddda9d.IReusableStackSynthesizer]:
        '''The stack synthesizer to use by default for all Stacks in the App.

        The Stack Synthesizer controls aspects of synthesis and deployment,
        like how assets are referenced and what IAM roles to use. For more
        information, see the README of the main CDK package.

        :default: - A ``DefaultStackSynthesizer`` with default settings
        '''
        result = self._values.get("default_stack_synthesizer")
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.IReusableStackSynthesizer], result)

    @builtins.property
    def outdir(self) -> typing.Optional[builtins.str]:
        '''The output directory into which to emit synthesized artifacts.

        You should never need to set this value. By default, the value you pass to
        the CLI's ``--output`` flag will be used, and if you change it to a different
        directory the CLI will fail to pick up the generated Cloud Assembly.

        This property is intended for internal and testing use.

        :default:

        - If this value is *not* set, considers the environment variable ``CDK_OUTDIR``.
        If ``CDK_OUTDIR`` is not defined, uses a temp directory.
        '''
        result = self._values.get("outdir")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def post_cli_context(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Additional context values for the application.

        Context provided here has precedence over context set by:

        - The CLI via --context
        - The ``context`` key in ``cdk.json``
        - The ``AppProps.context`` property

        This property is recommended over the ``AppProps.context`` property since you
        can make final decision over which context value to take in your app.

        Context can be read from any construct using ``node.getContext(key)``.

        :default: - no additional context

        Example::

            // context from the CLI and from `cdk.json` are stored in the
            // CDK_CONTEXT env variable
            const cliContext = JSON.parse(process.env.CDK_CONTEXT!);
            
            // determine whether to take the context passed in the CLI or not
            const determineValue = process.env.PROD ? cliContext.SOMEKEY : 'my-prod-value';
            new App({
              postCliContext: {
                SOMEKEY: determineValue,
              },
            });
        '''
        result = self._values.get("post_cli_context")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def stack_traces(self) -> typing.Optional[builtins.bool]:
        '''Include construct creation stack trace in the ``aws:cdk:trace`` metadata key of all constructs.

        :default: true stack traces are included unless ``aws:cdk:disable-stack-trace`` is set in the context.
        '''
        result = self._values.get("stack_traces")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def tree_metadata(self) -> typing.Optional[builtins.bool]:
        '''Include construct tree metadata as part of the Cloud Assembly.

        :default: true
        '''
        result = self._values.get("tree_metadata")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def pipeline(
        self,
    ) -> typing.List[typing.Union["WaveDeployment", "EnvironmentDeployment"]]:
        '''(experimental) CodePipeline deployment pipeline for the main repository branch.

        Can contain environments to deploy
        and waves that deploy multiple environments in parallel.

        Each environment and wave can have pre and post commands
        that will be executed before and after the environment or wave deployment.

        :stability: experimental
        '''
        result = self._values.get("pipeline")
        assert result is not None, "Required property 'pipeline' is missing"
        return typing.cast(typing.List[typing.Union["WaveDeployment", "EnvironmentDeployment"]], result)

    @builtins.property
    def repository(self) -> "RepositoryProps":
        '''
        :stability: experimental
        '''
        result = self._values.get("repository")
        assert result is not None, "Required property 'repository' is missing"
        return typing.cast("RepositoryProps", result)

    @builtins.property
    def stacks(self) -> "IStacksCreation":
        '''(experimental) An object with a create() method to create Stacks for the application.

        The same Stacks will be deployed with main pipeline, feature-branch builds, and local deployments.

        :stability: experimental
        '''
        result = self._values.get("stacks")
        assert result is not None, "Required property 'stacks' is missing"
        return typing.cast("IStacksCreation", result)

    @builtins.property
    def cdk_output_directory(self) -> typing.Optional[builtins.str]:
        '''(experimental) The location where CDK outputs synthetized files.

        Corresponds to the CDK Pipelines ShellStepProps#primaryOutputDirectory.

        :default: cdk.out

        :stability: experimental
        '''
        result = self._values.get("cdk_output_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def code_build(
        self,
    ) -> typing.Optional[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions]:
        '''(experimental) Override CodeBuild properties, used for the main pipeline as well as feature branch ephemeral environments deploys and destroys.

        :default: 30 minutes timeout, compute type MEDIUM with Linux build image Standard 6.0

        :stability: experimental
        '''
        result = self._values.get("code_build")
        return typing.cast(typing.Optional[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions], result)

    @builtins.property
    def code_pipeline(self) -> typing.Optional["CodePipelineOverrides"]:
        '''(experimental) Override CodePipeline properties.

        :stability: experimental
        '''
        result = self._values.get("code_pipeline")
        return typing.cast(typing.Optional["CodePipelineOverrides"], result)

    @builtins.property
    def commands(self) -> typing.Optional[BuildCommands]:
        '''(experimental) Commands executed to build and deploy the application.

        :stability: experimental
        '''
        result = self._values.get("commands")
        return typing.cast(typing.Optional[BuildCommands], result)

    @builtins.property
    def package_manager(self) -> typing.Optional[builtins.str]:
        '''(experimental) Package manager used in the repository.

        If provided, the install commands will be set to install dependencies using given package manager.

        :stability: experimental
        '''
        result = self._values.get("package_manager")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def slack_notifications(self) -> typing.Optional["SlackNotifications"]:
        '''(experimental) Configuration for Slack notifications.

        Requires configuring AWS Chatbot client manually first.

        :stability: experimental
        '''
        result = self._values.get("slack_notifications")
        return typing.cast(typing.Optional["SlackNotifications"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CDKApplicationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="opinionated-ci-pipeline.CodePipelineOverrides",
    jsii_struct_bases=[],
    name_mapping={
        "asset_publishing_code_build_defaults": "assetPublishingCodeBuildDefaults",
        "code_build_defaults": "codeBuildDefaults",
        "docker_credentials": "dockerCredentials",
        "docker_enabled_for_self_mutation": "dockerEnabledForSelfMutation",
        "docker_enabled_for_synth": "dockerEnabledForSynth",
        "enable_key_rotation": "enableKeyRotation",
        "pipeline_name": "pipelineName",
        "publish_assets_in_parallel": "publishAssetsInParallel",
        "reuse_cross_region_support_stacks": "reuseCrossRegionSupportStacks",
        "role": "role",
        "self_mutation": "selfMutation",
        "self_mutation_code_build_defaults": "selfMutationCodeBuildDefaults",
        "synth_code_build_defaults": "synthCodeBuildDefaults",
        "use_change_sets": "useChangeSets",
    },
)
class CodePipelineOverrides:
    def __init__(
        self,
        *,
        asset_publishing_code_build_defaults: typing.Optional[typing.Union[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        code_build_defaults: typing.Optional[typing.Union[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        docker_credentials: typing.Optional[typing.Sequence[_aws_cdk_pipelines_ceddda9d.DockerCredential]] = None,
        docker_enabled_for_self_mutation: typing.Optional[builtins.bool] = None,
        docker_enabled_for_synth: typing.Optional[builtins.bool] = None,
        enable_key_rotation: typing.Optional[builtins.bool] = None,
        pipeline_name: typing.Optional[builtins.str] = None,
        publish_assets_in_parallel: typing.Optional[builtins.bool] = None,
        reuse_cross_region_support_stacks: typing.Optional[builtins.bool] = None,
        role: typing.Optional[_aws_cdk_aws_iam_ceddda9d.IRole] = None,
        self_mutation: typing.Optional[builtins.bool] = None,
        self_mutation_code_build_defaults: typing.Optional[typing.Union[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        synth_code_build_defaults: typing.Optional[typing.Union[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        use_change_sets: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) Since jsii does not support Partial or Omit, we have to define all properties from CodePipelineProps that may be overriden manually.

        :param asset_publishing_code_build_defaults: 
        :param code_build_defaults: 
        :param docker_credentials: 
        :param docker_enabled_for_self_mutation: 
        :param docker_enabled_for_synth: 
        :param enable_key_rotation: 
        :param pipeline_name: 
        :param publish_assets_in_parallel: 
        :param reuse_cross_region_support_stacks: 
        :param role: 
        :param self_mutation: 
        :param self_mutation_code_build_defaults: 
        :param synth_code_build_defaults: 
        :param use_change_sets: 

        :stability: experimental
        '''
        if isinstance(asset_publishing_code_build_defaults, dict):
            asset_publishing_code_build_defaults = _aws_cdk_pipelines_ceddda9d.CodeBuildOptions(**asset_publishing_code_build_defaults)
        if isinstance(code_build_defaults, dict):
            code_build_defaults = _aws_cdk_pipelines_ceddda9d.CodeBuildOptions(**code_build_defaults)
        if isinstance(self_mutation_code_build_defaults, dict):
            self_mutation_code_build_defaults = _aws_cdk_pipelines_ceddda9d.CodeBuildOptions(**self_mutation_code_build_defaults)
        if isinstance(synth_code_build_defaults, dict):
            synth_code_build_defaults = _aws_cdk_pipelines_ceddda9d.CodeBuildOptions(**synth_code_build_defaults)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b7bac62224aec74b6d37f678f5bdede0184616abc1fb2eafb3a4aa19f1c20fd4)
            check_type(argname="argument asset_publishing_code_build_defaults", value=asset_publishing_code_build_defaults, expected_type=type_hints["asset_publishing_code_build_defaults"])
            check_type(argname="argument code_build_defaults", value=code_build_defaults, expected_type=type_hints["code_build_defaults"])
            check_type(argname="argument docker_credentials", value=docker_credentials, expected_type=type_hints["docker_credentials"])
            check_type(argname="argument docker_enabled_for_self_mutation", value=docker_enabled_for_self_mutation, expected_type=type_hints["docker_enabled_for_self_mutation"])
            check_type(argname="argument docker_enabled_for_synth", value=docker_enabled_for_synth, expected_type=type_hints["docker_enabled_for_synth"])
            check_type(argname="argument enable_key_rotation", value=enable_key_rotation, expected_type=type_hints["enable_key_rotation"])
            check_type(argname="argument pipeline_name", value=pipeline_name, expected_type=type_hints["pipeline_name"])
            check_type(argname="argument publish_assets_in_parallel", value=publish_assets_in_parallel, expected_type=type_hints["publish_assets_in_parallel"])
            check_type(argname="argument reuse_cross_region_support_stacks", value=reuse_cross_region_support_stacks, expected_type=type_hints["reuse_cross_region_support_stacks"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument self_mutation", value=self_mutation, expected_type=type_hints["self_mutation"])
            check_type(argname="argument self_mutation_code_build_defaults", value=self_mutation_code_build_defaults, expected_type=type_hints["self_mutation_code_build_defaults"])
            check_type(argname="argument synth_code_build_defaults", value=synth_code_build_defaults, expected_type=type_hints["synth_code_build_defaults"])
            check_type(argname="argument use_change_sets", value=use_change_sets, expected_type=type_hints["use_change_sets"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if asset_publishing_code_build_defaults is not None:
            self._values["asset_publishing_code_build_defaults"] = asset_publishing_code_build_defaults
        if code_build_defaults is not None:
            self._values["code_build_defaults"] = code_build_defaults
        if docker_credentials is not None:
            self._values["docker_credentials"] = docker_credentials
        if docker_enabled_for_self_mutation is not None:
            self._values["docker_enabled_for_self_mutation"] = docker_enabled_for_self_mutation
        if docker_enabled_for_synth is not None:
            self._values["docker_enabled_for_synth"] = docker_enabled_for_synth
        if enable_key_rotation is not None:
            self._values["enable_key_rotation"] = enable_key_rotation
        if pipeline_name is not None:
            self._values["pipeline_name"] = pipeline_name
        if publish_assets_in_parallel is not None:
            self._values["publish_assets_in_parallel"] = publish_assets_in_parallel
        if reuse_cross_region_support_stacks is not None:
            self._values["reuse_cross_region_support_stacks"] = reuse_cross_region_support_stacks
        if role is not None:
            self._values["role"] = role
        if self_mutation is not None:
            self._values["self_mutation"] = self_mutation
        if self_mutation_code_build_defaults is not None:
            self._values["self_mutation_code_build_defaults"] = self_mutation_code_build_defaults
        if synth_code_build_defaults is not None:
            self._values["synth_code_build_defaults"] = synth_code_build_defaults
        if use_change_sets is not None:
            self._values["use_change_sets"] = use_change_sets

    @builtins.property
    def asset_publishing_code_build_defaults(
        self,
    ) -> typing.Optional[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions]:
        '''
        :stability: experimental
        '''
        result = self._values.get("asset_publishing_code_build_defaults")
        return typing.cast(typing.Optional[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions], result)

    @builtins.property
    def code_build_defaults(
        self,
    ) -> typing.Optional[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions]:
        '''
        :stability: experimental
        '''
        result = self._values.get("code_build_defaults")
        return typing.cast(typing.Optional[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions], result)

    @builtins.property
    def docker_credentials(
        self,
    ) -> typing.Optional[typing.List[_aws_cdk_pipelines_ceddda9d.DockerCredential]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("docker_credentials")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_pipelines_ceddda9d.DockerCredential]], result)

    @builtins.property
    def docker_enabled_for_self_mutation(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        result = self._values.get("docker_enabled_for_self_mutation")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def docker_enabled_for_synth(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        result = self._values.get("docker_enabled_for_synth")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def enable_key_rotation(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        result = self._values.get("enable_key_rotation")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def pipeline_name(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        result = self._values.get("pipeline_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def publish_assets_in_parallel(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        result = self._values.get("publish_assets_in_parallel")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def reuse_cross_region_support_stacks(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        result = self._values.get("reuse_cross_region_support_stacks")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def role(self) -> typing.Optional[_aws_cdk_aws_iam_ceddda9d.IRole]:
        '''
        :stability: experimental
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_ceddda9d.IRole], result)

    @builtins.property
    def self_mutation(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        result = self._values.get("self_mutation")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def self_mutation_code_build_defaults(
        self,
    ) -> typing.Optional[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions]:
        '''
        :stability: experimental
        '''
        result = self._values.get("self_mutation_code_build_defaults")
        return typing.cast(typing.Optional[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions], result)

    @builtins.property
    def synth_code_build_defaults(
        self,
    ) -> typing.Optional[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions]:
        '''
        :stability: experimental
        '''
        result = self._values.get("synth_code_build_defaults")
        return typing.cast(typing.Optional[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions], result)

    @builtins.property
    def use_change_sets(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        result = self._values.get("use_change_sets")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CodePipelineOverrides(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="opinionated-ci-pipeline.EnvironmentDeployment",
    jsii_struct_bases=[],
    name_mapping={"environment": "environment", "post": "post", "pre": "pre"},
)
class EnvironmentDeployment:
    def __init__(
        self,
        *,
        environment: builtins.str,
        post: typing.Optional[typing.Sequence[builtins.str]] = None,
        pre: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''
        :param environment: (experimental) Environment name. Environment will be deployed to AWS account and region defined in cdk.json file ``context/environments`` properties, falling back to the ``default`` environment settings if given environment configuration is not found.
        :param post: (experimental) Commands to execute after the environment deployment.
        :param pre: (experimental) Commands to execute before the environment deployment.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8a56d8f8652385a6beef9373eed0946aec4a1e2ca901b17252542af97a91c99a)
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument post", value=post, expected_type=type_hints["post"])
            check_type(argname="argument pre", value=pre, expected_type=type_hints["pre"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "environment": environment,
        }
        if post is not None:
            self._values["post"] = post
        if pre is not None:
            self._values["pre"] = pre

    @builtins.property
    def environment(self) -> builtins.str:
        '''(experimental) Environment name.

        Environment will be deployed to AWS account and region
        defined in cdk.json file ``context/environments`` properties,
        falling back to the ``default`` environment settings if given environment configuration is not found.

        :stability: experimental
        '''
        result = self._values.get("environment")
        assert result is not None, "Required property 'environment' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def post(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) Commands to execute after the environment deployment.

        :stability: experimental
        '''
        result = self._values.get("post")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def pre(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) Commands to execute before the environment deployment.

        :stability: experimental
        '''
        result = self._values.get("pre")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EnvironmentDeployment(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.interface(jsii_type="opinionated-ci-pipeline.IStacksCreation")
class IStacksCreation(typing_extensions.Protocol):
    '''(experimental) To provide a method as parameter, jsii requires creating a behavioral interface, prefixed with "I".

    Mixing structural and behavioral interfaces is not always possible, hence we extract stacks creation
    to a separate object described by this behavioral interface.

    :stability: experimental
    '''

    @jsii.member(jsii_name="create")
    def create(
        self,
        scope: _constructs_77d1e7e8.Construct,
        project_name: builtins.str,
        env_name: builtins.str,
    ) -> None:
        '''(experimental) Create Stacks for the application.

        Use provided scope as stacks' parent (first constructor argument).

        Stacks must include provided environment name in their names
        to distinguish them when deploying multiple environments
        (like feature-branch environments) to the same account.

        :param scope: -
        :param project_name: -
        :param env_name: -

        :stability: experimental
        '''
        ...


class _IStacksCreationProxy:
    '''(experimental) To provide a method as parameter, jsii requires creating a behavioral interface, prefixed with "I".

    Mixing structural and behavioral interfaces is not always possible, hence we extract stacks creation
    to a separate object described by this behavioral interface.

    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "opinionated-ci-pipeline.IStacksCreation"

    @jsii.member(jsii_name="create")
    def create(
        self,
        scope: _constructs_77d1e7e8.Construct,
        project_name: builtins.str,
        env_name: builtins.str,
    ) -> None:
        '''(experimental) Create Stacks for the application.

        Use provided scope as stacks' parent (first constructor argument).

        Stacks must include provided environment name in their names
        to distinguish them when deploying multiple environments
        (like feature-branch environments) to the same account.

        :param scope: -
        :param project_name: -
        :param env_name: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6f94daa933ae3ad6d5e4b09fd8274268c1c6c4ce67db84eea5360a505e3c5d56)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument project_name", value=project_name, expected_type=type_hints["project_name"])
            check_type(argname="argument env_name", value=env_name, expected_type=type_hints["env_name"])
        return typing.cast(None, jsii.invoke(self, "create", [scope, project_name, env_name]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IStacksCreation).__jsii_proxy_class__ = lambda : _IStacksCreationProxy


@jsii.data_type(
    jsii_type="opinionated-ci-pipeline.RepositoryProps",
    jsii_struct_bases=[],
    name_mapping={"host": "host", "name": "name", "default_branch": "defaultBranch"},
)
class RepositoryProps:
    def __init__(
        self,
        *,
        host: builtins.str,
        name: builtins.str,
        default_branch: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param host: (experimental) Repository hosting.
        :param name: (experimental) Like "my-comapny/my-repo".
        :param default_branch: (experimental) Branch to deploy the environments from in the main pipeline. Default: main

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f6f830ab1d423354edc601071d8aae1b0849f4eb33644f06838e41508d8ba714)
            check_type(argname="argument host", value=host, expected_type=type_hints["host"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument default_branch", value=default_branch, expected_type=type_hints["default_branch"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "host": host,
            "name": name,
        }
        if default_branch is not None:
            self._values["default_branch"] = default_branch

    @builtins.property
    def host(self) -> builtins.str:
        '''(experimental) Repository hosting.

        :stability: experimental
        '''
        result = self._values.get("host")
        assert result is not None, "Required property 'host' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''(experimental) Like "my-comapny/my-repo".

        :stability: experimental
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def default_branch(self) -> typing.Optional[builtins.str]:
        '''(experimental) Branch to deploy the environments from in the main pipeline.

        :default: main

        :stability: experimental
        '''
        result = self._values.get("default_branch")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RepositoryProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="opinionated-ci-pipeline.SlackNotifications",
    jsii_struct_bases=[],
    name_mapping={
        "channel_id": "channelId",
        "workspace_id": "workspaceId",
        "feature_branch_failures": "featureBranchFailures",
        "main_pipeline_failures": "mainPipelineFailures",
    },
)
class SlackNotifications:
    def __init__(
        self,
        *,
        channel_id: builtins.str,
        workspace_id: builtins.str,
        feature_branch_failures: typing.Optional[builtins.bool] = None,
        main_pipeline_failures: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param channel_id: 
        :param workspace_id: 
        :param feature_branch_failures: (experimental) Whether to notify about feature branch deployment failures. Default: false
        :param main_pipeline_failures: (experimental) Whether to notify about main pipeline failures. Default: true

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__62e47fdb1eef2b0738fcc2152a329dd258aee175431814841e5687481f56c87b)
            check_type(argname="argument channel_id", value=channel_id, expected_type=type_hints["channel_id"])
            check_type(argname="argument workspace_id", value=workspace_id, expected_type=type_hints["workspace_id"])
            check_type(argname="argument feature_branch_failures", value=feature_branch_failures, expected_type=type_hints["feature_branch_failures"])
            check_type(argname="argument main_pipeline_failures", value=main_pipeline_failures, expected_type=type_hints["main_pipeline_failures"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "channel_id": channel_id,
            "workspace_id": workspace_id,
        }
        if feature_branch_failures is not None:
            self._values["feature_branch_failures"] = feature_branch_failures
        if main_pipeline_failures is not None:
            self._values["main_pipeline_failures"] = main_pipeline_failures

    @builtins.property
    def channel_id(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        result = self._values.get("channel_id")
        assert result is not None, "Required property 'channel_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def workspace_id(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        result = self._values.get("workspace_id")
        assert result is not None, "Required property 'workspace_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def feature_branch_failures(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to notify about feature branch deployment failures.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("feature_branch_failures")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def main_pipeline_failures(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to notify about main pipeline failures.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("main_pipeline_failures")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SlackNotifications(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="opinionated-ci-pipeline.WaveDeployment",
    jsii_struct_bases=[],
    name_mapping={
        "environments": "environments",
        "wave": "wave",
        "post": "post",
        "post_each_environment": "postEachEnvironment",
        "pre": "pre",
        "pre_each_environment": "preEachEnvironment",
    },
)
class WaveDeployment:
    def __init__(
        self,
        *,
        environments: typing.Sequence[typing.Union[EnvironmentDeployment, typing.Dict[builtins.str, typing.Any]]],
        wave: builtins.str,
        post: typing.Optional[typing.Sequence[builtins.str]] = None,
        post_each_environment: typing.Optional[typing.Sequence[builtins.str]] = None,
        pre: typing.Optional[typing.Sequence[builtins.str]] = None,
        pre_each_environment: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''
        :param environments: (experimental) List of environments to deploy in parallel.
        :param wave: (experimental) Wave name.
        :param post: (experimental) Commands to execute after the wave deployment.
        :param post_each_environment: (experimental) Commands to execute after environment deployment. If environment configuration also contains commands to execute post-deployment, they will be executed before the commands defined here.
        :param pre: (experimental) Commands to execute before the wave deployment.
        :param pre_each_environment: (experimental) Commands to execute before each environment deployment. If environment configuration also contains commands to execute pre-deployment, they will be executed after the commands defined here.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3e2f9daf95ae58e92db6a6c5f2f46d1d02b4f767b770eaf9116fc04ca2c5158e)
            check_type(argname="argument environments", value=environments, expected_type=type_hints["environments"])
            check_type(argname="argument wave", value=wave, expected_type=type_hints["wave"])
            check_type(argname="argument post", value=post, expected_type=type_hints["post"])
            check_type(argname="argument post_each_environment", value=post_each_environment, expected_type=type_hints["post_each_environment"])
            check_type(argname="argument pre", value=pre, expected_type=type_hints["pre"])
            check_type(argname="argument pre_each_environment", value=pre_each_environment, expected_type=type_hints["pre_each_environment"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "environments": environments,
            "wave": wave,
        }
        if post is not None:
            self._values["post"] = post
        if post_each_environment is not None:
            self._values["post_each_environment"] = post_each_environment
        if pre is not None:
            self._values["pre"] = pre
        if pre_each_environment is not None:
            self._values["pre_each_environment"] = pre_each_environment

    @builtins.property
    def environments(self) -> typing.List[EnvironmentDeployment]:
        '''(experimental) List of environments to deploy in parallel.

        :stability: experimental
        '''
        result = self._values.get("environments")
        assert result is not None, "Required property 'environments' is missing"
        return typing.cast(typing.List[EnvironmentDeployment], result)

    @builtins.property
    def wave(self) -> builtins.str:
        '''(experimental) Wave name.

        :stability: experimental
        '''
        result = self._values.get("wave")
        assert result is not None, "Required property 'wave' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def post(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) Commands to execute after the wave deployment.

        :stability: experimental
        '''
        result = self._values.get("post")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def post_each_environment(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) Commands to execute after environment deployment.

        If environment configuration also contains commands to execute post-deployment,
        they will be executed before the commands defined here.

        :stability: experimental
        '''
        result = self._values.get("post_each_environment")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def pre(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) Commands to execute before the wave deployment.

        :stability: experimental
        '''
        result = self._values.get("pre")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def pre_each_environment(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) Commands to execute before each environment deployment.

        If environment configuration also contains commands to execute pre-deployment,
        they will be executed after the commands defined here.

        :stability: experimental
        '''
        result = self._values.get("pre_each_environment")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "WaveDeployment(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


__all__ = [
    "ApplicationProps",
    "BuildCommands",
    "CDKApplication",
    "CDKApplicationProps",
    "CodePipelineOverrides",
    "EnvironmentDeployment",
    "IStacksCreation",
    "RepositoryProps",
    "SlackNotifications",
    "WaveDeployment",
]

publication.publish()

def _typecheckingstub__d22629910ce624c4144f7d3a67b45ba1c7b1018575c9f66d6177bc139efb6eb9(
    *,
    pipeline: typing.Sequence[typing.Union[typing.Union[WaveDeployment, typing.Dict[builtins.str, typing.Any]], typing.Union[EnvironmentDeployment, typing.Dict[builtins.str, typing.Any]]]],
    repository: typing.Union[RepositoryProps, typing.Dict[builtins.str, typing.Any]],
    stacks: IStacksCreation,
    cdk_output_directory: typing.Optional[builtins.str] = None,
    code_build: typing.Optional[typing.Union[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions, typing.Dict[builtins.str, typing.Any]]] = None,
    code_pipeline: typing.Optional[typing.Union[CodePipelineOverrides, typing.Dict[builtins.str, typing.Any]]] = None,
    commands: typing.Optional[typing.Union[BuildCommands, typing.Dict[builtins.str, typing.Any]]] = None,
    package_manager: typing.Optional[builtins.str] = None,
    slack_notifications: typing.Optional[typing.Union[SlackNotifications, typing.Dict[builtins.str, typing.Any]]] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__784009b669b30e314c38fdf9a7262dc95229d2370f3ae7a3bda7d6cf264f0c25(
    *,
    build_and_test: typing.Optional[typing.Sequence[builtins.str]] = None,
    deploy_environment: typing.Optional[typing.Sequence[builtins.str]] = None,
    destroy_environment: typing.Optional[typing.Sequence[builtins.str]] = None,
    install: typing.Optional[typing.Sequence[builtins.str]] = None,
    pre_install: typing.Optional[typing.Sequence[builtins.str]] = None,
    synth_pipeline: typing.Optional[typing.Sequence[builtins.str]] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__be38be79ae4e169cdad1ff5fa548afd5af816580eaf5ec1eca5799f6bf3ace32(
    *,
    analytics_reporting: typing.Optional[builtins.bool] = None,
    auto_synth: typing.Optional[builtins.bool] = None,
    context: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    default_stack_synthesizer: typing.Optional[_aws_cdk_ceddda9d.IReusableStackSynthesizer] = None,
    outdir: typing.Optional[builtins.str] = None,
    post_cli_context: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    stack_traces: typing.Optional[builtins.bool] = None,
    tree_metadata: typing.Optional[builtins.bool] = None,
    pipeline: typing.Sequence[typing.Union[typing.Union[WaveDeployment, typing.Dict[builtins.str, typing.Any]], typing.Union[EnvironmentDeployment, typing.Dict[builtins.str, typing.Any]]]],
    repository: typing.Union[RepositoryProps, typing.Dict[builtins.str, typing.Any]],
    stacks: IStacksCreation,
    cdk_output_directory: typing.Optional[builtins.str] = None,
    code_build: typing.Optional[typing.Union[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions, typing.Dict[builtins.str, typing.Any]]] = None,
    code_pipeline: typing.Optional[typing.Union[CodePipelineOverrides, typing.Dict[builtins.str, typing.Any]]] = None,
    commands: typing.Optional[typing.Union[BuildCommands, typing.Dict[builtins.str, typing.Any]]] = None,
    package_manager: typing.Optional[builtins.str] = None,
    slack_notifications: typing.Optional[typing.Union[SlackNotifications, typing.Dict[builtins.str, typing.Any]]] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__b7bac62224aec74b6d37f678f5bdede0184616abc1fb2eafb3a4aa19f1c20fd4(
    *,
    asset_publishing_code_build_defaults: typing.Optional[typing.Union[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions, typing.Dict[builtins.str, typing.Any]]] = None,
    code_build_defaults: typing.Optional[typing.Union[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions, typing.Dict[builtins.str, typing.Any]]] = None,
    docker_credentials: typing.Optional[typing.Sequence[_aws_cdk_pipelines_ceddda9d.DockerCredential]] = None,
    docker_enabled_for_self_mutation: typing.Optional[builtins.bool] = None,
    docker_enabled_for_synth: typing.Optional[builtins.bool] = None,
    enable_key_rotation: typing.Optional[builtins.bool] = None,
    pipeline_name: typing.Optional[builtins.str] = None,
    publish_assets_in_parallel: typing.Optional[builtins.bool] = None,
    reuse_cross_region_support_stacks: typing.Optional[builtins.bool] = None,
    role: typing.Optional[_aws_cdk_aws_iam_ceddda9d.IRole] = None,
    self_mutation: typing.Optional[builtins.bool] = None,
    self_mutation_code_build_defaults: typing.Optional[typing.Union[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions, typing.Dict[builtins.str, typing.Any]]] = None,
    synth_code_build_defaults: typing.Optional[typing.Union[_aws_cdk_pipelines_ceddda9d.CodeBuildOptions, typing.Dict[builtins.str, typing.Any]]] = None,
    use_change_sets: typing.Optional[builtins.bool] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__8a56d8f8652385a6beef9373eed0946aec4a1e2ca901b17252542af97a91c99a(
    *,
    environment: builtins.str,
    post: typing.Optional[typing.Sequence[builtins.str]] = None,
    pre: typing.Optional[typing.Sequence[builtins.str]] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__6f94daa933ae3ad6d5e4b09fd8274268c1c6c4ce67db84eea5360a505e3c5d56(
    scope: _constructs_77d1e7e8.Construct,
    project_name: builtins.str,
    env_name: builtins.str,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__f6f830ab1d423354edc601071d8aae1b0849f4eb33644f06838e41508d8ba714(
    *,
    host: builtins.str,
    name: builtins.str,
    default_branch: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__62e47fdb1eef2b0738fcc2152a329dd258aee175431814841e5687481f56c87b(
    *,
    channel_id: builtins.str,
    workspace_id: builtins.str,
    feature_branch_failures: typing.Optional[builtins.bool] = None,
    main_pipeline_failures: typing.Optional[builtins.bool] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__3e2f9daf95ae58e92db6a6c5f2f46d1d02b4f767b770eaf9116fc04ca2c5158e(
    *,
    environments: typing.Sequence[typing.Union[EnvironmentDeployment, typing.Dict[builtins.str, typing.Any]]],
    wave: builtins.str,
    post: typing.Optional[typing.Sequence[builtins.str]] = None,
    post_each_environment: typing.Optional[typing.Sequence[builtins.str]] = None,
    pre: typing.Optional[typing.Sequence[builtins.str]] = None,
    pre_each_environment: typing.Optional[typing.Sequence[builtins.str]] = None,
) -> None:
    """Type checking stubs"""
    pass
