from typing import Optional, Tuple
from jaxtyping import PyTree
import jax
import jax.numpy as jnp
import jax.tree_util as jtu
import equinox as eqx
from ..global_defs import get_real_dtype
from ..utils import filter_tree_map


@jtu.register_pytree_node_class
class Samples:
    r"""
    The samples generated by the sampler.
    This class is jittable, and there are 3 attributes.

    spins:
        The spin configurations

    wave_function:
        The wave_function of the spin configurations

    reweight_factor:
        According to

        .. math::

            \left< x \right>_p = \frac{\sum_s p_s x_s}{\sum_s p_s}
            = \frac{\sum_s q_s x_s p_s/q_s}{\sum_s q_s p_s/q_s}
            = \frac{\left< x p/q \right>_q}{\left< p/q \right>_q},

        the expectation value with probability distribution p can be computed from
        samples with a different probability distribution q.

        The reweighting factor is defined as

        .. math::
            r_s = \frac{p_s/q_s}{\left< p/q \right>_q},

        so that :math:`\left< x \right>_p = \left< r x \right>_q`.

        Usually, :math:`p_s = |\psi(s)|^2` is the target probability,
        and :math:`q_s` can be chosen as :math:`|\psi(s)|^n` or computed from a
        helper neural network. In the former case,
        :math:`r_s = \frac{|\psi_s|^{2-n}}{\left< |\psi|^{2-n} \right>}`

    state_internal:
        The internal status of samples for the forward pass.
    """

    def __init__(
        self,
        spins: jax.Array,
        wave_function: jax.Array,
        state_internal: PyTree = None,
        reweight_factor: Optional[jax.Array] = None
    ):
        """
        :param spins:
            The spin configurations

        :param wave_function:
            The wave_function of the spin configurations

        :param reweight:
            Either a number :math:`n` specifying the reweighting probability :math:`|\psi(s)|^n`,
            or the unnormalized reweighting factor :math:`r'_s = p_s/q_s`
        """
        self.spins = spins
        self.wave_function = wave_function
        self.state_internal = state_internal

        if reweight_factor is None and eqx.is_array(wave_function):
            reweight_factor = jnp.ones_like(wave_function, dtype=get_real_dtype())
        self.reweight_factor = reweight_factor

    @property
    def nsamples(self) -> int:
        return self.spins.shape[0]

    def tree_flatten(self) -> Tuple:
        children = (
            self.spins,
            self.wave_function,
            self.state_internal,
            self.reweight_factor,
        )
        aux_data = None
        return (children, aux_data)

    @classmethod
    def tree_unflatten(cls, aux_data, children):
        return cls(*children)

    def __getitem__(self, idx):
        f = lambda x: x[idx]
        return filter_tree_map(f, self)
