"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.importCert = exports.AcmCertificateImporter = void 0;
/* eslint-disable no-console */
const crypto = require("crypto");
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
const backoff_generator_1 = require("../lib/backoff-generator");
const custom_resource_1 = require("../lib/custom-resource");
const x509_certs_1 = require("../lib/x509-certs");
const types_1 = require("./types");
const ACM_VERSION = '2015-12-08';
const DYNAMODB_VERSION = '2012-08-10';
const SECRETS_MANAGER_VERSION = '2017-10-17';
class AcmCertificateImporter extends custom_resource_1.DynamoBackedCustomResource {
    constructor(acmClient, dynamoDbClient, secretsManagerClient) {
        super(dynamoDbClient);
        this.acmClient = acmClient;
        this.secretsManagerClient = secretsManagerClient;
    }
    /* istanbul ignore next */
    validateInput(data) {
        return types_1.implementsIAcmImportCertProps(data);
    }
    async doCreate(physicalId, resourceProperties) {
        const resourceTable = await this.getResourceTable();
        await Promise.all([
            this.databasePermissionsCheck(resourceTable),
        ]);
        const cert = await this.getSecretString(resourceProperties.X509CertificatePem.Cert);
        const certChainArn = resourceProperties.X509CertificatePem.CertChain;
        const certChain = certChainArn.length > 0 ? await this.getSecretString(certChainArn) : undefined;
        const key = await this.getSecretString(resourceProperties.X509CertificatePem.Key);
        const passphrase = await this.getSecretString(resourceProperties.X509CertificatePem.Passphrase);
        const decryptedKey = await x509_certs_1.Certificate.decryptKey(key, passphrase);
        const tags = resourceProperties.Tags;
        const certificateArn = await this.importAndStoreCertificate({
            resourceTable,
            key: decryptedKey,
            cert,
            certChain,
            physicalId,
            tags,
        });
        return { CertificateArn: certificateArn };
    }
    async doDelete(physicalId) {
        const resourceTable = await this.getResourceTable();
        await Promise.all([
            this.databasePermissionsCheck(resourceTable),
        ]);
        const resources = await resourceTable.query(physicalId);
        const maxAttempts = 10;
        for (const [key, resource] of Object.entries(resources)) {
            const arn = resource.ARN;
            let inUseByResources = [];
            const backoffGenerator = new backoff_generator_1.BackoffGenerator({
                base: 200,
                jitterDivisor: 4,
                maxAttempts,
            });
            do {
                const { Certificate: cert } = await this.acmClient.describeCertificate({
                    CertificateArn: arn,
                }).promise();
                inUseByResources = cert.InUseBy || [];
                if (inUseByResources.length) {
                    await backoffGenerator.backoff();
                }
                else {
                    break;
                }
            } while (backoffGenerator.shouldContinue());
            if (inUseByResources.length) {
                throw new Error(`Response from describeCertificate did not contain an empty InUseBy list after ${maxAttempts} attempts.`);
            }
            console.log(`Deleting resource for '${key}'`);
            try {
                await this.acmClient.deleteCertificate({ CertificateArn: arn }).promise();
            }
            catch (e) {
                // AccessDeniedException can happen if either:
                //  a) We do not have the required permission to delete the Certificate (unlikely)
                //  b) The Certificate has already been deleted (more likely)
                if (e.message.indexOf('AccessDeniedException')) {
                    console.warn(`Could not delete Certificate ${arn}. Please ensure it has been deleted.`);
                }
                throw e; // Rethrow so the custom resource handler will error-out.
            }
            await resourceTable.deleteItem({
                primaryKeyValue: physicalId,
                sortKeyValue: key,
            });
        }
    }
    async importAndStoreCertificate(args) {
        let certificateArn;
        const sortKey = crypto.createHash('md5').update(args.cert).digest('hex');
        const existingItem = await args.resourceTable.getItem({
            primaryKeyValue: args.physicalId,
            sortKeyValue: sortKey,
        });
        if (existingItem) {
            if (!existingItem.ARN) {
                throw Error("Database Item missing 'ARN' attribute");
            }
            // Verify that the cert is in ACM
            certificateArn = existingItem.ARN;
            try {
                await this.acmClient.getCertificate({ CertificateArn: certificateArn }).promise();
            }
            catch (e) {
                throw Error(`Database entry ${existingItem.ARN} could not be found in ACM: ${JSON.stringify(e)}`);
            }
            // Update the cert by performing an import again, with the new values.
            const importCertRequest = {
                CertificateArn: certificateArn,
                Certificate: args.cert,
                CertificateChain: args.certChain,
                PrivateKey: args.key,
                Tags: args.tags,
            };
            await this.importCertificate(importCertRequest);
        }
        else {
            const importCertRequest = {
                Certificate: args.cert,
                CertificateChain: args.certChain,
                PrivateKey: args.key,
                Tags: args.tags,
            };
            const resp = await this.importCertificate(importCertRequest);
            if (!resp.CertificateArn) {
                throw new Error(`CertificateArn was not properly populated after attempt to import ${args.cert}`);
            }
            certificateArn = resp.CertificateArn;
            await args.resourceTable.putItem({
                primaryKeyValue: args.physicalId,
                sortKeyValue: sortKey,
                attributes: {
                    ARN: certificateArn,
                },
                allow_overwrite: false,
            });
        }
        return certificateArn;
    }
    async importCertificate(importCertRequest) {
        var _a;
        // ACM cert imports are limited to 1 per second (see https://docs.aws.amazon.com/acm/latest/userguide/acm-limits.html#api-rate-limits)
        // We need to backoff & retry in the event that two imports happen in the same second
        const maxAttempts = 10;
        const backoffGenerator = new backoff_generator_1.BackoffGenerator({
            base: 200,
            jitterDivisor: 4,
            maxAttempts,
        });
        let retry = false;
        do {
            try {
                return await this.acmClient.importCertificate(importCertRequest).promise();
            }
            catch (e) {
                console.warn(`Could not import certificate: ${e}`);
                retry = await backoffGenerator.backoff();
                if (retry) {
                    console.log('Retrying...');
                }
            }
        } while (retry);
        throw new Error(`Failed to import certificate ${(_a = importCertRequest.CertificateArn) !== null && _a !== void 0 ? _a : ''} after ${maxAttempts} attempts.`);
    }
    async getSecretString(SecretId) {
        console.debug(`Retrieving secret: ${SecretId}`);
        const resp = await this.secretsManagerClient.getSecretValue({ SecretId }).promise();
        if (!resp.SecretString) {
            throw new Error(`Secret ${SecretId} did not contain a SecretString as expected`);
        }
        return resp.SecretString;
    }
}
exports.AcmCertificateImporter = AcmCertificateImporter;
/**
 * The handler used to import an X.509 certificate to ACM from a Secret
 */
/* istanbul ignore next */
async function importCert(event, context) {
    const handler = new AcmCertificateImporter(new aws_sdk_1.ACM({ apiVersion: ACM_VERSION }), new aws_sdk_1.DynamoDB({ apiVersion: DYNAMODB_VERSION }), new aws_sdk_1.SecretsManager({ apiVersion: SECRETS_MANAGER_VERSION }));
    return await handler.handler(event, context);
}
exports.importCert = importCert;
//# sourceMappingURL=data:application/json;base64,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