"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = exports.VersionProviderResource = void 0;
const custom_resource_1 = require("../lib/custom-resource");
const version_provider_1 = require("../lib/version-provider");
/**
 * This custom resource will parse and return the S3 URI's of the Deadline installers and Docker recipes for use in
 * any constructs that need to install Deadline.
 */
class VersionProviderResource extends custom_resource_1.SimpleCustomResource {
    constructor() {
        super();
        this.versionProvider = new version_provider_1.VersionProvider();
    }
    /**
     * @inheritdoc
     */
    validateInput(data) {
        return this.implementsIVersionProviderResourceProperties(data);
    }
    /**
     * @inheritdoc
     */
    // @ts-ignore  -- we do not use the physicalId
    async doCreate(physicalId, resourceProperties) {
        const deadlinePlatFormVersionedUris = await this.versionProvider.getVersionUris({
            versionString: resourceProperties.versionString,
            platform: version_provider_1.Platform.linux,
            product: version_provider_1.Product.deadline,
        });
        const deadlineLinux = deadlinePlatFormVersionedUris.get(version_provider_1.Platform.linux);
        const deadlineLinuxUris = deadlineLinux.Uris;
        const s3Bucket = this.parseS3BucketName(deadlineLinuxUris.bundle);
        const linuxRepoObjectKey = this.parseS3ObjectKey(deadlineLinuxUris.repositoryInstaller);
        return {
            S3Bucket: s3Bucket,
            MajorVersion: deadlineLinux.MajorVersion,
            MinorVersion: deadlineLinux.MinorVersion,
            ReleaseVersion: deadlineLinux.ReleaseVersion,
            LinuxPatchVersion: deadlineLinux.PatchVersion,
            LinuxRepositoryInstaller: linuxRepoObjectKey,
        };
    }
    /**
     * @inheritdoc
     */
    /* istanbul ignore next */ // @ts-ignore
    async doDelete(physicalId, resourceProperties) {
        // Nothing to do -- we don't modify anything.
        return;
    }
    implementsIVersionProviderResourceProperties(value) {
        if (!value || typeof (value) !== 'object') {
            return false;
        }
        if (value.versionString) {
            if (!version_provider_1.Version.validateVersionString(value.versionString)) {
                console.log(`Failed to validate the version string: ${value.versionString}`);
                return false;
            }
        }
        if (value.forceRun && typeof (value.forceRun) !== 'string') {
            return false;
        }
        return true;
    }
    /**
     * Parses the S3 bucket name from an S3 URI.
     */
    parseS3BucketName(uri) {
        let bucketName;
        try {
            bucketName = this.findRegex(uri, /^s3:\/\/([A-Za-z0-9\-]+)\//)[1];
        }
        catch (e) {
            throw new Error(`Could not parse S3 bucket name from ${uri}`);
        }
        return bucketName;
    }
    /**
     * Parses the S3 object key from an S3 URI.
     */
    parseS3ObjectKey(uri) {
        let objectKey;
        try {
            objectKey = this.findRegex(uri, /^s3:\/\/[A-Za-z0-9\-]+\/([A-Za-z0-9\-\/\.]+)$/)[1];
        }
        catch (e) {
            throw new Error(`Could not parse S3 object key from ${uri}`);
        }
        return objectKey;
    }
    // Assumes a single capture is in the regex
    findRegex(str, re) {
        const found = str.match(re);
        if (found === null) {
            throw new Error(`Couldn't find regular expression ${re} in ${str}`);
        }
        return found;
    }
}
exports.VersionProviderResource = VersionProviderResource;
/**
 * The handler used to provide the installer links for the requested version
 */
async function handler(event, context) {
    const versionProvider = new VersionProviderResource();
    return await versionProvider.handler(event, context);
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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