"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable dot-notation */
const events_1 = require("events");
const deadline_client_1 = require("../deadline-client");
jest.mock('http');
jest.mock('https');
describe('DeadlineClient', () => {
    let deadlineClient;
    class MockResponse extends events_1.EventEmitter {
        constructor(statusCode) {
            super();
            this.statusMessage = 'status message';
            this.statusCode = statusCode;
        }
    }
    class MockRequest extends events_1.EventEmitter {
        end() { }
        write(_data) { }
    }
    let consoleLogMock;
    let request;
    let response;
    /**
     * Mock implementation of the request
     *
     * @param _url The URL of the request
     * @param callback The callback to call when a response is available
     */
    function httpRequestMock(_url, callback) {
        if (callback) {
            callback(response);
        }
        return request;
    }
    describe('successful responses', () => {
        beforeEach(() => {
            consoleLogMock = jest.spyOn(console, 'log').mockReturnValue(undefined);
            request = new MockRequest();
            jest.requireMock('http').request.mockReset();
            jest.requireMock('https').request.mockReset();
            jest.requireMock('https').Agent.mockClear();
            response = new MockResponse(200);
        });
        test('successful http get request', async () => {
            // GIVEN
            jest.requireMock('http').request.mockImplementation(httpRequestMock);
            deadlineClient = new deadline_client_1.DeadlineClient({
                host: 'hostname',
                port: 8080,
                protocol: 'HTTP',
            });
            const responseData = {
                test: true,
            };
            // WHEN
            const promise = deadlineClient.GetRequest('/get/version/test');
            response.emit('data', Buffer.from(JSON.stringify(responseData), 'utf8'));
            response.emit('end');
            const result = await promise;
            // THEN
            // should make an HTTP request
            expect(jest.requireMock('http').request)
                .toBeCalledWith({
                agent: undefined,
                method: 'GET',
                port: 8080,
                host: 'hostname',
                path: '/get/version/test',
            }, expect.any(Function));
            expect(result.data).toEqual(responseData);
        });
        test('successful http get request with options', async () => {
            // GIVEN
            jest.requireMock('http').request.mockImplementation(httpRequestMock);
            deadlineClient = new deadline_client_1.DeadlineClient({
                host: 'hostname',
                port: 8080,
                protocol: 'HTTP',
            });
            const responseData = {
                test: true,
            };
            // WHEN
            const promise = deadlineClient.GetRequest('/get/version/test', {
                headers: {
                    'Content-Type': 'application/json',
                },
            });
            response.emit('data', Buffer.from(JSON.stringify(responseData), 'utf8'));
            response.emit('end');
            const result = await promise;
            // THEN
            // should make an HTTP request
            expect(jest.requireMock('http').request)
                .toBeCalledWith({
                agent: undefined,
                headers: {
                    'Content-Type': 'application/json',
                },
                method: 'GET',
                port: 8080,
                host: 'hostname',
                path: '/get/version/test',
            }, expect.any(Function));
            expect(result.data).toEqual(responseData);
        });
        test('successful https get request', async () => {
            // GIVEN
            jest.requireMock('https').request.mockImplementation(httpRequestMock);
            deadlineClient = new deadline_client_1.DeadlineClient({
                host: 'hostname',
                port: 4433,
                protocol: 'HTTPS',
            });
            const responseData = {
                test: true,
            };
            // WHEN
            const promise = deadlineClient.GetRequest('/get/version/test');
            response.emit('data', Buffer.from(JSON.stringify(responseData), 'utf8'));
            response.emit('end');
            const result = await promise;
            // THEN
            const agentMock = jest.requireMock('https').Agent;
            expect(agentMock).toHaveBeenCalledTimes(1);
            expect(agentMock).toBeCalledWith(expect.not.objectContaining({ ca: expect.any(String) }));
            expect(agentMock).toBeCalledWith(expect.not.objectContaining({ pfx: expect.any(String) }));
            expect(agentMock).toBeCalledWith(expect.not.objectContaining({ passphrase: expect.any(String) }));
            // should make an HTTPS request
            expect(jest.requireMock('https').request)
                .toBeCalledWith({
                agent: agentMock.mock.instances[0],
                method: 'GET',
                port: 4433,
                host: 'hostname',
                path: '/get/version/test',
            }, expect.any(Function));
            expect(result.data).toEqual(responseData);
        });
        test('successful https get request with tls', async () => {
            // GIVEN
            jest.requireMock('https').request.mockImplementation(httpRequestMock);
            deadlineClient = new deadline_client_1.DeadlineClient({
                host: 'hostname',
                port: 4433,
                protocol: 'HTTPS',
                tls: {
                    ca: 'cacontent',
                    pfx: 'pfxcontent',
                    passphrase: 'passphrasecontent',
                },
            });
            const responseData = {
                test: true,
            };
            // WHEN
            const promise = deadlineClient.GetRequest('/get/version/test');
            response.emit('data', Buffer.from(JSON.stringify(responseData), 'utf8'));
            response.emit('end');
            const result = await promise;
            // THEN
            const agentMock = jest.requireMock('https').Agent;
            expect(agentMock).toHaveBeenCalledTimes(1);
            expect(agentMock).toBeCalledWith(expect.objectContaining({
                ca: 'cacontent',
                pfx: 'pfxcontent',
                passphrase: 'passphrasecontent',
            }));
            // should make an HTTPS request
            expect(jest.requireMock('https').request)
                .toBeCalledWith({
                agent: agentMock.mock.instances[0],
                method: 'GET',
                port: 4433,
                host: 'hostname',
                path: '/get/version/test',
            }, expect.any(Function));
            expect(result.data).toEqual(responseData);
        });
        test('successful http post request', async () => {
            // GIVEN
            jest.requireMock('http').request.mockImplementation(httpRequestMock);
            deadlineClient = new deadline_client_1.DeadlineClient({
                host: 'hostname',
                port: 8080,
                protocol: 'HTTP',
            });
            const responseData = {
                test: true,
            };
            // WHEN
            const promise = deadlineClient.PostRequest('/save/version/test', 'anydata');
            response.emit('data', Buffer.from(JSON.stringify(responseData), 'utf8'));
            response.emit('end');
            const result = await promise;
            // THEN
            // should make an HTTP request
            expect(jest.requireMock('http').request)
                .toBeCalledWith({
                agent: undefined,
                method: 'POST',
                port: 8080,
                host: 'hostname',
                path: '/save/version/test',
            }, expect.any(Function));
            expect(result.data).toEqual(responseData);
        });
        test('successful https post request', async () => {
            // GIVEN
            jest.requireMock('https').request.mockImplementation(httpRequestMock);
            deadlineClient = new deadline_client_1.DeadlineClient({
                host: 'hostname',
                port: 4433,
                protocol: 'HTTPS',
            });
            const responseData = {
                test: true,
            };
            // WHEN
            const promise = deadlineClient.PostRequest('/save/version/test', 'anydata');
            response.emit('data', Buffer.from(JSON.stringify(responseData), 'utf8'));
            response.emit('end');
            const result = await promise;
            // THEN
            const agentMock = jest.requireMock('https').Agent;
            expect(agentMock).toHaveBeenCalledTimes(1);
            expect(agentMock).toBeCalledWith(expect.not.objectContaining({ ca: expect.any(String) }));
            expect(agentMock).toBeCalledWith(expect.not.objectContaining({ pfx: expect.any(String) }));
            expect(agentMock).toBeCalledWith(expect.not.objectContaining({ passphrase: expect.any(String) }));
            // should make an HTTP request
            expect(jest.requireMock('https').request)
                .toBeCalledWith({
                agent: agentMock.mock.instances[0],
                method: 'POST',
                port: 4433,
                host: 'hostname',
                path: '/save/version/test',
            }, expect.any(Function));
            expect(result.data).toEqual(responseData);
        });
        test('successful https post request with tls', async () => {
            // GIVEN
            jest.requireMock('https').request.mockImplementation(httpRequestMock);
            deadlineClient = new deadline_client_1.DeadlineClient({
                host: 'hostname',
                port: 4433,
                protocol: 'HTTPS',
                tls: {
                    ca: 'cacontent',
                    pfx: 'pfxcontent',
                    passphrase: 'passphrasecontent',
                },
            });
            const responseData = {
                test: true,
            };
            // WHEN
            const promise = deadlineClient.PostRequest('/save/version/test', 'anydata');
            response.emit('data', Buffer.from(JSON.stringify(responseData), 'utf8'));
            response.emit('end');
            const result = await promise;
            // THEN
            const agentMock = jest.requireMock('https').Agent;
            expect(agentMock).toHaveBeenCalledTimes(1);
            expect(agentMock).toBeCalledWith(expect.objectContaining({
                ca: 'cacontent',
                pfx: 'pfxcontent',
                passphrase: 'passphrasecontent',
            }));
            // should make an HTTPS request
            expect(jest.requireMock('https').request)
                .toBeCalledWith({
                agent: agentMock.mock.instances[0],
                method: 'POST',
                port: 4433,
                host: 'hostname',
                path: '/save/version/test',
            }, expect.any(Function));
            expect(result.data).toEqual(responseData);
        });
    });
    describe('failed responses', () => {
        beforeEach(() => {
            consoleLogMock = jest.spyOn(console, 'log').mockReturnValue(undefined);
            request = new MockRequest();
            jest.requireMock('http').request.mockImplementation(httpRequestMock);
            jest.requireMock('https').request.mockImplementation(httpRequestMock);
        });
        afterEach(() => {
            jest.clearAllMocks();
            jest.requireMock('http').request.mockReset();
            jest.requireMock('https').request.mockReset();
        });
        describe('returns a rejected promise on 400 responses', () => {
            test.each([
                ['HTTP', 'GET'],
                ['HTTP', 'POST'],
                ['HTTPS', 'GET'],
                ['HTTPS', 'POST'],
            ])('for %p %p', async (protocol, requestType) => {
                // GIVEN
                response = new MockResponse(400);
                deadlineClient = new deadline_client_1.DeadlineClient({
                    host: 'hostname',
                    port: 0,
                    protocol: protocol,
                });
                // WHEN
                function performRequest() {
                    if (requestType === 'GET') {
                        return deadlineClient.GetRequest('anypath');
                    }
                    return deadlineClient.PostRequest('anypath', 'anydata');
                }
                const promise = performRequest();
                // THEN
                await expect(promise)
                    .rejects
                    .toEqual(response.statusMessage);
                expect(consoleLogMock.mock.calls.length).toBe(0);
            });
        });
        describe('retries on 503 responses', () => {
            test.each([
                ['HTTP', 'GET'],
                ['HTTP', 'POST'],
                ['HTTPS', 'GET'],
                ['HTTPS', 'POST'],
            ])('for %p %p', async (protocol, requestType) => {
                // GIVEN
                response = new MockResponse(503);
                const retries = 3;
                deadlineClient = new deadline_client_1.DeadlineClient({
                    host: 'hostname',
                    port: 0,
                    protocol: protocol,
                    retries,
                    retryWaitMs: 0,
                });
                // WHEN
                function performRequest() {
                    if (requestType === 'GET') {
                        return deadlineClient.GetRequest('anypath');
                    }
                    return deadlineClient.PostRequest('anypath', 'anydata');
                }
                const promise = performRequest();
                // THEN
                await expect(promise)
                    .rejects
                    .toEqual(response.statusMessage);
                expect(consoleLogMock.mock.calls.length).toBe(retries * 2);
                expect(consoleLogMock.mock.calls[0][0]).toMatch(/Request failed with/);
                expect(consoleLogMock.mock.calls[1][0]).toMatch(/Retries left:/);
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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