/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
/**
 * Properties for BackoffGenerator.
 */
export interface BackoffGeneratorProps {
    /**
     * The base duration, in milliseconds, used to calculate exponential backoff.
     *
     * For example, when not using jitter, the backoff time per attempt will be calculated as:
     * 1. `base` * 2^0
     * 2. `base` * 2^1
     * 3. `base` * 2^2, etc.
     *
     * @default 200 milliseconds
     */
    readonly base?: number;
    /**
     * The maximum amount of time, in milliseconds, allowed between backoffs.
     * Each backoff will have its length clamped to a maximum of this value.
     * @default Number.MAX_SAFE_INTEGER
     */
    readonly maxIntervalMs?: number;
    /**
     * The divisor used to calculate the portion backoff time that will be subject to jitter.
     * Higher values indicate lower jitters (backoff times will differ by a smaller amount).
     *
     * For example, given a calculated `backoff` value, applying jitter would look like:
     * ```
     * backoffJitter = (backoff - backoff / jitterDivisor) + jitter(backoff / jitterDivisor)
     * ```
     * @default No jitter
     */
    readonly jitterDivisor?: number;
    /**
     * The maximum cumulative time, in milliseconds, to backoff before quitting.
     * @default No limit on how long this object can backoff for
     */
    readonly maxCumulativeBackoffTimeMs?: number;
    /**
     * The maximum number of times to backoff before quitting.
     * @default No limit on how many times this object can backoff
     */
    readonly maxAttempts?: number;
}
/**
 * Class to handle sleeping with exponential backoff.
 *
 * Reference: https://aws.amazon.com/blogs/architecture/exponential-backoff-and-jitter/
 */
export declare class BackoffGenerator {
    /**
     * Calculates the number of milliseconds to sleep based on the attempt count.
     * @param b The base value for the calculation.
     * @param attempt The attempt count.
     * @param maxIntervalMs The maximum interval between backoffs, in milliseconds.
     * @returns The number of milliseconds to sleep.
     */
    private static calculateSleepMs;
    private readonly base;
    private readonly maxIntervalMs;
    private readonly jitterDivisor;
    private readonly maxCumulativeBackoffTimeMs;
    private readonly maxAttempts;
    private cumulativeBackoffTimeMs;
    private attempt;
    constructor(props?: BackoffGeneratorProps);
    /**
     * Restarts the internal counters used by this class.
     */
    restart(): void;
    /**
     * Sleeps for an exponentially increasing time depending on how many times this class has backed off.
     * If `jitterDivisor` was provided, jitter will be applied to the backoff time.
     *
     * If any of the conditions to stop backing off are met, this method will not sleep and return false.
     * Otherwise, it sleeps and returns true.
     * @param force Force sleeping, regardless of the conditions that indicate when to stop backing off.
     */
    backoff(force?: boolean): Promise<boolean>;
    /**
     * Returns true if either the maximum number of attempts or maximum cumulative backoff time has been reached.
     * If neither are specified, this will always return true.
     */
    shouldContinue(): boolean;
}
