"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable dot-notation */
const assert_1 = require("@aws-cdk/assert");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../core/lib");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const token_regex_helpers_1 = require("../../core/test/token-regex-helpers");
const lib_2 = require("../lib");
const asset_constants_1 = require("./asset-constants");
let app;
let stack;
let wfstack;
let vpc;
let renderQueue;
let rcsImage;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'infraStack', {
        env: {
            region: 'us-east-1',
        },
    });
    vpc = new aws_ec2_1.Vpc(stack, 'VPC');
    rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
    const version = new lib_2.VersionQuery(stack, 'VersionQuery');
    renderQueue = new lib_2.RenderQueue(stack, 'RQ', {
        vpc,
        images: { remoteConnectionServer: rcsImage },
        repository: new lib_2.Repository(stack, 'Repository', {
            vpc,
            version,
        }),
        version,
    });
    wfstack = new core_1.Stack(app, 'workerFleetStack', {
        env: {
            region: 'us-east-1',
        },
    });
});
test('default worker fleet is created correctly', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
    });
    // THEN
    assert_1.expect(wfstack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
    assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't2.large',
        spotPrice: assert_1.ABSENT,
    }));
    assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        ToPort: parseInt(renderQueue.endpoint.portAsString(), 10),
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                stack.getLogicalId(fleet.fleet.connections.securityGroups[0].node.defaultChild),
                'GroupId',
            ],
        },
        GroupId: {
            'Fn::ImportValue': 'infraStack:ExportsOutputFnGetAttRQLBSecurityGroupAC643AEDGroupId8F9F7830',
        },
    }));
    assert_1.expect(wfstack).to(assert_1.haveResource('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: '/renderfarm/workerFleet',
    }));
    expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
    expect(fleet.node.metadataEntry[0].data).toMatch('being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
    expect(fleet.node.metadataEntry[1].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
    expect(fleet.node.metadataEntry[1].data).toContain('being created without a health monitor attached to it. This means that the fleet will not automatically scale-in to 0 if the workers are unhealthy');
});
test('security group is added to fleet after its creation', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
    });
    fleet.addSecurityGroup(aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
        allowAllOutbound: false,
    }));
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        SecurityGroups: [
            {
                'Fn::GetAtt': [
                    stack.getLogicalId(fleet.fleet.connections.securityGroups[0].node.defaultChild),
                    'GroupId',
                ],
            },
            'sg-123456789',
        ],
    }));
});
test('WorkerFleet uses given security group', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
        securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
            allowAllOutbound: false,
        }),
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        SecurityGroups: [
            'sg-123456789',
        ],
    }));
});
describe('allowing log listener port', () => {
    test('from CIDR', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        fleet.allowListenerPortFrom(aws_ec2_1.Peer.ipv4('127.0.0.1/24').connections);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [{ CidrIp: '0.0.0.0/0' }],
            SecurityGroupIngress: [
                {
                    CidrIp: '127.0.0.1/24',
                    Description: 'Worker remote command listening port',
                    FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
                    IpProtocol: 'tcp',
                    ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
                },
            ],
        }));
    });
    test('to CIDR', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        fleet.allowListenerPortTo(aws_ec2_1.Peer.ipv4('127.0.0.1/24').connections);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [{ CidrIp: '0.0.0.0/0' }],
            SecurityGroupIngress: [
                {
                    CidrIp: '127.0.0.1/24',
                    Description: 'Worker remote command listening port',
                    FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
                    IpProtocol: 'tcp',
                    ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
                },
            ],
        }));
    });
    test('from SecurityGroup', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789');
        fleet.allowListenerPortFrom(securityGroup);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
            IpProtocol: 'tcp',
            SourceSecurityGroupId: 'sg-123456789',
            ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
        }));
    });
    test('to SecurityGroup', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789');
        fleet.allowListenerPortTo(securityGroup);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
            IpProtocol: 'tcp',
            SourceSecurityGroupId: 'sg-123456789',
            ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
        }));
    });
    test('from other stack', () => {
        const otherStack = new core_1.Stack(app, 'otherStack', {
            env: { region: 'us-east-1' },
        });
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(otherStack, 'SG', 'sg-123456789');
        fleet.allowListenerPortFrom(securityGroup);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
            IpProtocol: 'tcp',
            SourceSecurityGroupId: 'sg-123456789',
            ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
        }));
    });
    test('to other stack', () => {
        const otherStack = new core_1.Stack(app, 'otherStack', {
            env: { region: 'us-east-1' },
        });
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(otherStack, 'SG', 'sg-123456789');
        fleet.allowListenerPortTo(securityGroup);
        // THEN
        assert_1.expect(otherStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
            IpProtocol: 'tcp',
            SourceSecurityGroupId: 'sg-123456789',
            ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
        }));
    });
});
test('default worker fleet is created correctly with linux image', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration'));
});
test('default worker fleet is created correctly with spot config', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        spotPrice: 2.5,
    });
    // THEN
    assert_1.expect(wfstack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
    assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        SpotPrice: '2.5',
    }));
});
test('default worker fleet is not created with incorrect spot config', () => {
    // WHEN
    expect(() => {
        new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            spotPrice: lib_2.WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT + 1,
        });
    }).toThrowError(/Invalid value: 256 for property 'spotPrice'. Valid values can be any decimal between 0.001 and 255./);
    // WHEN
    expect(() => {
        new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet2', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            spotPrice: lib_2.WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT / 2,
        });
    }).toThrowError(/Invalid value: 0.0005 for property 'spotPrice'. Valid values can be any decimal between 0.001 and 255./);
});
test('default worker fleet is created correctly custom Instance type', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't2.medium',
    }));
});
test.each([
    'test-prefix/',
    '',
])('default worker fleet is created correctly with custom LogGroup prefix %s', (testPrefix) => {
    // GIVEN
    const id = 'workerFleet';
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, id, {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        logGroupProps: {
            logGroupPrefix: testPrefix,
        },
    });
    assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: testPrefix + id,
    }));
});
test('default worker fleet is created correctly custom subnet values', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // WHEN
    const workers = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
        vpcSubnets: {
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        },
        healthCheckConfig: {
            port: 6161,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        VPCZoneIdentifier: [{
                Ref: 'VPC1AzPublicSubnet1Subnet9649CC17',
            }],
    }));
    const userData = stack.resolve(workers.fleet.userData.render());
    expect(userData).toStrictEqual({
        'Fn::Join': [
            '',
            [
                '#!/bin/bash\nfunction exitTrap(){\nexitCode=$?\n/opt/aws/bin/cfn-signal --stack infraStack --resource workerFleetASG25520D69 --region us-east-1 -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n}\ntrap exitTrap EXIT\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "\')\naws s3 cp 's3://",
                { Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\'\n' +
                    'set -e\n' +
                    'chmod +x \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\'\n\'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\' -i us-east-1 ',
                { Ref: 'workerFleetStringParameterDB3717DA' },
                '\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\')\naws s3 cp \'s3://',
                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\'\n' +
                    'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                    '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                    'fi\n' +
                    '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '" --render-queue "http://',
                {
                    'Fn::GetAtt': [
                        'RQLB3B7B1CBC',
                        'DNSName',
                    ],
                },
                ':8080" \n' +
                    'rm -f "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\"\n' +
                    "mkdir -p $(dirname '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                '\')\naws s3 cp \'s3://',
                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                '\' \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                "'\nmkdir -p $(dirname '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "')\naws s3 cp 's3://",
                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\nset -e\nchmod +x '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\n'/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                `' '' '' '' '${lib_2.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION}' ${lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} /tmp/`,
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
            ],
        ],
    });
});
test('default worker fleet is created correctly with groups, pools and region', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // WHEN
    const workers = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
        vpcSubnets: {
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        },
        groups: ['A', 'B'],
        pools: ['C', 'D'],
        region: 'E',
    });
    // THEN
    const userData = stack.resolve(workers.fleet.userData.render());
    expect(userData).toStrictEqual({
        'Fn::Join': ['', [
                '#!/bin/bash\nfunction exitTrap(){\nexitCode=$?\n/opt/aws/bin/cfn-signal --stack infraStack --resource workerFleetASG25520D69 --region us-east-1 -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n}\ntrap exitTrap EXIT\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\')\naws s3 cp \'s3://',
                { Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\nset -e\nchmod +x '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\n'/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' -i us-east-1 ",
                { Ref: 'workerFleetStringParameterDB3717DA' },
                '\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\')\naws s3 cp \'s3://',
                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\'\n' +
                    'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                    '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                    'fi\n' +
                    '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '" --render-queue "http://',
                {
                    'Fn::GetAtt': [
                        'RQLB3B7B1CBC',
                        'DNSName',
                    ],
                },
                ':8080" \n' +
                    'rm -f "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\"\n' +
                    "mkdir -p $(dirname '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                {
                                    Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key,
                                },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                {
                                    Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key,
                                },
                            ],
                        },
                    ],
                },
                "')\naws s3 cp 's3://",
                {
                    Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Bucket,
                },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                            ],
                        },
                    ],
                },
                "'\nmkdir -p $(dirname '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "')\naws s3 cp 's3://",
                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\nset -e\nchmod +x '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\n'/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                `' 'a,b' 'c,d' 'E' '${lib_2.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION}' ${lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} /tmp/`,
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                {
                                    Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key,
                                },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                {
                                    Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key,
                                },
                            ],
                        },
                    ],
                },
            ]],
    });
});
test('worker fleet does validation correctly with groups, pools and region', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // group name as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            groups: ['A', 'none'],
        });
    }).toThrowError();
    // group name with whitespace
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet1', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            groups: ['A', 'no ne'],
        });
    }).toThrowError(/Invalid value: no ne for property 'groups'/);
    // pool name with whitespace
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet2', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            pools: ['A', 'none'],
        });
    }).toThrowError(/Invalid value: none for property 'pools'/);
    // pool name as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet3', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            pools: ['A', 'none'],
        });
    }).toThrowError(/Invalid value: none for property 'pools'/);
    // region as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet4', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none',
        });
    }).toThrowError(/Invalid value: none for property 'region'/);
    // region as 'all'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet5', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'all',
        });
    }).toThrowError(/Invalid value: all for property 'region'/);
    // region as 'unrecognized'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet6', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'unrecognized',
        });
    }).toThrowError(/Invalid value: unrecognized for property 'region'/);
    // region with invalid characters
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet7', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none@123',
        });
    }).toThrowError(/Invalid value: none@123 for property 'region'/);
    // region with reserved name as substring
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet8', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none123',
        });
    }).not.toThrowError();
    // region with case-insensitive name
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet9', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'None',
        });
    }).toThrowError(/Invalid value: None for property 'region'/);
});
describe('Block Device Tests', () => {
    let healthMonitor;
    beforeEach(() => {
        // create a health monitor so it does not trigger warnings
        healthMonitor = new lib_1.HealthMonitor(wfstack, 'healthMonitor', {
            vpc,
        });
    });
    test('Warning if no BlockDevices provided', () => {
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
        });
        expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
        expect(fleet.node.metadataEntry[0].data).toMatch('being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
    });
    test('No Warnings if Encrypted BlockDevices Provided', () => {
        const VOLUME_SIZE = 50;
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: true }),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        Encrypted: true,
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        }));
        expect(fleet.node.metadataEntry).toHaveLength(0);
    });
    test('Warnings if non-Encrypted BlockDevices Provided', () => {
        const VOLUME_SIZE = 50;
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: false }),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        Encrypted: false,
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        }));
        expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
        expect(fleet.node.metadataEntry[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the worker-fleet workerFleet is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
    });
    test('Warnings for BlockDevices without encryption specified', () => {
        const VOLUME_SIZE = 50;
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        }));
        expect(fleet.node.metadataEntry[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
        expect(fleet.node.metadataEntry[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the worker-fleet workerFleet is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
    });
    test('No warnings for Ephemeral blockDeviceVolumes', () => {
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ephemeral(0),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    DeviceName: DEVICE_NAME,
                    VirtualName: 'ephemeral0',
                },
            ],
        }));
        expect(fleet.node.metadataEntry).toHaveLength(0);
    });
    test('No warnings for Suppressed blockDeviceVolumes', () => {
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.noDevice(),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    DeviceName: DEVICE_NAME,
                },
            ],
        }));
        expect(fleet.node.metadataEntry).toHaveLength(0);
    });
});
describe('HealthMonitor Tests', () => {
    let healthMonitor;
    beforeEach(() => {
        // create a health monitor so it does not trigger warnings
        healthMonitor = new lib_1.HealthMonitor(wfstack, 'healthMonitor', {
            vpc,
        });
    });
    test('Monitor is configured for Windows', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
        });
        const userData = fleet.fleet.userData.render();
        // THEN
        // Ensure the configuration script is executed with the expected arguments.
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('&\'C:/temp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \'63415\' \'10.1.9.2\'')));
        // Ensure that the health monitor target group has been set up.
        //  Note: It's sufficient to just check for any resource created by the HealthMonitor registration.
        //   The HealthMonitor tests cover ensuring that all of the resources are set up.
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '63415',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
    });
    test('Monitor is configured for Linux', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
        });
        const userData = fleet.fleet.userData.render();
        // THEN
        // Ensure the configuration script is executed with the expected arguments.
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('\'/tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \'63415\' \'10.1.9.2\'')));
        // Ensure that the health monitor target group has been set up.
        //  Note: It's sufficient to just check for any resource created by the HealthMonitor registration.
        //   The HealthMonitor tests cover ensuring that all of the resources are set up.
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '63415',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
    });
    test('UserData is added', () => {
        // WHEN
        class UserDataProvider extends lib_2.InstanceUserDataProvider {
            preCloudWatchAgent(host) {
                host.userData.addCommands('echo preCloudWatchAgent');
            }
            preRenderQueueConfiguration(host) {
                host.userData.addCommands('echo preRenderQueueConfiguration');
            }
            preWorkerConfiguration(host) {
                host.userData.addCommands('echo preWorkerConfiguration');
            }
            postWorkerLaunch(host) {
                host.userData.addCommands('echo postWorkerLaunch');
            }
        }
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            userDataProvider: new UserDataProvider(wfstack, 'UserDataProvider'),
        });
        const userData = fleet.fleet.userData.render();
        // THEN
        expect(userData).toContain('echo preCloudWatchAgent');
        expect(userData).toContain('echo preRenderQueueConfiguration');
        expect(userData).toContain('echo preWorkerConfiguration');
        expect(userData).toContain('echo postWorkerLaunch');
    });
});
describe('tagging', () => {
    tag_helpers_1.testConstructTags({
        constructName: 'WorkerInstanceFleet',
        createConstruct: () => {
            // GIVEN
            const healthMonitorStack = new core_1.Stack(app, 'HealthMonitorStack', {
                env: {
                    region: 'us-east-1',
                },
            });
            const healthMonitor = new lib_1.HealthMonitor(healthMonitorStack, 'healthMonitor', {
                vpc,
            });
            const deviceName = '/dev/xvda';
            // WHEN
            new lib_2.WorkerInstanceFleet(wfstack, 'WorkerFleet', {
                vpc,
                workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                    'us-east-1': 'ami-any',
                }),
                renderQueue,
                healthMonitor,
                blockDevices: [{
                        deviceName,
                        volume: aws_autoscaling_1.BlockDeviceVolume.noDevice(),
                    }],
            });
            return wfstack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::ElasticLoadBalancingV2::TargetGroup': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
test('worker fleet signals when non-zero minCapacity', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
        minCapacity: 1,
    });
    // WHEN
    const userData = fleet.fleet.userData.render();
    // THEN
    expect(userData).toContain('cfn-signal');
    assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        CreationPolicy: {
            ResourceSignal: {
                Count: 1,
            },
        },
    }, assert_1.ResourcePart.CompleteDefinition));
    // [0] = warning about block devices. [1] = warning about no health monitor
    expect(fleet.node.metadataEntry).toHaveLength(2);
});
test('worker fleet does not signal when zero minCapacity', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
        minCapacity: 0,
    });
    // WHEN
    const userData = fleet.fleet.userData.render();
    // THEN
    // There should be no cfn-signal call in the UserData.
    expect(userData).not.toContain('cfn-signal');
    // Make sure we don't have a CreationPolicy
    assert_1.expect(wfstack).notTo(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        CreationPolicy: assert_1.objectLike({}),
    }, assert_1.ResourcePart.CompleteDefinition));
    // There should be a warning in the construct's metadata about deploying with no capacity.
    expect(fleet.node.metadataEntry).toHaveLength(3);
    // [0] = warning about block devices. [2] = warning about no health monitor
    expect(fleet.node.metadataEntry[1].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
    expect(fleet.node.metadataEntry[1].data).toMatch(/Deploying with 0 minimum capacity./);
});
//# sourceMappingURL=data:application/json;base64,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