"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const __1 = require("../..");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
describe('RenderQueue', () => {
    let app;
    let dependencyStack;
    let stack;
    let vpc;
    let rcsImage;
    let images;
    let repository;
    let version;
    let renderQueueVersion;
    let renderQueueCommon;
    // GIVEN
    beforeEach(() => {
        app = new core_1.App();
        dependencyStack = new core_1.Stack(app, 'DepStack');
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'Vpc');
        version = new lib_1.VersionQuery(dependencyStack, 'Version');
        repository = new lib_1.Repository(dependencyStack, 'Repo', {
            version,
            vpc,
        });
        stack = new core_1.Stack(app, 'Stack');
        rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
        images = {
            remoteConnectionServer: rcsImage,
        };
        renderQueueVersion = new lib_1.VersionQuery(stack, 'Version');
        renderQueueCommon = new lib_1.RenderQueue(stack, 'RenderQueueCommon', {
            images,
            repository,
            version: renderQueueVersion,
            vpc,
        });
    });
    afterEach(() => {
        jest.resetAllMocks();
    });
    test('creates cluster', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Cluster'));
    });
    test('creates service', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service'));
    });
    test('creates task definition', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition'));
    });
    test('closed ingress by default', () => {
        // THEN
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            // The openListener=true option would create an ingress rule in the listener's SG.
            // make sure that we don't have that.
            SecurityGroupIngress: assert_1.arrayWith(assert_1.objectLike({})),
        }));
    });
    test('creates load balancer with default values', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 1, {
            LoadBalancerAttributes: [
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ],
            Scheme: 'internal',
        }));
    });
    test('creates a log group with default prefix of "/renderfarm/"', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
            LogGroupName: '/renderfarm/RenderQueueCommon',
            RetentionInDays: 3,
        }));
    });
    test('configure the container log driver', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                assert_1.objectLike({
                    LogConfiguration: {
                        LogDriver: 'awslogs',
                        Options: {
                            'awslogs-group': {
                                'Fn::GetAtt': [
                                    'RenderQueueCommonLogGroupWrapperA0EF7057',
                                    'LogGroupName',
                                ],
                            },
                            'awslogs-stream-prefix': 'RCS',
                            'awslogs-region': { Ref: 'AWS::Region' },
                        },
                    },
                }),
            ],
        }));
    });
    test('child dependencies added', () => {
        // GIVEN
        const host = new aws_ec2_1.Instance(stack, 'Host', {
            vpc,
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        renderQueueCommon.addChildDependency(host);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            DependsOn: assert_1.arrayWith('RenderQueueCommonLBPublicListener935F5635', 'RenderQueueCommonRCSTask2A4D5EA5', 'RenderQueueCommonAlbEc2ServicePatternService42BEFF4C', 'RenderQueueCommonWaitForStableServiceDB53E266'),
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    describe('renderQueueSize.min', () => {
        describe('defaults to 1', () => {
            function assertSpecifiesMinSize(stackToAssert) {
                assert_1.expect(stackToAssert).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    MinSize: '1',
                }));
            }
            test('renderQueueSize unspecified', () => {
                // THEN
                assertSpecifiesMinSize(stack);
            });
            test('renderQueueSize.min unspecified', () => {
                // GIVEN
                const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    renderQueueSize: {},
                });
                // THEN
                assertSpecifiesMinSize(isolatedStack);
            });
        });
        // Asserts that at least one RCS container and ASG instance must be created.
        test('throws error when minimum size is 0', () => {
            // GIVEN
            const props = {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
                renderQueueSize: {
                    min: 0,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(stack, 'RenderQueue', props);
            })
                // THEN
                .toThrow('renderQueueSize.min capacity must be at least 1: got 0');
        });
        // Deadline before 10.1.10 requires that successive API requests are serviced by a single RCS.
        test('validates Deadline pre 10.1.10 has min value of at most 1', () => {
            // GIVEN
            const min = 2;
            const newStack = new core_1.Stack(app, 'NewStack');
            const versionOld = new lib_1.VersionQuery(newStack, 'VersionOld', { version: '10.1.9' });
            const props = {
                images,
                repository,
                version: versionOld,
                vpc,
                renderQueueSize: {
                    min,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(newStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(`renderQueueSize.min for Deadline version less than 10.1.10.0 cannot be greater than 1 - got ${min}`);
        });
        // Asserts that when the renderQueueSize.min prop is specified, the underlying ASG's min property is set accordingly.
        test.each([
            [1],
            [2],
            [10],
        ])('configures minimum number of ASG instances to %d', (min) => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const props = {
                images,
                repository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                renderQueueSize: {
                    min,
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: min.toString(),
            }));
        });
    });
    describe('renderQueueSize.max', () => {
        describe('defaults to 1', () => {
            function assertSpecifiesMaxSize(stackToAssert) {
                assert_1.expect(stackToAssert).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    MaxSize: '1',
                }));
            }
            test('renderQueueSize unspecified', () => {
                // THEN
                assertSpecifiesMaxSize(stack);
            });
            test('renderQueueSize.max unspecified', () => {
                // GIVEN
                const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    renderQueueSize: {},
                });
                // THEN
                assertSpecifiesMaxSize(isolatedStack);
            });
        });
        // Deadline before 10.1.10 requires that successive API requests are serviced by a single RCS.
        test('validates Deadline pre 10.1.10 has max value of at most 1', () => {
            // GIVEN
            const max = 2;
            const newStack = new core_1.Stack(app, 'NewStack');
            const versionOld = new lib_1.VersionQuery(newStack, 'VersionOld', { version: '10.1.9' });
            const props = {
                images,
                repository,
                version: versionOld,
                vpc,
                renderQueueSize: {
                    max,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(newStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(`renderQueueSize.max for Deadline version less than 10.1.10.0 cannot be greater than 1 - got ${max}`);
        });
        // Asserts that when the renderQueueSize.max prop is specified, the underlying ASG's max property is set accordingly.
        test.each([
            [1],
            [2],
            [10],
        ])('configures maximum number of ASG instances to %d', (max) => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const props = {
                images,
                repository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                renderQueueSize: {
                    max,
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MaxSize: max.toString(),
            }));
        });
    });
    describe('renderQueueSize.desired', () => {
        describe('defaults', () => {
            test('unset ASG desired', () => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    DesiredCapacity: assert_1.ABSENT,
                }));
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: 1,
                }));
            });
        });
        test('validates Deadline pre 10.1.10 has desired value of at most 1', () => {
            // GIVEN
            const desired = 2;
            const newStack = new core_1.Stack(app, 'NewStack');
            const versionOld = new lib_1.VersionQuery(newStack, 'VersionOld', { version: '10.1.9' });
            const props = {
                images,
                repository,
                version: versionOld,
                vpc,
                renderQueueSize: {
                    desired,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(newStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(`renderQueueSize.desired for Deadline version less than 10.1.10.0 cannot be greater than 1 - got ${desired}`);
        });
        test.each([
            [1],
            [2],
            [10],
        ])('is specified to %d', (desired) => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const props = {
                images,
                repository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                renderQueueSize: {
                    desired,
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                DesiredCapacity: desired.toString(),
            }));
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                DesiredCount: desired,
            }));
        });
    });
    describe('trafficEncryption', () => {
        describe('defaults', () => {
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    trafficEncryption: {},
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('to HTTPS internally between ALB and RCS', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTPS',
                    Port: 4433,
                }));
            });
            test('to HTTP externally between clients and ALB', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTP',
                    Port: 8080,
                }));
            });
        });
        describe('when interalProtocol is HTTPS', () => {
            let isolatedStack;
            let renderQueue;
            let caCertPemLogicalId;
            let caCertPkcsLogicalId;
            let caCertPkcsPassphraseLogicalId;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    trafficEncryption: {
                        internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS,
                    },
                };
                // WHEN
                renderQueue = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
                caCertPemLogicalId = isolatedStack.getLogicalId(renderQueue.node.findChild('TlsCaCertPem').node.defaultChild);
                const caCertPkcs = renderQueue.node.findChild('TlsRcsCertBundle');
                const caCertPkcsPassphrase = caCertPkcs.node.findChild('Passphrase');
                caCertPkcsLogicalId = isolatedStack.getLogicalId(caCertPkcs.node.defaultChild);
                caCertPkcsPassphraseLogicalId = isolatedStack.getLogicalId(caCertPkcsPassphrase.node.defaultChild);
            });
            // THEN
            test('ALB connects with HTTPS to port 4433', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTPS',
                    Port: 4433,
                }));
            });
            test('creates RCS cert', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
                    ServiceToken: {
                        'Fn::GetAtt': assert_1.arrayWith('Arn'),
                    },
                    DistinguishedName: { CN: 'renderfarm.local' },
                    Secret: {
                        NamePrefix: 'IsolatedStack/RenderQueue/TlsCaCertPem',
                    },
                }));
            });
            test('grants read access to secrets containing the certs and passphrase', () => {
                const taskDef = renderQueue.node.findChild('RCSTask');
                const taskRoleLogicalId = isolatedStack.getLogicalId(taskDef.taskRole.node.defaultChild);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    caCertPemLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    caCertPkcsLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: { Ref: caCertPkcsPassphraseLogicalId },
                        }),
                        Version: '2012-10-17',
                    },
                    Roles: assert_1.arrayWith({ Ref: taskRoleLogicalId }),
                }));
            });
            test('configures environment variables for cert secret URIs', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                    ContainerDefinitions: assert_1.arrayWith(assert_1.deepObjectLike({
                        Environment: assert_1.arrayWith({
                            Name: 'RCS_TLS_CA_CERT_URI',
                            Value: {
                                'Fn::GetAtt': [
                                    caCertPemLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Name: 'RCS_TLS_CERT_URI',
                            Value: {
                                'Fn::GetAtt': [
                                    caCertPkcsLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Name: 'RCS_TLS_CERT_PASSPHRASE_URI',
                            Value: { Ref: caCertPkcsPassphraseLogicalId },
                        }),
                    })),
                }));
            });
        });
        describe('when internal protocol is HTTP', () => {
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    trafficEncryption: {
                        internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                    },
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('no certs are created', () => {
                assert_1.expect(isolatedStack).notTo(assert_1.haveResource('Custom::RFDK_X509Generator'));
            });
            test('ALB connects with HTTP to port 8080', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTP',
                    Port: 8080,
                }));
            });
        });
        describe('externalProtocol is HTTPS', () => {
            let isolatedStack;
            const CERT_ARN = 'certarn';
            const CA_ARN = 'arn:aws:secretsmanager:123456789012:secret:ca/arn';
            const ZONE_NAME = 'renderfarm.local';
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', CERT_ARN),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CA_Cert', CA_ARN),
                        },
                    },
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('sets the listener port to 4433', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Port: 4433,
                }));
            });
            test('sets the listener protocol to HTTPS', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                }));
            });
            test('configures the ALB listener to use the specified ACM certificate', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                    Certificates: assert_1.arrayWith({
                        CertificateArn: CERT_ARN,
                    }),
                }));
            });
            test('raises an error when a cert is specified without a hostname', () => {
                // GIVEN
                const props = {
                    images,
                    repository,
                    version: renderQueueVersion,
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Cert', 'certArn'),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CA_Cert2', CA_ARN),
                        },
                    },
                };
                // WHEN
                expect(() => {
                    new lib_1.RenderQueue(stack, 'RenderQueue', props);
                })
                    // THEN
                    .toThrow(/A hostname must be provided when the external protocol is HTTPS/);
            });
        });
        describe('externalProtocol is HTTPS importing cert', () => {
            let isolatedStack;
            let zone;
            const ZONE_NAME = 'renderfarm.local';
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const caCert = new __1.X509CertificatePem(isolatedStack, 'CaCert', {
                    subject: {
                        cn: `ca.${ZONE_NAME}`,
                    },
                });
                const serverCert = new __1.X509CertificatePem(isolatedStack, 'ServerCert', {
                    subject: {
                        cn: `server.${ZONE_NAME}`,
                    },
                    signingCertificate: caCert,
                });
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            rfdkCertificate: serverCert,
                        },
                        internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                    },
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('sets the listener port to 4433', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Port: 4433,
                }));
            });
            test('sets the listener protocol to HTTPS', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                }));
            });
            test('Imports Cert to ACM', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('Custom::RFDK_AcmImportedCertificate', {
                    X509CertificatePem: {
                        Cert: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'Cert',
                            ],
                        },
                        Key: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'Key',
                            ],
                        },
                        Passphrase: {
                            Ref: 'ServerCertPassphraseE4C3CB38',
                        },
                        CertChain: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'CertChain',
                            ],
                        },
                    },
                }));
            });
        });
        test('Throws if given ACM cert and RFDK Cert', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'certArn';
            const CA_ARN = 'arn:aws:secretsmanager:123456789012:secret:ca/arn';
            const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                vpc,
                zoneName: ZONE_NAME,
            });
            const caCert = new __1.X509CertificatePem(isolatedStack, 'CaCert', {
                subject: {
                    cn: `ca.${ZONE_NAME}`,
                },
            });
            const serverCert = new __1.X509CertificatePem(isolatedStack, 'ServerCert', {
                subject: {
                    cn: `server.${ZONE_NAME}`,
                },
                signingCertificate: caCert,
            });
            const props = {
                images,
                repository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                trafficEncryption: {
                    externalTLS: {
                        acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(isolatedStack, 'Certificate', CERT_ARN),
                        acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(isolatedStack, 'CA_Cert', CA_ARN),
                        rfdkCertificate: serverCert,
                    },
                },
                hostname: {
                    zone,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(/Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS/);
        });
        test('Throws if no Cert given', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const ZONE_NAME = 'renderfarm.local';
            const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                vpc,
                zoneName: ZONE_NAME,
            });
            const props = {
                images,
                repository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                trafficEncryption: {
                    externalTLS: {},
                },
                hostname: {
                    zone,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(/Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS/);
        });
        test('Throws if ACM Cert is given without a cert chain', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'certArn';
            const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                vpc,
                zoneName: ZONE_NAME,
            });
            const props = {
                images,
                repository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                trafficEncryption: {
                    externalTLS: {
                        acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(isolatedStack, 'Certificate', CERT_ARN),
                    },
                },
                hostname: {
                    zone,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(/externalTLS.acmCertificateChain must be provided when using externalTLS.acmCertificate./);
        });
    });
    describe('Client Connection', () => {
        describe('externalProtocol is http', () => {
            let isolatedStack;
            let zone;
            const ZONE_NAME = 'renderfarm.local';
            let rq;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                rq = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('ECS can connect', () => {
                // WHEN
                const hosts = [new aws_ec2_1.Instance(isolatedStack, 'Host', {
                        vpc,
                        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                    })];
                const role = new aws_iam_1.Role(isolatedStack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
                const env = rq.configureClientECS({
                    hosts,
                    grantee: role,
                });
                // THEN
                expect(env).toHaveProperty('RENDER_QUEUE_URI');
                expect(env.RENDER_QUEUE_URI).toMatch(/http:\/\/.*:8080$/);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(hosts[0].connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
                    DependsOn: assert_1.arrayWith('RenderQueueLBPublicListenerBBF15D5F', 'RenderQueueRCSTaskA9AE70D3'),
                }, assert_1.ResourcePart.CompleteDefinition));
            });
            test('Linux Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            "#!/bin/bash\nmkdir -p $(dirname '/tmp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\')\naws s3 cp \'s3://',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' \'/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\'\n' +
                                'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                                '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                                'fi\n' +
                                '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            `" --render-queue "http://renderqueue.${ZONE_NAME}:8080" \n` +
                                'rm -f "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                                '  service deadline10launcher restart\n' +
                                'fi',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
                    DependsOn: assert_1.arrayWith('RenderQueueLBPublicListenerBBF15D5F', 'RenderQueueRCSTaskA9AE70D3'),
                }, assert_1.ResourcePart.CompleteDefinition));
            });
            test('Windows Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_CORE_BASE),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' ) -ea 0\n' +
                                'Read-S3Object -BucketName \'',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '\' -key \'',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -file \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -ErrorAction Stop\n' +
                                '$ErrorActionPreference = "Stop"\n' +
                                '$DEADLINE_PATH = (get-item env:"DEADLINE_PATH").Value\n' +
                                '& "$DEADLINE_PATH/deadlinecommand.exe" -executeScriptNoGui "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            `" --render-queue "http://renderqueue.${ZONE_NAME}:8080"  2>&1\n` +
                                'Remove-Item -Path "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'If (Get-Service "deadline10launcherservice" -ErrorAction SilentlyContinue) {\n' +
                                '  Restart-Service "deadline10launcherservice"\n' +
                                '} Else {\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" -shutdownall 2>&1\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" -nogui 2>&1\n' +
                                '}</powershell>',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
                    DependsOn: assert_1.arrayWith('RenderQueueLBPublicListenerBBF15D5F', 'RenderQueueRCSTaskA9AE70D3'),
                }, assert_1.ResourcePart.CompleteDefinition));
            });
        });
        describe('externalProtocol is https', () => {
            let isolatedStack;
            let zone;
            let rq;
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'arn:a:b:c:dcertarn';
            const CA_ARN = 'arn:aws:secretsmanager:123456789012:secret:ca/arn';
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    hostname: {
                        zone,
                    },
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', CERT_ARN),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CA_Cert', CA_ARN),
                        },
                    },
                };
                // WHEN
                rq = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('ECS can connect', () => {
                // WHEN
                const hosts = [new aws_ec2_1.Instance(isolatedStack, 'Host', {
                        vpc,
                        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                    })];
                const role = new aws_iam_1.Role(isolatedStack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
                const env = rq.configureClientECS({
                    hosts,
                    grantee: role,
                });
                // THEN
                expect(env).toHaveProperty('RENDER_QUEUE_URI');
                expect(env.RENDER_QUEUE_URI).toMatch(/https:\/\/.*:4433$/);
                expect(env).toHaveProperty('RENDER_QUEUE_TLS_CA_CERT_URI', CA_ARN);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(hosts[0].connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
            });
            test('Linux Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            "#!/bin/bash\nmkdir -p $(dirname '/tmp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\')\naws s3 cp \'s3://',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' \'/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\'\n' +
                                'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                                '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                                'fi\n' +
                                '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            `" --render-queue "https://renderqueue.${ZONE_NAME}:4433" --tls-ca "${CA_ARN}"\n` +
                                'rm -f "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                                '  service deadline10launcher restart\n' +
                                'fi',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
            });
            test('Windows Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_CORE_BASE),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' ) -ea 0\n' +
                                'Read-S3Object -BucketName \'',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '\' -key \'',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -file \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -ErrorAction Stop\n' +
                                '$ErrorActionPreference = "Stop"\n' +
                                '$DEADLINE_PATH = (get-item env:"DEADLINE_PATH").Value\n' +
                                '& "$DEADLINE_PATH/deadlinecommand.exe" -executeScriptNoGui "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            `" --render-queue "https://renderqueue.${ZONE_NAME}:4433" --tls-ca \"${CA_ARN}\" 2>&1\n` +
                                'Remove-Item -Path "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'If (Get-Service "deadline10launcherservice" -ErrorAction SilentlyContinue) {\n' +
                                '  Restart-Service "deadline10launcherservice"\n' +
                                '} Else {\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" -shutdownall 2>&1\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" -nogui 2>&1\n' +
                                '}</powershell>',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
            });
        });
    });
    test('can specify subnets', () => {
        // GIVEN
        const subnets = [
            aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet1', {
                subnetId: 'SubnetID1',
                availabilityZone: 'us-west-2a',
            }),
            aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet2', {
                subnetId: 'SubnetID2',
                availabilityZone: 'us-west-2b',
            }),
        ];
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        const props = {
            images,
            repository,
            version: new lib_1.VersionQuery(isolatedStack, 'Version'),
            vpc,
            vpcSubnets: {
                subnets,
            },
            vpcSubnetsAlb: {
                subnets,
            },
        };
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        assert_1.expect(isolatedStack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            VPCZoneIdentifier: assert_1.arrayWith('SubnetID1', 'SubnetID2'),
        }));
        assert_1.expect(isolatedStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Subnets: [
                'SubnetID1',
                'SubnetID2',
            ],
        }));
    });
    test('can specify instance type', () => {
        // GIVEN
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        const props = {
            images,
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.C5, aws_ec2_1.InstanceSize.LARGE),
            repository,
            version: new lib_1.VersionQuery(isolatedStack, 'Version'),
            vpc,
        };
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: 'c5.large',
        }));
    });
    test('no deletion protection', () => {
        // GIVEN
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        const props = {
            images,
            repository,
            version: new lib_1.VersionQuery(isolatedStack, 'Version'),
            vpc,
            deletionProtection: false,
        };
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.not(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'deletion_protection.enabled',
                Value: 'true',
            }),
            Scheme: assert_1.ABSENT,
            Type: assert_1.ABSENT,
        })));
    });
    test('drop invalid http header fields enabled', () => {
        // GIVEN
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        const props = {
            images,
            repository,
            version: new lib_1.VersionQuery(isolatedStack, 'Version'),
            vpc,
        };
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'routing.http.drop_invalid_header_fields.enabled',
                Value: 'true',
            }),
        }));
    });
    describe('hostname', () => {
        // GIVEN
        const zoneName = 'mydomain.local';
        describe('not specified', () => {
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('does not create a record set', () => {
                assert_1.expect(isolatedStack).notTo(assert_1.haveResource('AWS::Route53::RecordSet'));
            });
        });
        describe('specified with zone but no hostname', () => {
            let zone;
            let isolatedStack;
            let renderQueue;
            beforeEach(() => {
                // GIVEN
                zone = new aws_route53_1.PrivateHostedZone(dependencyStack, 'Zone', {
                    vpc,
                    zoneName,
                });
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                renderQueue = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('creates a record set using default hostname', () => {
                const loadBalancerLogicalId = dependencyStack.getLogicalId(renderQueue.loadBalancer.node.defaultChild);
                assert_1.expect(isolatedStack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
                    Name: `renderqueue.${zoneName}.`,
                    Type: 'A',
                    AliasTarget: assert_1.objectLike({
                        HostedZoneId: {
                            'Fn::GetAtt': [
                                loadBalancerLogicalId,
                                'CanonicalHostedZoneID',
                            ],
                        },
                    }),
                }));
            });
        });
        test.each([
            ['rq.somedomain.local'],
            ['1startswithnumber'],
            ['-startswithhyphen'],
            ['endswith-'],
            ['contains$symbol'],
            ['thisstringisexactlysixtyfourcharacterswhichisonelargerthanthemax'],
        ])('.hostname validation - %s', (hostname) => {
            // GIVEN
            const zone = new aws_route53_1.PrivateHostedZone(dependencyStack, 'Zone', {
                zoneName: 'somedomain.local',
                vpc,
            });
            const props = {
                images,
                repository,
                version,
                vpc,
                hostname: {
                    hostname,
                    zone,
                },
            };
            // WHEN
            function when() {
                new lib_1.RenderQueue(stack, 'NewRenderQueue', props);
            }
            // THEN
            expect(when).toThrow(/Invalid RenderQueue hostname/);
        });
    });
    describe('Access Logs', () => {
        let isolatedStack;
        let isolatedVpc;
        let isolatedRepository;
        let isolatedVersion;
        let isolatedimages;
        let accessBucket;
        beforeEach(() => {
            // GIVEN
            isolatedStack = new core_1.Stack(app, 'IsolatedStack', {
                env: {
                    region: 'us-east-1',
                },
            });
            isolatedVpc = new aws_ec2_1.Vpc(isolatedStack, 'Vpc');
            isolatedVersion = new lib_1.VersionQuery(isolatedStack, 'Version');
            isolatedRepository = new lib_1.Repository(isolatedStack, 'Repo', {
                version: isolatedVersion,
                vpc: isolatedVpc,
            });
            isolatedimages = {
                remoteConnectionServer: rcsImage,
            };
            accessBucket = new aws_s3_1.Bucket(isolatedStack, 'AccessBucket');
        });
        test('enabling access logs sets attributes and policies', () => {
            // GIVEN
            const props = {
                images: isolatedimages,
                repository: isolatedRepository,
                version: isolatedVersion,
                vpc: isolatedVpc,
                accessLogs: {
                    destinationBucket: accessBucket,
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
                LoadBalancerAttributes: assert_1.arrayWith({
                    Key: 'access_logs.s3.enabled',
                    Value: 'true',
                }, {
                    Key: 'access_logs.s3.bucket',
                    Value: {
                        Ref: 'AccessBucketE2803D76',
                    },
                }),
            }));
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
                Bucket: {
                    Ref: 'AccessBucketE2803D76',
                },
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: 's3:PutObject',
                        Condition: {
                            StringEquals: {
                                's3:x-amz-acl': 'bucket-owner-full-control',
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }, {
                        Action: 's3:GetBucketAcl',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::GetAtt': [
                                'AccessBucketE2803D76',
                                'Arn',
                            ],
                        },
                    }, {
                        Action: [
                            's3:PutObject*',
                            's3:Abort*',
                        ],
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::127311923021:root',
                                    ],
                                ],
                            },
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/AWSLogs/',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }),
                },
            }));
        });
        test('enabling access logs works with prefix', () => {
            // GIVEN
            const props = {
                images: isolatedimages,
                repository: isolatedRepository,
                version: isolatedVersion,
                vpc: isolatedVpc,
                accessLogs: {
                    destinationBucket: accessBucket,
                    prefix: 'PREFIX_STRING',
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
                LoadBalancerAttributes: assert_1.arrayWith({
                    Key: 'access_logs.s3.enabled',
                    Value: 'true',
                }, {
                    Key: 'access_logs.s3.bucket',
                    Value: {
                        Ref: 'AccessBucketE2803D76',
                    },
                }, {
                    Key: 'access_logs.s3.prefix',
                    Value: 'PREFIX_STRING',
                }),
            }));
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
                Bucket: {
                    Ref: 'AccessBucketE2803D76',
                },
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: 's3:PutObject',
                        Condition: {
                            StringEquals: {
                                's3:x-amz-acl': 'bucket-owner-full-control',
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }, {
                        Action: 's3:GetBucketAcl',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::GetAtt': [
                                'AccessBucketE2803D76',
                                'Arn',
                            ],
                        },
                    }, {
                        Action: [
                            's3:PutObject*',
                            's3:Abort*',
                        ],
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::127311923021:root',
                                    ],
                                ],
                            },
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/PREFIX_STRING/AWSLogs/',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }),
                },
            }));
        });
    });
    describe('tagging', () => {
        tag_helpers_1.testConstructTags({
            constructName: 'RenderQueue',
            createConstruct: () => {
                return stack;
            },
            resourceTypeCounts: {
                'AWS::ECS::Cluster': 1,
                'AWS::EC2::SecurityGroup': 2,
                'AWS::IAM::Role': 8,
                'AWS::AutoScaling::AutoScalingGroup': 1,
                'AWS::Lambda::Function': 4,
                'AWS::SNS::Topic': 1,
                'AWS::ECS::TaskDefinition': 1,
                'AWS::DynamoDB::Table': 2,
                'AWS::SecretsManager::Secret': 2,
                'AWS::ElasticLoadBalancingV2::LoadBalancer': 1,
                'AWS::ElasticLoadBalancingV2::TargetGroup': 1,
                'AWS::ECS::Service': 1,
            },
        });
    });
    describe('SEP Policies', () => {
        test('with resource tracker', () => {
            // WHEN
            renderQueueCommon.addSEPPolicies();
            // THEN
            assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Role', 1, {
                ManagedPolicyArns: assert_1.arrayWith({
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/AWSThinkboxDeadlineSpotEventPluginAdminPolicy',
                        ],
                    ],
                }, {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/AWSThinkboxDeadlineResourceTrackerAdminPolicy',
                        ],
                    ],
                }),
            }));
        });
        test('no resource tracker', () => {
            // WHEN
            renderQueueCommon.addSEPPolicies(false);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                ManagedPolicyArns: assert_1.arrayWith({
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/AWSThinkboxDeadlineSpotEventPluginAdminPolicy',
                        ],
                    ],
                }),
            }));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Role', {
                ManagedPolicyArns: assert_1.arrayWith({
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/AWSThinkboxDeadlineResourceTrackerAdminPolicy',
                        ],
                    ],
                }),
            }));
        });
    });
    test('creates WaitForStableService by default', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_WaitForStableService', {
            cluster: stack.resolve(renderQueueCommon.cluster.clusterArn),
            // eslint-disable-next-line dot-notation
            services: [stack.resolve(renderQueueCommon['pattern'].service.serviceArn)],
        }));
    });
    describe('Security Groups', () => {
        let backendSecurityGroup;
        let frontendSecurityGroup;
        beforeEach(() => {
            backendSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'ASGSecurityGroup', { vpc });
            frontendSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'LBSecurityGroup', { vpc });
        });
        test('adds security groups on construction', () => {
            // GIVEN
            const securityGroups = {
                backend: backendSecurityGroup,
                frontend: frontendSecurityGroup,
            };
            // WHEN
            new lib_1.RenderQueue(stack, 'RenderQueue', {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
                securityGroups,
            });
            // THEN
            assertSecurityGroupsWereAdded(securityGroups);
        });
        test('adds backend security groups post-construction', () => {
            // GIVEN
            const renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
            });
            // WHEN
            renderQueue.addBackendSecurityGroups(backendSecurityGroup);
            // THEN
            assertSecurityGroupsWereAdded({
                backend: backendSecurityGroup,
            });
        });
        test('adds frontend security groups post-construction', () => {
            // GIVEN
            const renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
            });
            // WHEN
            renderQueue.addFrontendSecurityGroups(frontendSecurityGroup);
            // THEN
            assertSecurityGroupsWereAdded({
                frontend: frontendSecurityGroup,
            });
        });
        test('security groups added post-construction are not attached to Connections object', () => {
            // GIVEN
            const renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
            });
            renderQueue.addBackendSecurityGroups(backendSecurityGroup);
            renderQueue.addFrontendSecurityGroups(frontendSecurityGroup);
            const peerSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'PeerSecurityGroup', { vpc });
            // WHEN
            renderQueue.connections.allowFrom(peerSecurityGroup, aws_ec2_1.Port.tcp(22));
            // THEN
            // Existing LoadBalancer security groups shouldn't have the ingress rule added
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 22,
                ToPort: 22,
                GroupId: stack.resolve(frontendSecurityGroup.securityGroupId),
                SourceSecurityGroupId: stack.resolve(peerSecurityGroup.securityGroupId),
            }));
            // Existing AutoScalingGroup security groups shouldn't have the ingress rule added
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 22,
                ToPort: 22,
                GroupId: stack.resolve(backendSecurityGroup.securityGroupId),
                SourceSecurityGroupId: stack.resolve(peerSecurityGroup.securityGroupId),
            }));
        });
        function assertSecurityGroupsWereAdded(securityGroups) {
            if (securityGroups.backend !== undefined) {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                    SecurityGroups: assert_1.arrayWith(stack.resolve(securityGroups.backend.securityGroupId)),
                }));
            }
            if (securityGroups.frontend !== undefined) {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
                    SecurityGroups: assert_1.arrayWith(stack.resolve(securityGroups.frontend.securityGroupId)),
                }));
            }
        }
    });
    test('validates VersionQuery is not in a different stack', () => {
        // GIVEN
        const newStack = new core_1.Stack(app, 'NewStack');
        // WHEN
        new lib_1.RenderQueue(newStack, 'RenderQueueNew', {
            images,
            repository,
            version,
            vpc,
        });
        // WHEN
        function synth() {
            assert_1.SynthUtils.synthesize(newStack);
        }
        // THEN
        expect(synth).toThrow('A VersionQuery can not be supplied from a different stack');
    });
    test('Does not enable filesystem cache by default', () => {
        assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': assert_1.arrayWith(assert_1.arrayWith(' >> /etc/ecs/ecs.config\nsudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP\nsudo service iptables save\necho ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config\nyum install -yq awscli unzip\n# RenderQueue file caching enabled\nmkdir -p $(dirname \'/tmp/')),
                },
            },
        }));
    });
    test('Enables filesystem cache if required', () => {
        // GIVEN
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', {
            images,
            repository,
            version: new lib_1.VersionQuery(isolatedStack, 'Version'),
            vpc,
            enableLocalFileCaching: true,
        });
        // THEN
        // Note: If this test breaks/fails, then it is probable that the
        //  'Does not enable filesystem cache by default' test above will also require
        //  updating/fixing.
        assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': assert_1.arrayWith(assert_1.arrayWith(' >> /etc/ecs/ecs.config\nsudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP\nsudo service iptables save\necho ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config\nyum install -yq awscli unzip\n# RenderQueue file caching enabled\nmkdir -p $(dirname \'/tmp/')),
                },
            },
        }));
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVuZGVyLXF1ZXVlLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJyZW5kZXItcXVldWUudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7OztHQUdHOztBQUVILDRDQVl5QjtBQUN6Qiw0RUFFeUM7QUFDekMsOENBWTBCO0FBQzFCLDhDQUcwQjtBQUMxQixvRkFFNkM7QUFDN0MsOENBRzBCO0FBQzFCLHNEQUU4QjtBQUM5Qiw0Q0FFeUI7QUFDekIsb0VBQXFEO0FBQ3JELHdDQUl1QjtBQUV2Qiw2QkFFZTtBQUNmLDZEQUVxQztBQUNyQyxnQ0FRZ0I7QUFDaEIsdURBRTJCO0FBRTNCLFFBQVEsQ0FBQyxhQUFhLEVBQUUsR0FBRyxFQUFFO0lBQzNCLElBQUksR0FBUSxDQUFDO0lBQ2IsSUFBSSxlQUFzQixDQUFDO0lBQzNCLElBQUksS0FBWSxDQUFDO0lBQ2pCLElBQUksR0FBUSxDQUFDO0lBQ2IsSUFBSSxRQUF3QixDQUFDO0lBQzdCLElBQUksTUFBeUIsQ0FBQztJQUU5QixJQUFJLFVBQXNCLENBQUM7SUFDM0IsSUFBSSxPQUFpQixDQUFDO0lBQ3RCLElBQUksa0JBQTRCLENBQUM7SUFFakMsSUFBSSxpQkFBOEIsQ0FBQztJQUVuQyxRQUFRO0lBQ1IsVUFBVSxDQUFDLEdBQUcsRUFBRTtRQUNkLEdBQUcsR0FBRyxJQUFJLFVBQUcsRUFBRSxDQUFDO1FBQ2hCLGVBQWUsR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDN0MsR0FBRyxHQUFHLElBQUksYUFBRyxDQUFDLGVBQWUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUN0QyxPQUFPLEdBQUcsSUFBSSxrQkFBWSxDQUFDLGVBQWUsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUN2RCxVQUFVLEdBQUcsSUFBSSxnQkFBVSxDQUFDLGVBQWUsRUFBRSxNQUFNLEVBQUU7WUFDbkQsT0FBTztZQUNQLEdBQUc7U0FDSixDQUFDLENBQUM7UUFDSCxLQUFLLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ2hDLFFBQVEsR0FBRyx3QkFBYyxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUMvQyxNQUFNLEdBQUc7WUFDUCxzQkFBc0IsRUFBRSxRQUFRO1NBQ2pDLENBQUM7UUFDRixrQkFBa0IsR0FBRyxJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3hELGlCQUFpQixHQUFHLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsbUJBQW1CLEVBQUU7WUFDOUQsTUFBTTtZQUNOLFVBQVU7WUFDVixPQUFPLEVBQUUsa0JBQWtCO1lBQzNCLEdBQUc7U0FDSixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFNBQVMsQ0FBQyxHQUFHLEVBQUU7UUFDYixJQUFJLENBQUMsYUFBYSxFQUFFLENBQUM7SUFDdkIsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsaUJBQWlCLEVBQUUsR0FBRyxFQUFFO1FBQzNCLE9BQU87UUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxDQUFDO0lBQ3pELENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLGlCQUFpQixFQUFFLEdBQUcsRUFBRTtRQUMzQixPQUFPO1FBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLG1CQUFtQixDQUFDLENBQUMsQ0FBQztJQUN6RCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyx5QkFBeUIsRUFBRSxHQUFHLEVBQUU7UUFDbkMsT0FBTztRQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQywwQkFBMEIsQ0FBQyxDQUFDLENBQUM7SUFDaEUsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsMkJBQTJCLEVBQUUsR0FBRyxFQUFFO1FBQ3JDLE9BQU87UUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsS0FBSyxDQUFDLHFCQUFZLENBQUMseUJBQXlCLEVBQUU7WUFDN0Qsa0ZBQWtGO1lBQ2xGLHFDQUFxQztZQUNyQyxvQkFBb0IsRUFBRSxrQkFBUyxDQUFDLG1CQUFVLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDaEQsQ0FBQyxDQUFDLENBQUM7SUFDTixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQywyQ0FBMkMsRUFBRSxHQUFHLEVBQUU7UUFDckQsT0FBTztRQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsMkJBQWtCLENBQUMsMkNBQTJDLEVBQUUsQ0FBQyxFQUFFO1lBQ3JGLHNCQUFzQixFQUFFO2dCQUN0QjtvQkFDRSxHQUFHLEVBQUUsNkJBQTZCO29CQUNsQyxLQUFLLEVBQUUsTUFBTTtpQkFDZDthQUNGO1lBQ0QsTUFBTSxFQUFFLFVBQVU7U0FDbkIsQ0FBQyxDQUFDLENBQUM7SUFDTixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQywyREFBMkQsRUFBRSxHQUFHLEVBQUU7UUFDckUsT0FBTztRQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsc0JBQXNCLEVBQUU7WUFDM0QsWUFBWSxFQUFFLCtCQUErQjtZQUM3QyxlQUFlLEVBQUUsQ0FBQztTQUNuQixDQUFDLENBQUMsQ0FBQztJQUNOLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLG9DQUFvQyxFQUFFLEdBQUcsRUFBRTtRQUM5QyxPQUFPO1FBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywwQkFBMEIsRUFBRTtZQUMvRCxvQkFBb0IsRUFBRTtnQkFDcEIsbUJBQVUsQ0FBQztvQkFDVCxnQkFBZ0IsRUFBRTt3QkFDaEIsU0FBUyxFQUFFLFNBQVM7d0JBQ3BCLE9BQU8sRUFBRTs0QkFDUCxlQUFlLEVBQUU7Z0NBQ2YsWUFBWSxFQUFFO29DQUNaLDBDQUEwQztvQ0FDMUMsY0FBYztpQ0FDZjs2QkFDRjs0QkFDRCx1QkFBdUIsRUFBRSxLQUFLOzRCQUM5QixnQkFBZ0IsRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7eUJBQ3pDO3FCQUNGO2lCQUNGLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsMEJBQTBCLEVBQUUsR0FBRyxFQUFFO1FBQ3BDLFFBQVE7UUFDUixNQUFNLElBQUksR0FBRyxJQUFJLGtCQUFRLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtZQUN2QyxHQUFHO1lBQ0gsWUFBWSxFQUFFLHNCQUFZLENBQUMsRUFBRSxDQUMzQix1QkFBYSxDQUFDLEVBQUUsRUFDaEIsc0JBQVksQ0FBQyxLQUFLLENBQ25CO1lBQ0QsWUFBWSxFQUFFLHNCQUFZLENBQUMsaUJBQWlCLENBQUMsRUFBRSxVQUFVLEVBQUUsK0JBQXFCLENBQUMsY0FBYyxFQUFFLENBQUM7U0FDbkcsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGlCQUFpQixDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxDQUFDO1FBRTNDLE9BQU87UUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLG9CQUFvQixFQUFFO1lBQ3pELFNBQVMsRUFBRSxrQkFBUyxDQUNsQiwyQ0FBMkMsRUFDM0Msa0NBQWtDLEVBQ2xDLHNEQUFzRCxFQUN0RCwrQ0FBK0MsQ0FDaEQ7U0FDRixFQUFFLHFCQUFZLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO0lBQ3ZDLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixFQUFFLEdBQUcsRUFBRTtRQUNuQyxRQUFRLENBQUMsZUFBZSxFQUFFLEdBQUcsRUFBRTtZQUM3QixTQUFTLHNCQUFzQixDQUFDLGFBQW9CO2dCQUNsRCxlQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLG9DQUFvQyxFQUFFO29CQUNqRixPQUFPLEVBQUUsR0FBRztpQkFDYixDQUFDLENBQUMsQ0FBQztZQUNOLENBQUM7WUFFRCxJQUFJLENBQUMsNkJBQTZCLEVBQUUsR0FBRyxFQUFFO2dCQUN2QyxPQUFPO2dCQUNQLHNCQUFzQixDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQ2hDLENBQUMsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLGlDQUFpQyxFQUFFLEdBQUcsRUFBRTtnQkFDM0MsUUFBUTtnQkFDUixNQUFNLGFBQWEsR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsZUFBZSxDQUFDLENBQUM7Z0JBRXRELE9BQU87Z0JBQ1AsSUFBSSxpQkFBVyxDQUFDLGFBQWEsRUFBRSxhQUFhLEVBQUU7b0JBQzVDLE1BQU07b0JBQ04sVUFBVTtvQkFDVixPQUFPLEVBQUUsSUFBSSxrQkFBWSxDQUFDLGFBQWEsRUFBRSxTQUFTLENBQUM7b0JBQ25ELEdBQUc7b0JBQ0gsZUFBZSxFQUFFLEVBQUU7aUJBQ3BCLENBQUMsQ0FBQztnQkFFSCxPQUFPO2dCQUNQLHNCQUFzQixDQUFDLGFBQWEsQ0FBQyxDQUFDO1lBQ3hDLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCw0RUFBNEU7UUFDNUUsSUFBSSxDQUFDLHFDQUFxQyxFQUFFLEdBQUcsRUFBRTtZQUMvQyxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQXFCO2dCQUM5QixNQUFNO2dCQUNOLFVBQVU7Z0JBQ1YsT0FBTyxFQUFFLGtCQUFrQjtnQkFDM0IsR0FBRztnQkFDSCxlQUFlLEVBQUU7b0JBQ2YsR0FBRyxFQUFFLENBQUM7aUJBQ1A7YUFDRixDQUFDO1lBRUYsT0FBTztZQUNQLE1BQU0sQ0FBQyxHQUFHLEVBQUU7Z0JBQ1YsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxhQUFhLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDL0MsQ0FBQyxDQUFDO2dCQUNBLE9BQU87aUJBQ04sT0FBTyxDQUFDLHdEQUF3RCxDQUFDLENBQUM7UUFDdkUsQ0FBQyxDQUFDLENBQUM7UUFFSCw4RkFBOEY7UUFDOUYsSUFBSSxDQUFDLDJEQUEyRCxFQUFFLEdBQUcsRUFBRTtZQUNyRSxRQUFRO1lBQ1IsTUFBTSxHQUFHLEdBQUcsQ0FBQyxDQUFDO1lBQ2QsTUFBTSxRQUFRLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1lBQzVDLE1BQU0sVUFBVSxHQUFHLElBQUksa0JBQVksQ0FBQyxRQUFRLEVBQUUsWUFBWSxFQUFFLEVBQUMsT0FBTyxFQUFFLFFBQVEsRUFBQyxDQUFDLENBQUM7WUFDakYsTUFBTSxLQUFLLEdBQXFCO2dCQUM5QixNQUFNO2dCQUNOLFVBQVU7Z0JBQ1YsT0FBTyxFQUFFLFVBQVU7Z0JBQ25CLEdBQUc7Z0JBQ0gsZUFBZSxFQUFFO29CQUNmLEdBQUc7aUJBQ0o7YUFDRixDQUFDO1lBRUYsT0FBTztZQUNQLE1BQU0sQ0FBQyxHQUFHLEVBQUU7Z0JBQ1YsSUFBSSxpQkFBVyxDQUFDLFFBQVEsRUFBRSxhQUFhLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDbEQsQ0FBQyxDQUFDO2dCQUNGLE9BQU87aUJBQ0osT0FBTyxDQUFDLCtGQUErRixHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBQ25ILENBQUMsQ0FBQyxDQUFDO1FBRUgscUhBQXFIO1FBQ3JILElBQUksQ0FBQyxJQUFJLENBQUM7WUFDUixDQUFDLENBQUMsQ0FBQztZQUNILENBQUMsQ0FBQyxDQUFDO1lBQ0gsQ0FBQyxFQUFFLENBQUM7U0FDTCxDQUFDLENBQUMsa0RBQWtELEVBQUUsQ0FBQyxHQUFXLEVBQUUsRUFBRTtZQUNyRSxRQUFRO1lBQ1IsTUFBTSxhQUFhLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLGVBQWUsQ0FBQyxDQUFDO1lBQ3RELE1BQU0sS0FBSyxHQUFxQjtnQkFDOUIsTUFBTTtnQkFDTixVQUFVO2dCQUNWLE9BQU8sRUFBRSxJQUFJLGtCQUFZLENBQUMsYUFBYSxFQUFFLFNBQVMsQ0FBQztnQkFDbkQsR0FBRztnQkFDSCxlQUFlLEVBQUU7b0JBQ2YsR0FBRztpQkFDSjthQUNGLENBQUM7WUFFRixPQUFPO1lBQ1AsSUFBSSxpQkFBVyxDQUFDLGFBQWEsRUFBRSxhQUFhLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFFckQsT0FBTztZQUNQLGVBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsb0NBQW9DLEVBQUU7Z0JBQ2pGLE9BQU8sRUFBRSxHQUFHLENBQUMsUUFBUSxFQUFFO2FBQ3hCLENBQUMsQ0FBQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsRUFBRSxHQUFHLEVBQUU7UUFDbkMsUUFBUSxDQUFDLGVBQWUsRUFBRSxHQUFHLEVBQUU7WUFDN0IsU0FBUyxzQkFBc0IsQ0FBQyxhQUFvQjtnQkFDbEQsZUFBUyxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxvQ0FBb0MsRUFBRTtvQkFDakYsT0FBTyxFQUFFLEdBQUc7aUJBQ2IsQ0FBQyxDQUFDLENBQUM7WUFDTixDQUFDO1lBRUQsSUFBSSxDQUFDLDZCQUE2QixFQUFFLEdBQUcsRUFBRTtnQkFDdkMsT0FBTztnQkFDUCxzQkFBc0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUNoQyxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxpQ0FBaUMsRUFBRSxHQUFHLEVBQUU7Z0JBQzNDLFFBQVE7Z0JBQ1IsTUFBTSxhQUFhLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLGVBQWUsQ0FBQyxDQUFDO2dCQUV0RCxPQUFPO2dCQUNQLElBQUksaUJBQVcsQ0FBQyxhQUFhLEVBQUUsYUFBYSxFQUFFO29CQUM1QyxNQUFNO29CQUNOLFVBQVU7b0JBQ1YsT0FBTyxFQUFFLElBQUksa0JBQVksQ0FBQyxhQUFhLEVBQUUsU0FBUyxDQUFDO29CQUNuRCxHQUFHO29CQUNILGVBQWUsRUFBRSxFQUFFO2lCQUNwQixDQUFDLENBQUM7Z0JBRUgsT0FBTztnQkFDUCxzQkFBc0IsQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUN4QyxDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsOEZBQThGO1FBQzlGLElBQUksQ0FBQywyREFBMkQsRUFBRSxHQUFHLEVBQUU7WUFDckUsUUFBUTtZQUNSLE1BQU0sR0FBRyxHQUFHLENBQUMsQ0FBQztZQUNkLE1BQU0sUUFBUSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxVQUFVLENBQUMsQ0FBQztZQUM1QyxNQUFNLFVBQVUsR0FBRyxJQUFJLGtCQUFZLENBQUMsUUFBUSxFQUFFLFlBQVksRUFBRSxFQUFDLE9BQU8sRUFBRSxRQUFRLEVBQUMsQ0FBQyxDQUFDO1lBQ2pGLE1BQU0sS0FBSyxHQUFxQjtnQkFDOUIsTUFBTTtnQkFDTixVQUFVO2dCQUNWLE9BQU8sRUFBRSxVQUFVO2dCQUNuQixHQUFHO2dCQUNILGVBQWUsRUFBRTtvQkFDZixHQUFHO2lCQUNKO2FBQ0YsQ0FBQztZQUVGLE9BQU87WUFDUCxNQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNWLElBQUksaUJBQVcsQ0FBQyxRQUFRLEVBQUUsYUFBYSxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQ2xELENBQUMsQ0FBQztnQkFDRixPQUFPO2lCQUNKLE9BQU8sQ0FBQywrRkFBK0YsR0FBRyxFQUFFLENBQUMsQ0FBQztRQUNuSCxDQUFDLENBQUMsQ0FBQztRQUVILHFIQUFxSDtRQUNySCxJQUFJLENBQUMsSUFBSSxDQUFDO1lBQ1IsQ0FBQyxDQUFDLENBQUM7WUFDSCxDQUFDLENBQUMsQ0FBQztZQUNILENBQUMsRUFBRSxDQUFDO1NBQ0wsQ0FBQyxDQUFDLGtEQUFrRCxFQUFFLENBQUMsR0FBVyxFQUFFLEVBQUU7WUFDckUsUUFBUTtZQUNSLE1BQU0sYUFBYSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxlQUFlLENBQUMsQ0FBQztZQUN0RCxNQUFNLEtBQUssR0FBcUI7Z0JBQzlCLE1BQU07Z0JBQ04sVUFBVTtnQkFDVixPQUFPLEVBQUUsSUFBSSxrQkFBWSxDQUFDLGFBQWEsRUFBRSxTQUFTLENBQUM7Z0JBQ25ELEdBQUc7Z0JBQ0gsZUFBZSxFQUFFO29CQUNmLEdBQUc7aUJBQ0o7YUFDRixDQUFDO1lBRUYsT0FBTztZQUNQLElBQUksaUJBQVcsQ0FBQyxhQUFhLEVBQUUsYUFBYSxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBRXJELE9BQU87WUFDUCxlQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLG9DQUFvQyxFQUFFO2dCQUNqRixPQUFPLEVBQUUsR0FBRyxDQUFDLFFBQVEsRUFBRTthQUN4QixDQUFDLENBQUMsQ0FBQztRQUNOLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMseUJBQXlCLEVBQUUsR0FBRyxFQUFFO1FBQ3ZDLFFBQVEsQ0FBQyxVQUFVLEVBQUUsR0FBRyxFQUFFO1lBQ3hCLElBQUksQ0FBQyxtQkFBbUIsRUFBRSxHQUFHLEVBQUU7Z0JBQzdCLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsb0NBQW9DLEVBQUU7b0JBQ3pFLGVBQWUsRUFBRSxlQUFNO2lCQUN4QixDQUFDLENBQUMsQ0FBQztnQkFDSixlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLG1CQUFtQixFQUFFO29CQUN4RCxZQUFZLEVBQUUsQ0FBQztpQkFDaEIsQ0FBQyxDQUFDLENBQUM7WUFDTixDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLCtEQUErRCxFQUFFLEdBQUcsRUFBRTtZQUN6RSxRQUFRO1lBQ1IsTUFBTSxPQUFPLEdBQUcsQ0FBQyxDQUFDO1lBQ2xCLE1BQU0sUUFBUSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxVQUFVLENBQUMsQ0FBQztZQUM1QyxNQUFNLFVBQVUsR0FBRyxJQUFJLGtCQUFZLENBQUMsUUFBUSxFQUFFLFlBQVksRUFBRSxFQUFDLE9BQU8sRUFBRSxRQUFRLEVBQUMsQ0FBQyxDQUFDO1lBQ2pGLE1BQU0sS0FBSyxHQUFxQjtnQkFDOUIsTUFBTTtnQkFDTixVQUFVO2dCQUNWLE9BQU8sRUFBRSxVQUFVO2dCQUNuQixHQUFHO2dCQUNILGVBQWUsRUFBRTtvQkFDZixPQUFPO2lCQUNSO2FBQ0YsQ0FBQztZQUVGLE9BQU87WUFDUCxNQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNWLElBQUksaUJBQVcsQ0FBQyxRQUFRLEVBQUUsYUFBYSxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQ2xELENBQUMsQ0FBQztnQkFDQSxPQUFPO2lCQUNOLE9BQU8sQ0FBQyxtR0FBbUcsT0FBTyxFQUFFLENBQUMsQ0FBQztRQUMzSCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLENBQUM7WUFDUixDQUFDLENBQUMsQ0FBQztZQUNILENBQUMsQ0FBQyxDQUFDO1lBQ0gsQ0FBQyxFQUFFLENBQUM7U0FDTCxDQUFDLENBQUMsb0JBQW9CLEVBQUUsQ0FBQyxPQUFlLEVBQUUsRUFBRTtZQUMzQyxRQUFRO1lBQ1IsTUFBTSxhQUFhLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLGVBQWUsQ0FBQyxDQUFDO1lBQ3RELE1BQU0sS0FBSyxHQUFxQjtnQkFDOUIsTUFBTTtnQkFDTixVQUFVO2dCQUNWLE9BQU8sRUFBRSxJQUFJLGtCQUFZLENBQUMsYUFBYSxFQUFFLFNBQVMsQ0FBQztnQkFDbkQsR0FBRztnQkFDSCxlQUFlLEVBQUU7b0JBQ2YsT0FBTztpQkFDUjthQUNGLENBQUM7WUFFRixPQUFPO1lBQ1AsSUFBSSxpQkFBVyxDQUFDLGFBQWEsRUFBRSxhQUFhLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFFckQsT0FBTztZQUNQLGVBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsb0NBQW9DLEVBQUU7Z0JBQ2pGLGVBQWUsRUFBRSxPQUFPLENBQUMsUUFBUSxFQUFFO2FBQ3BDLENBQUMsQ0FBQyxDQUFDO1lBQ0osT0FBTztZQUNQLGVBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsbUJBQW1CLEVBQUU7Z0JBQ2hFLFlBQVksRUFBRSxPQUFPO2FBQ3RCLENBQUMsQ0FBQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxtQkFBbUIsRUFBRSxHQUFHLEVBQUU7UUFDakMsUUFBUSxDQUFDLFVBQVUsRUFBRSxHQUFHLEVBQUU7WUFDeEIsSUFBSSxhQUFvQixDQUFDO1lBRXpCLFVBQVUsQ0FBQyxHQUFHLEVBQUU7Z0JBQ2QsUUFBUTtnQkFDUixhQUFhLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLGVBQWUsQ0FBQyxDQUFDO2dCQUNoRCxNQUFNLEtBQUssR0FBcUI7b0JBQzlCLE1BQU07b0JBQ04sVUFBVTtvQkFDVixPQUFPLEVBQUUsSUFBSSxrQkFBWSxDQUFDLGFBQWEsRUFBRSxTQUFTLENBQUM7b0JBQ25ELEdBQUc7b0JBQ0gsaUJBQWlCLEVBQUUsRUFBRTtpQkFDdEIsQ0FBQztnQkFFRixPQUFPO2dCQUNQLElBQUksaUJBQVcsQ0FBQyxhQUFhLEVBQUUsYUFBYSxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQ3ZELENBQUMsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLElBQUksQ0FBQyx5Q0FBeUMsRUFBRSxHQUFHLEVBQUU7Z0JBQ25ELGVBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsMENBQTBDLEVBQUU7b0JBQ3ZGLFFBQVEsRUFBRSxPQUFPO29CQUNqQixJQUFJLEVBQUUsSUFBSTtpQkFDWCxDQUFDLENBQUMsQ0FBQztZQUNOLENBQUMsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLDRDQUE0QyxFQUFFLEdBQUcsRUFBRTtnQkFDdEQsZUFBUyxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1Q0FBdUMsRUFBRTtvQkFDcEYsUUFBUSxFQUFFLE1BQU07b0JBQ2hCLElBQUksRUFBRSxJQUFJO2lCQUNYLENBQUMsQ0FBQyxDQUFDO1lBQ04sQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILFFBQVEsQ0FBQywrQkFBK0IsRUFBRSxHQUFHLEVBQUU7WUFDN0MsSUFBSSxhQUFvQixDQUFDO1lBQ3pCLElBQUksV0FBd0IsQ0FBQztZQUM3QixJQUFJLGtCQUEwQixDQUFDO1lBQy9CLElBQUksbUJBQTJCLENBQUM7WUFDaEMsSUFBSSw2QkFBcUMsQ0FBQztZQUUxQyxVQUFVLENBQUMsR0FBRyxFQUFFO2dCQUNkLFFBQVE7Z0JBQ1IsYUFBYSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxlQUFlLENBQUMsQ0FBQztnQkFDaEQsTUFBTSxLQUFLLEdBQXFCO29CQUM5QixNQUFNO29CQUNOLFVBQVU7b0JBQ1YsT0FBTyxFQUFFLElBQUksa0JBQVksQ0FBQyxhQUFhLEVBQUUsU0FBUyxDQUFDO29CQUNuRCxHQUFHO29CQUNILGlCQUFpQixFQUFFO3dCQUNqQixnQkFBZ0IsRUFBRSxnREFBbUIsQ0FBQyxLQUFLO3FCQUM1QztpQkFDRixDQUFDO2dCQUVGLE9BQU87Z0JBQ1AsV0FBVyxHQUFHLElBQUksaUJBQVcsQ0FBQyxhQUFhLEVBQUUsYUFBYSxFQUFFLEtBQUssQ0FBQyxDQUFDO2dCQUVuRSxrQkFBa0IsR0FBRyxhQUFhLENBQUMsWUFBWSxDQUM3QyxXQUFXLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxJQUFJLENBQUMsWUFBMEIsQ0FDM0UsQ0FBQztnQkFDRixNQUFNLFVBQVUsR0FBRyxXQUFXLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO2dCQUNsRSxNQUFNLG9CQUFvQixHQUFHLFVBQVUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFlBQVksQ0FBQyxDQUFDO2dCQUNyRSxtQkFBbUIsR0FBRyxhQUFhLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsWUFBMEIsQ0FBQyxDQUFDO2dCQUM3Riw2QkFBNkIsR0FBRyxhQUFhLENBQUMsWUFBWSxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQyxZQUEwQixDQUFDLENBQUM7WUFDbkgsQ0FBQyxDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLEdBQUcsRUFBRTtnQkFDaEQsZUFBUyxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywwQ0FBMEMsRUFBRTtvQkFDdkYsUUFBUSxFQUFFLE9BQU87b0JBQ2pCLElBQUksRUFBRSxJQUFJO2lCQUNYLENBQUMsQ0FBQyxDQUFDO1lBQ04sQ0FBQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsa0JBQWtCLEVBQUUsR0FBRyxFQUFFO2dCQUM1QixlQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDRCQUE0QixFQUFFO29CQUN6RSxZQUFZLEVBQUU7d0JBQ1osWUFBWSxFQUFFLGtCQUFTLENBQUMsS0FBSyxDQUFDO3FCQUMvQjtvQkFDRCxpQkFBaUIsRUFBRSxFQUFFLEVBQUUsRUFBRSxrQkFBa0IsRUFBRTtvQkFDN0MsTUFBTSxFQUFFO3dCQUNOLFVBQVUsRUFBRSx3Q0FBd0M7cUJBQ3JEO2lCQUNGLENBQUMsQ0FBQyxDQUFDO1lBQ04sQ0FBQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsbUVBQW1FLEVBQUUsR0FBRyxFQUFFO2dCQUM3RSxNQUFNLE9BQU8sR0FBRyxXQUFXLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQW1CLENBQUM7Z0JBQ3hFLE1BQU0saUJBQWlCLEdBQUcsYUFBYSxDQUFDLFlBQVksQ0FBRSxPQUFPLENBQUMsUUFBaUIsQ0FBQyxJQUFJLENBQUMsWUFBMEIsQ0FBQyxDQUFDO2dCQUNqSCxlQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGtCQUFrQixFQUFFO29CQUMvRCxjQUFjLEVBQUU7d0JBQ2QsU0FBUyxFQUFFLGtCQUFTLENBQ2xCOzRCQUNFLE1BQU0sRUFBRTtnQ0FDTiwrQkFBK0I7Z0NBQy9CLCtCQUErQjs2QkFDaEM7NEJBQ0QsTUFBTSxFQUFFLE9BQU87NEJBQ2YsUUFBUSxFQUFFO2dDQUNSLFlBQVksRUFBRTtvQ0FDWixrQkFBa0I7b0NBQ2xCLE1BQU07aUNBQ1A7NkJBQ0Y7eUJBQ0YsRUFDRDs0QkFDRSxNQUFNLEVBQUU7Z0NBQ04sK0JBQStCO2dDQUMvQiwrQkFBK0I7NkJBQ2hDOzRCQUNELE1BQU0sRUFBRSxPQUFPOzRCQUNmLFFBQVEsRUFBRTtnQ0FDUixZQUFZLEVBQUU7b0NBQ1osbUJBQW1CO29DQUNuQixNQUFNO2lDQUNQOzZCQUNGO3lCQUNGLEVBQ0Q7NEJBQ0UsTUFBTSxFQUFFO2dDQUNOLCtCQUErQjtnQ0FDL0IsK0JBQStCOzZCQUNoQzs0QkFDRCxNQUFNLEVBQUUsT0FBTzs0QkFDZixRQUFRLEVBQUUsRUFBRSxHQUFHLEVBQUUsNkJBQTZCLEVBQUU7eUJBQ2pELENBQ0Y7d0JBQ0QsT0FBTyxFQUFFLFlBQVk7cUJBQ3RCO29CQUNELEtBQUssRUFBRSxrQkFBUyxDQUFDLEVBQUUsR0FBRyxFQUFFLGlCQUFpQixFQUFFLENBQUM7aUJBQzdDLENBQUMsQ0FBQyxDQUFDO1lBQ04sQ0FBQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsdURBQXVELEVBQUUsR0FBRyxFQUFFO2dCQUNqRSxlQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDBCQUEwQixFQUFFO29CQUN2RSxvQkFBb0IsRUFBRSxrQkFBUyxDQUFDLHVCQUFjLENBQUM7d0JBQzdDLFdBQVcsRUFBRSxrQkFBUyxDQUNwQjs0QkFDRSxJQUFJLEVBQUUscUJBQXFCOzRCQUMzQixLQUFLLEVBQUU7Z0NBQ0wsWUFBWSxFQUFFO29DQUNaLGtCQUFrQjtvQ0FDbEIsTUFBTTtpQ0FDUDs2QkFDRjt5QkFDRixFQUNEOzRCQUNFLElBQUksRUFBRSxrQkFBa0I7NEJBQ3hCLEtBQUssRUFBRTtnQ0FDTCxZQUFZLEVBQUU7b0NBQ1osbUJBQW1CO29DQUNuQixNQUFNO2lDQUNQOzZCQUNGO3lCQUNGLEVBQ0Q7NEJBQ0UsSUFBSSxFQUFFLDZCQUE2Qjs0QkFDbkMsS0FBSyxFQUFFLEVBQUUsR0FBRyxFQUFFLDZCQUE2QixFQUFFO3lCQUM5QyxDQUNGO3FCQUNGLENBQUMsQ0FBQztpQkFDSixDQUFDLENBQUMsQ0FBQztZQUNOLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsZ0NBQWdDLEVBQUUsR0FBRyxFQUFFO1lBQzlDLElBQUksYUFBb0IsQ0FBQztZQUV6QixVQUFVLENBQUMsR0FBRyxFQUFFO2dCQUNkLFFBQVE7Z0JBQ1IsYUFBYSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxlQUFlLENBQUMsQ0FBQztnQkFDaEQsTUFBTSxLQUFLLEdBQXFCO29CQUM5QixNQUFNO29CQUNOLFVBQVU7b0JBQ1YsT0FBTyxFQUFFLElBQUksa0JBQVksQ0FBQyxhQUFhLEVBQUUsU0FBUyxDQUFDO29CQUNuRCxHQUFHO29CQUNILGlCQUFpQixFQUFFO3dCQUNqQixnQkFBZ0IsRUFBRSxnREFBbUIsQ0FBQyxJQUFJO3FCQUMzQztpQkFDRixDQUFDO2dCQUVGLE9BQU87Z0JBQ1AsSUFBSSxpQkFBVyxDQUFDLGFBQWEsRUFBRSxhQUFhLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDdkQsQ0FBQyxDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsSUFBSSxDQUFDLHNCQUFzQixFQUFFLEdBQUcsRUFBRTtnQkFDaEMsZUFBUyxDQUFDLGFBQWEsQ0FBQyxDQUFDLEtBQUssQ0FBQyxxQkFBWSxDQUFDLDRCQUE0QixDQUFDLENBQUMsQ0FBQztZQUM3RSxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxxQ0FBcUMsRUFBRSxHQUFHLEVBQUU7Z0JBQy9DLGVBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsMENBQTBDLEVBQUU7b0JBQ3ZGLFFBQVEsRUFBRSxNQUFNO29CQUNoQixJQUFJLEVBQUUsSUFBSTtpQkFDWCxDQUFDLENBQUMsQ0FBQztZQUNOLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsMkJBQTJCLEVBQUUsR0FBRyxFQUFFO1lBQ3pDLElBQUksYUFBb0IsQ0FBQztZQUN6QixNQUFNLFFBQVEsR0FBRyxTQUFTLENBQUM7WUFDM0IsTUFBTSxNQUFNLEdBQUcsbURBQW1ELENBQUM7WUFDbkUsTUFBTSxTQUFTLEdBQUcsa0JBQWtCLENBQUM7WUFFckMsVUFBVSxDQUFDLEdBQUcsRUFBRTtnQkFDZCxRQUFRO2dCQUNSLGFBQWEsR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsZUFBZSxDQUFDLENBQUM7Z0JBQ2hELE1BQU0sSUFBSSxHQUFHLElBQUksK0JBQWlCLENBQUMsYUFBYSxFQUFFLGlCQUFpQixFQUFFO29CQUNuRSxHQUFHO29CQUNILFFBQVEsRUFBRSxTQUFTO2lCQUNwQixDQUFDLENBQUM7Z0JBQ0gsTUFBTSxLQUFLLEdBQXFCO29CQUM5QixNQUFNO29CQUNOLFVBQVU7b0JBQ1YsT0FBTyxFQUFFLElBQUksa0JBQVksQ0FBQyxhQUFhLEVBQUUsU0FBUyxDQUFDO29CQUNuRCxHQUFHO29CQUNILGlCQUFpQixFQUFFO3dCQUNqQixXQUFXLEVBQUU7NEJBQ1gsY0FBYyxFQUFFLG9DQUFXLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRSxRQUFRLENBQUM7NEJBQzlFLG1CQUFtQixFQUFFLDJCQUFNLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUUsTUFBTSxDQUFDO3lCQUNwRTtxQkFDRjtvQkFDRCxRQUFRLEVBQUU7d0JBQ1IsSUFBSTtxQkFDTDtpQkFDRixDQUFDO2dCQUVGLE9BQU87Z0JBQ1AsSUFBSSxpQkFBVyxDQUFDLGFBQWEsRUFBRSxhQUFhLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDdkQsQ0FBQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsZ0NBQWdDLEVBQUUsR0FBRyxFQUFFO2dCQUMxQyxPQUFPO2dCQUNQLGVBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsdUNBQXVDLEVBQUU7b0JBQ3BGLElBQUksRUFBRSxJQUFJO2lCQUNYLENBQUMsQ0FBQyxDQUFDO1lBQ04sQ0FBQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMscUNBQXFDLEVBQUUsR0FBRyxFQUFFO2dCQUMvQyxPQUFPO2dCQUNQLGVBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsdUNBQXVDLEVBQUU7b0JBQ3BGLFFBQVEsRUFBRSxPQUFPO2lCQUNsQixDQUFDLENBQUMsQ0FBQztZQUNOLENBQUMsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLGtFQUFrRSxFQUFFLEdBQUcsRUFBRTtnQkFDNUUsZUFBUyxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1Q0FBdUMsRUFBRTtvQkFDcEYsUUFBUSxFQUFFLE9BQU87b0JBQ2pCLFlBQVksRUFBRSxrQkFBUyxDQUFDO3dCQUN0QixjQUFjLEVBQUUsUUFBUTtxQkFDekIsQ0FBQztpQkFDSCxDQUFDLENBQUMsQ0FBQztZQUNOLENBQUMsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLDZEQUE2RCxFQUFFLEdBQUcsRUFBRTtnQkFDdkUsUUFBUTtnQkFDUixNQUFNLEtBQUssR0FBcUI7b0JBQzlCLE1BQU07b0JBQ04sVUFBVTtvQkFDVixPQUFPLEVBQUUsa0JBQWtCO29CQUMzQixHQUFHO29CQUNILGlCQUFpQixFQUFFO3dCQUNqQixXQUFXLEVBQUU7NEJBQ1gsY0FBYyxFQUFFLG9DQUFXLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxTQUFTLENBQUM7NEJBQ3hFLG1CQUFtQixFQUFFLDJCQUFNLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsTUFBTSxDQUFDO3lCQUNyRTtxQkFDRjtpQkFDRixDQUFDO2dCQUVGLE9BQU87Z0JBQ1AsTUFBTSxDQUFDLEdBQUcsRUFBRTtvQkFDVixJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRSxLQUFLLENBQUMsQ0FBQztnQkFDL0MsQ0FBQyxDQUFDO29CQUNBLE9BQU87cUJBQ04sT0FBTyxDQUFDLGlFQUFpRSxDQUFDLENBQUM7WUFDaEYsQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILFFBQVEsQ0FBQywwQ0FBMEMsRUFBRSxHQUFHLEVBQUU7WUFDeEQsSUFBSSxhQUFvQixDQUFDO1lBQ3pCLElBQUksSUFBdUIsQ0FBQztZQUM1QixNQUFNLFNBQVMsR0FBRyxrQkFBa0IsQ0FBQztZQUVyQyxVQUFVLENBQUMsR0FBRyxFQUFFO2dCQUNkLFFBQVE7Z0JBQ1IsYUFBYSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxlQUFlLENBQUMsQ0FBQztnQkFDaEQsSUFBSSxHQUFHLElBQUksK0JBQWlCLENBQUMsYUFBYSxFQUFFLGlCQUFpQixFQUFFO29CQUM3RCxHQUFHO29CQUNILFFBQVEsRUFBRSxTQUFTO2lCQUNwQixDQUFDLENBQUM7Z0JBRUgsTUFBTSxNQUFNLEdBQUcsSUFBSSxzQkFBa0IsQ0FBQyxhQUFhLEVBQUUsUUFBUSxFQUFFO29CQUM3RCxPQUFPLEVBQUU7d0JBQ1AsRUFBRSxFQUFFLE1BQU0sU0FBUyxFQUFFO3FCQUN0QjtpQkFDRixDQUFDLENBQUM7Z0JBQ0gsTUFBTSxVQUFVLEdBQUcsSUFBSSxzQkFBa0IsQ0FBQyxhQUFhLEVBQUUsWUFBWSxFQUFFO29CQUNyRSxPQUFPLEVBQUU7d0JBQ1AsRUFBRSxFQUFFLFVBQVUsU0FBUyxFQUFFO3FCQUMxQjtvQkFDRCxrQkFBa0IsRUFBRSxNQUFNO2lCQUMzQixDQUFDLENBQUM7Z0JBRUgsTUFBTSxLQUFLLEdBQXFCO29CQUM5QixNQUFNO29CQUNOLFVBQVU7b0JBQ1YsT0FBTyxFQUFFLElBQUksa0JBQVksQ0FBQyxhQUFhLEVBQUUsU0FBUyxDQUFDO29CQUNuRCxHQUFHO29CQUNILGlCQUFpQixFQUFFO3dCQUNqQixXQUFXLEVBQUU7NEJBQ1gsZUFBZSxFQUFFLFVBQVU7eUJBQzVCO3dCQUNELGdCQUFnQixFQUFFLGdEQUFtQixDQUFDLElBQUk7cUJBQzNDO29CQUNELFFBQVEsRUFBRTt3QkFDUixJQUFJO3FCQUNMO2lCQUNGLENBQUM7Z0JBRUYsT0FBTztnQkFDUCxJQUFJLGlCQUFXLENBQUMsYUFBYSxFQUFFLGFBQWEsRUFBRSxLQUFLLENBQUMsQ0FBQztZQUN2RCxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxnQ0FBZ0MsRUFBRSxHQUFHLEVBQUU7Z0JBQzFDLE9BQU87Z0JBQ1AsZUFBUyxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1Q0FBdUMsRUFBRTtvQkFDcEYsSUFBSSxFQUFFLElBQUk7aUJBQ1gsQ0FBQyxDQUFDLENBQUM7WUFDTixDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxxQ0FBcUMsRUFBRSxHQUFHLEVBQUU7Z0JBQy9DLE9BQU87Z0JBQ1AsZUFBUyxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1Q0FBdUMsRUFBRTtvQkFDcEYsUUFBUSxFQUFFLE9BQU87aUJBQ2xCLENBQUMsQ0FBQyxDQUFDO1lBQ04sQ0FBQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMscUJBQXFCLEVBQUUsR0FBRyxFQUFFO2dCQUMvQixPQUFPO2dCQUNQLGVBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMscUNBQXFDLEVBQUU7b0JBQ2xGLGtCQUFrQixFQUFFO3dCQUNsQixJQUFJLEVBQUU7NEJBQ0osWUFBWSxFQUFFO2dDQUNaLFlBQVk7Z0NBQ1osTUFBTTs2QkFDUDt5QkFDRjt3QkFDRCxHQUFHLEVBQUU7NEJBQ0gsWUFBWSxFQUFFO2dDQUNaLFlBQVk7Z0NBQ1osS0FBSzs2QkFDTjt5QkFDRjt3QkFDRCxVQUFVLEVBQUU7NEJBQ1YsR0FBRyxFQUFFLDhCQUE4Qjt5QkFDcEM7d0JBQ0QsU0FBUyxFQUFFOzRCQUNULFlBQVksRUFBRTtnQ0FDWixZQUFZO2dDQUNaLFdBQVc7NkJBQ1o7eUJBQ0Y7cUJBQ0Y7aUJBQ0YsQ0FBQyxDQUFDLENBQUM7WUFDTixDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHdDQUF3QyxFQUFFLEdBQUcsRUFBRTtZQUNsRCxRQUFRO1lBQ1IsTUFBTSxhQUFhLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLGVBQWUsQ0FBQyxDQUFDO1lBQ3RELE1BQU0sU0FBUyxHQUFHLGtCQUFrQixDQUFDO1lBQ3JDLE1BQU0sUUFBUSxHQUFHLFNBQVMsQ0FBQztZQUMzQixNQUFNLE1BQU0sR0FBRyxtREFBbUQsQ0FBQztZQUVuRSxNQUFNLElBQUksR0FBRyxJQUFJLCtCQUFpQixDQUFDLGFBQWEsRUFBRSxpQkFBaUIsRUFBRTtnQkFDbkUsR0FBRztnQkFDSCxRQUFRLEVBQUUsU0FBUzthQUNwQixDQUFDLENBQUM7WUFFSCxNQUFNLE1BQU0sR0FBRyxJQUFJLHNCQUFrQixDQUFDLGFBQWEsRUFBRSxRQUFRLEVBQUU7Z0JBQzdELE9BQU8sRUFBRTtvQkFDUCxFQUFFLEVBQUUsTUFBTSxTQUFTLEVBQUU7aUJBQ3RCO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsTUFBTSxVQUFVLEdBQUcsSUFBSSxzQkFBa0IsQ0FBQyxhQUFhLEVBQUUsWUFBWSxFQUFFO2dCQUNyRSxPQUFPLEVBQUU7b0JBQ1AsRUFBRSxFQUFFLFVBQVUsU0FBUyxFQUFFO2lCQUMxQjtnQkFDRCxrQkFBa0IsRUFBRSxNQUFNO2FBQzNCLENBQUMsQ0FBQztZQUVILE1BQU0sS0FBSyxHQUFxQjtnQkFDOUIsTUFBTTtnQkFDTixVQUFVO2dCQUNWLE9BQU8sRUFBRSxJQUFJLGtCQUFZLENBQUMsYUFBYSxFQUFFLFNBQVMsQ0FBQztnQkFDbkQsR0FBRztnQkFDSCxpQkFBaUIsRUFBRTtvQkFDakIsV0FBVyxFQUFFO3dCQUNYLGNBQWMsRUFBRSxvQ0FBVyxDQUFDLGtCQUFrQixDQUFDLGFBQWEsRUFBRSxhQUFhLEVBQUUsUUFBUSxDQUFDO3dCQUN0RixtQkFBbUIsRUFBRSwyQkFBTSxDQUFDLGFBQWEsQ0FBQyxhQUFhLEVBQUUsU0FBUyxFQUFFLE1BQU0sQ0FBQzt3QkFDM0UsZUFBZSxFQUFFLFVBQVU7cUJBQzVCO2lCQUNGO2dCQUNELFFBQVEsRUFBRTtvQkFDUixJQUFJO2lCQUNMO2FBQ0YsQ0FBQztZQUVGLE9BQU87WUFDUCxNQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNWLElBQUksaUJBQVcsQ0FBQyxhQUFhLEVBQUUsYUFBYSxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQ3ZELENBQUMsQ0FBQztnQkFDQSxPQUFPO2lCQUNOLE9BQU8sQ0FBQyxtSEFBbUgsQ0FBQyxDQUFDO1FBQ2xJLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHlCQUF5QixFQUFFLEdBQUcsRUFBRTtZQUNuQyxRQUFRO1lBQ1IsTUFBTSxhQUFhLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLGVBQWUsQ0FBQyxDQUFDO1lBQ3RELE1BQU0sU0FBUyxHQUFHLGtCQUFrQixDQUFDO1lBRXJDLE1BQU0sSUFBSSxHQUFHLElBQUksK0JBQWlCLENBQUMsYUFBYSxFQUFFLGlCQUFpQixFQUFFO2dCQUNuRSxHQUFHO2dCQUNILFFBQVEsRUFBRSxTQUFTO2FBQ3BCLENBQUMsQ0FBQztZQUVILE1BQU0sS0FBSyxHQUFxQjtnQkFDOUIsTUFBTTtnQkFDTixVQUFVO2dCQUNWLE9BQU8sRUFBRSxJQUFJLGtCQUFZLENBQUMsYUFBYSxFQUFFLFNBQVMsQ0FBQztnQkFDbkQsR0FBRztnQkFDSCxpQkFBaUIsRUFBRTtvQkFDakIsV0FBVyxFQUFFLEVBQ1o7aUJBQ0Y7Z0JBQ0QsUUFBUSxFQUFFO29CQUNSLElBQUk7aUJBQ0w7YUFDRixDQUFDO1lBRUYsT0FBTztZQUNQLE1BQU0sQ0FBQyxHQUFHLEVBQUU7Z0JBQ1YsSUFBSSxpQkFBVyxDQUFDLGFBQWEsRUFBRSxhQUFhLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDdkQsQ0FBQyxDQUFDO2dCQUNBLE9BQU87aUJBQ04sT0FBTyxDQUFDLG1IQUFtSCxDQUFDLENBQUM7UUFDbEksQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsa0RBQWtELEVBQUUsR0FBRyxFQUFFO1lBQzVELFFBQVE7WUFDUixNQUFNLGFBQWEsR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsZUFBZSxDQUFDLENBQUM7WUFDdEQsTUFBTSxTQUFTLEdBQUcsa0JBQWtCLENBQUM7WUFDckMsTUFBTSxRQUFRLEdBQUcsU0FBUyxDQUFDO1lBRTNCLE1BQU0sSUFBSSxHQUFHLElBQUksK0JBQWlCLENBQUMsYUFBYSxFQUFFLGlCQUFpQixFQUFFO2dCQUNuRSxHQUFHO2dCQUNILFFBQVEsRUFBRSxTQUFTO2FBQ3BCLENBQUMsQ0FBQztZQUVILE1BQU0sS0FBSyxHQUFxQjtnQkFDOUIsTUFBTTtnQkFDTixVQUFVO2dCQUNWLE9BQU8sRUFBRSxJQUFJLGtCQUFZLENBQUMsYUFBYSxFQUFFLFNBQVMsQ0FBQztnQkFDbkQsR0FBRztnQkFDSCxpQkFBaUIsRUFBRTtvQkFDakIsV0FBVyxFQUFFO3dCQUNYLGNBQWMsRUFBRSxvQ0FBVyxDQUFDLGtCQUFrQixDQUFDLGFBQWEsRUFBRSxhQUFhLEVBQUUsUUFBUSxDQUFDO3FCQUN2RjtpQkFDRjtnQkFDRCxRQUFRLEVBQUU7b0JBQ1IsSUFBSTtpQkFDTDthQUNGLENBQUM7WUFFRixPQUFPO1lBQ1AsTUFBTSxDQUFDLEdBQUcsRUFBRTtnQkFDVixJQUFJLGlCQUFXLENBQUMsYUFBYSxFQUFFLGFBQWEsRUFBRSxLQUFLLENBQUMsQ0FBQztZQUN2RCxDQUFDLENBQUM7Z0JBQ0EsT0FBTztpQkFDTixPQUFPLENBQUMseUZBQXlGLENBQUMsQ0FBQztRQUN4RyxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLG1CQUFtQixFQUFFLEdBQUcsRUFBRTtRQUNqQyxRQUFRLENBQUMsMEJBQTBCLEVBQUUsR0FBRyxFQUFFO1lBQ3hDLElBQUksYUFBb0IsQ0FBQztZQUN6QixJQUFJLElBQXVCLENBQUM7WUFDNUIsTUFBTSxTQUFTLEdBQUcsa0JBQWtCLENBQUM7WUFDckMsSUFBSSxFQUFlLENBQUM7WUFFcEIsVUFBVSxDQUFDLEdBQUcsRUFBRTtnQkFDZCxRQUFRO2dCQUNSLGFBQWEsR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsZUFBZSxDQUFDLENBQUM7Z0JBQ2hELElBQUksR0FBRyxJQUFJLCtCQUFpQixDQUFDLGFBQWEsRUFBRSxpQkFBaUIsRUFBRTtvQkFDN0QsR0FBRztvQkFDSCxRQUFRLEVBQUUsU0FBUztpQkFDcEIsQ0FBQyxDQUFDO2dCQUNILE1BQU0sS0FBSyxHQUFxQjtvQkFDOUIsTUFBTTtvQkFDTixVQUFVO29CQUNWLE9BQU8sRUFBRSxJQUFJLGtCQUFZLENBQUMsYUFBYSxFQUFFLFNBQVMsQ0FBQztvQkFDbkQsR0FBRztvQkFDSCxRQUFRLEVBQUU7d0JBQ1IsSUFBSTtxQkFDTDtpQkFDRixDQUFDO2dCQUVGLE9BQU87Z0JBQ1AsRUFBRSxHQUFHLElBQUksaUJBQVcsQ0FBQyxhQUFhLEVBQUUsYUFBYSxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQzVELENBQUMsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLGlCQUFpQixFQUFFLEdBQUcsRUFBRTtnQkFDM0IsT0FBTztnQkFDUCxNQUFNLEtBQUssR0FBRyxDQUFDLElBQUksa0JBQVEsQ0FBQyxhQUFhLEVBQUUsTUFBTSxFQUFFO3dCQUNqRCxHQUFHO3dCQUNILFlBQVksRUFBRSxzQkFBWSxDQUFDLEVBQUUsQ0FDM0IsdUJBQWEsQ0FBQyxFQUFFLEVBQ2hCLHNCQUFZLENBQUMsS0FBSyxDQUNuQjt3QkFDRCxZQUFZLEVBQUUsc0JBQVksQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLFVBQVUsRUFBRSwrQkFBcUIsQ0FBQyxjQUFjLEVBQUUsQ0FBQztxQkFDbkcsQ0FBQyxDQUFDLENBQUM7Z0JBQ0osTUFBTSxJQUFJLEdBQUcsSUFBSSxjQUFJLENBQUMsYUFBYSxFQUFFLE1BQU0sRUFBRSxFQUFDLFNBQVMsRUFBRSxJQUFJLDhCQUFvQixFQUFFLEVBQUMsQ0FBQyxDQUFDO2dCQUV0RixNQUFNLEdBQUcsR0FBRyxFQUFFLENBQUMsa0JBQWtCLENBQUM7b0JBQ2hDLEtBQUs7b0JBQ0wsT0FBTyxFQUFFLElBQUk7aUJBQ2QsQ0FBQyxDQUFDO2dCQUVILE9BQU87Z0JBQ1AsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO2dCQUMvQyxNQUFNLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDLENBQUMsT0FBTyxDQUFDLG1CQUFtQixDQUFDLENBQUM7Z0JBRTFELGVBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsZ0NBQWdDLEVBQUU7b0JBQzdFLFVBQVUsRUFBRSxLQUFLO29CQUNqQixNQUFNLEVBQUUsSUFBSTtvQkFDWixxQkFBcUIsRUFBRTt3QkFDckIsWUFBWSxFQUFFOzRCQUNaLGFBQWEsQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFlBQTBCLENBQUM7NEJBQ2xHLFNBQVM7eUJBQ1Y7cUJBQ0Y7aUJBQ0YsQ0FBQyxDQUFDLENBQUM7Z0JBRUosZUFBUyxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxvQkFBb0IsRUFBRTtvQkFDakUsU0FBUyxFQUFFLGtCQUFTLENBQ2xCLHFDQUFxQyxFQUNyQyw0QkFBNEIsQ0FDN0I7aUJBQ0YsRUFBRSxxQkFBWSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQztZQUN2QyxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyw0QkFBNEIsRUFBRSxHQUFHLEVBQUU7Z0JBQ3RDLE9BQU87Z0JBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxrQkFBUSxDQUFDLGFBQWEsRUFBRSxNQUFNLEVBQUU7b0JBQy9DLEdBQUc7b0JBQ0gsWUFBWSxFQUFFLHNCQUFZLENBQUMsRUFBRSxDQUMzQix1QkFBYSxDQUFDLEVBQUUsRUFDaEIsc0JBQVksQ0FBQyxLQUFLLENBQ25CO29CQUNELFlBQVksRUFBRSxzQkFBWSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsVUFBVSxFQUFFLCtCQUFxQixDQUFDLGNBQWMsRUFBRSxDQUFDO2lCQUNuRyxDQUFDLENBQUM7Z0JBRUgsRUFBRSxDQUFDLHVCQUF1QixDQUFDO29CQUN6QixJQUFJO2lCQUNMLENBQUMsQ0FBQztnQkFFSCxPQUFPO2dCQUNQLE1BQU0sUUFBUSxHQUFHLGFBQWEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO2dCQUMvRCxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsYUFBYSxDQUFDO29CQUM3QixVQUFVLEVBQUU7d0JBQ1YsRUFBRTt3QkFDRjs0QkFDRSx3Q0FBd0M7NEJBQ3hDO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLEdBQUcsRUFBRTt5Q0FDakM7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0Q7Z0NBQ0UsWUFBWSxFQUFFO29DQUNaLENBQUM7b0NBQ0Q7d0NBQ0UsV0FBVyxFQUFFOzRDQUNYLElBQUk7NENBQ0osRUFBRSxHQUFHLEVBQUUscUNBQW1CLENBQUMsR0FBRyxFQUFFO3lDQUNqQztxQ0FDRjtpQ0FDRjs2QkFDRjs0QkFDRCx3QkFBd0I7NEJBQ3hCLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLE1BQU0sRUFBRTs0QkFDbkMsR0FBRzs0QkFDSDtnQ0FDRSxZQUFZLEVBQUU7b0NBQ1osQ0FBQztvQ0FDRDt3Q0FDRSxXQUFXLEVBQUU7NENBQ1gsSUFBSTs0Q0FDSixFQUFFLEdBQUcsRUFBRSxxQ0FBbUIsQ0FBQyxHQUFHLEVBQUU7eUNBQ2pDO3FDQUNGO2lDQUNGOzZCQUNGOzRCQUNEO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLEdBQUcsRUFBRTt5Q0FDakM7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0QsWUFBWTs0QkFDWjtnQ0FDRSxZQUFZLEVBQUU7b0NBQ1osQ0FBQztvQ0FDRDt3Q0FDRSxXQUFXLEVBQUU7NENBQ1gsSUFBSTs0Q0FDSixFQUFFLEdBQUcsRUFBRSxxQ0FBbUIsQ0FBQyxHQUFHLEVBQUU7eUNBQ2pDO3FDQUNGO2lDQUNGOzZCQUNGOzRCQUNEO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLEdBQUcsRUFBRTt5Q0FDakM7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0QsTUFBTTtnQ0FDTix3REFBd0Q7Z0NBQ3hELGlEQUFpRDtnQ0FDakQsTUFBTTtnQ0FDTiwrREFBK0Q7NEJBQy9EO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLEdBQUcsRUFBRTt5Q0FDakM7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0Q7Z0NBQ0UsWUFBWSxFQUFFO29DQUNaLENBQUM7b0NBQ0Q7d0NBQ0UsV0FBVyxFQUFFOzRDQUNYLElBQUk7NENBQ0osRUFBRSxHQUFHLEVBQUUscUNBQW1CLENBQUMsR0FBRyxFQUFFO3lDQUNqQztxQ0FDRjtpQ0FDRjs2QkFDRjs0QkFDRCx3Q0FBd0MsU0FBUyxXQUFXO2dDQUM1RCxjQUFjOzRCQUNkO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLEdBQUcsRUFBRTt5Q0FDakM7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0Q7Z0NBQ0UsWUFBWSxFQUFFO29DQUNaLENBQUM7b0NBQ0Q7d0NBQ0UsV0FBVyxFQUFFOzRDQUNYLElBQUk7NENBQ0osRUFBRSxHQUFHLEVBQUUscUNBQW1CLENBQUMsR0FBRyxFQUFFO3lDQUNqQztxQ0FDRjtpQ0FDRjs2QkFDRjs0QkFDRCxLQUFLO2dDQUNMLGtFQUFrRTtnQ0FDbEUsd0NBQXdDO2dDQUN4QyxJQUFJO3lCQUNMO3FCQUNGO2lCQUNGLENBQUMsQ0FBQztnQkFFSCx3REFBd0Q7Z0JBQ3hELGVBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsZ0NBQWdDLEVBQUU7b0JBQzdFLFVBQVUsRUFBRSxLQUFLO29CQUNqQixNQUFNLEVBQUUsSUFBSTtvQkFDWixxQkFBcUIsRUFBRTt3QkFDckIsWUFBWSxFQUFFOzRCQUNaLGFBQWEsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFlBQTBCLENBQUM7NEJBQzlGLFNBQVM7eUJBQ1Y7cUJBQ0Y7aUJBQ0YsQ0FBQyxDQUFDLENBQUM7Z0JBRUosZUFBUyxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxvQkFBb0IsRUFBRTtvQkFDakUsU0FBUyxFQUFFLGtCQUFTLENBQ2xCLHFDQUFxQyxFQUNyQyw0QkFBNEIsQ0FDN0I7aUJBQ0YsRUFBRSxxQkFBWSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQztZQUN2QyxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyw4QkFBOEIsRUFBRSxHQUFHLEVBQUU7Z0JBQ3hDLE9BQU87Z0JBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxrQkFBUSxDQUFDLGFBQWEsRUFBRSxNQUFNLEVBQUU7b0JBQy9DLEdBQUc7b0JBQ0gsWUFBWSxFQUFFLHNCQUFZLENBQUMsRUFBRSxDQUMzQix1QkFBYSxDQUFDLEVBQUUsRUFDaEIsc0JBQVksQ0FBQyxLQUFLLENBQ25CO29CQUNELFlBQVksRUFBRSxzQkFBWSxDQUFDLGFBQWEsQ0FBRSx3QkFBYyxDQUFDLHFDQUFxQyxDQUFDO2lCQUNoRyxDQUFDLENBQUM7Z0JBRUgsRUFBRSxDQUFDLHVCQUF1QixDQUFDO29CQUN6QixJQUFJO2lCQUNMLENBQUMsQ0FBQztnQkFFSCxPQUFPO2dCQUNQLE1BQU0sUUFBUSxHQUFHLGFBQWEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO2dCQUMvRCxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsYUFBYSxDQUFDO29CQUM3QixVQUFVLEVBQUU7d0JBQ1YsRUFBRTt3QkFDRjs0QkFDRSxnREFBZ0Q7NEJBQ2hEO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLEdBQUcsRUFBRTt5Q0FDakM7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0Q7Z0NBQ0UsWUFBWSxFQUFFO29DQUNaLENBQUM7b0NBQ0Q7d0NBQ0UsV0FBVyxFQUFFOzRDQUNYLElBQUk7NENBQ0osRUFBRSxHQUFHLEVBQUUscUNBQW1CLENBQUMsR0FBRyxFQUFFO3lDQUNqQztxQ0FDRjtpQ0FDRjs2QkFDRjs0QkFDRCxjQUFjO2dDQUNkLDhCQUE4Qjs0QkFDOUIsRUFBRSxHQUFHLEVBQUUscUNBQW1CLENBQUMsTUFBTSxFQUFFOzRCQUNuQyxZQUFZOzRCQUNaO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLEdBQUcsRUFBRTt5Q0FDakM7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0Q7Z0NBQ0UsWUFBWSxFQUFFO29DQUNaLENBQUM7b0NBQ0Q7d0NBQ0UsV0FBVyxFQUFFOzRDQUNYLElBQUk7NENBQ0osRUFBRSxHQUFHLEVBQUUscUNBQW1CLENBQUMsR0FBRyxFQUFFO3lDQUNqQztxQ0FDRjtpQ0FDRjs2QkFDRjs0QkFDRCxxQkFBcUI7NEJBQ3JCO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLEdBQUcsRUFBRTt5Q0FDakM7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0Q7Z0NBQ0UsWUFBWSxFQUFFO29DQUNaLENBQUM7b0NBQ0Q7d0NBQ0UsV0FBVyxFQUFFOzRDQUNYLElBQUk7NENBQ0osRUFBRSxHQUFHLEVBQUUscUNBQW1CLENBQUMsR0FBRyxFQUFFO3lDQUNqQztxQ0FDRjtpQ0FDRjs2QkFDRjs0QkFDRCx3QkFBd0I7Z0NBQ3hCLG1DQUFtQztnQ0FDbkMseURBQXlEO2dDQUN6RCxzRUFBc0U7NEJBQ3RFO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLEdBQUcsRUFBRTt5Q0FDakM7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0Q7Z0NBQ0UsWUFBWSxFQUFFO29DQUNaLENBQUM7b0NBQ0Q7d0NBQ0UsV0FBVyxFQUFFOzRDQUNYLElBQUk7NENBQ0osRUFBRSxHQUFHLEVBQUUscUNBQW1CLENBQUMsR0FBRyxFQUFFO3lDQUNqQztxQ0FDRjtpQ0FDRjs2QkFDRjs0QkFDRCx3Q0FBd0MsU0FBUyxnQkFBZ0I7Z0NBQ2pFLDZCQUE2Qjs0QkFDN0I7Z0NBQ0UsWUFBWSxFQUFFO29DQUNaLENBQUM7b0NBQ0Q7d0NBQ0UsV0FBVyxFQUFFOzRDQUNYLElBQUk7NENBQ0osRUFBRSxHQUFHLEVBQUUscUNBQW1CLENBQUMsR0FBRyxFQUFFO3lDQUNqQztxQ0FDRjtpQ0FDRjs2QkFDRjs0QkFDRDtnQ0FDRSxZQUFZLEVBQUU7b0NBQ1osQ0FBQztvQ0FDRDt3Q0FDRSxXQUFXLEVBQUU7NENBQ1gsSUFBSTs0Q0FDSixFQUFFLEdBQUcsRUFBRSxxQ0FBbUIsQ0FBQyxHQUFHLEVBQUU7eUNBQ2pDO3FDQUNGO2lDQUNGOzZCQUNGOzRCQUNELEtBQUs7Z0NBQ0wsZ0ZBQWdGO2dDQUNoRixpREFBaUQ7Z0NBQ2pELFlBQVk7Z0NBQ1osK0RBQStEO2dDQUMvRCx5REFBeUQ7Z0NBQ3pELGdCQUFnQjt5QkFDakI7cUJBQ0Y7aUJBQ0YsQ0FBQyxDQUFDO2dCQUVILHdEQUF3RDtnQkFDeEQsZUFBUyxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxnQ0FBZ0MsRUFBRTtvQkFDN0UsVUFBVSxFQUFFLEtBQUs7b0JBQ2pCLE1BQU0sRUFBRSxJQUFJO29CQUNaLHFCQUFxQixFQUFFO3dCQUNyQixZQUFZLEVBQUU7NEJBQ1osYUFBYSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsWUFBMEIsQ0FBQzs0QkFDOUYsU0FBUzt5QkFDVjtxQkFDRjtpQkFDRixDQUFDLENBQUMsQ0FBQztnQkFFSixlQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLG9CQUFvQixFQUFFO29CQUNqRSxTQUFTLEVBQUUsa0JBQVMsQ0FDbEIscUNBQXFDLEVBQ3JDLDRCQUE0QixDQUM3QjtpQkFDRixFQUFFLHFCQUFZLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO1lBQ3ZDLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsMkJBQTJCLEVBQUUsR0FBRyxFQUFFO1lBQ3pDLElBQUksYUFBb0IsQ0FBQztZQUN6QixJQUFJLElBQXVCLENBQUM7WUFDNUIsSUFBSSxFQUFlLENBQUM7WUFDcEIsTUFBTSxTQUFTLEdBQUcsa0JBQWtCLENBQUM7WUFDckMsTUFBTSxRQUFRLEdBQUcsb0JBQW9CLENBQUM7WUFDdEMsTUFBTSxNQUFNLEdBQUcsbURBQW1ELENBQUM7WUFFbkUsVUFBVSxDQUFDLEdBQUcsRUFBRTtnQkFDZCxRQUFRO2dCQUNSLGFBQWEsR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsZUFBZSxDQUFDLENBQUM7Z0JBQ2hELElBQUksR0FBRyxJQUFJLCtCQUFpQixDQUFDLGFBQWEsRUFBRSxpQkFBaUIsRUFBRTtvQkFDN0QsR0FBRztvQkFDSCxRQUFRLEVBQUUsU0FBUztpQkFDcEIsQ0FBQyxDQUFDO2dCQUNILE1BQU0sS0FBSyxHQUFxQjtvQkFDOUIsTUFBTTtvQkFDTixVQUFVO29CQUNWLE9BQU8sRUFBRSxJQUFJLGtCQUFZLENBQUMsYUFBYSxFQUFFLFNBQVMsQ0FBQztvQkFDbkQsR0FBRztvQkFDSCxRQUFRLEVBQUU7d0JBQ1IsSUFBSTtxQkFDTDtvQkFDRCxpQkFBaUIsRUFBRTt3QkFDakIsV0FBVyxFQUFFOzRCQUNYLGNBQWMsRUFBRSxvQ0FBVyxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxhQUFhLEVBQUUsUUFBUSxDQUFDOzRCQUM5RSxtQkFBbUIsRUFBRSwyQkFBTSxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFLE1BQU0sQ0FBQzt5QkFDcEU7cUJBQ0Y7aUJBQ0YsQ0FBQztnQkFFRixPQUFPO2dCQUNQLEVBQUUsR0FBRyxJQUFJLGlCQUFXLENBQUMsYUFBYSxFQUFFLGFBQWEsRUFBRSxLQUFLLENBQUMsQ0FBQztZQUM1RCxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxpQkFBaUIsRUFBRSxHQUFHLEVBQUU7Z0JBQzNCLE9BQU87Z0JBQ1AsTUFBTSxLQUFLLEdBQUcsQ0FBQyxJQUFJLGtCQUFRLENBQUMsYUFBYSxFQUFFLE1BQU0sRUFBRTt3QkFDakQsR0FBRzt3QkFDSCxZQUFZLEVBQUUsc0JBQVksQ0FBQyxFQUFFLENBQzNCLHVCQUFhLENBQUMsRUFBRSxFQUNoQixzQkFBWSxDQUFDLEtBQUssQ0FDbkI7d0JBQ0QsWUFBWSxFQUFFLHNCQUFZLENBQUMsaUJBQWlCLENBQUMsRUFBRSxVQUFVLEVBQUUsK0JBQXFCLENBQUMsY0FBYyxFQUFFLENBQUM7cUJBQ25HLENBQUMsQ0FBQyxDQUFDO2dCQUNKLE1BQU0sSUFBSSxHQUFHLElBQUksY0FBSSxDQUFDLGFBQWEsRUFBRSxNQUFNLEVBQUUsRUFBQyxTQUFTLEVBQUUsSUFBSSw4QkFBb0IsRUFBRSxFQUFDLENBQUMsQ0FBQztnQkFFdEYsTUFBTSxHQUFHLEdBQUcsRUFBRSxDQUFDLGtCQUFrQixDQUFDO29CQUNoQyxLQUFLO29CQUNMLE9BQU8sRUFBRSxJQUFJO2lCQUNkLENBQUMsQ0FBQztnQkFFSCxPQUFPO2dCQUNQLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxjQUFjLENBQUMsa0JBQWtCLENBQUMsQ0FBQztnQkFDL0MsTUFBTSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO2dCQUMzRCxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsY0FBYyxDQUFDLDhCQUE4QixFQUFFLE1BQU0sQ0FBQyxDQUFDO2dCQUVuRSxlQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdDQUFnQyxFQUFFO29CQUM3RSxVQUFVLEVBQUUsS0FBSztvQkFDakIsTUFBTSxFQUFFLElBQUk7b0JBQ1oscUJBQXFCLEVBQUU7d0JBQ3JCLFlBQVksRUFBRTs0QkFDWixhQUFhLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxZQUEwQixDQUFDOzRCQUNsRyxTQUFTO3lCQUNWO3FCQUNGO2lCQUNGLENBQUMsQ0FBQyxDQUFDO1lBQ04sQ0FBQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsNEJBQTRCLEVBQUUsR0FBRyxFQUFFO2dCQUN0QyxPQUFPO2dCQUNQLE1BQU0sSUFBSSxHQUFHLElBQUksa0JBQVEsQ0FBQyxhQUFhLEVBQUUsTUFBTSxFQUFFO29CQUMvQyxHQUFHO29CQUNILFlBQVksRUFBRSxzQkFBWSxDQUFDLEVBQUUsQ0FDM0IsdUJBQWEsQ0FBQyxFQUFFLEVBQ2hCLHNCQUFZLENBQUMsS0FBSyxDQUNuQjtvQkFDRCxZQUFZLEVBQUUsc0JBQVksQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLFVBQVUsRUFBRSwrQkFBcUIsQ0FBQyxjQUFjLEVBQUUsQ0FBQztpQkFDbkcsQ0FBQyxDQUFDO2dCQUVILEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQztvQkFDekIsSUFBSTtpQkFDTCxDQUFDLENBQUM7Z0JBRUgsT0FBTztnQkFDUCxNQUFNLFFBQVEsR0FBRyxhQUFhLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQztnQkFDL0QsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLGFBQWEsQ0FBQztvQkFDN0IsVUFBVSxFQUFFO3dCQUNWLEVBQUU7d0JBQ0Y7NEJBQ0Usd0NBQXdDOzRCQUN4QztnQ0FDRSxZQUFZLEVBQUU7b0NBQ1osQ0FBQztvQ0FDRDt3Q0FDRSxXQUFXLEVBQUU7NENBQ1gsSUFBSTs0Q0FDSixFQUFFLEdBQUcsRUFBRSxxQ0FBbUIsQ0FBQyxHQUFHLEVBQUU7eUNBQ2pDO3FDQUNGO2lDQUNGOzZCQUNGOzRCQUNEO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLEdBQUcsRUFBRTt5Q0FDakM7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0Qsd0JBQXdCOzRCQUN4QixFQUFFLEdBQUcsRUFBRSxxQ0FBbUIsQ0FBQyxNQUFNLEVBQUU7NEJBQ25DLEdBQUc7NEJBQ0g7Z0NBQ0UsWUFBWSxFQUFFO29DQUNaLENBQUM7b0NBQ0Q7d0NBQ0UsV0FBVyxFQUFFOzRDQUNYLElBQUk7NENBQ0osRUFBRSxHQUFHLEVBQUUscUNBQW1CLENBQUMsR0FBRyxFQUFFO3lDQUNqQztxQ0FDRjtpQ0FDRjs2QkFDRjs0QkFDRDtnQ0FDRSxZQUFZLEVBQUU7b0NBQ1osQ0FBQztvQ0FDRDt3Q0FDRSxXQUFXLEVBQUU7NENBQ1gsSUFBSTs0Q0FDSixFQUFFLEdBQUcsRUFBRSxxQ0FBbUIsQ0FBQyxHQUFHLEVBQUU7eUNBQ2pDO3FDQUNGO2lDQUNGOzZCQUNGOzRCQUNELFlBQVk7NEJBQ1o7Z0NBQ0UsWUFBWSxFQUFFO29DQUNaLENBQUM7b0NBQ0Q7d0NBQ0UsV0FBVyxFQUFFOzRDQUNYLElBQUk7NENBQ0osRUFBRSxHQUFHLEVBQUUscUNBQW1CLENBQUMsR0FBRyxFQUFFO3lDQUNqQztxQ0FDRjtpQ0FDRjs2QkFDRjs0QkFDRDtnQ0FDRSxZQUFZLEVBQUU7b0NBQ1osQ0FBQztvQ0FDRDt3Q0FDRSxXQUFXLEVBQUU7NENBQ1gsSUFBSTs0Q0FDSixFQUFFLEdBQUcsRUFBRSxxQ0FBbUIsQ0FBQyxHQUFHLEVBQUU7eUNBQ2pDO3FDQUNGO2lDQUNGOzZCQUNGOzRCQUNELE1BQU07Z0NBQ04sd0RBQXdEO2dDQUN4RCxpREFBaUQ7Z0NBQ2pELE1BQU07Z0NBQ04sK0RBQStEOzRCQUMvRDtnQ0FDRSxZQUFZLEVBQUU7b0NBQ1osQ0FBQztvQ0FDRDt3Q0FDRSxXQUFXLEVBQUU7NENBQ1gsSUFBSTs0Q0FDSixFQUFFLEdBQUcsRUFBRSxxQ0FBbUIsQ0FBQyxHQUFHLEVBQUU7eUNBQ2pDO3FDQUNGO2lDQUNGOzZCQUNGOzRCQUNEO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLEdBQUcsRUFBRTt5Q0FDakM7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0QseUNBQXlDLFNBQVMsb0JBQW9CLE1BQU0sS0FBSztnQ0FDakYsY0FBYzs0QkFDZDtnQ0FDRSxZQUFZLEVBQUU7b0NBQ1osQ0FBQztvQ0FDRDt3Q0FDRSxXQUFXLEVBQUU7NENBQ1gsSUFBSTs0Q0FDSixFQUFFLEdBQUcsRUFBRSxxQ0FBbUIsQ0FBQyxHQUFHLEVBQUU7eUNBQ2pDO3FDQUNGO2lDQUNGOzZCQUNGOzRCQUNEO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLEdBQUcsRUFBRTt5Q0FDakM7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0QsS0FBSztnQ0FDTCxrRUFBa0U7Z0NBQ2xFLHdDQUF3QztnQ0FDeEMsSUFBSTt5QkFDTDtxQkFDRjtpQkFDRixDQUFDLENBQUM7Z0JBRUgsd0RBQXdEO2dCQUN4RCxlQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdDQUFnQyxFQUFFO29CQUM3RSxVQUFVLEVBQUUsS0FBSztvQkFDakIsTUFBTSxFQUFFLElBQUk7b0JBQ1oscUJBQXFCLEVBQUU7d0JBQ3JCLFlBQVksRUFBRTs0QkFDWixhQUFhLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxZQUEwQixDQUFDOzRCQUM5RixTQUFTO3lCQUNWO3FCQUNGO2lCQUNGLENBQUMsQ0FBQyxDQUFDO1lBQ04sQ0FBQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsOEJBQThCLEVBQUUsR0FBRyxFQUFFO2dCQUN4QyxPQUFPO2dCQUNQLE1BQU0sSUFBSSxHQUFHLElBQUksa0JBQVEsQ0FBQyxhQUFhLEVBQUUsTUFBTSxFQUFFO29CQUMvQyxHQUFHO29CQUNILFlBQVksRUFBRSxzQkFBWSxDQUFDLEVBQUUsQ0FDM0IsdUJBQWEsQ0FBQyxFQUFFLEVBQ2hCLHNCQUFZLENBQUMsS0FBSyxDQUNuQjtvQkFDRCxZQUFZLEVBQUUsc0JBQVksQ0FBQyxhQUFhLENBQUUsd0JBQWMsQ0FBQyxxQ0FBcUMsQ0FBQztpQkFDaEcsQ0FBQyxDQUFDO2dCQUVILEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQztvQkFDekIsSUFBSTtpQkFDTCxDQUFDLENBQUM7Z0JBRUgsT0FBTztnQkFDUCxNQUFNLFFBQVEsR0FBRyxhQUFhLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQztnQkFDL0QsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLGFBQWEsQ0FBQztvQkFDN0IsVUFBVSxFQUFFO3dCQUNWLEVBQUU7d0JBQ0Y7NEJBQ0UsZ0RBQWdEOzRCQUNoRDtnQ0FDRSxZQUFZLEVBQUU7b0NBQ1osQ0FBQztvQ0FDRDt3Q0FDRSxXQUFXLEVBQUU7NENBQ1gsSUFBSTs0Q0FDSixFQUFFLEdBQUcsRUFBRSxxQ0FBbUIsQ0FBQyxHQUFHLEVBQUU7eUNBQ2pDO3FDQUNGO2lDQUNGOzZCQUNGOzRCQUNEO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLEdBQUcsRUFBRTt5Q0FDakM7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0QsY0FBYztnQ0FDZCw4QkFBOEI7NEJBQzlCLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLE1BQU0sRUFBRTs0QkFDbkMsWUFBWTs0QkFDWjtnQ0FDRSxZQUFZLEVBQUU7b0NBQ1osQ0FBQztvQ0FDRDt3Q0FDRSxXQUFXLEVBQUU7NENBQ1gsSUFBSTs0Q0FDSixFQUFFLEdBQUcsRUFBRSxxQ0FBbUIsQ0FBQyxHQUFHLEVBQUU7eUNBQ2pDO3FDQUNGO2lDQUNGOzZCQUNGOzRCQUNEO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLEdBQUcsRUFBRTt5Q0FDakM7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0QscUJBQXFCOzRCQUNyQjtnQ0FDRSxZQUFZLEVBQUU7b0NBQ1osQ0FBQztvQ0FDRDt3Q0FDRSxXQUFXLEVBQUU7NENBQ1gsSUFBSTs0Q0FDSixFQUFFLEdBQUcsRUFBRSxxQ0FBbUIsQ0FBQyxHQUFHLEVBQUU7eUNBQ2pDO3FDQUNGO2lDQUNGOzZCQUNGOzRCQUNEO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLEdBQUcsRUFBRTt5Q0FDakM7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0Qsd0JBQXdCO2dDQUN4QixtQ0FBbUM7Z0NBQ25DLHlEQUF5RDtnQ0FDekQsc0VBQXNFOzRCQUN0RTtnQ0FDRSxZQUFZLEVBQUU7b0NBQ1osQ0FBQztvQ0FDRDt3Q0FDRSxXQUFXLEVBQUU7NENBQ1gsSUFBSTs0Q0FDSixFQUFFLEdBQUcsRUFBRSxxQ0FBbUIsQ0FBQyxHQUFHLEVBQUU7eUNBQ2pDO3FDQUNGO2lDQUNGOzZCQUNGOzRCQUNEO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLEdBQUcsRUFBRTt5Q0FDakM7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0QseUNBQXlDLFNBQVMscUJBQXFCLE1BQU0sV0FBVztnQ0FDeEYsNkJBQTZCOzRCQUM3QjtnQ0FDRSxZQUFZLEVBQUU7b0NBQ1osQ0FBQztvQ0FDRDt3Q0FDRSxXQUFXLEVBQUU7NENBQ1gsSUFBSTs0Q0FDSixFQUFFLEdBQUcsRUFBRSxxQ0FBbUIsQ0FBQyxHQUFHLEVBQUU7eUNBQ2pDO3FDQUNGO2lDQUNGOzZCQUNGOzRCQUNEO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKLEVBQUUsR0FBRyxFQUFFLHFDQUFtQixDQUFDLEdBQUcsRUFBRTt5Q0FDakM7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0QsS0FBSztnQ0FDTCxnRkFBZ0Y7Z0NBQ2hGLGlEQUFpRDtnQ0FDakQsWUFBWTtnQ0FDWiwrREFBK0Q7Z0NBQy9ELHlEQUF5RDtnQ0FDekQsZ0JBQWdCO3lCQUNqQjtxQkFDRjtpQkFDRixDQUFDLENBQUM7Z0JBRUgsd0RBQXdEO2dCQUN4RCxlQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdDQUFnQyxFQUFFO29CQUM3RSxVQUFVLEVBQUUsS0FBSztvQkFDakIsTUFBTSxFQUFFLElBQUk7b0JBQ1oscUJBQXFCLEVBQUU7d0JBQ3JCLFlBQVksRUFBRTs0QkFDWixhQUFhLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxZQUEwQixDQUFDOzRCQUM5RixTQUFTO3lCQUNWO3FCQUNGO2lCQUNGLENBQUMsQ0FBQyxDQUFDO1lBQ04sQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHFCQUFxQixFQUFFLEdBQUcsRUFBRTtRQUMvQixRQUFRO1FBQ1IsTUFBTSxPQUFPLEdBQUc7WUFDZCxnQkFBTSxDQUFDLG9CQUFvQixDQUFDLGVBQWUsRUFBRSxTQUFTLEVBQUU7Z0JBQ3RELFFBQVEsRUFBRSxXQUFXO2dCQUNyQixnQkFBZ0IsRUFBRSxZQUFZO2FBQy9CLENBQUM7WUFDRixnQkFBTSxDQUFDLG9CQUFvQixDQUFDLGVBQWUsRUFBRSxTQUFTLEVBQUU7Z0JBQ3RELFFBQVEsRUFBRSxXQUFXO2dCQUNyQixnQkFBZ0IsRUFBRSxZQUFZO2FBQy9CLENBQUM7U0FDSCxDQUFDO1FBQ0YsTUFBTSxhQUFhLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLGVBQWUsQ0FBQyxDQUFDO1FBQ3RELE1BQU0sS0FBSyxHQUFxQjtZQUM5QixNQUFNO1lBQ04sVUFBVTtZQUNWLE9BQU8sRUFBRSxJQUFJLGtCQUFZLENBQUMsYUFBYSxFQUFFLFNBQVMsQ0FBQztZQUNuRCxHQUFHO1lBQ0gsVUFBVSxFQUFFO2dCQUNWLE9BQU87YUFDUjtZQUNELGFBQWEsRUFBRTtnQkFDYixPQUFPO2FBQ1I7U0FDRixDQUFDO1FBRUYsT0FBTztRQUNQLElBQUksaUJBQVcsQ0FBQyxhQUFhLEVBQUUsYUFBYSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRXJELGVBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxvQ0FBb0MsRUFBRTtZQUM3RSxpQkFBaUIsRUFBRSxrQkFBUyxDQUMxQixXQUFXLEVBQ1gsV0FBVyxDQUNaO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDSixlQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsMkNBQTJDLEVBQUU7WUFDcEYsT0FBTyxFQUFFO2dCQUNQLFdBQVc7Z0JBQ1gsV0FBVzthQUNaO1NBQ0YsQ0FBQyxDQUFDLENBQUM7SUFDTixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQywyQkFBMkIsRUFBRSxHQUFHLEVBQUU7UUFDckMsUUFBUTtRQUNSLE1BQU0sYUFBYSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxlQUFlLENBQUMsQ0FBQztRQUN0RCxNQUFNLEtBQUssR0FBcUI7WUFDOUIsTUFBTTtZQUNOLFlBQVksRUFBRSxzQkFBWSxDQUFDLEVBQUUsQ0FBQyx1QkFBYSxDQUFDLEVBQUUsRUFBRSxzQkFBWSxDQUFDLEtBQUssQ0FBQztZQUNuRSxVQUFVO1lBQ1YsT0FBTyxFQUFFLElBQUksa0JBQVksQ0FBQyxhQUFhLEVBQUUsU0FBUyxDQUFDO1lBQ25ELEdBQUc7U0FDSixDQUFDO1FBRUYsT0FBTztRQUNQLElBQUksaUJBQVcsQ0FBQyxhQUFhLEVBQUUsYUFBYSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRXJELE9BQU87UUFDUCxlQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLHVDQUF1QyxFQUFFO1lBQ3BGLFlBQVksRUFBRSxVQUFVO1NBQ3pCLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsd0JBQXdCLEVBQUUsR0FBRyxFQUFFO1FBQ2xDLFFBQVE7UUFDUixNQUFNLGFBQWEsR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsZUFBZSxDQUFDLENBQUM7UUFDdEQsTUFBTSxLQUFLLEdBQXFCO1lBQzlCLE1BQU07WUFDTixVQUFVO1lBQ1YsT0FBTyxFQUFFLElBQUksa0JBQVksQ0FBQyxhQUFhLEVBQUUsU0FBUyxDQUFDO1lBQ25ELEdBQUc7WUFDSCxrQkFBa0IsRUFBRSxLQUFLO1NBQzFCLENBQUM7UUFFRixPQUFPO1FBQ1AsSUFBSSxpQkFBVyxDQUFDLGFBQWEsRUFBRSxhQUFhLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFckQsT0FBTztRQUNQLGVBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLENBQUMsWUFBRyxDQUFDLHlCQUFnQixDQUFDLDJDQUEyQyxFQUFFO1lBQzVGLHNCQUFzQixFQUFFLGtCQUFTLENBQy9CO2dCQUNFLEdBQUcsRUFBRSw2QkFBNkI7Z0JBQ2xDLEtBQUssRUFBRSxNQUFNO2FBQ2QsQ0FDRjtZQUNELE1BQU0sRUFBRSxlQUFNO1lBQ2QsSUFBSSxFQUFFLGVBQU07U0FDYixDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ1AsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMseUNBQXlDLEVBQUUsR0FBRyxFQUFFO1FBQ25ELFFBQVE7UUFDUixNQUFNLGFBQWEsR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsZUFBZSxDQUFDLENBQUM7UUFDdEQsTUFBTSxLQUFLLEdBQXFCO1lBQzlCLE1BQU07WUFDTixVQUFVO1lBQ1YsT0FBTyxFQUFFLElBQUksa0JBQVksQ0FBQyxhQUFhLEVBQUUsU0FBUyxDQUFDO1lBQ25ELEdBQUc7U0FDSixDQUFDO1FBRUYsT0FBTztRQUNQLElBQUksaUJBQVcsQ0FBQyxhQUFhLEVBQUUsYUFBYSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRXJELE9BQU87UUFDUCxlQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDJDQUEyQyxFQUFFO1lBQ3hGLHNCQUFzQixFQUFFLGtCQUFTLENBQy9CO2dCQUNFLEdBQUcsRUFBRSxpREFBaUQ7Z0JBQ3RELEtBQUssRUFBRSxNQUFNO2FBQ2QsQ0FDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsVUFBVSxFQUFFLEdBQUcsRUFBRTtRQUN4QixRQUFRO1FBQ1IsTUFBTSxRQUFRLEdBQUcsZ0JBQWdCLENBQUM7UUFFbEMsUUFBUSxDQUFDLGVBQWUsRUFBRSxHQUFHLEVBQUU7WUFDN0IsSUFBSSxhQUFvQixDQUFDO1lBRXpCLFVBQVUsQ0FBQyxHQUFHLEVBQUU7Z0JBQ2QsUUFBUTtnQkFDUixhQUFhLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLGVBQWUsQ0FBQyxDQUFDO2dCQUNoRCxNQUFNLEtBQUssR0FBcUI7b0JBQzlCLE1BQU07b0JBQ04sVUFBVTtvQkFDVixPQUFPLEVBQUUsSUFBSSxrQkFBWSxDQUFDLGFBQWEsRUFBRSxTQUFTLENBQUM7b0JBQ25ELEdBQUc7aUJBQ0osQ0FBQztnQkFFRixPQUFPO2dCQUNQLElBQUksaUJBQVcsQ0FBQyxhQUFhLEVBQUUsYUFBYSxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQ3ZELENBQUMsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLElBQUksQ0FBQyw4QkFBOEIsRUFBRSxHQUFHLEVBQUU7Z0JBQ3hDLGVBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxLQUFLLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDLENBQUM7WUFDMUUsQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILFFBQVEsQ0FBQyxxQ0FBcUMsRUFBRSxHQUFHLEVBQUU7WUFDbkQsSUFBSSxJQUF1QixDQUFDO1lBQzVCLElBQUksYUFBb0IsQ0FBQztZQUN6QixJQUFJLFdBQXdCLENBQUM7WUFFN0IsVUFBVSxDQUFDLEdBQUcsRUFBRTtnQkFDZCxRQUFRO2dCQUNSLElBQUksR0FBRyxJQUFJLCtCQUFpQixDQUFDLGVBQWUsRUFBRSxNQUFNLEVBQUU7b0JBQ3BELEdBQUc7b0JBQ0gsUUFBUTtpQkFDVCxDQUFDLENBQUM7Z0JBQ0gsYUFBYSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxlQUFlLENBQUMsQ0FBQztnQkFDaEQsTUFBTSxLQUFLLEdBQXFCO29CQUM5QixNQUFNO29CQUNOLFVBQVU7b0JBQ1YsT0FBTyxFQUFFLElBQUksa0JBQVksQ0FBQyxhQUFhLEVBQUUsU0FBUyxDQUFDO29CQUNuRCxHQUFHO29CQUNILFFBQVEsRUFBRTt3QkFDUixJQUFJO3FCQUNMO2lCQUNGLENBQUM7Z0JBRUYsT0FBTztnQkFDUCxXQUFXLEdBQUcsSUFBSSxpQkFBVyxDQUFDLGFBQWEsRUFBRSxhQUFhLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDckUsQ0FBQyxDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsSUFBSSxDQUFDLDZDQUE2QyxFQUFFLEdBQUcsRUFBRTtnQkFDdkQsTUFBTSxxQkFBcUIsR0FBRyxlQUFlLENBQUMsWUFBWSxDQUN4RCxXQUFXLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxZQUEwQixDQUN6RCxDQUFDO2dCQUNGLGVBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRTtvQkFDbEUsSUFBSSxFQUFFLGVBQWUsUUFBUSxHQUFHO29CQUNoQyxJQUFJLEVBQUUsR0FBRztvQkFDVCxXQUFXLEVBQUUsbUJBQVUsQ0FBQzt3QkFDdEIsWUFBWSxFQUFFOzRCQUNaLFlBQVksRUFBRTtnQ0FDWixxQkFBcUI7Z0NBQ3JCLHVCQUF1Qjs2QkFDeEI7eUJBQ0Y7cUJBQ0YsQ0FBQztpQkFDSCxDQUFDLENBQUMsQ0FBQztZQUNOLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsSUFBSSxDQUFDO1lBQ1IsQ0FBQyxxQkFBcUIsQ0FBQztZQUN2QixDQUFDLG1CQUFtQixDQUFDO1lBQ3JCLENBQUMsbUJBQW1CLENBQUM7WUFDckIsQ0FBQyxXQUFXLENBQUM7WUFDYixDQUFDLGlCQUFpQixDQUFDO1lBQ25CLENBQUMsa0VBQWtFLENBQUM7U0FDckUsQ0FBQyxDQUFDLDJCQUEyQixFQUFFLENBQUMsUUFBZ0IsRUFBRSxFQUFFO1lBQ25ELFFBQVE7WUFDUixNQUFNLElBQUksR0FBRyxJQUFJLCtCQUFpQixDQUFDLGVBQWUsRUFBRSxNQUFNLEVBQUU7Z0JBQzFELFFBQVEsRUFBRSxrQkFBa0I7Z0JBQzVCLEdBQUc7YUFDSixDQUFDLENBQUM7WUFDSCxNQUFNLEtBQUssR0FBcUI7Z0JBQzlCLE1BQU07Z0JBQ04sVUFBVTtnQkFDVixPQUFPO2dCQUNQLEdBQUc7Z0JBQ0gsUUFBUSxFQUFFO29CQUNSLFFBQVE7b0JBQ1IsSUFBSTtpQkFDTDthQUNGLENBQUM7WUFFRixPQUFPO1lBQ1AsU0FBUyxJQUFJO2dCQUNYLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDbEQsQ0FBQztZQUVELE9BQU87WUFDUCxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDdkQsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxhQUFhLEVBQUUsR0FBRyxFQUFFO1FBQzNCLElBQUksYUFBb0IsQ0FBQztRQUN6QixJQUFJLFdBQWdCLENBQUM7UUFDckIsSUFBSSxrQkFBOEIsQ0FBQztRQUNuQyxJQUFJLGVBQXlCLENBQUM7UUFDOUIsSUFBSSxjQUFpQyxDQUFDO1FBRXRDLElBQUksWUFBb0IsQ0FBQztRQUV6QixVQUFVLENBQUMsR0FBRyxFQUFFO1lBQ2QsUUFBUTtZQUNSLGFBQWEsR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsZUFBZSxFQUFFO2dCQUM5QyxHQUFHLEVBQUU7b0JBQ0gsTUFBTSxFQUFFLFdBQVc7aUJBQ3BCO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsV0FBVyxHQUFHLElBQUksYUFBRyxDQUFDLGFBQWEsRUFBRSxLQUFLLENBQUMsQ0FBQztZQUM1QyxlQUFlLEdBQUcsSUFBSSxrQkFBWSxDQUFDLGFBQWEsRUFBRSxTQUFTLENBQUMsQ0FBQztZQUU3RCxrQkFBa0IsR0FBRyxJQUFJLGdCQUFVLENBQUMsYUFBYSxFQUFFLE1BQU0sRUFBRTtnQkFDekQsT0FBTyxFQUFFLGVBQWU7Z0JBQ3hCLEdBQUcsRUFBRSxXQUFXO2FBQ2pCLENBQUMsQ0FBQztZQUVILGNBQWMsR0FBRztnQkFDZixzQkFBc0IsRUFBRSxRQUFRO2FBQ2pDLENBQUM7WUFFRixZQUFZLEdBQUcsSUFBSSxlQUFNLENBQUMsYUFBYSxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBRTNELENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLG1EQUFtRCxFQUFFLEdBQUcsRUFBRTtZQUM3RCxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQXFCO2dCQUM5QixNQUFNLEVBQUUsY0FBYztnQkFDdEIsVUFBVSxFQUFFLGtCQUFrQjtnQkFDOUIsT0FBTyxFQUFFLGVBQWU7Z0JBQ3hCLEdBQUcsRUFBRSxXQUFXO2dCQUNoQixVQUFVLEVBQUU7b0JBQ1YsaUJBQWlCLEVBQUUsWUFBWTtpQkFDaEM7YUFDRixDQUFDO1lBRUYsT0FBTztZQUNQLElBQUksaUJBQVcsQ0FBQyxhQUFhLEVBQUUsYUFBYSxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBRXJELE9BQU87WUFDUCxlQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDJDQUEyQyxFQUFFO2dCQUN4RixzQkFBc0IsRUFBRSxrQkFBUyxDQUMvQjtvQkFDRSxHQUFHLEVBQUUsd0JBQXdCO29CQUM3QixLQUFLLEVBQUUsTUFBTTtpQkFDZCxFQUNEO29CQUNFLEdBQUcsRUFBRSx1QkFBdUI7b0JBQzVCLEtBQUssRUFBRTt3QkFDTCxHQUFHLEVBQUUsc0JBQXNCO3FCQUM1QjtpQkFDRixDQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixlQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLHVCQUF1QixFQUFFO2dCQUNwRSxNQUFNLEVBQUU7b0JBQ04sR0FBRyxFQUFFLHNCQUFzQjtpQkFDNUI7Z0JBQ0QsY0FBYyxFQUFFO29CQUNkLFNBQVMsRUFBRSxrQkFBUyxDQUNsQjt3QkFDRSxNQUFNLEVBQUUsY0FBYzt3QkFDdEIsU0FBUyxFQUFFOzRCQUNULFlBQVksRUFBRTtnQ0FDWixjQUFjLEVBQUUsMkJBQTJCOzZCQUM1Qzt5QkFDRjt3QkFDRCxNQUFNLEVBQUUsT0FBTzt3QkFDZixTQUFTLEVBQUU7NEJBQ1QsT0FBTyxFQUFFLDZCQUE2Qjt5QkFDdkM7d0JBQ0QsUUFBUSxFQUFFOzRCQUNSLFVBQVUsRUFBRTtnQ0FDVixFQUFFO2dDQUNGO29DQUNFO3dDQUNFLFlBQVksRUFBRTs0Q0FDWixzQkFBc0I7NENBQ3RCLEtBQUs7eUNBQ047cUNBQ0Y7b0NBQ0QsSUFBSTtpQ0FDTDs2QkFDRjt5QkFDRjtxQkFDRixFQUNEO3dCQUNFLE1BQU0sRUFBRSxpQkFBaUI7d0JBQ3pCLE1BQU0sRUFBRSxPQUFPO3dCQUNmLFNBQVMsRUFBRTs0QkFDVCxPQUFPLEVBQUUsNkJBQTZCO3lCQUN2Qzt3QkFDRCxRQUFRLEVBQUU7NEJBQ1IsWUFBWSxFQUFFO2dDQUNaLHNCQUFzQjtnQ0FDdEIsS0FBSzs2QkFDTjt5QkFDRjtxQkFDRixFQUNEO3dCQUNFLE1BQU0sRUFBRTs0QkFDTixlQUFlOzRCQUNmLFdBQVc7eUJBQ1o7d0JBQ0QsTUFBTSxFQUFFLE9BQU87d0JBQ2YsU0FBUyxFQUFFOzRCQUNULEdBQUcsRUFBRTtnQ0FDSCxVQUFVLEVBQUU7b0NBQ1YsRUFBRTtvQ0FDRjt3Q0FDRSxNQUFNO3dDQUNOOzRDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7eUNBQ3RCO3dDQUNELHlCQUF5QjtxQ0FDMUI7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7d0JBQ0QsUUFBUSxFQUFFOzRCQUNSLFVBQVUsRUFBRTtnQ0FDVixFQUFFO2dDQUNGO29DQUNFO3dDQUNFLFlBQVksRUFBRTs0Q0FDWixzQkFBc0I7NENBQ3RCLEtBQUs7eUNBQ047cUNBQ0Y7b0NBQ0QsV0FBVztvQ0FDWDt3Q0FDRSxHQUFHLEVBQUUsZ0JBQWdCO3FDQUN0QjtvQ0FDRCxJQUFJO2lDQUNMOzZCQUNGO3lCQUNGO3FCQUNGLENBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztRQUNOLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHdDQUF3QyxFQUFFLEdBQUcsRUFBRTtZQUNsRCxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQXFCO2dCQUM5QixNQUFNLEVBQUUsY0FBYztnQkFDdEIsVUFBVSxFQUFFLGtCQUFrQjtnQkFDOUIsT0FBTyxFQUFFLGVBQWU7Z0JBQ3hCLEdBQUcsRUFBRSxXQUFXO2dCQUNoQixVQUFVLEVBQUU7b0JBQ1YsaUJBQWlCLEVBQUUsWUFBWTtvQkFDL0IsTUFBTSxFQUFFLGVBQWU7aUJBQ3hCO2FBQ0YsQ0FBQztZQUVGLE9BQU87WUFDUCxJQUFJLGlCQUFXLENBQUMsYUFBYSxFQUFFLGFBQWEsRUFBRSxLQUFLLENBQUMsQ0FBQztZQUVyRCxPQUFPO1lBQ1AsZUFBUyxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywyQ0FBMkMsRUFBRTtnQkFDeEYsc0JBQXNCLEVBQUUsa0JBQVMsQ0FDL0I7b0JBQ0UsR0FBRyxFQUFFLHdCQUF3QjtvQkFDN0IsS0FBSyxFQUFFLE1BQU07aUJBQ2QsRUFDRDtvQkFDRSxHQUFHLEVBQUUsdUJBQXVCO29CQUM1QixLQUFLLEVBQUU7d0JBQ0wsR0FBRyxFQUFFLHNCQUFzQjtxQkFDNUI7aUJBQ0YsRUFDRDtvQkFDRSxHQUFHLEVBQUUsdUJBQXVCO29CQUM1QixLQUFLLEVBQUUsZUFBZTtpQkFDdkIsQ0FDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosZUFBUyxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1QkFBdUIsRUFBRTtnQkFDcEUsTUFBTSxFQUFFO29CQUNOLEdBQUcsRUFBRSxzQkFBc0I7aUJBQzVCO2dCQUNELGNBQWMsRUFBRTtvQkFDZCxTQUFTLEVBQUUsa0JBQVMsQ0FDbEI7d0JBQ0UsTUFBTSxFQUFFLGNBQWM7d0JBQ3RCLFNBQVMsRUFBRTs0QkFDVCxZQUFZLEVBQUU7Z0NBQ1osY0FBYyxFQUFFLDJCQUEyQjs2QkFDNUM7eUJBQ0Y7d0JBQ0QsTUFBTSxFQUFFLE9BQU87d0JBQ2YsU0FBUyxFQUFFOzRCQUNULE9BQU8sRUFBRSw2QkFBNkI7eUJBQ3ZDO3dCQUNELFFBQVEsRUFBRTs0QkFDUixVQUFVLEVBQUU7Z0NBQ1YsRUFBRTtnQ0FDRjtvQ0FDRTt3Q0FDRSxZQUFZLEVBQUU7NENBQ1osc0JBQXNCOzRDQUN0QixLQUFLO3lDQUNOO3FDQUNGO29DQUNELElBQUk7aUNBQ0w7NkJBQ0Y7eUJBQ0Y7cUJBQ0YsRUFDRDt3QkFDRSxNQUFNLEVBQUUsaUJBQWlCO3dCQUN6QixNQUFNLEVBQUUsT0FBTzt3QkFDZixTQUFTLEVBQUU7NEJBQ1QsT0FBTyxFQUFFLDZCQUE2Qjt5QkFDdkM7d0JBQ0QsUUFBUSxFQUFFOzRCQUNSLFlBQVksRUFBRTtnQ0FDWixzQkFBc0I7Z0NBQ3RCLEtBQUs7NkJBQ047eUJBQ0Y7cUJBQ0YsRUFDRDt3QkFDRSxNQUFNLEVBQUU7NEJBQ04sZUFBZTs0QkFDZixXQUFXO3lCQUNaO3dCQUNELE1BQU0sRUFBRSxPQUFPO3dCQUNmLFNBQVMsRUFBRTs0QkFDVCxHQUFHLEVBQUU7Z0NBQ0gsVUFBVSxFQUFFO29DQUNWLEVBQUU7b0NBQ0Y7d0NBQ0UsTUFBTTt3Q0FDTjs0Q0FDRSxHQUFHLEVBQUUsZ0JBQWdCO3lDQUN0Qjt3Q0FDRCx5QkFBeUI7cUNBQzFCO2lDQUNGOzZCQUNGO3lCQUNGO3dCQUNELFFBQVEsRUFBRTs0QkFDUixVQUFVLEVBQUU7Z0NBQ1YsRUFBRTtnQ0FDRjtvQ0FDRTt3Q0FDRSxZQUFZLEVBQUU7NENBQ1osc0JBQXNCOzRDQUN0QixLQUFLO3lDQUNOO3FDQUNGO29DQUNELHlCQUF5QjtvQ0FDekI7d0NBQ0UsR0FBRyxFQUFFLGdCQUFnQjtxQ0FDdEI7b0NBQ0QsSUFBSTtpQ0FDTDs2QkFDRjt5QkFDRjtxQkFDRixDQUNGO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDTixDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLFNBQVMsRUFBRSxHQUFHLEVBQUU7UUFDdkIsK0JBQWlCLENBQUM7WUFDaEIsYUFBYSxFQUFFLGFBQWE7WUFDNUIsZUFBZSxFQUFFLEdBQUcsRUFBRTtnQkFDcEIsT0FBTyxLQUFLLENBQUM7WUFDZixDQUFDO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLG1CQUFtQixFQUFFLENBQUM7Z0JBQ3RCLHlCQUF5QixFQUFFLENBQUM7Z0JBQzVCLGdCQUFnQixFQUFFLENBQUM7Z0JBQ25CLG9DQUFvQyxFQUFFLENBQUM7Z0JBQ3ZDLHVCQUF1QixFQUFFLENBQUM7Z0JBQzFCLGlCQUFpQixFQUFFLENBQUM7Z0JBQ3BCLDBCQUEwQixFQUFFLENBQUM7Z0JBQzdCLHNCQUFzQixFQUFFLENBQUM7Z0JBQ3pCLDZCQUE2QixFQUFFLENBQUM7Z0JBQ2hDLDJDQUEyQyxFQUFFLENBQUM7Z0JBQzlDLDBDQUEwQyxFQUFFLENBQUM7Z0JBQzdDLG1CQUFtQixFQUFFLENBQUM7YUFDdkI7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxjQUFjLEVBQUUsR0FBRyxFQUFFO1FBQzVCLElBQUksQ0FBQyx1QkFBdUIsRUFBRSxHQUFHLEVBQUU7WUFDakMsT0FBTztZQUNQLGlCQUFpQixDQUFDLGNBQWMsRUFBRSxDQUFDO1lBRW5DLE9BQU87WUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLDJCQUFrQixDQUFDLGdCQUFnQixFQUFFLENBQUMsRUFBRTtnQkFDMUQsaUJBQWlCLEVBQUUsa0JBQVMsQ0FDMUI7b0JBQ0UsVUFBVSxFQUFFO3dCQUNWLEVBQUU7d0JBQ0Y7NEJBQ0UsTUFBTTs0QkFDTjtnQ0FDRSxHQUFHLEVBQUUsZ0JBQWdCOzZCQUN0Qjs0QkFDRCxnRUFBZ0U7eUJBQ2pFO3FCQUNGO2lCQUNGLEVBQ0Q7b0JBQ0UsVUFBVSxFQUFFO3dCQUNWLEVBQUU7d0JBQ0Y7NEJBQ0UsTUFBTTs0QkFDTjtnQ0FDRSxHQUFHLEVBQUUsZ0JBQWdCOzZCQUN0Qjs0QkFDRCxnRUFBZ0U7eUJBQ2pFO3FCQUNGO2lCQUNGLENBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztRQUNOLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHFCQUFxQixFQUFFLEdBQUcsRUFBRTtZQUMvQixPQUFPO1lBQ1AsaUJBQWlCLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBRXhDLE9BQU87WUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdCQUFnQixFQUFFO2dCQUNyRCxpQkFBaUIsRUFBRSxrQkFBUyxDQUMxQjtvQkFDRSxVQUFVLEVBQUU7d0JBQ1YsRUFBRTt3QkFDRjs0QkFDRSxNQUFNOzRCQUNOO2dDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7NkJBQ3RCOzRCQUNELGdFQUFnRTt5QkFDakU7cUJBQ0Y7aUJBQ0YsQ0FDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBQ0osZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEtBQUssQ0FBQyx5QkFBZ0IsQ0FBQyxnQkFBZ0IsRUFBRTtnQkFDeEQsaUJBQWlCLEVBQUUsa0JBQVMsQ0FDMUI7b0JBQ0UsVUFBVSxFQUFFO3dCQUNWLEVBQUU7d0JBQ0Y7NEJBQ0UsTUFBTTs0QkFDTjtnQ0FDRSxHQUFHLEVBQUUsZ0JBQWdCOzZCQUN0Qjs0QkFDRCxnRUFBZ0U7eUJBQ2pFO3FCQUNGO2lCQUNGLENBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztRQUNOLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMseUNBQXlDLEVBQUUsR0FBRyxFQUFFO1FBQ25ELE9BQU87UUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLG1DQUFtQyxFQUFFO1lBQ3hFLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUM7WUFDNUQsd0NBQXdDO1lBQ3hDLFFBQVEsRUFBRSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsaUJBQWlCLENBQUMsU0FBUyxDQUFDLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQzNFLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsaUJBQWlCLEVBQUUsR0FBRyxFQUFFO1FBQy9CLElBQUksb0JBQW1DLENBQUM7UUFDeEMsSUFBSSxxQkFBb0MsQ0FBQztRQUV6QyxVQUFVLENBQUMsR0FBRyxFQUFFO1lBQ2Qsb0JBQW9CLEdBQUcsSUFBSSx1QkFBYSxDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7WUFDN0UscUJBQXFCLEdBQUcsSUFBSSx1QkFBYSxDQUFDLEtBQUssRUFBRSxpQkFBaUIsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7UUFDL0UsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsc0NBQXNDLEVBQUUsR0FBRyxFQUFFO1lBQ2hELFFBQVE7WUFDUixNQUFNLGNBQWMsR0FBOEI7Z0JBQ2hELE9BQU8sRUFBRSxvQkFBb0I7Z0JBQzdCLFFBQVEsRUFBRSxxQkFBcUI7YUFDaEMsQ0FBQztZQUVGLE9BQU87WUFDUCxJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRTtnQkFDcEMsTUFBTTtnQkFDTixVQUFVO2dCQUNWLE9BQU8sRUFBRSxrQkFBa0I7Z0JBQzNCLEdBQUc7Z0JBQ0gsY0FBYzthQUNmLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCw2QkFBNkIsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUNoRCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxnREFBZ0QsRUFBRSxHQUFHLEVBQUU7WUFDMUQsUUFBUTtZQUNSLE1BQU0sV0FBVyxHQUFHLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsYUFBYSxFQUFFO2dCQUN4RCxNQUFNO2dCQUNOLFVBQVU7Z0JBQ1YsT0FBTyxFQUFFLGtCQUFrQjtnQkFDM0IsR0FBRzthQUNKLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxXQUFXLENBQUMsd0JBQXdCLENBQUMsb0JBQW9CLENBQUMsQ0FBQztZQUUzRCxPQUFPO1lBQ1AsNkJBQTZCLENBQUM7Z0JBQzVCLE9BQU8sRUFBRSxvQkFBb0I7YUFDOUIsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsaURBQWlELEVBQUUsR0FBRyxFQUFFO1lBQzNELFFBQVE7WUFDUixNQUFNLFdBQVcsR0FBRyxJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRTtnQkFDeEQsTUFBTTtnQkFDTixVQUFVO2dCQUNWLE9BQU8sRUFBRSxrQkFBa0I7Z0JBQzNCLEdBQUc7YUFDSixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsV0FBVyxDQUFDLHlCQUF5QixDQUFDLHFCQUFxQixDQUFDLENBQUM7WUFFN0QsT0FBTztZQUNQLDZCQUE2QixDQUFDO2dCQUM1QixRQUFRLEVBQUUscUJBQXFCO2FBQ2hDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGdGQUFnRixFQUFFLEdBQUcsRUFBRTtZQUMxRixRQUFRO1lBQ1IsTUFBTSxXQUFXLEdBQUcsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxhQUFhLEVBQUU7Z0JBQ3hELE1BQU07Z0JBQ04sVUFBVTtnQkFDVixPQUFPLEVBQUUsa0JBQWtCO2dCQUMzQixHQUFHO2FBQ0osQ0FBQyxDQUFDO1lBQ0gsV0FBVyxDQUFDLHdCQUF3QixDQUFDLG9CQUFvQixDQUFDLENBQUM7WUFDM0QsV0FBVyxDQUFDLHlCQUF5QixDQUFDLHFCQUFxQixDQUFDLENBQUM7WUFDN0QsTUFBTSxpQkFBaUIsR0FBRyxJQUFJLHVCQUFhLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQztZQUVqRixPQUFPO1lBQ1AsV0FBVyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsaUJBQWlCLEVBQUUsY0FBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBRW5FLE9BQU87WUFDUCw4RUFBOEU7WUFDOUUsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEtBQUssQ0FBQyx5QkFBZ0IsQ0FBQyxnQ0FBZ0MsRUFBRTtnQkFDeEUsVUFBVSxFQUFFLEtBQUs7Z0JBQ2pCLFFBQVEsRUFBRSxFQUFFO2dCQUNaLE1BQU0sRUFBRSxFQUFFO2dCQUNWLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLHFCQUFxQixDQUFDLGVBQWUsQ0FBQztnQkFDN0QscUJBQXFCLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxlQUFlLENBQUM7YUFDeEUsQ0FBQyxDQUFDLENBQUM7WUFDSixrRkFBa0Y7WUFDbEYsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEtBQUssQ0FBQyx5QkFBZ0IsQ0FBQyxnQ0FBZ0MsRUFBRTtnQkFDeEUsVUFBVSxFQUFFLEtBQUs7Z0JBQ2pCLFFBQVEsRUFBRSxFQUFFO2dCQUNaLE1BQU0sRUFBRSxFQUFFO2dCQUNWLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLG9CQUFvQixDQUFDLGVBQWUsQ0FBQztnQkFDNUQscUJBQXFCLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxlQUFlLENBQUM7YUFDeEUsQ0FBQyxDQUFDLENBQUM7UUFDTixDQUFDLENBQUMsQ0FBQztRQUVILFNBQVMsNkJBQTZCLENBQUMsY0FBeUM7WUFDOUUsSUFBSSxjQUFjLENBQUMsT0FBTyxLQUFLLFNBQVMsRUFBRTtnQkFDeEMsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1Q0FBdUMsRUFBRTtvQkFDNUUsY0FBYyxFQUFFLGtCQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyxDQUFDO2lCQUNqRixDQUFDLENBQUMsQ0FBQzthQUNMO1lBQ0QsSUFBSSxjQUFjLENBQUMsUUFBUSxLQUFLLFNBQVMsRUFBRTtnQkFDekMsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywyQ0FBMkMsRUFBRTtvQkFDaEYsY0FBYyxFQUFFLGtCQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDLGVBQWUsQ0FBQyxDQUFDO2lCQUNsRixDQUFDLENBQUMsQ0FBQzthQUNMO1FBQ0gsQ0FBQztJQUNILENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLG9EQUFvRCxFQUFFLEdBQUcsRUFBRTtRQUM5RCxRQUFRO1FBQ1IsTUFBTSxRQUFRLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQzVDLE9BQU87UUFDUCxJQUFJLGlCQUFXLENBQUMsUUFBUSxFQUFFLGdCQUFnQixFQUFFO1lBQzFDLE1BQU07WUFDTixVQUFVO1lBQ1YsT0FBTztZQUNQLEdBQUc7U0FDSixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsU0FBUyxLQUFLO1lBQ1osbUJBQVUsQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDbEMsQ0FBQztRQUVELE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDLDJEQUEyRCxDQUFDLENBQUM7SUFDckYsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsNkNBQTZDLEVBQUUsR0FBRyxFQUFFO1FBQ3ZELGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxLQUFLLENBQUMseUJBQWdCLENBQUMsdUNBQXVDLEVBQUU7WUFDL0UsUUFBUSxFQUFFO2dCQUNSLFlBQVksRUFBRTtvQkFDWixVQUFVLEVBQUUsa0JBQVMsQ0FBQyxrQkFBUyxDQUFDLGlUQUFpVCxDQUFDLENBQUM7aUJBQ3BWO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztJQUNOLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLEdBQUcsRUFBRTtRQUNoRCxRQUFRO1FBQ1IsTUFBTSxhQUFhLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLGVBQWUsQ0FBQyxDQUFDO1FBRXRELE9BQU87UUFDUCxJQUFJLGlCQUFXLENBQUMsYUFBYSxFQUFFLGFBQWEsRUFBRTtZQUM1QyxNQUFNO1lBQ04sVUFBVTtZQUNWLE9BQU8sRUFBRSxJQUFJLGtCQUFZLENBQUMsYUFBYSxFQUFFLFNBQVMsQ0FBQztZQUNuRCxHQUFHO1lBQ0gsc0JBQXNCLEVBQUUsSUFBSTtTQUM3QixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZ0VBQWdFO1FBQ2hFLDhFQUE4RTtRQUM5RSxvQkFBb0I7UUFDcEIsZUFBUyxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1Q0FBdUMsRUFBRTtZQUNwRixRQUFRLEVBQUU7Z0JBQ1IsWUFBWSxFQUFFO29CQUNaLFVBQVUsRUFBRSxrQkFBUyxDQUFDLGtCQUFTLENBQUMsaVRBQWlULENBQUMsQ0FBQztpQkFDcFY7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQyxDQUFDLENBQUM7QUFFTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuICovXG5cbmltcG9ydCB7XG4gIEFCU0VOVCxcbiAgYXJyYXlXaXRoLFxuICBjb3VudFJlc291cmNlc0xpa2UsXG4gIGRlZXBPYmplY3RMaWtlLFxuICBleHBlY3QgYXMgZXhwZWN0Q0RLLFxuICBoYXZlUmVzb3VyY2UsXG4gIGhhdmVSZXNvdXJjZUxpa2UsXG4gIG5vdCxcbiAgb2JqZWN0TGlrZSxcbiAgUmVzb3VyY2VQYXJ0LFxuICBTeW50aFV0aWxzLFxufSBmcm9tICdAYXdzLWNkay9hc3NlcnQnO1xuaW1wb3J0IHtcbiAgQ2VydGlmaWNhdGUsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1jZXJ0aWZpY2F0ZW1hbmFnZXInO1xuaW1wb3J0IHtcbiAgQW1hem9uTGludXhHZW5lcmF0aW9uLFxuICBJbnN0YW5jZSxcbiAgSW5zdGFuY2VDbGFzcyxcbiAgSW5zdGFuY2VTaXplLFxuICBJbnN0YW5jZVR5cGUsXG4gIE1hY2hpbmVJbWFnZSxcbiAgUG9ydCxcbiAgU2VjdXJpdHlHcm91cCxcbiAgU3VibmV0LFxuICBWcGMsXG4gIFdpbmRvd3NWZXJzaW9uLFxufSBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCB7XG4gIENvbnRhaW5lckltYWdlLFxuICBUYXNrRGVmaW5pdGlvbixcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWVjcyc7XG5pbXBvcnQge1xuICBBcHBsaWNhdGlvblByb3RvY29sLFxufSBmcm9tICdAYXdzLWNkay9hd3MtZWxhc3RpY2xvYWRiYWxhbmNpbmd2Mic7XG5pbXBvcnQge1xuICBBY2NvdW50Um9vdFByaW5jaXBhbCxcbiAgUm9sZSxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQge1xuICBQcml2YXRlSG9zdGVkWm9uZSxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLXJvdXRlNTMnO1xuaW1wb3J0IHtcbiAgQnVja2V0LFxufSBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuaW1wb3J0IHsgU2VjcmV0IH0gZnJvbSAnQGF3cy1jZGsvYXdzLXNlY3JldHNtYW5hZ2VyJztcbmltcG9ydCB7XG4gIEFwcCxcbiAgQ2ZuRWxlbWVudCxcbiAgU3RhY2ssXG59IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG5pbXBvcnQge1xuICBYNTA5Q2VydGlmaWNhdGVQZW0sXG59IGZyb20gJy4uLy4uJztcbmltcG9ydCB7XG4gIHRlc3RDb25zdHJ1Y3RUYWdzLFxufSBmcm9tICcuLi8uLi9jb3JlL3Rlc3QvdGFnLWhlbHBlcnMnO1xuaW1wb3J0IHtcbiAgSVZlcnNpb24sXG4gIFJlbmRlclF1ZXVlLFxuICBSZW5kZXJRdWV1ZUltYWdlcyxcbiAgUmVuZGVyUXVldWVQcm9wcyxcbiAgUmVuZGVyUXVldWVTZWN1cml0eUdyb3VwcyxcbiAgUmVwb3NpdG9yeSxcbiAgVmVyc2lvblF1ZXJ5LFxufSBmcm9tICcuLi9saWInO1xuaW1wb3J0IHtcbiAgUlFfQ09OTkVDVElPTl9BU1NFVCxcbn0gZnJvbSAnLi9hc3NldC1jb25zdGFudHMnO1xuXG5kZXNjcmliZSgnUmVuZGVyUXVldWUnLCAoKSA9PiB7XG4gIGxldCBhcHA6IEFwcDtcbiAgbGV0IGRlcGVuZGVuY3lTdGFjazogU3RhY2s7XG4gIGxldCBzdGFjazogU3RhY2s7XG4gIGxldCB2cGM6IFZwYztcbiAgbGV0IHJjc0ltYWdlOiBDb250YWluZXJJbWFnZTtcbiAgbGV0IGltYWdlczogUmVuZGVyUXVldWVJbWFnZXM7XG5cbiAgbGV0IHJlcG9zaXRvcnk6IFJlcG9zaXRvcnk7XG4gIGxldCB2ZXJzaW9uOiBJVmVyc2lvbjtcbiAgbGV0IHJlbmRlclF1ZXVlVmVyc2lvbjogSVZlcnNpb247XG5cbiAgbGV0IHJlbmRlclF1ZXVlQ29tbW9uOiBSZW5kZXJRdWV1ZTtcblxuICAvLyBHSVZFTlxuICBiZWZvcmVFYWNoKCgpID0+IHtcbiAgICBhcHAgPSBuZXcgQXBwKCk7XG4gICAgZGVwZW5kZW5jeVN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ0RlcFN0YWNrJyk7XG4gICAgdnBjID0gbmV3IFZwYyhkZXBlbmRlbmN5U3RhY2ssICdWcGMnKTtcbiAgICB2ZXJzaW9uID0gbmV3IFZlcnNpb25RdWVyeShkZXBlbmRlbmN5U3RhY2ssICdWZXJzaW9uJyk7XG4gICAgcmVwb3NpdG9yeSA9IG5ldyBSZXBvc2l0b3J5KGRlcGVuZGVuY3lTdGFjaywgJ1JlcG8nLCB7XG4gICAgICB2ZXJzaW9uLFxuICAgICAgdnBjLFxuICAgIH0pO1xuICAgIHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrJyk7XG4gICAgcmNzSW1hZ2UgPSBDb250YWluZXJJbWFnZS5mcm9tQXNzZXQoX19kaXJuYW1lKTtcbiAgICBpbWFnZXMgPSB7XG4gICAgICByZW1vdGVDb25uZWN0aW9uU2VydmVyOiByY3NJbWFnZSxcbiAgICB9O1xuICAgIHJlbmRlclF1ZXVlVmVyc2lvbiA9IG5ldyBWZXJzaW9uUXVlcnkoc3RhY2ssICdWZXJzaW9uJyk7XG4gICAgcmVuZGVyUXVldWVDb21tb24gPSBuZXcgUmVuZGVyUXVldWUoc3RhY2ssICdSZW5kZXJRdWV1ZUNvbW1vbicsIHtcbiAgICAgIGltYWdlcyxcbiAgICAgIHJlcG9zaXRvcnksXG4gICAgICB2ZXJzaW9uOiByZW5kZXJRdWV1ZVZlcnNpb24sXG4gICAgICB2cGMsXG4gICAgfSk7XG4gIH0pO1xuXG4gIGFmdGVyRWFjaCgoKSA9PiB7XG4gICAgamVzdC5yZXNldEFsbE1vY2tzKCk7XG4gIH0pO1xuXG4gIHRlc3QoJ2NyZWF0ZXMgY2x1c3RlcicsICgpID0+IHtcbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUNTOjpDbHVzdGVyJykpO1xuICB9KTtcblxuICB0ZXN0KCdjcmVhdGVzIHNlcnZpY2UnLCAoKSA9PiB7XG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDUzo6U2VydmljZScpKTtcbiAgfSk7XG5cbiAgdGVzdCgnY3JlYXRlcyB0YXNrIGRlZmluaXRpb24nLCAoKSA9PiB7XG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDUzo6VGFza0RlZmluaXRpb24nKSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2Nsb3NlZCBpbmdyZXNzIGJ5IGRlZmF1bHQnLCAoKSA9PiB7XG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdENESyhzdGFjaykubm90VG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cCcsIHtcbiAgICAgIC8vIFRoZSBvcGVuTGlzdGVuZXI9dHJ1ZSBvcHRpb24gd291bGQgY3JlYXRlIGFuIGluZ3Jlc3MgcnVsZSBpbiB0aGUgbGlzdGVuZXIncyBTRy5cbiAgICAgIC8vIG1ha2Ugc3VyZSB0aGF0IHdlIGRvbid0IGhhdmUgdGhhdC5cbiAgICAgIFNlY3VyaXR5R3JvdXBJbmdyZXNzOiBhcnJheVdpdGgob2JqZWN0TGlrZSh7fSkpLFxuICAgIH0pKTtcbiAgfSk7XG5cbiAgdGVzdCgnY3JlYXRlcyBsb2FkIGJhbGFuY2VyIHdpdGggZGVmYXVsdCB2YWx1ZXMnLCAoKSA9PiB7XG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdENESyhzdGFjaykudG8oY291bnRSZXNvdXJjZXNMaWtlKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6OkxvYWRCYWxhbmNlcicsIDEsIHtcbiAgICAgIExvYWRCYWxhbmNlckF0dHJpYnV0ZXM6IFtcbiAgICAgICAge1xuICAgICAgICAgIEtleTogJ2RlbGV0aW9uX3Byb3RlY3Rpb24uZW5hYmxlZCcsXG4gICAgICAgICAgVmFsdWU6ICd0cnVlJyxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgICBTY2hlbWU6ICdpbnRlcm5hbCcsXG4gICAgfSkpO1xuICB9KTtcblxuICB0ZXN0KCdjcmVhdGVzIGEgbG9nIGdyb3VwIHdpdGggZGVmYXVsdCBwcmVmaXggb2YgXCIvcmVuZGVyZmFybS9cIicsICgpID0+IHtcbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdDdXN0b206OkxvZ1JldGVudGlvbicsIHtcbiAgICAgIExvZ0dyb3VwTmFtZTogJy9yZW5kZXJmYXJtL1JlbmRlclF1ZXVlQ29tbW9uJyxcbiAgICAgIFJldGVudGlvbkluRGF5czogMyxcbiAgICB9KSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2NvbmZpZ3VyZSB0aGUgY29udGFpbmVyIGxvZyBkcml2ZXInLCAoKSA9PiB7XG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uJywge1xuICAgICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgb2JqZWN0TGlrZSh7XG4gICAgICAgICAgTG9nQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgICAgTG9nRHJpdmVyOiAnYXdzbG9ncycsXG4gICAgICAgICAgICBPcHRpb25zOiB7XG4gICAgICAgICAgICAgICdhd3Nsb2dzLWdyb3VwJzoge1xuICAgICAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICAgICAgJ1JlbmRlclF1ZXVlQ29tbW9uTG9nR3JvdXBXcmFwcGVyQTBFRjcwNTcnLFxuICAgICAgICAgICAgICAgICAgJ0xvZ0dyb3VwTmFtZScsXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgJ2F3c2xvZ3Mtc3RyZWFtLXByZWZpeCc6ICdSQ1MnLFxuICAgICAgICAgICAgICAnYXdzbG9ncy1yZWdpb24nOiB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KSxcbiAgICAgIF0sXG4gICAgfSkpO1xuICB9KTtcblxuICB0ZXN0KCdjaGlsZCBkZXBlbmRlbmNpZXMgYWRkZWQnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBob3N0ID0gbmV3IEluc3RhbmNlKHN0YWNrLCAnSG9zdCcsIHtcbiAgICAgIHZwYyxcbiAgICAgIGluc3RhbmNlVHlwZTogSW5zdGFuY2VUeXBlLm9mKFxuICAgICAgICBJbnN0YW5jZUNsYXNzLlI0LFxuICAgICAgICBJbnN0YW5jZVNpemUuTEFSR0UsXG4gICAgICApLFxuICAgICAgbWFjaGluZUltYWdlOiBNYWNoaW5lSW1hZ2UubGF0ZXN0QW1hem9uTGludXgoeyBnZW5lcmF0aW9uOiBBbWF6b25MaW51eEdlbmVyYXRpb24uQU1BWk9OX0xJTlVYXzIgfSksXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgcmVuZGVyUXVldWVDb21tb24uYWRkQ2hpbGREZXBlbmRlbmN5KGhvc3QpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQzI6Okluc3RhbmNlJywge1xuICAgICAgRGVwZW5kc09uOiBhcnJheVdpdGgoXG4gICAgICAgICdSZW5kZXJRdWV1ZUNvbW1vbkxCUHVibGljTGlzdGVuZXI5MzVGNTYzNScsXG4gICAgICAgICdSZW5kZXJRdWV1ZUNvbW1vblJDU1Rhc2syQTRENUVBNScsXG4gICAgICAgICdSZW5kZXJRdWV1ZUNvbW1vbkFsYkVjMlNlcnZpY2VQYXR0ZXJuU2VydmljZTQyQkVGRjRDJyxcbiAgICAgICAgJ1JlbmRlclF1ZXVlQ29tbW9uV2FpdEZvclN0YWJsZVNlcnZpY2VEQjUzRTI2NicsXG4gICAgICApLFxuICAgIH0sIFJlc291cmNlUGFydC5Db21wbGV0ZURlZmluaXRpb24pKTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ3JlbmRlclF1ZXVlU2l6ZS5taW4nLCAoKSA9PiB7XG4gICAgZGVzY3JpYmUoJ2RlZmF1bHRzIHRvIDEnLCAoKSA9PiB7XG4gICAgICBmdW5jdGlvbiBhc3NlcnRTcGVjaWZpZXNNaW5TaXplKHN0YWNrVG9Bc3NlcnQ6IFN0YWNrKSB7XG4gICAgICAgIGV4cGVjdENESyhzdGFja1RvQXNzZXJ0KS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkF1dG9TY2FsaW5nOjpBdXRvU2NhbGluZ0dyb3VwJywge1xuICAgICAgICAgIE1pblNpemU6ICcxJyxcbiAgICAgICAgfSkpO1xuICAgICAgfVxuXG4gICAgICB0ZXN0KCdyZW5kZXJRdWV1ZVNpemUgdW5zcGVjaWZpZWQnLCAoKSA9PiB7XG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgYXNzZXJ0U3BlY2lmaWVzTWluU2l6ZShzdGFjayk7XG4gICAgICB9KTtcblxuICAgICAgdGVzdCgncmVuZGVyUXVldWVTaXplLm1pbiB1bnNwZWNpZmllZCcsICgpID0+IHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3QgaXNvbGF0ZWRTdGFjayA9IG5ldyBTdGFjayhhcHAsICdJc29sYXRlZFN0YWNrJyk7XG5cbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBuZXcgUmVuZGVyUXVldWUoaXNvbGF0ZWRTdGFjaywgJ1JlbmRlclF1ZXVlJywge1xuICAgICAgICAgIGltYWdlcyxcbiAgICAgICAgICByZXBvc2l0b3J5LFxuICAgICAgICAgIHZlcnNpb246IG5ldyBWZXJzaW9uUXVlcnkoaXNvbGF0ZWRTdGFjaywgJ1ZlcnNpb24nKSxcbiAgICAgICAgICB2cGMsXG4gICAgICAgICAgcmVuZGVyUXVldWVTaXplOiB7fSxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBhc3NlcnRTcGVjaWZpZXNNaW5TaXplKGlzb2xhdGVkU3RhY2spO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICAvLyBBc3NlcnRzIHRoYXQgYXQgbGVhc3Qgb25lIFJDUyBjb250YWluZXIgYW5kIEFTRyBpbnN0YW5jZSBtdXN0IGJlIGNyZWF0ZWQuXG4gICAgdGVzdCgndGhyb3dzIGVycm9yIHdoZW4gbWluaW11bSBzaXplIGlzIDAnLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgcHJvcHM6IFJlbmRlclF1ZXVlUHJvcHMgPSB7XG4gICAgICAgIGltYWdlcyxcbiAgICAgICAgcmVwb3NpdG9yeSxcbiAgICAgICAgdmVyc2lvbjogcmVuZGVyUXVldWVWZXJzaW9uLFxuICAgICAgICB2cGMsXG4gICAgICAgIHJlbmRlclF1ZXVlU2l6ZToge1xuICAgICAgICAgIG1pbjogMCxcbiAgICAgICAgfSxcbiAgICAgIH07XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICAgIG5ldyBSZW5kZXJRdWV1ZShzdGFjaywgJ1JlbmRlclF1ZXVlJywgcHJvcHMpO1xuICAgICAgfSlcbiAgICAgICAgLy8gVEhFTlxuICAgICAgICAudG9UaHJvdygncmVuZGVyUXVldWVTaXplLm1pbiBjYXBhY2l0eSBtdXN0IGJlIGF0IGxlYXN0IDE6IGdvdCAwJyk7XG4gICAgfSk7XG5cbiAgICAvLyBEZWFkbGluZSBiZWZvcmUgMTAuMS4xMCByZXF1aXJlcyB0aGF0IHN1Y2Nlc3NpdmUgQVBJIHJlcXVlc3RzIGFyZSBzZXJ2aWNlZCBieSBhIHNpbmdsZSBSQ1MuXG4gICAgdGVzdCgndmFsaWRhdGVzIERlYWRsaW5lIHByZSAxMC4xLjEwIGhhcyBtaW4gdmFsdWUgb2YgYXQgbW9zdCAxJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IG1pbiA9IDI7XG4gICAgICBjb25zdCBuZXdTdGFjayA9IG5ldyBTdGFjayhhcHAsICdOZXdTdGFjaycpO1xuICAgICAgY29uc3QgdmVyc2lvbk9sZCA9IG5ldyBWZXJzaW9uUXVlcnkobmV3U3RhY2ssICdWZXJzaW9uT2xkJywge3ZlcnNpb246ICcxMC4xLjknfSk7XG4gICAgICBjb25zdCBwcm9wczogUmVuZGVyUXVldWVQcm9wcyA9IHtcbiAgICAgICAgaW1hZ2VzLFxuICAgICAgICByZXBvc2l0b3J5LFxuICAgICAgICB2ZXJzaW9uOiB2ZXJzaW9uT2xkLFxuICAgICAgICB2cGMsXG4gICAgICAgIHJlbmRlclF1ZXVlU2l6ZToge1xuICAgICAgICAgIG1pbixcbiAgICAgICAgfSxcbiAgICAgIH07XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICAgIG5ldyBSZW5kZXJRdWV1ZShuZXdTdGFjaywgJ1JlbmRlclF1ZXVlJywgcHJvcHMpO1xuICAgICAgfSlcbiAgICAgIC8vIFRIRU5cbiAgICAgICAgLnRvVGhyb3coYHJlbmRlclF1ZXVlU2l6ZS5taW4gZm9yIERlYWRsaW5lIHZlcnNpb24gbGVzcyB0aGFuIDEwLjEuMTAuMCBjYW5ub3QgYmUgZ3JlYXRlciB0aGFuIDEgLSBnb3QgJHttaW59YCk7XG4gICAgfSk7XG5cbiAgICAvLyBBc3NlcnRzIHRoYXQgd2hlbiB0aGUgcmVuZGVyUXVldWVTaXplLm1pbiBwcm9wIGlzIHNwZWNpZmllZCwgdGhlIHVuZGVybHlpbmcgQVNHJ3MgbWluIHByb3BlcnR5IGlzIHNldCBhY2NvcmRpbmdseS5cbiAgICB0ZXN0LmVhY2goW1xuICAgICAgWzFdLFxuICAgICAgWzJdLFxuICAgICAgWzEwXSxcbiAgICBdKSgnY29uZmlndXJlcyBtaW5pbXVtIG51bWJlciBvZiBBU0cgaW5zdGFuY2VzIHRvICVkJywgKG1pbjogbnVtYmVyKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgaXNvbGF0ZWRTdGFjayA9IG5ldyBTdGFjayhhcHAsICdJc29sYXRlZFN0YWNrJyk7XG4gICAgICBjb25zdCBwcm9wczogUmVuZGVyUXVldWVQcm9wcyA9IHtcbiAgICAgICAgaW1hZ2VzLFxuICAgICAgICByZXBvc2l0b3J5LFxuICAgICAgICB2ZXJzaW9uOiBuZXcgVmVyc2lvblF1ZXJ5KGlzb2xhdGVkU3RhY2ssICdWZXJzaW9uJyksXG4gICAgICAgIHZwYyxcbiAgICAgICAgcmVuZGVyUXVldWVTaXplOiB7XG4gICAgICAgICAgbWluLFxuICAgICAgICB9LFxuICAgICAgfTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgbmV3IFJlbmRlclF1ZXVlKGlzb2xhdGVkU3RhY2ssICdSZW5kZXJRdWV1ZScsIHByb3BzKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0Q0RLKGlzb2xhdGVkU3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QXV0b1NjYWxpbmc6OkF1dG9TY2FsaW5nR3JvdXAnLCB7XG4gICAgICAgIE1pblNpemU6IG1pbi50b1N0cmluZygpLFxuICAgICAgfSkpO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgncmVuZGVyUXVldWVTaXplLm1heCcsICgpID0+IHtcbiAgICBkZXNjcmliZSgnZGVmYXVsdHMgdG8gMScsICgpID0+IHtcbiAgICAgIGZ1bmN0aW9uIGFzc2VydFNwZWNpZmllc01heFNpemUoc3RhY2tUb0Fzc2VydDogU3RhY2spIHtcbiAgICAgICAgZXhwZWN0Q0RLKHN0YWNrVG9Bc3NlcnQpLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QXV0b1NjYWxpbmc6OkF1dG9TY2FsaW5nR3JvdXAnLCB7XG4gICAgICAgICAgTWF4U2l6ZTogJzEnLFxuICAgICAgICB9KSk7XG4gICAgICB9XG5cbiAgICAgIHRlc3QoJ3JlbmRlclF1ZXVlU2l6ZSB1bnNwZWNpZmllZCcsICgpID0+IHtcbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBhc3NlcnRTcGVjaWZpZXNNYXhTaXplKHN0YWNrKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCdyZW5kZXJRdWV1ZVNpemUubWF4IHVuc3BlY2lmaWVkJywgKCkgPT4ge1xuICAgICAgICAvLyBHSVZFTlxuICAgICAgICBjb25zdCBpc29sYXRlZFN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ0lzb2xhdGVkU3RhY2snKTtcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIG5ldyBSZW5kZXJRdWV1ZShpc29sYXRlZFN0YWNrLCAnUmVuZGVyUXVldWUnLCB7XG4gICAgICAgICAgaW1hZ2VzLFxuICAgICAgICAgIHJlcG9zaXRvcnksXG4gICAgICAgICAgdmVyc2lvbjogbmV3IFZlcnNpb25RdWVyeShpc29sYXRlZFN0YWNrLCAnVmVyc2lvbicpLFxuICAgICAgICAgIHZwYyxcbiAgICAgICAgICByZW5kZXJRdWV1ZVNpemU6IHt9LFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGFzc2VydFNwZWNpZmllc01heFNpemUoaXNvbGF0ZWRTdGFjayk7XG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIC8vIERlYWRsaW5lIGJlZm9yZSAxMC4xLjEwIHJlcXVpcmVzIHRoYXQgc3VjY2Vzc2l2ZSBBUEkgcmVxdWVzdHMgYXJlIHNlcnZpY2VkIGJ5IGEgc2luZ2xlIFJDUy5cbiAgICB0ZXN0KCd2YWxpZGF0ZXMgRGVhZGxpbmUgcHJlIDEwLjEuMTAgaGFzIG1heCB2YWx1ZSBvZiBhdCBtb3N0IDEnLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgbWF4ID0gMjtcbiAgICAgIGNvbnN0IG5ld1N0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ05ld1N0YWNrJyk7XG4gICAgICBjb25zdCB2ZXJzaW9uT2xkID0gbmV3IFZlcnNpb25RdWVyeShuZXdTdGFjaywgJ1ZlcnNpb25PbGQnLCB7dmVyc2lvbjogJzEwLjEuOSd9KTtcbiAgICAgIGNvbnN0IHByb3BzOiBSZW5kZXJRdWV1ZVByb3BzID0ge1xuICAgICAgICBpbWFnZXMsXG4gICAgICAgIHJlcG9zaXRvcnksXG4gICAgICAgIHZlcnNpb246IHZlcnNpb25PbGQsXG4gICAgICAgIHZwYyxcbiAgICAgICAgcmVuZGVyUXVldWVTaXplOiB7XG4gICAgICAgICAgbWF4LFxuICAgICAgICB9LFxuICAgICAgfTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgZXhwZWN0KCgpID0+IHtcbiAgICAgICAgbmV3IFJlbmRlclF1ZXVlKG5ld1N0YWNrLCAnUmVuZGVyUXVldWUnLCBwcm9wcyk7XG4gICAgICB9KVxuICAgICAgLy8gVEhFTlxuICAgICAgICAudG9UaHJvdyhgcmVuZGVyUXVldWVTaXplLm1heCBmb3IgRGVhZGxpbmUgdmVyc2lvbiBsZXNzIHRoYW4gMTAuMS4xMC4wIGNhbm5vdCBiZSBncmVhdGVyIHRoYW4gMSAtIGdvdCAke21heH1gKTtcbiAgICB9KTtcblxuICAgIC8vIEFzc2VydHMgdGhhdCB3aGVuIHRoZSByZW5kZXJRdWV1ZVNpemUubWF4IHByb3AgaXMgc3BlY2lmaWVkLCB0aGUgdW5kZXJseWluZyBBU0cncyBtYXggcHJvcGVydHkgaXMgc2V0IGFjY29yZGluZ2x5LlxuICAgIHRlc3QuZWFjaChbXG4gICAgICBbMV0sXG4gICAgICBbMl0sXG4gICAgICBbMTBdLFxuICAgIF0pKCdjb25maWd1cmVzIG1heGltdW0gbnVtYmVyIG9mIEFTRyBpbnN0YW5jZXMgdG8gJWQnLCAobWF4OiBudW1iZXIpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBpc29sYXRlZFN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ0lzb2xhdGVkU3RhY2snKTtcbiAgICAgIGNvbnN0IHByb3BzOiBSZW5kZXJRdWV1ZVByb3BzID0ge1xuICAgICAgICBpbWFnZXMsXG4gICAgICAgIHJlcG9zaXRvcnksXG4gICAgICAgIHZlcnNpb246IG5ldyBWZXJzaW9uUXVlcnkoaXNvbGF0ZWRTdGFjaywgJ1ZlcnNpb24nKSxcbiAgICAgICAgdnBjLFxuICAgICAgICByZW5kZXJRdWV1ZVNpemU6IHtcbiAgICAgICAgICBtYXgsXG4gICAgICAgIH0sXG4gICAgICB9O1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBuZXcgUmVuZGVyUXVldWUoaXNvbGF0ZWRTdGFjaywgJ1JlbmRlclF1ZXVlJywgcHJvcHMpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3RDREsoaXNvbGF0ZWRTdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBdXRvU2NhbGluZzo6QXV0b1NjYWxpbmdHcm91cCcsIHtcbiAgICAgICAgTWF4U2l6ZTogbWF4LnRvU3RyaW5nKCksXG4gICAgICB9KSk7XG4gICAgfSk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCdyZW5kZXJRdWV1ZVNpemUuZGVzaXJlZCcsICgpID0+IHtcbiAgICBkZXNjcmliZSgnZGVmYXVsdHMnLCAoKSA9PiB7XG4gICAgICB0ZXN0KCd1bnNldCBBU0cgZGVzaXJlZCcsICgpID0+IHtcbiAgICAgICAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkF1dG9TY2FsaW5nOjpBdXRvU2NhbGluZ0dyb3VwJywge1xuICAgICAgICAgIERlc2lyZWRDYXBhY2l0eTogQUJTRU5ULFxuICAgICAgICB9KSk7XG4gICAgICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQ1M6OlNlcnZpY2UnLCB7XG4gICAgICAgICAgRGVzaXJlZENvdW50OiAxLFxuICAgICAgICB9KSk7XG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3ZhbGlkYXRlcyBEZWFkbGluZSBwcmUgMTAuMS4xMCBoYXMgZGVzaXJlZCB2YWx1ZSBvZiBhdCBtb3N0IDEnLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgZGVzaXJlZCA9IDI7XG4gICAgICBjb25zdCBuZXdTdGFjayA9IG5ldyBTdGFjayhhcHAsICdOZXdTdGFjaycpO1xuICAgICAgY29uc3QgdmVyc2lvbk9sZCA9IG5ldyBWZXJzaW9uUXVlcnkobmV3U3RhY2ssICdWZXJzaW9uT2xkJywge3ZlcnNpb246ICcxMC4xLjknfSk7XG4gICAgICBjb25zdCBwcm9wczogUmVuZGVyUXVldWVQcm9wcyA9IHtcbiAgICAgICAgaW1hZ2VzLFxuICAgICAgICByZXBvc2l0b3J5LFxuICAgICAgICB2ZXJzaW9uOiB2ZXJzaW9uT2xkLFxuICAgICAgICB2cGMsXG4gICAgICAgIHJlbmRlclF1ZXVlU2l6ZToge1xuICAgICAgICAgIGRlc2lyZWQsXG4gICAgICAgIH0sXG4gICAgICB9O1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBleHBlY3QoKCkgPT4ge1xuICAgICAgICBuZXcgUmVuZGVyUXVldWUobmV3U3RhY2ssICdSZW5kZXJRdWV1ZScsIHByb3BzKTtcbiAgICAgIH0pXG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgLnRvVGhyb3coYHJlbmRlclF1ZXVlU2l6ZS5kZXNpcmVkIGZvciBEZWFkbGluZSB2ZXJzaW9uIGxlc3MgdGhhbiAxMC4xLjEwLjAgY2Fubm90IGJlIGdyZWF0ZXIgdGhhbiAxIC0gZ290ICR7ZGVzaXJlZH1gKTtcbiAgICB9KTtcblxuICAgIHRlc3QuZWFjaChbXG4gICAgICBbMV0sXG4gICAgICBbMl0sXG4gICAgICBbMTBdLFxuICAgIF0pKCdpcyBzcGVjaWZpZWQgdG8gJWQnLCAoZGVzaXJlZDogbnVtYmVyKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgaXNvbGF0ZWRTdGFjayA9IG5ldyBTdGFjayhhcHAsICdJc29sYXRlZFN0YWNrJyk7XG4gICAgICBjb25zdCBwcm9wczogUmVuZGVyUXVldWVQcm9wcyA9IHtcbiAgICAgICAgaW1hZ2VzLFxuICAgICAgICByZXBvc2l0b3J5LFxuICAgICAgICB2ZXJzaW9uOiBuZXcgVmVyc2lvblF1ZXJ5KGlzb2xhdGVkU3RhY2ssICdWZXJzaW9uJyksXG4gICAgICAgIHZwYyxcbiAgICAgICAgcmVuZGVyUXVldWVTaXplOiB7XG4gICAgICAgICAgZGVzaXJlZCxcbiAgICAgICAgfSxcbiAgICAgIH07XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIG5ldyBSZW5kZXJRdWV1ZShpc29sYXRlZFN0YWNrLCAnUmVuZGVyUXVldWUnLCBwcm9wcyk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkF1dG9TY2FsaW5nOjpBdXRvU2NhbGluZ0dyb3VwJywge1xuICAgICAgICBEZXNpcmVkQ2FwYWNpdHk6IGRlc2lyZWQudG9TdHJpbmcoKSxcbiAgICAgIH0pKTtcbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDUzo6U2VydmljZScsIHtcbiAgICAgICAgRGVzaXJlZENvdW50OiBkZXNpcmVkLFxuICAgICAgfSkpO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgndHJhZmZpY0VuY3J5cHRpb24nLCAoKSA9PiB7XG4gICAgZGVzY3JpYmUoJ2RlZmF1bHRzJywgKCkgPT4ge1xuICAgICAgbGV0IGlzb2xhdGVkU3RhY2s6IFN0YWNrO1xuXG4gICAgICBiZWZvcmVFYWNoKCgpID0+IHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgaXNvbGF0ZWRTdGFjayA9IG5ldyBTdGFjayhhcHAsICdJc29sYXRlZFN0YWNrJyk7XG4gICAgICAgIGNvbnN0IHByb3BzOiBSZW5kZXJRdWV1ZVByb3BzID0ge1xuICAgICAgICAgIGltYWdlcyxcbiAgICAgICAgICByZXBvc2l0b3J5LFxuICAgICAgICAgIHZlcnNpb246IG5ldyBWZXJzaW9uUXVlcnkoaXNvbGF0ZWRTdGFjaywgJ1ZlcnNpb24nKSxcbiAgICAgICAgICB2cGMsXG4gICAgICAgICAgdHJhZmZpY0VuY3J5cHRpb246IHt9LFxuICAgICAgICB9O1xuXG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgbmV3IFJlbmRlclF1ZXVlKGlzb2xhdGVkU3RhY2ssICdSZW5kZXJRdWV1ZScsIHByb3BzKTtcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICB0ZXN0KCd0byBIVFRQUyBpbnRlcm5hbGx5IGJldHdlZW4gQUxCIGFuZCBSQ1MnLCAoKSA9PiB7XG4gICAgICAgIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6OlRhcmdldEdyb3VwJywge1xuICAgICAgICAgIFByb3RvY29sOiAnSFRUUFMnLFxuICAgICAgICAgIFBvcnQ6IDQ0MzMsXG4gICAgICAgIH0pKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCd0byBIVFRQIGV4dGVybmFsbHkgYmV0d2VlbiBjbGllbnRzIGFuZCBBTEInLCAoKSA9PiB7XG4gICAgICAgIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6Okxpc3RlbmVyJywge1xuICAgICAgICAgIFByb3RvY29sOiAnSFRUUCcsXG4gICAgICAgICAgUG9ydDogODA4MCxcbiAgICAgICAgfSkpO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICBkZXNjcmliZSgnd2hlbiBpbnRlcmFsUHJvdG9jb2wgaXMgSFRUUFMnLCAoKSA9PiB7XG4gICAgICBsZXQgaXNvbGF0ZWRTdGFjazogU3RhY2s7XG4gICAgICBsZXQgcmVuZGVyUXVldWU6IFJlbmRlclF1ZXVlO1xuICAgICAgbGV0IGNhQ2VydFBlbUxvZ2ljYWxJZDogc3RyaW5nO1xuICAgICAgbGV0IGNhQ2VydFBrY3NMb2dpY2FsSWQ6IHN0cmluZztcbiAgICAgIGxldCBjYUNlcnRQa2NzUGFzc3BocmFzZUxvZ2ljYWxJZDogc3RyaW5nO1xuXG4gICAgICBiZWZvcmVFYWNoKCgpID0+IHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgaXNvbGF0ZWRTdGFjayA9IG5ldyBTdGFjayhhcHAsICdJc29sYXRlZFN0YWNrJyk7XG4gICAgICAgIGNvbnN0IHByb3BzOiBSZW5kZXJRdWV1ZVByb3BzID0ge1xuICAgICAgICAgIGltYWdlcyxcbiAgICAgICAgICByZXBvc2l0b3J5LFxuICAgICAgICAgIHZlcnNpb246IG5ldyBWZXJzaW9uUXVlcnkoaXNvbGF0ZWRTdGFjaywgJ1ZlcnNpb24nKSxcbiAgICAgICAgICB2cGMsXG4gICAgICAgICAgdHJhZmZpY0VuY3J5cHRpb246IHtcbiAgICAgICAgICAgIGludGVybmFsUHJvdG9jb2w6IEFwcGxpY2F0aW9uUHJvdG9jb2wuSFRUUFMsXG4gICAgICAgICAgfSxcbiAgICAgICAgfTtcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIHJlbmRlclF1ZXVlID0gbmV3IFJlbmRlclF1ZXVlKGlzb2xhdGVkU3RhY2ssICdSZW5kZXJRdWV1ZScsIHByb3BzKTtcblxuICAgICAgICBjYUNlcnRQZW1Mb2dpY2FsSWQgPSBpc29sYXRlZFN0YWNrLmdldExvZ2ljYWxJZChcbiAgICAgICAgICByZW5kZXJRdWV1ZS5ub2RlLmZpbmRDaGlsZCgnVGxzQ2FDZXJ0UGVtJykubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuRWxlbWVudCxcbiAgICAgICAgKTtcbiAgICAgICAgY29uc3QgY2FDZXJ0UGtjcyA9IHJlbmRlclF1ZXVlLm5vZGUuZmluZENoaWxkKCdUbHNSY3NDZXJ0QnVuZGxlJyk7XG4gICAgICAgIGNvbnN0IGNhQ2VydFBrY3NQYXNzcGhyYXNlID0gY2FDZXJ0UGtjcy5ub2RlLmZpbmRDaGlsZCgnUGFzc3BocmFzZScpO1xuICAgICAgICBjYUNlcnRQa2NzTG9naWNhbElkID0gaXNvbGF0ZWRTdGFjay5nZXRMb2dpY2FsSWQoY2FDZXJ0UGtjcy5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5FbGVtZW50KTtcbiAgICAgICAgY2FDZXJ0UGtjc1Bhc3NwaHJhc2VMb2dpY2FsSWQgPSBpc29sYXRlZFN0YWNrLmdldExvZ2ljYWxJZChjYUNlcnRQa2NzUGFzc3BocmFzZS5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5FbGVtZW50KTtcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICB0ZXN0KCdBTEIgY29ubmVjdHMgd2l0aCBIVFRQUyB0byBwb3J0IDQ0MzMnLCAoKSA9PiB7XG4gICAgICAgIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6OlRhcmdldEdyb3VwJywge1xuICAgICAgICAgIFByb3RvY29sOiAnSFRUUFMnLFxuICAgICAgICAgIFBvcnQ6IDQ0MzMsXG4gICAgICAgIH0pKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCdjcmVhdGVzIFJDUyBjZXJ0JywgKCkgPT4ge1xuICAgICAgICBleHBlY3RDREsoaXNvbGF0ZWRTdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQ3VzdG9tOjpSRkRLX1g1MDlHZW5lcmF0b3InLCB7XG4gICAgICAgICAgU2VydmljZVRva2VuOiB7XG4gICAgICAgICAgICAnRm46OkdldEF0dCc6IGFycmF5V2l0aCgnQXJuJyksXG4gICAgICAgICAgfSxcbiAgICAgICAgICBEaXN0aW5ndWlzaGVkTmFtZTogeyBDTjogJ3JlbmRlcmZhcm0ubG9jYWwnIH0sXG4gICAgICAgICAgU2VjcmV0OiB7XG4gICAgICAgICAgICBOYW1lUHJlZml4OiAnSXNvbGF0ZWRTdGFjay9SZW5kZXJRdWV1ZS9UbHNDYUNlcnRQZW0nLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCdncmFudHMgcmVhZCBhY2Nlc3MgdG8gc2VjcmV0cyBjb250YWluaW5nIHRoZSBjZXJ0cyBhbmQgcGFzc3BocmFzZScsICgpID0+IHtcbiAgICAgICAgY29uc3QgdGFza0RlZiA9IHJlbmRlclF1ZXVlLm5vZGUuZmluZENoaWxkKCdSQ1NUYXNrJykgYXMgVGFza0RlZmluaXRpb247XG4gICAgICAgIGNvbnN0IHRhc2tSb2xlTG9naWNhbElkID0gaXNvbGF0ZWRTdGFjay5nZXRMb2dpY2FsSWQoKHRhc2tEZWYudGFza1JvbGUgYXMgUm9sZSkubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuRWxlbWVudCk7XG4gICAgICAgIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgICAgICBTdGF0ZW1lbnQ6IGFycmF5V2l0aChcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAgICAgJ3NlY3JldHNtYW5hZ2VyOkdldFNlY3JldFZhbHVlJyxcbiAgICAgICAgICAgICAgICAgICdzZWNyZXRzbWFuYWdlcjpEZXNjcmliZVNlY3JldCcsXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICAgICAgICBjYUNlcnRQZW1Mb2dpY2FsSWQsXG4gICAgICAgICAgICAgICAgICAgICdDZXJ0JyxcbiAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAgICAgJ3NlY3JldHNtYW5hZ2VyOkdldFNlY3JldFZhbHVlJyxcbiAgICAgICAgICAgICAgICAgICdzZWNyZXRzbWFuYWdlcjpEZXNjcmliZVNlY3JldCcsXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICAgICAgICBjYUNlcnRQa2NzTG9naWNhbElkLFxuICAgICAgICAgICAgICAgICAgICAnQ2VydCcsXG4gICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgICAgICAgICdzZWNyZXRzbWFuYWdlcjpHZXRTZWNyZXRWYWx1ZScsXG4gICAgICAgICAgICAgICAgICAnc2VjcmV0c21hbmFnZXI6RGVzY3JpYmVTZWNyZXQnLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgICAgIFJlc291cmNlOiB7IFJlZjogY2FDZXJ0UGtjc1Bhc3NwaHJhc2VMb2dpY2FsSWQgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICksXG4gICAgICAgICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBSb2xlczogYXJyYXlXaXRoKHsgUmVmOiB0YXNrUm9sZUxvZ2ljYWxJZCB9KSxcbiAgICAgICAgfSkpO1xuICAgICAgfSk7XG5cbiAgICAgIHRlc3QoJ2NvbmZpZ3VyZXMgZW52aXJvbm1lbnQgdmFyaWFibGVzIGZvciBjZXJ0IHNlY3JldCBVUklzJywgKCkgPT4ge1xuICAgICAgICBleHBlY3RDREsoaXNvbGF0ZWRTdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uJywge1xuICAgICAgICAgIENvbnRhaW5lckRlZmluaXRpb25zOiBhcnJheVdpdGgoZGVlcE9iamVjdExpa2Uoe1xuICAgICAgICAgICAgRW52aXJvbm1lbnQ6IGFycmF5V2l0aChcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIE5hbWU6ICdSQ1NfVExTX0NBX0NFUlRfVVJJJyxcbiAgICAgICAgICAgICAgICBWYWx1ZToge1xuICAgICAgICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICAgICAgIGNhQ2VydFBlbUxvZ2ljYWxJZCxcbiAgICAgICAgICAgICAgICAgICAgJ0NlcnQnLFxuICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgTmFtZTogJ1JDU19UTFNfQ0VSVF9VUkknLFxuICAgICAgICAgICAgICAgIFZhbHVlOiB7XG4gICAgICAgICAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICAgICAgICAgY2FDZXJ0UGtjc0xvZ2ljYWxJZCxcbiAgICAgICAgICAgICAgICAgICAgJ0NlcnQnLFxuICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgTmFtZTogJ1JDU19UTFNfQ0VSVF9QQVNTUEhSQVNFX1VSSScsXG4gICAgICAgICAgICAgICAgVmFsdWU6IHsgUmVmOiBjYUNlcnRQa2NzUGFzc3BocmFzZUxvZ2ljYWxJZCB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgKSxcbiAgICAgICAgICB9KSksXG4gICAgICAgIH0pKTtcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgZGVzY3JpYmUoJ3doZW4gaW50ZXJuYWwgcHJvdG9jb2wgaXMgSFRUUCcsICgpID0+IHtcbiAgICAgIGxldCBpc29sYXRlZFN0YWNrOiBTdGFjaztcblxuICAgICAgYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgICAgIC8vIEdJVkVOXG4gICAgICAgIGlzb2xhdGVkU3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnSXNvbGF0ZWRTdGFjaycpO1xuICAgICAgICBjb25zdCBwcm9wczogUmVuZGVyUXVldWVQcm9wcyA9IHtcbiAgICAgICAgICBpbWFnZXMsXG4gICAgICAgICAgcmVwb3NpdG9yeSxcbiAgICAgICAgICB2ZXJzaW9uOiBuZXcgVmVyc2lvblF1ZXJ5KGlzb2xhdGVkU3RhY2ssICdWZXJzaW9uJyksXG4gICAgICAgICAgdnBjLFxuICAgICAgICAgIHRyYWZmaWNFbmNyeXB0aW9uOiB7XG4gICAgICAgICAgICBpbnRlcm5hbFByb3RvY29sOiBBcHBsaWNhdGlvblByb3RvY29sLkhUVFAsXG4gICAgICAgICAgfSxcbiAgICAgICAgfTtcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIG5ldyBSZW5kZXJRdWV1ZShpc29sYXRlZFN0YWNrLCAnUmVuZGVyUXVldWUnLCBwcm9wcyk7XG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgdGVzdCgnbm8gY2VydHMgYXJlIGNyZWF0ZWQnLCAoKSA9PiB7XG4gICAgICAgIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS5ub3RUbyhoYXZlUmVzb3VyY2UoJ0N1c3RvbTo6UkZES19YNTA5R2VuZXJhdG9yJykpO1xuICAgICAgfSk7XG5cbiAgICAgIHRlc3QoJ0FMQiBjb25uZWN0cyB3aXRoIEhUVFAgdG8gcG9ydCA4MDgwJywgKCkgPT4ge1xuICAgICAgICBleHBlY3RDREsoaXNvbGF0ZWRTdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpUYXJnZXRHcm91cCcsIHtcbiAgICAgICAgICBQcm90b2NvbDogJ0hUVFAnLFxuICAgICAgICAgIFBvcnQ6IDgwODAsXG4gICAgICAgIH0pKTtcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgZGVzY3JpYmUoJ2V4dGVybmFsUHJvdG9jb2wgaXMgSFRUUFMnLCAoKSA9PiB7XG4gICAgICBsZXQgaXNvbGF0ZWRTdGFjazogU3RhY2s7XG4gICAgICBjb25zdCBDRVJUX0FSTiA9ICdjZXJ0YXJuJztcbiAgICAgIGNvbnN0IENBX0FSTiA9ICdhcm46YXdzOnNlY3JldHNtYW5hZ2VyOjEyMzQ1Njc4OTAxMjpzZWNyZXQ6Y2EvYXJuJztcbiAgICAgIGNvbnN0IFpPTkVfTkFNRSA9ICdyZW5kZXJmYXJtLmxvY2FsJztcblxuICAgICAgYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgICAgIC8vIEdJVkVOXG4gICAgICAgIGlzb2xhdGVkU3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnSXNvbGF0ZWRTdGFjaycpO1xuICAgICAgICBjb25zdCB6b25lID0gbmV3IFByaXZhdGVIb3N0ZWRab25lKGlzb2xhdGVkU3RhY2ssICdSZW5kZXJRdWV1ZVpvbmUnLCB7XG4gICAgICAgICAgdnBjLFxuICAgICAgICAgIHpvbmVOYW1lOiBaT05FX05BTUUsXG4gICAgICAgIH0pO1xuICAgICAgICBjb25zdCBwcm9wczogUmVuZGVyUXVldWVQcm9wcyA9IHtcbiAgICAgICAgICBpbWFnZXMsXG4gICAgICAgICAgcmVwb3NpdG9yeSxcbiAgICAgICAgICB2ZXJzaW9uOiBuZXcgVmVyc2lvblF1ZXJ5KGlzb2xhdGVkU3RhY2ssICdWZXJzaW9uJyksXG4gICAgICAgICAgdnBjLFxuICAgICAgICAgIHRyYWZmaWNFbmNyeXB0aW9uOiB7XG4gICAgICAgICAgICBleHRlcm5hbFRMUzoge1xuICAgICAgICAgICAgICBhY21DZXJ0aWZpY2F0ZTogQ2VydGlmaWNhdGUuZnJvbUNlcnRpZmljYXRlQXJuKHN0YWNrLCAnQ2VydGlmaWNhdGUnLCBDRVJUX0FSTiksXG4gICAgICAgICAgICAgIGFjbUNlcnRpZmljYXRlQ2hhaW46IFNlY3JldC5mcm9tU2VjcmV0QXJuKHN0YWNrLCAnQ0FfQ2VydCcsIENBX0FSTiksXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAgaG9zdG5hbWU6IHtcbiAgICAgICAgICAgIHpvbmUsXG4gICAgICAgICAgfSxcbiAgICAgICAgfTtcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIG5ldyBSZW5kZXJRdWV1ZShpc29sYXRlZFN0YWNrLCAnUmVuZGVyUXVldWUnLCBwcm9wcyk7XG4gICAgICB9KTtcblxuICAgICAgdGVzdCgnc2V0cyB0aGUgbGlzdGVuZXIgcG9ydCB0byA0NDMzJywgKCkgPT4ge1xuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6Okxpc3RlbmVyJywge1xuICAgICAgICAgIFBvcnQ6IDQ0MzMsXG4gICAgICAgIH0pKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCdzZXRzIHRoZSBsaXN0ZW5lciBwcm90b2NvbCB0byBIVFRQUycsICgpID0+IHtcbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBleHBlY3RDREsoaXNvbGF0ZWRTdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMaXN0ZW5lcicsIHtcbiAgICAgICAgICBQcm90b2NvbDogJ0hUVFBTJyxcbiAgICAgICAgfSkpO1xuICAgICAgfSk7XG5cbiAgICAgIHRlc3QoJ2NvbmZpZ3VyZXMgdGhlIEFMQiBsaXN0ZW5lciB0byB1c2UgdGhlIHNwZWNpZmllZCBBQ00gY2VydGlmaWNhdGUnLCAoKSA9PiB7XG4gICAgICAgIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6Okxpc3RlbmVyJywge1xuICAgICAgICAgIFByb3RvY29sOiAnSFRUUFMnLFxuICAgICAgICAgIENlcnRpZmljYXRlczogYXJyYXlXaXRoKHtcbiAgICAgICAgICAgIENlcnRpZmljYXRlQXJuOiBDRVJUX0FSTixcbiAgICAgICAgICB9KSxcbiAgICAgICAgfSkpO1xuICAgICAgfSk7XG5cbiAgICAgIHRlc3QoJ3JhaXNlcyBhbiBlcnJvciB3aGVuIGEgY2VydCBpcyBzcGVjaWZpZWQgd2l0aG91dCBhIGhvc3RuYW1lJywgKCkgPT4ge1xuICAgICAgICAvLyBHSVZFTlxuICAgICAgICBjb25zdCBwcm9wczogUmVuZGVyUXVldWVQcm9wcyA9IHtcbiAgICAgICAgICBpbWFnZXMsXG4gICAgICAgICAgcmVwb3NpdG9yeSxcbiAgICAgICAgICB2ZXJzaW9uOiByZW5kZXJRdWV1ZVZlcnNpb24sXG4gICAgICAgICAgdnBjLFxuICAgICAgICAgIHRyYWZmaWNFbmNyeXB0aW9uOiB7XG4gICAgICAgICAgICBleHRlcm5hbFRMUzoge1xuICAgICAgICAgICAgICBhY21DZXJ0aWZpY2F0ZTogQ2VydGlmaWNhdGUuZnJvbUNlcnRpZmljYXRlQXJuKHN0YWNrLCAnQ2VydCcsICdjZXJ0QXJuJyksXG4gICAgICAgICAgICAgIGFjbUNlcnRpZmljYXRlQ2hhaW46IFNlY3JldC5mcm9tU2VjcmV0QXJuKHN0YWNrLCAnQ0FfQ2VydDInLCBDQV9BUk4pLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9O1xuXG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgZXhwZWN0KCgpID0+IHtcbiAgICAgICAgICBuZXcgUmVuZGVyUXVldWUoc3RhY2ssICdSZW5kZXJRdWV1ZScsIHByb3BzKTtcbiAgICAgICAgfSlcbiAgICAgICAgICAvLyBUSEVOXG4gICAgICAgICAgLnRvVGhyb3coL0EgaG9zdG5hbWUgbXVzdCBiZSBwcm92aWRlZCB3aGVuIHRoZSBleHRlcm5hbCBwcm90b2NvbCBpcyBIVFRQUy8pO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICBkZXNjcmliZSgnZXh0ZXJuYWxQcm90b2NvbCBpcyBIVFRQUyBpbXBvcnRpbmcgY2VydCcsICgpID0+IHtcbiAgICAgIGxldCBpc29sYXRlZFN0YWNrOiBTdGFjaztcbiAgICAgIGxldCB6b25lOiBQcml2YXRlSG9zdGVkWm9uZTtcbiAgICAgIGNvbnN0IFpPTkVfTkFNRSA9ICdyZW5kZXJmYXJtLmxvY2FsJztcblxuICAgICAgYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgICAgIC8vIEdJVkVOXG4gICAgICAgIGlzb2xhdGVkU3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnSXNvbGF0ZWRTdGFjaycpO1xuICAgICAgICB6b25lID0gbmV3IFByaXZhdGVIb3N0ZWRab25lKGlzb2xhdGVkU3RhY2ssICdSZW5kZXJRdWV1ZVpvbmUnLCB7XG4gICAgICAgICAgdnBjLFxuICAgICAgICAgIHpvbmVOYW1lOiBaT05FX05BTUUsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIGNvbnN0IGNhQ2VydCA9IG5ldyBYNTA5Q2VydGlmaWNhdGVQZW0oaXNvbGF0ZWRTdGFjaywgJ0NhQ2VydCcsIHtcbiAgICAgICAgICBzdWJqZWN0OiB7XG4gICAgICAgICAgICBjbjogYGNhLiR7Wk9ORV9OQU1FfWAsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IHNlcnZlckNlcnQgPSBuZXcgWDUwOUNlcnRpZmljYXRlUGVtKGlzb2xhdGVkU3RhY2ssICdTZXJ2ZXJDZXJ0Jywge1xuICAgICAgICAgIHN1YmplY3Q6IHtcbiAgICAgICAgICAgIGNuOiBgc2VydmVyLiR7Wk9ORV9OQU1FfWAsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBzaWduaW5nQ2VydGlmaWNhdGU6IGNhQ2VydCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgY29uc3QgcHJvcHM6IFJlbmRlclF1ZXVlUHJvcHMgPSB7XG4gICAgICAgICAgaW1hZ2VzLFxuICAgICAgICAgIHJlcG9zaXRvcnksXG4gICAgICAgICAgdmVyc2lvbjogbmV3IFZlcnNpb25RdWVyeShpc29sYXRlZFN0YWNrLCAnVmVyc2lvbicpLFxuICAgICAgICAgIHZwYyxcbiAgICAgICAgICB0cmFmZmljRW5jcnlwdGlvbjoge1xuICAgICAgICAgICAgZXh0ZXJuYWxUTFM6IHtcbiAgICAgICAgICAgICAgcmZka0NlcnRpZmljYXRlOiBzZXJ2ZXJDZXJ0LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIGludGVybmFsUHJvdG9jb2w6IEFwcGxpY2F0aW9uUHJvdG9jb2wuSFRUUCxcbiAgICAgICAgICB9LFxuICAgICAgICAgIGhvc3RuYW1lOiB7XG4gICAgICAgICAgICB6b25lLFxuICAgICAgICAgIH0sXG4gICAgICAgIH07XG5cbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBuZXcgUmVuZGVyUXVldWUoaXNvbGF0ZWRTdGFjaywgJ1JlbmRlclF1ZXVlJywgcHJvcHMpO1xuICAgICAgfSk7XG5cbiAgICAgIHRlc3QoJ3NldHMgdGhlIGxpc3RlbmVyIHBvcnQgdG8gNDQzMycsICgpID0+IHtcbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBleHBlY3RDREsoaXNvbGF0ZWRTdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMaXN0ZW5lcicsIHtcbiAgICAgICAgICBQb3J0OiA0NDMzLFxuICAgICAgICB9KSk7XG4gICAgICB9KTtcblxuICAgICAgdGVzdCgnc2V0cyB0aGUgbGlzdGVuZXIgcHJvdG9jb2wgdG8gSFRUUFMnLCAoKSA9PiB7XG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgZXhwZWN0Q0RLKGlzb2xhdGVkU3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TGlzdGVuZXInLCB7XG4gICAgICAgICAgUHJvdG9jb2w6ICdIVFRQUycsXG4gICAgICAgIH0pKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCdJbXBvcnRzIENlcnQgdG8gQUNNJywgKCkgPT4ge1xuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdDdXN0b206OlJGREtfQWNtSW1wb3J0ZWRDZXJ0aWZpY2F0ZScsIHtcbiAgICAgICAgICBYNTA5Q2VydGlmaWNhdGVQZW06IHtcbiAgICAgICAgICAgIENlcnQ6IHtcbiAgICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICAgJ1NlcnZlckNlcnQnLFxuICAgICAgICAgICAgICAgICdDZXJ0JyxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBLZXk6IHtcbiAgICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICAgJ1NlcnZlckNlcnQnLFxuICAgICAgICAgICAgICAgICdLZXknLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIFBhc3NwaHJhc2U6IHtcbiAgICAgICAgICAgICAgUmVmOiAnU2VydmVyQ2VydFBhc3NwaHJhc2VFNEMzQ0IzOCcsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgQ2VydENoYWluOiB7XG4gICAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICAgICdTZXJ2ZXJDZXJ0JyxcbiAgICAgICAgICAgICAgICAnQ2VydENoYWluJyxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSkpO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdUaHJvd3MgaWYgZ2l2ZW4gQUNNIGNlcnQgYW5kIFJGREsgQ2VydCcsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBpc29sYXRlZFN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ0lzb2xhdGVkU3RhY2snKTtcbiAgICAgIGNvbnN0IFpPTkVfTkFNRSA9ICdyZW5kZXJmYXJtLmxvY2FsJztcbiAgICAgIGNvbnN0IENFUlRfQVJOID0gJ2NlcnRBcm4nO1xuICAgICAgY29uc3QgQ0FfQVJOID0gJ2Fybjphd3M6c2VjcmV0c21hbmFnZXI6MTIzNDU2Nzg5MDEyOnNlY3JldDpjYS9hcm4nO1xuXG4gICAgICBjb25zdCB6b25lID0gbmV3IFByaXZhdGVIb3N0ZWRab25lKGlzb2xhdGVkU3RhY2ssICdSZW5kZXJRdWV1ZVpvbmUnLCB7XG4gICAgICAgIHZwYyxcbiAgICAgICAgem9uZU5hbWU6IFpPTkVfTkFNRSxcbiAgICAgIH0pO1xuXG4gICAgICBjb25zdCBjYUNlcnQgPSBuZXcgWDUwOUNlcnRpZmljYXRlUGVtKGlzb2xhdGVkU3RhY2ssICdDYUNlcnQnLCB7XG4gICAgICAgIHN1YmplY3Q6IHtcbiAgICAgICAgICBjbjogYGNhLiR7Wk9ORV9OQU1FfWAsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICAgIGNvbnN0IHNlcnZlckNlcnQgPSBuZXcgWDUwOUNlcnRpZmljYXRlUGVtKGlzb2xhdGVkU3RhY2ssICdTZXJ2ZXJDZXJ0Jywge1xuICAgICAgICBzdWJqZWN0OiB7XG4gICAgICAgICAgY246IGBzZXJ2ZXIuJHtaT05FX05BTUV9YCxcbiAgICAgICAgfSxcbiAgICAgICAgc2lnbmluZ0NlcnRpZmljYXRlOiBjYUNlcnQsXG4gICAgICB9KTtcblxuICAgICAgY29uc3QgcHJvcHM6IFJlbmRlclF1ZXVlUHJvcHMgPSB7XG4gICAgICAgIGltYWdlcyxcbiAgICAgICAgcmVwb3NpdG9yeSxcbiAgICAgICAgdmVyc2lvbjogbmV3IFZlcnNpb25RdWVyeShpc29sYXRlZFN0YWNrLCAnVmVyc2lvbicpLFxuICAgICAgICB2cGMsXG4gICAgICAgIHRyYWZmaWNFbmNyeXB0aW9uOiB7XG4gICAgICAgICAgZXh0ZXJuYWxUTFM6IHtcbiAgICAgICAgICAgIGFjbUNlcnRpZmljYXRlOiBDZXJ0aWZpY2F0ZS5mcm9tQ2VydGlmaWNhdGVBcm4oaXNvbGF0ZWRTdGFjaywgJ0NlcnRpZmljYXRlJywgQ0VSVF9BUk4pLFxuICAgICAgICAgICAgYWNtQ2VydGlmaWNhdGVDaGFpbjogU2VjcmV0LmZyb21TZWNyZXRBcm4oaXNvbGF0ZWRTdGFjaywgJ0NBX0NlcnQnLCBDQV9BUk4pLFxuICAgICAgICAgICAgcmZka0NlcnRpZmljYXRlOiBzZXJ2ZXJDZXJ0LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIGhvc3RuYW1lOiB7XG4gICAgICAgICAgem9uZSxcbiAgICAgICAgfSxcbiAgICAgIH07XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICAgIG5ldyBSZW5kZXJRdWV1ZShpc29sYXRlZFN0YWNrLCAnUmVuZGVyUXVldWUnLCBwcm9wcyk7XG4gICAgICB9KVxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIC50b1Rocm93KC9FeGFjdGx5IG9uZSBvZiBleHRlcm5hbFRMUy5hY21DZXJ0aWZpY2F0ZSBhbmQgZXh0ZXJuYWxUTFMucmZka0NlcnRpZmljYXRlIG11c3QgYmUgcHJvdmlkZWQgd2hlbiB1c2luZyBleHRlcm5hbFRMUy8pO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnVGhyb3dzIGlmIG5vIENlcnQgZ2l2ZW4nLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgaXNvbGF0ZWRTdGFjayA9IG5ldyBTdGFjayhhcHAsICdJc29sYXRlZFN0YWNrJyk7XG4gICAgICBjb25zdCBaT05FX05BTUUgPSAncmVuZGVyZmFybS5sb2NhbCc7XG5cbiAgICAgIGNvbnN0IHpvbmUgPSBuZXcgUHJpdmF0ZUhvc3RlZFpvbmUoaXNvbGF0ZWRTdGFjaywgJ1JlbmRlclF1ZXVlWm9uZScsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICB6b25lTmFtZTogWk9ORV9OQU1FLFxuICAgICAgfSk7XG5cbiAgICAgIGNvbnN0IHByb3BzOiBSZW5kZXJRdWV1ZVByb3BzID0ge1xuICAgICAgICBpbWFnZXMsXG4gICAgICAgIHJlcG9zaXRvcnksXG4gICAgICAgIHZlcnNpb246IG5ldyBWZXJzaW9uUXVlcnkoaXNvbGF0ZWRTdGFjaywgJ1ZlcnNpb24nKSxcbiAgICAgICAgdnBjLFxuICAgICAgICB0cmFmZmljRW5jcnlwdGlvbjoge1xuICAgICAgICAgIGV4dGVybmFsVExTOiB7XG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAgaG9zdG5hbWU6IHtcbiAgICAgICAgICB6b25lLFxuICAgICAgICB9LFxuICAgICAgfTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgZXhwZWN0KCgpID0+IHtcbiAgICAgICAgbmV3IFJlbmRlclF1ZXVlKGlzb2xhdGVkU3RhY2ssICdSZW5kZXJRdWV1ZScsIHByb3BzKTtcbiAgICAgIH0pXG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgLnRvVGhyb3coL0V4YWN0bHkgb25lIG9mIGV4dGVybmFsVExTLmFjbUNlcnRpZmljYXRlIGFuZCBleHRlcm5hbFRMUy5yZmRrQ2VydGlmaWNhdGUgbXVzdCBiZSBwcm92aWRlZCB3aGVuIHVzaW5nIGV4dGVybmFsVExTLyk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdUaHJvd3MgaWYgQUNNIENlcnQgaXMgZ2l2ZW4gd2l0aG91dCBhIGNlcnQgY2hhaW4nLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgaXNvbGF0ZWRTdGFjayA9IG5ldyBTdGFjayhhcHAsICdJc29sYXRlZFN0YWNrJyk7XG4gICAgICBjb25zdCBaT05FX05BTUUgPSAncmVuZGVyZmFybS5sb2NhbCc7XG4gICAgICBjb25zdCBDRVJUX0FSTiA9ICdjZXJ0QXJuJztcblxuICAgICAgY29uc3Qgem9uZSA9IG5ldyBQcml2YXRlSG9zdGVkWm9uZShpc29sYXRlZFN0YWNrLCAnUmVuZGVyUXVldWVab25lJywge1xuICAgICAgICB2cGMsXG4gICAgICAgIHpvbmVOYW1lOiBaT05FX05BTUUsXG4gICAgICB9KTtcblxuICAgICAgY29uc3QgcHJvcHM6IFJlbmRlclF1ZXVlUHJvcHMgPSB7XG4gICAgICAgIGltYWdlcyxcbiAgICAgICAgcmVwb3NpdG9yeSxcbiAgICAgICAgdmVyc2lvbjogbmV3IFZlcnNpb25RdWVyeShpc29sYXRlZFN0YWNrLCAnVmVyc2lvbicpLFxuICAgICAgICB2cGMsXG4gICAgICAgIHRyYWZmaWNFbmNyeXB0aW9uOiB7XG4gICAgICAgICAgZXh0ZXJuYWxUTFM6IHtcbiAgICAgICAgICAgIGFjbUNlcnRpZmljYXRlOiBDZXJ0aWZpY2F0ZS5mcm9tQ2VydGlmaWNhdGVBcm4oaXNvbGF0ZWRTdGFjaywgJ0NlcnRpZmljYXRlJywgQ0VSVF9BUk4pLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIGhvc3RuYW1lOiB7XG4gICAgICAgICAgem9uZSxcbiAgICAgICAgfSxcbiAgICAgIH07XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICAgIG5ldyBSZW5kZXJRdWV1ZShpc29sYXRlZFN0YWNrLCAnUmVuZGVyUXVldWUnLCBwcm9wcyk7XG4gICAgICB9KVxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIC50b1Rocm93KC9leHRlcm5hbFRMUy5hY21DZXJ0aWZpY2F0ZUNoYWluIG11c3QgYmUgcHJvdmlkZWQgd2hlbiB1c2luZyBleHRlcm5hbFRMUy5hY21DZXJ0aWZpY2F0ZS4vKTtcbiAgICB9KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ0NsaWVudCBDb25uZWN0aW9uJywgKCkgPT4ge1xuICAgIGRlc2NyaWJlKCdleHRlcm5hbFByb3RvY29sIGlzIGh0dHAnLCAoKSA9PiB7XG4gICAgICBsZXQgaXNvbGF0ZWRTdGFjazogU3RhY2s7XG4gICAgICBsZXQgem9uZTogUHJpdmF0ZUhvc3RlZFpvbmU7XG4gICAgICBjb25zdCBaT05FX05BTUUgPSAncmVuZGVyZmFybS5sb2NhbCc7XG4gICAgICBsZXQgcnE6IFJlbmRlclF1ZXVlO1xuXG4gICAgICBiZWZvcmVFYWNoKCgpID0+IHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgaXNvbGF0ZWRTdGFjayA9IG5ldyBTdGFjayhhcHAsICdJc29sYXRlZFN0YWNrJyk7XG4gICAgICAgIHpvbmUgPSBuZXcgUHJpdmF0ZUhvc3RlZFpvbmUoaXNvbGF0ZWRTdGFjaywgJ1JlbmRlclF1ZXVlWm9uZScsIHtcbiAgICAgICAgICB2cGMsXG4gICAgICAgICAgem9uZU5hbWU6IFpPTkVfTkFNRSxcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IHByb3BzOiBSZW5kZXJRdWV1ZVByb3BzID0ge1xuICAgICAgICAgIGltYWdlcyxcbiAgICAgICAgICByZXBvc2l0b3J5LFxuICAgICAgICAgIHZlcnNpb246IG5ldyBWZXJzaW9uUXVlcnkoaXNvbGF0ZWRTdGFjaywgJ1ZlcnNpb24nKSxcbiAgICAgICAgICB2cGMsXG4gICAgICAgICAgaG9zdG5hbWU6IHtcbiAgICAgICAgICAgIHpvbmUsXG4gICAgICAgICAgfSxcbiAgICAgICAgfTtcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIHJxID0gbmV3IFJlbmRlclF1ZXVlKGlzb2xhdGVkU3RhY2ssICdSZW5kZXJRdWV1ZScsIHByb3BzKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCdFQ1MgY2FuIGNvbm5lY3QnLCAoKSA9PiB7XG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgY29uc3QgaG9zdHMgPSBbbmV3IEluc3RhbmNlKGlzb2xhdGVkU3RhY2ssICdIb3N0Jywge1xuICAgICAgICAgIHZwYyxcbiAgICAgICAgICBpbnN0YW5jZVR5cGU6IEluc3RhbmNlVHlwZS5vZihcbiAgICAgICAgICAgIEluc3RhbmNlQ2xhc3MuUjQsXG4gICAgICAgICAgICBJbnN0YW5jZVNpemUuTEFSR0UsXG4gICAgICAgICAgKSxcbiAgICAgICAgICBtYWNoaW5lSW1hZ2U6IE1hY2hpbmVJbWFnZS5sYXRlc3RBbWF6b25MaW51eCh7IGdlbmVyYXRpb246IEFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVhfMiB9KSxcbiAgICAgICAgfSldO1xuICAgICAgICBjb25zdCByb2xlID0gbmV3IFJvbGUoaXNvbGF0ZWRTdGFjaywgJ1JvbGUnLCB7YXNzdW1lZEJ5OiBuZXcgQWNjb3VudFJvb3RQcmluY2lwYWwoKX0pO1xuXG4gICAgICAgIGNvbnN0IGVudiA9IHJxLmNvbmZpZ3VyZUNsaWVudEVDUyh7XG4gICAgICAgICAgaG9zdHMsXG4gICAgICAgICAgZ3JhbnRlZTogcm9sZSxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBleHBlY3QoZW52KS50b0hhdmVQcm9wZXJ0eSgnUkVOREVSX1FVRVVFX1VSSScpO1xuICAgICAgICBleHBlY3QoZW52LlJFTkRFUl9RVUVVRV9VUkkpLnRvTWF0Y2goL2h0dHA6XFwvXFwvLio6ODA4MCQvKTtcblxuICAgICAgICBleHBlY3RDREsoaXNvbGF0ZWRTdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXBJbmdyZXNzJywge1xuICAgICAgICAgIElwUHJvdG9jb2w6ICd0Y3AnLFxuICAgICAgICAgIFRvUG9ydDogODA4MCxcbiAgICAgICAgICBTb3VyY2VTZWN1cml0eUdyb3VwSWQ6IHtcbiAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICBpc29sYXRlZFN0YWNrLmdldExvZ2ljYWxJZChob3N0c1swXS5jb25uZWN0aW9ucy5zZWN1cml0eUdyb3Vwc1swXS5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5FbGVtZW50KSxcbiAgICAgICAgICAgICAgJ0dyb3VwSWQnLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KSk7XG5cbiAgICAgICAgZXhwZWN0Q0RLKGlzb2xhdGVkU3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUMyOjpJbnN0YW5jZScsIHtcbiAgICAgICAgICBEZXBlbmRzT246IGFycmF5V2l0aChcbiAgICAgICAgICAgICdSZW5kZXJRdWV1ZUxCUHVibGljTGlzdGVuZXJCQkYxNUQ1RicsXG4gICAgICAgICAgICAnUmVuZGVyUXVldWVSQ1NUYXNrQTlBRTcwRDMnLFxuICAgICAgICAgICksXG4gICAgICAgIH0sIFJlc291cmNlUGFydC5Db21wbGV0ZURlZmluaXRpb24pKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCdMaW51eCBJbnN0YW5jZSBjYW4gY29ubmVjdCcsICgpID0+IHtcbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBjb25zdCBob3N0ID0gbmV3IEluc3RhbmNlKGlzb2xhdGVkU3RhY2ssICdIb3N0Jywge1xuICAgICAgICAgIHZwYyxcbiAgICAgICAgICBpbnN0YW5jZVR5cGU6IEluc3RhbmNlVHlwZS5vZihcbiAgICAgICAgICAgIEluc3RhbmNlQ2xhc3MuUjQsXG4gICAgICAgICAgICBJbnN0YW5jZVNpemUuTEFSR0UsXG4gICAgICAgICAgKSxcbiAgICAgICAgICBtYWNoaW5lSW1hZ2U6IE1hY2hpbmVJbWFnZS5sYXRlc3RBbWF6b25MaW51eCh7IGdlbmVyYXRpb246IEFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVhfMiB9KSxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgcnEuY29uZmlndXJlQ2xpZW50SW5zdGFuY2Uoe1xuICAgICAgICAgIGhvc3QsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgY29uc3QgdXNlckRhdGEgPSBpc29sYXRlZFN0YWNrLnJlc29sdmUoaG9zdC51c2VyRGF0YS5yZW5kZXIoKSk7XG4gICAgICAgIGV4cGVjdCh1c2VyRGF0YSkudG9TdHJpY3RFcXVhbCh7XG4gICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgJycsXG4gICAgICAgICAgICBbXG4gICAgICAgICAgICAgIFwiIyEvYmluL2Jhc2hcXG5ta2RpciAtcCAkKGRpcm5hbWUgJy90bXAvXCIsXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgIDAsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgJ3x8JyxcbiAgICAgICAgICAgICAgICAgICAgICB7IFJlZjogUlFfQ09OTkVDVElPTl9BU1NFVC5LZXkgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgMSxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICAgIHsgUmVmOiBSUV9DT05ORUNUSU9OX0FTU0VULktleSB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAnXFwnKVxcbmF3cyBzMyBjcCBcXCdzMzovLycsXG4gICAgICAgICAgICAgIHsgUmVmOiBSUV9DT05ORUNUSU9OX0FTU0VULkJ1Y2tldCB9LFxuICAgICAgICAgICAgICAnLycsXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgIDAsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgJ3x8JyxcbiAgICAgICAgICAgICAgICAgICAgICB7IFJlZjogUlFfQ09OTkVDVElPTl9BU1NFVC5LZXkgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgMSxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICAgIHsgUmVmOiBSUV9DT05ORUNUSU9OX0FTU0VULktleSB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAnXFwnIFxcJy90bXAvJyxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgMCxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICAgIHsgUmVmOiBSUV9DT05ORUNUSU9OX0FTU0VULktleSB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpTZWxlY3QnOiBbXG4gICAgICAgICAgICAgICAgICAxLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAnRm46OlNwbGl0JzogW1xuICAgICAgICAgICAgICAgICAgICAgICd8fCcsXG4gICAgICAgICAgICAgICAgICAgICAgeyBSZWY6IFJRX0NPTk5FQ1RJT05fQVNTRVQuS2V5IH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICdcXCdcXG4nICtcbiAgICAgICAgICAgICAgJ2lmIFsgLWYgXFxcIi9ldGMvcHJvZmlsZS5kL2RlYWRsaW5lY2xpZW50LnNoXFxcIiBdOyB0aGVuXFxuJyArXG4gICAgICAgICAgICAgICcgIHNvdXJjZSBcXFwiL2V0Yy9wcm9maWxlLmQvZGVhZGxpbmVjbGllbnQuc2hcXFwiXFxuJyArXG4gICAgICAgICAgICAgICdmaVxcbicgK1xuICAgICAgICAgICAgICAnXCIke0RFQURMSU5FX1BBVEh9L2RlYWRsaW5lY29tbWFuZFwiIC1leGVjdXRlU2NyaXB0Tm9HdWkgXCIvdG1wLycsXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgIDAsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgJ3x8JyxcbiAgICAgICAgICAgICAgICAgICAgICB7IFJlZjogUlFfQ09OTkVDVElPTl9BU1NFVC5LZXkgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgMSxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICAgIHsgUmVmOiBSUV9DT05ORUNUSU9OX0FTU0VULktleSB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBgXCIgLS1yZW5kZXItcXVldWUgXCJodHRwOi8vcmVuZGVycXVldWUuJHtaT05FX05BTUV9OjgwODBcIiBcXG5gICtcbiAgICAgICAgICAgICAgJ3JtIC1mIFwiL3RtcC8nLFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpTZWxlY3QnOiBbXG4gICAgICAgICAgICAgICAgICAwLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAnRm46OlNwbGl0JzogW1xuICAgICAgICAgICAgICAgICAgICAgICd8fCcsXG4gICAgICAgICAgICAgICAgICAgICAgeyBSZWY6IFJRX0NPTk5FQ1RJT05fQVNTRVQuS2V5IH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgIDEsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgJ3x8JyxcbiAgICAgICAgICAgICAgICAgICAgICB7IFJlZjogUlFfQ09OTkVDVElPTl9BU1NFVC5LZXkgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgJ1wiXFxuJyArXG4gICAgICAgICAgICAgICdpZiBzZXJ2aWNlIC0tc3RhdHVzLWFsbCB8IGdyZXAgLXEgXCJEZWFkbGluZSAxMCBMYXVuY2hlclwiOyB0aGVuXFxuJyArXG4gICAgICAgICAgICAgICcgIHNlcnZpY2UgZGVhZGxpbmUxMGxhdW5jaGVyIHJlc3RhcnRcXG4nICtcbiAgICAgICAgICAgICAgJ2ZpJyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgLy8gTWFrZSBzdXJlIHdlIGV4ZWN1dGUgdGhlIHNjcmlwdCB3aXRoIHRoZSBjb3JyZWN0IGFyZ3NcbiAgICAgICAgZXhwZWN0Q0RLKGlzb2xhdGVkU3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUMyOjpTZWN1cml0eUdyb3VwSW5ncmVzcycsIHtcbiAgICAgICAgICBJcFByb3RvY29sOiAndGNwJyxcbiAgICAgICAgICBUb1BvcnQ6IDgwODAsXG4gICAgICAgICAgU291cmNlU2VjdXJpdHlHcm91cElkOiB7XG4gICAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICAgaXNvbGF0ZWRTdGFjay5nZXRMb2dpY2FsSWQoaG9zdC5jb25uZWN0aW9ucy5zZWN1cml0eUdyb3Vwc1swXS5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5FbGVtZW50KSxcbiAgICAgICAgICAgICAgJ0dyb3VwSWQnLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KSk7XG5cbiAgICAgICAgZXhwZWN0Q0RLKGlzb2xhdGVkU3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUMyOjpJbnN0YW5jZScsIHtcbiAgICAgICAgICBEZXBlbmRzT246IGFycmF5V2l0aChcbiAgICAgICAgICAgICdSZW5kZXJRdWV1ZUxCUHVibGljTGlzdGVuZXJCQkYxNUQ1RicsXG4gICAgICAgICAgICAnUmVuZGVyUXVldWVSQ1NUYXNrQTlBRTcwRDMnLFxuICAgICAgICAgICksXG4gICAgICAgIH0sIFJlc291cmNlUGFydC5Db21wbGV0ZURlZmluaXRpb24pKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCdXaW5kb3dzIEluc3RhbmNlIGNhbiBjb25uZWN0JywgKCkgPT4ge1xuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGNvbnN0IGhvc3QgPSBuZXcgSW5zdGFuY2UoaXNvbGF0ZWRTdGFjaywgJ0hvc3QnLCB7XG4gICAgICAgICAgdnBjLFxuICAgICAgICAgIGluc3RhbmNlVHlwZTogSW5zdGFuY2VUeXBlLm9mKFxuICAgICAgICAgICAgSW5zdGFuY2VDbGFzcy5SNCxcbiAgICAgICAgICAgIEluc3RhbmNlU2l6ZS5MQVJHRSxcbiAgICAgICAgICApLFxuICAgICAgICAgIG1hY2hpbmVJbWFnZTogTWFjaGluZUltYWdlLmxhdGVzdFdpbmRvd3MoIFdpbmRvd3NWZXJzaW9uLldJTkRPV1NfU0VSVkVSXzIwMTlfRU5HTElTSF9DT1JFX0JBU0UpLFxuICAgICAgICB9KTtcblxuICAgICAgICBycS5jb25maWd1cmVDbGllbnRJbnN0YW5jZSh7XG4gICAgICAgICAgaG9zdCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBjb25zdCB1c2VyRGF0YSA9IGlzb2xhdGVkU3RhY2sucmVzb2x2ZShob3N0LnVzZXJEYXRhLnJlbmRlcigpKTtcbiAgICAgICAgZXhwZWN0KHVzZXJEYXRhKS50b1N0cmljdEVxdWFsKHtcbiAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAnJyxcbiAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgJzxwb3dlcnNoZWxsPm1rZGlyIChTcGxpdC1QYXRoIC1QYXRoIFxcJ0M6L3RlbXAvJyxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgMCxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICAgIHsgUmVmOiBSUV9DT05ORUNUSU9OX0FTU0VULktleSB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpTZWxlY3QnOiBbXG4gICAgICAgICAgICAgICAgICAxLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAnRm46OlNwbGl0JzogW1xuICAgICAgICAgICAgICAgICAgICAgICd8fCcsXG4gICAgICAgICAgICAgICAgICAgICAgeyBSZWY6IFJRX0NPTk5FQ1RJT05fQVNTRVQuS2V5IH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICdcXCcgKSAtZWEgMFxcbicgK1xuICAgICAgICAgICAgICAnUmVhZC1TM09iamVjdCAtQnVja2V0TmFtZSBcXCcnLFxuICAgICAgICAgICAgICB7IFJlZjogUlFfQ09OTkVDVElPTl9BU1NFVC5CdWNrZXQgfSxcbiAgICAgICAgICAgICAgJ1xcJyAta2V5IFxcJycsXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgIDAsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgJ3x8JyxcbiAgICAgICAgICAgICAgICAgICAgICB7IFJlZjogUlFfQ09OTkVDVElPTl9BU1NFVC5LZXkgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgMSxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICAgIHsgUmVmOiBSUV9DT05ORUNUSU9OX0FTU0VULktleSB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAnXFwnIC1maWxlIFxcJ0M6L3RlbXAvJyxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgMCxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICAgIHsgUmVmOiBSUV9DT05ORUNUSU9OX0FTU0VULktleSB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpTZWxlY3QnOiBbXG4gICAgICAgICAgICAgICAgICAxLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAnRm46OlNwbGl0JzogW1xuICAgICAgICAgICAgICAgICAgICAgICd8fCcsXG4gICAgICAgICAgICAgICAgICAgICAgeyBSZWY6IFJRX0NPTk5FQ1RJT05fQVNTRVQuS2V5IH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICdcXCcgLUVycm9yQWN0aW9uIFN0b3BcXG4nICtcbiAgICAgICAgICAgICAgJyRFcnJvckFjdGlvblByZWZlcmVuY2UgPSBcIlN0b3BcIlxcbicgK1xuICAgICAgICAgICAgICAnJERFQURMSU5FX1BBVEggPSAoZ2V0LWl0ZW0gZW52OlwiREVBRExJTkVfUEFUSFwiKS5WYWx1ZVxcbicgK1xuICAgICAgICAgICAgICAnJiBcIiRERUFETElORV9QQVRIL2RlYWRsaW5lY29tbWFuZC5leGVcIiAtZXhlY3V0ZVNjcmlwdE5vR3VpIFwiQzovdGVtcC8nLFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpTZWxlY3QnOiBbXG4gICAgICAgICAgICAgICAgICAwLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAnRm46OlNwbGl0JzogW1xuICAgICAgICAgICAgICAgICAgICAgICd8fCcsXG4gICAgICAgICAgICAgICAgICAgICAgeyBSZWY6IFJRX0NPTk5FQ1RJT05fQVNTRVQuS2V5IH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgIDEsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgJ3x8JyxcbiAgICAgICAgICAgICAgICAgICAgICB7IFJlZjogUlFfQ09OTkVDVElPTl9BU1NFVC5LZXkgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgYFwiIC0tcmVuZGVyLXF1ZXVlIFwiaHR0cDovL3JlbmRlcnF1ZXVlLiR7Wk9ORV9OQU1FfTo4MDgwXCIgIDI+JjFcXG5gICtcbiAgICAgICAgICAgICAgJ1JlbW92ZS1JdGVtIC1QYXRoIFwiQzovdGVtcC8nLFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpTZWxlY3QnOiBbXG4gICAgICAgICAgICAgICAgICAwLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAnRm46OlNwbGl0JzogW1xuICAgICAgICAgICAgICAgICAgICAgICd8fCcsXG4gICAgICAgICAgICAgICAgICAgICAgeyBSZWY6IFJRX0NPTk5FQ1RJT05fQVNTRVQuS2V5IH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgIDEsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgJ3x8JyxcbiAgICAgICAgICAgICAgICAgICAgICB7IFJlZjogUlFfQ09OTkVDVElPTl9BU1NFVC5LZXkgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgJ1wiXFxuJyArXG4gICAgICAgICAgICAgICdJZiAoR2V0LVNlcnZpY2UgXCJkZWFkbGluZTEwbGF1bmNoZXJzZXJ2aWNlXCIgLUVycm9yQWN0aW9uIFNpbGVudGx5Q29udGludWUpIHtcXG4nICtcbiAgICAgICAgICAgICAgJyAgUmVzdGFydC1TZXJ2aWNlIFwiZGVhZGxpbmUxMGxhdW5jaGVyc2VydmljZVwiXFxuJyArXG4gICAgICAgICAgICAgICd9IEVsc2Uge1xcbicgK1xuICAgICAgICAgICAgICAnICAmIFwiJERFQURMSU5FX1BBVEgvZGVhZGxpbmVsYXVuY2hlci5leGVcIiAtc2h1dGRvd25hbGwgMj4mMVxcbicgK1xuICAgICAgICAgICAgICAnICAmIFwiJERFQURMSU5FX1BBVEgvZGVhZGxpbmVsYXVuY2hlci5leGVcIiAtbm9ndWkgMj4mMVxcbicgK1xuICAgICAgICAgICAgICAnfTwvcG93ZXJzaGVsbD4nLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBNYWtlIHN1cmUgd2UgZXhlY3V0ZSB0aGUgc2NyaXB0IHdpdGggdGhlIGNvcnJlY3QgYXJnc1xuICAgICAgICBleHBlY3RDREsoaXNvbGF0ZWRTdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXBJbmdyZXNzJywge1xuICAgICAgICAgIElwUHJvdG9jb2w6ICd0Y3AnLFxuICAgICAgICAgIFRvUG9ydDogODA4MCxcbiAgICAgICAgICBTb3VyY2VTZWN1cml0eUdyb3VwSWQ6IHtcbiAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICBpc29sYXRlZFN0YWNrLmdldExvZ2ljYWxJZChob3N0LmNvbm5lY3Rpb25zLnNlY3VyaXR5R3JvdXBzWzBdLm5vZGUuZGVmYXVsdENoaWxkIGFzIENmbkVsZW1lbnQpLFxuICAgICAgICAgICAgICAnR3JvdXBJZCcsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pKTtcblxuICAgICAgICBleHBlY3RDREsoaXNvbGF0ZWRTdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQzI6Okluc3RhbmNlJywge1xuICAgICAgICAgIERlcGVuZHNPbjogYXJyYXlXaXRoKFxuICAgICAgICAgICAgJ1JlbmRlclF1ZXVlTEJQdWJsaWNMaXN0ZW5lckJCRjE1RDVGJyxcbiAgICAgICAgICAgICdSZW5kZXJRdWV1ZVJDU1Rhc2tBOUFFNzBEMycsXG4gICAgICAgICAgKSxcbiAgICAgICAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbikpO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICBkZXNjcmliZSgnZXh0ZXJuYWxQcm90b2NvbCBpcyBodHRwcycsICgpID0+IHtcbiAgICAgIGxldCBpc29sYXRlZFN0YWNrOiBTdGFjaztcbiAgICAgIGxldCB6b25lOiBQcml2YXRlSG9zdGVkWm9uZTtcbiAgICAgIGxldCBycTogUmVuZGVyUXVldWU7XG4gICAgICBjb25zdCBaT05FX05BTUUgPSAncmVuZGVyZmFybS5sb2NhbCc7XG4gICAgICBjb25zdCBDRVJUX0FSTiA9ICdhcm46YTpiOmM6ZGNlcnRhcm4nO1xuICAgICAgY29uc3QgQ0FfQVJOID0gJ2Fybjphd3M6c2VjcmV0c21hbmFnZXI6MTIzNDU2Nzg5MDEyOnNlY3JldDpjYS9hcm4nO1xuXG4gICAgICBiZWZvcmVFYWNoKCgpID0+IHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgaXNvbGF0ZWRTdGFjayA9IG5ldyBTdGFjayhhcHAsICdJc29sYXRlZFN0YWNrJyk7XG4gICAgICAgIHpvbmUgPSBuZXcgUHJpdmF0ZUhvc3RlZFpvbmUoaXNvbGF0ZWRTdGFjaywgJ1JlbmRlclF1ZXVlWm9uZScsIHtcbiAgICAgICAgICB2cGMsXG4gICAgICAgICAgem9uZU5hbWU6IFpPTkVfTkFNRSxcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IHByb3BzOiBSZW5kZXJRdWV1ZVByb3BzID0ge1xuICAgICAgICAgIGltYWdlcyxcbiAgICAgICAgICByZXBvc2l0b3J5LFxuICAgICAgICAgIHZlcnNpb246IG5ldyBWZXJzaW9uUXVlcnkoaXNvbGF0ZWRTdGFjaywgJ1ZlcnNpb24nKSxcbiAgICAgICAgICB2cGMsXG4gICAgICAgICAgaG9zdG5hbWU6IHtcbiAgICAgICAgICAgIHpvbmUsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB0cmFmZmljRW5jcnlwdGlvbjoge1xuICAgICAgICAgICAgZXh0ZXJuYWxUTFM6IHtcbiAgICAgICAgICAgICAgYWNtQ2VydGlmaWNhdGU6IENlcnRpZmljYXRlLmZyb21DZXJ0aWZpY2F0ZUFybihzdGFjaywgJ0NlcnRpZmljYXRlJywgQ0VSVF9BUk4pLFxuICAgICAgICAgICAgICBhY21DZXJ0aWZpY2F0ZUNoYWluOiBTZWNyZXQuZnJvbVNlY3JldEFybihzdGFjaywgJ0NBX0NlcnQnLCBDQV9BUk4pLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9O1xuXG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgcnEgPSBuZXcgUmVuZGVyUXVldWUoaXNvbGF0ZWRTdGFjaywgJ1JlbmRlclF1ZXVlJywgcHJvcHMpO1xuICAgICAgfSk7XG5cbiAgICAgIHRlc3QoJ0VDUyBjYW4gY29ubmVjdCcsICgpID0+IHtcbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBjb25zdCBob3N0cyA9IFtuZXcgSW5zdGFuY2UoaXNvbGF0ZWRTdGFjaywgJ0hvc3QnLCB7XG4gICAgICAgICAgdnBjLFxuICAgICAgICAgIGluc3RhbmNlVHlwZTogSW5zdGFuY2VUeXBlLm9mKFxuICAgICAgICAgICAgSW5zdGFuY2VDbGFzcy5SNCxcbiAgICAgICAgICAgIEluc3RhbmNlU2l6ZS5MQVJHRSxcbiAgICAgICAgICApLFxuICAgICAgICAgIG1hY2hpbmVJbWFnZTogTWFjaGluZUltYWdlLmxhdGVzdEFtYXpvbkxpbnV4KHsgZ2VuZXJhdGlvbjogQW1hem9uTGludXhHZW5lcmF0aW9uLkFNQVpPTl9MSU5VWF8yIH0pLFxuICAgICAgICB9KV07XG4gICAgICAgIGNvbnN0IHJvbGUgPSBuZXcgUm9sZShpc29sYXRlZFN0YWNrLCAnUm9sZScsIHthc3N1bWVkQnk6IG5ldyBBY2NvdW50Um9vdFByaW5jaXBhbCgpfSk7XG5cbiAgICAgICAgY29uc3QgZW52ID0gcnEuY29uZmlndXJlQ2xpZW50RUNTKHtcbiAgICAgICAgICBob3N0cyxcbiAgICAgICAgICBncmFudGVlOiByb2xlLFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGV4cGVjdChlbnYpLnRvSGF2ZVByb3BlcnR5KCdSRU5ERVJfUVVFVUVfVVJJJyk7XG4gICAgICAgIGV4cGVjdChlbnYuUkVOREVSX1FVRVVFX1VSSSkudG9NYXRjaCgvaHR0cHM6XFwvXFwvLio6NDQzMyQvKTtcbiAgICAgICAgZXhwZWN0KGVudikudG9IYXZlUHJvcGVydHkoJ1JFTkRFUl9RVUVVRV9UTFNfQ0FfQ0VSVF9VUkknLCBDQV9BUk4pO1xuXG4gICAgICAgIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cEluZ3Jlc3MnLCB7XG4gICAgICAgICAgSXBQcm90b2NvbDogJ3RjcCcsXG4gICAgICAgICAgVG9Qb3J0OiA0NDMzLFxuICAgICAgICAgIFNvdXJjZVNlY3VyaXR5R3JvdXBJZDoge1xuICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgIGlzb2xhdGVkU3RhY2suZ2V0TG9naWNhbElkKGhvc3RzWzBdLmNvbm5lY3Rpb25zLnNlY3VyaXR5R3JvdXBzWzBdLm5vZGUuZGVmYXVsdENoaWxkIGFzIENmbkVsZW1lbnQpLFxuICAgICAgICAgICAgICAnR3JvdXBJZCcsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCdMaW51eCBJbnN0YW5jZSBjYW4gY29ubmVjdCcsICgpID0+IHtcbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBjb25zdCBob3N0ID0gbmV3IEluc3RhbmNlKGlzb2xhdGVkU3RhY2ssICdIb3N0Jywge1xuICAgICAgICAgIHZwYyxcbiAgICAgICAgICBpbnN0YW5jZVR5cGU6IEluc3RhbmNlVHlwZS5vZihcbiAgICAgICAgICAgIEluc3RhbmNlQ2xhc3MuUjQsXG4gICAgICAgICAgICBJbnN0YW5jZVNpemUuTEFSR0UsXG4gICAgICAgICAgKSxcbiAgICAgICAgICBtYWNoaW5lSW1hZ2U6IE1hY2hpbmVJbWFnZS5sYXRlc3RBbWF6b25MaW51eCh7IGdlbmVyYXRpb246IEFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVhfMiB9KSxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgcnEuY29uZmlndXJlQ2xpZW50SW5zdGFuY2Uoe1xuICAgICAgICAgIGhvc3QsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgY29uc3QgdXNlckRhdGEgPSBpc29sYXRlZFN0YWNrLnJlc29sdmUoaG9zdC51c2VyRGF0YS5yZW5kZXIoKSk7XG4gICAgICAgIGV4cGVjdCh1c2VyRGF0YSkudG9TdHJpY3RFcXVhbCh7XG4gICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgJycsXG4gICAgICAgICAgICBbXG4gICAgICAgICAgICAgIFwiIyEvYmluL2Jhc2hcXG5ta2RpciAtcCAkKGRpcm5hbWUgJy90bXAvXCIsXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgIDAsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgJ3x8JyxcbiAgICAgICAgICAgICAgICAgICAgICB7IFJlZjogUlFfQ09OTkVDVElPTl9BU1NFVC5LZXkgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgMSxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICAgIHsgUmVmOiBSUV9DT05ORUNUSU9OX0FTU0VULktleSB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAnXFwnKVxcbmF3cyBzMyBjcCBcXCdzMzovLycsXG4gICAgICAgICAgICAgIHsgUmVmOiBSUV9DT05ORUNUSU9OX0FTU0VULkJ1Y2tldCB9LFxuICAgICAgICAgICAgICAnLycsXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgIDAsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgJ3x8JyxcbiAgICAgICAgICAgICAgICAgICAgICB7IFJlZjogUlFfQ09OTkVDVElPTl9BU1NFVC5LZXkgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgMSxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICAgIHsgUmVmOiBSUV9DT05ORUNUSU9OX0FTU0VULktleSB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAnXFwnIFxcJy90bXAvJyxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgMCxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICAgIHsgUmVmOiBSUV9DT05ORUNUSU9OX0FTU0VULktleSB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpTZWxlY3QnOiBbXG4gICAgICAgICAgICAgICAgICAxLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAnRm46OlNwbGl0JzogW1xuICAgICAgICAgICAgICAgICAgICAgICd8fCcsXG4gICAgICAgICAgICAgICAgICAgICAgeyBSZWY6IFJRX0NPTk5FQ1RJT05fQVNTRVQuS2V5IH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICdcXCdcXG4nICtcbiAgICAgICAgICAgICAgJ2lmIFsgLWYgXFxcIi9ldGMvcHJvZmlsZS5kL2RlYWRsaW5lY2xpZW50LnNoXFxcIiBdOyB0aGVuXFxuJyArXG4gICAgICAgICAgICAgICcgIHNvdXJjZSBcXFwiL2V0Yy9wcm9maWxlLmQvZGVhZGxpbmVjbGllbnQuc2hcXFwiXFxuJyArXG4gICAgICAgICAgICAgICdmaVxcbicgK1xuICAgICAgICAgICAgICAnXCIke0RFQURMSU5FX1BBVEh9L2RlYWRsaW5lY29tbWFuZFwiIC1leGVjdXRlU2NyaXB0Tm9HdWkgXCIvdG1wLycsXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgIDAsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgJ3x8JyxcbiAgICAgICAgICAgICAgICAgICAgICB7IFJlZjogUlFfQ09OTkVDVElPTl9BU1NFVC5LZXkgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgMSxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICAgIHsgUmVmOiBSUV9DT05ORUNUSU9OX0FTU0VULktleSB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBgXCIgLS1yZW5kZXItcXVldWUgXCJodHRwczovL3JlbmRlcnF1ZXVlLiR7Wk9ORV9OQU1FfTo0NDMzXCIgLS10bHMtY2EgXCIke0NBX0FSTn1cIlxcbmAgK1xuICAgICAgICAgICAgICAncm0gLWYgXCIvdG1wLycsXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgIDAsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgJ3x8JyxcbiAgICAgICAgICAgICAgICAgICAgICB7IFJlZjogUlFfQ09OTkVDVElPTl9BU1NFVC5LZXkgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgMSxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICAgIHsgUmVmOiBSUV9DT05ORUNUSU9OX0FTU0VULktleSB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAnXCJcXG4nICtcbiAgICAgICAgICAgICAgJ2lmIHNlcnZpY2UgLS1zdGF0dXMtYWxsIHwgZ3JlcCAtcSBcIkRlYWRsaW5lIDEwIExhdW5jaGVyXCI7IHRoZW5cXG4nICtcbiAgICAgICAgICAgICAgJyAgc2VydmljZSBkZWFkbGluZTEwbGF1bmNoZXIgcmVzdGFydFxcbicgK1xuICAgICAgICAgICAgICAnZmknLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBNYWtlIHN1cmUgd2UgZXhlY3V0ZSB0aGUgc2NyaXB0IHdpdGggdGhlIGNvcnJlY3QgYXJnc1xuICAgICAgICBleHBlY3RDREsoaXNvbGF0ZWRTdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXBJbmdyZXNzJywge1xuICAgICAgICAgIElwUHJvdG9jb2w6ICd0Y3AnLFxuICAgICAgICAgIFRvUG9ydDogNDQzMyxcbiAgICAgICAgICBTb3VyY2VTZWN1cml0eUdyb3VwSWQ6IHtcbiAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICBpc29sYXRlZFN0YWNrLmdldExvZ2ljYWxJZChob3N0LmNvbm5lY3Rpb25zLnNlY3VyaXR5R3JvdXBzWzBdLm5vZGUuZGVmYXVsdENoaWxkIGFzIENmbkVsZW1lbnQpLFxuICAgICAgICAgICAgICAnR3JvdXBJZCcsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCdXaW5kb3dzIEluc3RhbmNlIGNhbiBjb25uZWN0JywgKCkgPT4ge1xuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGNvbnN0IGhvc3QgPSBuZXcgSW5zdGFuY2UoaXNvbGF0ZWRTdGFjaywgJ0hvc3QnLCB7XG4gICAgICAgICAgdnBjLFxuICAgICAgICAgIGluc3RhbmNlVHlwZTogSW5zdGFuY2VUeXBlLm9mKFxuICAgICAgICAgICAgSW5zdGFuY2VDbGFzcy5SNCxcbiAgICAgICAgICAgIEluc3RhbmNlU2l6ZS5MQVJHRSxcbiAgICAgICAgICApLFxuICAgICAgICAgIG1hY2hpbmVJbWFnZTogTWFjaGluZUltYWdlLmxhdGVzdFdpbmRvd3MoIFdpbmRvd3NWZXJzaW9uLldJTkRPV1NfU0VSVkVSXzIwMTlfRU5HTElTSF9DT1JFX0JBU0UpLFxuICAgICAgICB9KTtcblxuICAgICAgICBycS5jb25maWd1cmVDbGllbnRJbnN0YW5jZSh7XG4gICAgICAgICAgaG9zdCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBjb25zdCB1c2VyRGF0YSA9IGlzb2xhdGVkU3RhY2sucmVzb2x2ZShob3N0LnVzZXJEYXRhLnJlbmRlcigpKTtcbiAgICAgICAgZXhwZWN0KHVzZXJEYXRhKS50b1N0cmljdEVxdWFsKHtcbiAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAnJyxcbiAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgJzxwb3dlcnNoZWxsPm1rZGlyIChTcGxpdC1QYXRoIC1QYXRoIFxcJ0M6L3RlbXAvJyxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgMCxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICAgIHsgUmVmOiBSUV9DT05ORUNUSU9OX0FTU0VULktleSB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpTZWxlY3QnOiBbXG4gICAgICAgICAgICAgICAgICAxLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAnRm46OlNwbGl0JzogW1xuICAgICAgICAgICAgICAgICAgICAgICd8fCcsXG4gICAgICAgICAgICAgICAgICAgICAgeyBSZWY6IFJRX0NPTk5FQ1RJT05fQVNTRVQuS2V5IH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICdcXCcgKSAtZWEgMFxcbicgK1xuICAgICAgICAgICAgICAnUmVhZC1TM09iamVjdCAtQnVja2V0TmFtZSBcXCcnLFxuICAgICAgICAgICAgICB7IFJlZjogUlFfQ09OTkVDVElPTl9BU1NFVC5CdWNrZXQgfSxcbiAgICAgICAgICAgICAgJ1xcJyAta2V5IFxcJycsXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgIDAsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgJ3x8JyxcbiAgICAgICAgICAgICAgICAgICAgICB7IFJlZjogUlFfQ09OTkVDVElPTl9BU1NFVC5LZXkgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgMSxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICAgIHsgUmVmOiBSUV9DT05ORUNUSU9OX0FTU0VULktleSB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAnXFwnIC1maWxlIFxcJ0M6L3RlbXAvJyxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgMCxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICAgIHsgUmVmOiBSUV9DT05ORUNUSU9OX0FTU0VULktleSB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpTZWxlY3QnOiBbXG4gICAgICAgICAgICAgICAgICAxLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAnRm46OlNwbGl0JzogW1xuICAgICAgICAgICAgICAgICAgICAgICd8fCcsXG4gICAgICAgICAgICAgICAgICAgICAgeyBSZWY6IFJRX0NPTk5FQ1RJT05fQVNTRVQuS2V5IH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICdcXCcgLUVycm9yQWN0aW9uIFN0b3BcXG4nICtcbiAgICAgICAgICAgICAgJyRFcnJvckFjdGlvblByZWZlcmVuY2UgPSBcIlN0b3BcIlxcbicgK1xuICAgICAgICAgICAgICAnJERFQURMSU5FX1BBVEggPSAoZ2V0LWl0ZW0gZW52OlwiREVBRExJTkVfUEFUSFwiKS5WYWx1ZVxcbicgK1xuICAgICAgICAgICAgICAnJiBcIiRERUFETElORV9QQVRIL2RlYWRsaW5lY29tbWFuZC5leGVcIiAtZXhlY3V0ZVNjcmlwdE5vR3VpIFwiQzovdGVtcC8nLFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpTZWxlY3QnOiBbXG4gICAgICAgICAgICAgICAgICAwLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAnRm46OlNwbGl0JzogW1xuICAgICAgICAgICAgICAgICAgICAgICd8fCcsXG4gICAgICAgICAgICAgICAgICAgICAgeyBSZWY6IFJRX0NPTk5FQ1RJT05fQVNTRVQuS2V5IH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgIDEsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgJ3x8JyxcbiAgICAgICAgICAgICAgICAgICAgICB7IFJlZjogUlFfQ09OTkVDVElPTl9BU1NFVC5LZXkgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgYFwiIC0tcmVuZGVyLXF1ZXVlIFwiaHR0cHM6Ly9yZW5kZXJxdWV1ZS4ke1pPTkVfTkFNRX06NDQzM1wiIC0tdGxzLWNhIFxcXCIke0NBX0FSTn1cXFwiIDI+JjFcXG5gICtcbiAgICAgICAgICAgICAgJ1JlbW92ZS1JdGVtIC1QYXRoIFwiQzovdGVtcC8nLFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpTZWxlY3QnOiBbXG4gICAgICAgICAgICAgICAgICAwLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAnRm46OlNwbGl0JzogW1xuICAgICAgICAgICAgICAgICAgICAgICd8fCcsXG4gICAgICAgICAgICAgICAgICAgICAgeyBSZWY6IFJRX0NPTk5FQ1RJT05fQVNTRVQuS2V5IH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgIDEsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgJ3x8JyxcbiAgICAgICAgICAgICAgICAgICAgICB7IFJlZjogUlFfQ09OTkVDVElPTl9BU1NFVC5LZXkgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgJ1wiXFxuJyArXG4gICAgICAgICAgICAgICdJZiAoR2V0LVNlcnZpY2UgXCJkZWFkbGluZTEwbGF1bmNoZXJzZXJ2aWNlXCIgLUVycm9yQWN0aW9uIFNpbGVudGx5Q29udGludWUpIHtcXG4nICtcbiAgICAgICAgICAgICAgJyAgUmVzdGFydC1TZXJ2aWNlIFwiZGVhZGxpbmUxMGxhdW5jaGVyc2VydmljZVwiXFxuJyArXG4gICAgICAgICAgICAgICd9IEVsc2Uge1xcbicgK1xuICAgICAgICAgICAgICAnICAmIFwiJERFQURMSU5FX1BBVEgvZGVhZGxpbmVsYXVuY2hlci5leGVcIiAtc2h1dGRvd25hbGwgMj4mMVxcbicgK1xuICAgICAgICAgICAgICAnICAmIFwiJERFQURMSU5FX1BBVEgvZGVhZGxpbmVsYXVuY2hlci5leGVcIiAtbm9ndWkgMj4mMVxcbicgK1xuICAgICAgICAgICAgICAnfTwvcG93ZXJzaGVsbD4nLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBNYWtlIHN1cmUgd2UgZXhlY3V0ZSB0aGUgc2NyaXB0IHdpdGggdGhlIGNvcnJlY3QgYXJnc1xuICAgICAgICBleHBlY3RDREsoaXNvbGF0ZWRTdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXBJbmdyZXNzJywge1xuICAgICAgICAgIElwUHJvdG9jb2w6ICd0Y3AnLFxuICAgICAgICAgIFRvUG9ydDogNDQzMyxcbiAgICAgICAgICBTb3VyY2VTZWN1cml0eUdyb3VwSWQ6IHtcbiAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICBpc29sYXRlZFN0YWNrLmdldExvZ2ljYWxJZChob3N0LmNvbm5lY3Rpb25zLnNlY3VyaXR5R3JvdXBzWzBdLm5vZGUuZGVmYXVsdENoaWxkIGFzIENmbkVsZW1lbnQpLFxuICAgICAgICAgICAgICAnR3JvdXBJZCcsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pKTtcbiAgICAgIH0pO1xuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdjYW4gc3BlY2lmeSBzdWJuZXRzJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3VibmV0cyA9IFtcbiAgICAgIFN1Ym5ldC5mcm9tU3VibmV0QXR0cmlidXRlcyhkZXBlbmRlbmN5U3RhY2ssICdTdWJuZXQxJywge1xuICAgICAgICBzdWJuZXRJZDogJ1N1Ym5ldElEMScsXG4gICAgICAgIGF2YWlsYWJpbGl0eVpvbmU6ICd1cy13ZXN0LTJhJyxcbiAgICAgIH0pLFxuICAgICAgU3VibmV0LmZyb21TdWJuZXRBdHRyaWJ1dGVzKGRlcGVuZGVuY3lTdGFjaywgJ1N1Ym5ldDInLCB7XG4gICAgICAgIHN1Ym5ldElkOiAnU3VibmV0SUQyJyxcbiAgICAgICAgYXZhaWxhYmlsaXR5Wm9uZTogJ3VzLXdlc3QtMmInLFxuICAgICAgfSksXG4gICAgXTtcbiAgICBjb25zdCBpc29sYXRlZFN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ0lzb2xhdGVkU3RhY2snKTtcbiAgICBjb25zdCBwcm9wczogUmVuZGVyUXVldWVQcm9wcyA9IHtcbiAgICAgIGltYWdlcyxcbiAgICAgIHJlcG9zaXRvcnksXG4gICAgICB2ZXJzaW9uOiBuZXcgVmVyc2lvblF1ZXJ5KGlzb2xhdGVkU3RhY2ssICdWZXJzaW9uJyksXG4gICAgICB2cGMsXG4gICAgICB2cGNTdWJuZXRzOiB7XG4gICAgICAgIHN1Ym5ldHMsXG4gICAgICB9LFxuICAgICAgdnBjU3VibmV0c0FsYjoge1xuICAgICAgICBzdWJuZXRzLFxuICAgICAgfSxcbiAgICB9O1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBSZW5kZXJRdWV1ZShpc29sYXRlZFN0YWNrLCAnUmVuZGVyUXVldWUnLCBwcm9wcyk7XG5cbiAgICBleHBlY3RDREsoaXNvbGF0ZWRTdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkF1dG9TY2FsaW5nOjpBdXRvU2NhbGluZ0dyb3VwJywge1xuICAgICAgVlBDWm9uZUlkZW50aWZpZXI6IGFycmF5V2l0aChcbiAgICAgICAgJ1N1Ym5ldElEMScsXG4gICAgICAgICdTdWJuZXRJRDInLFxuICAgICAgKSxcbiAgICB9KSk7XG4gICAgZXhwZWN0Q0RLKGlzb2xhdGVkU3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMb2FkQmFsYW5jZXInLCB7XG4gICAgICBTdWJuZXRzOiBbXG4gICAgICAgICdTdWJuZXRJRDEnLFxuICAgICAgICAnU3VibmV0SUQyJyxcbiAgICAgIF0sXG4gICAgfSkpO1xuICB9KTtcblxuICB0ZXN0KCdjYW4gc3BlY2lmeSBpbnN0YW5jZSB0eXBlJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgaXNvbGF0ZWRTdGFjayA9IG5ldyBTdGFjayhhcHAsICdJc29sYXRlZFN0YWNrJyk7XG4gICAgY29uc3QgcHJvcHM6IFJlbmRlclF1ZXVlUHJvcHMgPSB7XG4gICAgICBpbWFnZXMsXG4gICAgICBpbnN0YW5jZVR5cGU6IEluc3RhbmNlVHlwZS5vZihJbnN0YW5jZUNsYXNzLkM1LCBJbnN0YW5jZVNpemUuTEFSR0UpLFxuICAgICAgcmVwb3NpdG9yeSxcbiAgICAgIHZlcnNpb246IG5ldyBWZXJzaW9uUXVlcnkoaXNvbGF0ZWRTdGFjaywgJ1ZlcnNpb24nKSxcbiAgICAgIHZwYyxcbiAgICB9O1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBSZW5kZXJRdWV1ZShpc29sYXRlZFN0YWNrLCAnUmVuZGVyUXVldWUnLCBwcm9wcyk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0Q0RLKGlzb2xhdGVkU3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QXV0b1NjYWxpbmc6OkxhdW5jaENvbmZpZ3VyYXRpb24nLCB7XG4gICAgICBJbnN0YW5jZVR5cGU6ICdjNS5sYXJnZScsXG4gICAgfSkpO1xuICB9KTtcblxuICB0ZXN0KCdubyBkZWxldGlvbiBwcm90ZWN0aW9uJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgaXNvbGF0ZWRTdGFjayA9IG5ldyBTdGFjayhhcHAsICdJc29sYXRlZFN0YWNrJyk7XG4gICAgY29uc3QgcHJvcHM6IFJlbmRlclF1ZXVlUHJvcHMgPSB7XG4gICAgICBpbWFnZXMsXG4gICAgICByZXBvc2l0b3J5LFxuICAgICAgdmVyc2lvbjogbmV3IFZlcnNpb25RdWVyeShpc29sYXRlZFN0YWNrLCAnVmVyc2lvbicpLFxuICAgICAgdnBjLFxuICAgICAgZGVsZXRpb25Qcm90ZWN0aW9uOiBmYWxzZSxcbiAgICB9O1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBSZW5kZXJRdWV1ZShpc29sYXRlZFN0YWNrLCAnUmVuZGVyUXVldWUnLCBwcm9wcyk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0Q0RLKGlzb2xhdGVkU3RhY2spLnRvKG5vdChoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6OkxvYWRCYWxhbmNlcicsIHtcbiAgICAgIExvYWRCYWxhbmNlckF0dHJpYnV0ZXM6IGFycmF5V2l0aChcbiAgICAgICAge1xuICAgICAgICAgIEtleTogJ2RlbGV0aW9uX3Byb3RlY3Rpb24uZW5hYmxlZCcsXG4gICAgICAgICAgVmFsdWU6ICd0cnVlJyxcbiAgICAgICAgfSxcbiAgICAgICksXG4gICAgICBTY2hlbWU6IEFCU0VOVCxcbiAgICAgIFR5cGU6IEFCU0VOVCxcbiAgICB9KSkpO1xuICB9KTtcblxuICB0ZXN0KCdkcm9wIGludmFsaWQgaHR0cCBoZWFkZXIgZmllbGRzIGVuYWJsZWQnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBpc29sYXRlZFN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ0lzb2xhdGVkU3RhY2snKTtcbiAgICBjb25zdCBwcm9wczogUmVuZGVyUXVldWVQcm9wcyA9IHtcbiAgICAgIGltYWdlcyxcbiAgICAgIHJlcG9zaXRvcnksXG4gICAgICB2ZXJzaW9uOiBuZXcgVmVyc2lvblF1ZXJ5KGlzb2xhdGVkU3RhY2ssICdWZXJzaW9uJyksXG4gICAgICB2cGMsXG4gICAgfTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgUmVuZGVyUXVldWUoaXNvbGF0ZWRTdGFjaywgJ1JlbmRlclF1ZXVlJywgcHJvcHMpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6OkxvYWRCYWxhbmNlcicsIHtcbiAgICAgIExvYWRCYWxhbmNlckF0dHJpYnV0ZXM6IGFycmF5V2l0aChcbiAgICAgICAge1xuICAgICAgICAgIEtleTogJ3JvdXRpbmcuaHR0cC5kcm9wX2ludmFsaWRfaGVhZGVyX2ZpZWxkcy5lbmFibGVkJyxcbiAgICAgICAgICBWYWx1ZTogJ3RydWUnLFxuICAgICAgICB9LFxuICAgICAgKSxcbiAgICB9KSk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCdob3N0bmFtZScsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHpvbmVOYW1lID0gJ215ZG9tYWluLmxvY2FsJztcblxuICAgIGRlc2NyaWJlKCdub3Qgc3BlY2lmaWVkJywgKCkgPT4ge1xuICAgICAgbGV0IGlzb2xhdGVkU3RhY2s6IFN0YWNrO1xuXG4gICAgICBiZWZvcmVFYWNoKCgpID0+IHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgaXNvbGF0ZWRTdGFjayA9IG5ldyBTdGFjayhhcHAsICdJc29sYXRlZFN0YWNrJyk7XG4gICAgICAgIGNvbnN0IHByb3BzOiBSZW5kZXJRdWV1ZVByb3BzID0ge1xuICAgICAgICAgIGltYWdlcyxcbiAgICAgICAgICByZXBvc2l0b3J5LFxuICAgICAgICAgIHZlcnNpb246IG5ldyBWZXJzaW9uUXVlcnkoaXNvbGF0ZWRTdGFjaywgJ1ZlcnNpb24nKSxcbiAgICAgICAgICB2cGMsXG4gICAgICAgIH07XG5cbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBuZXcgUmVuZGVyUXVldWUoaXNvbGF0ZWRTdGFjaywgJ1JlbmRlclF1ZXVlJywgcHJvcHMpO1xuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QoJ2RvZXMgbm90IGNyZWF0ZSBhIHJlY29yZCBzZXQnLCAoKSA9PiB7XG4gICAgICAgIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS5ub3RUbyhoYXZlUmVzb3VyY2UoJ0FXUzo6Um91dGU1Mzo6UmVjb3JkU2V0JykpO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICBkZXNjcmliZSgnc3BlY2lmaWVkIHdpdGggem9uZSBidXQgbm8gaG9zdG5hbWUnLCAoKSA9PiB7XG4gICAgICBsZXQgem9uZTogUHJpdmF0ZUhvc3RlZFpvbmU7XG4gICAgICBsZXQgaXNvbGF0ZWRTdGFjazogU3RhY2s7XG4gICAgICBsZXQgcmVuZGVyUXVldWU6IFJlbmRlclF1ZXVlO1xuXG4gICAgICBiZWZvcmVFYWNoKCgpID0+IHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgem9uZSA9IG5ldyBQcml2YXRlSG9zdGVkWm9uZShkZXBlbmRlbmN5U3RhY2ssICdab25lJywge1xuICAgICAgICAgIHZwYyxcbiAgICAgICAgICB6b25lTmFtZSxcbiAgICAgICAgfSk7XG4gICAgICAgIGlzb2xhdGVkU3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnSXNvbGF0ZWRTdGFjaycpO1xuICAgICAgICBjb25zdCBwcm9wczogUmVuZGVyUXVldWVQcm9wcyA9IHtcbiAgICAgICAgICBpbWFnZXMsXG4gICAgICAgICAgcmVwb3NpdG9yeSxcbiAgICAgICAgICB2ZXJzaW9uOiBuZXcgVmVyc2lvblF1ZXJ5KGlzb2xhdGVkU3RhY2ssICdWZXJzaW9uJyksXG4gICAgICAgICAgdnBjLFxuICAgICAgICAgIGhvc3RuYW1lOiB7XG4gICAgICAgICAgICB6b25lLFxuICAgICAgICAgIH0sXG4gICAgICAgIH07XG5cbiAgICAgICAgLy8gV0hFTlxuICAgICAgICByZW5kZXJRdWV1ZSA9IG5ldyBSZW5kZXJRdWV1ZShpc29sYXRlZFN0YWNrLCAnUmVuZGVyUXVldWUnLCBwcm9wcyk7XG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgdGVzdCgnY3JlYXRlcyBhIHJlY29yZCBzZXQgdXNpbmcgZGVmYXVsdCBob3N0bmFtZScsICgpID0+IHtcbiAgICAgICAgY29uc3QgbG9hZEJhbGFuY2VyTG9naWNhbElkID0gZGVwZW5kZW5jeVN0YWNrLmdldExvZ2ljYWxJZChcbiAgICAgICAgICByZW5kZXJRdWV1ZS5sb2FkQmFsYW5jZXIubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuRWxlbWVudCxcbiAgICAgICAgKTtcbiAgICAgICAgZXhwZWN0Q0RLKGlzb2xhdGVkU3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpSb3V0ZTUzOjpSZWNvcmRTZXQnLCB7XG4gICAgICAgICAgTmFtZTogYHJlbmRlcnF1ZXVlLiR7em9uZU5hbWV9LmAsXG4gICAgICAgICAgVHlwZTogJ0EnLFxuICAgICAgICAgIEFsaWFzVGFyZ2V0OiBvYmplY3RMaWtlKHtcbiAgICAgICAgICAgIEhvc3RlZFpvbmVJZDoge1xuICAgICAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICAgICBsb2FkQmFsYW5jZXJMb2dpY2FsSWQsXG4gICAgICAgICAgICAgICAgJ0Nhbm9uaWNhbEhvc3RlZFpvbmVJRCcsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0pLFxuICAgICAgICB9KSk7XG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QuZWFjaChbXG4gICAgICBbJ3JxLnNvbWVkb21haW4ubG9jYWwnXSxcbiAgICAgIFsnMXN0YXJ0c3dpdGhudW1iZXInXSxcbiAgICAgIFsnLXN0YXJ0c3dpdGhoeXBoZW4nXSxcbiAgICAgIFsnZW5kc3dpdGgtJ10sXG4gICAgICBbJ2NvbnRhaW5zJHN5bWJvbCddLFxuICAgICAgWyd0aGlzc3RyaW5naXNleGFjdGx5c2l4dHlmb3VyY2hhcmFjdGVyc3doaWNoaXNvbmVsYXJnZXJ0aGFudGhlbWF4J10sXG4gICAgXSkoJy5ob3N0bmFtZSB2YWxpZGF0aW9uIC0gJXMnLCAoaG9zdG5hbWU6IHN0cmluZykgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHpvbmUgPSBuZXcgUHJpdmF0ZUhvc3RlZFpvbmUoZGVwZW5kZW5jeVN0YWNrLCAnWm9uZScsIHtcbiAgICAgICAgem9uZU5hbWU6ICdzb21lZG9tYWluLmxvY2FsJyxcbiAgICAgICAgdnBjLFxuICAgICAgfSk7XG4gICAgICBjb25zdCBwcm9wczogUmVuZGVyUXVldWVQcm9wcyA9IHtcbiAgICAgICAgaW1hZ2VzLFxuICAgICAgICByZXBvc2l0b3J5LFxuICAgICAgICB2ZXJzaW9uLFxuICAgICAgICB2cGMsXG4gICAgICAgIGhvc3RuYW1lOiB7XG4gICAgICAgICAgaG9zdG5hbWUsXG4gICAgICAgICAgem9uZSxcbiAgICAgICAgfSxcbiAgICAgIH07XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGZ1bmN0aW9uIHdoZW4oKSB7XG4gICAgICAgIG5ldyBSZW5kZXJRdWV1ZShzdGFjaywgJ05ld1JlbmRlclF1ZXVlJywgcHJvcHMpO1xuICAgICAgfVxuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qod2hlbikudG9UaHJvdygvSW52YWxpZCBSZW5kZXJRdWV1ZSBob3N0bmFtZS8pO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgnQWNjZXNzIExvZ3MnLCAoKSA9PiB7XG4gICAgbGV0IGlzb2xhdGVkU3RhY2s6IFN0YWNrO1xuICAgIGxldCBpc29sYXRlZFZwYzogVnBjO1xuICAgIGxldCBpc29sYXRlZFJlcG9zaXRvcnk6IFJlcG9zaXRvcnk7XG4gICAgbGV0IGlzb2xhdGVkVmVyc2lvbjogSVZlcnNpb247XG4gICAgbGV0IGlzb2xhdGVkaW1hZ2VzOiBSZW5kZXJRdWV1ZUltYWdlcztcblxuICAgIGxldCBhY2Nlc3NCdWNrZXQ6IEJ1Y2tldDtcblxuICAgIGJlZm9yZUVhY2goKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGlzb2xhdGVkU3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnSXNvbGF0ZWRTdGFjaycsIHtcbiAgICAgICAgZW52OiB7XG4gICAgICAgICAgcmVnaW9uOiAndXMtZWFzdC0xJyxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgICAgaXNvbGF0ZWRWcGMgPSBuZXcgVnBjKGlzb2xhdGVkU3RhY2ssICdWcGMnKTtcbiAgICAgIGlzb2xhdGVkVmVyc2lvbiA9IG5ldyBWZXJzaW9uUXVlcnkoaXNvbGF0ZWRTdGFjaywgJ1ZlcnNpb24nKTtcblxuICAgICAgaXNvbGF0ZWRSZXBvc2l0b3J5ID0gbmV3IFJlcG9zaXRvcnkoaXNvbGF0ZWRTdGFjaywgJ1JlcG8nLCB7XG4gICAgICAgIHZlcnNpb246IGlzb2xhdGVkVmVyc2lvbixcbiAgICAgICAgdnBjOiBpc29sYXRlZFZwYyxcbiAgICAgIH0pO1xuXG4gICAgICBpc29sYXRlZGltYWdlcyA9IHtcbiAgICAgICAgcmVtb3RlQ29ubmVjdGlvblNlcnZlcjogcmNzSW1hZ2UsXG4gICAgICB9O1xuXG4gICAgICBhY2Nlc3NCdWNrZXQgPSBuZXcgQnVja2V0KGlzb2xhdGVkU3RhY2ssICdBY2Nlc3NCdWNrZXQnKTtcblxuICAgIH0pO1xuXG4gICAgdGVzdCgnZW5hYmxpbmcgYWNjZXNzIGxvZ3Mgc2V0cyBhdHRyaWJ1dGVzIGFuZCBwb2xpY2llcycsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBwcm9wczogUmVuZGVyUXVldWVQcm9wcyA9IHtcbiAgICAgICAgaW1hZ2VzOiBpc29sYXRlZGltYWdlcyxcbiAgICAgICAgcmVwb3NpdG9yeTogaXNvbGF0ZWRSZXBvc2l0b3J5LFxuICAgICAgICB2ZXJzaW9uOiBpc29sYXRlZFZlcnNpb24sXG4gICAgICAgIHZwYzogaXNvbGF0ZWRWcGMsXG4gICAgICAgIGFjY2Vzc0xvZ3M6IHtcbiAgICAgICAgICBkZXN0aW5hdGlvbkJ1Y2tldDogYWNjZXNzQnVja2V0LFxuICAgICAgICB9LFxuICAgICAgfTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgbmV3IFJlbmRlclF1ZXVlKGlzb2xhdGVkU3RhY2ssICdSZW5kZXJRdWV1ZScsIHByb3BzKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0Q0RLKGlzb2xhdGVkU3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TG9hZEJhbGFuY2VyJywge1xuICAgICAgICBMb2FkQmFsYW5jZXJBdHRyaWJ1dGVzOiBhcnJheVdpdGgoXG4gICAgICAgICAge1xuICAgICAgICAgICAgS2V5OiAnYWNjZXNzX2xvZ3MuczMuZW5hYmxlZCcsXG4gICAgICAgICAgICBWYWx1ZTogJ3RydWUnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgS2V5OiAnYWNjZXNzX2xvZ3MuczMuYnVja2V0JyxcbiAgICAgICAgICAgIFZhbHVlOiB7XG4gICAgICAgICAgICAgIFJlZjogJ0FjY2Vzc0J1Y2tldEUyODAzRDc2JyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgKSxcbiAgICAgIH0pKTtcblxuICAgICAgZXhwZWN0Q0RLKGlzb2xhdGVkU3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6UzM6OkJ1Y2tldFBvbGljeScsIHtcbiAgICAgICAgQnVja2V0OiB7XG4gICAgICAgICAgUmVmOiAnQWNjZXNzQnVja2V0RTI4MDNENzYnLFxuICAgICAgICB9LFxuICAgICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICAgIFN0YXRlbWVudDogYXJyYXlXaXRoKFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBBY3Rpb246ICdzMzpQdXRPYmplY3QnLFxuICAgICAgICAgICAgICBDb25kaXRpb246IHtcbiAgICAgICAgICAgICAgICBTdHJpbmdFcXVhbHM6IHtcbiAgICAgICAgICAgICAgICAgICdzMzp4LWFtei1hY2wnOiAnYnVja2V0LW93bmVyLWZ1bGwtY29udHJvbCcsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgICBQcmluY2lwYWw6IHtcbiAgICAgICAgICAgICAgICBTZXJ2aWNlOiAnZGVsaXZlcnkubG9ncy5hbWF6b25hd3MuY29tJyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICAgICAnJyxcbiAgICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICAgICAgICAgICAgJ0FjY2Vzc0J1Y2tldEUyODAzRDc2JyxcbiAgICAgICAgICAgICAgICAgICAgICAgICdBcm4nLFxuICAgICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgICcvKicsXG4gICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBBY3Rpb246ICdzMzpHZXRCdWNrZXRBY2wnLFxuICAgICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICAgIFByaW5jaXBhbDoge1xuICAgICAgICAgICAgICAgIFNlcnZpY2U6ICdkZWxpdmVyeS5sb2dzLmFtYXpvbmF3cy5jb20nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICAgICAgJ0FjY2Vzc0J1Y2tldEUyODAzRDc2JyxcbiAgICAgICAgICAgICAgICAgICdBcm4nLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgICAgICAnczM6UHV0T2JqZWN0KicsXG4gICAgICAgICAgICAgICAgJ3MzOkFib3J0KicsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgICAgUHJpbmNpcGFsOiB7XG4gICAgICAgICAgICAgICAgQVdTOiB7XG4gICAgICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAgICAgJ2FybjonLFxuICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIFJlZjogJ0FXUzo6UGFydGl0aW9uJyxcbiAgICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICAgICc6aWFtOjoxMjczMTE5MjMwMjE6cm9vdCcsXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAgICdBY2Nlc3NCdWNrZXRFMjgwM0Q3NicsXG4gICAgICAgICAgICAgICAgICAgICAgICAnQXJuJyxcbiAgICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICAnL0FXU0xvZ3MvJyxcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgIFJlZjogJ0FXUzo6QWNjb3VudElkJyxcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgJy8qJyxcbiAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgKSxcbiAgICAgICAgfSxcbiAgICAgIH0pKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ2VuYWJsaW5nIGFjY2VzcyBsb2dzIHdvcmtzIHdpdGggcHJlZml4JywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHByb3BzOiBSZW5kZXJRdWV1ZVByb3BzID0ge1xuICAgICAgICBpbWFnZXM6IGlzb2xhdGVkaW1hZ2VzLFxuICAgICAgICByZXBvc2l0b3J5OiBpc29sYXRlZFJlcG9zaXRvcnksXG4gICAgICAgIHZlcnNpb246IGlzb2xhdGVkVmVyc2lvbixcbiAgICAgICAgdnBjOiBpc29sYXRlZFZwYyxcbiAgICAgICAgYWNjZXNzTG9nczoge1xuICAgICAgICAgIGRlc3RpbmF0aW9uQnVja2V0OiBhY2Nlc3NCdWNrZXQsXG4gICAgICAgICAgcHJlZml4OiAnUFJFRklYX1NUUklORycsXG4gICAgICAgIH0sXG4gICAgICB9O1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBuZXcgUmVuZGVyUXVldWUoaXNvbGF0ZWRTdGFjaywgJ1JlbmRlclF1ZXVlJywgcHJvcHMpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3RDREsoaXNvbGF0ZWRTdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMb2FkQmFsYW5jZXInLCB7XG4gICAgICAgIExvYWRCYWxhbmNlckF0dHJpYnV0ZXM6IGFycmF5V2l0aChcbiAgICAgICAgICB7XG4gICAgICAgICAgICBLZXk6ICdhY2Nlc3NfbG9ncy5zMy5lbmFibGVkJyxcbiAgICAgICAgICAgIFZhbHVlOiAndHJ1ZScsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBLZXk6ICdhY2Nlc3NfbG9ncy5zMy5idWNrZXQnLFxuICAgICAgICAgICAgVmFsdWU6IHtcbiAgICAgICAgICAgICAgUmVmOiAnQWNjZXNzQnVja2V0RTI4MDNENzYnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIEtleTogJ2FjY2Vzc19sb2dzLnMzLnByZWZpeCcsXG4gICAgICAgICAgICBWYWx1ZTogJ1BSRUZJWF9TVFJJTkcnLFxuICAgICAgICAgIH0sXG4gICAgICAgICksXG4gICAgICB9KSk7XG5cbiAgICAgIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OlMzOjpCdWNrZXRQb2xpY3knLCB7XG4gICAgICAgIEJ1Y2tldDoge1xuICAgICAgICAgIFJlZjogJ0FjY2Vzc0J1Y2tldEUyODAzRDc2JyxcbiAgICAgICAgfSxcbiAgICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgICBTdGF0ZW1lbnQ6IGFycmF5V2l0aChcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgQWN0aW9uOiAnczM6UHV0T2JqZWN0JyxcbiAgICAgICAgICAgICAgQ29uZGl0aW9uOiB7XG4gICAgICAgICAgICAgICAgU3RyaW5nRXF1YWxzOiB7XG4gICAgICAgICAgICAgICAgICAnczM6eC1hbXotYWNsJzogJ2J1Y2tldC1vd25lci1mdWxsLWNvbnRyb2wnLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgICAgUHJpbmNpcGFsOiB7XG4gICAgICAgICAgICAgICAgU2VydmljZTogJ2RlbGl2ZXJ5LmxvZ3MuYW1hem9uYXdzLmNvbScsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAgICdBY2Nlc3NCdWNrZXRFMjgwM0Q3NicsXG4gICAgICAgICAgICAgICAgICAgICAgICAnQXJuJyxcbiAgICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICAnLyonLFxuICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgQWN0aW9uOiAnczM6R2V0QnVja2V0QWNsJyxcbiAgICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgICBQcmluY2lwYWw6IHtcbiAgICAgICAgICAgICAgICBTZXJ2aWNlOiAnZGVsaXZlcnkubG9ncy5hbWF6b25hd3MuY29tJyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICAgICAgICdBY2Nlc3NCdWNrZXRFMjgwM0Q3NicsXG4gICAgICAgICAgICAgICAgICAnQXJuJyxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICAgJ3MzOlB1dE9iamVjdConLFxuICAgICAgICAgICAgICAgICdzMzpBYm9ydConLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICAgIFByaW5jaXBhbDoge1xuICAgICAgICAgICAgICAgIEFXUzoge1xuICAgICAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICAgICAnJyxcbiAgICAgICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgICAnOmlhbTo6MTI3MzExOTIzMDIxOnJvb3QnLFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgICAnQWNjZXNzQnVja2V0RTI4MDNENzYnLFxuICAgICAgICAgICAgICAgICAgICAgICAgJ0FybicsXG4gICAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgJy9QUkVGSVhfU1RSSU5HL0FXU0xvZ3MvJyxcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgIFJlZjogJ0FXUzo6QWNjb3VudElkJyxcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgJy8qJyxcbiAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgKSxcbiAgICAgICAgfSxcbiAgICAgIH0pKTtcbiAgICB9KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ3RhZ2dpbmcnLCAoKSA9PiB7XG4gICAgdGVzdENvbnN0cnVjdFRhZ3Moe1xuICAgICAgY29uc3RydWN0TmFtZTogJ1JlbmRlclF1ZXVlJyxcbiAgICAgIGNyZWF0ZUNvbnN0cnVjdDogKCkgPT4ge1xuICAgICAgICByZXR1cm4gc3RhY2s7XG4gICAgICB9LFxuICAgICAgcmVzb3VyY2VUeXBlQ291bnRzOiB7XG4gICAgICAgICdBV1M6OkVDUzo6Q2x1c3Rlcic6IDEsXG4gICAgICAgICdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cCc6IDIsXG4gICAgICAgICdBV1M6OklBTTo6Um9sZSc6IDgsXG4gICAgICAgICdBV1M6OkF1dG9TY2FsaW5nOjpBdXRvU2NhbGluZ0dyb3VwJzogMSxcbiAgICAgICAgJ0FXUzo6TGFtYmRhOjpGdW5jdGlvbic6IDQsXG4gICAgICAgICdBV1M6OlNOUzo6VG9waWMnOiAxLFxuICAgICAgICAnQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uJzogMSxcbiAgICAgICAgJ0FXUzo6RHluYW1vREI6OlRhYmxlJzogMixcbiAgICAgICAgJ0FXUzo6U2VjcmV0c01hbmFnZXI6OlNlY3JldCc6IDIsXG4gICAgICAgICdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6OkxvYWRCYWxhbmNlcic6IDEsXG4gICAgICAgICdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6OlRhcmdldEdyb3VwJzogMSxcbiAgICAgICAgJ0FXUzo6RUNTOjpTZXJ2aWNlJzogMSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCdTRVAgUG9saWNpZXMnLCAoKSA9PiB7XG4gICAgdGVzdCgnd2l0aCByZXNvdXJjZSB0cmFja2VyJywgKCkgPT4ge1xuICAgICAgLy8gV0hFTlxuICAgICAgcmVuZGVyUXVldWVDb21tb24uYWRkU0VQUG9saWNpZXMoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0Q0RLKHN0YWNrKS50byhjb3VudFJlc291cmNlc0xpa2UoJ0FXUzo6SUFNOjpSb2xlJywgMSwge1xuICAgICAgICBNYW5hZ2VkUG9saWN5QXJuczogYXJyYXlXaXRoKFxuICAgICAgICAgIHtcbiAgICAgICAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgJzppYW06OmF3czpwb2xpY3kvQVdTVGhpbmtib3hEZWFkbGluZVNwb3RFdmVudFBsdWdpbkFkbWluUG9saWN5JyxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgJ2FybjonLFxuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIFJlZjogJ0FXUzo6UGFydGl0aW9uJyxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICc6aWFtOjphd3M6cG9saWN5L0FXU1RoaW5rYm94RGVhZGxpbmVSZXNvdXJjZVRyYWNrZXJBZG1pblBvbGljeScsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgICksXG4gICAgICB9KSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdubyByZXNvdXJjZSB0cmFja2VyJywgKCkgPT4ge1xuICAgICAgLy8gV0hFTlxuICAgICAgcmVuZGVyUXVldWVDb21tb24uYWRkU0VQUG9saWNpZXMoZmFsc2UpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6SUFNOjpSb2xlJywge1xuICAgICAgICBNYW5hZ2VkUG9saWN5QXJuczogYXJyYXlXaXRoKFxuICAgICAgICAgIHtcbiAgICAgICAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgJzppYW06OmF3czpwb2xpY3kvQVdTVGhpbmtib3hEZWFkbGluZVNwb3RFdmVudFBsdWdpbkFkbWluUG9saWN5JyxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgfSxcbiAgICAgICAgKSxcbiAgICAgIH0pKTtcbiAgICAgIGV4cGVjdENESyhzdGFjaykubm90VG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpJQU06OlJvbGUnLCB7XG4gICAgICAgIE1hbmFnZWRQb2xpY3lBcm5zOiBhcnJheVdpdGgoXG4gICAgICAgICAge1xuICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAnJyxcbiAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAnOmlhbTo6YXdzOnBvbGljeS9BV1NUaGlua2JveERlYWRsaW5lUmVzb3VyY2VUcmFja2VyQWRtaW5Qb2xpY3knLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICApLFxuICAgICAgfSkpO1xuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdjcmVhdGVzIFdhaXRGb3JTdGFibGVTZXJ2aWNlIGJ5IGRlZmF1bHQnLCAoKSA9PiB7XG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQ3VzdG9tOjpSRkRLX1dhaXRGb3JTdGFibGVTZXJ2aWNlJywge1xuICAgICAgY2x1c3Rlcjogc3RhY2sucmVzb2x2ZShyZW5kZXJRdWV1ZUNvbW1vbi5jbHVzdGVyLmNsdXN0ZXJBcm4pLFxuICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGRvdC1ub3RhdGlvblxuICAgICAgc2VydmljZXM6IFtzdGFjay5yZXNvbHZlKHJlbmRlclF1ZXVlQ29tbW9uWydwYXR0ZXJuJ10uc2VydmljZS5zZXJ2aWNlQXJuKV0sXG4gICAgfSkpO1xuICB9KTtcblxuICBkZXNjcmliZSgnU2VjdXJpdHkgR3JvdXBzJywgKCkgPT4ge1xuICAgIGxldCBiYWNrZW5kU2VjdXJpdHlHcm91cDogU2VjdXJpdHlHcm91cDtcbiAgICBsZXQgZnJvbnRlbmRTZWN1cml0eUdyb3VwOiBTZWN1cml0eUdyb3VwO1xuXG4gICAgYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgICBiYWNrZW5kU2VjdXJpdHlHcm91cCA9IG5ldyBTZWN1cml0eUdyb3VwKHN0YWNrLCAnQVNHU2VjdXJpdHlHcm91cCcsIHsgdnBjIH0pO1xuICAgICAgZnJvbnRlbmRTZWN1cml0eUdyb3VwID0gbmV3IFNlY3VyaXR5R3JvdXAoc3RhY2ssICdMQlNlY3VyaXR5R3JvdXAnLCB7IHZwYyB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ2FkZHMgc2VjdXJpdHkgZ3JvdXBzIG9uIGNvbnN0cnVjdGlvbicsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzZWN1cml0eUdyb3VwczogUmVuZGVyUXVldWVTZWN1cml0eUdyb3VwcyA9IHtcbiAgICAgICAgYmFja2VuZDogYmFja2VuZFNlY3VyaXR5R3JvdXAsXG4gICAgICAgIGZyb250ZW5kOiBmcm9udGVuZFNlY3VyaXR5R3JvdXAsXG4gICAgICB9O1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBuZXcgUmVuZGVyUXVldWUoc3RhY2ssICdSZW5kZXJRdWV1ZScsIHtcbiAgICAgICAgaW1hZ2VzLFxuICAgICAgICByZXBvc2l0b3J5LFxuICAgICAgICB2ZXJzaW9uOiByZW5kZXJRdWV1ZVZlcnNpb24sXG4gICAgICAgIHZwYyxcbiAgICAgICAgc2VjdXJpdHlHcm91cHMsXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgYXNzZXJ0U2VjdXJpdHlHcm91cHNXZXJlQWRkZWQoc2VjdXJpdHlHcm91cHMpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnYWRkcyBiYWNrZW5kIHNlY3VyaXR5IGdyb3VwcyBwb3N0LWNvbnN0cnVjdGlvbicsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCByZW5kZXJRdWV1ZSA9IG5ldyBSZW5kZXJRdWV1ZShzdGFjaywgJ1JlbmRlclF1ZXVlJywge1xuICAgICAgICBpbWFnZXMsXG4gICAgICAgIHJlcG9zaXRvcnksXG4gICAgICAgIHZlcnNpb246IHJlbmRlclF1ZXVlVmVyc2lvbixcbiAgICAgICAgdnBjLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIHJlbmRlclF1ZXVlLmFkZEJhY2tlbmRTZWN1cml0eUdyb3VwcyhiYWNrZW5kU2VjdXJpdHlHcm91cCk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGFzc2VydFNlY3VyaXR5R3JvdXBzV2VyZUFkZGVkKHtcbiAgICAgICAgYmFja2VuZDogYmFja2VuZFNlY3VyaXR5R3JvdXAsXG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ2FkZHMgZnJvbnRlbmQgc2VjdXJpdHkgZ3JvdXBzIHBvc3QtY29uc3RydWN0aW9uJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHJlbmRlclF1ZXVlID0gbmV3IFJlbmRlclF1ZXVlKHN0YWNrLCAnUmVuZGVyUXVldWUnLCB7XG4gICAgICAgIGltYWdlcyxcbiAgICAgICAgcmVwb3NpdG9yeSxcbiAgICAgICAgdmVyc2lvbjogcmVuZGVyUXVldWVWZXJzaW9uLFxuICAgICAgICB2cGMsXG4gICAgICB9KTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgcmVuZGVyUXVldWUuYWRkRnJvbnRlbmRTZWN1cml0eUdyb3Vwcyhmcm9udGVuZFNlY3VyaXR5R3JvdXApO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBhc3NlcnRTZWN1cml0eUdyb3Vwc1dlcmVBZGRlZCh7XG4gICAgICAgIGZyb250ZW5kOiBmcm9udGVuZFNlY3VyaXR5R3JvdXAsXG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3NlY3VyaXR5IGdyb3VwcyBhZGRlZCBwb3N0LWNvbnN0cnVjdGlvbiBhcmUgbm90IGF0dGFjaGVkIHRvIENvbm5lY3Rpb25zIG9iamVjdCcsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCByZW5kZXJRdWV1ZSA9IG5ldyBSZW5kZXJRdWV1ZShzdGFjaywgJ1JlbmRlclF1ZXVlJywge1xuICAgICAgICBpbWFnZXMsXG4gICAgICAgIHJlcG9zaXRvcnksXG4gICAgICAgIHZlcnNpb246IHJlbmRlclF1ZXVlVmVyc2lvbixcbiAgICAgICAgdnBjLFxuICAgICAgfSk7XG4gICAgICByZW5kZXJRdWV1ZS5hZGRCYWNrZW5kU2VjdXJpdHlHcm91cHMoYmFja2VuZFNlY3VyaXR5R3JvdXApO1xuICAgICAgcmVuZGVyUXVldWUuYWRkRnJvbnRlbmRTZWN1cml0eUdyb3Vwcyhmcm9udGVuZFNlY3VyaXR5R3JvdXApO1xuICAgICAgY29uc3QgcGVlclNlY3VyaXR5R3JvdXAgPSBuZXcgU2VjdXJpdHlHcm91cChzdGFjaywgJ1BlZXJTZWN1cml0eUdyb3VwJywgeyB2cGMgfSk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIHJlbmRlclF1ZXVlLmNvbm5lY3Rpb25zLmFsbG93RnJvbShwZWVyU2VjdXJpdHlHcm91cCwgUG9ydC50Y3AoMjIpKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgLy8gRXhpc3RpbmcgTG9hZEJhbGFuY2VyIHNlY3VyaXR5IGdyb3VwcyBzaG91bGRuJ3QgaGF2ZSB0aGUgaW5ncmVzcyBydWxlIGFkZGVkXG4gICAgICBleHBlY3RDREsoc3RhY2spLm5vdFRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUMyOjpTZWN1cml0eUdyb3VwSW5ncmVzcycsIHtcbiAgICAgICAgSXBQcm90b2NvbDogJ3RjcCcsXG4gICAgICAgIEZyb21Qb3J0OiAyMixcbiAgICAgICAgVG9Qb3J0OiAyMixcbiAgICAgICAgR3JvdXBJZDogc3RhY2sucmVzb2x2ZShmcm9udGVuZFNlY3VyaXR5R3JvdXAuc2VjdXJpdHlHcm91cElkKSxcbiAgICAgICAgU291cmNlU2VjdXJpdHlHcm91cElkOiBzdGFjay5yZXNvbHZlKHBlZXJTZWN1cml0eUdyb3VwLnNlY3VyaXR5R3JvdXBJZCksXG4gICAgICB9KSk7XG4gICAgICAvLyBFeGlzdGluZyBBdXRvU2NhbGluZ0dyb3VwIHNlY3VyaXR5IGdyb3VwcyBzaG91bGRuJ3QgaGF2ZSB0aGUgaW5ncmVzcyBydWxlIGFkZGVkXG4gICAgICBleHBlY3RDREsoc3RhY2spLm5vdFRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUMyOjpTZWN1cml0eUdyb3VwSW5ncmVzcycsIHtcbiAgICAgICAgSXBQcm90b2NvbDogJ3RjcCcsXG4gICAgICAgIEZyb21Qb3J0OiAyMixcbiAgICAgICAgVG9Qb3J0OiAyMixcbiAgICAgICAgR3JvdXBJZDogc3RhY2sucmVzb2x2ZShiYWNrZW5kU2VjdXJpdHlHcm91cC5zZWN1cml0eUdyb3VwSWQpLFxuICAgICAgICBTb3VyY2VTZWN1cml0eUdyb3VwSWQ6IHN0YWNrLnJlc29sdmUocGVlclNlY3VyaXR5R3JvdXAuc2VjdXJpdHlHcm91cElkKSxcbiAgICAgIH0pKTtcbiAgICB9KTtcblxuICAgIGZ1bmN0aW9uIGFzc2VydFNlY3VyaXR5R3JvdXBzV2VyZUFkZGVkKHNlY3VyaXR5R3JvdXBzOiBSZW5kZXJRdWV1ZVNlY3VyaXR5R3JvdXBzKSB7XG4gICAgICBpZiAoc2VjdXJpdHlHcm91cHMuYmFja2VuZCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBdXRvU2NhbGluZzo6TGF1bmNoQ29uZmlndXJhdGlvbicsIHtcbiAgICAgICAgICBTZWN1cml0eUdyb3VwczogYXJyYXlXaXRoKHN0YWNrLnJlc29sdmUoc2VjdXJpdHlHcm91cHMuYmFja2VuZC5zZWN1cml0eUdyb3VwSWQpKSxcbiAgICAgICAgfSkpO1xuICAgICAgfVxuICAgICAgaWYgKHNlY3VyaXR5R3JvdXBzLmZyb250ZW5kICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6OkxvYWRCYWxhbmNlcicsIHtcbiAgICAgICAgICBTZWN1cml0eUdyb3VwczogYXJyYXlXaXRoKHN0YWNrLnJlc29sdmUoc2VjdXJpdHlHcm91cHMuZnJvbnRlbmQuc2VjdXJpdHlHcm91cElkKSksXG4gICAgICAgIH0pKTtcbiAgICAgIH1cbiAgICB9XG4gIH0pO1xuXG4gIHRlc3QoJ3ZhbGlkYXRlcyBWZXJzaW9uUXVlcnkgaXMgbm90IGluIGEgZGlmZmVyZW50IHN0YWNrJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgbmV3U3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnTmV3U3RhY2snKTtcbiAgICAvLyBXSEVOXG4gICAgbmV3IFJlbmRlclF1ZXVlKG5ld1N0YWNrLCAnUmVuZGVyUXVldWVOZXcnLCB7XG4gICAgICBpbWFnZXMsXG4gICAgICByZXBvc2l0b3J5LFxuICAgICAgdmVyc2lvbixcbiAgICAgIHZwYyxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBmdW5jdGlvbiBzeW50aCgpIHtcbiAgICAgIFN5bnRoVXRpbHMuc3ludGhlc2l6ZShuZXdTdGFjayk7XG4gICAgfVxuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzeW50aCkudG9UaHJvdygnQSBWZXJzaW9uUXVlcnkgY2FuIG5vdCBiZSBzdXBwbGllZCBmcm9tIGEgZGlmZmVyZW50IHN0YWNrJyk7XG4gIH0pO1xuXG4gIHRlc3QoJ0RvZXMgbm90IGVuYWJsZSBmaWxlc3lzdGVtIGNhY2hlIGJ5IGRlZmF1bHQnLCAoKSA9PiB7XG4gICAgZXhwZWN0Q0RLKHN0YWNrKS5ub3RUbyhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkF1dG9TY2FsaW5nOjpMYXVuY2hDb25maWd1cmF0aW9uJywge1xuICAgICAgVXNlckRhdGE6IHtcbiAgICAgICAgJ0ZuOjpCYXNlNjQnOiB7XG4gICAgICAgICAgJ0ZuOjpKb2luJzogYXJyYXlXaXRoKGFycmF5V2l0aCgnID4+IC9ldGMvZWNzL2Vjcy5jb25maWdcXG5zdWRvIGlwdGFibGVzIC0taW5zZXJ0IEZPUldBUkQgMSAtLWluLWludGVyZmFjZSBkb2NrZXIrIC0tZGVzdGluYXRpb24gMTY5LjI1NC4xNjkuMjU0LzMyIC0tanVtcCBEUk9QXFxuc3VkbyBzZXJ2aWNlIGlwdGFibGVzIHNhdmVcXG5lY2hvIEVDU19BV1NWUENfQkxPQ0tfSU1EUz10cnVlID4+IC9ldGMvZWNzL2Vjcy5jb25maWdcXG55dW0gaW5zdGFsbCAteXEgYXdzY2xpIHVuemlwXFxuIyBSZW5kZXJRdWV1ZSBmaWxlIGNhY2hpbmcgZW5hYmxlZFxcbm1rZGlyIC1wICQoZGlybmFtZSBcXCcvdG1wLycpKSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSkpO1xuICB9KTtcblxuICB0ZXN0KCdFbmFibGVzIGZpbGVzeXN0ZW0gY2FjaGUgaWYgcmVxdWlyZWQnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBpc29sYXRlZFN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ0lzb2xhdGVkU3RhY2snKTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgUmVuZGVyUXVldWUoaXNvbGF0ZWRTdGFjaywgJ1JlbmRlclF1ZXVlJywge1xuICAgICAgaW1hZ2VzLFxuICAgICAgcmVwb3NpdG9yeSxcbiAgICAgIHZlcnNpb246IG5ldyBWZXJzaW9uUXVlcnkoaXNvbGF0ZWRTdGFjaywgJ1ZlcnNpb24nKSxcbiAgICAgIHZwYyxcbiAgICAgIGVuYWJsZUxvY2FsRmlsZUNhY2hpbmc6IHRydWUsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgLy8gTm90ZTogSWYgdGhpcyB0ZXN0IGJyZWFrcy9mYWlscywgdGhlbiBpdCBpcyBwcm9iYWJsZSB0aGF0IHRoZVxuICAgIC8vICAnRG9lcyBub3QgZW5hYmxlIGZpbGVzeXN0ZW0gY2FjaGUgYnkgZGVmYXVsdCcgdGVzdCBhYm92ZSB3aWxsIGFsc28gcmVxdWlyZVxuICAgIC8vICB1cGRhdGluZy9maXhpbmcuXG4gICAgZXhwZWN0Q0RLKGlzb2xhdGVkU3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QXV0b1NjYWxpbmc6OkxhdW5jaENvbmZpZ3VyYXRpb24nLCB7XG4gICAgICBVc2VyRGF0YToge1xuICAgICAgICAnRm46OkJhc2U2NCc6IHtcbiAgICAgICAgICAnRm46OkpvaW4nOiBhcnJheVdpdGgoYXJyYXlXaXRoKCcgPj4gL2V0Yy9lY3MvZWNzLmNvbmZpZ1xcbnN1ZG8gaXB0YWJsZXMgLS1pbnNlcnQgRk9SV0FSRCAxIC0taW4taW50ZXJmYWNlIGRvY2tlcisgLS1kZXN0aW5hdGlvbiAxNjkuMjU0LjE2OS4yNTQvMzIgLS1qdW1wIERST1BcXG5zdWRvIHNlcnZpY2UgaXB0YWJsZXMgc2F2ZVxcbmVjaG8gRUNTX0FXU1ZQQ19CTE9DS19JTURTPXRydWUgPj4gL2V0Yy9lY3MvZWNzLmNvbmZpZ1xcbnl1bSBpbnN0YWxsIC15cSBhd3NjbGkgdW56aXBcXG4jIFJlbmRlclF1ZXVlIGZpbGUgY2FjaGluZyBlbmFibGVkXFxubWtkaXIgLXAgJChkaXJuYW1lIFxcJy90bXAvJykpLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KSk7XG4gIH0pO1xuXG59KTtcbiJdfQ==