"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const lib_1 = require("../lib");
const spot_event_plugin_fleet_1 = require("../lib/spot-event-plugin-fleet");
describe('ConfigureSpotEventPlugin', () => {
    let stack;
    let vpc;
    let region;
    let renderQueue;
    let version;
    let app;
    let fleet;
    let groupName;
    const workerMachineImage = new aws_ec2_1.GenericWindowsImage({
        'us-east-1': 'ami-any',
    });
    beforeEach(() => {
        region = 'us-east-1';
        app = new core_1.App();
        stack = new core_1.Stack(app, 'stack', {
            env: {
                region,
            },
        });
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        version = new lib_1.VersionQuery(stack, 'Version');
        renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
            vpc,
            images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
            repository: new lib_1.Repository(stack, 'Repository', {
                vpc,
                version,
            }),
            version,
        });
        groupName = 'group_name1';
        fleet = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotFleet', {
            vpc,
            renderQueue: renderQueue,
            deadlineGroups: [
                groupName,
            ],
            instanceTypes: [
                aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
            ],
            workerMachineImage,
            maxCapacity: 1,
        });
    });
    describe('creates a custom resource', () => {
        test('with default spot event plugin properties', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotPluginConfigurations: assert_1.objectLike({
                    AWSInstanceStatus: 'Disabled',
                    DeleteInterruptedSlaves: false,
                    DeleteTerminatedSlaves: false,
                    IdleShutdown: 10,
                    Logging: 'Standard',
                    PreJobTaskMode: 'Conservative',
                    Region: core_1.Stack.of(renderQueue).region,
                    ResourceTracker: true,
                    StaggerInstances: 50,
                    State: 'Global Enabled',
                    StrictHardCap: false,
                }),
            })));
        });
        test('with custom spot event plugin properties', () => {
            // GIVEN
            const configuration = {
                awsInstanceStatus: lib_1.SpotEventPluginDisplayInstanceStatus.EXTRA_INFO_0,
                deleteEC2SpotInterruptedWorkers: true,
                deleteSEPTerminatedWorkers: true,
                idleShutdown: core_1.Duration.minutes(20),
                loggingLevel: lib_1.SpotEventPluginLoggingLevel.VERBOSE,
                preJobTaskMode: lib_1.SpotEventPluginPreJobTaskMode.NORMAL,
                region: 'us-west-2',
                enableResourceTracker: false,
                maximumInstancesStartedPerCycle: 10,
                state: lib_1.SpotEventPluginState.DISABLED,
                strictHardCap: true,
            };
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
                configuration,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotPluginConfigurations: assert_1.objectLike({
                    AWSInstanceStatus: 'ExtraInfo0',
                    DeleteInterruptedSlaves: true,
                    DeleteTerminatedSlaves: true,
                    IdleShutdown: 20,
                    Logging: 'Verbose',
                    PreJobTaskMode: 'Normal',
                    Region: 'us-west-2',
                    ResourceTracker: false,
                    StaggerInstances: 10,
                    State: 'Disabled',
                    StrictHardCap: true,
                }),
            })));
        });
        test('without spot fleets', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', {
                spotFleetRequestConfigurations: assert_1.ABSENT,
            }));
        });
        test('provides RQ connection parameters to custom resource', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                connection: assert_1.objectLike({
                    hostname: stack.resolve(renderQueue.endpoint.hostname),
                    port: stack.resolve(renderQueue.endpoint.portAsString()),
                    protocol: stack.resolve(renderQueue.endpoint.applicationProtocol.toString()),
                }),
            })));
        });
        test('with default spot fleet request configuration', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            const rfdkTag = runtime_info_1.tagFields(fleet);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.objectLike({
                    [groupName]: assert_1.objectLike({
                        AllocationStrategy: fleet.allocationStrategy.toString(),
                        IamFleetRole: stack.resolve(fleet.fleetRole.roleArn),
                        LaunchSpecifications: assert_1.arrayWith(assert_1.objectLike({
                            IamInstanceProfile: {
                                Arn: stack.resolve(fleet.instanceProfile.attrArn),
                            },
                            ImageId: fleet.imageId,
                            SecurityGroups: assert_1.arrayWith(assert_1.objectLike({
                                GroupId: stack.resolve(fleet.securityGroups[0].securityGroupId),
                            })),
                            SubnetId: stack.resolve(core_1.Fn.join('', [vpc.privateSubnets[0].subnetId, ',', vpc.privateSubnets[1].subnetId])),
                            TagSpecifications: assert_1.arrayWith(assert_1.objectLike({
                                ResourceType: 'instance',
                                Tags: assert_1.arrayWith(assert_1.objectLike({
                                    Key: rfdkTag.name,
                                    Value: rfdkTag.value,
                                })),
                            })),
                            UserData: stack.resolve(core_1.Fn.base64(fleet.userData.render())),
                            InstanceType: fleet.instanceTypes[0].toString(),
                        })),
                        ReplaceUnhealthyInstances: true,
                        TargetCapacity: fleet.maxCapacity,
                        TerminateInstancesWithExpiration: true,
                        Type: 'maintain',
                        TagSpecifications: assert_1.arrayWith(assert_1.objectLike({
                            ResourceType: 'spot-fleet-request',
                            Tags: assert_1.arrayWith(assert_1.objectLike({
                                Key: rfdkTag.name,
                                Value: rfdkTag.value,
                            })),
                        })),
                    }),
                }),
            })));
        });
        test('adds policies to the render queue', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Role', 1, {
                ManagedPolicyArns: assert_1.arrayWith(stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy').managedPolicyArn), stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy').managedPolicyArn)),
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Condition: {
                                StringLike: {
                                    'iam:PassedToService': 'ec2.amazonaws.com',
                                },
                            },
                            Effect: 'Allow',
                            Resource: [
                                stack.resolve(fleet.fleetRole.roleArn),
                                stack.resolve(fleet.fleetInstanceRole.roleArn),
                            ],
                        },
                        {
                            Action: 'ec2:CreateTags',
                            Effect: 'Allow',
                            Resource: 'arn:aws:ec2:*:*:spot-fleet-request/*',
                        },
                    ],
                },
                Roles: [{
                        Ref: 'RQRCSTaskTaskRole00DC9B43',
                    }],
            }));
        });
        test('adds resource tracker policy even if rt disabled', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
                configuration: {
                    enableResourceTracker: false,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                ManagedPolicyArns: assert_1.arrayWith(stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy').managedPolicyArn)),
            }));
        });
        test.each([
            undefined,
            [],
        ])('without spot fleet', (noFleets) => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: noFleets,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.ABSENT,
            })));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Role', {
                ManagedPolicyArns: assert_1.arrayWith(stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy').managedPolicyArn), stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy').managedPolicyArn)),
            }));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Condition: {
                                StringLike: {
                                    'iam:PassedToService': 'ec2.amazonaws.com',
                                },
                            },
                            Effect: 'Allow',
                            Resource: [
                                stack.resolve(fleet.fleetRole.roleArn),
                                stack.resolve(fleet.fleetInstanceRole.roleArn),
                            ],
                        },
                        {
                            Action: 'ec2:CreateTags',
                            Effect: 'Allow',
                            Resource: 'arn:aws:ec2:*:*:spot-fleet-request/*',
                        },
                    ],
                },
                Roles: [{
                        Ref: 'RQRCSTaskTaskRole00DC9B43',
                    }],
            }));
        });
        test('fleet with validUntil', () => {
            // GIVEN
            const validUntil = core_1.Expiration.atDate(new Date(2022, 11, 17));
            const fleetWithCustomProps = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                vpc,
                renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                ],
                workerMachineImage,
                maxCapacity: 1,
                validUntil,
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleetWithCustomProps,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.objectLike({
                    [groupName]: assert_1.objectLike({
                        ValidUntil: validUntil.date.toISOString(),
                    }),
                }),
            })));
        });
        test('fleet with block devices', () => {
            // GIVEN
            const deviceName = '/dev/xvda';
            const volumeSize = 50;
            const encrypted = true;
            const deleteOnTermination = true;
            const iops = 100;
            const volumeType = aws_autoscaling_1.EbsDeviceVolumeType.STANDARD;
            const fleetWithCustomProps = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                vpc,
                renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                ],
                workerMachineImage,
                maxCapacity: 1,
                blockDevices: [{
                        deviceName,
                        volume: aws_autoscaling_1.BlockDeviceVolume.ebs(volumeSize, {
                            encrypted,
                            deleteOnTermination,
                            iops,
                            volumeType,
                        }),
                    }],
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleetWithCustomProps,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.objectLike({
                    [groupName]: assert_1.objectLike({
                        LaunchSpecifications: assert_1.arrayWith(assert_1.objectLike({
                            BlockDeviceMappings: assert_1.arrayWith(assert_1.objectLike({
                                DeviceName: deviceName,
                                Ebs: assert_1.objectLike({
                                    DeleteOnTermination: deleteOnTermination,
                                    Iops: iops,
                                    VolumeSize: volumeSize,
                                    VolumeType: volumeType,
                                    Encrypted: encrypted,
                                }),
                            })),
                        })),
                    }),
                }),
            })));
        });
        test('fleet with block devices with custom volume', () => {
            // GIVEN
            const deviceName = '/dev/xvda';
            const virtualName = 'name';
            const snapshotId = 'snapshotId';
            const volumeSize = 50;
            const deleteOnTermination = true;
            const iops = 100;
            const volumeType = aws_autoscaling_1.EbsDeviceVolumeType.STANDARD;
            const fleetWithCustomProps = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                vpc,
                renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                ],
                workerMachineImage,
                maxCapacity: 1,
                blockDevices: [{
                        deviceName: deviceName,
                        volume: {
                            ebsDevice: {
                                deleteOnTermination,
                                iops,
                                volumeSize,
                                volumeType,
                                snapshotId,
                            },
                            virtualName,
                        },
                    }],
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleetWithCustomProps,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.objectLike({
                    [groupName]: assert_1.objectLike({
                        LaunchSpecifications: assert_1.arrayWith(assert_1.objectLike({
                            BlockDeviceMappings: assert_1.arrayWith(assert_1.objectLike({
                                DeviceName: deviceName,
                                Ebs: assert_1.objectLike({
                                    SnapshotId: snapshotId,
                                    DeleteOnTermination: deleteOnTermination,
                                    Iops: iops,
                                    VolumeSize: volumeSize,
                                    VolumeType: volumeType,
                                    Encrypted: assert_1.ABSENT,
                                }),
                                VirtualName: virtualName,
                            })),
                        })),
                    }),
                }),
            })));
        });
        test('fleet with block devices with no device', () => {
            // GIVEN
            const deviceName = '/dev/xvda';
            const volume = aws_autoscaling_1.BlockDeviceVolume.noDevice();
            const fleetWithCustomProps = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                vpc,
                renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                ],
                workerMachineImage,
                maxCapacity: 1,
                blockDevices: [{
                        deviceName: deviceName,
                        volume,
                    }],
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleetWithCustomProps,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.objectLike({
                    [groupName]: assert_1.objectLike({
                        LaunchSpecifications: assert_1.arrayWith(assert_1.objectLike({
                            BlockDeviceMappings: assert_1.arrayWith(assert_1.objectLike({
                                DeviceName: deviceName,
                                NoDevice: '',
                            })),
                        })),
                    }),
                }),
            })));
        });
        test('fleet with deprecated mappingEnabled', () => {
            // GIVEN
            const deviceName = '/dev/xvda';
            const mappingEnabled = false;
            const fleetWithCustomProps = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                vpc,
                renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                ],
                workerMachineImage,
                maxCapacity: 1,
                blockDevices: [{
                        deviceName: deviceName,
                        volume: aws_autoscaling_1.BlockDeviceVolume.ebs(50),
                        mappingEnabled,
                    }],
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleetWithCustomProps,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.objectLike({
                    [groupName]: assert_1.objectLike({
                        LaunchSpecifications: assert_1.arrayWith(assert_1.objectLike({
                            BlockDeviceMappings: assert_1.arrayWith(assert_1.objectLike({
                                DeviceName: deviceName,
                                NoDevice: '',
                            })),
                        })),
                    }),
                }),
            })));
        });
    });
    test('only one object allowed per render queue', () => {
        // GIVEN
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        // WHEN
        function createConfigureSpotEventPlugin() {
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin2', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).toThrowError(/Only one ConfigureSpotEventPlugin construct is allowed per render queue./);
    });
    test('can create multiple objects with different render queues', () => {
        // GIVEN
        const renderQueue2 = new lib_1.RenderQueue(stack, 'RQ2', {
            vpc,
            images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
            repository: new lib_1.Repository(stack, 'Repository2', {
                vpc,
                version,
            }),
            version,
        });
        // WHEN
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin2', {
            vpc,
            renderQueue: renderQueue2,
            spotFleets: [
                fleet,
            ],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('Custom::RFDK_ConfigureSpotEventPlugin', 2));
    });
    test('throws with not supported render queue', () => {
        // GIVEN
        const invalidRenderQueue = {};
        // WHEN
        function createConfigureSpotEventPlugin() {
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin2', {
                vpc,
                renderQueue: invalidRenderQueue,
                spotFleets: [
                    fleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).toThrowError(/The provided render queue is not an instance of RenderQueue class. Some functionality is not supported./);
    });
    test('tagSpecifications returns undefined if fleet does not have tags', () => {
        // GIVEN
        const mockFleet = {
            tags: {
                hasTags: jest.fn().mockReturnValue(false),
            },
        };
        const mockedFleet = mockFleet;
        const config = new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        // WHEN
        // eslint-disable-next-line dot-notation
        const result = stack.resolve(config['tagSpecifications'](mockedFleet, lib_1.SpotFleetResourceType.INSTANCE));
        // THEN
        expect(result).toBeUndefined();
    });
    describe('with TLS', () => {
        let renderQueueWithTls;
        let caCert;
        beforeEach(() => {
            const host = 'renderqueue';
            const zoneName = 'deadline-test.internal';
            caCert = new core_2.X509CertificatePem(stack, 'RootCA', {
                subject: {
                    cn: 'SampleRootCA',
                },
            });
            renderQueueWithTls = new lib_1.RenderQueue(stack, 'RQ with TLS', {
                vpc,
                images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
                repository: new lib_1.Repository(stack, 'Repository2', {
                    vpc,
                    version,
                }),
                version,
                hostname: {
                    zone: new aws_route53_1.PrivateHostedZone(stack, 'DnsZone', {
                        vpc,
                        zoneName: zoneName,
                    }),
                    hostname: host,
                },
                trafficEncryption: {
                    externalTLS: {
                        rfdkCertificate: new core_2.X509CertificatePem(stack, 'RQCert', {
                            subject: {
                                cn: `${host}.${zoneName}`,
                            },
                            signingCertificate: caCert,
                        }),
                    },
                },
            });
        });
        test('Lambda role can get the ca secret', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueueWithTls,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: stack.resolve(renderQueueWithTls.certChain.secretArn),
                        },
                    ],
                },
                Roles: [
                    {
                        Ref: 'ConfigureSpotEventPluginConfiguratorServiceRole341B4735',
                    },
                ],
            }));
        });
        test('creates a custom resource with connection', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueueWithTls,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                connection: assert_1.objectLike({
                    hostname: stack.resolve(renderQueueWithTls.endpoint.hostname),
                    port: stack.resolve(renderQueueWithTls.endpoint.portAsString()),
                    protocol: stack.resolve(renderQueueWithTls.endpoint.applicationProtocol.toString()),
                    caCertificateArn: stack.resolve(renderQueueWithTls.certChain.secretArn),
                }),
            })));
        });
    });
    test('throws with the same group name', () => {
        // WHEN
        function createConfigureSpotEventPlugin() {
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                    fleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).toThrowError(`Bad Group Name: ${groupName}. Group names in Spot Fleet Request Configurations should be unique.`);
    });
    test('uses selected subnets', () => {
        // WHEN
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            vpcSubnets: { subnets: [vpc.privateSubnets[0]] },
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Handler: 'configure-spot-event-plugin.configureSEP',
            VpcConfig: {
                SubnetIds: [
                    stack.resolve(vpc.privateSubnets[0].subnetId),
                ],
            },
        }));
    });
    describe('throws with wrong deadline version', () => {
        test.each([
            ['10.1.9'],
            ['10.1.10'],
        ])('%s', (versionString) => {
            // GIVEN
            const newStack = new core_1.Stack(app, 'NewStack');
            version = new lib_1.VersionQuery(newStack, 'OldVersion', {
                version: versionString,
            });
            renderQueue = new lib_1.RenderQueue(newStack, 'OldRenderQueue', {
                vpc,
                images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
                repository: new lib_1.Repository(newStack, 'Repository', {
                    vpc,
                    version,
                }),
                version,
            });
            // WHEN
            function createConfigureSpotEventPlugin() {
                new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                    vpc,
                    renderQueue: renderQueue,
                    spotFleets: [
                        fleet,
                    ],
                });
            }
            // THEN
            expect(createConfigureSpotEventPlugin).toThrowError(`Minimum supported Deadline version for ConfigureSpotEventPlugin is 10.1.12.0. Received: ${versionString}.`);
        });
    });
    test('does not throw with min deadline version', () => {
        // GIVEN
        const versionString = '10.1.12';
        const newStack = new core_1.Stack(app, 'NewStack');
        version = new lib_1.VersionQuery(newStack, 'OldVersion', {
            version: versionString,
        });
        renderQueue = new lib_1.RenderQueue(newStack, 'OldRenderQueue', {
            vpc,
            images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
            repository: new lib_1.Repository(newStack, 'Repository', {
                vpc,
                version,
            }),
            version,
        });
        // WHEN
        function createConfigureSpotEventPlugin() {
            new lib_1.ConfigureSpotEventPlugin(newStack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).not.toThrow();
    });
});
//# sourceMappingURL=data:application/json;base64,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