"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkerInstanceFleet = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const version_1 = require("./version");
const worker_configuration_1 = require("./worker-configuration");
/**
 *  A new or imported Deadline Worker Fleet.
 */
class WorkerInstanceFleetBase extends core_1.Construct {
}
/**
 * This construct reperesents a fleet of Deadline Workers.
 *
 * The construct consists of an Auto Scaling Group (ASG) of instances using a provided AMI which has Deadline and any number
 * of render applications installed.  Whenever an instance in the ASG start it will connect Deadline to the desired render queue.
 *
 * When the worker fleet is deployed if it has been provided a HealthMonitor the Worker fleet will register itself against the Monitor
 * to ensure that the fleet remains healthy.
 *
 * Resources Deployed
 * ------------------------
 * - An EC2 Auto Scaling Group to maintain the number of instances.
 * - An Instance Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Worker, Deadline Launcher, and instance-startup logs for the instances
 *    in the fleet.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The data that is stored on your Worker's local EBS volume can include temporary working files from the applications
 *    that are rendering your jobs and tasks. That data can be sensitive or privileged, so we recommend that you encrypt
 *    the data volumes of these instances using either the provided option or by using an encrypted AMI as your source.
 * - The software on the AMI that is being used by this construct may pose a security risk. We recommend that you adopt a
 *    patching strategy to keep this software current with the latest security patches. Please see
 *    https://docs.aws.amazon.com/rfdk/latest/guide/patching-software.html for more information.
 *
 * @stability stable
 */
class WorkerInstanceFleet extends WorkerInstanceFleetBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        this.stack = core_1.Stack.of(scope);
        this.env = {
            account: this.stack.account,
            region: this.stack.region,
        };
        this.validateProps(props);
        const minCapacity = (_b = props.minCapacity) !== null && _b !== void 0 ? _b : 1;
        const signals = minCapacity > 0 ? aws_autoscaling_1.Signals.waitForMinCapacity({
            timeout: WorkerInstanceFleet.RESOURCE_SIGNAL_TIMEOUT,
        }) : undefined;
        if (signals === undefined) {
            core_1.Annotations.of(this).addWarning('Deploying with 0 minimum capacity. If there is an error in the EC2 UserData for this fleet, then your stack deployment will not fail. Watch for errors in your CloudWatch logs.');
        }
        // Launching the fleet with deadline workers.
        this.fleet = new aws_autoscaling_1.AutoScalingGroup(this, 'Default', {
            vpc: props.vpc,
            instanceType: (props.instanceType ? props.instanceType : aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE)),
            machineImage: props.workerMachineImage,
            keyName: props.keyName,
            vpcSubnets: props.vpcSubnets ? props.vpcSubnets : {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            securityGroup: props.securityGroup,
            minCapacity,
            maxCapacity: props.maxCapacity,
            desiredCapacity: props.desiredCapacity,
            signals,
            healthCheck: aws_autoscaling_1.HealthCheck.elb({
                grace: WorkerInstanceFleet.DEFAULT_HEALTH_CHECK_INTERVAL,
            }),
            role: props.role,
            spotPrice: (_c = props.spotPrice) === null || _c === void 0 ? void 0 : _c.toString(),
            blockDevices: props.blockDevices,
        });
        this.targetCapacity = parseInt(this.fleet.node.defaultChild.maxSize, 10);
        this.targetScope = this;
        this.targetToMonitor = this.fleet;
        this.targetCapacityMetric = new aws_cloudwatch_1.Metric({
            namespace: 'AWS/AutoScaling',
            metricName: 'GroupDesiredCapacity',
            dimensions: {
                AutoScalingGroupName: this.fleet.autoScalingGroupName,
            },
            label: 'GroupDesiredCapacity',
        });
        this.targetUpdatePolicy = new aws_iam_1.Policy(this, 'ASGUpdatePolicy', {
            statements: [new aws_iam_1.PolicyStatement({
                    actions: ['autoscaling:UpdateAutoScalingGroup'],
                    resources: [this.fleet.autoScalingGroupArn],
                })],
        });
        this.fleet.node.defaultChild.metricsCollection = [{
                granularity: '1Minute',
                metrics: ['GroupDesiredCapacity'],
            }];
        this.grantPrincipal = this.fleet.grantPrincipal;
        this.connections = this.fleet.connections;
        // Configure the health monitoring if provided.
        // Note: This must be done *BEFORE* configuring the worker. We rely on the worker configuration
        // script restarting the launcher.
        this.configureHealthMonitor(props);
        const workerConfig = new worker_configuration_1.WorkerInstanceConfiguration(this, id, {
            worker: this.fleet,
            cloudWatchLogSettings: {
                logGroupPrefix: WorkerInstanceFleet.DEFAULT_LOG_GROUP_PREFIX,
                ...props.logGroupProps,
            },
            renderQueue: props.renderQueue,
            workerSettings: props,
            userDataProvider: props.userDataProvider,
        });
        this.listeningPorts = aws_ec2_1.Port.tcpRange(workerConfig.listenerPort, workerConfig.listenerPort + WorkerInstanceFleet.MAX_WORKERS_PER_HOST);
        // Updating the user data with successful cfn-signal commands.
        if (signals) {
            this.fleet.userData.addSignalOnExitCommand(this.fleet);
        }
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Add the security group to all workers.
     *
     * @param securityGroup : The security group to add.
     * @stability stable
     */
    addSecurityGroup(securityGroup) {
        this.fleet.addSecurityGroup(securityGroup);
    }
    /**
     * Allow access to the worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that depends on this stack.
     *
     * If this stack depends on the other stack, use allowListenerPortTo().
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowListenerPortFrom(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowListenerPortFrom(Peer.ipv4('10.0.0.0/24').connections)`
     *
     * @stability stable
     * @inheritdoc true
     */
    allowListenerPortFrom(other) {
        this.connections.allowFrom(other.connections, this.listeningPorts, 'Worker remote command listening port');
    }
    /**
     * Allow access to the worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that this stack depends on.
     *
     * If the other stack depends on this stack, use allowListenerPortFrom().
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowListenerPortTo(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowListenerPortTo(Peer.ipv4('10.0.0.0/24').connections)`
     *
     * @stability stable
     * @inheritdoc true
     */
    allowListenerPortTo(other) {
        other.connections.allowTo(this.connections, this.listeningPorts, 'Worker remote command listening port');
    }
    validateProps(props) {
        this.validateSpotPrice(props.spotPrice);
        this.validateArrayGroupsPoolsSyntax(props.groups, /^(?!none$)[a-zA-Z0-9-_]+$/i, 'groups');
        this.validateArrayGroupsPoolsSyntax(props.pools, /^(?!none$)[a-zA-Z0-9-_]+$/i, 'pools');
        this.validateRegion(props.region, /^(?!none$|all$|unrecognized$)[a-zA-Z0-9-_]+$/i);
        this.validateBlockDevices(props.blockDevices);
    }
    validateSpotPrice(spotPrice) {
        if (spotPrice && !(spotPrice >= WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT && spotPrice <= WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT)) {
            throw new Error(`Invalid value: ${spotPrice} for property 'spotPrice'. Valid values can be any decimal between ${WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT} and ${WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT}.`);
        }
    }
    validateRegion(region, regex) {
        if (region && !regex.test(region)) {
            throw new Error(`Invalid value: ${region} for property 'region'. Valid characters are A-Z, a-z, 0-9, - and _. ‘All’, ‘none’ and ‘unrecognized’ are reserved names that cannot be used.`);
        }
    }
    validateArrayGroupsPoolsSyntax(array, regex, property) {
        if (array) {
            array.forEach(value => {
                if (!regex.test(value)) {
                    throw new Error(`Invalid value: ${value} for property '${property}'. Valid characters are A-Z, a-z, 0-9, - and _. Also, group 'none' is reserved as the default group.`);
                }
            });
        }
    }
    validateBlockDevices(blockDevices) {
        if (blockDevices === undefined) {
            core_1.Annotations.of(this).addWarning(`The worker-fleet ${this.node.id} is being created without being provided any block devices so the Source AMI's devices will be used. ` +
                'Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
        }
        else {
            blockDevices.forEach(device => {
                if (device.volume.ebsDevice === undefined) {
                    // Suppressed or Ephemeral Block Device
                    return;
                }
                // encrypted is not exposed as part of ebsDeviceProps so we need to confirm it exists then access it via [].
                // eslint-disable-next-line dot-notation
                if (('encrypted' in device.volume.ebsDevice === false) || ('encrypted' in device.volume.ebsDevice && !device.volume.ebsDevice['encrypted'])) {
                    core_1.Annotations.of(this).addWarning(`The BlockDevice "${device.deviceName}" on the worker-fleet ${this.node.id} is not encrypted. ` +
                        'Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.');
                }
            });
        }
    }
    configureHealthMonitor(props) {
        var _b, _c;
        if (props.healthMonitor) {
            const healthCheckPort = (_c = (_b = props.healthCheckConfig) === null || _b === void 0 ? void 0 : _b.port) !== null && _c !== void 0 ? _c : core_2.HealthMonitor.DEFAULT_HEALTH_CHECK_PORT;
            const configureHealthMonitorScriptAsset = core_2.ScriptAsset.fromPathConvention(this, 'WorkerConfigurationScript', {
                osType: this.fleet.osType,
                baseName: 'configureWorkerHealthCheck',
                rootDir: path.join(__dirname, '..', 'scripts/'),
            });
            configureHealthMonitorScriptAsset.executeOn({
                host: this.fleet,
                args: [
                    `'${healthCheckPort}'`,
                    `'${version_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}'`,
                ],
            });
            props.healthMonitor.registerFleet(this, props.healthCheckConfig || {
                port: healthCheckPort,
            });
        }
        else {
            core_1.Annotations.of(this).addWarning(`The worker-fleet ${this.node.id} is being created without a health monitor attached to it. This means that the fleet will not automatically scale-in to 0 if the workers are unhealthy.`);
        }
    }
}
exports.WorkerInstanceFleet = WorkerInstanceFleet;
_a = JSII_RTTI_SYMBOL_1;
WorkerInstanceFleet[_a] = { fqn: "aws-rfdk.deadline.WorkerInstanceFleet", version: "0.34.0" };
/**
 * The min limit for spot price.
 *
 * @stability stable
 */
WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT = 0.001;
/**
 * The max limit for spot price.
 *
 * @stability stable
 */
WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT = 255;
/**
 * This determines worker's health based on any hardware or software issues of EC2 instance.
 * Resource Tracker does deep ping every 5 minutes. These checks should be more frequent so
 * that any EC2 level issues are identified ASAP. Hence setting it to 1 minute.
 */
WorkerInstanceFleet.DEFAULT_HEALTH_CHECK_INTERVAL = core_1.Duration.minutes(1);
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
WorkerInstanceFleet.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * This is the current maximum for number of workers that can be started on a single host. Currently the
 * only thing using this limit is the configuration of the listener ports. More than 8 workers can be started,
 * but only the first 8 will have their ports opened in the workers' security group.
 */
WorkerInstanceFleet.MAX_WORKERS_PER_HOST = 8;
/**
 * Setting the default signal timeout to 15 min. This is the max time, a single instance is expected
 * to take for launch and execute the user-data for deadline worker configuration. As we are setting
 * failure signals in the user-data, any failure will terminate deployment immediately.
 */
WorkerInstanceFleet.RESOURCE_SIGNAL_TIMEOUT = core_1.Duration.minutes(15);
//# sourceMappingURL=data:application/json;base64,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