"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkerInstanceConfiguration = exports.InstanceUserDataProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const version_1 = require("./version");
/**
 * Implementation of {@link IInstanceUserDataProvider}.
 *
 * Can be used as sub-class with override the desired methods
 * to add custom user data commands for WorkerInstanceFleet or WorkerInstanceConfiguration.
 *
 * @stability stable
 */
class InstanceUserDataProvider extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id) {
        super(scope, id);
    }
    /**
     * Method that is invoked before configuring the Cloud Watch Agent.
     *
     * @stability stable
     * @inheritdoc true
     */
    preCloudWatchAgent(_host) {
    }
    /**
     * Method that is invoked before the render queue configuration.
     *
     * @stability stable
     * @inheritdoc true
     */
    preRenderQueueConfiguration(_host) {
    }
    /**
     * Method that is invoked after configuring the connection to the render queue and before configuring the Deadline Worker.
     *
     * @stability stable
     * @inheritdoc true
     */
    preWorkerConfiguration(_host) {
    }
    /**
     * Method that is invoked after all configuration is done and worker started.
     *
     * @stability stable
     * @inheritdoc true
     */
    postWorkerLaunch(_host) {
    }
}
exports.InstanceUserDataProvider = InstanceUserDataProvider;
_a = JSII_RTTI_SYMBOL_1;
InstanceUserDataProvider[_a] = { fqn: "aws-rfdk.deadline.InstanceUserDataProvider", version: "0.34.0" };
/**
 * This construct can be used to configure Deadline Workers on an instance to connect to a RenderQueue, stream their log files to CloudWatch, and configure various settings of the Deadline Worker.
 *
 * The configuration happens on instance start-up using user data scripting.
 *
 * This configuration performs the following steps in order:
 * 1) Configure Cloud Watch Agent
 * 2) Configure Deadline Worker RenderQueue connection
 * 3) Configure Deadline Worker settings
 *
 * A `userDataProvider` can be specified that defines callback functions.
 * These callbacks can be used to inject user data commands at different points during the Worker instance configuration.
 *
 * Security Considerations
 * ------------------------
 * - The instances configured by this construct will download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 *
 * @stability stable
 */
class WorkerInstanceConfiguration extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h, _j;
        super(scope, id);
        (_c = props.userDataProvider) === null || _c === void 0 ? void 0 : _c.preCloudWatchAgent(props.worker);
        if (props.cloudWatchLogSettings) {
            this.configureCloudWatchLogStream(props.worker, id, props.cloudWatchLogSettings, props.shouldInstallCloudWatchAgent);
        }
        (_d = props.userDataProvider) === null || _d === void 0 ? void 0 : _d.preRenderQueueConfiguration(props.worker);
        (_e = props.renderQueue) === null || _e === void 0 ? void 0 : _e.configureClientInstance({
            host: props.worker,
            // Don't restart the Deadline Launcher service after configuring the connection to the Render Queue. We need to
            // restart it later anyways, and the Windows service for the Deadline Launcher can get locked in the "stopping"
            // state if you attempt to stop or restart it while it is still restarting. This can cause the user data execution
            // to get locked waiting for the service to finish stopping/restarting.
            restartLauncher: false,
        });
        (_f = props.userDataProvider) === null || _f === void 0 ? void 0 : _f.preWorkerConfiguration(props.worker);
        this.listenerPort = (_h = (_g = props.workerSettings) === null || _g === void 0 ? void 0 : _g.listenerPort) !== null && _h !== void 0 ? _h : WorkerInstanceConfiguration.DEFAULT_LISTENER_PORT;
        this.configureWorkerSettings(props.worker, id, props.workerSettings);
        (_j = props.userDataProvider) === null || _j === void 0 ? void 0 : _j.postWorkerLaunch(props.worker);
    }
    /**
     * This method can be used to configure a Deadline Worker instance to stream its logs to the AWS CloudWatch service.
     *
     * The logs that this configures to stream are:
     * - EC2 Instance UserData execution; this is the startup scripting that is run when the instance launches
     *    for the first time.
     * - Deadline Worker logs.
     * - Deadline Launcher logs.
     *
     * @param worker The worker to configure.
     * @param id Identifier to disambiguate the resources that are created.
     * @param logGroupProps Configuration for the log group in CloudWatch.
     * @param shouldInstallAgent Boolean for if the worker's User Data should attempt to install the CloudWatch agent.
     * @stability stable
     */
    configureCloudWatchLogStream(worker, id, logGroupProps, shouldInstallAgent) {
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, `${id}LogGroupWrapper`, id, logGroupProps);
        logGroup.grantWrite(worker);
        const cloudWatchConfigurationBuilder = new core_2.CloudWatchConfigBuilder(core_1.Duration.seconds(15));
        if (worker.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'UserdataExecution', 'C:\\ProgramData\\Amazon\\EC2-Windows\\Launch\\Log\\UserdataExecution.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'WorkerLogs', 'C:\\ProgramData\\Thinkbox\\Deadline10\\logs\\deadlineslave*.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'LauncherLogs', 'C:\\ProgramData\\Thinkbox\\Deadline10\\logs\\deadlinelauncher*.log');
        }
        else {
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'WorkerLogs', '/var/log/Thinkbox/Deadline10/deadlineslave*.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'LauncherLogs', '/var/log/Thinkbox/Deadline10/deadlinelauncher*.log');
        }
        new core_2.CloudWatchAgent(this, `${id}LogsConfig`, {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: worker,
            shouldInstallAgent,
        });
    }
    /**
     * This method can be used to set up the Deadline Worker application on an EC2 instance.
     *
     * From a practical
     * perspective, this is executing the script found in aws-rfdk/lib/deadline/scripts/[bash,powershell]/configureWorker.[sh,ps1]
     * to configure the Deadline Worker application.
     *
     * @param worker The worker to configure.
     * @param id Identifier to disambiguate the resources that are created.
     * @param settings The Deadline Worker settings to apply.
     * @stability stable
     */
    configureWorkerSettings(worker, id, settings) {
        var _c, _d, _e, _f, _g;
        const configureWorkerScriptAsset = core_2.ScriptAsset.fromPathConvention(this, `${id}ConfigScript`, {
            osType: worker.osType,
            baseName: 'configureWorker',
            rootDir: path.join(__dirname, '..', 'scripts/'),
        });
        const configureWorkerPortAsset = new aws_s3_assets_1.Asset(this, `${id}WorkerListenerScript`, {
            path: path.join(__dirname, '..', 'scripts', 'python', 'worker-listening-port.py'),
        });
        const configWorkerPortLocalPath = worker.userData.addS3DownloadCommand({
            bucket: configureWorkerPortAsset.bucket,
            bucketKey: configureWorkerPortAsset.s3ObjectKey,
        });
        // Converting to lower case, as groups and pools are all stored in lower case in deadline.
        const groups = (_d = (_c = settings === null || settings === void 0 ? void 0 : settings.groups) === null || _c === void 0 ? void 0 : _c.map(val => val.toLowerCase()).join(',')) !== null && _d !== void 0 ? _d : '';
        const pools = (_f = (_e = settings === null || settings === void 0 ? void 0 : settings.pools) === null || _e === void 0 ? void 0 : _e.map(val => val.toLowerCase()).join(',')) !== null && _f !== void 0 ? _f : '';
        configureWorkerScriptAsset.executeOn({
            host: worker,
            args: [
                `'${groups}'`,
                `'${pools}'`,
                `'${(_g = settings === null || settings === void 0 ? void 0 : settings.region) !== null && _g !== void 0 ? _g : ''}'`,
                `'${version_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}'`,
                this.listenerPort.toString(),
                configWorkerPortLocalPath,
            ],
        });
    }
}
exports.WorkerInstanceConfiguration = WorkerInstanceConfiguration;
_b = JSII_RTTI_SYMBOL_1;
WorkerInstanceConfiguration[_b] = { fqn: "aws-rfdk.deadline.WorkerInstanceConfiguration", version: "0.34.0" };
/**
 * The default port to use for a worker to listen on for remote commands.
 */
WorkerInstanceConfiguration.DEFAULT_LISTENER_PORT = 56032;
//# sourceMappingURL=data:application/json;base64,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