"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionQuery = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto_1 = require("crypto");
const path_1 = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
/**
 * The abstract class for new or imported(custom) Deadline Version.
 */
class VersionQueryBase extends core_1.Construct {
}
/**
 * This class encapsulates information about a particular version of Thinkbox's Deadline software.
 *
 * Information such as the version number, and where to get installers for that version from Amazon S3.
 *
 * The version of an official release of Deadline is always four numeric version components separated by dots.
 * ex: 10.1.8.5. We refer to the components in this version, in order from left-to-right, as the
 * major, minor, release, and patch versions. For example, Deadline version 10.1.8.5 is majorVersion 10, minorVersion 1,
 * releaseVersion 8, and patchVersion 5.
 *
 * All of the installers provided by an instance of this class must be for the same Deadline release (ex: 10.1.8),
 * but the patch versions may differ between operating systems depending on the particulars of that release of Deadline.
 * This class provides a simple way to query a version of Deadline prior to or during deployment of a
 * CDK app.
 *
 * You pass an instance of this class to various Deadline constructs in this library to tell those
 * constructs which version of Deadline you want them to use, and be configured for.
 *
 * @stability stable
 */
class VersionQuery extends VersionQueryBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        this.expression = props === null || props === void 0 ? void 0 : props.version;
        const match = ((_b = props === null || props === void 0 ? void 0 : props.version) !== null && _b !== void 0 ? _b : '').match(VersionQuery.EXPRESSION_REGEX);
        if (match === null) {
            throw new Error(`Invalid version expression "${props.version}`);
        }
        this.pinnedVersionComponents = (match
            // First capture group is the entire matched string, so slice it off
            .slice(1)
            // Capture groups that are not matched return as undefined, so we filter them out
            .filter(component => component)
            // Parse strings to numbers
            .map(component => Number(component)));
        const lambdaCode = aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs'));
        const lambdaFunc = new aws_lambda_1.SingletonFunction(this, 'VersionProviderFunction', {
            uuid: '2e19e243-16ee-4d1a-a3c9-18d35eddd446',
            description: 'Used by the Version construct to get installer locations for a specific Deadline version.',
            code: lambdaCode,
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'version-provider.handler',
            timeout: core_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        const deadlineProperties = {
            versionString: props === null || props === void 0 ? void 0 : props.version,
            // If we don't have a full static version string, create a random string that will force the Lambda to always
            // run on redeploys, effectively checking for version updates.
            forceRun: this.forceRun(props === null || props === void 0 ? void 0 : props.version),
        };
        this.deadlineResource = new core_1.CustomResource(this, 'DeadlineResource', {
            serviceToken: lambdaFunc.functionArn,
            properties: deadlineProperties,
            resourceType: 'Custom::RFDK_DEADLINE_INSTALLERS',
        });
        this.majorVersion = this.versionComponent({
            expressionIndex: 0,
            customResourceAttribute: 'MajorVersion',
        });
        this.minorVersion = this.versionComponent({
            expressionIndex: 1,
            customResourceAttribute: 'MinorVersion',
        });
        this.releaseVersion = this.versionComponent({
            expressionIndex: 2,
            customResourceAttribute: 'ReleaseVersion',
        });
        this.linuxInstallers = {
            patchVersion: core_1.Token.asNumber(this.deadlineResource.getAtt('LinuxPatchVersion')),
            repository: {
                objectKey: this.deadlineResource.getAttString('LinuxRepositoryInstaller'),
                s3Bucket: aws_s3_1.Bucket.fromBucketName(scope, 'InstallerBucket', this.deadlineResource.getAttString('S3Bucket')),
            },
        };
    }
    versionComponent(args) {
        const { expressionIndex, customResourceAttribute } = args;
        return this.pinnedVersionComponents.length > expressionIndex
            ? this.pinnedVersionComponents[expressionIndex]
            : core_1.Token.asNumber(this.deadlineResource.getAtt(customResourceAttribute));
    }
    /**
     * Construct the full version string for the linux patch release referenced in this version object.
     *
     * This is constructed by joining the major, minor,
     * release, and patch versions by dots.
     *
     * @stability stable
     */
    linuxFullVersionString() {
        const major = core_1.Token.isUnresolved(this.majorVersion) ? core_1.Token.asString(this.majorVersion) : this.majorVersion.toString();
        const minor = core_1.Token.isUnresolved(this.minorVersion) ? core_1.Token.asString(this.minorVersion) : this.minorVersion.toString();
        const release = core_1.Token.isUnresolved(this.releaseVersion) ? core_1.Token.asString(this.releaseVersion) : this.releaseVersion.toString();
        const patch = core_1.Token.isUnresolved(this.linuxInstallers.patchVersion)
            ? core_1.Token.asString(this.linuxInstallers.patchVersion)
            : this.linuxInstallers.patchVersion.toString();
        return `${major}.${minor}.${release}.${patch}`;
    }
    /**
     * Returns whether this version is less than another version.
     *
     * @stability stable
     */
    isLessThan(other) {
        if (other.patchVersion !== 0) {
            throw new Error('Cannot compare a VersionQuery to a fully-qualified version with a non-zero patch number');
        }
        // We compare each component from highest order to lowest
        const componentGetters = [
            v => v.majorVersion,
            v => v.minorVersion,
            v => v.releaseVersion,
        ];
        for (const componentGetter of componentGetters) {
            const thisComponent = componentGetter(this);
            const otherComponent = componentGetter(other);
            if (core_1.Token.isUnresolved(thisComponent)) {
                // Unresolved components are unpinned. These will resolve to the latest and are not less than any provided
                // version
                return false;
            }
            else {
                const componentDiff = thisComponent - otherComponent;
                if (componentDiff !== 0) {
                    // If the components are different, return whether this component is smaller than the other component
                    return componentDiff < 0;
                }
            }
        }
        // If we've exited the loop naturally, it means all version components are pinned and equal. This means the version
        // is not less than the other, they are the same
        return false;
    }
    /**
     * Check if we have a full version in the supplied version string. If we don't, we want to make sure the Lambda
     * that fetches the full version number and the installers for it is always run. This allows for Deadline updates
     * to be discovered.
     */
    forceRun(version) {
        return !this.isFullVersion(version) ? crypto_1.randomBytes(32).toString('base64').slice(0, 32) : undefined;
    }
    /**
     * Checks if the supplied version contains the major, minor, release, and patch version numbers,
     * and returns true only if all 4 are supplied.
     */
    isFullVersion(version) {
        const components = version === null || version === void 0 ? void 0 : version.split('.').map(x => parseInt(x));
        if (!components || (components === null || components === void 0 ? void 0 : components.length) != 4) {
            return false;
        }
        return true;
    }
    /**
     * A string representation of the version using the best available information at synthesis-time.
     *
     * This value is not guaranteed to be resolved, and is intended for output to CDK users.
     *
     * @stability stable
     */
    get versionString() {
        var _b;
        return (_b = this.expression) !== null && _b !== void 0 ? _b : '(latest)';
    }
}
exports.VersionQuery = VersionQuery;
_a = JSII_RTTI_SYMBOL_1;
VersionQuery[_a] = { fqn: "aws-rfdk.deadline.VersionQuery", version: "0.34.0" };
/**
 * Regular expression for valid version query expressions
 */
VersionQuery.EXPRESSION_REGEX = /^(?:(\d+)(?:\.(\d+)(?:\.(\d+)(?:\.(\d+))?)?)?)?$/;
//# sourceMappingURL=data:application/json;base64,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