"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UsageBasedLicensing = exports.UsageBasedLicense = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
/**
 * Instances of this class represent a usage-based license for a particular product.
 *
 * It encapsulates all of the information specific to a product that the UsageBasedLicensing
 * construct requires to interoperate with that product.
 *
 * @stability stable
 */
class UsageBasedLicense {
    /**
     * @stability stable
     */
    constructor(props) {
        this.licenseName = props.licenseName;
        this.ports = props.ports;
        this.limit = props.limit;
    }
    /**
     * Method for 3dsMax license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     */
    static for3dsMax(limit) {
        return new UsageBasedLicense({
            licenseName: 'max',
            ports: [aws_ec2_1.Port.tcp(27002)],
            limit,
        });
    }
    /**
     * Method for Arnold license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     */
    static forArnold(limit) {
        return new UsageBasedLicense({
            licenseName: 'arnold',
            ports: [aws_ec2_1.Port.tcp(5056), aws_ec2_1.Port.tcp(7056)],
            limit,
        });
    }
    /**
     * Method for Cinema 4D license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forCinema4D(limit) {
        return new UsageBasedLicense({
            licenseName: 'cinema4d',
            ports: [aws_ec2_1.Port.tcp(5057), aws_ec2_1.Port.tcp(7057)],
            limit,
        });
    }
    /**
     * Method for Clarisse license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forClarisse(limit) {
        return new UsageBasedLicense({
            licenseName: 'clarisse',
            ports: [aws_ec2_1.Port.tcp(40500)],
            limit,
        });
    }
    /**
     * Method for Houdini license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forHoudini(limit) {
        return new UsageBasedLicense({
            licenseName: 'houdini',
            ports: [aws_ec2_1.Port.tcp(1715)],
            limit,
        });
    }
    /**
     * Method for Katana license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forKatana(limit) {
        return new UsageBasedLicense({
            licenseName: 'katana',
            ports: [aws_ec2_1.Port.tcp(4151), aws_ec2_1.Port.tcp(6101)],
            limit,
        });
    }
    /**
     * Method for KeyShot license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forKeyShot(limit) {
        return new UsageBasedLicense({
            licenseName: 'keyshot',
            ports: [aws_ec2_1.Port.tcp(27003), aws_ec2_1.Port.tcp(2703)],
            limit,
        });
    }
    /**
     * Method for krakatoa license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forKrakatoa(limit) {
        return new UsageBasedLicense({
            licenseName: 'krakatoa',
            ports: [aws_ec2_1.Port.tcp(27000), aws_ec2_1.Port.tcp(2700)],
            limit,
        });
    }
    /**
     * Method for Mantra license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forMantra(limit) {
        return new UsageBasedLicense({
            licenseName: 'mantra',
            ports: [aws_ec2_1.Port.tcp(1716)],
            limit,
        });
    }
    /**
     * Method for maxwell license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forMaxwell(limit) {
        return new UsageBasedLicense({
            licenseName: 'maxwell',
            ports: [aws_ec2_1.Port.tcp(5555), aws_ec2_1.Port.tcp(7055)],
            limit,
        });
    }
    /**
     * Method for Maya license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     */
    static forMaya(limit) {
        return new UsageBasedLicense({
            licenseName: 'maya',
            ports: [aws_ec2_1.Port.tcp(27002), aws_ec2_1.Port.tcp(2702)],
            limit,
        });
    }
    /**
     * Method for Nuke license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forNuke(limit) {
        return new UsageBasedLicense({
            licenseName: 'nuke',
            ports: [aws_ec2_1.Port.tcp(4101), aws_ec2_1.Port.tcp(6101)],
            limit,
        });
    }
    /**
     * Method for RealFlow license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forRealFlow(limit) {
        return new UsageBasedLicense({
            licenseName: 'realflow',
            ports: [aws_ec2_1.Port.tcp(5055), aws_ec2_1.Port.tcp(7055)],
            limit,
        });
    }
    /**
     * Method for RedShift license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forRedShift(limit) {
        return new UsageBasedLicense({
            licenseName: 'redshift',
            ports: [aws_ec2_1.Port.tcp(5054), aws_ec2_1.Port.tcp(7054)],
            limit,
        });
    }
    /**
     * Method for V-Ray license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forVray(limit) {
        return new UsageBasedLicense({
            licenseName: 'vray',
            ports: [aws_ec2_1.Port.tcp(30306)],
            limit,
        });
    }
    /**
     * Method for Yeti license limit.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     * @default - limit will be set to unlimited
     * @stability stable
     */
    static forYeti(limit) {
        return new UsageBasedLicense({
            licenseName: 'yeti',
            ports: [aws_ec2_1.Port.tcp(5053), aws_ec2_1.Port.tcp(7053)],
            limit,
        });
    }
}
exports.UsageBasedLicense = UsageBasedLicense;
_a = JSII_RTTI_SYMBOL_1;
UsageBasedLicense[_a] = { fqn: "aws-rfdk.deadline.UsageBasedLicense", version: "0.34.0" };
/**
 * Constant used to signify unlimited overage.
 *
 * @stability stable
 */
UsageBasedLicense.UNLIMITED = 2147483647;
/**
 * This construct is an implementation of the Deadline component that is required for Usage-based Licensing (UBL) (see: https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/licensing-usage-based.html ) in a render farm.
 *
 * Internally this is implemented as one or more instances of the Deadline License Forwarder application set up
 * to communicate to the render queue and Thinkbox’s licensing system, and to allow ingress connections
 * from the worker nodes so that they can acquire licenses as needed.
 *
 * The Deadline License Forwarder is set up to run within an AWS ECS task.
 *
 * Access to the running License Forwarder is gated by a security group that, by default, allows no ingress;
 * when a Deadline Worker requires access to licensing, then the RFDK constructs will grant that worker’s security group
 * ingress on TCP port 17004 as well as other ports as required by the specific licenses being used.
 *
 * Note: This construct does not currently implement the Deadline License Forwarder's Web Forwarding functionality.
 * This construct is not usable in any China region.
 *
 * Resources Deployed
 * ------------------------
 * - The Auto Scaling Group (ASG) added to the Amazon Elastic Container Service cluster that is hosting the Deadline
 *    License Forwarder for UBL. This creates one C5 Large instance by default.
 * - Amazon Elastic Block Store (EBS) device(s) associated with the EC2 instance(s) in the ASG. The default volume size is 30 GiB.
 * - An Amazon CloudWatch log group that contains the logs from the Deadline License Forwarder application.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The Deadline License Forwarder is designed to be secured by restricting network access to it. For security, only the Deadline
 *    Workers that require access to Usage-based Licenses should be granted network access to the instances deployed by this construct.
 *    Futhermore, you should restrict that access to only the product(s) that those workers require when deploying this construct.
 *
 * @stability stable
 */
class UsageBasedLicensing extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h;
        super(scope, id);
        const usageBasedLicenses = new Array();
        props.licenses.forEach(license => {
            usageBasedLicenses.push(`${license.licenseName}:${license.limit ? license.limit : UsageBasedLicense.UNLIMITED}`);
        });
        if (usageBasedLicenses.length < 1) {
            throw new Error('Should be specified at least one license with defined limit.');
        }
        this.cluster = new aws_ecs_1.Cluster(this, 'Cluster', { vpc: props.vpc });
        this.asg = this.cluster.addCapacity('ASG', {
            vpcSubnets: (_c = props.vpcSubnets) !== null && _c !== void 0 ? _c : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            instanceType: props.instanceType ? props.instanceType : aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.C5, aws_ec2_1.InstanceSize.LARGE),
            minCapacity: (_d = props.desiredCount) !== null && _d !== void 0 ? _d : 1,
            maxCapacity: (_e = props.desiredCount) !== null && _e !== void 0 ? _e : 1,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(30, { encrypted: true }),
                }],
            // addCapacity doesn't specifically take a securityGroup, but it passes on its properties to the ASG it creates,
            // so this security group will get applied there
            // @ts-ignore
            securityGroup: props.securityGroup,
        });
        const taskDefinition = new aws_ecs_1.TaskDefinition(this, 'TaskDefinition', {
            compatibility: aws_ecs_1.Compatibility.EC2,
            networkMode: aws_ecs_1.NetworkMode.HOST,
        });
        this.grantPrincipal = taskDefinition.taskRole;
        const containerEnv = {
            UBL_CERTIFICATES_URI: '',
            UBL_LIMITS: usageBasedLicenses.join(';'),
            ...props.renderQueue.configureClientECS({
                hosts: [this.asg],
                grantee: this,
            }),
        };
        containerEnv.UBL_CERTIFICATES_URI = props.certificateSecret.secretArn;
        props.certificateSecret.grantRead(taskDefinition.taskRole);
        const prefix = (_g = (_f = props.logGroupProps) === null || _f === void 0 ? void 0 : _f.logGroupPrefix) !== null && _g !== void 0 ? _g : UsageBasedLicensing.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...props.logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, 'LogGroupWrapper', id, defaultedLogGroupProps);
        logGroup.grantWrite(this.asg);
        const container = taskDefinition.addContainer('LicenseForwarderContainer', {
            image: props.images.licenseForwarder,
            environment: containerEnv,
            memoryReservationMiB: 1024,
            logging: aws_ecs_1.LogDriver.awsLogs({
                logGroup,
                streamPrefix: 'LicenseForwarder',
            }),
        });
        // Increase ulimits
        container.addUlimits({
            name: aws_ecs_1.UlimitName.NOFILE,
            softLimit: 200000,
            hardLimit: 200000,
        }, {
            name: aws_ecs_1.UlimitName.NPROC,
            softLimit: 64000,
            hardLimit: 64000,
        });
        this.service = new aws_ecs_1.Ec2Service(this, 'Service', {
            cluster: this.cluster,
            taskDefinition,
            desiredCount: (_h = props.desiredCount) !== null && _h !== void 0 ? _h : 1,
            placementConstraints: [aws_ecs_1.PlacementConstraint.distinctInstances()],
            // This is required to right-size our host capacity and not have the ECS service block on updates. We set a memory
            // reservation, but no memory limit on the container. This allows the container's memory usage to grow unbounded.
            // We want 1:1 container to container instances to not over-spend, but this comes at the price of down-time during
            // cloudformation updates.
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
        });
        // An explicit dependency is required from the service to the ASG providing its capacity.
        // See: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html
        this.service.node.addDependency(this.asg);
        this.node.defaultChild = this.service;
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * This method grant access of worker fleet to ports that required.
     *
     * @param workerFleet - worker fleet.
     * @param licenses - UBL licenses.
     * @stability stable
     */
    grantPortAccess(workerFleet, licenses) {
        licenses.forEach(license => {
            license.ports.forEach(port => {
                workerFleet.connections.allowTo(this, port);
            });
        });
        workerFleet.connections.allowTo(this, aws_ec2_1.Port.tcp(UsageBasedLicensing.LF_PORT));
    }
    /**
     * The connections object that allows you to control network egress/ingress to the License Forwarder.
     *
     * @stability stable
     */
    get connections() {
        return this.service.connections;
    }
}
exports.UsageBasedLicensing = UsageBasedLicensing;
_b = JSII_RTTI_SYMBOL_1;
UsageBasedLicensing[_b] = { fqn: "aws-rfdk.deadline.UsageBasedLicensing", version: "0.34.0" };
/**
 * The port that the License Forwarder listens on
 */
UsageBasedLicensing.LF_PORT = 17004;
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
UsageBasedLicensing.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
//# sourceMappingURL=data:application/json;base64,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