"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ThinkboxDockerImages = exports.AwsThinkboxEulaAcceptance = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto_1 = require("crypto");
const path = require("path");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
/**
 * Choices for signifying the user's stance on the terms of the AWS Thinkbox End-User License Agreement (EULA).
 *
 * See: https://www.awsthinkbox.com/end-user-license-agreement
 *
 * @stability stable
 */
var AwsThinkboxEulaAcceptance;
(function (AwsThinkboxEulaAcceptance) {
    AwsThinkboxEulaAcceptance[AwsThinkboxEulaAcceptance["USER_REJECTS_AWS_THINKBOX_EULA"] = 0] = "USER_REJECTS_AWS_THINKBOX_EULA";
    AwsThinkboxEulaAcceptance[AwsThinkboxEulaAcceptance["USER_ACCEPTS_AWS_THINKBOX_EULA"] = 1] = "USER_ACCEPTS_AWS_THINKBOX_EULA";
})(AwsThinkboxEulaAcceptance = exports.AwsThinkboxEulaAcceptance || (exports.AwsThinkboxEulaAcceptance = {}));
/**
 * An API for interacting with publicly available Deadline container images published by AWS Thinkbox.
 *
 * This provides container images as required by RFDK's Deadline constructs such as
 *
 * * {@link @aws-rfdk/deadline#RenderQueue}
 * * {@link @aws-rfdk/deadline#UsageBasedLicensing}
 *
 * Successful usage of the published Deadline container images with this class requires:
 *
 * 1) Explicit acceptance of the terms of the AWS Thinkbox End User License Agreement, under which Deadline is
 *     distributed; and
 * 2) The lambda on which the custom resource looks up the Thinkbox container images is able to make HTTPS
 *     requests to the official AWS Thinbox download site: https://downloads.thinkboxsoftware.com
 *
 * Resources Deployed
 * ------------------------
 * - A Lambda function containing a script to look up the AWS Thinkbox container image registry
 *
 * Security Considerations
 * ------------------------
 * - CDK deploys the code for this lambda as an S3 object in the CDK bootstrap bucket. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly
 *    recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS
 *    CloudTrail on your account to assist in post-incident analysis of compromised production environments.
 *
 * For example, to construct a RenderQueue using the images:
 *
 * ```ts
 * import { App, Stack, Vpc } from '@aws-rfdk/core';
 * import { AwsThinkboxEulaAcceptance, RenderQueue, Repository, ThinkboxDockerImages, VersionQuery } from '@aws-rfdk/deadline';
 * const app = new App();
 * const stack = new Stack(app, 'Stack');
 * const vpc = new Vpc(stack, 'Vpc');
 * const version = new VersionQuery(stack, 'Version', {
 *    version: '10.1.12',
 * });
 * const images = new ThinkboxDockerImages(stack, 'Image', {
 *    version,
 *    // Change this to AwsThinkboxEulaAcceptance.USER_ACCEPTS_AWS_THINKBOX_EULA to accept the terms
 *    // of the AWS Thinkbox End User License Agreement
 *    userAwsThinkboxEulaAcceptance: AwsThinkboxEulaAcceptance.USER_REJECTS_AWS_THINKBOX_EULA,
 * });
 * const repository = new Repository(stack, 'Repository', {
 *    vpc,
 *    version,
 * });
 *
 * const renderQueue = new RenderQueue(stack, 'RenderQueue', {
 *    images: images.forRenderQueue(),
 *    // ...
 * });
 * ```
 *
 * @stability stable
 */
class ThinkboxDockerImages extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.userAwsThinkboxEulaAcceptance = props.userAwsThinkboxEulaAcceptance;
        this.version = props === null || props === void 0 ? void 0 : props.version;
        const lambdaCode = aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'));
        const lambdaFunc = new aws_lambda_1.SingletonFunction(this, 'VersionProviderFunction', {
            uuid: '08553416-1fc9-4be9-a818-609a31ae1b5b',
            description: 'Used by the ThinkboxDockerImages construct to look up the ECR repositories where AWS Thinkbox publishes Deadline container images.',
            code: lambdaCode,
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'ecr-provider.handler',
            timeout: core_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        const ecrProvider = new core_1.CustomResource(this, 'ThinkboxEcrProvider', {
            serviceToken: lambdaFunc.functionArn,
            properties: {
                // create a random string that will force the Lambda to "update" on each deployment. Changes to its output will
                // be propagated to any CloudFormation resource providers that reference the output ARN
                ForceRun: this.forceRun(),
            },
            resourceType: 'Custom::RFDK_EcrProvider',
        });
        this.node.defaultChild = ecrProvider;
        this.ecrBaseURI = ecrProvider.getAtt('EcrURIPrefix').toString();
        this.remoteConnectionServer = this.ecrImageForRecipe(_1.ThinkboxManagedDeadlineDockerRecipes.REMOTE_CONNECTION_SERVER);
        this.licenseForwarder = this.ecrImageForRecipe(_1.ThinkboxManagedDeadlineDockerRecipes.LICENSE_FORWARDER);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    onValidate() {
        const errors = [];
        // Users must accept the AWS Thinkbox EULA to use the container images
        if (this.userAwsThinkboxEulaAcceptance !== AwsThinkboxEulaAcceptance.USER_ACCEPTS_AWS_THINKBOX_EULA) {
            errors.push(ThinkboxDockerImages.AWS_THINKBOX_EULA_MESSAGE);
        }
        // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
        // a resource in another stack is referencing it.
        if (this.version instanceof _1.VersionQuery) {
            const versionStack = core_1.Stack.of(this.version);
            const thisStack = core_1.Stack.of(this);
            if (versionStack != thisStack) {
                errors.push('A VersionQuery can not be supplied from a different stack');
            }
        }
        return errors;
    }
    ecrImageForRecipe(recipe) {
        let registryName = `${this.ecrBaseURI}${recipe}`;
        if (this.versionString) {
            registryName += `:${this.versionString}`;
        }
        return aws_ecs_1.ContainerImage.fromRegistry(registryName);
    }
    /**
     * Returns container images for use with the {@link RenderQueue} construct.
     *
     * @stability stable
     */
    forRenderQueue() {
        return this;
    }
    /**
     * Returns container images for use with the {@link UsageBasedLicensing} construct.
     *
     * @stability stable
     */
    forUsageBasedLicensing() {
        return this;
    }
    /**
     * A string representation of the Deadline version to retrieve images for.
     *
     * This can be undefined - in which case the latest available version of Deadline is used.
     */
    get versionString() {
        function numAsString(num) {
            return core_1.Token.isUnresolved(num) ? core_1.Token.asString(num) : num.toString();
        }
        const version = this.version;
        if (version) {
            const major = numAsString(version.majorVersion);
            const minor = numAsString(version.minorVersion);
            const release = numAsString(version.releaseVersion);
            return `${major}.${minor}.${release}`;
        }
        return undefined;
    }
    forceRun() {
        return crypto_1.randomBytes(32).toString('base64').slice(0, 32);
    }
}
exports.ThinkboxDockerImages = ThinkboxDockerImages;
_a = JSII_RTTI_SYMBOL_1;
ThinkboxDockerImages[_a] = { fqn: "aws-rfdk.deadline.ThinkboxDockerImages", version: "0.34.0" };
/**
 * The AWS Thinkbox licensing message that is presented to the user if they create an instance of
 * this class without explicitly accepting the AWS Thinkbox EULA.
 *
 * Note to developers: The text of this string is a legal requirement, and must not be altered
 * witout approval.
 */
ThinkboxDockerImages.AWS_THINKBOX_EULA_MESSAGE = `
The ThinkboxDockerImages will install Deadline onto one or more EC2 instances.

Deadline is provided by AWS Thinkbox under the AWS Thinkbox End User License
Agreement (EULA). By installing Deadline, you are agreeing to the terms of this
license. Follow the link below to read the terms of the AWS Thinkbox EULA.

https://www.awsthinkbox.com/end-user-license-agreement

By using the ThinkboxDockerImages to install Deadline you agree to the terms of
the AWS Thinkbox EULA.

Please set the userAwsThinkboxEulaAcceptance property to
USER_ACCEPTS_AWS_THINKBOX_EULA to signify your acceptance of the terms of the
AWS Thinkbox EULA.
`;
//# sourceMappingURL=data:application/json;base64,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