"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RenderQueue = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path_1 = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_ecs_patterns_1 = require("@aws-cdk/aws-ecs-patterns");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const rq_connection_1 = require("./rq-connection");
const version_1 = require("./version");
const wait_for_stable_service_1 = require("./wait-for-stable-service");
/**
 * Base class for Render Queue providers
 */
class RenderQueueBase extends core_1.Construct {
}
/**
 * The RenderQueue construct deploys an Elastic Container Service (ECS) service that serves Deadline's REST HTTP API to Deadline Clients.
 *
 * Most Deadline clients will connect to a Deadline render farm via the the RenderQueue. The API provides Deadline
 * clients access to Deadline's database and repository file-system in a way that is secure, performant, and scalable.
 *
 * Resources Deployed
 * ------------------------
 * - An Amazon Elastic Container Service (ECS) cluster.
 * - An AWS EC2 auto-scaling group that provides the instances that host the ECS service.
 * - An ECS service with a task definition that deploys the Deadline Remote Connetion Server (RCS) in a container.
 * - A Amazon CloudWatch log group for streaming logs from the Deadline RCS.
 * - An application load balancer, listener and target group that balance incoming traffic among the RCS containers.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - Care must be taken to secure what can connect to the RenderQueue. The RenderQueue does not authenticate API
 *    requests made against it. You must limit access to the RenderQueue endpoint to only trusted hosts. Those hosts
 *    should be governed carefully, as malicious software could use the API to remotely execute code across the entire render farm.
 * - The RenderQueue can be deployed with network encryption through Transport Layer Security (TLS) or without it. Unencrypted
 *    network communications can be eavesdropped upon or modified in transit. We strongly recommend deploying the RenderQueue
 *    with TLS enabled in production environments.
 *
 * @stability stable
 */
class RenderQueue extends RenderQueueBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0;
        super(scope, id);
        /**
         * Whether SEP policies have been added
         */
        this.haveAddedSEPPolicies = false;
        /**
         * Whether Resource Tracker policies have been added
         */
        this.haveAddedResourceTrackerPolicies = false;
        this.renderQueueSize = (_b = props === null || props === void 0 ? void 0 : props.renderQueueSize) !== null && _b !== void 0 ? _b : { min: 1, max: 1 };
        if (props.version.isLessThan(RenderQueue.MINIMUM_LOAD_BALANCING_VERSION)) {
            // Deadline versions earlier than 10.1.10 do not support horizontal scaling behind a load-balancer, so we limit to at most one instance
            if (((_c = this.renderQueueSize.min) !== null && _c !== void 0 ? _c : 0) > 1) {
                throw new Error(`renderQueueSize.min for Deadline version less than ${RenderQueue.MINIMUM_LOAD_BALANCING_VERSION.toString()} cannot be greater than 1 - got ${this.renderQueueSize.min}`);
            }
            if (((_d = this.renderQueueSize.desired) !== null && _d !== void 0 ? _d : 0) > 1) {
                throw new Error(`renderQueueSize.desired for Deadline version less than ${RenderQueue.MINIMUM_LOAD_BALANCING_VERSION.toString()} cannot be greater than 1 - got ${this.renderQueueSize.desired}`);
            }
            if (((_e = this.renderQueueSize.max) !== null && _e !== void 0 ? _e : 0) > 1) {
                throw new Error(`renderQueueSize.max for Deadline version less than ${RenderQueue.MINIMUM_LOAD_BALANCING_VERSION.toString()} cannot be greater than 1 - got ${this.renderQueueSize.max}`);
            }
        }
        this.version = props === null || props === void 0 ? void 0 : props.version;
        let externalProtocol;
        if ((_f = props.trafficEncryption) === null || _f === void 0 ? void 0 : _f.externalTLS) {
            externalProtocol = aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
            if ((props.trafficEncryption.externalTLS.acmCertificate === undefined) ===
                (props.trafficEncryption.externalTLS.rfdkCertificate === undefined)) {
                throw new Error('Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS.');
            }
            else if (props.trafficEncryption.externalTLS.rfdkCertificate) {
                if (props.trafficEncryption.externalTLS.rfdkCertificate.certChain === undefined) {
                    throw new Error('Provided rfdkCertificate does not contain a certificate chain.');
                }
                this.clientCert = new core_2.ImportedAcmCertificate(this, 'AcmCert', props.trafficEncryption.externalTLS.rfdkCertificate);
                this.certChain = props.trafficEncryption.externalTLS.rfdkCertificate.certChain;
            }
            else {
                if (props.trafficEncryption.externalTLS.acmCertificateChain === undefined) {
                    throw new Error('externalTLS.acmCertificateChain must be provided when using externalTLS.acmCertificate.');
                }
                this.clientCert = props.trafficEncryption.externalTLS.acmCertificate;
                this.certChain = props.trafficEncryption.externalTLS.acmCertificateChain;
            }
        }
        else {
            externalProtocol = aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP;
        }
        this.version = props.version;
        const internalProtocol = (_h = (_g = props.trafficEncryption) === null || _g === void 0 ? void 0 : _g.internalProtocol) !== null && _h !== void 0 ? _h : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS && !props.hostname) {
            throw new Error('A hostname must be provided when the external protocol is HTTPS');
        }
        this.cluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            vpc: props.vpc,
        });
        const minCapacity = (_k = (_j = props.renderQueueSize) === null || _j === void 0 ? void 0 : _j.min) !== null && _k !== void 0 ? _k : 1;
        if (minCapacity < 1) {
            throw new Error(`renderQueueSize.min capacity must be at least 1: got ${minCapacity}`);
        }
        const maxCapacity = (_l = this.renderQueueSize.max) !== null && _l !== void 0 ? _l : (_m = this.renderQueueSize) === null || _m === void 0 ? void 0 : _m.desired;
        if (((_o = this.renderQueueSize) === null || _o === void 0 ? void 0 : _o.desired) && maxCapacity && ((_p = this.renderQueueSize) === null || _p === void 0 ? void 0 : _p.desired) > maxCapacity) {
            throw new Error(`renderQueueSize.desired capacity cannot be more than ${maxCapacity}: got ${this.renderQueueSize.desired}`);
        }
        this.asg = this.cluster.addCapacity('RCS Capacity', {
            vpcSubnets: (_q = props.vpcSubnets) !== null && _q !== void 0 ? _q : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            instanceType: (_r = props.instanceType) !== null && _r !== void 0 ? _r : new aws_ec2_1.InstanceType('c5.large'),
            minCapacity,
            desiredCapacity: (_s = this.renderQueueSize) === null || _s === void 0 ? void 0 : _s.desired,
            maxCapacity,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    // See: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-ami-storage-config.html
                    // We want the volume to be encrypted. The default AMI size is 30-GiB.
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(30, { encrypted: true }),
                }],
            updateType: undefined,
            updatePolicy: aws_autoscaling_1.UpdatePolicy.rollingUpdate(),
            // addCapacity doesn't specifically take a securityGroup, but it passes on its properties to the ASG it creates,
            // so this security group will get applied there
            // @ts-ignore
            securityGroup: (_t = props.securityGroups) === null || _t === void 0 ? void 0 : _t.backend,
        });
        /**
         * The ECS-optimized AMI that is defaulted to when adding capacity to a cluster does not include the awscli or unzip
         * packages as is the case with the standard Amazon Linux AMI. These are required by RFDK scripts to configure the
         * direct connection on the host container instances.
         */
        this.asg.userData.addCommands('yum install -yq awscli unzip');
        if ((_u = props.enableLocalFileCaching) !== null && _u !== void 0 ? _u : false) {
            // Has to be done before any filesystems mount.
            this.enableFilecaching(this.asg);
        }
        const externalPortNumber = RenderQueue.RCS_PROTO_PORTS[externalProtocol];
        const internalPortNumber = RenderQueue.RCS_PROTO_PORTS[internalProtocol];
        this.logGroup = core_2.LogGroupFactory.createOrFetch(this, 'LogGroupWrapper', id, {
            logGroupPrefix: '/renderfarm/',
            ...props.logGroupProps,
        });
        this.logGroup.grantWrite(this.asg);
        const taskDefinition = this.createTaskDefinition({
            image: props.images.remoteConnectionServer,
            portNumber: internalPortNumber,
            protocol: internalProtocol,
            repository: props.repository,
        });
        this.taskDefinition = taskDefinition;
        // The fully-qualified domain name to use for the ALB
        let loadBalancerFQDN;
        if (props.hostname) {
            const label = (_v = props.hostname.hostname) !== null && _v !== void 0 ? _v : 'renderqueue';
            if (props.hostname.hostname && !RenderQueue.RE_VALID_HOSTNAME.test(label)) {
                throw new Error(`Invalid RenderQueue hostname: ${label}`);
            }
            loadBalancerFQDN = `${label}.${props.hostname.zone.zoneName}`;
        }
        const loadBalancer = new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, 'LB', {
            vpc: this.cluster.vpc,
            vpcSubnets: (_w = props.vpcSubnetsAlb) !== null && _w !== void 0 ? _w : { subnetType: aws_ec2_1.SubnetType.PRIVATE, onePerAz: true },
            internetFacing: false,
            deletionProtection: (_x = props.deletionProtection) !== null && _x !== void 0 ? _x : true,
            securityGroup: (_y = props.securityGroups) === null || _y === void 0 ? void 0 : _y.frontend,
        });
        this.pattern = new aws_ecs_patterns_1.ApplicationLoadBalancedEc2Service(this, 'AlbEc2ServicePattern', {
            certificate: this.clientCert,
            cluster: this.cluster,
            desiredCount: (_z = this.renderQueueSize) === null || _z === void 0 ? void 0 : _z.desired,
            domainZone: (_0 = props.hostname) === null || _0 === void 0 ? void 0 : _0.zone,
            domainName: loadBalancerFQDN,
            listenerPort: externalPortNumber,
            loadBalancer,
            protocol: externalProtocol,
            taskDefinition,
            // This is required to right-size our host capacity and not have the ECS service block on updates. We set a memory
            // reservation, but no memory limit on the container. This allows the container's memory usage to grow unbounded.
            // We want 1:1 container to container instances to not over-spend, but this comes at the price of down-time during
            // cloudformation updates.
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
            // This is required to ensure that the ALB listener's security group does not allow any ingress by default.
            openListener: false,
        });
        // An explicit dependency is required from the Service to the Client certificate
        // Otherwise cloud formation will try to remove the cert before the ALB using it is disposed.
        if (this.clientCert) {
            this.pattern.node.addDependency(this.clientCert);
        }
        // An explicit dependency is required from the service to the ASG providing its capacity.
        // See: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html
        this.pattern.service.node.addDependency(this.asg);
        this.loadBalancer = this.pattern.loadBalancer;
        // Enabling dropping of invalid HTTP header fields on the load balancer to prevent http smuggling attacks.
        this.loadBalancer.setAttribute('routing.http.drop_invalid_header_fields.enabled', 'true');
        if (props.accessLogs) {
            const accessLogsBucket = props.accessLogs.destinationBucket;
            // Policies are applied according to
            // https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-access-logs.html
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:PutObject'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [`${accessLogsBucket.bucketArn}/*`],
                conditions: {
                    StringEquals: {
                        's3:x-amz-acl': 'bucket-owner-full-control',
                    },
                },
            }));
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:GetBucketAcl'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [accessLogsBucket.bucketArn],
            }));
            this.loadBalancer.logAccessLogs(accessLogsBucket, props.accessLogs.prefix);
        }
        // Ensure tasks are run on separate container instances
        this.pattern.service.addPlacementConstraints(aws_ecs_1.PlacementConstraint.distinctInstances());
        /**
         * Uses an escape-hatch to set the target group protocol to HTTPS. We cannot configure server certificate
         * validation, but at least traffic is encrypted and terminated at the application layer.
         */
        const listener = this.loadBalancer.node.findChild('PublicListener');
        this.listener = listener;
        const targetGroup = listener.node.findChild('ECSGroup');
        const targetGroupResource = targetGroup.node.defaultChild;
        targetGroupResource.protocol = aws_elasticloadbalancingv2_1.ApplicationProtocol[internalProtocol];
        targetGroupResource.port = internalPortNumber;
        this.grantPrincipal = taskDefinition.taskRole;
        this.connections = new aws_ec2_1.Connections({
            defaultPort: aws_ec2_1.Port.tcp(externalPortNumber),
            securityGroups: this.pattern.loadBalancer.connections.securityGroups,
        });
        this.endpoint = new core_2.ConnectableApplicationEndpoint({
            address: loadBalancerFQDN !== null && loadBalancerFQDN !== void 0 ? loadBalancerFQDN : this.pattern.loadBalancer.loadBalancerDnsName,
            port: externalPortNumber,
            connections: this.connections,
            protocol: externalProtocol,
        });
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP) {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttp({
                endpoint: this.endpoint,
            });
        }
        else {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttps({
                endpoint: this.endpoint,
                caCert: this.certChain,
            });
        }
        this.ecsServiceStabilized = new wait_for_stable_service_1.WaitForStableService(this, 'WaitForStableService', {
            service: this.pattern.service,
        });
        this.node.defaultChild = taskDefinition;
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    onValidate() {
        const validationErrors = [];
        // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
        // a resource in another stack is referencing it.
        if (this.version instanceof _1.VersionQuery) {
            const versionStack = core_1.Stack.of(this.version);
            const thisStack = core_1.Stack.of(this);
            if (versionStack != thisStack) {
                validationErrors.push('A VersionQuery can not be supplied from a different stack');
            }
        }
        return validationErrors;
    }
    /**
     * Configures an ECS cluster to be able to connect to a RenderQueue.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientECS(param) {
        param.hosts.forEach(host => this.addChildDependency(host));
        return this.rqConnection.configureClientECS(param);
    }
    /**
     * Configure an Instance/Autoscaling group to connect to a RenderQueue.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientInstance(param) {
        this.addChildDependency(param.host);
        this.rqConnection.configureClientInstance(param);
    }
    /**
     * Adds AWS Managed Policies to the Render Queue so it is able to control Deadline's Spot Event Plugin.
     *
     * See: https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html for additonal information.
     *
     * @param includeResourceTracker Whether or not the Resource tracker admin policy should also be added (Default: True).
     * @stability stable
     */
    addSEPPolicies(includeResourceTracker = true) {
        if (!this.haveAddedSEPPolicies) {
            const sepPolicy = aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy');
            this.taskDefinition.taskRole.addManagedPolicy(sepPolicy);
            this.haveAddedSEPPolicies = true;
        }
        if (!this.haveAddedResourceTrackerPolicies) {
            if (includeResourceTracker) {
                const rtPolicy = aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy');
                this.taskDefinition.taskRole.addManagedPolicy(rtPolicy);
                this.haveAddedResourceTrackerPolicies = true;
            }
        }
    }
    /**
     * Add an ordering dependency to another Construct.
     *
     * All constructs in the child's scope will be deployed after the RenderQueue has been deployed and is ready to recieve traffic.
     *
     * This can be used to ensure that the RenderQueue is fully up and serving queries before a client attempts to connect to it.
     *
     * @param child The child to make dependent upon this RenderQueue.
     * @stability stable
     */
    addChildDependency(child) {
        // Narrowly define the dependencies to reduce the probability of cycles
        // ex: cycles that involve the security group of the RenderQueue & child.
        child.node.addDependency(this.listener);
        child.node.addDependency(this.taskDefinition);
        child.node.addDependency(this.pattern.service);
        child.node.addDependency(this.ecsServiceStabilized);
    }
    /**
     * Adds security groups to the frontend of the Render Queue, which is its load balancer.
     *
     * @param securityGroups The security groups to add.
     * @stability stable
     */
    addFrontendSecurityGroups(...securityGroups) {
        securityGroups.forEach(securityGroup => this.loadBalancer.addSecurityGroup(securityGroup));
    }
    /**
     * Adds security groups to the backend components of the Render Queue, which consists of the AutoScalingGroup for the Deadline RCS.
     *
     * @param securityGroups The security groups to add.
     * @stability stable
     */
    addBackendSecurityGroups(...securityGroups) {
        securityGroups.forEach(securityGroup => this.asg.addSecurityGroup(securityGroup));
    }
    enableFilecaching(asg) {
        const script = core_2.ScriptAsset.fromPathConvention(this, 'FilecachingScript', {
            osType: asg.osType,
            baseName: 'enableCacheFilesd',
            rootDir: path_1.join(__dirname, '..', 'scripts'),
        });
        // A comment in userData to make this easier to test.
        asg.userData.addCommands('# RenderQueue file caching enabled');
        script.executeOn({
            host: asg,
        });
    }
    createTaskDefinition(props) {
        const { image, portNumber, protocol, repository } = props;
        const taskDefinition = new aws_ecs_1.Ec2TaskDefinition(this, 'RCSTask');
        // Mount the repo filesystem to RenderQueue.HOST_REPO_FS_MOUNT_PATH
        const connection = repository.configureClientECS({
            containerInstances: {
                hosts: [this.asg],
            },
            containers: {
                taskDefinition,
            },
        });
        const environment = connection.containerEnvironment;
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            // Generate a self-signed X509 certificate, private key and passphrase for use by the RCS containers.
            // Note: the Application Load Balancer does not validate the certificate in any way.
            const rcsCertPem = new core_2.X509CertificatePem(this, 'TlsCaCertPem', {
                subject: {
                    cn: 'renderfarm.local',
                },
            });
            const rcsCertPkcs = new core_2.X509CertificatePkcs12(this, 'TlsRcsCertBundle', {
                sourceCertificate: rcsCertPem,
            });
            [rcsCertPem.cert, rcsCertPkcs.cert, rcsCertPkcs.passphrase].forEach(secret => {
                secret.grantRead(taskDefinition.taskRole);
            });
            environment.RCS_TLS_CA_CERT_URI = rcsCertPem.cert.secretArn;
            environment.RCS_TLS_CERT_URI = rcsCertPkcs.cert.secretArn;
            environment.RCS_TLS_CERT_PASSPHRASE_URI = rcsCertPkcs.passphrase.secretArn;
            environment.RCS_TLS_REQUIRE_CLIENT_CERT = 'no';
        }
        const containerDefinition = taskDefinition.addContainer('ContainerDefinition', {
            image,
            memoryReservationMiB: 2048,
            environment,
            logging: aws_ecs_1.LogDriver.awsLogs({
                logGroup: this.logGroup,
                streamPrefix: 'RCS',
            }),
        });
        containerDefinition.addMountPoints(connection.readWriteMountPoint);
        // Increase ulimits
        containerDefinition.addUlimits({
            name: aws_ecs_1.UlimitName.NOFILE,
            softLimit: 200000,
            hardLimit: 200000,
        }, {
            name: aws_ecs_1.UlimitName.NPROC,
            softLimit: 64000,
            hardLimit: 64000,
        });
        containerDefinition.addPortMappings({
            containerPort: portNumber,
            hostPort: portNumber,
        });
        return taskDefinition;
    }
}
exports.RenderQueue = RenderQueue;
_a = JSII_RTTI_SYMBOL_1;
RenderQueue[_a] = { fqn: "aws-rfdk.deadline.RenderQueue", version: "0.34.0" };
/**
 * Container listening ports for each protocol.
 */
RenderQueue.RCS_PROTO_PORTS = {
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP]: 8080,
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS]: 4433,
};
/**
* The minimum Deadline version required for the Remote Connection Server to support load-balancing
*/
RenderQueue.MINIMUM_LOAD_BALANCING_VERSION = new version_1.Version([10, 1, 10, 0]);
/**
 * Regular expression that validates a hostname (portion in front of the subdomain).
 */
RenderQueue.RE_VALID_HOSTNAME = /^[a-z](?:[a-z0-9-]{0,61}[a-z0-9])?$/i;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVuZGVyLXF1ZXVlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsicmVuZGVyLXF1ZXVlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUE7OztHQUdHO0FBRUgsK0JBRWM7QUFDZCw4REFJa0M7QUFJbEMsOENBTzBCO0FBQzFCLDhDQU8wQjtBQUMxQixnRUFFbUM7QUFDbkMsb0ZBTTZDO0FBQzdDLDhDQU0wQjtBQU8xQix3Q0FJdUI7QUFFdkIsd0JBUVc7QUFFWCxxQ0FPb0I7QUFDcEIsOERBRXFDO0FBQ3JDLG1EQUV5QjtBQUN6Qix1Q0FBb0M7QUFDcEMsdUVBRW1DO0FBY25DOztHQUVHO0FBQ0gsTUFBZSxlQUFnQixTQUFRLGdCQUFTO0NBcUIvQzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztBQUdELE1BQWEsV0FBWSxTQUFRLGVBQWU7Ozs7SUE4RjlDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBdUI7O1FBQy9ELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFwRG5COztXQUVHO1FBQ0sseUJBQW9CLEdBQVksS0FBSyxDQUFDO1FBRTlDOztXQUVHO1FBQ0sscUNBQWdDLEdBQVksS0FBSyxDQUFDO1FBOEN4RCxJQUFJLENBQUMsZUFBZSxTQUFHLEtBQUssYUFBTCxLQUFLLHVCQUFMLEtBQUssQ0FBRSxlQUFlLG1DQUFJLEVBQUMsR0FBRyxFQUFFLENBQUMsRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFDLENBQUM7UUFFbEUsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsOEJBQThCLENBQUMsRUFBRTtZQUN4RSx1SUFBdUk7WUFDdkksSUFBSSxPQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsR0FBRyxtQ0FBSSxDQUFDLENBQUMsR0FBRyxDQUFDLEVBQUU7Z0JBQ3ZDLE1BQU0sSUFBSSxLQUFLLENBQUMsc0RBQXNELFdBQVcsQ0FBQyw4QkFBOEIsQ0FBQyxRQUFRLEVBQUUsbUNBQW1DLElBQUksQ0FBQyxlQUFlLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQzthQUMzTDtZQUNELElBQUksT0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLE9BQU8sbUNBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxFQUFFO2dCQUMzQyxNQUFNLElBQUksS0FBSyxDQUFDLDBEQUEwRCxXQUFXLENBQUMsOEJBQThCLENBQUMsUUFBUSxFQUFFLG1DQUFtQyxJQUFJLENBQUMsZUFBZSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7YUFDbk07WUFDRCxJQUFJLE9BQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxHQUFHLG1DQUFJLENBQUMsQ0FBQyxHQUFHLENBQUMsRUFBRTtnQkFDdkMsTUFBTSxJQUFJLEtBQUssQ0FBQyxzREFBc0QsV0FBVyxDQUFDLDhCQUE4QixDQUFDLFFBQVEsRUFBRSxtQ0FBbUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO2FBQzNMO1NBQ0Y7UUFFRCxJQUFJLENBQUMsT0FBTyxHQUFHLEtBQUssYUFBTCxLQUFLLHVCQUFMLEtBQUssQ0FBRSxPQUFPLENBQUM7UUFFOUIsSUFBSSxnQkFBcUMsQ0FBQztRQUMxQyxVQUFLLEtBQUssQ0FBQyxpQkFBaUIsMENBQUUsV0FBVyxFQUFHO1lBQzFDLGdCQUFnQixHQUFHLGdEQUFtQixDQUFDLEtBQUssQ0FBQztZQUU3QyxJQUFLLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLFdBQVcsQ0FBQyxjQUFjLEtBQUssU0FBUyxDQUFFO2dCQUN4RSxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxXQUFXLENBQUMsZUFBZSxLQUFLLFNBQVMsQ0FBQyxFQUFHO2dCQUNwRSxNQUFNLElBQUksS0FBSyxDQUFDLG9IQUFvSCxDQUFDLENBQUM7YUFDdkk7aUJBQU0sSUFBSSxLQUFLLENBQUMsaUJBQWlCLENBQUMsV0FBVyxDQUFDLGVBQWUsRUFBRztnQkFDL0QsSUFBSSxLQUFLLENBQUMsaUJBQWlCLENBQUMsV0FBVyxDQUFDLGVBQWUsQ0FBQyxTQUFTLEtBQUssU0FBUyxFQUFFO29CQUMvRSxNQUFNLElBQUksS0FBSyxDQUFDLGdFQUFnRSxDQUFDLENBQUM7aUJBQ25GO2dCQUNELElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSw2QkFBc0IsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxXQUFXLENBQUMsZUFBZSxDQUFFLENBQUM7Z0JBQ3BILElBQUksQ0FBQyxTQUFTLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDLFdBQVcsQ0FBQyxlQUFlLENBQUMsU0FBUyxDQUFDO2FBQ2hGO2lCQUFNO2dCQUNMLElBQUksS0FBSyxDQUFDLGlCQUFpQixDQUFDLFdBQVcsQ0FBQyxtQkFBbUIsS0FBSyxTQUFTLEVBQUU7b0JBQ3pFLE1BQU0sSUFBSSxLQUFLLENBQUMseUZBQXlGLENBQUMsQ0FBQztpQkFDNUc7Z0JBQ0QsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsaUJBQWlCLENBQUMsV0FBVyxDQUFDLGNBQWMsQ0FBQztnQkFDckUsSUFBSSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsaUJBQWlCLENBQUMsV0FBVyxDQUFDLG1CQUFtQixDQUFDO2FBQzFFO1NBQ0Y7YUFBTTtZQUNMLGdCQUFnQixHQUFHLGdEQUFtQixDQUFDLElBQUksQ0FBQztTQUM3QztRQUVELElBQUksQ0FBQyxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUU3QixNQUFNLGdCQUFnQixlQUFHLEtBQUssQ0FBQyxpQkFBaUIsMENBQUUsZ0JBQWdCLG1DQUFJLGdEQUFtQixDQUFDLEtBQUssQ0FBQztRQUVoRyxJQUFJLGdCQUFnQixLQUFLLGdEQUFtQixDQUFDLEtBQUssSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUU7WUFDckUsTUFBTSxJQUFJLEtBQUssQ0FBQyxpRUFBaUUsQ0FBQyxDQUFDO1NBQ3BGO1FBRUQsSUFBSSxDQUFDLE9BQU8sR0FBRyxJQUFJLGlCQUFPLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUMxQyxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7U0FDZixDQUFDLENBQUM7UUFFSCxNQUFNLFdBQVcsZUFBRyxLQUFLLENBQUMsZUFBZSwwQ0FBRSxHQUFHLG1DQUFJLENBQUMsQ0FBQztRQUNwRCxJQUFJLFdBQVcsR0FBRyxDQUFDLEVBQUU7WUFDbkIsTUFBTSxJQUFJLEtBQUssQ0FBQyx3REFBd0QsV0FBVyxFQUFFLENBQUMsQ0FBQztTQUN4RjtRQUNELE1BQU0sV0FBVyxTQUFHLElBQUksQ0FBQyxlQUFlLENBQUMsR0FBRyx5Q0FBSSxJQUFJLENBQUMsZUFBZSwwQ0FBRSxPQUFPLENBQUM7UUFDOUUsSUFBSSxPQUFBLElBQUksQ0FBQyxlQUFlLDBDQUFFLE9BQU8sS0FBSSxXQUFXLElBQUksT0FBQSxJQUFJLENBQUMsZUFBZSwwQ0FBRSxPQUFPLElBQUcsV0FBVyxFQUFFO1lBQy9GLE1BQU0sSUFBSSxLQUFLLENBQUMsd0RBQXdELFdBQVcsU0FBUyxJQUFJLENBQUMsZUFBZSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7U0FDN0g7UUFDRCxJQUFJLENBQUMsR0FBRyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLGNBQWMsRUFBRTtZQUNsRCxVQUFVLFFBQUUsS0FBSyxDQUFDLFVBQVUsbUNBQUksRUFBRSxVQUFVLEVBQUUsb0JBQVUsQ0FBQyxPQUFPLEVBQUU7WUFDbEUsWUFBWSxRQUFFLEtBQUssQ0FBQyxZQUFZLG1DQUFJLElBQUksc0JBQVksQ0FBQyxVQUFVLENBQUM7WUFDaEUsV0FBVztZQUNYLGVBQWUsUUFBRSxJQUFJLENBQUMsZUFBZSwwQ0FBRSxPQUFPO1lBQzlDLFdBQVc7WUFDWCxZQUFZLEVBQUUsQ0FBQztvQkFDYixVQUFVLEVBQUUsV0FBVztvQkFDdkIsK0ZBQStGO29CQUMvRixzRUFBc0U7b0JBQ3RFLE1BQU0sRUFBRSxtQ0FBaUIsQ0FBQyxHQUFHLENBQUMsRUFBRSxFQUFFLEVBQUUsU0FBUyxFQUFFLElBQUksRUFBRSxDQUFDO2lCQUN2RCxDQUFDO1lBQ0YsVUFBVSxFQUFFLFNBQVM7WUFDckIsWUFBWSxFQUFFLDhCQUFZLENBQUMsYUFBYSxFQUFFO1lBQzFDLGdIQUFnSDtZQUNoSCxnREFBZ0Q7WUFDaEQsYUFBYTtZQUNiLGFBQWEsUUFBRSxLQUFLLENBQUMsY0FBYywwQ0FBRSxPQUFPO1NBQzdDLENBQUMsQ0FBQztRQUVIOzs7O1dBSUc7UUFDSCxJQUFJLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQzNCLDhCQUE4QixDQUMvQixDQUFDO1FBQ0YsVUFBSSxLQUFLLENBQUMsc0JBQXNCLG1DQUFJLEtBQUssRUFBRTtZQUN6QywrQ0FBK0M7WUFDL0MsSUFBSSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUNsQztRQUVELE1BQU0sa0JBQWtCLEdBQUcsV0FBVyxDQUFDLGVBQWUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ3pFLE1BQU0sa0JBQWtCLEdBQUcsV0FBVyxDQUFDLGVBQWUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBRXpFLElBQUksQ0FBQyxRQUFRLEdBQUcsc0JBQWUsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFLEVBQUUsRUFBRTtZQUN6RSxjQUFjLEVBQUUsY0FBYztZQUM5QixHQUFHLEtBQUssQ0FBQyxhQUFhO1NBQ3ZCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUVuQyxNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsb0JBQW9CLENBQUM7WUFDL0MsS0FBSyxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsc0JBQXNCO1lBQzFDLFVBQVUsRUFBRSxrQkFBa0I7WUFDOUIsUUFBUSxFQUFFLGdCQUFnQjtZQUMxQixVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7U0FDN0IsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLGNBQWMsR0FBRyxjQUFjLENBQUM7UUFFckMscURBQXFEO1FBQ3JELElBQUksZ0JBQW9DLENBQUM7UUFDekMsSUFBSSxLQUFLLENBQUMsUUFBUSxFQUFFO1lBQ2xCLE1BQU0sS0FBSyxTQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsUUFBUSxtQ0FBSSxhQUFhLENBQUM7WUFDdkQsSUFBSSxLQUFLLENBQUMsUUFBUSxDQUFDLFFBQVEsSUFBSSxDQUFDLFdBQVcsQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUU7Z0JBQ3pFLE1BQU0sSUFBSSxLQUFLLENBQUMsaUNBQWlDLEtBQUssRUFBRSxDQUFDLENBQUM7YUFDM0Q7WUFDRCxnQkFBZ0IsR0FBRyxHQUFHLEtBQUssSUFBSSxLQUFLLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQztTQUMvRDtRQUVELE1BQU0sWUFBWSxHQUFHLElBQUksb0RBQXVCLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRTtZQUMzRCxHQUFHLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHO1lBQ3JCLFVBQVUsUUFBRSxLQUFLLENBQUMsYUFBYSxtQ0FBSSxFQUFFLFVBQVUsRUFBRSxvQkFBVSxDQUFDLE9BQU8sRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFO1lBQ3JGLGNBQWMsRUFBRSxLQUFLO1lBQ3JCLGtCQUFrQixRQUFFLEtBQUssQ0FBQyxrQkFBa0IsbUNBQUksSUFBSTtZQUNwRCxhQUFhLFFBQUUsS0FBSyxDQUFDLGNBQWMsMENBQUUsUUFBUTtTQUM5QyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksb0RBQWlDLENBQUMsSUFBSSxFQUFFLHNCQUFzQixFQUFFO1lBQ2pGLFdBQVcsRUFBRSxJQUFJLENBQUMsVUFBVTtZQUM1QixPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87WUFDckIsWUFBWSxRQUFFLElBQUksQ0FBQyxlQUFlLDBDQUFFLE9BQU87WUFDM0MsVUFBVSxRQUFFLEtBQUssQ0FBQyxRQUFRLDBDQUFFLElBQUk7WUFDaEMsVUFBVSxFQUFFLGdCQUFnQjtZQUM1QixZQUFZLEVBQUUsa0JBQWtCO1lBQ2hDLFlBQVk7WUFDWixRQUFRLEVBQUUsZ0JBQWdCO1lBQzFCLGNBQWM7WUFDZCxrSEFBa0g7WUFDbEgsaUhBQWlIO1lBQ2pILGtIQUFrSDtZQUNsSCwwQkFBMEI7WUFDMUIsaUJBQWlCLEVBQUUsQ0FBQztZQUNwQixpQkFBaUIsRUFBRSxHQUFHO1lBQ3RCLDJHQUEyRztZQUMzRyxZQUFZLEVBQUUsS0FBSztTQUNwQixDQUFDLENBQUM7UUFFSCxnRkFBZ0Y7UUFDaEYsNkZBQTZGO1FBQzdGLElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNuQixJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQ2xEO1FBRUQseUZBQXlGO1FBQ3pGLG1HQUFtRztRQUNuRyxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUVsRCxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDO1FBQzlDLDBHQUEwRztRQUMxRyxJQUFJLENBQUMsWUFBWSxDQUFDLFlBQVksQ0FBQyxpREFBaUQsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUUxRixJQUFJLEtBQUssQ0FBQyxVQUFVLEVBQUU7WUFDcEIsTUFBTSxnQkFBZ0IsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDLGlCQUFpQixDQUFDO1lBRTVELG9DQUFvQztZQUNwQyxxR0FBcUc7WUFDckcsZ0JBQWdCLENBQUMsbUJBQW1CLENBQUUsSUFBSSx5QkFBZSxDQUFDO2dCQUN4RCxPQUFPLEVBQUUsQ0FBQyxjQUFjLENBQUM7Z0JBQ3pCLFVBQVUsRUFBRSxDQUFDLElBQUksMEJBQWdCLENBQUMsNkJBQTZCLENBQUMsQ0FBQztnQkFDakUsU0FBUyxFQUFFLENBQUMsR0FBRyxnQkFBZ0IsQ0FBQyxTQUFTLElBQUksQ0FBQztnQkFDOUMsVUFBVSxFQUFFO29CQUNWLFlBQVksRUFBRTt3QkFDWixjQUFjLEVBQUUsMkJBQTJCO3FCQUM1QztpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBQ0osZ0JBQWdCLENBQUMsbUJBQW1CLENBQUMsSUFBSSx5QkFBZSxDQUFDO2dCQUN2RCxPQUFPLEVBQUUsQ0FBRSxpQkFBaUIsQ0FBRTtnQkFDOUIsVUFBVSxFQUFFLENBQUUsSUFBSSwwQkFBZ0IsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO2dCQUNsRSxTQUFTLEVBQUUsQ0FBRSxnQkFBZ0IsQ0FBQyxTQUFTLENBQUU7YUFDMUMsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsWUFBWSxDQUFDLGFBQWEsQ0FDN0IsZ0JBQWdCLEVBQ2hCLEtBQUssQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDNUI7UUFFRCx1REFBdUQ7UUFDdkQsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsdUJBQXVCLENBQUMsNkJBQW1CLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyxDQUFDO1FBRXRGOzs7V0FHRztRQUNILE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ3BFLElBQUksQ0FBQyxRQUFRLEdBQUcsUUFBK0IsQ0FBQztRQUNoRCxNQUFNLFdBQVcsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQTJCLENBQUM7UUFDbEYsTUFBTSxtQkFBbUIsR0FBRyxXQUFXLENBQUMsSUFBSSxDQUFDLFlBQThCLENBQUM7UUFDNUUsbUJBQW1CLENBQUMsUUFBUSxHQUFHLGdEQUFtQixDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFDckUsbUJBQW1CLENBQUMsSUFBSSxHQUFHLGtCQUFrQixDQUFDO1FBRTlDLElBQUksQ0FBQyxjQUFjLEdBQUcsY0FBYyxDQUFDLFFBQVEsQ0FBQztRQUU5QyxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUkscUJBQVcsQ0FBQztZQUNqQyxXQUFXLEVBQUUsY0FBSSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQztZQUN6QyxjQUFjLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLGNBQWM7U0FDckUsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLHFDQUE4QixDQUFDO1lBQ2pELE9BQU8sRUFBRSxnQkFBZ0IsYUFBaEIsZ0JBQWdCLGNBQWhCLGdCQUFnQixHQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLG1CQUFtQjtZQUMxRSxJQUFJLEVBQUUsa0JBQWtCO1lBQ3hCLFdBQVcsRUFBRSxJQUFJLENBQUMsV0FBVztZQUM3QixRQUFRLEVBQUUsZ0JBQWdCO1NBQzNCLENBQUMsQ0FBQztRQUVILElBQUssZ0JBQWdCLEtBQUssZ0RBQW1CLENBQUMsSUFBSSxFQUFHO1lBQ25ELElBQUksQ0FBQyxZQUFZLEdBQUcscUNBQXFCLENBQUMsT0FBTyxDQUFDO2dCQUNoRCxRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7YUFDeEIsQ0FBQyxDQUFDO1NBQ0o7YUFBTTtZQUNMLElBQUksQ0FBQyxZQUFZLEdBQUcscUNBQXFCLENBQUMsUUFBUSxDQUFDO2dCQUNqRCxRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7Z0JBQ3ZCLE1BQU0sRUFBRSxJQUFJLENBQUMsU0FBVTthQUN4QixDQUFDLENBQUM7U0FDSjtRQUVELElBQUksQ0FBQyxvQkFBb0IsR0FBRyxJQUFJLDhDQUFvQixDQUFDLElBQUksRUFBRSxzQkFBc0IsRUFBRTtZQUNqRixPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPO1NBQzlCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxHQUFHLGNBQWMsQ0FBQztRQUV4Qyw2Q0FBNkM7UUFDN0MsMkJBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNyQixDQUFDOzs7Ozs7Ozs7SUFFUyxVQUFVO1FBQ2xCLE1BQU0sZ0JBQWdCLEdBQUcsRUFBRSxDQUFDO1FBRTVCLGlIQUFpSDtRQUNqSCxpREFBaUQ7UUFDakQsSUFBSSxJQUFJLENBQUMsT0FBTyxZQUFZLGVBQVksRUFBRTtZQUN4QyxNQUFNLFlBQVksR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUM1QyxNQUFNLFNBQVMsR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ2pDLElBQUksWUFBWSxJQUFJLFNBQVMsRUFBRTtnQkFDN0IsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLDJEQUEyRCxDQUFDLENBQUM7YUFDcEY7U0FDRjtRQUVELE9BQU8sZ0JBQWdCLENBQUM7SUFDMUIsQ0FBQzs7Ozs7OztJQUdNLGtCQUFrQixDQUFDLEtBQXdCO1FBQ2hELEtBQUssQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFFLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxDQUFFLENBQUM7UUFDN0QsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3JELENBQUM7Ozs7Ozs7SUFHTSx1QkFBdUIsQ0FBQyxLQUE2QjtRQUMxRCxJQUFJLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3BDLElBQUksQ0FBQyxZQUFZLENBQUMsdUJBQXVCLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDbkQsQ0FBQzs7Ozs7Ozs7O0lBR00sY0FBYyxDQUFDLHlCQUFrQyxJQUFJO1FBQzFELElBQUksQ0FBQyxJQUFJLENBQUMsb0JBQW9CLEVBQUU7WUFDOUIsTUFBTSxTQUFTLEdBQUcsdUJBQWEsQ0FBQyx3QkFBd0IsQ0FBQywrQ0FBK0MsQ0FBQyxDQUFDO1lBQzFHLElBQUksQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDLGdCQUFnQixDQUFDLFNBQVMsQ0FBQyxDQUFDO1lBQ3pELElBQUksQ0FBQyxvQkFBb0IsR0FBRyxJQUFJLENBQUM7U0FDbEM7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLGdDQUFnQyxFQUFFO1lBQzFDLElBQUksc0JBQXNCLEVBQUU7Z0JBQzFCLE1BQU0sUUFBUSxHQUFHLHVCQUFhLENBQUMsd0JBQXdCLENBQUMsK0NBQStDLENBQUMsQ0FBQztnQkFDekcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxDQUFDLENBQUM7Z0JBQ3hELElBQUksQ0FBQyxnQ0FBZ0MsR0FBRyxJQUFJLENBQUM7YUFDOUM7U0FDRjtJQUNILENBQUM7Ozs7Ozs7Ozs7O0lBR00sa0JBQWtCLENBQUMsS0FBaUI7UUFDekMsdUVBQXVFO1FBQ3ZFLHlFQUF5RTtRQUN6RSxLQUFLLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDeEMsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQzlDLEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDL0MsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdEQsQ0FBQzs7Ozs7OztJQUdNLHlCQUF5QixDQUFDLEdBQUcsY0FBZ0M7UUFDbEUsY0FBYyxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQztJQUM3RixDQUFDOzs7Ozs7O0lBR00sd0JBQXdCLENBQUMsR0FBRyxjQUFnQztRQUNqRSxjQUFjLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDO0lBQ3BGLENBQUM7SUFFTyxpQkFBaUIsQ0FBQyxHQUFxQjtRQUM3QyxNQUFNLE1BQU0sR0FBRyxrQkFBVyxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRSxtQkFBbUIsRUFBRTtZQUN2RSxNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU07WUFDbEIsUUFBUSxFQUFFLG1CQUFtQjtZQUM3QixPQUFPLEVBQUUsV0FBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLEVBQUUsU0FBUyxDQUFDO1NBQzFDLENBQUMsQ0FBQztRQUNILHFEQUFxRDtRQUNyRCxHQUFHLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO1FBQy9ELE1BQU0sQ0FBQyxTQUFTLENBQUM7WUFDZixJQUFJLEVBQUUsR0FBRztTQUNWLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTyxvQkFBb0IsQ0FBQyxLQUs1QjtRQUNDLE1BQU0sRUFBRSxLQUFLLEVBQUUsVUFBVSxFQUFFLFFBQVEsRUFBRSxVQUFVLEVBQUUsR0FBRyxLQUFLLENBQUM7UUFFMUQsTUFBTSxjQUFjLEdBQUcsSUFBSSwyQkFBaUIsQ0FBQyxJQUFJLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFFOUQsbUVBQW1FO1FBQ25FLE1BQU0sVUFBVSxHQUFHLFVBQVUsQ0FBQyxrQkFBa0IsQ0FBQztZQUMvQyxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQzthQUNsQjtZQUNELFVBQVUsRUFBRTtnQkFDVixjQUFjO2FBQ2Y7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLFdBQVcsR0FBRyxVQUFVLENBQUMsb0JBQW9CLENBQUM7UUFFcEQsSUFBSSxRQUFRLEtBQUssZ0RBQW1CLENBQUMsS0FBSyxFQUFFO1lBQzFDLHFHQUFxRztZQUNyRyxvRkFBb0Y7WUFDcEYsTUFBTSxVQUFVLEdBQUcsSUFBSSx5QkFBa0IsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO2dCQUM5RCxPQUFPLEVBQUU7b0JBQ1AsRUFBRSxFQUFFLGtCQUFrQjtpQkFDdkI7YUFDRixDQUFDLENBQUM7WUFDSCxNQUFNLFdBQVcsR0FBRyxJQUFJLDRCQUFxQixDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtnQkFDdEUsaUJBQWlCLEVBQUUsVUFBVTthQUM5QixDQUFDLENBQUM7WUFDSCxDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsV0FBVyxDQUFDLElBQUksRUFBRSxXQUFXLENBQUMsVUFBVSxDQUFDLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFO2dCQUMzRSxNQUFNLENBQUMsU0FBUyxDQUFDLGNBQWMsQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUM1QyxDQUFDLENBQUMsQ0FBQztZQUNILFdBQVcsQ0FBQyxtQkFBbUIsR0FBRyxVQUFVLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUM1RCxXQUFXLENBQUMsZ0JBQWdCLEdBQUcsV0FBVyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDMUQsV0FBVyxDQUFDLDJCQUEyQixHQUFHLFdBQVcsQ0FBQyxVQUFVLENBQUMsU0FBUyxDQUFDO1lBQzNFLFdBQVcsQ0FBQywyQkFBMkIsR0FBRyxJQUFJLENBQUM7U0FDaEQ7UUFFRCxNQUFNLG1CQUFtQixHQUFHLGNBQWMsQ0FBQyxZQUFZLENBQUMscUJBQXFCLEVBQUU7WUFDN0UsS0FBSztZQUNMLG9CQUFvQixFQUFFLElBQUk7WUFDMUIsV0FBVztZQUNYLE9BQU8sRUFBRSxtQkFBUyxDQUFDLE9BQU8sQ0FBQztnQkFDekIsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRO2dCQUN2QixZQUFZLEVBQUUsS0FBSzthQUNwQixDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBRUgsbUJBQW1CLENBQUMsY0FBYyxDQUFDLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBRW5FLG1CQUFtQjtRQUNuQixtQkFBbUIsQ0FBQyxVQUFVLENBQzVCO1lBQ0UsSUFBSSxFQUFFLG9CQUFVLENBQUMsTUFBTTtZQUN2QixTQUFTLEVBQUUsTUFBTTtZQUNqQixTQUFTLEVBQUUsTUFBTTtTQUNsQixFQUFFO1lBQ0QsSUFBSSxFQUFFLG9CQUFVLENBQUMsS0FBSztZQUN0QixTQUFTLEVBQUUsS0FBSztZQUNoQixTQUFTLEVBQUUsS0FBSztTQUNqQixDQUNGLENBQUM7UUFFRixtQkFBbUIsQ0FBQyxlQUFlLENBQUM7WUFDbEMsYUFBYSxFQUFFLFVBQVU7WUFDekIsUUFBUSxFQUFFLFVBQVU7U0FDckIsQ0FBQyxDQUFDO1FBRUgsT0FBTyxjQUFjLENBQUM7SUFDeEIsQ0FBQzs7QUF0ZUgsa0NBdWVDOzs7QUF0ZUM7O0dBRUc7QUFDcUIsMkJBQWUsR0FBRztJQUN4QyxDQUFDLGdEQUFtQixDQUFDLElBQUksQ0FBQyxFQUFFLElBQUk7SUFDaEMsQ0FBQyxnREFBbUIsQ0FBQyxLQUFLLENBQUMsRUFBRSxJQUFJO0NBQ2xDLENBQUM7QUFFRjs7RUFFRTtBQUNzQiwwQ0FBOEIsR0FBRyxJQUFJLGlCQUFPLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO0FBRXJGOztHQUVHO0FBQ3FCLDZCQUFpQixHQUFHLHNDQUFzQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0IHtcbiAgam9pbixcbn0gZnJvbSAncGF0aCc7XG5pbXBvcnQge1xuICBBdXRvU2NhbGluZ0dyb3VwLFxuICBCbG9ja0RldmljZVZvbHVtZSxcbiAgVXBkYXRlUG9saWN5LFxufSBmcm9tICdAYXdzLWNkay9hd3MtYXV0b3NjYWxpbmcnO1xuaW1wb3J0IHtcbiAgSUNlcnRpZmljYXRlLFxufSBmcm9tICdAYXdzLWNkay9hd3MtY2VydGlmaWNhdGVtYW5hZ2VyJztcbmltcG9ydCB7XG4gIENvbm5lY3Rpb25zLFxuICBJQ29ubmVjdGFibGUsXG4gIEluc3RhbmNlVHlwZSxcbiAgSVNlY3VyaXR5R3JvdXAsXG4gIFBvcnQsXG4gIFN1Ym5ldFR5cGUsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0IHtcbiAgQ2x1c3RlcixcbiAgQ29udGFpbmVySW1hZ2UsXG4gIEVjMlRhc2tEZWZpbml0aW9uLFxuICBMb2dEcml2ZXIsXG4gIFBsYWNlbWVudENvbnN0cmFpbnQsXG4gIFVsaW1pdE5hbWUsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1lY3MnO1xuaW1wb3J0IHtcbiAgQXBwbGljYXRpb25Mb2FkQmFsYW5jZWRFYzJTZXJ2aWNlLFxufSBmcm9tICdAYXdzLWNkay9hd3MtZWNzLXBhdHRlcm5zJztcbmltcG9ydCB7XG4gIEFwcGxpY2F0aW9uTGlzdGVuZXIsXG4gIEFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyLFxuICBBcHBsaWNhdGlvblByb3RvY29sLFxuICBBcHBsaWNhdGlvblRhcmdldEdyb3VwLFxuICBDZm5UYXJnZXRHcm91cCxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWVsYXN0aWNsb2FkYmFsYW5jaW5ndjInO1xuaW1wb3J0IHtcbiAgSUdyYW50YWJsZSxcbiAgSVByaW5jaXBhbCxcbiAgTWFuYWdlZFBvbGljeSxcbiAgUG9saWN5U3RhdGVtZW50LFxuICBTZXJ2aWNlUHJpbmNpcGFsLFxufSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCB7XG4gIElMb2dHcm91cCxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWxvZ3MnO1xuaW1wb3J0IHtcbiAgSVNlY3JldCxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLXNlY3JldHNtYW5hZ2VyJztcbmltcG9ydCB7XG4gIENvbnN0cnVjdCxcbiAgSUNvbnN0cnVjdCxcbiAgU3RhY2ssXG59IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG5pbXBvcnQge1xuICBFQ1NDb25uZWN0T3B0aW9ucyxcbiAgSW5zdGFuY2VDb25uZWN0T3B0aW9ucyxcbiAgSVJlcG9zaXRvcnksXG4gIElWZXJzaW9uLFxuICBSZW5kZXJRdWV1ZVByb3BzLFxuICBSZW5kZXJRdWV1ZVNpemVDb25zdHJhaW50cyxcbiAgVmVyc2lvblF1ZXJ5LFxufSBmcm9tICcuJztcblxuaW1wb3J0IHtcbiAgQ29ubmVjdGFibGVBcHBsaWNhdGlvbkVuZHBvaW50LFxuICBJbXBvcnRlZEFjbUNlcnRpZmljYXRlLFxuICBMb2dHcm91cEZhY3RvcnksXG4gIFNjcmlwdEFzc2V0LFxuICBYNTA5Q2VydGlmaWNhdGVQZW0sXG4gIFg1MDlDZXJ0aWZpY2F0ZVBrY3MxMixcbn0gZnJvbSAnLi4vLi4vY29yZSc7XG5pbXBvcnQge1xuICB0YWdDb25zdHJ1Y3QsXG59IGZyb20gJy4uLy4uL2NvcmUvbGliL3J1bnRpbWUtaW5mbyc7XG5pbXBvcnQge1xuICBSZW5kZXJRdWV1ZUNvbm5lY3Rpb24sXG59IGZyb20gJy4vcnEtY29ubmVjdGlvbic7XG5pbXBvcnQgeyBWZXJzaW9uIH0gZnJvbSAnLi92ZXJzaW9uJztcbmltcG9ydCB7XG4gIFdhaXRGb3JTdGFibGVTZXJ2aWNlLFxufSBmcm9tICcuL3dhaXQtZm9yLXN0YWJsZS1zZXJ2aWNlJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSVJlbmRlclF1ZXVlIGV4dGVuZHMgSUNvbnN0cnVjdCwgSUNvbm5lY3RhYmxlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGVuZHBvaW50OiBDb25uZWN0YWJsZUFwcGxpY2F0aW9uRW5kcG9pbnQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBjb25maWd1cmVDbGllbnRFQ1MocGFyYW1zOiBFQ1NDb25uZWN0T3B0aW9ucyk6IHsgW25hbWU6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgY29uZmlndXJlQ2xpZW50SW5zdGFuY2UocGFyYW1zOiBJbnN0YW5jZUNvbm5lY3RPcHRpb25zKTogdm9pZDtcbn1cblxuLyoqXG4gKiBCYXNlIGNsYXNzIGZvciBSZW5kZXIgUXVldWUgcHJvdmlkZXJzXG4gKi9cbmFic3RyYWN0IGNsYXNzIFJlbmRlclF1ZXVlQmFzZSBleHRlbmRzIENvbnN0cnVjdCBpbXBsZW1lbnRzIElSZW5kZXJRdWV1ZSB7XG4gIC8qKlxuICAgKiBUaGUgZW5kcG9pbnQgdGhhdCBEZWFkbGluZSBjbGllbnRzIGNhbiB1c2UgdG8gY29ubmVjdCB0byB0aGUgUmVuZGVyIFF1ZXVlXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgZW5kcG9pbnQ6IENvbm5lY3RhYmxlQXBwbGljYXRpb25FbmRwb2ludDtcblxuICAvKipcbiAgICogQWxsb3dzIHNwZWNpZnlpbmcgc2VjdXJpdHkgZ3JvdXAgY29ubmVjdGlvbnMgZm9yIHRoZSBSZW5kZXIgUXVldWUuXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgY29ubmVjdGlvbnM6IENvbm5lY3Rpb25zO1xuXG4gIC8qKlxuICAgKiBDb25maWd1cmVzIGFuIEVDUyBjbHVzdGVyIHRvIGJlIGFibGUgdG8gY29ubmVjdCB0byBhIFJlbmRlclF1ZXVlXG4gICAqIEByZXR1cm5zIEFuIGVudmlyb25tZW50IG1hcHBpbmcgdGhhdCBpcyB1c2VkIHRvIGNvbmZpZ3VyZSB0aGUgRG9ja2VyIEltYWdlc1xuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IGNvbmZpZ3VyZUNsaWVudEVDUyhwYXJhbXM6IEVDU0Nvbm5lY3RPcHRpb25zKTogeyBbbmFtZTogc3RyaW5nXTogc3RyaW5nIH07XG5cbiAgLyoqXG4gICAqIENvbmZpZ3VyZSBhbiBJbnN0YW5jZS9BdXRvc2NhbGluZyBncm91cCB0byBjb25uZWN0IHRvIGEgUmVuZGVyUXVldWVcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCBjb25maWd1cmVDbGllbnRJbnN0YW5jZShwYXJhbXM6IEluc3RhbmNlQ29ubmVjdE9wdGlvbnMpOiB2b2lkO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIFJlbmRlclF1ZXVlIGV4dGVuZHMgUmVuZGVyUXVldWVCYXNlIGltcGxlbWVudHMgSUdyYW50YWJsZSB7XG4gIC8qKlxuICAgKiBDb250YWluZXIgbGlzdGVuaW5nIHBvcnRzIGZvciBlYWNoIHByb3RvY29sLlxuICAgKi9cbiAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgUkNTX1BST1RPX1BPUlRTID0ge1xuICAgIFtBcHBsaWNhdGlvblByb3RvY29sLkhUVFBdOiA4MDgwLFxuICAgIFtBcHBsaWNhdGlvblByb3RvY29sLkhUVFBTXTogNDQzMyxcbiAgfTtcblxuICAvKipcbiAgKiBUaGUgbWluaW11bSBEZWFkbGluZSB2ZXJzaW9uIHJlcXVpcmVkIGZvciB0aGUgUmVtb3RlIENvbm5lY3Rpb24gU2VydmVyIHRvIHN1cHBvcnQgbG9hZC1iYWxhbmNpbmdcbiAgKi9cbiAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgTUlOSU1VTV9MT0FEX0JBTEFOQ0lOR19WRVJTSU9OID0gbmV3IFZlcnNpb24oWzEwLCAxLCAxMCwgMF0pO1xuXG4gIC8qKlxuICAgKiBSZWd1bGFyIGV4cHJlc3Npb24gdGhhdCB2YWxpZGF0ZXMgYSBob3N0bmFtZSAocG9ydGlvbiBpbiBmcm9udCBvZiB0aGUgc3ViZG9tYWluKS5cbiAgICovXG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IFJFX1ZBTElEX0hPU1ROQU1FID0gL15bYS16XSg/OlthLXowLTktXXswLDYxfVthLXowLTldKT8kL2k7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsOiBJUHJpbmNpcGFsO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgY2x1c3RlcjogQ2x1c3RlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgY29ubmVjdGlvbnM6IENvbm5lY3Rpb25zO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBlbmRwb2ludDogQ29ubmVjdGFibGVBcHBsaWNhdGlvbkVuZHBvaW50O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgbG9hZEJhbGFuY2VyOiBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBhc2c6IEF1dG9TY2FsaW5nR3JvdXA7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgdmVyc2lvbjogSVZlcnNpb247XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGNlcnRDaGFpbj86IElTZWNyZXQ7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgU0VQIHBvbGljaWVzIGhhdmUgYmVlbiBhZGRlZFxuICAgKi9cbiAgcHJpdmF0ZSBoYXZlQWRkZWRTRVBQb2xpY2llczogYm9vbGVhbiA9IGZhbHNlO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIFJlc291cmNlIFRyYWNrZXIgcG9saWNpZXMgaGF2ZSBiZWVuIGFkZGVkXG4gICAqL1xuICBwcml2YXRlIGhhdmVBZGRlZFJlc291cmNlVHJhY2tlclBvbGljaWVzOiBib29sZWFuID0gZmFsc2U7XG5cbiAgLyoqXG4gICAqIFRoZSBsb2cgZ3JvdXAgd2hlcmUgdGhlIFJDUyBjb250YWluZXIgd2lsbCBsb2cgdG9cbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgbG9nR3JvdXA6IElMb2dHcm91cDtcblxuICAvKipcbiAgICogSW5zdGFuY2Ugb2YgdGhlIEFwcGxpY2F0aW9uIExvYWQgQmFsYW5jZWQgRUMyIHNlcnZpY2UgcGF0dGVybi5cbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgcGF0dGVybjogQXBwbGljYXRpb25Mb2FkQmFsYW5jZWRFYzJTZXJ2aWNlO1xuXG4gIC8qKlxuICAgKiBUaGUgY2VydGlmaWNhdGUgdXNlZCBieSB0aGUgQUxCIGZvciBleHRlcm5hbCBUcmFmZmljXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IGNsaWVudENlcnQ/OiBJQ2VydGlmaWNhdGU7XG5cbiAgLyoqXG4gICAqIFRoZSBjb25uZWN0aW9uIG9iamVjdCB0aGF0IGNvbnRhaW5zIHRoZSBsb2dpYyBmb3IgaG93IGNsaWVudHMgY2FuIGNvbm5lY3QgdG8gdGhlIFJlbmRlciBRdWV1ZS5cbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgcnFDb25uZWN0aW9uOiBSZW5kZXJRdWV1ZUNvbm5lY3Rpb247XG5cbiAgLyoqXG4gICAqIENvbnN0cmFpbnRzIG9uIHRoZSBudW1iZXIgb2YgRGVhZGxpbmUgUkNTIHByb2Nlc3NlcyB0aGF0IGNhbiBiZSBydW4gYXMgcGFydCBvZiB0aGlzXG4gICAqIFJlbmRlclF1ZXVlLlxuICAgKi9cbiAgcHJpdmF0ZSByZWFkb25seSByZW5kZXJRdWV1ZVNpemU6IFJlbmRlclF1ZXVlU2l6ZUNvbnN0cmFpbnRzO1xuXG4gIC8qKlxuICAgKiBUaGUgbGlzdGVuZXIgb24gdGhlIEFMQiB0aGF0IGlzIHJlZGlyZWN0aW5nIHRyYWZmaWMgdG8gdGhlIFJDUy5cbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgbGlzdGVuZXI6IEFwcGxpY2F0aW9uTGlzdGVuZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBFQ1MgdGFzayBmb3IgdGhlIFJDUy5cbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgdGFza0RlZmluaXRpb246IEVjMlRhc2tEZWZpbml0aW9uO1xuXG4gIC8qKlxuICAgKiBEZXBlbmQgb24gdGhpcyB0byBlbnN1cmUgdGhhdCBFQ1MgU2VydmljZSBpcyBzdGFibGUuXG4gICAqL1xuICBwcml2YXRlIGVjc1NlcnZpY2VTdGFiaWxpemVkOiBXYWl0Rm9yU3RhYmxlU2VydmljZTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUmVuZGVyUXVldWVQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLnJlbmRlclF1ZXVlU2l6ZSA9IHByb3BzPy5yZW5kZXJRdWV1ZVNpemUgPz8ge21pbjogMSwgbWF4OiAxfTtcblxuICAgIGlmIChwcm9wcy52ZXJzaW9uLmlzTGVzc1RoYW4oUmVuZGVyUXVldWUuTUlOSU1VTV9MT0FEX0JBTEFOQ0lOR19WRVJTSU9OKSkge1xuICAgICAgLy8gRGVhZGxpbmUgdmVyc2lvbnMgZWFybGllciB0aGFuIDEwLjEuMTAgZG8gbm90IHN1cHBvcnQgaG9yaXpvbnRhbCBzY2FsaW5nIGJlaGluZCBhIGxvYWQtYmFsYW5jZXIsIHNvIHdlIGxpbWl0IHRvIGF0IG1vc3Qgb25lIGluc3RhbmNlXG4gICAgICBpZiAoKHRoaXMucmVuZGVyUXVldWVTaXplLm1pbiA/PyAwKSA+IDEpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGByZW5kZXJRdWV1ZVNpemUubWluIGZvciBEZWFkbGluZSB2ZXJzaW9uIGxlc3MgdGhhbiAke1JlbmRlclF1ZXVlLk1JTklNVU1fTE9BRF9CQUxBTkNJTkdfVkVSU0lPTi50b1N0cmluZygpfSBjYW5ub3QgYmUgZ3JlYXRlciB0aGFuIDEgLSBnb3QgJHt0aGlzLnJlbmRlclF1ZXVlU2l6ZS5taW59YCk7XG4gICAgICB9XG4gICAgICBpZiAoKHRoaXMucmVuZGVyUXVldWVTaXplLmRlc2lyZWQgPz8gMCkgPiAxKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgcmVuZGVyUXVldWVTaXplLmRlc2lyZWQgZm9yIERlYWRsaW5lIHZlcnNpb24gbGVzcyB0aGFuICR7UmVuZGVyUXVldWUuTUlOSU1VTV9MT0FEX0JBTEFOQ0lOR19WRVJTSU9OLnRvU3RyaW5nKCl9IGNhbm5vdCBiZSBncmVhdGVyIHRoYW4gMSAtIGdvdCAke3RoaXMucmVuZGVyUXVldWVTaXplLmRlc2lyZWR9YCk7XG4gICAgICB9XG4gICAgICBpZiAoKHRoaXMucmVuZGVyUXVldWVTaXplLm1heCA/PyAwKSA+IDEpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGByZW5kZXJRdWV1ZVNpemUubWF4IGZvciBEZWFkbGluZSB2ZXJzaW9uIGxlc3MgdGhhbiAke1JlbmRlclF1ZXVlLk1JTklNVU1fTE9BRF9CQUxBTkNJTkdfVkVSU0lPTi50b1N0cmluZygpfSBjYW5ub3QgYmUgZ3JlYXRlciB0aGFuIDEgLSBnb3QgJHt0aGlzLnJlbmRlclF1ZXVlU2l6ZS5tYXh9YCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgdGhpcy52ZXJzaW9uID0gcHJvcHM/LnZlcnNpb247XG5cbiAgICBsZXQgZXh0ZXJuYWxQcm90b2NvbDogQXBwbGljYXRpb25Qcm90b2NvbDtcbiAgICBpZiAoIHByb3BzLnRyYWZmaWNFbmNyeXB0aW9uPy5leHRlcm5hbFRMUyApIHtcbiAgICAgIGV4dGVybmFsUHJvdG9jb2wgPSBBcHBsaWNhdGlvblByb3RvY29sLkhUVFBTO1xuXG4gICAgICBpZiAoIChwcm9wcy50cmFmZmljRW5jcnlwdGlvbi5leHRlcm5hbFRMUy5hY21DZXJ0aWZpY2F0ZSA9PT0gdW5kZWZpbmVkICkgPT09XG4gICAgICAocHJvcHMudHJhZmZpY0VuY3J5cHRpb24uZXh0ZXJuYWxUTFMucmZka0NlcnRpZmljYXRlID09PSB1bmRlZmluZWQpICkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0V4YWN0bHkgb25lIG9mIGV4dGVybmFsVExTLmFjbUNlcnRpZmljYXRlIGFuZCBleHRlcm5hbFRMUy5yZmRrQ2VydGlmaWNhdGUgbXVzdCBiZSBwcm92aWRlZCB3aGVuIHVzaW5nIGV4dGVybmFsVExTLicpO1xuICAgICAgfSBlbHNlIGlmIChwcm9wcy50cmFmZmljRW5jcnlwdGlvbi5leHRlcm5hbFRMUy5yZmRrQ2VydGlmaWNhdGUgKSB7XG4gICAgICAgIGlmIChwcm9wcy50cmFmZmljRW5jcnlwdGlvbi5leHRlcm5hbFRMUy5yZmRrQ2VydGlmaWNhdGUuY2VydENoYWluID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Byb3ZpZGVkIHJmZGtDZXJ0aWZpY2F0ZSBkb2VzIG5vdCBjb250YWluIGEgY2VydGlmaWNhdGUgY2hhaW4uJyk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5jbGllbnRDZXJ0ID0gbmV3IEltcG9ydGVkQWNtQ2VydGlmaWNhdGUodGhpcywgJ0FjbUNlcnQnLCBwcm9wcy50cmFmZmljRW5jcnlwdGlvbi5leHRlcm5hbFRMUy5yZmRrQ2VydGlmaWNhdGUgKTtcbiAgICAgICAgdGhpcy5jZXJ0Q2hhaW4gPSBwcm9wcy50cmFmZmljRW5jcnlwdGlvbi5leHRlcm5hbFRMUy5yZmRrQ2VydGlmaWNhdGUuY2VydENoYWluO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgaWYgKHByb3BzLnRyYWZmaWNFbmNyeXB0aW9uLmV4dGVybmFsVExTLmFjbUNlcnRpZmljYXRlQ2hhaW4gPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcignZXh0ZXJuYWxUTFMuYWNtQ2VydGlmaWNhdGVDaGFpbiBtdXN0IGJlIHByb3ZpZGVkIHdoZW4gdXNpbmcgZXh0ZXJuYWxUTFMuYWNtQ2VydGlmaWNhdGUuJyk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5jbGllbnRDZXJ0ID0gcHJvcHMudHJhZmZpY0VuY3J5cHRpb24uZXh0ZXJuYWxUTFMuYWNtQ2VydGlmaWNhdGU7XG4gICAgICAgIHRoaXMuY2VydENoYWluID0gcHJvcHMudHJhZmZpY0VuY3J5cHRpb24uZXh0ZXJuYWxUTFMuYWNtQ2VydGlmaWNhdGVDaGFpbjtcbiAgICAgIH1cbiAgICB9IGVsc2Uge1xuICAgICAgZXh0ZXJuYWxQcm90b2NvbCA9IEFwcGxpY2F0aW9uUHJvdG9jb2wuSFRUUDtcbiAgICB9XG5cbiAgICB0aGlzLnZlcnNpb24gPSBwcm9wcy52ZXJzaW9uO1xuXG4gICAgY29uc3QgaW50ZXJuYWxQcm90b2NvbCA9IHByb3BzLnRyYWZmaWNFbmNyeXB0aW9uPy5pbnRlcm5hbFByb3RvY29sID8/IEFwcGxpY2F0aW9uUHJvdG9jb2wuSFRUUFM7XG5cbiAgICBpZiAoZXh0ZXJuYWxQcm90b2NvbCA9PT0gQXBwbGljYXRpb25Qcm90b2NvbC5IVFRQUyAmJiAhcHJvcHMuaG9zdG5hbWUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQSBob3N0bmFtZSBtdXN0IGJlIHByb3ZpZGVkIHdoZW4gdGhlIGV4dGVybmFsIHByb3RvY29sIGlzIEhUVFBTJyk7XG4gICAgfVxuXG4gICAgdGhpcy5jbHVzdGVyID0gbmV3IENsdXN0ZXIodGhpcywgJ0NsdXN0ZXInLCB7XG4gICAgICB2cGM6IHByb3BzLnZwYyxcbiAgICB9KTtcblxuICAgIGNvbnN0IG1pbkNhcGFjaXR5ID0gcHJvcHMucmVuZGVyUXVldWVTaXplPy5taW4gPz8gMTtcbiAgICBpZiAobWluQ2FwYWNpdHkgPCAxKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYHJlbmRlclF1ZXVlU2l6ZS5taW4gY2FwYWNpdHkgbXVzdCBiZSBhdCBsZWFzdCAxOiBnb3QgJHttaW5DYXBhY2l0eX1gKTtcbiAgICB9XG4gICAgY29uc3QgbWF4Q2FwYWNpdHkgPSB0aGlzLnJlbmRlclF1ZXVlU2l6ZS5tYXggPz8gdGhpcy5yZW5kZXJRdWV1ZVNpemU/LmRlc2lyZWQ7XG4gICAgaWYgKHRoaXMucmVuZGVyUXVldWVTaXplPy5kZXNpcmVkICYmIG1heENhcGFjaXR5ICYmIHRoaXMucmVuZGVyUXVldWVTaXplPy5kZXNpcmVkID4gbWF4Q2FwYWNpdHkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgcmVuZGVyUXVldWVTaXplLmRlc2lyZWQgY2FwYWNpdHkgY2Fubm90IGJlIG1vcmUgdGhhbiAke21heENhcGFjaXR5fTogZ290ICR7dGhpcy5yZW5kZXJRdWV1ZVNpemUuZGVzaXJlZH1gKTtcbiAgICB9XG4gICAgdGhpcy5hc2cgPSB0aGlzLmNsdXN0ZXIuYWRkQ2FwYWNpdHkoJ1JDUyBDYXBhY2l0eScsIHtcbiAgICAgIHZwY1N1Ym5ldHM6IHByb3BzLnZwY1N1Ym5ldHMgPz8geyBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEUgfSxcbiAgICAgIGluc3RhbmNlVHlwZTogcHJvcHMuaW5zdGFuY2VUeXBlID8/IG5ldyBJbnN0YW5jZVR5cGUoJ2M1LmxhcmdlJyksXG4gICAgICBtaW5DYXBhY2l0eSxcbiAgICAgIGRlc2lyZWRDYXBhY2l0eTogdGhpcy5yZW5kZXJRdWV1ZVNpemU/LmRlc2lyZWQsXG4gICAgICBtYXhDYXBhY2l0eSxcbiAgICAgIGJsb2NrRGV2aWNlczogW3tcbiAgICAgICAgZGV2aWNlTmFtZTogJy9kZXYveHZkYScsXG4gICAgICAgIC8vIFNlZTogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvZWNzLWFtaS1zdG9yYWdlLWNvbmZpZy5odG1sXG4gICAgICAgIC8vIFdlIHdhbnQgdGhlIHZvbHVtZSB0byBiZSBlbmNyeXB0ZWQuIFRoZSBkZWZhdWx0IEFNSSBzaXplIGlzIDMwLUdpQi5cbiAgICAgICAgdm9sdW1lOiBCbG9ja0RldmljZVZvbHVtZS5lYnMoMzAsIHsgZW5jcnlwdGVkOiB0cnVlIH0pLFxuICAgICAgfV0sXG4gICAgICB1cGRhdGVUeXBlOiB1bmRlZmluZWQsIC8vIFdvcmthcm91bmQgLS0gU2VlOiBodHRwczovL2dpdGh1Yi5jb20vYXdzL2F3cy1jZGsvaXNzdWVzLzExNTgxXG4gICAgICB1cGRhdGVQb2xpY3k6IFVwZGF0ZVBvbGljeS5yb2xsaW5nVXBkYXRlKCksXG4gICAgICAvLyBhZGRDYXBhY2l0eSBkb2Vzbid0IHNwZWNpZmljYWxseSB0YWtlIGEgc2VjdXJpdHlHcm91cCwgYnV0IGl0IHBhc3NlcyBvbiBpdHMgcHJvcGVydGllcyB0byB0aGUgQVNHIGl0IGNyZWF0ZXMsXG4gICAgICAvLyBzbyB0aGlzIHNlY3VyaXR5IGdyb3VwIHdpbGwgZ2V0IGFwcGxpZWQgdGhlcmVcbiAgICAgIC8vIEB0cy1pZ25vcmVcbiAgICAgIHNlY3VyaXR5R3JvdXA6IHByb3BzLnNlY3VyaXR5R3JvdXBzPy5iYWNrZW5kLFxuICAgIH0pO1xuXG4gICAgLyoqXG4gICAgICogVGhlIEVDUy1vcHRpbWl6ZWQgQU1JIHRoYXQgaXMgZGVmYXVsdGVkIHRvIHdoZW4gYWRkaW5nIGNhcGFjaXR5IHRvIGEgY2x1c3RlciBkb2VzIG5vdCBpbmNsdWRlIHRoZSBhd3NjbGkgb3IgdW56aXBcbiAgICAgKiBwYWNrYWdlcyBhcyBpcyB0aGUgY2FzZSB3aXRoIHRoZSBzdGFuZGFyZCBBbWF6b24gTGludXggQU1JLiBUaGVzZSBhcmUgcmVxdWlyZWQgYnkgUkZESyBzY3JpcHRzIHRvIGNvbmZpZ3VyZSB0aGVcbiAgICAgKiBkaXJlY3QgY29ubmVjdGlvbiBvbiB0aGUgaG9zdCBjb250YWluZXIgaW5zdGFuY2VzLlxuICAgICAqL1xuICAgIHRoaXMuYXNnLnVzZXJEYXRhLmFkZENvbW1hbmRzKFxuICAgICAgJ3l1bSBpbnN0YWxsIC15cSBhd3NjbGkgdW56aXAnLFxuICAgICk7XG4gICAgaWYgKHByb3BzLmVuYWJsZUxvY2FsRmlsZUNhY2hpbmcgPz8gZmFsc2UpIHtcbiAgICAgIC8vIEhhcyB0byBiZSBkb25lIGJlZm9yZSBhbnkgZmlsZXN5c3RlbXMgbW91bnQuXG4gICAgICB0aGlzLmVuYWJsZUZpbGVjYWNoaW5nKHRoaXMuYXNnKTtcbiAgICB9XG5cbiAgICBjb25zdCBleHRlcm5hbFBvcnROdW1iZXIgPSBSZW5kZXJRdWV1ZS5SQ1NfUFJPVE9fUE9SVFNbZXh0ZXJuYWxQcm90b2NvbF07XG4gICAgY29uc3QgaW50ZXJuYWxQb3J0TnVtYmVyID0gUmVuZGVyUXVldWUuUkNTX1BST1RPX1BPUlRTW2ludGVybmFsUHJvdG9jb2xdO1xuXG4gICAgdGhpcy5sb2dHcm91cCA9IExvZ0dyb3VwRmFjdG9yeS5jcmVhdGVPckZldGNoKHRoaXMsICdMb2dHcm91cFdyYXBwZXInLCBpZCwge1xuICAgICAgbG9nR3JvdXBQcmVmaXg6ICcvcmVuZGVyZmFybS8nLFxuICAgICAgLi4ucHJvcHMubG9nR3JvdXBQcm9wcyxcbiAgICB9KTtcbiAgICB0aGlzLmxvZ0dyb3VwLmdyYW50V3JpdGUodGhpcy5hc2cpO1xuXG4gICAgY29uc3QgdGFza0RlZmluaXRpb24gPSB0aGlzLmNyZWF0ZVRhc2tEZWZpbml0aW9uKHtcbiAgICAgIGltYWdlOiBwcm9wcy5pbWFnZXMucmVtb3RlQ29ubmVjdGlvblNlcnZlcixcbiAgICAgIHBvcnROdW1iZXI6IGludGVybmFsUG9ydE51bWJlcixcbiAgICAgIHByb3RvY29sOiBpbnRlcm5hbFByb3RvY29sLFxuICAgICAgcmVwb3NpdG9yeTogcHJvcHMucmVwb3NpdG9yeSxcbiAgICB9KTtcbiAgICB0aGlzLnRhc2tEZWZpbml0aW9uID0gdGFza0RlZmluaXRpb247XG5cbiAgICAvLyBUaGUgZnVsbHktcXVhbGlmaWVkIGRvbWFpbiBuYW1lIHRvIHVzZSBmb3IgdGhlIEFMQlxuICAgIGxldCBsb2FkQmFsYW5jZXJGUUROOiBzdHJpbmcgfCB1bmRlZmluZWQ7XG4gICAgaWYgKHByb3BzLmhvc3RuYW1lKSB7XG4gICAgICBjb25zdCBsYWJlbCA9IHByb3BzLmhvc3RuYW1lLmhvc3RuYW1lID8/ICdyZW5kZXJxdWV1ZSc7XG4gICAgICBpZiAocHJvcHMuaG9zdG5hbWUuaG9zdG5hbWUgJiYgIVJlbmRlclF1ZXVlLlJFX1ZBTElEX0hPU1ROQU1FLnRlc3QobGFiZWwpKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBSZW5kZXJRdWV1ZSBob3N0bmFtZTogJHtsYWJlbH1gKTtcbiAgICAgIH1cbiAgICAgIGxvYWRCYWxhbmNlckZRRE4gPSBgJHtsYWJlbH0uJHtwcm9wcy5ob3N0bmFtZS56b25lLnpvbmVOYW1lfWA7XG4gICAgfVxuXG4gICAgY29uc3QgbG9hZEJhbGFuY2VyID0gbmV3IEFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyKHRoaXMsICdMQicsIHtcbiAgICAgIHZwYzogdGhpcy5jbHVzdGVyLnZwYyxcbiAgICAgIHZwY1N1Ym5ldHM6IHByb3BzLnZwY1N1Ym5ldHNBbGIgPz8geyBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEUsIG9uZVBlckF6OiB0cnVlIH0sXG4gICAgICBpbnRlcm5ldEZhY2luZzogZmFsc2UsXG4gICAgICBkZWxldGlvblByb3RlY3Rpb246IHByb3BzLmRlbGV0aW9uUHJvdGVjdGlvbiA/PyB0cnVlLFxuICAgICAgc2VjdXJpdHlHcm91cDogcHJvcHMuc2VjdXJpdHlHcm91cHM/LmZyb250ZW5kLFxuICAgIH0pO1xuXG4gICAgdGhpcy5wYXR0ZXJuID0gbmV3IEFwcGxpY2F0aW9uTG9hZEJhbGFuY2VkRWMyU2VydmljZSh0aGlzLCAnQWxiRWMyU2VydmljZVBhdHRlcm4nLCB7XG4gICAgICBjZXJ0aWZpY2F0ZTogdGhpcy5jbGllbnRDZXJ0LFxuICAgICAgY2x1c3RlcjogdGhpcy5jbHVzdGVyLFxuICAgICAgZGVzaXJlZENvdW50OiB0aGlzLnJlbmRlclF1ZXVlU2l6ZT8uZGVzaXJlZCxcbiAgICAgIGRvbWFpblpvbmU6IHByb3BzLmhvc3RuYW1lPy56b25lLFxuICAgICAgZG9tYWluTmFtZTogbG9hZEJhbGFuY2VyRlFETixcbiAgICAgIGxpc3RlbmVyUG9ydDogZXh0ZXJuYWxQb3J0TnVtYmVyLFxuICAgICAgbG9hZEJhbGFuY2VyLFxuICAgICAgcHJvdG9jb2w6IGV4dGVybmFsUHJvdG9jb2wsXG4gICAgICB0YXNrRGVmaW5pdGlvbixcbiAgICAgIC8vIFRoaXMgaXMgcmVxdWlyZWQgdG8gcmlnaHQtc2l6ZSBvdXIgaG9zdCBjYXBhY2l0eSBhbmQgbm90IGhhdmUgdGhlIEVDUyBzZXJ2aWNlIGJsb2NrIG9uIHVwZGF0ZXMuIFdlIHNldCBhIG1lbW9yeVxuICAgICAgLy8gcmVzZXJ2YXRpb24sIGJ1dCBubyBtZW1vcnkgbGltaXQgb24gdGhlIGNvbnRhaW5lci4gVGhpcyBhbGxvd3MgdGhlIGNvbnRhaW5lcidzIG1lbW9yeSB1c2FnZSB0byBncm93IHVuYm91bmRlZC5cbiAgICAgIC8vIFdlIHdhbnQgMToxIGNvbnRhaW5lciB0byBjb250YWluZXIgaW5zdGFuY2VzIHRvIG5vdCBvdmVyLXNwZW5kLCBidXQgdGhpcyBjb21lcyBhdCB0aGUgcHJpY2Ugb2YgZG93bi10aW1lIGR1cmluZ1xuICAgICAgLy8gY2xvdWRmb3JtYXRpb24gdXBkYXRlcy5cbiAgICAgIG1pbkhlYWx0aHlQZXJjZW50OiAwLFxuICAgICAgbWF4SGVhbHRoeVBlcmNlbnQ6IDEwMCxcbiAgICAgIC8vIFRoaXMgaXMgcmVxdWlyZWQgdG8gZW5zdXJlIHRoYXQgdGhlIEFMQiBsaXN0ZW5lcidzIHNlY3VyaXR5IGdyb3VwIGRvZXMgbm90IGFsbG93IGFueSBpbmdyZXNzIGJ5IGRlZmF1bHQuXG4gICAgICBvcGVuTGlzdGVuZXI6IGZhbHNlLFxuICAgIH0pO1xuXG4gICAgLy8gQW4gZXhwbGljaXQgZGVwZW5kZW5jeSBpcyByZXF1aXJlZCBmcm9tIHRoZSBTZXJ2aWNlIHRvIHRoZSBDbGllbnQgY2VydGlmaWNhdGVcbiAgICAvLyBPdGhlcndpc2UgY2xvdWQgZm9ybWF0aW9uIHdpbGwgdHJ5IHRvIHJlbW92ZSB0aGUgY2VydCBiZWZvcmUgdGhlIEFMQiB1c2luZyBpdCBpcyBkaXNwb3NlZC5cbiAgICBpZiAodGhpcy5jbGllbnRDZXJ0KSB7XG4gICAgICB0aGlzLnBhdHRlcm4ubm9kZS5hZGREZXBlbmRlbmN5KHRoaXMuY2xpZW50Q2VydCk7XG4gICAgfVxuXG4gICAgLy8gQW4gZXhwbGljaXQgZGVwZW5kZW5jeSBpcyByZXF1aXJlZCBmcm9tIHRoZSBzZXJ2aWNlIHRvIHRoZSBBU0cgcHJvdmlkaW5nIGl0cyBjYXBhY2l0eS5cbiAgICAvLyBTZWU6IGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1hdHRyaWJ1dGUtZGVwZW5kc29uLmh0bWxcbiAgICB0aGlzLnBhdHRlcm4uc2VydmljZS5ub2RlLmFkZERlcGVuZGVuY3kodGhpcy5hc2cpO1xuXG4gICAgdGhpcy5sb2FkQmFsYW5jZXIgPSB0aGlzLnBhdHRlcm4ubG9hZEJhbGFuY2VyO1xuICAgIC8vIEVuYWJsaW5nIGRyb3BwaW5nIG9mIGludmFsaWQgSFRUUCBoZWFkZXIgZmllbGRzIG9uIHRoZSBsb2FkIGJhbGFuY2VyIHRvIHByZXZlbnQgaHR0cCBzbXVnZ2xpbmcgYXR0YWNrcy5cbiAgICB0aGlzLmxvYWRCYWxhbmNlci5zZXRBdHRyaWJ1dGUoJ3JvdXRpbmcuaHR0cC5kcm9wX2ludmFsaWRfaGVhZGVyX2ZpZWxkcy5lbmFibGVkJywgJ3RydWUnKTtcblxuICAgIGlmIChwcm9wcy5hY2Nlc3NMb2dzKSB7XG4gICAgICBjb25zdCBhY2Nlc3NMb2dzQnVja2V0ID0gcHJvcHMuYWNjZXNzTG9ncy5kZXN0aW5hdGlvbkJ1Y2tldDtcblxuICAgICAgLy8gUG9saWNpZXMgYXJlIGFwcGxpZWQgYWNjb3JkaW5nIHRvXG4gICAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2xvYWRiYWxhbmNpbmcvbGF0ZXN0L2FwcGxpY2F0aW9uL2xvYWQtYmFsYW5jZXItYWNjZXNzLWxvZ3MuaHRtbFxuICAgICAgYWNjZXNzTG9nc0J1Y2tldC5hZGRUb1Jlc291cmNlUG9saWN5KCBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgYWN0aW9uczogWydzMzpQdXRPYmplY3QnXSxcbiAgICAgICAgcHJpbmNpcGFsczogW25ldyBTZXJ2aWNlUHJpbmNpcGFsKCdkZWxpdmVyeS5sb2dzLmFtYXpvbmF3cy5jb20nKV0sXG4gICAgICAgIHJlc291cmNlczogW2Ake2FjY2Vzc0xvZ3NCdWNrZXQuYnVja2V0QXJufS8qYF0sXG4gICAgICAgIGNvbmRpdGlvbnM6IHtcbiAgICAgICAgICBTdHJpbmdFcXVhbHM6IHtcbiAgICAgICAgICAgICdzMzp4LWFtei1hY2wnOiAnYnVja2V0LW93bmVyLWZ1bGwtY29udHJvbCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pKTtcbiAgICAgIGFjY2Vzc0xvZ3NCdWNrZXQuYWRkVG9SZXNvdXJjZVBvbGljeShuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgYWN0aW9uczogWyAnczM6R2V0QnVja2V0QWNsJyBdLFxuICAgICAgICBwcmluY2lwYWxzOiBbIG5ldyBTZXJ2aWNlUHJpbmNpcGFsKCdkZWxpdmVyeS5sb2dzLmFtYXpvbmF3cy5jb20nKV0sXG4gICAgICAgIHJlc291cmNlczogWyBhY2Nlc3NMb2dzQnVja2V0LmJ1Y2tldEFybiBdLFxuICAgICAgfSkpO1xuXG4gICAgICB0aGlzLmxvYWRCYWxhbmNlci5sb2dBY2Nlc3NMb2dzKFxuICAgICAgICBhY2Nlc3NMb2dzQnVja2V0LFxuICAgICAgICBwcm9wcy5hY2Nlc3NMb2dzLnByZWZpeCk7XG4gICAgfVxuXG4gICAgLy8gRW5zdXJlIHRhc2tzIGFyZSBydW4gb24gc2VwYXJhdGUgY29udGFpbmVyIGluc3RhbmNlc1xuICAgIHRoaXMucGF0dGVybi5zZXJ2aWNlLmFkZFBsYWNlbWVudENvbnN0cmFpbnRzKFBsYWNlbWVudENvbnN0cmFpbnQuZGlzdGluY3RJbnN0YW5jZXMoKSk7XG5cbiAgICAvKipcbiAgICAgKiBVc2VzIGFuIGVzY2FwZS1oYXRjaCB0byBzZXQgdGhlIHRhcmdldCBncm91cCBwcm90b2NvbCB0byBIVFRQUy4gV2UgY2Fubm90IGNvbmZpZ3VyZSBzZXJ2ZXIgY2VydGlmaWNhdGVcbiAgICAgKiB2YWxpZGF0aW9uLCBidXQgYXQgbGVhc3QgdHJhZmZpYyBpcyBlbmNyeXB0ZWQgYW5kIHRlcm1pbmF0ZWQgYXQgdGhlIGFwcGxpY2F0aW9uIGxheWVyLlxuICAgICAqL1xuICAgIGNvbnN0IGxpc3RlbmVyID0gdGhpcy5sb2FkQmFsYW5jZXIubm9kZS5maW5kQ2hpbGQoJ1B1YmxpY0xpc3RlbmVyJyk7XG4gICAgdGhpcy5saXN0ZW5lciA9IGxpc3RlbmVyIGFzIEFwcGxpY2F0aW9uTGlzdGVuZXI7XG4gICAgY29uc3QgdGFyZ2V0R3JvdXAgPSBsaXN0ZW5lci5ub2RlLmZpbmRDaGlsZCgnRUNTR3JvdXAnKSBhcyBBcHBsaWNhdGlvblRhcmdldEdyb3VwO1xuICAgIGNvbnN0IHRhcmdldEdyb3VwUmVzb3VyY2UgPSB0YXJnZXRHcm91cC5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5UYXJnZXRHcm91cDtcbiAgICB0YXJnZXRHcm91cFJlc291cmNlLnByb3RvY29sID0gQXBwbGljYXRpb25Qcm90b2NvbFtpbnRlcm5hbFByb3RvY29sXTtcbiAgICB0YXJnZXRHcm91cFJlc291cmNlLnBvcnQgPSBpbnRlcm5hbFBvcnROdW1iZXI7XG5cbiAgICB0aGlzLmdyYW50UHJpbmNpcGFsID0gdGFza0RlZmluaXRpb24udGFza1JvbGU7XG5cbiAgICB0aGlzLmNvbm5lY3Rpb25zID0gbmV3IENvbm5lY3Rpb25zKHtcbiAgICAgIGRlZmF1bHRQb3J0OiBQb3J0LnRjcChleHRlcm5hbFBvcnROdW1iZXIpLFxuICAgICAgc2VjdXJpdHlHcm91cHM6IHRoaXMucGF0dGVybi5sb2FkQmFsYW5jZXIuY29ubmVjdGlvbnMuc2VjdXJpdHlHcm91cHMsXG4gICAgfSk7XG5cbiAgICB0aGlzLmVuZHBvaW50ID0gbmV3IENvbm5lY3RhYmxlQXBwbGljYXRpb25FbmRwb2ludCh7XG4gICAgICBhZGRyZXNzOiBsb2FkQmFsYW5jZXJGUUROID8/IHRoaXMucGF0dGVybi5sb2FkQmFsYW5jZXIubG9hZEJhbGFuY2VyRG5zTmFtZSxcbiAgICAgIHBvcnQ6IGV4dGVybmFsUG9ydE51bWJlcixcbiAgICAgIGNvbm5lY3Rpb25zOiB0aGlzLmNvbm5lY3Rpb25zLFxuICAgICAgcHJvdG9jb2w6IGV4dGVybmFsUHJvdG9jb2wsXG4gICAgfSk7XG5cbiAgICBpZiAoIGV4dGVybmFsUHJvdG9jb2wgPT09IEFwcGxpY2F0aW9uUHJvdG9jb2wuSFRUUCApIHtcbiAgICAgIHRoaXMucnFDb25uZWN0aW9uID0gUmVuZGVyUXVldWVDb25uZWN0aW9uLmZvckh0dHAoe1xuICAgICAgICBlbmRwb2ludDogdGhpcy5lbmRwb2ludCxcbiAgICAgIH0pO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLnJxQ29ubmVjdGlvbiA9IFJlbmRlclF1ZXVlQ29ubmVjdGlvbi5mb3JIdHRwcyh7XG4gICAgICAgIGVuZHBvaW50OiB0aGlzLmVuZHBvaW50LFxuICAgICAgICBjYUNlcnQ6IHRoaXMuY2VydENoYWluISxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHRoaXMuZWNzU2VydmljZVN0YWJpbGl6ZWQgPSBuZXcgV2FpdEZvclN0YWJsZVNlcnZpY2UodGhpcywgJ1dhaXRGb3JTdGFibGVTZXJ2aWNlJywge1xuICAgICAgc2VydmljZTogdGhpcy5wYXR0ZXJuLnNlcnZpY2UsXG4gICAgfSk7XG5cbiAgICB0aGlzLm5vZGUuZGVmYXVsdENoaWxkID0gdGFza0RlZmluaXRpb247XG5cbiAgICAvLyBUYWcgZGVwbG95ZWQgcmVzb3VyY2VzIHdpdGggUkZESyBtZXRhLWRhdGFcbiAgICB0YWdDb25zdHJ1Y3QodGhpcyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgb25WYWxpZGF0ZSgpOiBzdHJpbmdbXSB7XG4gICAgY29uc3QgdmFsaWRhdGlvbkVycm9ycyA9IFtdO1xuXG4gICAgLy8gVXNpbmcgdGhlIG91dHB1dCBvZiBWZXJzaW9uUXVlcnkgYWNyb3NzIHN0YWNrcyBjYW4gY2F1c2UgaXNzdWVzLiBDbG91ZEZvcm1hdGlvbiBzdGFjayBvdXRwdXRzIGNhbm5vdCBjaGFuZ2UgaWZcbiAgICAvLyBhIHJlc291cmNlIGluIGFub3RoZXIgc3RhY2sgaXMgcmVmZXJlbmNpbmcgaXQuXG4gICAgaWYgKHRoaXMudmVyc2lvbiBpbnN0YW5jZW9mIFZlcnNpb25RdWVyeSkge1xuICAgICAgY29uc3QgdmVyc2lvblN0YWNrID0gU3RhY2sub2YodGhpcy52ZXJzaW9uKTtcbiAgICAgIGNvbnN0IHRoaXNTdGFjayA9IFN0YWNrLm9mKHRoaXMpO1xuICAgICAgaWYgKHZlcnNpb25TdGFjayAhPSB0aGlzU3RhY2spIHtcbiAgICAgICAgdmFsaWRhdGlvbkVycm9ycy5wdXNoKCdBIFZlcnNpb25RdWVyeSBjYW4gbm90IGJlIHN1cHBsaWVkIGZyb20gYSBkaWZmZXJlbnQgc3RhY2snKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gdmFsaWRhdGlvbkVycm9ycztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBjb25maWd1cmVDbGllbnRFQ1MocGFyYW06IEVDU0Nvbm5lY3RPcHRpb25zKTogeyBbbmFtZTogc3RyaW5nXTogc3RyaW5nIH0ge1xuICAgIHBhcmFtLmhvc3RzLmZvckVhY2goIGhvc3QgPT4gdGhpcy5hZGRDaGlsZERlcGVuZGVuY3koaG9zdCkgKTtcbiAgICByZXR1cm4gdGhpcy5ycUNvbm5lY3Rpb24uY29uZmlndXJlQ2xpZW50RUNTKHBhcmFtKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBjb25maWd1cmVDbGllbnRJbnN0YW5jZShwYXJhbTogSW5zdGFuY2VDb25uZWN0T3B0aW9ucyk6IHZvaWQge1xuICAgIHRoaXMuYWRkQ2hpbGREZXBlbmRlbmN5KHBhcmFtLmhvc3QpO1xuICAgIHRoaXMucnFDb25uZWN0aW9uLmNvbmZpZ3VyZUNsaWVudEluc3RhbmNlKHBhcmFtKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZFNFUFBvbGljaWVzKGluY2x1ZGVSZXNvdXJjZVRyYWNrZXI6IGJvb2xlYW4gPSB0cnVlKTogdm9pZCB7XG4gICAgaWYgKCF0aGlzLmhhdmVBZGRlZFNFUFBvbGljaWVzKSB7XG4gICAgICBjb25zdCBzZXBQb2xpY3kgPSBNYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnQVdTVGhpbmtib3hEZWFkbGluZVNwb3RFdmVudFBsdWdpbkFkbWluUG9saWN5Jyk7XG4gICAgICB0aGlzLnRhc2tEZWZpbml0aW9uLnRhc2tSb2xlLmFkZE1hbmFnZWRQb2xpY3koc2VwUG9saWN5KTtcbiAgICAgIHRoaXMuaGF2ZUFkZGVkU0VQUG9saWNpZXMgPSB0cnVlO1xuICAgIH1cblxuICAgIGlmICghdGhpcy5oYXZlQWRkZWRSZXNvdXJjZVRyYWNrZXJQb2xpY2llcykge1xuICAgICAgaWYgKGluY2x1ZGVSZXNvdXJjZVRyYWNrZXIpIHtcbiAgICAgICAgY29uc3QgcnRQb2xpY3kgPSBNYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnQVdTVGhpbmtib3hEZWFkbGluZVJlc291cmNlVHJhY2tlckFkbWluUG9saWN5Jyk7XG4gICAgICAgIHRoaXMudGFza0RlZmluaXRpb24udGFza1JvbGUuYWRkTWFuYWdlZFBvbGljeShydFBvbGljeSk7XG4gICAgICAgIHRoaXMuaGF2ZUFkZGVkUmVzb3VyY2VUcmFja2VyUG9saWNpZXMgPSB0cnVlO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZENoaWxkRGVwZW5kZW5jeShjaGlsZDogSUNvbnN0cnVjdCk6IHZvaWQge1xuICAgIC8vIE5hcnJvd2x5IGRlZmluZSB0aGUgZGVwZW5kZW5jaWVzIHRvIHJlZHVjZSB0aGUgcHJvYmFiaWxpdHkgb2YgY3ljbGVzXG4gICAgLy8gZXg6IGN5Y2xlcyB0aGF0IGludm9sdmUgdGhlIHNlY3VyaXR5IGdyb3VwIG9mIHRoZSBSZW5kZXJRdWV1ZSAmIGNoaWxkLlxuICAgIGNoaWxkLm5vZGUuYWRkRGVwZW5kZW5jeSh0aGlzLmxpc3RlbmVyKTtcbiAgICBjaGlsZC5ub2RlLmFkZERlcGVuZGVuY3kodGhpcy50YXNrRGVmaW5pdGlvbik7XG4gICAgY2hpbGQubm9kZS5hZGREZXBlbmRlbmN5KHRoaXMucGF0dGVybi5zZXJ2aWNlKTtcbiAgICBjaGlsZC5ub2RlLmFkZERlcGVuZGVuY3kodGhpcy5lY3NTZXJ2aWNlU3RhYmlsaXplZCk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkRnJvbnRlbmRTZWN1cml0eUdyb3VwcyguLi5zZWN1cml0eUdyb3VwczogSVNlY3VyaXR5R3JvdXBbXSk6IHZvaWQge1xuICAgIHNlY3VyaXR5R3JvdXBzLmZvckVhY2goc2VjdXJpdHlHcm91cCA9PiB0aGlzLmxvYWRCYWxhbmNlci5hZGRTZWN1cml0eUdyb3VwKHNlY3VyaXR5R3JvdXApKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkQmFja2VuZFNlY3VyaXR5R3JvdXBzKC4uLnNlY3VyaXR5R3JvdXBzOiBJU2VjdXJpdHlHcm91cFtdKTogdm9pZCB7XG4gICAgc2VjdXJpdHlHcm91cHMuZm9yRWFjaChzZWN1cml0eUdyb3VwID0+IHRoaXMuYXNnLmFkZFNlY3VyaXR5R3JvdXAoc2VjdXJpdHlHcm91cCkpO1xuICB9XG5cbiAgcHJpdmF0ZSBlbmFibGVGaWxlY2FjaGluZyhhc2c6IEF1dG9TY2FsaW5nR3JvdXApOiB2b2lkIHtcbiAgICBjb25zdCBzY3JpcHQgPSBTY3JpcHRBc3NldC5mcm9tUGF0aENvbnZlbnRpb24odGhpcywgJ0ZpbGVjYWNoaW5nU2NyaXB0Jywge1xuICAgICAgb3NUeXBlOiBhc2cub3NUeXBlLFxuICAgICAgYmFzZU5hbWU6ICdlbmFibGVDYWNoZUZpbGVzZCcsXG4gICAgICByb290RGlyOiBqb2luKF9fZGlybmFtZSwgJy4uJywgJ3NjcmlwdHMnKSxcbiAgICB9KTtcbiAgICAvLyBBIGNvbW1lbnQgaW4gdXNlckRhdGEgdG8gbWFrZSB0aGlzIGVhc2llciB0byB0ZXN0LlxuICAgIGFzZy51c2VyRGF0YS5hZGRDb21tYW5kcygnIyBSZW5kZXJRdWV1ZSBmaWxlIGNhY2hpbmcgZW5hYmxlZCcpO1xuICAgIHNjcmlwdC5leGVjdXRlT24oe1xuICAgICAgaG9zdDogYXNnLFxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVUYXNrRGVmaW5pdGlvbihwcm9wczoge1xuICAgIGltYWdlOiBDb250YWluZXJJbWFnZSxcbiAgICBwb3J0TnVtYmVyOiBudW1iZXIsXG4gICAgcHJvdG9jb2w6IEFwcGxpY2F0aW9uUHJvdG9jb2wsXG4gICAgcmVwb3NpdG9yeTogSVJlcG9zaXRvcnksXG4gIH0pIHtcbiAgICBjb25zdCB7IGltYWdlLCBwb3J0TnVtYmVyLCBwcm90b2NvbCwgcmVwb3NpdG9yeSB9ID0gcHJvcHM7XG5cbiAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBFYzJUYXNrRGVmaW5pdGlvbih0aGlzLCAnUkNTVGFzaycpO1xuXG4gICAgLy8gTW91bnQgdGhlIHJlcG8gZmlsZXN5c3RlbSB0byBSZW5kZXJRdWV1ZS5IT1NUX1JFUE9fRlNfTU9VTlRfUEFUSFxuICAgIGNvbnN0IGNvbm5lY3Rpb24gPSByZXBvc2l0b3J5LmNvbmZpZ3VyZUNsaWVudEVDUyh7XG4gICAgICBjb250YWluZXJJbnN0YW5jZXM6IHtcbiAgICAgICAgaG9zdHM6IFt0aGlzLmFzZ10sXG4gICAgICB9LFxuICAgICAgY29udGFpbmVyczoge1xuICAgICAgICB0YXNrRGVmaW5pdGlvbixcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBlbnZpcm9ubWVudCA9IGNvbm5lY3Rpb24uY29udGFpbmVyRW52aXJvbm1lbnQ7XG5cbiAgICBpZiAocHJvdG9jb2wgPT09IEFwcGxpY2F0aW9uUHJvdG9jb2wuSFRUUFMpIHtcbiAgICAgIC8vIEdlbmVyYXRlIGEgc2VsZi1zaWduZWQgWDUwOSBjZXJ0aWZpY2F0ZSwgcHJpdmF0ZSBrZXkgYW5kIHBhc3NwaHJhc2UgZm9yIHVzZSBieSB0aGUgUkNTIGNvbnRhaW5lcnMuXG4gICAgICAvLyBOb3RlOiB0aGUgQXBwbGljYXRpb24gTG9hZCBCYWxhbmNlciBkb2VzIG5vdCB2YWxpZGF0ZSB0aGUgY2VydGlmaWNhdGUgaW4gYW55IHdheS5cbiAgICAgIGNvbnN0IHJjc0NlcnRQZW0gPSBuZXcgWDUwOUNlcnRpZmljYXRlUGVtKHRoaXMsICdUbHNDYUNlcnRQZW0nLCB7XG4gICAgICAgIHN1YmplY3Q6IHtcbiAgICAgICAgICBjbjogJ3JlbmRlcmZhcm0ubG9jYWwnLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgICBjb25zdCByY3NDZXJ0UGtjcyA9IG5ldyBYNTA5Q2VydGlmaWNhdGVQa2NzMTIodGhpcywgJ1Rsc1Jjc0NlcnRCdW5kbGUnLCB7XG4gICAgICAgIHNvdXJjZUNlcnRpZmljYXRlOiByY3NDZXJ0UGVtLFxuICAgICAgfSk7XG4gICAgICBbcmNzQ2VydFBlbS5jZXJ0LCByY3NDZXJ0UGtjcy5jZXJ0LCByY3NDZXJ0UGtjcy5wYXNzcGhyYXNlXS5mb3JFYWNoKHNlY3JldCA9PiB7XG4gICAgICAgIHNlY3JldC5ncmFudFJlYWQodGFza0RlZmluaXRpb24udGFza1JvbGUpO1xuICAgICAgfSk7XG4gICAgICBlbnZpcm9ubWVudC5SQ1NfVExTX0NBX0NFUlRfVVJJID0gcmNzQ2VydFBlbS5jZXJ0LnNlY3JldEFybjtcbiAgICAgIGVudmlyb25tZW50LlJDU19UTFNfQ0VSVF9VUkkgPSByY3NDZXJ0UGtjcy5jZXJ0LnNlY3JldEFybjtcbiAgICAgIGVudmlyb25tZW50LlJDU19UTFNfQ0VSVF9QQVNTUEhSQVNFX1VSSSA9IHJjc0NlcnRQa2NzLnBhc3NwaHJhc2Uuc2VjcmV0QXJuO1xuICAgICAgZW52aXJvbm1lbnQuUkNTX1RMU19SRVFVSVJFX0NMSUVOVF9DRVJUID0gJ25vJztcbiAgICB9XG5cbiAgICBjb25zdCBjb250YWluZXJEZWZpbml0aW9uID0gdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdDb250YWluZXJEZWZpbml0aW9uJywge1xuICAgICAgaW1hZ2UsXG4gICAgICBtZW1vcnlSZXNlcnZhdGlvbk1pQjogMjA0OCxcbiAgICAgIGVudmlyb25tZW50LFxuICAgICAgbG9nZ2luZzogTG9nRHJpdmVyLmF3c0xvZ3Moe1xuICAgICAgICBsb2dHcm91cDogdGhpcy5sb2dHcm91cCxcbiAgICAgICAgc3RyZWFtUHJlZml4OiAnUkNTJyxcbiAgICAgIH0pLFxuICAgIH0pO1xuXG4gICAgY29udGFpbmVyRGVmaW5pdGlvbi5hZGRNb3VudFBvaW50cyhjb25uZWN0aW9uLnJlYWRXcml0ZU1vdW50UG9pbnQpO1xuXG4gICAgLy8gSW5jcmVhc2UgdWxpbWl0c1xuICAgIGNvbnRhaW5lckRlZmluaXRpb24uYWRkVWxpbWl0cyhcbiAgICAgIHtcbiAgICAgICAgbmFtZTogVWxpbWl0TmFtZS5OT0ZJTEUsXG4gICAgICAgIHNvZnRMaW1pdDogMjAwMDAwLFxuICAgICAgICBoYXJkTGltaXQ6IDIwMDAwMCxcbiAgICAgIH0sIHtcbiAgICAgICAgbmFtZTogVWxpbWl0TmFtZS5OUFJPQyxcbiAgICAgICAgc29mdExpbWl0OiA2NDAwMCxcbiAgICAgICAgaGFyZExpbWl0OiA2NDAwMCxcbiAgICAgIH0sXG4gICAgKTtcblxuICAgIGNvbnRhaW5lckRlZmluaXRpb24uYWRkUG9ydE1hcHBpbmdzKHtcbiAgICAgIGNvbnRhaW5lclBvcnQ6IHBvcnROdW1iZXIsXG4gICAgICBob3N0UG9ydDogcG9ydE51bWJlcixcbiAgICB9KTtcblxuICAgIHJldHVybiB0YXNrRGVmaW5pdGlvbjtcbiAgfVxufVxuIl19