"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseConnection = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
/**
 * Helper class for connecting Thinkbox's Deadline to a specific Database.
 *
 * @stability stable
 */
class DatabaseConnection {
    /**
     * Creates a DatabaseConnection which allows Deadline to connect to Amazon DocumentDB.
     *
     * Note: Deadline officially supports only databases that are compatible with MongoDB 3.6.
     *
     * Resources Deployed
     * ------------------------
     * This construct does not deploy any resources
     *
     * @stability stable
     */
    static forDocDB(options) {
        return new DocDBDatabaseConnection(options);
    }
    /**
     * Creates a DatabaseConnection which allows Deadline to connect to MongoDB.
     *
     * Note: Deadline officially supports only databases that are compatible with MongoDB 3.6.
     *
     * Resources Deployed
     * ------------------------
     * This construct does not deploy any resources
     *
     * @stability stable
     */
    static forMongoDbInstance(options) {
        return new MongoDbInstanceDatabaseConnection(options);
    }
}
exports.DatabaseConnection = DatabaseConnection;
_a = JSII_RTTI_SYMBOL_1;
DatabaseConnection[_a] = { fqn: "aws-rfdk.deadline.DatabaseConnection", version: "0.34.0" };
/**
 * Specialization of {@link DatabaseConnection} targetting Amazon DocumentDB.
 */
class DocDBDatabaseConnection extends DatabaseConnection {
    constructor(props) {
        super();
        this.props = props;
        if (!this.isCompatibleDocDBVersion()) {
            core_1.Annotations.of(props.database).addError('engineVersion must be 3.6.0 to be compatible with Deadline');
        }
        this.containerEnvironment = {
            // The container must fetch the credentials from Secrets Manager
            DB_CREDENTIALS_URI: this.props.login.secretArn,
        };
    }
    /**
     * @inheritdoc
     */
    addInstallerDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Can only install Deadline from a Linux instance.');
        }
        host.userData.addCommands('configure_database_installation_args(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export SECRET_STRING=\`aws secretsmanager get-secret-value --secret-id ${this.props.login.secretArn} --region ${core_1.Stack.of(this.props.login).region} | getJsonVal 'SecretString'\``, "DB_USERNAME=`printenv SECRET_STRING | getJsonVal 'username'`", "DB_PASSWORD=`printenv SECRET_STRING | getJsonVal 'password'`", 'unset SECRET_STRING', `INSTALLER_DB_ARGS=( ["--dbuser"]=$DB_USERNAME ["--dbpassword"]=$DB_PASSWORD ["--dbhost"]=${this.props.database.clusterEndpoint.hostname}` +
            ` ["--dbport"]=${this.props.database.clusterEndpoint.portAsString()} ["--dbtype"]=DocumentDB )`, 'unset DB_USERNAME', 'unset DB_PASSWORD', 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_database_installation_args');
    }
    /**
     * @inheritdoc
     */
    addConnectionDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Connecting to the Deadline Database is currently only supported for Linux.');
        }
        host.userData.addCommands('configure_deadline_database(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export SECRET_STRING=\`aws secretsmanager get-secret-value --secret-id ${this.props.login.secretArn} --region ${core_1.Stack.of(this.props.login).region} | getJsonVal 'SecretString'\``, "DB_USERNAME=`printenv SECRET_STRING | getJsonVal 'username'`", "DB_PASSWORD=`printenv SECRET_STRING | getJsonVal 'password'`", 'unset SECRET_STRING', 'sudo -u ec2-user "${deadlinecommand}" -StoreDatabasecredentials "${DB_USERNAME}" "${DB_PASSWORD}"', 'unset DB_USERNAME', 'unset DB_PASSWORD', 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_deadline_database');
    }
    /**
     * @inheritdoc
     */
    allowConnectionsFrom(other) {
        other.connections.allowTo(this.props.database, this.props.database.connections.defaultPort);
    }
    /**
     * @inheritdoc
     */
    grantRead(grantee) {
        this.props.login.grantRead(grantee);
    }
    /**
     * @inheritdoc
     */
    addChildDependency(child) {
        // To depend on document DB it is not sufficient to depend on the Cluster. The instances are what serves data, so
        // we must add a dependency to an instance in the DocDB cluster.
        // The DocDB L2 does not expose any of its instances as properties, so we have to escape-hatch to gain access.
        const docdbInstance = this.props.database.node.tryFindChild('Instance1');
        // We won't find an instance in two situations:
        //  1) The DocDB Cluster was created from attributes. In this case, the DocDB pre-exists the stack and there's no need
        //     to depend on anything.
        //  2) The DocDB Cluster was constructed, but the internal name for the instance has been changed from 'Instance1'; this is
        //     unlikely, but not impossible.
        // We can differentiate cases (1) & (2) by looking for the defaultChild on the cluster. The version from attributes will not have one.
        if (docdbInstance) {
            child.node.addDependency(docdbInstance);
        }
        else if (this.props.database.node.defaultChild) {
            throw new Error('The internal implementation of the AWS CDK\'s DocumentDB cluster construct may have changed. Please update to a newer RFDK for an updated implementation, or file a ticket if this is the latest release.');
        }
    }
    /**
     * @inheritdoc
     */
    addSecurityGroup(...securityGroups) {
        let added = false;
        const errorReasons = [];
        if (this.props.database instanceof aws_docdb_1.DatabaseCluster) {
            const resource = this.props.database.node.tryFindChild('Resource');
            // TODO: Replace this code with the addSecurityGroup method of DatabaseCluster once this PR is merged:
            // https://github.com/aws/aws-cdk/pull/13290
            if (resource instanceof aws_docdb_1.CfnDBCluster) {
                const cfnCluster = resource;
                const securityGroupIds = securityGroups.map(sg => sg.securityGroupId);
                if (cfnCluster.vpcSecurityGroupIds === undefined) {
                    cfnCluster.vpcSecurityGroupIds = securityGroupIds;
                }
                else {
                    cfnCluster.vpcSecurityGroupIds.push(...securityGroupIds);
                }
                added = true;
            }
            else {
                errorReasons.push('The internal implementation of AWS CDK\'s DocumentDB cluster construct has changed.');
            }
        }
        else {
            errorReasons.push('The "database" property passed to this class is not an instance of AWS CDK\'s DocumentDB cluster construct.');
        }
        if (!added) {
            core_1.Annotations.of(this.props.database).addWarning(`Failed to add the following security groups to ${this.props.database.node.id}: ${securityGroups.map(sg => sg.node.id).join(', ')}. ` +
                errorReasons.join(' '));
        }
    }
    /**
     * Deadline is only compatible with MongoDB 3.6. This function attempts to determine whether
     * the given DocDB version is compatible.
     */
    isCompatibleDocDBVersion() {
        var _b, _c;
        // The defaultChild of a DocDB DatabaseCluster is a CfnDBCluster, but we only have this
        // child if the customer didn't import from attributes. We can check the DB version by
        // checking the value of the engineVersion property of that object.
        if (this.props.database.node.defaultChild) {
            const cluster = this.props.database.node.defaultChild;
            return (_c = (_b = cluster.engineVersion) === null || _b === void 0 ? void 0 : _b.startsWith('3.6')) !== null && _c !== void 0 ? _c : false;
        }
        return true; // No information, assume it's compatible.
    }
}
/**
 * Specialization of {@link DatabaseConnection} targetting MongoDB.
 */
class MongoDbInstanceDatabaseConnection extends DatabaseConnection {
    constructor(props) {
        super();
        this.props = props;
        this.containerEnvironment = {
            DB_TLS_CLIENT_CERT_URI: props.clientCertificate.cert.secretArn,
            DB_TLS_CLIENT_CERT_PASSWORD_URI: props.clientCertificate.passphrase.secretArn,
        };
    }
    /**
     * @inheritdoc
     */
    allowConnectionsFrom(other) {
        other.connections.allowTo(this.props.database, aws_ec2_1.Port.tcp(this.props.database.port));
    }
    /**
     * @inheritdoc
     */
    addInstallerDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Can only install Deadline from a Linux instance.');
        }
        this.downloadCertificate(host);
        const certPwSecret = this.props.clientCertificate.passphrase;
        host.userData.addCommands('configure_database_installation_args(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 
        // Suppress -x, so no secrets go to the logs
        'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `CERT_PASSWORD=$(aws secretsmanager get-secret-value --secret-id ${certPwSecret.secretArn} --region ${core_1.Stack.of(certPwSecret).region} | getJsonVal 'SecretString')`, 'INSTALLER_DB_ARGS=( ["--dbssl"]=true ["--dbauth"]=true ["--dbsslauth"]=true ' +
            `["--dbhost"]="${this.props.database.fullHostname}" ["--dbport"]=${this.props.database.port} ` +
            `["--dbclientcert"]="${MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION}" ["--dbcertpass"]=$CERT_PASSWORD )`, 'unset CERT_PASSWORD', 
        // Restore -x, if it was set.
        'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_database_installation_args');
    }
    /**
     * @inheritdoc
     */
    addConnectionDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Connecting to the Deadline Database is currently only supported for Linux.');
        }
        this.downloadCertificate(host);
        const certPwSecret = this.props.clientCertificate.passphrase;
        host.userData.addCommands('configure_deadline_database(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export DB_CERT_FILE="${MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION}"`, `export DB_CERT_PASSWORD=$(aws secretsmanager get-secret-value --secret-id ${certPwSecret.secretArn} --region ${core_1.Stack.of(certPwSecret).region} | getJsonVal 'SecretString')`, 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_deadline_database');
    }
    /**
     * @inheritdoc
     */
    grantRead(grantee) {
        this.props.clientCertificate.cert.grantRead(grantee);
        this.props.clientCertificate.passphrase.grantRead(grantee);
    }
    /**
     * @inheritdoc
     */
    addChildDependency(child) {
        if (this.props.database.hasOwnProperty('server')) {
            const db = this.props.database;
            child.node.addDependency(db.server.autoscalingGroup.node.defaultChild);
        }
    }
    /**
     * @inheritdoc
     */
    addSecurityGroup(...securityGroups) {
        this.props.database.addSecurityGroup(...securityGroups);
    }
    /**
     * Download the client PKCS#12 certificate for authenticating to the MongoDB, and place it into
     * the path defined by: DB_CERT_LOCATION
     * @param host
     */
    downloadCertificate(host) {
        var _b;
        const stack = core_1.Stack.of(host);
        const uuid = 'e8125dd2-ab2c-4861-8ee4-998c26b30ee0';
        const uniqueId = 'GetSecretToFile' + host.osType + uuid.replace(/[-]/g, '');
        const getSecretsScript = (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : core_2.ScriptAsset.fromPathConvention(stack, uniqueId, {
            osType: host.osType,
            baseName: 'getSecretToFile',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        getSecretsScript.executeOn({
            host,
            args: [
                this.props.clientCertificate.cert.secretArn,
                MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION,
            ],
        });
    }
}
MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION = '/opt/Thinkbox/certs/mongo_client.pfx';
//# sourceMappingURL=data:application/json;base64,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