"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
const tag_helpers_1 = require("./tag-helpers");
const token_regex_helpers_1 = require("./token-regex-helpers");
describe('Test MongoDbInstance', () => {
    let app;
    let stack;
    let vpc;
    let dnsZone;
    let caCert;
    let serverCert;
    const hostname = 'hostname';
    const zoneName = 'testZone';
    const version = lib_1.MongoDbVersion.COMMUNITY_3_6;
    const userSsplAcceptance = lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL;
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app, 'Stack');
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        dnsZone = new aws_route53_1.PrivateHostedZone(stack, 'PrivateHostedZone', {
            vpc,
            zoneName,
        });
        caCert = new lib_1.X509CertificatePem(stack, 'CaCert', {
            subject: {
                cn: 'DistinguishedName',
            },
        });
        serverCert = new lib_1.X509CertificatePem(stack, 'ServerCert', {
            subject: {
                cn: `${hostname}.${zoneName}`,
            },
            signingCertificate: caCert,
        });
    });
    test('default mongodb instance is created correctly', () => {
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: 'r5.large',
            BlockDeviceMappings: assert_1.arrayWith(assert_1.objectLike({
                Ebs: assert_1.objectLike({
                    Encrypted: true,
                }),
            })),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Route53::RecordSet', {
            Name: hostname + '.' + zoneName + '.',
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SecretsManager::Secret', {
            Description: `Admin credentials for the MongoDB database ${core_1.Names.uniqueId(instance)}`,
            GenerateSecretString: {
                ExcludeCharacters: '\"()$\'',
                ExcludePunctuation: true,
                GenerateStringKey: 'password',
                IncludeSpace: false,
                PasswordLength: 24,
                RequireEachIncludedType: true,
                SecretStringTemplate: '{\"username\":\"admin\"}',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Encrypted: true,
            Tags: assert_1.arrayWith(assert_1.objectLike({
                Key: 'VolumeGrantAttach-6238D22B12',
                Value: '6238d22b121db8094cb816e2a49d2b61',
            })),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: assert_1.objectLike({
                Statement: assert_1.arrayWith({
                    Action: [
                        'logs:CreateLogStream',
                        'logs:PutLogEvents',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':logs:',
                                {
                                    Ref: 'AWS::Region',
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':log-group:',
                                {
                                    'Fn::GetAtt': [
                                        'MongoDbInstanceMongoDbInstanceLogGroupWrapperEAF733BB',
                                        'LogGroupName',
                                    ],
                                },
                                ':*',
                            ],
                        ],
                    },
                }, {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Resource: [
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket,
                                    },
                                ],
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket,
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'Cert',
                        ],
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'CertChain',
                        ],
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'Key',
                        ],
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'ServerCertPassphraseE4C3CB38',
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'MongoDbInstanceAdminUser54147F2B',
                    },
                }),
            }),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
            LogGroupName: '/renderfarm/MongoDbInstance',
        }));
        const cloudInitLogPath = '/var/log/cloud-init-output.log';
        const cloudInitLogPrefix = 'cloud-init-output';
        const mongoLogPath = '/var/log/mongodb/mongod.log';
        const mongoLogPrefix = 'MongoDB';
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SSM::Parameter', {
            Description: 'config file for Repository logs config',
            Value: assert_1.objectLike({
                'Fn::Join': assert_1.arrayWith(assert_1.arrayWith('\",\"log_stream_name\":\"' + cloudInitLogPrefix + '-{instance_id}\",\"file_path\":\"' + cloudInitLogPath + '\",' +
                    '\"timezone\":\"Local\"},{\"log_group_name\":\"', '\",\"log_stream_name\":\"' + mongoLogPrefix + '-{instance_id}\",\"file_path\":\"' + mongoLogPath + '\"' +
                    ',\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}')),
            }),
        }));
        const userData = instance.userData.render();
        const token = '${Token[TOKEN.\\d+]}';
        // Make sure we add signal on exit
        const exitTrap = '#!/bin/bash\n' +
            'function exitTrap(){\n' +
            'exitCode=$?\n' +
            '/opt/aws/bin/cfn-signal --stack Stack --resource MongoDbInstanceServerAsgASG2643AD1D --region ' + token +
            ' -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n' +
            'test \"${MONGO_SETUP_DIR} != \"\" && sudo umount \"${MONGO_SETUP_DIR}\n' +
            '}';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(exitTrap)));
        const callExitTrap = 'trap exitTrap EXIT';
        expect(userData).toMatch(new RegExp(callExitTrap));
        const settings = 'set -xefuo pipefail';
        expect(userData).toMatch(new RegExp(settings));
        const createTempDir = 'mkdir -p $\\(dirname \'/tmp/' + token + token + '\'\\)\n';
        const s3Copy = 'aws s3 cp \'s3://' + token + '/' + token + token + '\' \'/tmp/' + token + token + '\'\n';
        // CloudWatch Agent
        const setE = 'set -e\n';
        const setChmod = 'chmod \\+x \'/tmp/' + token + token + '\'\n';
        const execute = '\'/tmp/' + token + token + '\' -i ${Token[AWS.Region.\\d+]} ' + token + '\n';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + setE + setChmod + execute)));
        // Make sure we mount EBS volume
        const mount = 'TMPDIR=$\\(mktemp -d\\)\n' +
            'pushd \"$TMPDIR\"\n' +
            'unzip /tmp/' + token + token + '\n' +
            'bash ./mountEbsBlockVolume.sh ' + token + ' xfs /var/lib/mongo rw \"\"\n' +
            'popd\n' +
            'rm -f /tmp/' + token + token;
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + mount)));
        // install mongodb
        const bashCmd = 'bash /tmp/' + token + token;
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + bashCmd)));
        // configureMongoDb
        const monogdbConfig = 'which mongod && test -f /etc/mongod.conf\n' +
            'sudo service mongod stop\n' +
            'MONGO_SETUP_DIR=$\\(mktemp -d\\)\n' +
            'mkdir -p \"${MONGO_SETUP_DIR}\"\n' +
            'sudo mount -t tmpfs -o size=50M tmpfs \"${MONGO_SETUP_DIR}\"\n' +
            'pushd \"${MONGO_SETUP_DIR}\"\n' +
            'unzip /tmp/' + token + token + '\n' +
            'cp /etc/mongod.conf .';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + monogdbConfig)));
        // Getting the server certificate
        const serverCertCmd = 'bash serverCertFromSecrets.sh \\"' + token + '\\" \\"' + token + '\\" \\"' + token + '\\" \\"' + token + '\\"';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(serverCertCmd)));
        // set mongodb certificates and credentials
        const monogdbCredentials = 'sudo mkdir -p /etc/mongod_certs\n' +
            'sudo mv ./ca.crt ./key.pem /etc/mongod_certs\n' +
            'sudo chown root.mongod -R /etc/mongod_certs/\n' +
            'sudo chmod 640 -R /etc/mongod_certs/\n' +
            'sudo chmod 750 /etc/mongod_certs/\n' +
            'sudo chown mongod.mongod -R /var/lib/mongo\n' +
            'bash ./setMongoLimits.sh\n' +
            'bash ./setStoragePath.sh \"/var/lib/mongo\"\n' +
            'bash ./setMongoNoAuth.sh\n' +
            'sudo service mongod start\n' +
            'bash ./setAdminCredentials.sh \"' + token + '\"';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(monogdbCredentials)));
        // Setup for live deployment, and start mongod
        const startMongo = 'sudo service mongod stop\n' +
            'bash ./setLiveConfiguration.sh\n' +
            'sudo systemctl enable mongod\n' +
            'sudo service mongod start\n' +
            'popd';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(startMongo)));
        // Make sure all the required public members are set
        expect(instance.version).toBe(version);
        expect(instance.connections).toBeDefined();
        expect(instance.connections).toBe(instance.server.connections);
        expect(instance.grantPrincipal).toBeDefined();
        expect(instance.grantPrincipal).toBe(instance.server.grantPrincipal);
        expect(instance.port).toBeDefined();
        expect(instance.role).toBeDefined();
        expect(instance.role).toBe(instance.server.role);
        expect(instance.userData).toBeDefined();
        expect(instance.userData).toBe(instance.server.userData);
        expect(instance.fullHostname).toBeDefined();
    });
    test('throw exception when no available subnets', () => {
        // GIVEN
        const invalidSubnets = {
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
            availabilityZones: ['dummy zone'],
        };
        // THEN
        expect(() => {
            new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
                mongoDb: {
                    version,
                    dnsZone,
                    hostname,
                    serverCertificate: serverCert,
                    userSsplAcceptance,
                },
                vpc,
                vpcSubnets: invalidSubnets,
            });
        }).toThrowError(/Did not find any subnets matching/);
    });
    test('changing instance type works correctly', () => {
        // GIVEN
        const expectedInstanceType = 'm4.micro';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            instanceType: new aws_ec2_1.InstanceType(expectedInstanceType),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: expectedInstanceType,
        }));
    });
    test('allowing ssh connection with key name', () => {
        // GIVEN
        const expectedKeyName = 'someKeyName';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            keyName: expectedKeyName,
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            KeyName: expectedKeyName,
        }));
    });
    test('using custom admin user works correctly', () => {
        // GIVEN
        const expectedAdminUser = new aws_secretsmanager_1.Secret(stack, 'AdminUser', {
            description: 'Custom admin credentials for the MongoDB database',
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
                generateStringKey: 'test_password',
                secretStringTemplate: JSON.stringify({ username: 'admin' }),
            },
        });
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                adminUser: expectedAdminUser,
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        expect(instance.adminUser).toBe(expectedAdminUser);
    });
    test('setting security group works correctly', () => {
        // GIVEN
        const actualSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'SecurityGroup', {
            securityGroupName: 'CustomSecurityGroup',
            vpc,
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            securityGroup: actualSecurityGroup,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::SecurityGroup', 1));
    });
    test('setting role works correctly', () => {
        // GIVEN
        const expectedRole = new aws_iam_1.Role(stack, 'Role', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            roleName: 'CustomRole',
        });
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            role: expectedRole,
        });
        // THEN
        expect(instance.server.role).toBe(expectedRole);
        expect(instance.role).toBe(expectedRole);
    });
    test('setting custom data volume works correctly', () => {
        // GIVEN
        const actualVolume = new aws_ec2_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: core_1.Size.gibibytes(50),
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volume: actualVolume,
                },
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Volume', 1));
    });
    test('setting custom encryption key for data volume works correctly', () => {
        // GIVEN
        // KmsKeyId is Key961B73FD
        const actualEncryptionKey = new aws_kms_1.Key(stack, 'Key', {
            description: 'Key for testing',
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volumeProps: {
                        encryptionKey: actualEncryptionKey,
                    },
                },
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Encrypted: true,
            KmsKeyId: assert_1.objectLike({
                'Fn::GetAtt': assert_1.arrayWith('Key961B73FD'),
            }),
        }));
    });
    test('setting custom size for data volume works correctly', () => {
        // GIVEN
        const volumeSize = 123;
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volumeProps: {
                        size: core_1.Size.gibibytes(volumeSize),
                    },
                },
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Size: volumeSize,
        }));
    });
    test('setting LogGroup bucket name enables export to S3', () => {
        // GIVEN
        const bucketName = 'test-bucket';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                bucketName,
            },
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Targets: assert_1.arrayWith(assert_1.objectLike({
                Input: '{\"BucketName\":\"' + bucketName + '\",\"ExportFrequencyInHours\":1,\"LogGroupName\":\"/renderfarm/MongoDbInstance\",\"RetentionInHours\":72}',
            })),
        }));
    });
    test.each([
        'test-prefix/',
        '',
    ])('is created with correct LogGroup prefix %s', (testPrefix) => {
        // GIVEN
        const id = 'MongoDbInstance';
        // WHEN
        new lib_1.MongoDbInstance(stack, id, {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                logGroupPrefix: testPrefix,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            LogGroupName: testPrefix + id,
        }));
    });
    test('is created with correct LogGroup retention', () => {
        // GIVEN
        const retention = aws_logs_1.RetentionDays.ONE_DAY;
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                retention,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            RetentionInDays: retention,
        }));
    });
    test('adds security group', () => {
        // GIVEN
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'NewSecurityGroup', {
            vpc,
        });
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // WHEN
        instance.addSecurityGroup(securityGroup);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            SecurityGroups: assert_1.arrayWith(stack.resolve(securityGroup.securityGroupId)),
        }));
    });
    tag_helpers_1.testConstructTags({
        constructName: 'MongoDbInstance',
        createConstruct: () => {
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            new lib_1.MongoDbInstance(isolatedStack, 'MongoDbInstance', {
                mongoDb: {
                    version,
                    dnsZone,
                    hostname,
                    serverCertificate: serverCert,
                    userSsplAcceptance,
                },
                vpc,
            });
            return isolatedStack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::EC2::NetworkInterface': 1,
            'AWS::SecretsManager::Secret': 1,
            'AWS::EC2::Volume': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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