"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.X509CertificatePkcs12 = exports.X509CertificatePem = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto = require("crypto");
const path_1 = require("path");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const lambdaLayerVersionArns_1 = require("../../lambdas/lambdaLayerVersionArns");
class X509CertificateBase extends core_1.Construct {
    constructor(scope, id, props) {
        var _c;
        super(scope, id);
        this.database = new aws_dynamodb_1.Table(this, 'Table', {
            partitionKey: { name: 'PhysicalId', type: aws_dynamodb_1.AttributeType.STRING },
            sortKey: { name: 'CustomResource', type: aws_dynamodb_1.AttributeType.STRING },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            serverSideEncryption: true,
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
        });
        this.passphrase = new aws_secretsmanager_1.Secret(this, 'Passphrase', {
            description: `Passphrase for the private key of the X509Certificate ${core_1.Names.uniqueId(this)}`,
            encryptionKey: props.encryptionKey,
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
            },
        });
        const region = core_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        /*
         * We cannot make this a singleton function; doing so would create circular references in the lambda role (to sign
         * a cert we need a cert that this lambda generated).
         */
        this.lambdaFunc = new aws_lambda_1.Function(this, 'Generator', {
            description: `Used by a X509Certificate ${core_1.Names.uniqueId(this)} to generate certificates.`,
            code: props.lambdaCode,
            environment: {
                DATABASE: this.database.tableName,
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            layers: [openSslLayer],
            handler: props.lambdaHandler,
            timeout: core_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        this.database.grantReadWriteData(this.lambdaFunc);
        this.database.grant(this.lambdaFunc, 'dynamodb:DescribeTable');
        (_c = props.encryptionKey) === null || _c === void 0 ? void 0 : _c.grantEncrypt(this.lambdaFunc);
        this.passphrase.grantRead(this.lambdaFunc);
        const uniqueValue = crypto.createHash('md5').update(core_1.Names.uniqueId(this)).digest('hex');
        this.uniqueTag = new core_1.Tag(`X509SecretGrant-${uniqueValue.slice(0, 8).toUpperCase()}`, uniqueValue);
        const tagCondition = {};
        tagCondition[`secretsmanager:ResourceTag/${this.uniqueTag.key}`] = this.uniqueTag.value;
        this.lambdaFunc.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'secretsmanager:CreateSecret',
                'secretsmanager:DeleteSecret',
                'secretsmanager:TagResource',
                'secretsmanager:PutSecretValue',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
    }
}
/**
 * A Construct that uses a Lambda to generate an X.509 certificate and then saves the certificate's components into Secrets. On an update, if any properties of the construct are changed, then a new certificate will be generated. When the Stack is destroyed or the Construct is removed, the Secrets will all be deleted. An X.509 certificate is comprised of the certificate, a certificate chain with the chain of signing certificates (if any), and a private key that is password protected by a randomly generated passphrase.
 *
 * Cost:
 * The cost of four AWS SecretsManager Secrets in the deployed region.
 * The other resources created by this construct have negligible ongoing costs.
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the Custom Resource.
 * - Secrets - 4 in total, for the certificate, it's private key, the passphrase to the key, and the cert chain.
 * - Lambda Function, with role - Used to create/update/delete the Custom Resource
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - Access to the AWS SecretsManager Secrets that are created by this construct should be tightly restricted
 *    to only the principal(s) that require access.
 *
 * @stability stable
 */
class X509CertificatePem extends X509CertificateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        super(scope, id, {
            lambdaCode: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            lambdaHandler: 'x509-certificate.generate',
            encryptionKey: props.encryptionKey,
        });
        if (((_c = props.validFor) !== null && _c !== void 0 ? _c : 1) < 1 && !core_1.Token.isUnresolved(props.validFor)) {
            core_1.Annotations.of(this).addError('Certificates must be valid for at least one day.');
        }
        (_d = props.signingCertificate) === null || _d === void 0 ? void 0 : _d.cert.grantRead(this.lambdaFunc);
        (_e = props.signingCertificate) === null || _e === void 0 ? void 0 : _e.key.grantRead(this.lambdaFunc);
        (_f = props.signingCertificate) === null || _f === void 0 ? void 0 : _f.passphrase.grantRead(this.lambdaFunc);
        (_h = (_g = props.signingCertificate) === null || _g === void 0 ? void 0 : _g.certChain) === null || _h === void 0 ? void 0 : _h.grantRead(this.lambdaFunc);
        const signingCertificate = props.signingCertificate
            ? {
                Cert: props.signingCertificate.cert.secretArn,
                Key: props.signingCertificate.key.secretArn,
                Passphrase: props.signingCertificate.passphrase.secretArn,
                CertChain: props.signingCertificate.certChain ? props.signingCertificate.certChain.secretArn : '',
            }
            : undefined;
        const properties = {
            DistinguishedName: {
                CN: props.subject.cn,
                O: (_j = props.subject.o) !== null && _j !== void 0 ? _j : 'AWS',
                OU: (_k = props.subject.ou) !== null && _k !== void 0 ? _k : 'Thinkbox',
            },
            Passphrase: this.passphrase.secretArn,
            Secret: {
                NamePrefix: this.node.path,
                Description: this.node.path,
                EncryptionKey: (_l = props.encryptionKey) === null || _l === void 0 ? void 0 : _l.keyArn,
                Tags: [
                    {
                        Key: this.uniqueTag.key,
                        Value: this.uniqueTag.value,
                    },
                ],
            },
            SigningCertificate: signingCertificate,
            CertificateValidFor: (_m = props.validFor) === null || _m === void 0 ? void 0 : _m.toString(),
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: this.lambdaFunc.functionArn,
            properties,
            resourceType: 'Custom::RFDK_X509Generator',
        });
        if (this.lambdaFunc.role) {
            // There's a race on update where this resource could execute before the role has updated.
            resource.node.addDependency(this.lambdaFunc.role);
        }
        this.cert = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Cert', {
            secretArn: core_1.Token.asString(resource.getAtt('Cert')),
            encryptionKey: props.encryptionKey,
        });
        this.key = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Key', {
            secretArn: core_1.Token.asString(resource.getAtt('Key')),
            encryptionKey: props.encryptionKey,
        });
        // We'll only have a chain if we used a ca to sign this cert. We cannot check for certChainResource being an empty
        // string because it is an unresolved token at this point.
        if (signingCertificate) {
            const certChainResource = resource.getAtt('CertChain');
            this.certChain = certChainResource
                ? aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'CertChain', {
                    secretArn: core_1.Token.asString(certChainResource),
                    encryptionKey: props.encryptionKey,
                })
                : undefined;
        }
    }
    /**
     * Grant read permissions for the certificate.
     *
     * @stability stable
     */
    grantCertRead(grantee) {
        var _c;
        const result = this.cert.grantRead(grantee);
        (_c = this.certChain) === null || _c === void 0 ? void 0 : _c.grantRead(grantee);
        return result;
    }
    /**
     * Grant read permissions for the certificate, key, and passphrase.
     *
     * @stability stable
     */
    grantFullRead(grantee) {
        var _c;
        const result = this.cert.grantRead(grantee);
        (_c = this.certChain) === null || _c === void 0 ? void 0 : _c.grantRead(grantee);
        this.key.grantRead(grantee);
        this.passphrase.grantRead(grantee);
        return result;
    }
}
exports.X509CertificatePem = X509CertificatePem;
_a = JSII_RTTI_SYMBOL_1;
X509CertificatePem[_a] = { fqn: "aws-rfdk.X509CertificatePem", version: "0.34.0" };
/**
 * This Construct will generate a PKCS #12 file from an X.509 certificate in PEM format. The PEM certificate must be provided through an instance of the X509CertificatePem Construct. A Lambda Function is used to do the conversion and the result is stored in a Secret. The PKCS #12 file is password protected with a passphrase that is randomly generated and stored in a Secret.
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the CustomResource.
 * - Secrets - 2 in total, The binary of the PKCS #12 certificate and its passphrase.
 * - Lambda Function, with role - Used to create/update/delete the CustomResource.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - Access to the AWS SecretsManager Secrets that are created by this construct should be tightly restricted
 *    to only the principal(s) that require access.
 *
 * @stability stable
 */
class X509CertificatePkcs12 extends X509CertificateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c;
        super(scope, id, {
            lambdaCode: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            lambdaHandler: 'x509-certificate.convert',
            encryptionKey: props.encryptionKey,
        });
        props.sourceCertificate.grantFullRead(this.lambdaFunc);
        const properties = {
            Passphrase: this.passphrase.secretArn,
            Secret: {
                NamePrefix: this.node.path,
                Description: this.node.path,
                EncryptionKey: (_c = props.encryptionKey) === null || _c === void 0 ? void 0 : _c.keyArn,
                Tags: [
                    {
                        Key: this.uniqueTag.key,
                        Value: this.uniqueTag.value,
                    },
                ],
            },
            Certificate: {
                Cert: props.sourceCertificate.cert.secretArn,
                CertChain: props.sourceCertificate.certChain ? props.sourceCertificate.certChain.secretArn : '',
                Key: props.sourceCertificate.key.secretArn,
                Passphrase: props.sourceCertificate.passphrase.secretArn,
            },
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: this.lambdaFunc.functionArn,
            properties,
            resourceType: 'Custom::RFDK_X509_PKCS12',
        });
        this.cert = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Cert', {
            secretArn: core_1.Token.asString(resource.getAtt('Cert')),
            encryptionKey: props.encryptionKey,
        });
    }
}
exports.X509CertificatePkcs12 = X509CertificatePkcs12;
_b = JSII_RTTI_SYMBOL_1;
X509CertificatePkcs12[_b] = { fqn: "aws-rfdk.X509CertificatePkcs12", version: "0.34.0" };
//# sourceMappingURL=data:application/json;base64,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