"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountableEfs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const efs = require("@aws-cdk/aws-efs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const mount_permissions_helper_1 = require("./mount-permissions-helper");
/**
 * This class encapsulates scripting that can be used to mount an Amazon Elastic File System onto an instance.
 *
 * An optional EFS access point can be specified for mounting the EFS file-system. For more information on using EFS
 * Access Points, see https://docs.aws.amazon.com/efs/latest/ug/efs-access-points.html. For this to work properly, the
 * EFS mount helper is required. The EFS Mount helper comes pre-installed on Amazon Linux 2. For other Linux
 * distributions, the host machine must have the Amazon EFS client installed. We advise installing the Amazon EFS Client
 * when building your AMI. For instructions on installing the Amazon EFS client for other distributions, see
 * https://docs.aws.amazon.com/efs/latest/ug/installing-amazon-efs-utils.html#installing-other-distro.
 *
 * NOTE: Without an EFS access point, the file-system is writeable only by the root user.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 */
class MountableEfs {
    /**
     * @stability stable
     */
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
        this.fileSystem = props.filesystem;
        this.accessPoint = props.accessPoint;
    }
    /**
     * Mount the filesystem to the given instance at instance startup.
     *
     * This is accomplished by
     * adding scripting to the UserData of the instance to mount the filesystem on startup.
     * If required, the instance's security group is granted ingress to the filesystem's security
     * group on the required ports.
     *
     * @stability stable
     * @inheritdoc true
     */
    mountToLinuxInstance(target, mount) {
        var _b;
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        if (core_1.Construct.isConstruct(target)) {
            target.node.addDependency(this.props.filesystem.mountTargetsAvailable);
        }
        if (this.props.accessPoint) {
            const grantActions = mount_permissions_helper_1.MountPermissionsHelper.toEfsIAMActions(mount === null || mount === void 0 ? void 0 : mount.permissions);
            if (this.accessPointRequiresClientRootAccess(this.props.accessPoint)) {
                grantActions.push('elasticfilesystem:ClientRootAccess');
            }
            target.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                resources: [
                    this.props.filesystem.node.defaultChild.attrArn,
                ],
                actions: grantActions,
                conditions: {
                    StringEquals: {
                        'elasticfilesystem:AccessPointArn': this.props.accessPoint.accessPointArn,
                    },
                },
            }));
        }
        target.connections.allowTo(this.props.filesystem, this.props.filesystem.connections.defaultPort);
        const mountScriptAsset = this.mountAssetSingleton(target);
        mountScriptAsset.grantRead(target.grantPrincipal);
        const mountScript = target.userData.addS3DownloadCommand({
            bucket: mountScriptAsset.bucket,
            bucketKey: mountScriptAsset.s3ObjectKey,
        });
        const mountDir = path.posix.normalize(mount.location);
        const mountOptions = [mount_permissions_helper_1.MountPermissionsHelper.toLinuxMountOption(mount.permissions)];
        if (this.props.accessPoint) {
            mountOptions.push('iam', `accesspoint=${this.props.accessPoint.accessPointId}`);
        }
        if (this.props.extraMountOptions) {
            mountOptions.push(...this.props.extraMountOptions);
        }
        const mountOptionsStr = mountOptions.join(',');
        const resolveMountTargetDnsWithApi = (_b = this.props.resolveMountTargetDnsWithApi) !== null && _b !== void 0 ? _b : false;
        if (resolveMountTargetDnsWithApi) {
            const describeMountTargetResources = [
                this.props.filesystem.node.defaultChild.attrArn,
            ];
            if (this.props.accessPoint) {
                describeMountTargetResources.push(this.props.accessPoint.accessPointArn);
            }
            target.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                resources: describeMountTargetResources,
                actions: ['elasticfilesystem:DescribeMountTargets'],
            }));
        }
        target.userData.addCommands('TMPDIR=$(mktemp -d)', 'pushd "$TMPDIR"', `unzip ${mountScript}`, `bash ./mountEfs.sh ${this.props.filesystem.fileSystemId} ${mountDir} ${resolveMountTargetDnsWithApi} ${mountOptionsStr}`, 'popd', `rm -f ${mountScript}`);
    }
    /**
     * Uses a CDK escape-hatch to fetch the UID/GID of the access point POSIX user.
     *
     * @param accessPoint The access point to obtain the POSIX user for
     */
    getAccessPointPosixUser(accessPoint) {
        const accessPointResource = accessPoint.node.defaultChild;
        return accessPointResource.posixUser;
    }
    /**
     * Uses a synthesis-time check to determine whether an access point is setting its UID/GID to 0 (root). Mounting such
     * an access point requires the `ClientRootAccess` IAM permission.
     *
     * If this introspection is possible and the access point is determined to require root access, the method returns
     * true.
     *
     * If there is no information at synthesis-time, the method returns false as a secure default.
     *
     * @param accessPoint The access point to introspect
     */
    accessPointRequiresClientRootAccess(accessPoint) {
        if (accessPoint instanceof efs.AccessPoint) {
            const posixUser = this.getAccessPointPosixUser(accessPoint);
            // The following code path is cannot be triggered using the L2 construct for EFS Access Points. It currently
            // accepts a PosixUser struct. We will skip coverage for the time-being.
            /* istanbul ignore next */
            if (core_1.isResolvableObject(posixUser)) {
                // We can't know at synthesis time whether this POSIX user is root. Use secure defaults.
                return false;
            }
            if (!posixUser) {
                // No POSIX user specified we will not grant ClientRootAccess permission to opt on the side of secure defaults.
                return false;
            }
            // We have synthesis-time values for the UID/GID being set in the access point. Return true if either is 0 (root).
            return Number(posixUser.uid) === 0 || Number(posixUser.gid) === 0;
        }
        else {
            // This code path is for imported or custom-implementations of efs.AccessPoint
            // We cannot introspect the access point, so we will impose secure defaults and not grant ClientRootAccess.
            return false;
        }
    }
    /**
     * Fetch the Asset singleton for the EFS mounting scripts, or generate it if needed.
     *
     * @stability stable
     */
    mountAssetSingleton(scope) {
        var _b;
        const stack = core_1.Stack.of(scope);
        const uuid = '2b31c419-5b0b-4bb8-99ad-5b2575b2c06b';
        const uniqueId = 'MountableEfsAsset' + uuid.replace(/[-]/g, '');
        return (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'bash'),
            exclude: ['**/*', '!mountEfs.sh', '!metadataUtilities.sh', '!ec2-certificates.crt'],
        });
    }
}
exports.MountableEfs = MountableEfs;
_a = JSII_RTTI_SYMBOL_1;
MountableEfs[_a] = { fqn: "aws-rfdk.MountableEfs", version: "0.34.0" };
//# sourceMappingURL=data:application/json;base64,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